/*******************************************************************************
 *
 * FILE:          ecnr.cpp
 *
 * SW-COMPONENT:  FC_Phone application
 *
 * PROJECT:
 *
 * DESCRIPTION:   CCA Application.
 *
 * AUTHOR:
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "FC_Phone_ECNR.h"
#include "../FC_Phone_main.h"
#include <qmutex.h>

#include "FC_Phone_ecnr-introspection.h"

#include "FC_Phone_Audio.h"

static GDBusConnection* qbus = NULLPTR;

static OrgBoschEcnrService *ecnrInterfaceProxy;
static _T_CCA_ECNR_If* EcnrCCA_callbacks = NULLPTR;

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
#include "trcGenProj/Header/FC_Phone_ECNR.cpp.trc.h"
#endif

tBool m_bRequestRunning;
std::vector<tEcnrData> m_oEventBuffer;
QMutex m_oBufferLock;

/*******************************************************************************
 *
 * FUNCTION:    _FB_ecnr_init_completion_callback
 *
 * DESCRIPTION: _FB_ecnr_init_completion_callback received from stack when the ecnr_init is done.
 *
 * PARAMETER:      proxy, poResult, userdata
 *
 * RETURNVALUE:    void
 *
 *******************************************************************************/
static void _FB_ecnr_init_completion_callback(GObject* source_object, GAsyncResult* poResult, gpointer user_data)
{
   ETG_TRACE_USR4(( "_FB_ecnr_init_completion_callback " ));
   /* To remove lint warnings */
   (tVoid) user_data;
   (tVoid) source_object;
   GError *error = NULLPTR;
   ecnrInterfaceProxy = org_bosch_ecnr_service_proxy_new_finish(poResult, &error);
   if (!ecnrInterfaceProxy)
   {
      ETG_TRACE_ERRMEM(("-PHONE-ERR-: ecnrInterfaceProxy == NULL!"));
      ETG_TRACE_FATAL(( "-PHONE-ERR-: ecnrInterfaceProxy == NULL!"));
      ETG_TRACE_ERR(("  ERROR ::Proxy not created due to : %s",error->message));
      g_error_free(error);
      return;
   }
   else
   {
      ETG_TRACE_USR4(("ECNR proxy created successfully"));
   }
}

/*******************************************************************************
 *
 * FUNCTION:    FB_ecnr_init
 *
 * DESCRIPTION: This method is called to establish GDbus connection.
 *
 * PARAMETER:      dataset
 *
 * RETURNVALUE:    gboolean
 *
 *******************************************************************************/
void FB_ecnr_init(GDBusConnection *connection)
{
   ETG_TRACE_USR4(( "FB_ecnr_init " ));
   m_bRequestRunning = FALSE;

   if (connection)
   {
      ETG_TRACE_USR4(("GDBusConnection is NOT NULL"));
      qbus = connection;

      org_bosch_ecnr_service_proxy_new (qbus,
            G_DBUS_PROXY_FLAGS_NONE,
            ECNR_SERVICE_NAME,
            ECNR_OBJECT_PATH,
            NULLPTR,
            &_FB_ecnr_init_completion_callback,
            NULLPTR);
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::GDBusConnection is null"));
   }
}

/*******************************************************************************
 *
 * FUNCTION:    FB_ecnr_deinit
 *
 * DESCRIPTION: This method is called to remove GDbus connection.
 *
 * PARAMETER:      dataset
 *
 * RETURNVALUE:    void
 *
 *******************************************************************************/
void FB_ecnr_deinit(GDBusConnection *connection)
{
   ETG_TRACE_USR4(( "FB_ecnr_deinit " ));

   if (!connection)
   {
      ETG_TRACE_FATAL(("FATAL_ERROR ::GDBusConnection is null"));
   }
   else
   {
      ETG_TRACE_FATAL(("Bus and proxy will be removed"));
      g_object_unref(connection);
      g_object_unref(ecnrInterfaceProxy);
      ecnrInterfaceProxy = NULLPTR;
   }
}

/*******************************************************************************
 *
 * FUNCTION:    _ecnrInitialize_completion_callback
 *
 * DESCRIPTION: _ecnrInitialize_completion_callback received from stack when the ecnrInitialize is done.
 *
 * PARAMETER:      proxy, poResult, userdata
 *
 * RETURNVALUE:    void
 *
 *******************************************************************************/
static void _ecnrInitialize_completion_callback(GObject *proxy, GAsyncResult *poResult, gpointer userdata)
{   
   ETG_TRACE_USR4(( "_ecnrInitialize_completion_callback " ));
   /* To remove lint warnings */
   (tVoid) userdata;
   GError *error = NULLPTR;
   gboolean retval = FALSE;
   retval = org_bosch_ecnr_service_call_ecnr_initialize_finish((OrgBoschEcnrService*)proxy, poResult, &error);
   if ( FALSE == retval )
   {
      ETG_TRACE_ERR(("  ERROR ::ecnrInitialize_completion_callback error : %s",error->message));
   }

   vProcessPendingRequest();
}

/*******************************************************************************
 *
 * FUNCTION:    ecnrInitialize
 *
 * DESCRIPTION: initialise ecnr with dataset.
 *
 * PARAMETER:      dataset
 *
 * RETURNVALUE:    gboolean
 *
 *******************************************************************************/
gboolean ecnrInitialize(tUInt dataset, tBool bisFromQueue)
{	
   ETG_TRACE_USR4(("_ecnrInitialize dataset = %d", dataset));
   QMutexLocker lock(&m_oBufferLock);
   ETG_TRACE_USR4(( " Lock aquired." ));
   if( m_bRequestRunning && !bisFromQueue)
   {
      ETG_TRACE_USR4(( " Inserting into the queue.." ));
      tEcnrData oData;
      oData.eventType = ECNR_INITIALIZE;
      oData.u16Data = static_cast<tU16>(dataset);
      m_oEventBuffer.push_back(oData);
      lock.unlock();
      return FALSE;
   }
   if (!ecnrInterfaceProxy)
   {
      ETG_TRACE_USR4(( "ecnrInterfaceProxy is NULL " ));
      lock.unlock();
      return TRUE;
   }
   m_bRequestRunning = TRUE;
   lock.unlock();

   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetSaScoEcnrInit(EN_ECNR_INIT);
   org_bosch_ecnr_service_call_ecnr_initialize(ecnrInterfaceProxy,(guchar)ECNR_APPID_PHONE,(guint)dataset,NULLPTR,&_ecnrInitialize_completion_callback, NULLPTR);

   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION:    _ecnrStartAudio_completion_callback
 *
 * DESCRIPTION: Audio packets will be transferred once we receive the _ecnrStartAudio_completion_callback.
 *
 * PARAMETER:      proxy, pResult, userdata
 *
 * RETURNVALUE:    void
 *
 *******************************************************************************/
 
static void _ecnrStartAudio_completion_callback(GObject *proxy, GAsyncResult *pResult, gpointer userdata)
{
   ETG_TRACE_USR4(( "_ecnrStartAudio_completion_callback " ));
   /* To remove lint warnings */
   (tVoid) proxy;
   (tVoid) userdata;
   GError *error = NULLPTR;
   gboolean retval = FALSE;
   tBool bCallBackStatus = TRUE;
   retval = org_bosch_ecnr_service_call_ecnr_start_audio_finish((OrgBoschEcnrService*)proxy, pResult, &error);
   if ( FALSE == retval )
   {
      bCallBackStatus = FALSE;
      ETG_TRACE_ERR(("  ERROR ::ecnrStartAudio_completion_callback error : %s",error->message));
   }

   EcnrCCA_callbacks->ecnrStartAudioCB(bCallBackStatus);
   vProcessPendingRequest();
}

/*******************************************************************************
 *
 * FUNCTION:    ecnrStartAudio
 *
 * DESCRIPTION: ecnrStartAudio will be called to transfer the audio packets.
 *
 * PARAMETER:      bisFromQueue
 *
 * RETURNVALUE:    gboolean
 *
 *******************************************************************************/
gboolean ecnrStartAudio(tBool bisFromQueue)
{
   ETG_TRACE_USR4(( "_ecnrStartAudio " ));
   QMutexLocker lock(&m_oBufferLock);
   ETG_TRACE_USR4(( " Lock aquired." ));
   if( m_bRequestRunning && !bisFromQueue)
   {
      ETG_TRACE_USR4(( " Inserting into the queue.." ));
      tEcnrData oData;
      oData.eventType = ECNR_STARTAUDIO;
      oData.u16Data = INTEGER_DEFAULT;
      m_oEventBuffer.push_back(oData);
      lock.unlock();
      return FALSE;
   }

   if (!ecnrInterfaceProxy)
   {
      ETG_TRACE_USR4(( "ecnrInterfaceProxy is NULL " ));
      lock.unlock();
      return TRUE;
   }
   m_bRequestRunning = TRUE;
   lock.unlock();

   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetSaScoEcnrInit(EN_ECNR_STARTAUDIO);
   org_bosch_ecnr_service_call_ecnr_start_audio(ecnrInterfaceProxy,(guchar)ECNR_APPID_PHONE, NULLPTR, &_ecnrStartAudio_completion_callback, NULLPTR);

   return FALSE;
}

 /*******************************************************************************
 *
 * FUNCTION:    _ecnrStopAudio_completion_callback
 *
 * DESCRIPTION: Transfer of audio packets will be stopped once we receive the _ecnrStopAudio_completion_callback.
 *
 * PARAMETER:      proxy, pResult, userdata
 *
 * RETURNVALUE:    void
 *
 *******************************************************************************/
 
static void _ecnrStopAudio_completion_callback(GObject *proxy, GAsyncResult *pResult, gpointer userdata)
{
   /* To remove lint warnings */
   ETG_TRACE_USR4(( "_ecnrStopAudio_completion_callback " ));
   (tVoid) proxy;
   (tVoid) userdata;
   GError *error = NULLPTR;
   gboolean retval = FALSE;
   tBool bCallBackStatus = TRUE;
   retval = org_bosch_ecnr_service_call_ecnr_stop_audio_finish((OrgBoschEcnrService*)proxy, pResult, &error);
   if ( FALSE == retval )
   {
      bCallBackStatus = FALSE;
      ETG_TRACE_ERR(("  ERROR ::ecnrStopAudio_completion_callback error : %s",error->message));
   }
   EcnrCCA_callbacks->ecnrStopAudioCB(bCallBackStatus);
   vProcessPendingRequest();
}

/*******************************************************************************
 *
 * FUNCTION:    ecnrStopAudio
 *
 * DESCRIPTION: Audio packet transfer will be stopped.
 *
 * PARAMETER:      bisFromQueue
 *
 * RETURNVALUE:    gboolean
 *
 *******************************************************************************/
gboolean ecnrStopAudio(tBool bisFromQueue) //CMG3G-4847 - To solve Lint
{
   ETG_TRACE_USR4(( "_ecnrStopAudio " ));
   QMutexLocker lock(&m_oBufferLock);
   ETG_TRACE_USR4(( " Lock aquired." ));
   if( m_bRequestRunning && !bisFromQueue)
   {
      ETG_TRACE_USR4(( " Inserting into the queue.." ));
      tEcnrData oData;
      oData.eventType = ECNR_STOPAUDIO;
      oData.u16Data = INTEGER_DEFAULT;
      m_oEventBuffer.push_back(oData);
      lock.unlock();
      return FALSE;
   }

   if (!ecnrInterfaceProxy)
   {
      ETG_TRACE_USR4(( "ecnrInterfaceProxy is NULL " ));
      lock.unlock();
      return TRUE;
   }

   tU8 u8EcnrSaStatus = (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vGetEcnrSaScoStatus();
   ETG_TRACE_USR4(("u8EcnrSaStatus: %d ",u8EcnrSaStatus ));

   if(EN_ECNR_STARTAUDIO == (u8EcnrSaStatus & EN_ECNR_STARTAUDIO))
   {
      m_bRequestRunning = TRUE;
      lock.unlock();
      (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vResetSaScoEcnrInit(EN_ECNR_STARTAUDIO);
      org_bosch_ecnr_service_call_ecnr_stop_audio(ecnrInterfaceProxy,(guchar)ECNR_APPID_PHONE, NULLPTR, &_ecnrStopAudio_completion_callback, NULLPTR);
   }

   return FALSE;
}

 /*******************************************************************************
 *
 * FUNCTION:    _ecnrDestroy_completion_callback
 *
 * DESCRIPTION: ecnr will be reset once we receive the _ecnrDestroy_completion_callback.
 *
 * PARAMETER:      proxy, pResult, userdata
 *
 * RETURNVALUE:    void
 *
 *******************************************************************************/

static void _ecnrDestroy_completion_callback(GObject *proxy, GAsyncResult *pResult, gpointer userdata){

   /* To remove lint warnings */
   ETG_TRACE_USR4(( "_ecnrDestroy_completion_callback " ));
   (tVoid) proxy;	
   (tVoid) userdata;
   GError *error = NULLPTR;
   gboolean retval = FALSE;  
   retval = org_bosch_ecnr_service_call_ecnr_destroy_finish((OrgBoschEcnrService*)proxy, pResult, &error);
   if ( FALSE == retval )
   {
      ETG_TRACE_ERR(("  ERROR ::_ecnrDestroy_completion_callback error : %s",error->message));

   }
   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vProcessSourceActivity();
   vProcessPendingRequest();
}

/*******************************************************************************
 *
 * FUNCTION:    ecnrDestroy
 *
 * DESCRIPTION: Audio channel will be destroyed.
 *
 * PARAMETER:      bisFromQueue
 *
 * RETURNVALUE:    gboolean
 *
 *******************************************************************************/
gboolean ecnrDestroy(tBool bisFromQueue) //CMG3G-4847 - To solve Lint
{
   ETG_TRACE_USR4(( "_ecnrDestroy " ));
   QMutexLocker lock(&m_oBufferLock);
   ETG_TRACE_USR4(( " Lock aquired." ));
   if( m_bRequestRunning && !bisFromQueue )
   {
      ETG_TRACE_USR4(( " Inserting into the queue.." ));
      tEcnrData oData;
      oData.eventType = ECNR_DESTROY;
      oData.u16Data = INTEGER_DEFAULT;
      m_oEventBuffer.push_back(oData);
      lock.unlock();
      return FALSE;
   }

   if (!ecnrInterfaceProxy)
   {
      ETG_TRACE_USR4(( "ecnrInterfaceProxy is NULL " ));
      lock.unlock();
      return TRUE;
   }

   m_bRequestRunning = TRUE;
   lock.unlock();

   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vResetSaScoEcnrInit(EN_ECNR_STARTAUDIO);
   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vResetSaScoEcnrInit( EN_ECNR_INIT );
   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vResetSaScoEcnrInit( EN_ECNR_CONFIGURATION );

   org_bosch_ecnr_service_call_ecnr_destroy(ecnrInterfaceProxy,(guchar)ECNR_APPID_PHONE, NULLPTR, &_ecnrDestroy_completion_callback, NULLPTR);

   return FALSE;
}

 /*******************************************************************************
 *
 * FUNCTION:    _ecnrconfiguration_completion_callback
 *
 * DESCRIPTION: ecnr is configured successfully once we receive the _ecnrconfiguration_completion_callback.
 *
 * PARAMETER:      proxy, pResult, userdata
 *
 * RETURNVALUE:    void
 *
 *******************************************************************************/

static void _ecnrconfiguration_completion_callback(GObject *proxy, GAsyncResult *pResult, gpointer userdata)
{
   ETG_TRACE_USR4(( "_ecnrconfiguration_completion_callback " ));
   /* To remove lint warnings */
   (tVoid) userdata;
   gboolean retval = FALSE;
   GError *error = NULLPTR;
   retval = org_bosch_ecnr_service_call_ecnr_set_configuration_finish((OrgBoschEcnrService*)proxy,pResult,&error);
   if ( FALSE == retval )
   {
      ETG_TRACE_ERR(("  ERROR ::ecnrconfiguration_completion_callback error : %s",error->message));
   }

   vProcessPendingRequest();
}

/*******************************************************************************
 *
 * FUNCTION:    ecnrconfiguration
 *
 * DESCRIPTION: ecnr will be configured for audio packet transfer(Wideband/Narrowband).
 *
 * PARAMETER:      u8ConfigID, bisFromQueue
 *
 * RETURNVALUE:    gboolean
 *
 *******************************************************************************/ 
gboolean ecnrconfiguration(tU8 u8ConfigID, tBool bisFromQueue)
{
   ETG_TRACE_USR4(("_ecnrconfiguration dataset = %d", u8ConfigID));
   ETG_TRACE_USR4(("Testing ecnr"));
   QMutexLocker lock(&m_oBufferLock);
   ETG_TRACE_USR4(( " Lock aquired." ));
   if( m_bRequestRunning && !bisFromQueue)
   {
      ETG_TRACE_USR4(( " Inserting into the queue.." ));
      tEcnrData oData;
      oData.eventType = ECNR_CONFIGURATION;
      oData.u16Data = static_cast<tU16>(u8ConfigID);
      m_oEventBuffer.push_back(oData);
      lock.unlock();
      return FALSE;
   }

   if (!ecnrInterfaceProxy)
   {
      ETG_TRACE_USR4(( "ecnrInterfaceProxy is NULL " ));
      lock.unlock();
      return TRUE;
   }

   m_bRequestRunning = TRUE;
   lock.unlock();
   (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vSetSaScoEcnrInit(EN_ECNR_CONFIGURATION);

   org_bosch_ecnr_service_call_ecnr_set_configuration(ecnrInterfaceProxy,(guchar)ECNR_APPID_PHONE,(guint)u8ConfigID, NULLPTR,&_ecnrconfiguration_completion_callback, NULLPTR);
   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION:    _ecnrMuteSwitch_completion_callback
 *
 * DESCRIPTION: ecnr mute switch is done once we receive the _ecnrMuteSwitch_completion_callback.
 *
 * PARAMETER:      proxy, pResult, userdata
 *
 * RETURNVALUE:    void
 *
 *******************************************************************************/
 
static void
_ecnrMuteSwitch_completion_callback(GObject *proxy, GAsyncResult *pResult, gpointer userdata)
{
   ETG_TRACE_USR4(( "_ecnrMuteSwitch_completion_callback " ));
   /* To remove lint warnings */
   (tVoid) proxy;
   (tVoid) userdata;
   gboolean retval = FALSE;
   GError *error = NULLPTR;
   retval = org_bosch_ecnr_service_call_ecnr_set_send_mute_switch_finish((OrgBoschEcnrService*)proxy, pResult, &error);
   if ( FALSE == retval )
   {   
      ETG_TRACE_ERR(("  ERROR ::ecnrMuteSwitch_completion_callback error : %s",error->message));
   }
}

/*******************************************************************************
 *
 * FUNCTION:    ecnrMuteSwitch
 *
 * DESCRIPTION: ecnr mute switch can be intitiated.
 *
 * PARAMETER:      ecnrSendMuteSwitch
 *
 * RETURNVALUE:    gboolean
 *
 *******************************************************************************/ 
gboolean ecnrMuteSwitch(tBool ecnrSendMuteSwitch)
{
   ETG_TRACE_USR4(( " ecnrMuteSwitch %d", ecnrSendMuteSwitch ));
   if (!ecnrInterfaceProxy)
   {
      ETG_TRACE_USR4(( "ecnrInterfaceProxy is NULL " ));
      return TRUE;
   }
   org_bosch_ecnr_service_call_ecnr_set_send_mute_switch(ecnrInterfaceProxy,(guchar)ECNR_APPID_PHONE,(guint)ecnrSendMuteSwitch, NULLPTR,&_ecnrMuteSwitch_completion_callback, NULLPTR);
   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION:    register_ecnr_callbacks
 *
 * DESCRIPTION: Sets qwa parameters.
 *
 * PARAMETER:      None
 *
 * RETURNVALUE:    None
 *
 *******************************************************************************/
void register_ecnr_callbacks(_T_CCA_ECNR_If *vCCA_callbacks)
{
   EcnrCCA_callbacks = vCCA_callbacks;
}
/*******************************************************************************
 *
 * FUNCTION:    vProcessPendingRequest
 *
 * DESCRIPTION: Process the pending request.
 *
 * PARAMETER:      None
 *
 * RETURNVALUE:    None
 *
 *******************************************************************************/
void vProcessPendingRequest()
{
   ETG_TRACE_USR4(( " vProcessPendingRequest.." ));

   tEcnrData oEcnrData;
   // process pending request if any.
   QMutexLocker lock(&m_oBufferLock);
   ETG_TRACE_USR4(( " Lock aquired." ));
   if( m_oEventBuffer.size() > 0 )
   {
      ETG_TRACE_USR4(( " Retrieving from the queue.." ));

      oEcnrData = m_oEventBuffer[0];
      m_oEventBuffer.erase(m_oEventBuffer.begin());
      lock.unlock();
   }
   else
   {
      // No pending request available.
      m_bRequestRunning = FALSE;
      lock.unlock();
      return;
   }

   tU8 u8EcnrSaStatus = (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioUtility->vGetEcnrSaScoStatus();
   switch(oEcnrData.eventType)
   {
      case ECNR_INITIALIZE:
      {
         if( !(u8EcnrSaStatus & EN_ECNR_INIT ))
         {
            ecnrInitialize(static_cast<tUInt> (oEcnrData.u16Data), TRUE);
         }
         else
         {
            vProcessPendingRequest();
         }
      }
      break;
      case ECNR_CONFIGURATION:
      {
         if( !(u8EcnrSaStatus & EN_ECNR_CONFIGURATION ))
         {
            ecnrconfiguration(static_cast<tU8> (oEcnrData.u16Data), TRUE);
         }
         else
         {
            vProcessPendingRequest();
         }
      }
      break;
      case ECNR_STARTAUDIO:
      {
         ETG_TRACE_USR4(( "ECNR_STARTAUDIO :: %d  %d ", u8EcnrSaStatus, EN_ECNR_STARTAUDIO ));
         if( ( EN_ECNR_START_AUDIO ==  ( EN_ECNR_START_AUDIO & u8EcnrSaStatus ) )
               && ( EN_ECNR_STARTAUDIO !=  ( EN_ECNR_STARTAUDIO & u8EcnrSaStatus )))
         {
            ecnrStartAudio(TRUE);
         }
         else
         {
            vProcessPendingRequest();
         }
      }
      break;
      case ECNR_STOPAUDIO:
      {
         if( u8EcnrSaStatus & EN_ECNR_STARTAUDIO )
         {
            ecnrStopAudio(TRUE);
         }
         else
         {
            vProcessPendingRequest();
         }
      }
      break;
    case ECNR_DESTROY:
      {
         if( u8EcnrSaStatus & EN_ECNR_INIT )
         {
            ecnrDestroy(TRUE);
         }
         else
         {
            vProcessPendingRequest();
         }
      }
      break;
      default:
         break;
   }
}
/*******************************************************************************
 *
 * FUNCTION:    clearEventBuffer
 *
 * DESCRIPTION: Clears the Eventbuffer.
 *
 * PARAMETER:      None
 *
 * RETURNVALUE:    None
 *
 *******************************************************************************/
void clearEventBuffer()
{
   ETG_TRACE_USR4(( "clearEventBuffer" ));

   QMutexLocker lock(&m_oBufferLock);
   if (m_bRequestRunning)
   {
      m_oEventBuffer.clear();
   }
   lock.unlock();
}
