/**
 * @file FC_Phone_AudioPlayer.cpp
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 *
 * @addtogroup FC_Phone
 *
 * @brief Implementation to Play audio files through GStreamer.
  */

#include "../FC_Phone_main.h"
#include "../FC_Phone_service_Telephone.h"

#include "FC_Phone_AudioPlayer.h"
#include "../Interface/FC_Phone_DBusInterface.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
#include "trcGenProj/Header/FC_Phone_AudioPlayer.cpp.trc.h"
#endif

GMainLoop *FC_Phone_AudioPlayer::gstMainLoop = NULLPTR;
FC_Phone_AudioPlayer* FC_Phone_AudioPlayer::poAudioPlayer = NULLPTR;
tBool FC_Phone_AudioPlayer::m_bPlayRepeated = true;

tBool FC_Phone_AudioPlayer::m_bIsPlayStarted = false;
bool FC_Phone_AudioPlayer::m_bWaitforCondSignal = false;

pthread_mutex_t PlayerThread_Mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t threadWait_CondSignal = PTHREAD_COND_INITIALIZER;
pthread_attr_t playerThread_Attr;

std::stringstream FC_Phone_AudioPlayer::m_szAlsaDeviceName;

FC_Phone_AudioPlayer::FC_Phone_AudioPlayer()
{
   ETG_TRACE_USR4(("  FC_Phone_AudioPlayer::FC_Phone_AudioPlayer entered. "));

   pthread_cond_init(&threadWait_CondSignal, NULLPTR);
   pthread_mutex_init(&PlayerThread_Mutex, NULLPTR);

   m_bSetAttributes = false;
   vSetAttributeToPthread();
}

FC_Phone_AudioPlayer::~FC_Phone_AudioPlayer()
{
   ETG_TRACE_USR4(("  FC_Phone_AudioPlayer::~FC_Phone_AudioPlayer entered. "));
}

FC_Phone_AudioPlayer* FC_Phone_AudioPlayer::getInstance()
{
   ETG_TRACE_USR4(("  FC_Phone_AudioPlayer::getInstance entered. "));

   if (!poAudioPlayer)
   {
      poAudioPlayer = OSAL_NEW FC_Phone_AudioPlayer();
   }

   return poAudioPlayer;
}

void FC_Phone_AudioPlayer::deleteInstance()
{
   ETG_TRACE_USR4((" FC_Phone_AudioPlayer::deleteInstance entered "));
   if (poAudioPlayer)
   {
      delete poAudioPlayer;
      poAudioPlayer = NULLPTR;
   }
}

tVoid FC_Phone_AudioPlayer::vSetAttributeToPthread()
{
   ETG_TRACE_USR4((" FC_Phone_AudioPlayer::vSetAttributeToPthread entered "));
   if(m_bSetAttributes == false)
   {
      sched_param param;
      int ret = 0;
      tBool bTemp = true;

      /* initialized with default attributes */
      ret = pthread_attr_init(&playerThread_Attr);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_init(&playerThread_Attr) returned with ret= %d ", ret ));
         bTemp = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_init(&playerThread_Attr) returned ret= %d ", ret ));
      }

      ret = pthread_attr_setschedpolicy(&playerThread_Attr, SCHED_FIFO);  // <-- RT prio policy
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setschedpolicy(&playerThread_Attr, SCHED_FIFO) returned with ret= %d ", ret ));
         bTemp = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setschedpolicy(&playerThread_Attr, SCHED_FIFO) returned ret= %d ", ret ));
      }

      /* safe to get existing scheduling param */
      ret = 0;
      ret = pthread_attr_getschedparam(&playerThread_Attr, &param);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_getschedparam(&playerThread_Attr, &param) returned with ret= %d ", ret ));
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_getschedparam(&playerThread_Attr, &param) returned ret= %d ", ret ));
      }

      /* set the priority; others are unchanged */
      param.sched_priority = PTHREAD_RT_PRIO;

      /* setting the new scheduling param */
      ret = pthread_attr_setschedparam(&playerThread_Attr, &param);
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setschedparam(&playerThread_Attr, &param) returned with ret= %d ", ret ));
         bTemp = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setschedparam(&playerThread_Attr, &param) returned ret= %d ", ret ));
      }

      ret = pthread_attr_setinheritsched(&playerThread_Attr, PTHREAD_EXPLICIT_SCHED);  // <-- set RT prio to be inherited
      if(0 != ret)
      {
         ETG_TRACE_ERR(( " ERROR: pthread_attr_setinheritsched(&playerThread_Attr, PTHREAD_EXPLICIT_SCHED) returned with ret= %d ", ret ));
         bTemp = false;
      }
      else
      {
         ETG_TRACE_USR1(( " SUCCESS: pthread_attr_setinheritsched(&playerThread_Attr, PTHREAD_EXPLICIT_SCHED) returned ret= %d ", ret ));
      }

      m_bSetAttributes = bTemp;
   }
   else
   {
      ETG_TRACE_USR1((" SUCCESS:Attributes are already set "));
   }
   // set thread prio: END

   return;

}

bool FC_Phone_AudioPlayer::playFile(const char *filePath, bool repeat)
{
   ETG_TRACE_USR4(("  FC_Phone_AudioPlayer::playFile entered %d ", repeat));

   bool playStarted = false;
   const char *filename = NULLPTR;

   if (filePath)
   {
      vSetAttributeToPthread();

      if(m_bSetAttributes == true)
      {
         filename = strdup(filePath);

         ETG_TRACE_USR2(("playFile: File To Play: %s", filename));

         pthread_t threadHandle = 0;

         int result = pthread_create(&threadHandle, NULLPTR, FC_Phone_AudioPlayer::startPlayback,
               (tVoid*)const_cast<char*>(filename));

         if(0 == result)
         {
            pthread_mutex_lock(&PlayerThread_Mutex);
            ETG_TRACE_USR4(("PlayerThread_Mutex lock acquired from playFile"));

            ETG_TRACE_USR2(("Wait for the conditional signal"));

            m_bWaitforCondSignal = false;
            ETG_TRACE_USR4(("m_bWaitforCondSignal :%d", m_bWaitforCondSignal));

            // This condition check is used to overcome the Spurious wakeup of pthread without conditional signal(threadWait_CondSignal)
            while(!m_bWaitforCondSignal)
            {
               pthread_cond_wait(&threadWait_CondSignal, &PlayerThread_Mutex);
            }

            ETG_TRACE_USR2(("Condition signal received"));

            pthread_mutex_unlock(&PlayerThread_Mutex);
            ETG_TRACE_USR4(("PlayerThread_Mutex lock released from playFile"));

            if (m_bIsPlayStarted)
            {
               playStarted = true;
               m_bPlayRepeated = repeat;
            }
            else
            {
               ETG_TRACE_USR4(("Something went wrong with the Player thread"));
               stopPlayback();
            }

         }
         else
         {
            ETG_TRACE_ERR(("Player thread creation failed"));
            free((void*)filename);
         }
      }
      else
      {
         ETG_TRACE_ERR((" ERROR: Attributes are not set properly "));
      }
   }
   else
   {
      ETG_TRACE_ERR(("No file received to play"));
   }

   return playStarted;
}

gboolean FC_Phone_AudioPlayer::gstBusCallBack(GstBus *bus_local, GstMessage *msg, gpointer data)
{
   ETG_TRACE_USR4(("  FC_Phone_AudioPlayer::gstBusCallBack entered. "));
   (tVoid) bus_local;

   GstElement *playElement = (GstElement *)data;

   if (!msg)
   {
      ETG_TRACE_USR4(("GstBusCallBack: msg is NULL"));
      return TRUE;
   }

   switch (GST_MESSAGE_TYPE(msg))
   {
      case GST_MESSAGE_BUFFERING:
      {
         gint percent = 0;
         gst_message_parse_buffering(msg, &percent);
         ETG_TRACE_USR4(("gstBusCallBack: Buffering"));
      }
      break;

      case GST_MESSAGE_EOS:
      {
         // PlayBack finished
         ETG_TRACE_USR4(("gstBusCallBack: EOS STREAM RECEIVED STOPPING THE PLAYER"));

         // For playing the waiting mode tone(from FC_Phone perspective), Repeat will be true always
         // TODO: The else portion needs to be discussed
         if (m_bPlayRepeated)
         {
            gint64 time_nanoseconds = 0;
            if (!gst_element_seek(playElement, 1.0, GST_FORMAT_TIME, GST_SEEK_FLAG_FLUSH, GST_SEEK_TYPE_SET,
                  time_nanoseconds, GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE))
            {
               // TODO: Need to discuss what shall be done in this case
               ETG_TRACE_ERR(("gstBusCallBack: Seeking failed"));
            }
            else
            {
               if (GST_STATE_CHANGE_FAILURE == gst_element_set_state(playElement, GST_STATE_PLAYING))
               {
                  ETG_TRACE_USR4(("gstBusCallBack: State change(GST_STATE_PLAYING) failed- GST_STATE_CHANGE_FAILURE"));
                  ETG_TRACE_ERRMEM(("gstBusCallBack: State change(GST_STATE_PLAYING) failed- GST_STATE_CHANGE_FAILURE"));
               }
            }
         }
      }
      break;

      case GST_MESSAGE_STATE_CHANGED:
      {
         // gst_play_pipeline changed state
         GstState oldState;
         GstState newState;
         GstState pending;
         gst_message_parse_state_changed(msg, &oldState, &newState, &pending);
         gstStateChanged(newState, oldState);
      }
      break;

      case GST_MESSAGE_WARNING:
      {
         // Warning received from GST
         gchar *debug = NULLPTR;
         GError *error = NULLPTR;
         gst_message_parse_warning(msg, &error, &debug);
         ETG_TRACE_ERR(("gstBusCallBack: Warning is =%s", error->message));
         g_free(debug);
         g_error_free(error);
      }
      break;

      case GST_MESSAGE_ERROR:
      {
         // Error received from GST
         gchar *debug = NULLPTR;
         GError *error = NULLPTR;
         gst_message_parse_error(msg, &error, &debug);
         if (error && error->message)
         {
            ETG_TRACE_ERR(("GstBusCallBack: Error is  =%s", error->message));
         }
         g_free(debug);
         g_error_free(error);

         // Stop the playback and kill the thread
         FC_Phone_AudioPlayer* poAudioPlayerInstance = FC_Phone_AudioPlayer::getInstance();
         poAudioPlayerInstance->vSendSignaltoWaitingThread(false);
      }
      break;

      default:
      {
         ETG_TRACE_USR2 (("GstBusCallBack :got %s message", GST_MESSAGE_TYPE_NAME (msg)));
         break;
      }
   }
   return TRUE;
}

void FC_Phone_AudioPlayer::gstStateChanged(GstState new_state, GstState old_state)
{
   ETG_TRACE_USR4(("  FC_Phone_AudioPlayer::gstStateChanged entered. "));

   ETG_TRACE_USR2(("gstStateChanged: %u => %u", ETG_ENUM(TR_GSTSTATE, old_state), ETG_ENUM(TR_GSTSTATE, new_state)));

   if (GST_STATE_PLAYING == new_state)
   {
      FC_Phone_AudioPlayer* poAudioPlayerInstance = FC_Phone_AudioPlayer::getInstance();
      poAudioPlayerInstance->vSendSignaltoWaitingThread(true);
   }
}

bool FC_Phone_AudioPlayer::setAlsaDevice(const char *deviceName)
{
   ETG_TRACE_USR4(("  FC_Phone_AudioPlayer::setAlsaDevice entered. "));

   bool returnVal = true;

   if(!deviceName)
   {
      ETG_TRACE_ERR(("ALSA deviceName is received as Null"));
      returnVal = false;
   }
   else
   {
      ETG_TRACE_USR2(("setAlsaDevice: device =%s", deviceName));
      m_szAlsaDeviceName.str("");
      m_szAlsaDeviceName << deviceName;
   }

   ETG_TRACE_USR2(("setAlsaDevice: Hardware name is %s", m_szAlsaDeviceName.str().c_str()));

   return returnVal;
}

void FC_Phone_AudioPlayer::stopPlayback()
{
   ETG_TRACE_USR4(("FC_Phone_AudioPlayer::stopPlayback entered"));

   pthread_mutex_lock(&PlayerThread_Mutex);
   ETG_TRACE_USR4(("PlayerThread_Mutex lock acquired from stopPlayback"));

   if (gstMainLoop)
   {
      if (g_main_loop_is_running(gstMainLoop))
      {
         g_main_loop_quit(gstMainLoop);

         m_bWaitforCondSignal = false;
         ETG_TRACE_USR4(("m_bWaitforCondSignal :%d", m_bWaitforCondSignal));

         // This condition check is used to overcome the Spurious wakeup of pthread without conditional signal(threadWait_CondSignal)
         while(!m_bWaitforCondSignal)
         {
            pthread_cond_wait(&threadWait_CondSignal, &PlayerThread_Mutex);
         }

         ETG_TRACE_USR2(("Condition signal received in stopPlayback"));

         m_bWaitforCondSignal = false;
         ETG_TRACE_USR4(("m_bWaitforCondSignal :%d", m_bWaitforCondSignal));
      }
      else
      {
         ETG_TRACE_USR4(("g_main_loop is not RUNNING"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("gstMainLoop is NULL"));
   }

   pthread_mutex_unlock(&PlayerThread_Mutex);
   ETG_TRACE_USR4(("PlayerThread_Mutex lock released from stopPlayback"));

   ETG_TRACE_USR4(("stopPlayback: exit"));
}

void *FC_Phone_AudioPlayer::startPlayback(void* fileInfo)
{
   ETG_TRACE_USR4(("  FC_Phone_AudioPlayer::startPlayback entered. "));

   FC_Phone_AudioPlayer* poAudioPlayerInstance = FC_Phone_AudioPlayer::getInstance();

   if (!fileInfo)
   {
      ETG_TRACE_USR4(("startPlayback:fileInfo is null"));
      poAudioPlayerInstance->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   if (m_szAlsaDeviceName.str().empty())
   {
      ETG_TRACE_USR4(("startPlayback:m_szAlsaDeviceName is empty"));
      poAudioPlayerInstance->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   char* file = NULLPTR;
   file = (char*)fileInfo;

   std::stringstream fileToPlay;

   fileToPlay.str("");
   fileToPlay<<"file://";
   fileToPlay<<file;

   ETG_TRACE_USR4(("startPlayback: filename:%s", file));
   ETG_TRACE_USR4(("startPlayback: filename to play:%s", fileToPlay.str().c_str()));

   free(fileInfo);

   /* Initialise GStreamer */
   GError *err = NULLPTR;
   if ( false == gst_init_check(NULLPTR, NULLPTR, &err))
   {
      ETG_TRACE_USR4(("GST initialization failed: %s", (err && err->message) ? err->message : "unknown error occurred"));
      ETG_TRACE_ERRMEM(("GST initialization failed: %s", (err && err->message) ? err->message : "unknown error occurred"));

      if (err)
      {
         g_error_free (err);
      }

      poAudioPlayerInstance->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   /* set up the playbin */
   GstElement *playBinElement = gst_element_factory_make("playbin", "play");
   GstElement *sinkElement = gst_element_factory_make("alsasink", "sink");

   if ((!playBinElement ) || (!sinkElement))
   {
      ETG_TRACE_USR4(("Not able to create new gst element"));
      poAudioPlayerInstance->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   g_object_set((GObject*) playBinElement, "uri", fileToPlay.str().c_str(), NULL);
   g_object_set((GObject*) sinkElement, "device", m_szAlsaDeviceName.str().c_str(), NULL);
   g_object_set((GObject*) playBinElement, "audio-sink", sinkElement, NULL);

   /* Adding a message handler */
   GstBus* gstBus = gst_pipeline_get_bus(GST_PIPELINE_CAST((tVoid*) playBinElement));

   if (!gstBus)
   {
      ETG_TRACE_USR4(("startPlayback: gstBus is null: Unable to create pipeline"));
      gst_object_unref((GstObject*) playBinElement);
      poAudioPlayerInstance->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   gstMainLoop = g_main_loop_new(NULLPTR, FALSE);

   if(!gstMainLoop)
   {
      ETG_TRACE_USR4(("startPlayback: gstMainLoop is null"));
      gst_object_unref((GstObject*) playBinElement);
      poAudioPlayerInstance->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   guint gstbus_watch_id = gst_bus_add_watch(gstBus, gstBusCallBack, playBinElement);
   (tVoid) gstbus_watch_id;
   gst_object_unref(gstBus);

   if (GST_STATE_CHANGE_FAILURE == gst_element_set_state(playBinElement, GST_STATE_PLAYING))
   {
      ETG_TRACE_USR4(("startPlayback: State change(GST_STATE_PLAYING) failed- GST_STATE_CHANGE_FAILURE"));
      ETG_TRACE_ERRMEM(("startPlayback: State change(GST_STATE_PLAYING) failed- GST_STATE_CHANGE_FAILURE"));

      gst_object_unref((GstObject*) playBinElement);
      g_source_remove(gstbus_watch_id);
      poAudioPlayerInstance->vSendSignaltoWaitingThread(false);
      return NULL;
   }

   /* Run the PlayingLoop*/
   ETG_TRACE_USR4(("startPlayback: Playing the file"));
   g_main_loop_run(gstMainLoop);

   ETG_TRACE_USR4(("CLEANING PLAYING THREAD"));

   if (GST_STATE_CHANGE_FAILURE == gst_element_set_state(playBinElement, GST_STATE_NULL))
   {
      ETG_TRACE_USR4(("startPlayback: State change(GST_STATE_NULL) failed- GST_STATE_CHANGE_FAILURE"));
      ETG_TRACE_ERRMEM(("startPlayback: State change(GST_STATE_NULL) failed- GST_STATE_CHANGE_FAILURE"));
   }

   gst_object_unref((GstObject*) playBinElement);
   g_source_remove(gstbus_watch_id);
   if(gstMainLoop)
   {
      g_main_loop_unref (gstMainLoop);
      gstMainLoop = NULLPTR;
   }

   poAudioPlayerInstance->vSendSignaltoWaitingThread(false);

   ETG_TRACE_USR4(("startPlayback: PLAYING THREAD STOPPED"));

   return NULLPTR;
}

tVoid FC_Phone_AudioPlayer::vSendSignaltoWaitingThread(tBool bResponseStatus)
{
   ETG_TRACE_USR4(("FC_Phone_AudioPlayer::vSendSignaltoWaitingThread Entered with bResponseStatus = %d", bResponseStatus));

   m_bIsPlayStarted = bResponseStatus;
   ETG_TRACE_USR4(("m_bIsPlayStarted :%d", m_bIsPlayStarted));

   pthread_mutex_lock(&PlayerThread_Mutex);
   ETG_TRACE_USR4(("PlayerThread_Mutex lock acquired from vSendSignaltoWaitingThread"));

   m_bWaitforCondSignal = true;
   ETG_TRACE_USR4(("m_bWaitforCondSignal :%d", m_bWaitforCondSignal));

   ETG_TRACE_USR4(("Signal the waiting thread from vSendSignaltoWaitingThread"));
   pthread_cond_signal(&threadWait_CondSignal);
   pthread_mutex_unlock(&PlayerThread_Mutex);
   ETG_TRACE_USR4(("PlayerThread_Mutex lock released from vSendSignaltoWaitingThread"));
}
