/*******************************************************************************
 *
 * FILE:          FC_Messaging_service_Messaging.cpp
 *
 * SW-COMPONENT:  FC_Messaging application
 *
 * PROJECT:       Bosch
 *
 * DESCRIPTION:   CCA service Messaging.
 *
 * AUTHOR:        Arun Choudhary
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "FC_Messaging_Debug.h"
#include "FC_Messaging_main.h"
#include "FC_Messaging_clienthandler_BluetoothSetings.h"
#include "SendMessage/FC_Messaging_SendMessage.h"
#include "DbusClient/FC_Messaging_EvoDbusClient.h"
#include "DbusClient/FC_Messaging_DbusClientInterface.h"
#include "ReadMessageAloud/Gen3/FC_Messaging_ReadMessageAloud.h"
#include "FC_Messaging_StateMachine.h"
#include "FC_Messaging_WorkQueue.h"
#include "DbusClient/FC_Messaging_Datatype.h"
#include "./Database/FC_Messaging_SQLite.h"
#include "FileSystemManager/FC_Messaging_FileSystemManager.h"
#include "MsgList/FC_Messaging_Settings.h"
#include "FC_Messaging_clienthandler_Phonebook.h"
// Bugfix for GMMY15-5992 Large empty gap is present in text messages list
#include "FC_Messaging_phonebook.h"

#include <unistd.h>

#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_STREAMROUTERFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_STREAMROUTERFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_STREAMROUTERFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_STREAMROUTERFI_SERVICEINFO
#include "midw_fi_if.h"

#include "FC_Messaging_service_Messaging.h"

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_SERVICE
#include "trcGenProj/Header/FC_Messaging_service_Messaging.cpp.trc.h"
#endif

#define FC_MESSAGING_S_IMPORT_INTERFACE_MSG
#include "FC_Messaging_if.h"
#include "MsgList/FC_Messaging_alltypes.h"
//
/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

AudioRouteAllocateStatus_t
AllocateStatusState::aoAudioRouteAllocateStatus_t[SOURCE_NUM_MAX];
tU8 AllocateStatusState::u8CurrentAllocateSourceNr;

fc_messaging_tclService_Messaging
* fc_messaging_tclService_Messaging::m_poMessagingService = NULLPTR;
tBool fc_messaging_tclService_Messaging::m_bServerAvailable = FALSE;
most_msgfi_tclMsgMessagingDeviceConnectionStatus
			fc_messaging_tclService_Messaging::m_oMessagingDeviceConnection_Status;
most_msgfi_tclMsgSendMessageMethodStart* fc_messaging_tclService_Messaging::m_l_poSendMessageMethodStart = NULLPTR;
struct Job* fc_messaging_tclService_Messaging::m_l_prJob = NULLPTR;

/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/

BEGIN_MSG_MAP(fc_messaging_tclService_Messaging, ahl_tclBaseWork)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_CHANGEACCOUNT,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_ChangeAccount_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_REQUESTACCOUNTLIST,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_RequestAccountList_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_REQUESTFOLDERLIST,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_RequestFolderList_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_SENDMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_SendMessage_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_CREATEMESSAGELIST,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_CreateMessageList_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_REQUESTSLICEMESSAGELIST,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_RequestSliceMessageList_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_REQUESTSLICECONVERSATIONLIST,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_RequestSliceConversationList_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_GETSLICECONVERSATIONDETAILS,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_GetSliceConversationDetails_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_RELEASEMESSAGELIST,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_ReleaseMessageList_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_GETMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_GetMessage_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_GETATTACHMENTLIST,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_GetAttachmentList_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_GETBODYPARTLIST,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_GetBodyPartList_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_UPDATEINBOX,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_UpdateInbox_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_DELETEMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_DeleteMessage_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_PROVIDEMESSAGEHEADER,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_ProvideMessageHeader_Method )

ON_MESSAGE_SVCDATA(MOST_MSGFI_C_U16_CREATEMESSAGE,
                   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                   vHandle_CreateMessage_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_PROVIDEMESSAGEBODY,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_ProvideMessageBody_Method )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_ADDPREDEFINEDMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_AddPredefinedMessage_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_DELETEPREDEFINEDMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_DeletePredefinedMessage_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_SOURCEACTIVITY,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_SourceActivity_MethodStart)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_ALLOCATE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_Allocate_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_DEALLOCATE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_DeAllocate_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_READMESSAGEALOUD,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_ReadMessageAloud_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_READMESSAGEALOUDEXTENDED,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_ReadMessageAloudExtended_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_GETIMAGEATTACHMENT,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_GetImageAttachment_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_GETIMAGEATTACHMENTINFO,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_GetImageAttachmentInfo_MethodStart )

ON_MESSAGE_SVCDATA(MOST_MSGFI_C_U16_AUDIOATTACHMENTPLAYBACK,
                   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                   vHandle_MessagingAudioAttachmentPlayback_MethodStart)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_GETCONVERSATIONMESSAGES,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_GetConversationMessages_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_RECORDAUDIOMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_MessagingRecordAudio_MethodStart)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_PLAYBACKAUDIOMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_MessagingPlaybackRecordedAudio_MethodStart)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_RECOVERMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_RecoverMessage_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_FOLDERLISTCHANGE,
                    AMT_C_U8_CCAMSG_OPCODE_RELUPREG,
                    vHandle_FolderListChange_RelUpReg )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_GETMSGBODYACTIONELEMENT,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_GetMsgBodyActionElement_MethodStart )

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_CLEARUSERMESSAGINGINFO,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_ClearUserMessagingInfo_MethodStart)

//CMG3G-8286, CMG3G-8160, CMG3G-8287- ConnectMapProfile and MapConnectionStatus
ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_CONNECTMAPPROFILE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_ConnectMapProfile_MethodStart)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_EDITPREDEFINEDMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_EditPredefinedMessage_MethodStart)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_ADDPREDEFINEDMESSAGEEXTENDED,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_AddPredefinedMessageExtended_MethodStart)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_SWITCHMAPONOFF,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_SwitchMapOnOff_MethodStart)

ON_MESSAGE_SVCDATA( MOST_MSGFI_C_U16_ADDCUSTOMMESSAGE,
                    AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
                    vHandle_AddCustomMessage_MethodStart)
END_MSG_MAP()

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
 *
 * FUNCTION: fc_messaging_tclService_Messaging:: \
             fc_messaging_tclService_Messaging(fc_messaging_tclApp* poMainAppl)
 *
 * DESCRIPTION: Constructor.
 *
 *              Create an object of the base class ahl_tclBaseOneThreadService
 *              with a pointer to this application, the service identifier,
 *              the application identifier and the service version as parameters.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_messaging_tclService_Messaging::fc_messaging_tclService_Messaging(fc_messaging_tclApp* poMainAppl) :
    ahl_tclBaseOneThreadService(
        /* Application Pointer              */poMainAppl,
        /* ID of offered Service            */CCA_C_U16_SRV_FB_MESSAGING,
        /* MajorVersion of offered Service  */MOST_MSGFI_C_U16_SERVICE_MAJORVERSION,
        /* MinoreVersion of offered Service */MOST_MSGFI_C_U16_SERVICE_MINORVERSION,
        /* PatchVersion of offered Service  */0)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging() entered."));

   //MessagingSettings::poGetMessagingSettingsInstance();
   MessagingMutex::messagingMutexInit();
   WorkQueue::vCreateWorkQueue();
   SendMessage::vCreateSendMessageObject();
   vLoopBackDataItems.reserve(5);

   m_bServerAvailable = FALSE;
   fc_messaging_tclService_Messaging::m_poMessagingService = this;
   m_oMsgSettings  = MessagingSettings::poGetMessagingSettingsInstance();

   // Set Automatic handling of Properties
   ahl_sServiceConfig oServiceConfig;
   oServiceConfig.bAutoHandleGet = TRUE;
   oServiceConfig.bAutoHandleSet = TRUE;
   oServiceConfig.bAutoHandleIncDec = FALSE;
   oServiceConfig.bEnableHighPrioFIDs = FALSE; // CMG3G-12982- Fix for COVERITY

   m_oSourceAvailableStatus.bSourceAvailable = TRUE;

   m_u8SourceActivityErrorParameterPosition = 0x02;
   vSetConfiguration(oServiceConfig);

   m_oMessagingAudioPlaybackState_Status.e8PlaybackState.enType
   = (most_fi_tcl_e8_MsgPlaybackState::tenType) most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED;

   vResetMapDeviceCapabilities();   // Task CMG3G-7698
   vResetMapDeviceCapabilitiesExtended();
   vResetMessagingDeviceConnectionStatus();
   vResetMessageNotificationStatus();
   vResetMessageListChangeStatus();
   vResetSignatureTextStatus();
}

/*******************************************************************************
 *
 * FUNCTION: fc_messaging_tclService_Messaging:: \
                                      ~fc_messaging_tclService_Messaging(tVoid)
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_messaging_tclService_Messaging::~fc_messaging_tclService_Messaging(tVoid)
{
   ETG_TRACE_USR4(("~fc_messaging_tclService_Messaging() entered."));

   ReadMessageAloud::vDestroyReadMessageAloudObject();
   SendMessage::vDestroySendMessageObject();
   WorkQueue::vDestroyWorkQueue();
   MessagingMutex::messagingMutexDestroy();

   MessagingSettings::vDeleteMessagingSettingsInstance();
   m_oMsgSettings = NULLPTR;

   // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
   m_oMsgNotificationStatus.vDestroy();
   m_oSignatureSettingStatus.vDestroy();
   m_oSignatureSettingExtendedStatus.vDestroy();
   m_oMapDeviceCapabilitiesExtendedStatus.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_tclService_Messaging::vOnServiceAvailable()
 *
 * DESCRIPTION: This function is called by the CCA framework when the service
 *              which is offered by this server has become available.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadService::vOnServiceAvailable().
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vOnServiceAvailable()
{
   m_bServerAvailable = TRUE;
   ETG_TRACE_USR4(("vOnServiceAvailable()...Messaging entered."));

   //Fix for issue 'Messaging not requesting for MAP session'
   DbusClientInterface::getInstance().getSystemBus();
   DbusClientInterface::getInstance().messagingCreateEvoProxies();

   if (m_bServerAvailable)
   {
      //Set the ARL src available
      eUpdatePropertyToClients(MOST_MSGFI_C_U16_SOURCEAVAILABLE);
      eUpdatePropertyToClients(MOST_MSGFI_C_U16_TEXTALERTENABLE);
   }
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_tclService_Messaging::vOnServiceUnavailable()
 *
 * DESCRIPTION: This function is called by the CCA framework when the service
 *              which is offered by this server has become unavailable.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadService::vOnServiceUnavailable().
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vOnServiceUnavailable()
{
   ETG_TRACE_USR4(("vOnServiceUnavailable() entered."));
}

/*******************************************************************************
 *
 * FUNCTION: tBool fc_messaging_tclService_Messaging
 *                 ::bStatusMessageFactory(tU16 u16FunctionId,
 *                                         amt_tclServiceData& roOutMsg,
 *                                         amt_tclServiceData* poInMsg)
 *
 * DESCRIPTION: This function is called by the CCA framework to request ANY
 *              property which is offered by this service. For each property
 *              accessed via parameter 'u16FunctionId' the user has to prepare
 *              the corresponding FI data object which is then copied to the
 *              referenced parameter 'roOutMsg'.
 *
 * PARAMETER: [IN] u16FunctionId = Function ID of the requested property.
 *            [OUT] roOutMsg = Reference to the service data object to which the
 *                             content of the prepared FI data object should be
 *                             copied to.
 *            [IN] poInMsg = Selector message which is used to select dedicated
 *                           content to be copied to 'roOutMsg' instead of
 *                           updating the entire FI data object.
 *
 * RETURNVALUE: TRUE = Status message for property successfully generated.
 *              FALSE = Failed to generate status message for property.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadService::bStatusMessageFactory().
 *******************************************************************************/
 

tBool fc_messaging_tclService_Messaging::bStatusMessageFactory(tU16 u16FunctionId,
      amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::bStatusMessageFactory entered"));
   ETG_TRACE_USR4(("bStatusMessageFactory() FID = 0x%4x.", u16FunctionId));

   tBool bSuccess = FALSE;

   switch (u16FunctionId)
   {
      case MOST_MSGFI_C_U16_UPDATEINBOXSETTINGS:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_UPDATEINBOXSETTINGS"));
         most_msgfi_tclMsgUpdateInboxSettingsStatus oUpdateInboxSettings_Status;
         MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
         if (l_poMessagingSettings)
         {
            oUpdateInboxSettings_Status.u32UpdateInboxPeriod = l_poMessagingSettings->u32GetUpdateInboxTime();

            if (oUpdateInboxSettings_Status.u32UpdateInboxPeriod > 0)
            {
               oUpdateInboxSettings_Status.bAutoUpdateInboxOnOff = true;
            }
            else
            {
               oUpdateInboxSettings_Status.bAutoUpdateInboxOnOff = false;
            }
         }
         bSuccess = fi_tclVisitorMessage(
               oUpdateInboxSettings_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oUpdateInboxSettings_Status.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_AUDIOATTACHMENTPLAYBACKSTATE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_AUDIOATTACHMENTPLAYBACKSTATE"));
         ETG_TRACE_USR1((" Playback State:: %u",
               ETG_ENUM(TR_MSG_PLAYSTATUS, m_oAudioAttachmentPlaybackState_Status.e8PlaybackState.enType)));
         bSuccess = fi_tclVisitorMessage(
               m_oAudioAttachmentPlaybackState_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;

      case MOST_MSGFI_C_U16_SIGNATURESETTING:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_SIGNATURESETTING"));

         bSuccess= fi_tclVisitorMessage(
               m_oSignatureSettingStatus, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;

      case MOST_MSGFI_C_U16_SIGNATURESETTINGEXTENDED:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_SIGNATURESETTINGEXTENDED"));

         tMap_DevHandle_SettingState mapPairedDevicesSettingList;
         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->szQueryGetDeviceSpecificSettingList(SIGNATURE_ON_OFF_SETTING, mapPairedDevicesSettingList);

         tMap_DevHandle_SettingState::const_iterator MapIterator;
         for (MapIterator = mapPairedDevicesSettingList.begin(); MapIterator != mapPairedDevicesSettingList.end(); ++MapIterator)
         {
            tBool isDeviceAvailable = false;
            for (tU8 i = 0; i < m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.size(); i++)
            {
               if (MapIterator->first == m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems[i].u8DeviceHandle)
               {
                  //Update the device specific signature text, if signature is available already for the device.
                  m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems[i].bSignatureOnOff = MapIterator->second;
                  isDeviceAvailable = true;
                  break;
               }
            }
            if(!isDeviceAvailable)
            {
               //Device specific signature text is not available, so add the new signature text for the device
               most_fi_tcl_SignatureTextStreamItem oItem;
               oItem.u8DeviceHandle = MapIterator->first;
               oItem.bSignatureOnOff = MapIterator->second;
               oItem.sSignatureText.bSet(""); //set as empty signature text initially, later client can set this param if required

               m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.push_back(oItem);
            }
         }

         bSuccess= fi_tclVisitorMessage(
               m_oSignatureSettingExtendedStatus, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;

      case MOST_MSGFI_C_U16_AUDIOMESSAGERECORDINGSTATE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_AUDIOMESSAGERECORDINGSTATE"));
         ETG_TRACE_USR1((" RecordMessageState:: %u", ETG_ENUM(TR_MSG_RECORD, m_oAudioMessageRecordingState_Status.e8RecordMessageState.enType)));
         bSuccess= fi_tclVisitorMessage(
               m_oAudioMessageRecordingState_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);

      }
      break;

      case MOST_MSGFI_C_U16_MESSAGINGAUDIOPLAYBACKSTATE:
      {
      #ifdef READ_MESSAGE_ALOUD_SUPPORT
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MESSAGINGAUDIOPLAYBACKSTATE"));
         ETG_TRACE_USR1((" Playback State:: %u", ETG_ENUM(TR_MSG_PLAYSTATUS, m_oMessagingAudioPlaybackState_Status.e8PlaybackState.enType)));
         bSuccess = fi_tclVisitorMessage(
               m_oMessagingAudioPlaybackState_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      #endif
      }
      break;

      case MOST_MSGFI_C_U16_PREDEFINEDMESSAGELIST:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_PREDEFINEDMESSAGELIST"));
         bSuccess = fi_tclVisitorMessage(
               m_oPredefinedMessageList_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      // CMG3G-8602 - IS2543_PredefinedMessageListExtended@FC_Messaging
      case MOST_MSGFI_C_U16_PREDEFINEDMESSAGELISTEXTENDED:
       {
          ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_PREDEFINEDMESSAGELISTEXTENDED"));
          bSuccess = fi_tclVisitorMessage(
               m_oPredefinedMessageListExtended_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
       }
      break;
      case MOST_MSGFI_C_U16_TEXTALERTENABLE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_TEXTALERTENABLE"));

         most_msgfi_tclMsgTextAlertEnableStatus oTextAlertEnable_Status;
         MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();

         if (l_poMessagingSettings)
         {
            oTextAlertEnable_Status.bTextAlertOnOff = l_poMessagingSettings->bGetTextAlertSetting();
         }
         bSuccess = fi_tclVisitorMessage(
               oTextAlertEnable_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oTextAlertEnable_Status.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_EMAILALERTENABLE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_EMAILALERTENABLE"));

         most_msgfi_tclMsgEmailAlertEnableStatus oEmailAlertEnable_Status;

         if (m_oMsgSettings)
         {
            oEmailAlertEnable_Status.bEmailAlertOnOff = m_oMsgSettings->bGetEmailAlertSetting();
         }
         bSuccess = fi_tclVisitorMessage(
               oEmailAlertEnable_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oEmailAlertEnable_Status.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_SAVESENTTEXT:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_SAVESENTTEXT"));

         most_msgfi_tclMsgSaveSentTextStatus oSaveSentText_Status;
         MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();

         if (l_poMessagingSettings)
         {
            oSaveSentText_Status.bSaveSentTextOnOff = l_poMessagingSettings->bGetSaveSentTextSetting();
         }
         bSuccess = fi_tclVisitorMessage(
                     oSaveSentText_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oSaveSentText_Status.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_SAVESENTEMAIL:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_SAVESENTEMAIL"));

         most_msgfi_tclMsgSaveSentEmailStatus oSaveSentEmail_Status;

         if (m_oMsgSettings)
         {
            oSaveSentEmail_Status.bSaveSentEmailOnOff = m_oMsgSettings->bGetSaveSentEmailSetting();
         }
         bSuccess = fi_tclVisitorMessage(
               oSaveSentEmail_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oSaveSentEmail_Status.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_MESSAGELISTCHANGE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MESSAGELISTCHANGE"));
         ETG_TRACE_USR1((" ListChangeType:: %u", ETG_ENUM(TR_LISTCHANGETYPE, m_oMessageListChange_Status.e8ListChangeType.enType)));
         bSuccess = fi_tclVisitorMessage(m_oMessageListChange_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;

      case MOST_MSGFI_C_U16_MESSAGINGDEVICECONNECTION:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MESSAGINGDEVICECONNECTION"));
         bSuccess = fi_tclVisitorMessage(
               m_oMessagingDeviceConnection_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;

      case MOST_MSGFI_C_U16_RECORDEDAUDIOMESSAGEINFO:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_RECORDEDAUDIOMESSAGEINFO"));
         bSuccess = fi_tclVisitorMessage(
               m_oRecordedAudioMessageInfo_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;

      case MOST_MSGFI_C_U16_FOLDERLISTCHANGE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_FOLDERLISTCHANGE"));
         bSuccess = fi_tclVisitorMessage(
               m_oFolderListChange_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_MSGFI_C_U16_SOURCEAVAILABLE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_SOURCEAVAILABLE"));
         bSuccess = fi_tclVisitorMessage(
               m_oSourceAvailableStatus, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_MSGFI_C_U16_SOURCEINFO:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_SOURCEINFO"));

         fi_tclVisitorMessage oVisitorMsg(poInMsg);
         most_msgfi_tclMsgSourceInfoGet oSourceInfo_Get;

         if (oVisitorMsg.s32GetData(oSourceInfo_Get, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
         {
            ETG_TRACE_USR1(("CALLED FOR SOURCE NR = %d", oSourceInfo_Get.u8SourceNr));
            if (!isSourceNrValid(oSourceInfo_Get.u8SourceNr))
            {
               ETG_TRACE_USR1(("wrong sourceNr::%d", oSourceInfo_Get.u8SourceNr));
               vHandle_SourceInfo_Error(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE, poInMsg);
            }
            else
            {
               most_msgfi_tclMsgSourceInfoStatus oSourceInfo;

               oSourceInfo.u8SourceNr = oSourceInfo_Get.u8SourceNr;
               oSourceInfo.e8TransmissionClass.enType = most_fi_tcl_e8_SrcTransmissionClass::FI_EN_E8TRANSMISSION_CLASS_SYNCHRONOUS;
               oSourceInfo.e8ContentProtection.enType = most_fi_tcl_e8_SrcContentProtection::FI_EN_E8CONTENT_PROTECTION_NONE;
               oSourceInfo.e8ContentType.enType = most_fi_tcl_e8_SrcContentType::FI_EN_E8CONTENT_TYPE_AUDIO;

               ETG_TRACE_USR1(("SOURCE INFO SENDIGN SOURCE NR =%d", oSourceInfo_Get.u8SourceNr));
               ETG_TRACE_USR1(("SOURCE INFO SENDIGN u16BlockWidth =%d", oSourceInfo.u16BlockWidth));
               ETG_TRACE_USR1(("SOURCE INFO SENDIGN u16ConnectionLabel =%d", oSourceInfo.u16ConnectionLabel));
               ETG_TRACE_USR1(("SOURCE INFO SENDIGN e8TransmissionClass =%d", oSourceInfo.e8TransmissionClass.enType));
               ETG_TRACE_USR1(("SOURCE INFO SENDIGN e8ContentProtection =%d", oSourceInfo.e8ContentProtection.enType));
               ETG_TRACE_USR1(("SOURCE INFO SENDIGN e8ContentType =%d", oSourceInfo.e8ContentType.enType));

               if ((oSourceInfo.u8SourceNr == 1) || (oSourceInfo.u8SourceNr == 2) || (oSourceInfo.u8SourceNr == 5) || (oSourceInfo.u8SourceNr == 3))
               {
                  oSourceInfo.oContentDescription.oAudioChannels_Resolution().u8AudioChannels = 1;
               }
               else if (oSourceInfo.u8SourceNr == 4)
               {
                  oSourceInfo.oContentDescription.oAudioChannels_Resolution().u8AudioChannels = 2;
               }

               oSourceInfo.oContentDescription.oAudioChannels_Resolution().u8Resolution = 2;
               oSourceInfo.oTransmissionParameter.vGetData(NULLPTR, oSourceInfo.oTransmissionParameter.u32GetDataSize());
               bSuccess = fi_tclVisitorMessage(oSourceInfo, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
               oSourceInfo.vDestroy();
            }
         }
         else
         {
            ETG_TRACE_ERR(("parameter not available"));
            vHandle_SourceInfo_Error(most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH, poInMsg);
         }
         oSourceInfo_Get.vDestroy();
      }
      break;

      // Task CMG3G-7698: IS2201_API for device capabilities@FC_Messaging
      case MOST_MSGFI_C_U16_MAPDEVICECAPABILITIES:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MAPDEVICECAPABILITIES"));

         bSuccess = fi_tclVisitorMessage(m_oMapDeviceCapabilitiesStatus,
               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;

      case MOST_MSGFI_C_U16_MAPDEVICECAPABILITIESEXTENDED:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MAPDEVICECAPABILITIESEXTENDED"));

         bSuccess = fi_tclVisitorMessage(m_oMapDeviceCapabilitiesExtendedStatus,
               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);

         for (tU8 i = 0; i < m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems.size(); i++)
         {
            if (most_fi_tcl_e8_MapConnectionStatus::FI_EN_E8MAP_DISCONNECTED ==
                  m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems[i].e8MapConnectionState.enType)
            {
               m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems.erase(
                     m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems.begin() + i);
            }
         }
      }
      break;

      // Task CMG3G-7697 - IS2200_Switching MAP ON/OFF@FC_Messaging
      case MOST_MSGFI_C_U16_MESSAGINGONOFF:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MESSAGINGONOFF"));

         most_msgfi_tclMsgMessagingOnOffStatus oMessagingOnOff_Status;
         oMessagingOnOff_Status.MapOnOff = FALSE;

         tMap_DevHandle_SettingState mapPairedDevicesSettingList;
         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->szQueryGetDeviceSpecificSettingList(MAP_ON_OFF_SETTING, mapPairedDevicesSettingList);

         tMap_DevHandle_SettingState::const_iterator MapIterator;
         for (MapIterator = mapPairedDevicesSettingList.begin(); MapIterator != mapPairedDevicesSettingList.end(); ++MapIterator)
         {
            if(TRUE == MapIterator->second)
            {
               oMessagingOnOff_Status.MapOnOff = MapIterator->second;
               break;
            }
         }

         bSuccess = fi_tclVisitorMessage(oMessagingOnOff_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);

         oMessagingOnOff_Status.vDestroy();
      }
      break;
      // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging	  
      case MOST_MSGFI_C_U16_MESSAGENOTIFICATION:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MESSAGENOTIFICATION"));
         bSuccess = fi_tclVisitorMessage(m_oMsgNotificationStatus,
              MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_MSGFI_C_U16_MESSAGINGLANGUAGE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MESSAGINGLANGUAGE"));
         most_msgfi_tclMsgMessagingLanguageStatus oMessagingLanguage_Status;
         MessagingSettings* l_poMessagingSettings =
               MessagingSettings::poGetMessagingSettingsInstance();
         if (l_poMessagingSettings)
         {
            oMessagingLanguage_Status.e8LanguageSet.enType
                  = (most_fi_tcl_e8_MsgLanguageSet::tenType) l_poMessagingSettings->u8GetMessagingLanguage();
         }
         bSuccess
               = fi_tclVisitorMessage(oMessagingLanguage_Status, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oMessagingLanguage_Status.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_MESSAGINGONOFFLIST:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_MESSAGINGONOFFLIST"));
         most_msgfi_tclMsgMessagingOnOffListStatus oMessagingOnOffListStatus;

         tMap_DevHandle_SettingState mapPairedDevicesSettingList;
         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->szQueryGetDeviceSpecificSettingList(MAP_ON_OFF_SETTING, mapPairedDevicesSettingList);

         tMap_DevHandle_SettingState::const_iterator MapIterator;
         for (MapIterator = mapPairedDevicesSettingList.begin(); MapIterator != mapPairedDevicesSettingList.end(); ++MapIterator)
         {
            most_fi_tcl_OnOffListItem oOnOffListItem;

            oOnOffListItem.u8DeviceHandle = MapIterator->first;
            oOnOffListItem.bOnOffState = MapIterator->second;

            oMessagingOnOffListStatus.oMessagingOnOffList.oItems.push_back(oOnOffListItem);
         }

         bSuccess = fi_tclVisitorMessage(oMessagingOnOffListStatus, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oMessagingOnOffListStatus.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_TEXTRINGTONEONOFFLIST:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_TEXTRINGTONEONOFFLIST"));
         most_msgfi_tclMsgTextRingtoneOnOffListStatus oTextRingtoneOnOffListStatus;

         tMap_DevHandle_SettingState mapPairedDevicesSettingList;
         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->szQueryGetDeviceSpecificSettingList(TEXT_RINGTONE_ON_OFF_SETTING, mapPairedDevicesSettingList);

         tMap_DevHandle_SettingState::const_iterator MapIterator;
         for (MapIterator = mapPairedDevicesSettingList.begin(); MapIterator != mapPairedDevicesSettingList.end(); ++MapIterator)
         {
            most_fi_tcl_OnOffListItem oOnOffListItem;

            oOnOffListItem.u8DeviceHandle = MapIterator->first;
            oOnOffListItem.bOnOffState = MapIterator->second;

            oTextRingtoneOnOffListStatus.oTextRingtoneOnOffList.oItems.push_back(oOnOffListItem);
         }

         bSuccess = fi_tclVisitorMessage(oTextRingtoneOnOffListStatus, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oTextRingtoneOnOffListStatus.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_AUTOREPLYONOFFLIST:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_AUTOREPLYONOFFLIST"));
         most_msgfi_tclMsgAutoReplyOnOffListStatus oAutoReplyOnOffListStatus;

         tMap_DevHandle_SettingState mapPairedDevicesSettingList;
         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->szQueryGetDeviceSpecificSettingList(AUTO_REPLY_ON_OFF_SETTING, mapPairedDevicesSettingList);

         tMap_DevHandle_SettingState::const_iterator MapIterator;
         for (MapIterator = mapPairedDevicesSettingList.begin(); MapIterator != mapPairedDevicesSettingList.end(); ++MapIterator)
         {
            most_fi_tcl_OnOffListItem oOnOffListItem;

            oOnOffListItem.u8DeviceHandle = MapIterator->first;
            oOnOffListItem.bOnOffState = MapIterator->second;

            oAutoReplyOnOffListStatus.oAutoReplyOnOffList.oItems.push_back(oOnOffListItem);
         }

         bSuccess = fi_tclVisitorMessage(oAutoReplyOnOffListStatus, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oAutoReplyOnOffListStatus.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_AUTOREPLYTEXTLIST:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_MSGFI_C_U16_AUTOREPLYTEXTLIST"));
         most_msgfi_tclMsgAutoReplyTextListStatus oAutoReplyTextListStatus;

         tMap_DevHandle_MsgHandle mapPairedDevicesMsgHandleList;
         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->szQueryGetDeviceSpecificAutoReplyMsgHandleList(mapPairedDevicesMsgHandleList);

         tMap_DevHandle_MsgHandle::const_iterator MapIterator;
         for (MapIterator = mapPairedDevicesMsgHandleList.begin(); MapIterator != mapPairedDevicesMsgHandleList.end(); ++MapIterator)
         {
            most_fi_tcl_AutoReplyTextListItem oAutoReplyTextListItem;

            oAutoReplyTextListItem.u8DeviceHandle = MapIterator->first;
            oAutoReplyTextListItem.u16PredefinedMsgHandle = MapIterator->second;

            oAutoReplyTextListStatus.oAutoReplyTextList.oItems.push_back(oAutoReplyTextListItem);
         }

         bSuccess = fi_tclVisitorMessage(oAutoReplyTextListStatus, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
         oAutoReplyTextListStatus.vDestroy();
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory default"));
         bSuccess = TRUE; // To handle undefined properties.
      }
   }

   if (FALSE == bSuccess)
   {
      ETG_TRACE_ERR(("bStatusMessageFactory(). Creation of message with 'FID = 0x%4x.' failed.", u16FunctionId));
   }

   return bSuccess;
}

/*******************************************************************************
 *
 * FUNCTION: tBool fc_messaging_tclService_Messaging
 *                 ::bProcessSet(amt_tclServiceData* poMessage,
 *                               tBool& bPropertyChanged,
 *                               tU16& u16ErrorCode)
 *
 * DESCRIPTION: This function is called by the CCA framework when it has
 *              received a message for a property with Opcode 'SET' or 'PURESET'
 *              and there is no dedicated handler function defined in the
 *              message map for this pair of FID and opcode. The user has to
 *              set the application specific property to the requested value
 *              and the CCA framework then cares about informing the requesting
 *              client as well as other registered clients.
 *
 * PARAMETER: [IN] poMessage = Property to be set.
 *            [OUT] bPropertyChanged = Property changed flag to be set to TRUE
 *                                     if property has changed. Otherwise to be
 *                                     set to FALSE (default).
 *            [OUT] u16ErrorCode = Error code to be set if a CCA error occurs,
 *                                 otherwise don't touch.
 *
 * RETURNVALUE: TRUE = Send 'STATUS' message to the requesting client if opcode
 *                     was 'SET'. Send 'STATUS' message to other registered
 *                     clients as well if [OUT] parameter 'bPropertyChanged'
 *                     is set to TRUE.
 *              FALSE = Send an error message to the requesting client.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadService::bProcessSet().
 *******************************************************************************/
tBool fc_messaging_tclService_Messaging::bProcessSet(amt_tclServiceData* poMessage, tBool& bPropertyChanged, tU16& u16ErrorCode)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::bProcessSet entered"));

   tBool bSuccess = FALSE;
   tU16 u16FunctionId = poMessage->u16GetFunctionID();

   ETG_TRACE_USR4(("bProcessSet() FID = 0x%4x.", u16FunctionId));

   switch (u16FunctionId)
   {
      case MOST_MSGFI_C_U16_UPDATEINBOXSETTINGS:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_UPDATEINBOXSETTINGS"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgUpdateInboxSettingsPureSet
               oUpdateInboxSettings_PureSet;

         if (oVisitorMsg.s32GetData(oUpdateInboxSettings_PureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
         {
            fc_messaging_tclApp::m_poMainAppInstance->u32DoUpdateInboxTime =
                  oUpdateInboxSettings_PureSet.u32UpdateInboxPeriod;

            if (FALSE == bStopTimer(FC_MESSAGING_UPDATEINBOX_TIMER_ID))
            {
               ETG_TRACE_ERR(("Unable to stop the timer"));
            }

            if (oUpdateInboxSettings_PureSet.bAutoUpdateInboxOnOff)
            {
               if (FALSE == bStartTimer(FC_MESSAGING_UPDATEINBOX_TIMER_ID,
                           oUpdateInboxSettings_PureSet.u32UpdateInboxPeriod * 1000,
                           oUpdateInboxSettings_PureSet.u32UpdateInboxPeriod * 1000))
               {
                  ETG_TRACE_ERR(("Unable to start the timer"));
               }
            }

            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();

            if (l_poMessagingSettings)
            {
               tU32 l_u32UpdateInboxTimePeroid;
               l_u32UpdateInboxTimePeroid = oUpdateInboxSettings_PureSet.u32UpdateInboxPeriod;
               if (oUpdateInboxSettings_PureSet.bAutoUpdateInboxOnOff == false)
               {
                  l_u32UpdateInboxTimePeroid = 0;
               }

               tBool bResult = l_poMessagingSettings->bSetUpdateInboxTime(l_u32UpdateInboxTimePeroid,
                     oUpdateInboxSettings_PureSet.bAutoUpdateInboxOnOff);

               if (bResult)
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oUpdateInboxSettings_PureSet.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_SIGNATURESETTING:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_SIGNATURESETTING"));
         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgSignatureSettingPureSet oSignatureSettingPureSet;

         if (oVisitorMsg.s32GetData(oSignatureSettingPureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
         {
            //Set Signature Text
            m_oSignatureSettingStatus.sSignatureText.bSet(oSignatureSettingPureSet.sSignatureText.szValue);
            bPropertyChanged = TRUE;
            bSuccess = TRUE;
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oSignatureSettingPureSet.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_SIGNATURESETTINGEXTENDED:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_SIGNATURESETTINGEXTENDED"));
         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgSignatureSettingExtendedPureSet oSignatureSettingExtendedPureSet;

         if (oVisitorMsg.s32GetData(oSignatureSettingExtendedPureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
         {
            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
            if (l_poMessagingSettings)
            {
               tBool bResult = TRUE;

               for (tU8 index = 0; index < oSignatureSettingExtendedPureSet.oSignatureTextStream.oItems.size(); index++)
               {
                  tU8 u8DeviceHandle = oSignatureSettingExtendedPureSet.oSignatureTextStream.oItems[index].u8DeviceHandle;
                  tBool bSignatureOnOff = oSignatureSettingExtendedPureSet.oSignatureTextStream.oItems[index].bSignatureOnOff;

                  ETG_TRACE_USR4(("bProcessSet SignatureSettingExtendedPureSet- u8DeviceHandle: %d", u8DeviceHandle));
                  ETG_TRACE_USR4(("bProcessSet SignatureSettingExtendedPureSet- bSignatureOnOff: %d", bSignatureOnOff));
                  //ETG_TRACE_USR4(("bProcessSet SignatureSettingExtendedPureSet- sSignatureText: %s", sSignatureText));

                  bResult = bResult && l_poMessagingSettings->bSetDeviceSpecificSetting(u8DeviceHandle, SIGNATURE_ON_OFF_SETTING, bSignatureOnOff);

                  if (bResult)
                  {
                     if(bSignatureOnOff)
                     {
                        (void)DbusClientInterface::getInstance().setSignatureText(u8DeviceHandle,
                              oSignatureSettingExtendedPureSet.oSignatureTextStream.oItems[index].sSignatureText.szValue);
                     }
                     else
                     {
                        (void)DbusClientInterface::getInstance().setSignatureText(u8DeviceHandle, "");
                     }

                     tBool isDeviceSpecificTextAvailable = false;
                     for (tU8 i = 0; i < m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.size(); i++)
                     {
                        if (u8DeviceHandle == m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems[i].u8DeviceHandle)
                        {
                           //Update the device specific signature text, if signature is available already for the device.
                           m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems[i].sSignatureText.bSet(
                                 oSignatureSettingExtendedPureSet.oSignatureTextStream.oItems[index].sSignatureText.szValue);
                           isDeviceSpecificTextAvailable = true;
                           break;
                        }
                     }
                     if(!isDeviceSpecificTextAvailable)
                     {
                        //Device specific signature text is not available, so add the new signature text for the device
                        m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.push_back(
                              oSignatureSettingExtendedPureSet.oSignatureTextStream.oItems[index]);
                     }
                  }
               }

               if(bResult && (oSignatureSettingExtendedPureSet.oSignatureTextStream.oItems.size() > 0))
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oSignatureSettingExtendedPureSet.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_SAVESENTEMAIL:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_SAVESENTEMAIL"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgSaveSentEmailPureSet oSaveSentEmail_PureSet;

         if (oVisitorMsg.s32GetData(oSaveSentEmail_PureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
         {
            ETG_TRACE_USR4(("bSaveSentEmailOnOff: %u", oSaveSentEmail_PureSet.bSaveSentEmailOnOff));

            if (m_oMsgSettings)
            {
               tBool bResult =  m_oMsgSettings->bSetSaveSentEmailSetting(oSaveSentEmail_PureSet.bSaveSentEmailOnOff);

               if (bResult)
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oSaveSentEmail_PureSet.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_SAVESENTTEXT:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_SAVESENTTEXT"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgSaveSentTextPureSet oSaveSentText_PureSet;

         if (oVisitorMsg.s32GetData(oSaveSentText_PureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
         {
            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
            if (l_poMessagingSettings)
            {
               tBool bResult = l_poMessagingSettings->bSetSaveSentTextSetting(oSaveSentText_PureSet.bSaveSentTextOnOff);

               if (bResult)
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oSaveSentText_PureSet.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_EMAILALERTENABLE:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_EMAILALERTENABLE"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgEmailAlertEnablePureSet oEmailAlertEnable_PureSet;

         if (oVisitorMsg.s32GetData(oEmailAlertEnable_PureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
         {
            ETG_TRACE_USR1(("bEmailAlertOnOff: %u", oEmailAlertEnable_PureSet.bEmailAlertOnOff));

            if (m_oMsgSettings)
            {
               tBool bResult = m_oMsgSettings->bSetEmailAlertSetting(oEmailAlertEnable_PureSet.bEmailAlertOnOff);

               if (bResult)
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oEmailAlertEnable_PureSet.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_TEXTALERTENABLE:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_TEXTALERTENABLE"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgTextAlertEnablePureSet oTextAlertEnable_PureSet;

         if (oVisitorMsg.s32GetData(oTextAlertEnable_PureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
         {
            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
            if (l_poMessagingSettings)
            {
               tBool bResult = l_poMessagingSettings->bSetTextAlertSetting(oTextAlertEnable_PureSet.bTextAlertOnOff);

               if (bResult)
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oTextAlertEnable_PureSet.vDestroy();
      }
      break;
      // Task CMG3G-7697 - IS2200_Switching MAP ON/OFF@FC_Messaging
      case MOST_MSGFI_C_U16_MESSAGINGONOFF:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_MESSAGINGONOFF"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgMessagingOnOffSet oMessagingOnOff_Set;

         if (oVisitorMsg.s32GetData(oMessagingOnOff_Set, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
         {
            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
            if (l_poMessagingSettings)
            {
               tU8 deviceHandle = 0;
               tBool bResult = l_poMessagingSettings->bSetDeviceSpecificSetting(deviceHandle, MAP_ON_OFF_SETTING, oMessagingOnOff_Set.MapOnOff);
               if (bResult)
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
               vSwitchMapOnOff(oMessagingOnOff_Set.MapOnOff);
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oMessagingOnOff_Set.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_MESSAGINGLANGUAGE:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_MESSAGINGLANGUAGE"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgMessagingLanguageSet oMessagingLanguage_Set;

         if (oVisitorMsg.s32GetData(oMessagingLanguage_Set, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
         {
            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
            if (l_poMessagingSettings)
            {
               tBool bResult = l_poMessagingSettings->bSetMessagingLanguage(oMessagingLanguage_Set.e8LanguageSet.enType);
               if (bResult)
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oMessagingLanguage_Set.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_TEXTRINGTONEONOFFLIST:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_TEXTRINGTONEONOFFLIST"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgTextRingtoneOnOffListPureSet oTextRingtoneOnOffListPureSet;

         if (oVisitorMsg.s32GetData(oTextRingtoneOnOffListPureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
         {
            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
            if (l_poMessagingSettings)
            {
               tBool bResult = TRUE;

               for (tU8 index = 0; index < oTextRingtoneOnOffListPureSet.oTextRingtoneOnOffList.oItems.size(); ++index)
               {
                  tU8 u8DeviceHandle = oTextRingtoneOnOffListPureSet.oTextRingtoneOnOffList.oItems[index].u8DeviceHandle;
                  tBool bOnOffState = oTextRingtoneOnOffListPureSet.oTextRingtoneOnOffList.oItems[index].bOnOffState;

                  ETG_TRACE_USR4(("bProcessSet TextRingtoneOnOffListPureSet- u8DeviceHandle: %d", u8DeviceHandle));
                  ETG_TRACE_USR4(("bProcessSet TextRingtoneOnOffListPureSet- bOnOffState: %d", bOnOffState));

                  bResult = bResult && l_poMessagingSettings->bSetDeviceSpecificSetting(u8DeviceHandle, TEXT_RINGTONE_ON_OFF_SETTING, bOnOffState);
               }

               if (bResult && (oTextRingtoneOnOffListPureSet.oTextRingtoneOnOffList.oItems.size() > 0))
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oTextRingtoneOnOffListPureSet.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_AUTOREPLYONOFFLIST:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_AUTOREPLYONOFFLIST"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgAutoReplyOnOffListPureSet oAutoReplyOnOffListPureSet;

         if (oVisitorMsg.s32GetData(oAutoReplyOnOffListPureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
         {
            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
            if (l_poMessagingSettings)
            {
               tBool bResult = TRUE;

               for (tU8 index = 0; index < oAutoReplyOnOffListPureSet.oAutoReplyOnOffList.oItems.size(); ++index)
               {
                  tU8 u8DeviceHandle = oAutoReplyOnOffListPureSet.oAutoReplyOnOffList.oItems[index].u8DeviceHandle;
                  tBool bOnOffState = oAutoReplyOnOffListPureSet.oAutoReplyOnOffList.oItems[index].bOnOffState;

                  ETG_TRACE_USR4(("bProcessSet AutoReplyOnOffListPureSet- u8DeviceHandle: %d", u8DeviceHandle));
                  ETG_TRACE_USR4(("bProcessSet AutoReplyOnOffListPureSet- bOnOffState: %d", bOnOffState));

                  bResult = bResult && l_poMessagingSettings->bSetDeviceSpecificSetting(u8DeviceHandle, AUTO_REPLY_ON_OFF_SETTING, bOnOffState);
               }

               if (bResult && (oAutoReplyOnOffListPureSet.oAutoReplyOnOffList.oItems.size() > 0))
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oAutoReplyOnOffListPureSet.vDestroy();
      }
      break;

      case MOST_MSGFI_C_U16_AUTOREPLYTEXTLIST:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_AUTOREPLYTEXTLIST"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);

         most_msgfi_tclMsgAutoReplyTextListPureSet oAutoReplyTextListPureSet;

         if (oVisitorMsg.s32GetData(oAutoReplyTextListPureSet, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
         {
            MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
            if (l_poMessagingSettings)
            {
               tBool bResult = TRUE;

               for (tU8 index = 0; index < oAutoReplyTextListPureSet.oAutoReplyTextList.oItems.size(); ++index)
               {
                  tU8 u8DeviceHandle = oAutoReplyTextListPureSet.oAutoReplyTextList.oItems[index].u8DeviceHandle;
                  tU16 u16PredefinedMsgHandle = oAutoReplyTextListPureSet.oAutoReplyTextList.oItems[index].u16PredefinedMsgHandle;

                  ETG_TRACE_USR4(("bProcessSet AutoReplyTextListPureSet- u8DeviceHandle: %d", u8DeviceHandle));
                  ETG_TRACE_USR4(("bProcessSet AutoReplyTextListPureSet- u16PredefinedMsgHandle: %d", u16PredefinedMsgHandle));

                  bResult = bResult && l_poMessagingSettings->bSetAutoReplyText(u8DeviceHandle, u16PredefinedMsgHandle);
               }

               if (bResult && (oAutoReplyTextListPureSet.oAutoReplyTextList.oItems.size() > 0))
               {
                  bPropertyChanged = TRUE;
                  bSuccess = TRUE;
               }
            }
         }
         else
         {
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oAutoReplyTextListPureSet.vDestroy();
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("bProcessSet default: Invalid u16FunctionId"));
      }
   }

   if (FALSE == bSuccess)
   {
      ETG_TRACE_USR4(("bProcessSet(). Setting of property with 'FID = 0x%4x.' failed.", u16FunctionId));
   }
   return bSuccess;
}

tVoid fc_messaging_tclService_Messaging::vOff2Normal()
{
   ETG_TRACE_USR4(("vOff2Normal() entered."));
   //CMG3G-8753 : Severe DB error("malformed database schema ") during runtime should be used to decide
   //if the database is damaged in a critical way to be in FC_Messaging.
   // Here we check if the "database re-create trigger" file 'msg.db.recreate.trigger' exists in the database
   // folder. If so, we just delete/rename the database. [It will then be re-created from scratch in
   // method FC_Messaging_SQLite::bInitSQLDatabase()]
   int iResult = 0;
   struct stat oStatBuf;

   iResult = stat(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_RECREATE_TRIGGER_FILE_NAME, &oStatBuf);
   if (iResult == 0)
   { // The "database re-create trigger" file exists.

      // Delete the current (probably corrupted) database 'messaging.db'
      if ( 0 != remove(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME) )
      {
         iResult = -1;
         ETG_TRACE_FATAL((" -MSG-QUERY-ERR-: Database RE-CREATION has FAILED: DB file CANNOT be renamed. "));
         ETG_TRACE_ERRMEM((" -MSG-QUERY-ERR-: Database RE-CREATION has FAILED: DB file CANNOT be renamed. "));
      }

      // Remove the database journal file, if it exists, also.
      if ( 0 == stat(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_JOURNAL_FILE_NAME, &oStatBuf) )
      {
         ETG_TRACE_FATAL((" -MSG-QUERY-ERR-: Info: The DB journal file exists and will be deleted. "));
         ETG_TRACE_ERRMEM((" -MSG-QUERY-ERR-: Info: The DB journal file exists and will be deleted. "));

         if ( 0 != remove(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_JOURNAL_FILE_NAME) )
         {
            iResult = -1;
            ETG_TRACE_FATAL((" -MSG-QUERY-ERR-: Database RE-CREATION is INCOMPLETE: Existing DB journal file CANNOT be deleted. "));
            ETG_TRACE_ERRMEM((" -MSG-QUERY-ERR-: Database RE-CREATION is INCOMPLETE: Existing DB journal file CANNOT be deleted. "));
         }
      }

      // As final step the "database re-create trigger" file is removed.
      if ( 0 != remove(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_RECREATE_TRIGGER_FILE_NAME) )
      {
         iResult = -1;
         ETG_TRACE_FATAL((" -MSG-QUERY-ERR-: Database RE-CREATION is INCOMPLETE: The 'recreate trigger' file CANNOT be deleted. "));
         ETG_TRACE_ERRMEM((" -MSG-QUERY-ERR-: Database RE-CREATION is INCOMPLETE: The 'recreate trigger' file CANNOT be deleted. "));
      }

      if (0 == iResult)
      {
         ETG_TRACE_FATAL((" -MSG-QUERY-ERR-: Database RE-CREATION: All affected files have been deleted/renamed with SUCCESS. "));
         ETG_TRACE_ERRMEM((" -MSG-QUERY-ERR-: Database RE-CREATION: All affected files have been deleted/renamed with SUCCESS. "));
      }
   }
   FC_Messaging_PredefinedMsgList
         * ptrMsgList =
               MessagingSettings::poGetMessagingSettingsInstance()->poGetPredefinedMessageListInstance();

   if (ptrMsgList)
   {
      // GMMY17-7826 - Unable to turn on/off text alerts
      // Initialise the DB every time when system switches from OFF -> NORMAL state
      ptrMsgList->bInitSQLDatabase();
      // CMG3G-8602 - IS2543_PredefinedMessageListExtended@FC_Messaging
      ptrMsgList->newUserPredefinedMessageAdded();
   }
   else
   {
      ETG_TRACE_ERR(("ptrMsgList instance is NULL in vOff2Normal"));
      NORMAL_M_ASSERT_ALWAYS();
   }
}

tVoid fc_messaging_tclService_Messaging::vNormal2Off()
{
   ETG_TRACE_USR4(("vNormal2Off() entered."));
   // DB is closed before releasing other resources
   FC_Messaging_PredefinedMsgList* dataBase = MessagingSettings::poGetMessagingSettingsInstance()->poGetPredefinedMessageListInstance();
   if (dataBase)
   {
      dataBase->setDataBaseState(false);
   }
   else
   {
      ETG_TRACE_ERR(("dataBase instance is NULL in vNormal2Off"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   //Clear the data when system state is going on shutdown
   vResetMessagingDeviceConnectionStatus();
   vResetMapDeviceCapabilities();
   vResetMapDeviceCapabilitiesExtended();
   vResetMessageNotificationStatus();
   vResetMessageListChangeStatus();
   vResetSignatureTextStatus();
   fc_messaging_tclClientHandler_BluetoothSetings::clearDevicesFromRetryList();

   if (FALSE == bStopTimer(FC_MESSAGING_MAPCAPABILITYRETRY_TIMER_ID))
   {
      ETG_TRACE_USR4(("Unable to stop the timer FC_MESSAGING_MAPCAPABILITYRETRY_TIMER_ID"));
   }
   if (FALSE == bStopTimer(FC_MESSAGING_MAPERRORRETRY_TIMER_ID))
   {
      ETG_TRACE_USR4(("Unable to stop the timer FC_MESSAGING_MAPERRORRETRY_TIMER_ID"));
   }
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_tclService_Messaging::\
                              vOnLoopbackService(amt_tclServiceData* poMessage)
 *
 * DESCRIPTION: This function is called from the vOnLoopback() function of this
 *              CCA application if a message from one of this applications
 *              services is received (so called self or loopback messages).
 *
 * PARAMETER: [IN] u16ServiceID = Identifier of the service from where the
 *                                message was sent.
 *            [IN] poMessage = Message object.
 *
 * RETURNVALUE: -
 *
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vOnLoopbackService(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vOnLoopbackService Entered"));

    QMutexLocker lock(&mLoopBackMessageDataVectorMutex);

    gm_tclU32Message oVisitorMsg(poMessage);

    tU16 u16FunctionId = poMessage->u16GetFunctionID();
    ETG_TRACE_USR4(("vOnLoopbackService() FID = 0x%4x.", u16FunctionId));

    tU32 u32MsgContent = oVisitorMsg.u32GetDWord();

    LoopBackMessageData oLoopBackMessageData;
    QVector<LoopBackMessageData>::Iterator it;

    it = findMessagePosition(u32MsgContent);
    if (!it)
    {
        ETG_TRACE_ERR(("Wrong Function ID oops"));
        return;
    }

    oLoopBackMessageData = *it;

    FileSystemManager* l_poFileSystemManager = NULLPTR;

    ETG_TRACE_USR1(("Service Name::%s", oLoopBackMessageData.sServiceName.toUtf8().constData()));

    if(true == DbusClientInterface::getInstance().isDbusClientAvailable(oLoopBackMessageData.u8DeviceHandle))
    {
       l_poFileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(oLoopBackMessageData.u8DeviceHandle, oLoopBackMessageData.sServiceName);
    }
    else
    {
       ETG_TRACE_USR1(("DBusClient unavailable, erase currect iterator from LoopBackDataItems list"));

       if((FC_MSG_LB_FID_JOB_FROM_WORKQUEUE != u32MsgContent) || (TRUE == fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.empty()))
       {
          vLoopBackDataItems.erase(it);
          return;
       }
    }

    switch (u16FunctionId)
    {
       case MESSAGING_C_U16_IFID:
       {
          switch (u32MsgContent)
          {
             case FC_MSG_LB_FID_GETMESSAGE:
             {
                ETG_TRACE_USR3(("FC_MSG_LB_FID_GETMESSAGE"));

                if(l_poFileSystemManager)
                {
                   l_poFileSystemManager->vGetMessageResult(
                         DbusClientInterface::getInstance().getMsgReadObject(oLoopBackMessageData.u8DeviceHandle));
                }
             }
             break;

             case FC_MSG_LB_FID_NEW_INCOMING_MESSAGE:
             {
                ETG_TRACE_USR3(("FC_MSG_LB_FID_NEW_INCOMING_MESSAGE"));

                if(l_poFileSystemManager)
                {
                   MsgListing_t* l_poMessageList;
                   l_poMessageList = (MsgListing_t*) oLoopBackMessageData.vpData;

                   // Two consecutive New Incoming Message Bug-Fix GMMY15-1259
                   for (tU16 u16Index = 0; u16Index < oLoopBackMessageData.u16Private; u16Index++)
                   {
                      if(false == l_poMessageList[u16Index].sGetMessageHandle().isEmpty())
                      {
                         l_poFileSystemManager->vNewIncomingMessageReceived(
                               oLoopBackMessageData.u8DeviceHandle, &l_poMessageList[u16Index]);
                      }
                   }

                   delete[] l_poMessageList;
                }
             }
             break;

             // Bug fix for GMMY15-73 - MMS message not displayed
             case FC_MSG_LB_FID_NEW_INC_MSG_HANDLE:
             {
                ETG_TRACE_USR3(("FC_MSG_LB_FID_NEW_INC_MSG_HANDLE"));

                if(l_poFileSystemManager)
                {
                   l_poFileSystemManager->vNewIncomingMessageReceived(oLoopBackMessageData.u8DeviceHandle,
                         DbusClientInterface::getInstance().getMsgListObject(oLoopBackMessageData.u8DeviceHandle));
                }
             }
             break;

             case FC_MSG_LB_FID_MESSAGE_DELETED_MESSAGE:
             {
                ETG_TRACE_USR3(("FC_MSG_LB_FID_MESSAGE_DELETED_MESSAGE"));

                if(l_poFileSystemManager)
                {
                   l_poFileSystemManager->vDeleteMessageMethodReturn(
                         oLoopBackMessageData.ucStatus);
                }

             }
             break;

             case FC_MSG_LB_FID_MAP_LIST_MSG_DATA:
             {
                ETG_TRACE_USR3(("FC_MSG_LB_FID_MAP_LIST_MSG_DATA"));

                if(l_poFileSystemManager)
                {
                   MsgListing_t* MsgList = NULLPTR;
                   tU16 u16Index;

                   MsgList = (MsgListing_t*) oLoopBackMessageData.vpData;

                   for (u16Index = 0; u16Index < oLoopBackMessageData.u16Private; u16Index++)
                   {
                      l_poFileSystemManager->u32AddMessageInList(oLoopBackMessageData.u8DeviceHandle, &MsgList[u16Index]);

                      // Bugfix for GMMY16-5286: [CT} Sender Name in Text Inbox view is not presented in the format
                      // that the contact list was last view. Sender name is always shown as First Name - Last Name
                      if (FALSE == MsgList[u16Index].sGetSenderAddress().isEmpty())
                      {
                         // Bugfix for GMMY16-7664: Contact name displayed as mass code in the incoming text alert
                         QString sSenderAddress = QString::fromUtf8(MsgList[u16Index].sGetSenderAddress().constData());
                         ETG_TRACE_USR4(("Lookup SenderAddress %s in PhonebookMap", sSenderAddress.toUtf8().constData()));

                         if(Phonebook_Detail::pGetPhonebookInstance(oLoopBackMessageData.u8DeviceHandle))
                         {
                            Phonebook_Detail::pGetPhonebookInstance(oLoopBackMessageData.u8DeviceHandle)->vSearchPhonebookMap(sSenderAddress,
                                  e8CreateMsgList,
                                  oLoopBackMessageData.u8DeviceHandle);
                         }
                      }
                   }

#ifdef CONVERSATION_MESSAGES_SUPPORT
                   //Here lock is not required
                   //if(MessagingMutex::messagingMutexTrylock(NULL) == 0)
                   {
                      l_poFileSystemManager->vGetMessagesForConversation();
                   }
#endif
                   delete[] MsgList;
                }
             }
             break;

             // GMMY15-9119 FC_Messaging : logiscope code complexity value >= 20
             default:
             {
                vOnLoopbackServiceHelper(u32MsgContent, &oLoopBackMessageData, l_poFileSystemManager);
             }
          }
       }
       break;

       default:
       {
          ETG_TRACE_ERR(("vOnLoopbackService(): Unknown FID = 0x%4x.", u16FunctionId));
       }
    }

    // Clear the message from the vector
    vLoopBackDataItems.erase(it);
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_tclService_Messaging::\
                                                  vProcessTimer(tU16 u16TimerId)
 *
 * DESCRIPTION: This function is called from the vOnTimer() function of this
 *              CCA application on the expiration of a previously via function
 *              bStartTimer() started timer.
 *
 * PARAMETER: [IN] u16TimerId = Identifier of the expired timer.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vProcessTimer(tU16 /*u16TimerId*/)
{
	ETG_TRACE_USR4(("vProcessTimer entered."));

    gm_tclU32Message oServiceReadyMsg(_poMainAppl->u16GetAppId(), // Source AppID
                                      _poMainAppl->u16GetAppId(), // Target AppID
                                      0, // RegisterID
                                      0, // CmdCounter
                                      _u16ServiceID, // ServiceID
                                      MESSAGING_C_U16_IFID_EXAMPLE_1,// Function ID
                                      AMT_C_U8_CCAMSG_OPCODE_STATUS // Opcode
                                      );
    oServiceReadyMsg.vSetDWord(MESSAGING_C_U32_MSGID_EXAMPLE_1);

    if (!oServiceReadyMsg.bIsValid())
    {
       NORMAL_M_ASSERT_ALWAYS();
    }
}

/*******************************************************************************
 *
 * FUNCTION: ail_tenCommunicationError fc_messaging_tclService_Messaging::
 *                 eUpdatePropertyToClients(tU16 u16FunctionId)
 *
 * DESCRIPTION:Just a wrapper functionto update clients
 *
 * PARAMETER: [IN] u16FunctionId = Property Id to be updated.
 *
 * RETURNVALUE: ail_tenCommunicationError.
 *
 *******************************************************************************/
ail_tenCommunicationError fc_messaging_tclService_Messaging::eUpdatePropertyToClients(
        tU16 u16FunctionId)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::eUpdatePropertyToClients..Entered"));
   ETG_TRACE_USR4(("eUpdatePropertyToClients() FunctionId = 0x%4x.", u16FunctionId));
   return (eUpdateClients(u16FunctionId));
}

/*******************************************************************************
 *
 * FUNCTION: ail_tenCommunicationError vStartTimer
 *
 * DESCRIPTION:Just a wrapper function to start timer of the application
 *
 * PARAMETER: [IN] nTimerID
 *            [IN] nTimeUntilFirstTimerTick
 *            [IN] nTimerTickInterval
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tBool fc_messaging_tclService_Messaging::bStartTimer(tU16 nTimerID,
                                                     OSAL_tMSecond nTimeUntilFirstTimerTick,
                                                     OSAL_tMSecond nTimerTickInterval)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::bStartTimer TimerID = '%d'", nTimerID));

   tBool bTimerStarted = _poMainAppl->bStartTimer(nTimerID, nTimeUntilFirstTimerTick, nTimerTickInterval);

   if (FALSE == bTimerStarted)
   {
      ETG_TRACE_ERR(("Failed to Start Timer"));
      //FATAL_M_ASSERT_ALWAYS(); // This option with the fatal assert would issue a direct reset and could be used to avoid
   }
   else
   {
      ETG_TRACE_USR4(("Timer started"));
   }

   return bTimerStarted;
}
/*******************************************************************************
 *
 * FUNCTION: ail_tenCommunicationError vStopTimer
 *
 * DESCRIPTION:Just a wrapper function to stop timer of the application
 *
 * PARAMETER: [IN] nTimerID

 *
 * RETURNVALUE: None.
 */

tBool fc_messaging_tclService_Messaging::bStopTimer(tU16 nTimerID)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::bStopTimer TimerID = '%d'", nTimerID));

   tBool bTimerStopped = _poMainAppl->bStopTimer(nTimerID);

   if (FALSE == bTimerStopped)
   {
      ETG_TRACE_USR4(("Timer stop failed!!"));
   }

   return bTimerStopped;
}



/*******************************************************************************
 ** FUNCTION:   vHandle_CreateMessageList_MethodStart
 *******************************************************************************
 * DESCRIPTION:
 *   This is a handler function for CreateMessageList method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_CreateMessageList_MethodStart(
        amt_tclServiceData* poMessage, struct Job* f_prJob)
{
   ETG_TRACE_USR4(("vHandle_CreateMessageList_MethodStart : Called"));
   struct Job* l_prJob = NULLPTR;
   most_msgfi_tclMsgCreateMessageListMethodStart* l_poCreateMessageListMethodStart = NULLPTR;

   if(poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);

      l_poCreateMessageListMethodStart = new most_msgfi_tclMsgCreateMessageListMethodStart;
      if(l_poCreateMessageListMethodStart)
      {
         if (oVisitorMsg.s32GetData(*l_poCreateMessageListMethodStart,
               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
         {
            vHandleGetDataError(poMessage, "vHandle_CreateMessageList_MethodStart");
            delete l_poCreateMessageListMethodStart;
            return;
         }
         if ((l_poCreateMessageListMethodStart->u8DeviceHandle>= MINIMUM_DEVICE_HANDLE) && (l_poCreateMessageListMethodStart->u8DeviceHandle<= MAXIMUM_DEVICE_HANDLE))
         {
            l_prJob = new Job;
            if(l_prJob)
            {
               l_prJob->m_pfMethod  = &fc_messaging_tclService_Messaging::vHandle_CreateMessageList_MethodStart;
               l_prJob->u8DeviceHandle = l_poCreateMessageListMethodStart->u8DeviceHandle;
               l_prJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               l_prJob->u16FunctionID = poMessage->u16GetFunctionID();
               l_prJob->u16RegisterID = poMessage->u16GetRegisterID();
               l_prJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               l_prJob->m_pvMethodStart = l_poCreateMessageListMethodStart;
            }
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_CreateMessageList_MethodStart : Invalid Device Handle"));
            vHandleInvalidDataError(poMessage);
            delete l_poCreateMessageListMethodStart;
            return;
         }
      }
   }
   else if(f_prJob)
   {
      l_prJob = f_prJob;
   }
   else
   {
      return; // Should not come here
   }
   if(l_prJob)
   {
      if (MessagingMutex::messagingMutexTrylock(l_prJob) != 0)
      {
         tVoid*  pCustodialLose = l_prJob ;  //To remove LINT warning : Custodial Pointer
         (tVoid) pCustodialLose ;
         return; // Didn't get the lock :(
      }
   }
   if(l_prJob)
   {
      m_oCreateMessageList.u16SourceAppID = l_prJob->u16SourceAppID;
      m_oCreateMessageList.u16RegisterID = l_prJob->u16RegisterID;
      m_oCreateMessageList.u16CmdCounter = l_prJob->u16CmdCounter;
      m_oCreateMessageList.u16FunctionID = l_prJob->u16FunctionID;

      if(l_poCreateMessageListMethodStart) //LINT fix
      {
         tVoid*  pCustodialLose = l_poCreateMessageListMethodStart ;  //To remove LINT warning
         (tVoid) pCustodialLose ;
      }
      l_poCreateMessageListMethodStart = (most_msgfi_tclMsgCreateMessageListMethodStart*) l_prJob->m_pvMethodStart;
   }

   if(l_poCreateMessageListMethodStart)
   {
      QString serviceName;

      ETG_TRACE_USR1(("Message List Type:: %u", ETG_ENUM(TR_MESSAGELISTTYPE,l_poCreateMessageListMethodStart->e8MessageListType.enType)));
      ETG_TRACE_USR1(("Message Folder Type:: %u", ETG_ENUM(TR_FOLDER_TYPE,l_poCreateMessageListMethodStart->e8FolderType.enType)));
      ETG_TRACE_USR1(("Message Filter Type:: %d", l_poCreateMessageListMethodStart->e8MessageListFilterType.enType));

      serviceName = DbusClientInterface::getInstance().getCurrentActiveService(l_poCreateMessageListMethodStart->u8DeviceHandle,
            l_poCreateMessageListMethodStart->e8MessageListType.enType);
      FileSystemManager* l_poFileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(l_poCreateMessageListMethodStart->u8DeviceHandle,
            serviceName);

      if (l_poFileSystemManager)
      {
         if(most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_CONVERSATION == l_poCreateMessageListMethodStart->e8FolderType.enType)
         {
            FolderNode* l_poFolderNode = l_poFileSystemManager->poSearchNodeByFolderName(MSG_DBUS_FOLDER_TYPE_CONVERSATION);
            if(l_poFolderNode && l_poFolderNode->m_pvConversationList)
            {
               if(most_fi_tcl_e8_MsgMessageListFilterType::FI_EN_E8MSG_LIST_FILTER_CURRENT_MAP_SESSION ==
                     l_poCreateMessageListMethodStart->e8MessageListFilterType.enType)
               {
                  vHandle_CreateMessageList_Return(static_cast<tU16>(l_poFolderNode->m_u32ListHandle),
                        static_cast<unsigned int>(l_poFolderNode->m_pvConversationList->size()));
               }
               else
               {
                  tU8 u8Result = l_poFileSystemManager->u8CheckAllMessageLists(l_poCreateMessageListMethodStart->u8DeviceHandle,
                        e8CreateMessageListRequestForConversationFolder);

                  if(0 == u8Result)
                  {
                     vHandle_CreateMessageList_Return(static_cast<tU16>(l_poFolderNode->m_u32ListHandle),
                           static_cast<unsigned int>(l_poFolderNode->m_pvConversationList->size()));
                  }
               }
            }
            else
            {
               ETG_TRACE_USR4(("vHandle_CreateMessageList_MethodStart : FolderNode NULL"));
               vHandle_CreateMessageList_Error();
            }
         }
         else
         {
            //TODO ListFilterType FI_EN_E8MSG_LIST_FILTER_CURRENT_MAP_SESSION needs to checked and added for INBOX and SENT folder.
            l_poFileSystemManager->vCreateMessageList((*l_poCreateMessageListMethodStart));
         }
      }
      else
      {
         ETG_TRACE_USR4(("vHandle_CreateMessageList_MethodStart : FileSystemManager NULL"));
         vHandle_CreateMessageList_Error();
      }

      l_poCreateMessageListMethodStart->vDestroy();
      delete l_poCreateMessageListMethodStart;
   }
   delete l_prJob;
}

/*******************************************************************************
 ** FUNCTION:   vHandle_CreateMessageList_Return
 *******************************************************************************/
/* DESCRIPTION:
 *  function is used to send the result of CreateMessageList to client.
 *
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_CreateMessageList_Return(
        tU16 u16ListHandle, tU32 u32NumberMessage)
{
    ETG_TRACE_USR4(("vHandle_CreateMessageList_Return : Called  ListHandle = %d", u16ListHandle));

    most_msgfi_tclMsgCreateMessageListMethodResult
            oCreateMessageListMethodResult_MR;

    oCreateMessageListMethodResult_MR.u16ListHandle = u16ListHandle;
    oCreateMessageListMethodResult_MR.u16ListLength = static_cast<tU16>(u32NumberMessage);

    fi_tclVisitorMessage oMRMsg(oCreateMessageListMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oCreateMessageList.u16SourceAppID, // ServiceDataMsg
                     m_oCreateMessageList.u16RegisterID, // RegisterID
                     m_oCreateMessageList.u16CmdCounter, // CmdCounter,
                     m_oCreateMessageList.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_USR4(("vHandle_CreateMessageList_Return: Post failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }
    else
    {
        ETG_TRACE_USR4(("vHandle_CreateMessageList_Return: Invalid message"));
    }


    MessagingMutex::messagingMutexUnlock();

    oCreateMessageListMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_CreateMessageList_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of CreateMessageList to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_CreateMessageList_Error()
{
    ETG_TRACE_USR4(("vHandle_CreateMessageList_Error : Called"));
    most_msgfi_tclMsgCreateMessageListError l_oCreateMessageListError_ER;

    l_oCreateMessageListError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(l_oCreateMessageListError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oCreateMessageList.u16SourceAppID, // ServiceDataMsg
                     m_oCreateMessageList.u16RegisterID, // RegisterID
                     m_oCreateMessageList.u16CmdCounter, // CmdCounter,
                     m_oCreateMessageList.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_CreateMessageList_Error: enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }
    else
    {
        ETG_TRACE_ERR(("vHandle_CreateMessageList_Error: Invalid Message"));
    }

    MessagingMutex::messagingMutexUnlock();

    l_oCreateMessageListError_ER.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_CreateMessage_Method
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for CreateMessage method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_CreateMessage_MethodStart(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandle_CreateMessage_MethodStart : Called"));
    // LINT WARNING tBool bReturnValue = false;

    fi_tclVisitorMessage oVisitorMsg(poMessage);

    most_msgfi_tclMsgCreateMessageMethodStart oMsgCreateMessageMethodStart_MS;

    if (oVisitorMsg.s32GetData(oMsgCreateMessageMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        SendMessage* l_poSendMessage;
        l_poSendMessage = SendMessage::poGetSendMessageInstance();

        if (l_poSendMessage)
        {
            l_poSendMessage->vSetCreateMessageData(
                        oMsgCreateMessageMethodStart_MS);
        }
    }
    else
    {
        vHandleGetDataError(poMessage, "vHandle_CreateMessage_Method");
    }

    most_msgfi_tclMsgCreateMessageMethodResult oMsgCreateMessageMethodResult_MR;

    oMsgCreateMessageMethodResult_MR.u8CreateMessageHandle
            = oMsgCreateMessageMethodStart_MS.u8CreateMessageHandle;

    fi_tclVisitorMessage oMRMsg(oMsgCreateMessageMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
                     poMessage->u16GetRegisterID(), // RegisterID
                     poMessage->u16GetCmdCounter(), // CmdCounter,
                     poMessage->u16GetFunctionID(), // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
    	ETG_TRACE_USR1((" Message Type:: %u",ETG_ENUM(TR_MESSAGELIST,oMsgCreateMessageMethodStart_MS.e8MessageType.enType)));
    	ETG_TRACE_USR1((" Create Message Type:: %u",ETG_ENUM(TR_CREATEMESSAGETYPE,oMsgCreateMessageMethodStart_MS.e8CreateMessageType.enType)));
    	// Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_CreateMessage_Method: ail ERROR', enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    oMsgCreateMessageMethodResult_MR.vDestroy();
    oMsgCreateMessageMethodStart_MS.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestSliceMessageList_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for RequestSliceMessageList method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestSliceMessageList_MethodStart(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandle_RequestSliceMessageList_MethodStart : Called"));

    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgRequestSliceMessageListMethodStart
            oRequestSliceMessageListMethodStart_MS;

    if (oVisitorMsg.s32GetData(oRequestSliceMessageListMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {

        m_oRequestSliceMessageList.u16SourceAppID
                = poMessage->u16GetSourceAppID();
        m_oRequestSliceMessageList.u16RegisterID
                = poMessage->u16GetRegisterID();
        m_oRequestSliceMessageList.u16CmdCounter
                = poMessage->u16GetCmdCounter();
        m_oRequestSliceMessageList.u16FunctionID
                = poMessage->u16GetFunctionID();

        ETG_TRACE_USR1(("List Handle is::%d",
                  oRequestSliceMessageListMethodStart_MS.u16ListHandle));
        ETG_TRACE_USR1(("Window start is::%d",
                  oRequestSliceMessageListMethodStart_MS.u32WindowStart));
        ETG_TRACE_USR1(("Window size is::%d",
                  oRequestSliceMessageListMethodStart_MS.u32WindowSize));

        FolderNode* l_poFolderNode = NULLPTR;
        QString serviceName;

        tU8 deviceHandle = static_cast<tU8>(oRequestSliceMessageListMethodStart_MS.u16ListHandle >> 8);

        serviceName = DbusClientInterface::getInstance().getCurrentActiveService(deviceHandle);
        FileSystemManager* l_poFileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(deviceHandle, serviceName);

        if (l_poFileSystemManager)
        {
           l_poFolderNode = l_poFileSystemManager->poSearchNodeByListHandle(oRequestSliceMessageListMethodStart_MS.u16ListHandle);

           if (!l_poFolderNode)
           {
              // SMS list not available for received ListHandle, check for EMAIL list
              serviceName = DbusClientInterface::getInstance().getCurrentActiveService(deviceHandle, most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_EMAIL);
              l_poFileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(deviceHandle, serviceName);

              if (l_poFileSystemManager)
              {
                 l_poFolderNode = l_poFileSystemManager->poSearchNodeByListHandle(oRequestSliceMessageListMethodStart_MS.u16ListHandle);
              }
           }
        }

        // Bugfix for GMMY15-5992 Large empty gap is present in text messages list
        if (l_poFileSystemManager && l_poFolderNode && l_poFolderNode->m_pvMessageList &&
              (oRequestSliceMessageListMethodStart_MS.u32WindowStart < static_cast<tU32>(l_poFolderNode->m_pvMessageList->size())))
        {
           vHandle_RequestSliceMessageList_Return(
                 l_poFolderNode->m_pvMessageList,
                 oRequestSliceMessageListMethodStart_MS.u32WindowStart,
                 oRequestSliceMessageListMethodStart_MS.u32WindowSize);
        }
        else
        {
           // SMS & EMAIL lists are not available, sent error response to client
           ETG_TRACE_ERR(("Invalid u16ListHandle (or) u32WindowStart has negative value"));
           vHandle_RequestSliceMessageList_Error();
        }
    }
    else
    {
        vHandleGetDataError(poMessage, "vHandle_RequestSliceMessageList_MethodStart");
    }

    oRequestSliceMessageListMethodStart_MS.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestSliceMessageList_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of RequestSlice to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestSliceMessageList_Return(
      QVector<most_fi_tcl_MsgMessageListSliceResultItem>* pMessageList,
      tU32 u32WindowStart, tU32 u32WindowSize)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vHandle_RequestSliceMessageList_Return Entered"));

   most_msgfi_tclMsgRequestSliceMessageListMethodResult oRequestSliceMessageListMethodResult_MR;

   tInt u32ListLength = pMessageList->size();
   tU32 u32Counter = 0;

   if (0 != u32ListLength)
   {
      most_fi_tcl_MsgMessageListSliceResult oMessageListSliceResultItem;

      for (tInt u32Index = (tInt)u32WindowStart;
            ((u32Counter < u32WindowSize) && (0 <= u32Index) && (u32Index < u32ListLength));
            u32Index++, u32Counter++)
      {
         ETG_TRACE_USR4(("vHandle_RequestSliceMessageList_Return u32Counter: %d", u32Counter));

         // Changes done to resolve the reset issue - GMMY-16 - 21822 caused by the MOST-Fi's changes
         // Do a deep copy to the result message from the local member.
         oMessageListSliceResultItem[u32Counter].oMessageHandle =
               pMessageList->at(u32Index).oMessageHandle;
         oMessageListSliceResultItem[u32Counter].oMessageDateTime.oMessageDate.sCldrYear.bSet(
               pMessageList->at(u32Index).oMessageDateTime.oMessageDate.sCldrYear.szValue);
         oMessageListSliceResultItem[u32Counter].oMessageDateTime.oMessageDate.sCldrMonth.bSet(
               pMessageList->at(u32Index).oMessageDateTime.oMessageDate.sCldrMonth.szValue);
         oMessageListSliceResultItem[u32Counter].oMessageDateTime.oMessageDate.sCldrDay.bSet(
               pMessageList->at(u32Index).oMessageDateTime.oMessageDate.sCldrDay.szValue);
         oMessageListSliceResultItem[u32Counter].oMessageDateTime.oMessageTime.sHours.bSet(
               pMessageList->at(u32Index).oMessageDateTime.oMessageTime.sHours.szValue);
         oMessageListSliceResultItem[u32Counter].oMessageDateTime.oMessageTime.sMinutes.bSet(
               pMessageList->at(u32Index).oMessageDateTime.oMessageTime.sMinutes.szValue);
         oMessageListSliceResultItem[u32Counter].oMessageDateTime.oMessageTime.sSeconds.bSet(
               pMessageList->at(u32Index).oMessageDateTime.oMessageTime.sSeconds.szValue);
         oMessageListSliceResultItem[u32Counter].sSubject.bSet(
               pMessageList->at(u32Index).sSubject.szValue);
         oMessageListSliceResultItem[u32Counter].sShortMessage.bSet(
               pMessageList->at(u32Index).sShortMessage.szValue);
         oMessageListSliceResultItem[u32Counter].bAttachmentFlag =
               pMessageList->at(u32Index).bAttachmentFlag;
         oMessageListSliceResultItem[u32Counter].e8MessageType.enType =
               pMessageList->at(u32Index).e8MessageType.enType;
         oMessageListSliceResultItem[u32Counter].e8SendMessageResultType.enType =
               pMessageList->at(u32Index).e8SendMessageResultType.enType;
         oMessageListSliceResultItem[u32Counter].bReadStatusFlag =
               pMessageList->at(u32Index).bReadStatusFlag;
         oMessageListSliceResultItem[u32Counter].sFirstName.bSet(
               pMessageList->at(u32Index).sFirstName.szValue);
         oMessageListSliceResultItem[u32Counter].sLastName.bSet(
               pMessageList->at(u32Index).sLastName.szValue);
         oMessageListSliceResultItem[u32Counter].sEmailAddress.bSet(
               pMessageList->at(u32Index).sEmailAddress.szValue);
         oMessageListSliceResultItem[u32Counter].sPhoneNumber.bSet(
               pMessageList->at(u32Index).sPhoneNumber.szValue);
         oMessageListSliceResultItem[u32Counter].u32ContactHandle =
               pMessageList->at(u32Index).u32ContactHandle;
         oMessageListSliceResultItem[u32Counter].e8PlaybackState.enType =
               pMessageList->at(u32Index).e8PlaybackState.enType;
      }

      oRequestSliceMessageListMethodResult_MR.oMessageListSliceResult = oMessageListSliceResultItem;
   }

   fi_tclVisitorMessage oMRMsg(oRequestSliceMessageListMethodResult_MR, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg, m_oRequestSliceMessageList.u16SourceAppID,
         m_oRequestSliceMessageList.u16RegisterID, m_oRequestSliceMessageList.u16CmdCounter,
         m_oRequestSliceMessageList.u16FunctionID, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_RequestSliceMessageList_Return: PostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oRequestSliceMessageListMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestSliceMessageList_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of RequestSlice to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestSliceMessageList_Error()
{
    ETG_TRACE_USR4(("vHandle_RequestSliceMessageList_Error : Called"));
    most_msgfi_tclMsgRequestSliceMessageListError
            l_oRequestSliceMessageListError_ER;

    l_oRequestSliceMessageListError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(l_oRequestSliceMessageListError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oRequestSliceMessageList.u16SourceAppID, // ServiceDataMsg
                     m_oRequestSliceMessageList.u16RegisterID, // RegisterID
                     m_oRequestSliceMessageList.u16CmdCounter, // CmdCounter,
                     m_oRequestSliceMessageList.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_RequestSliceMessageList_Error: PostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    l_oRequestSliceMessageListError_ER.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vHandle_RequestSliceConversationList_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_RequestSliceConversationList_MethodStart() called"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgRequestSliceConversationListMethodStart  oRequestSliceConversationListMethodStart;

   if (oVisitorMsg.s32GetData(oRequestSliceConversationListMethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      m_oRequestSliceConversationList.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oRequestSliceConversationList.u16RegisterID  = poMessage->u16GetRegisterID();
      m_oRequestSliceConversationList.u16CmdCounter  = poMessage->u16GetCmdCounter();
      m_oRequestSliceConversationList.u16FunctionID  = poMessage->u16GetFunctionID();

      ETG_TRACE_USR1(("List Handle is::%d", oRequestSliceConversationListMethodStart.u16ListHandle));
      ETG_TRACE_USR1(("Window Start is::%d", oRequestSliceConversationListMethodStart.u32WindowStart));
      ETG_TRACE_USR1(("Window Size is::%d", oRequestSliceConversationListMethodStart.u32WindowSize));

      FolderNode* folderNode = NULLPTR;
      QString serviceName;

      tU8 deviceHandle = static_cast<tU8>(oRequestSliceConversationListMethodStart.u16ListHandle >> 8);

      serviceName = DbusClientInterface::getInstance().getCurrentActiveService(deviceHandle);
      FileSystemManager* fileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(deviceHandle, serviceName);

      if (fileSystemManager)
      {
         folderNode = fileSystemManager->poSearchNodeByListHandle(oRequestSliceConversationListMethodStart.u16ListHandle);
      }

      if (fileSystemManager && folderNode && folderNode->m_pvConversationList &&
            (oRequestSliceConversationListMethodStart.u32WindowStart < static_cast<tU32>(folderNode->m_pvConversationList->size())))
      {
         fileSystemManager->bSortConversationList(folderNode->m_pvConversationList);

         vHandle_RequestSliceConversationList_Return(
               folderNode->m_pvConversationList,
               oRequestSliceConversationListMethodStart.u32WindowStart,
               oRequestSliceConversationListMethodStart.u32WindowSize);
      }
      else
      {
         // SMS & EMAIL lists are not available, sent error response to client
         ETG_TRACE_ERR(("Invalid u16ListHandle (or) u32WindowStart has negative value"));
         vHandle_RequestSliceConversationList_Error();
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_RequestSliceConversationList_MethodStart");
   }

   oRequestSliceConversationListMethodStart.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vHandle_RequestSliceConversationList_Return(
      QVector<most_fi_tcl_MsgConversationListSliceResultItem>* pConversationList,
      tU32 u32WindowStart, tU32 u32WindowSize)
{
   ETG_TRACE_USR4(("vHandle_RequestSliceConversationList_Return() called"));

   most_msgfi_tclMsgRequestSliceConversationListMethodResult oRequestSliceConversationListMethodResult;

   tInt u32ListLength = pConversationList->size();
   tU32 u32Counter = 0;

   if (0 != u32ListLength)
   {
      most_fi_tcl_MsgConversationListSliceResult oConversationListSliceResultItem;

      for (tInt u32Index = (tInt)u32WindowStart;
            ((u32Counter < u32WindowSize) && (0 <= u32Index) && (u32Index < u32ListLength));
            u32Index++, u32Counter++)
      {
         ETG_TRACE_USR4(("vHandle_RequestSliceConversationList_Return u32Counter: %d", u32Counter));

         // Do a deep copy to the result message from the local member.
         oConversationListSliceResultItem[u32Counter].oConversationHandle =
               pConversationList->at(u32Index).oConversationHandle;

         ETG_TRACE_USR4(("ConversationHandle : %d %u", pConversationList->at(u32Index).oConversationHandle.u8DeviceHandle,
               pConversationList->at(u32Index).oConversationHandle.u32ConversationID));

         oConversationListSliceResultItem[u32Counter].oLastActivity.oMessageDate.sCldrYear.bSet(
               pConversationList->at(u32Index).oLastActivity.oMessageDate.sCldrYear.szValue);
         oConversationListSliceResultItem[u32Counter].oLastActivity.oMessageDate.sCldrMonth.bSet(
               pConversationList->at(u32Index).oLastActivity.oMessageDate.sCldrMonth.szValue);
         oConversationListSliceResultItem[u32Counter].oLastActivity.oMessageDate.sCldrDay.bSet(
               pConversationList->at(u32Index).oLastActivity.oMessageDate.sCldrDay.szValue);
         oConversationListSliceResultItem[u32Counter].oLastActivity.oMessageTime.sHours.bSet(
               pConversationList->at(u32Index).oLastActivity.oMessageTime.sHours.szValue);
         oConversationListSliceResultItem[u32Counter].oLastActivity.oMessageTime.sMinutes.bSet(
               pConversationList->at(u32Index).oLastActivity.oMessageTime.sMinutes.szValue);
         oConversationListSliceResultItem[u32Counter].oLastActivity.oMessageTime.sSeconds.bSet(
               pConversationList->at(u32Index).oLastActivity.oMessageTime.sSeconds.szValue);

         ETG_TRACE_USR4(("LastActivity Time: %2s:%2s:%2s", pConversationList->at(u32Index).oLastActivity.oMessageTime.sHours.szValue,
               pConversationList->at(u32Index).oLastActivity.oMessageTime.sMinutes.szValue,
               pConversationList->at(u32Index).oLastActivity.oMessageTime.sSeconds.szValue));

         ETG_TRACE_USR4(("LastActivity Date: %2s/%2s/%4s", pConversationList->at(u32Index).oLastActivity.oMessageDate.sCldrDay.szValue,
               pConversationList->at(u32Index).oLastActivity.oMessageDate.sCldrMonth.szValue,
               pConversationList->at(u32Index).oLastActivity.oMessageDate.sCldrYear.szValue));

         oConversationListSliceResultItem[u32Counter].sSubject.bSet(
               pConversationList->at(u32Index).sSubject.szValue);

         ETG_TRACE_USR4(("Subject : %s", pConversationList->at(u32Index).sSubject.szValue));

         oConversationListSliceResultItem[u32Counter].u32AttachmentSize =
               pConversationList->at(u32Index).u32AttachmentSize;

         ETG_TRACE_USR4(("AttachmentSize : %u", pConversationList->at(u32Index).u32AttachmentSize));

         oConversationListSliceResultItem[u32Counter].e8MessageType.enType =
               pConversationList->at(u32Index).e8MessageType.enType;

         ETG_TRACE_USR4(("MessageType : %u", ETG_ENUM(TR_MESSAGELIST,pConversationList->at(u32Index).e8MessageType.enType)));

         oConversationListSliceResultItem[u32Counter].u32NumberOfMessages =
               pConversationList->at(u32Index).u32NumberOfMessages;

         ETG_TRACE_USR4(("NumberOfMessages : %u", pConversationList->at(u32Index).u32NumberOfMessages));

         oConversationListSliceResultItem[u32Counter].u8UnreadMessageCount =
               pConversationList->at(u32Index).u8UnreadMessageCount;

         ETG_TRACE_USR4(("UnreadMessageCount : %d", pConversationList->at(u32Index).u8UnreadMessageCount));

         oConversationListSliceResultItem[u32Counter].oAddressField.e8AddressFieldType.enType =
                        pConversationList->at(u32Index).oAddressField.e8AddressFieldType.enType;

         oConversationListSliceResultItem[u32Counter].oAddressField.sFirstName.bSet(
               pConversationList->at(u32Index).oAddressField.sFirstName.szValue);

         ETG_TRACE_USR4(("FirstName : %s", pConversationList->at(u32Index).oAddressField.sFirstName.szValue));

         oConversationListSliceResultItem[u32Counter].oAddressField.sLastName.bSet(
               pConversationList->at(u32Index).oAddressField.sLastName.szValue);

         ETG_TRACE_USR4(("LastName : %s", pConversationList->at(u32Index).oAddressField.sLastName.szValue));

         oConversationListSliceResultItem[u32Counter].oAddressField.sEmailAddress.bSet(
               pConversationList->at(u32Index).oAddressField.sEmailAddress.szValue);

         ETG_TRACE_USR4(("EmailAddress : %s", pConversationList->at(u32Index).oAddressField.sEmailAddress.szValue));

         oConversationListSliceResultItem[u32Counter].oAddressField.sPhoneNumber.bSet(
               pConversationList->at(u32Index).oAddressField.sPhoneNumber.szValue);

         ETG_TRACE_USR4(("PhoneNumber : %s", pConversationList->at(u32Index).oAddressField.sPhoneNumber.szValue));

         oConversationListSliceResultItem[u32Counter].u32ContactHandle =
               pConversationList->at(u32Index).u32ContactHandle;

         ETG_TRACE_USR4(("ContactHandle : %u", pConversationList->at(u32Index).u32ContactHandle));

      }

      oRequestSliceConversationListMethodResult.oConversationListSliceResult = oConversationListSliceResultItem;
   }

   fi_tclVisitorMessage oMRMsg(oRequestSliceConversationListMethodResult, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg, m_oRequestSliceConversationList.u16SourceAppID,
         m_oRequestSliceConversationList.u16RegisterID, m_oRequestSliceConversationList.u16CmdCounter,
         m_oRequestSliceConversationList.u16FunctionID, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_RequestSliceConversationList_Return: PostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oRequestSliceConversationListMethodResult.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestSliceConversationList_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of RequestSliceConversation to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestSliceConversationList_Error()
{
    ETG_TRACE_USR4(("vHandle_RequestSliceConversationList_Error() called"));

    most_msgfi_tclMsgRequestSliceConversationListError oRequestSliceConversationListError;

    oRequestSliceConversationListError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(oRequestSliceConversationListError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oRequestSliceConversationList.u16SourceAppID, // ServiceDataMsg
                             m_oRequestSliceConversationList.u16RegisterID,  // RegisterID
                             m_oRequestSliceConversationList.u16CmdCounter,  // CmdCounter,
                             m_oRequestSliceConversationList.u16FunctionID,  // Function ID
                             AMT_C_U8_CCAMSG_OPCODE_ERROR                    // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(("vHandle_RequestSliceConversationList_Error: PostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    oRequestSliceConversationListError.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vHandle_GetSliceConversationDetails_MethodStart(
      amt_tclServiceData* poMessage, struct Job* f_prJob)
{
   ETG_TRACE_USR4(("vHandle_GetSliceConversationDetails_MethodStart() called"));

   struct Job* l_prJob = NULLPTR;
   most_msgfi_tclMsgGetSliceConversationDetailsMethodStart* oGetSliceConversationDetailsMethodStart = NULLPTR;

   if(poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);

      oGetSliceConversationDetailsMethodStart = new most_msgfi_tclMsgGetSliceConversationDetailsMethodStart;
      if(oGetSliceConversationDetailsMethodStart)
      {
         if (oVisitorMsg.s32GetData(*oGetSliceConversationDetailsMethodStart,
               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
         {
            vHandleGetDataError(poMessage, "vHandle_GetSliceConversationDetails_MethodStart");
            delete oGetSliceConversationDetailsMethodStart;
            return;
         }

         l_prJob = new Job;
         if(l_prJob)
         {
            l_prJob->m_pfMethod  = &fc_messaging_tclService_Messaging::vHandle_GetSliceConversationDetails_MethodStart;
            l_prJob->u8DeviceHandle = oGetSliceConversationDetailsMethodStart->oConversationHandle.u8DeviceHandle;
            l_prJob->u16CmdCounter = poMessage->u16GetCmdCounter();
            l_prJob->u16FunctionID = poMessage->u16GetFunctionID();
            l_prJob->u16RegisterID = poMessage->u16GetRegisterID();
            l_prJob->u16SourceAppID = poMessage->u16GetSourceAppID();
            l_prJob->m_pvMethodStart = oGetSliceConversationDetailsMethodStart;
         }
      }
   }
   else if(f_prJob)
   {
      l_prJob = f_prJob;
   }
   else
   {
      return; // Should not come here
   }
   if(l_prJob)
   {
      if (MessagingMutex::messagingMutexTrylock(l_prJob) != 0)
      {
         tVoid*  pCustodialLose = l_prJob ;  //To remove LINT warning : Custodial Pointer
         (tVoid) pCustodialLose ;
         return; // Didn't get the lock :(
      }

      m_oGetSliceConversationDetails.u16SourceAppID = l_prJob->u16SourceAppID;
      m_oGetSliceConversationDetails.u16RegisterID = l_prJob->u16RegisterID;
      m_oGetSliceConversationDetails.u16CmdCounter = l_prJob->u16CmdCounter;
      m_oGetSliceConversationDetails.u16FunctionID = l_prJob->u16FunctionID;

      if(oGetSliceConversationDetailsMethodStart) //LINT fix
      {
         tVoid*  pCustodialLose = oGetSliceConversationDetailsMethodStart ;  //To remove LINT warning
         (tVoid) pCustodialLose ;
      }
      oGetSliceConversationDetailsMethodStart = (most_msgfi_tclMsgGetSliceConversationDetailsMethodStart*) l_prJob->m_pvMethodStart;
   }

   if(oGetSliceConversationDetailsMethodStart)
   {
      tU8 deviceHandle = oGetSliceConversationDetailsMethodStart->oConversationHandle.u8DeviceHandle;
      tU32 u32ConversationID = oGetSliceConversationDetailsMethodStart->oConversationHandle.u32ConversationID;

      ETG_TRACE_USR1(("DeviceHandle : %d", deviceHandle));
      ETG_TRACE_USR1(("ConversationID : %u", static_cast<unsigned int> (u32ConversationID)));
      ETG_TRACE_USR1(("WindowStart : %d", oGetSliceConversationDetailsMethodStart->u32WindowStart));
      ETG_TRACE_USR1(("WindowSize : %d", oGetSliceConversationDetailsMethodStart->u32WindowSize));

      FolderNode* folderNode = NULLPTR;
      QString serviceName;
      tBool bResult = false;

      serviceName = DbusClientInterface::getInstance().getCurrentActiveService(deviceHandle);
      FileSystemManager* fileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(deviceHandle, serviceName);

      if (fileSystemManager)
      {
         folderNode = fileSystemManager->poSearchNodeByFolderName(MSG_DBUS_FOLDER_TYPE_CONVERSATION);

         QMap< tU32, QVector<most_fi_tcl_MsgConversationDetailListItem> >::iterator iter;

         if (folderNode)
         {
            iter = folderNode->m_pvConversationDetailList.find(u32ConversationID);

            if((iter != folderNode->m_pvConversationDetailList.end()) && (oGetSliceConversationDetailsMethodStart->u32WindowStart < static_cast<tU32>(iter.value().size())))
            {
               bResult = true;

               fileSystemManager->bSortConversationDetailList(&iter.value());

               vHandle_GetSliceConversationDetails_Return(fileSystemManager,
                     iter.value(),
                     oGetSliceConversationDetailsMethodStart->u32WindowStart,
                     oGetSliceConversationDetailsMethodStart->u32WindowSize);
            }
         }
      }

      if(false == bResult)
      {
         // SMS & EMAIL lists are not available, sent error response to client
         ETG_TRACE_ERR(("Invalid u16ListHandle (or) u32WindowStart has negative value"));
         vHandle_GetSliceConversationDetails_Error();
      }
      oGetSliceConversationDetailsMethodStart->vDestroy();
      delete oGetSliceConversationDetailsMethodStart;
   }
   delete l_prJob;
}

tVoid fc_messaging_tclService_Messaging::vHandle_GetSliceConversationDetails_Return(
      FileSystemManager* fileSystemManager,
      QVector<most_fi_tcl_MsgConversationDetailListItem> pConversationDetailListItem,
      tU32 u32WindowStart, tU32 u32WindowSize)
{
   ETG_TRACE_USR4(("vHandle_GetSliceConversationDetails_Return() called"));

   most_msgfi_tclMsgGetSliceConversationDetailsMethodResult oGetSliceConversationDetailsMethodResult;

   QList<most_fi_tcl_MsgMessageHandle> oUnreadMsgHandles;

   tInt u32ListLength = pConversationDetailListItem.size();
   tU32 u32Counter = 0;

   if (0 != u32ListLength)
   {
      most_fi_tcl_MsgConversationDetailList oConversationDetailListItem;

      for (tInt u32Index = (tInt)u32WindowStart;
            ((u32Counter < u32WindowSize) && (0 <= u32Index) && (u32Index < u32ListLength));
            u32Index++, u32Counter++)
      {
         ETG_TRACE_USR4(("vHandle_GetSliceConversationDetails_Return u32Counter: %d", u32Counter));

         // Do a deep copy to the result message from the local member.
         oConversationDetailListItem[u32Counter].oMessageHandle =
               pConversationDetailListItem.at(u32Index).oMessageHandle;

         ETG_TRACE_USR4(("MessageHandle : %d %x %x", pConversationDetailListItem.at(u32Index).oMessageHandle.u8DeviceHandle,
               pConversationDetailListItem.at(u32Index).oMessageHandle.u32MsgHandleUpper,
               pConversationDetailListItem.at(u32Index).oMessageHandle.u32MsgHandleLower));

         oConversationDetailListItem[u32Counter].oMessageDateTime.oMessageDate.sCldrYear.bSet(
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageDate.sCldrYear.szValue);
         oConversationDetailListItem[u32Counter].oMessageDateTime.oMessageDate.sCldrMonth.bSet(
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageDate.sCldrMonth.szValue);
         oConversationDetailListItem[u32Counter].oMessageDateTime.oMessageDate.sCldrDay.bSet(
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageDate.sCldrDay.szValue);
         oConversationDetailListItem[u32Counter].oMessageDateTime.oMessageTime.sHours.bSet(
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageTime.sHours.szValue);
         oConversationDetailListItem[u32Counter].oMessageDateTime.oMessageTime.sMinutes.bSet(
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageTime.sMinutes.szValue);
         oConversationDetailListItem[u32Counter].oMessageDateTime.oMessageTime.sSeconds.bSet(
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageTime.sSeconds.szValue);

         ETG_TRACE_USR4(("MessageDateTime Time: %2s:%2s:%2s", pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageTime.sHours.szValue,
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageTime.sMinutes.szValue,
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageTime.sSeconds.szValue));

         ETG_TRACE_USR4(("MessageDateTime Date: %2s/%2s/%4s", pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageDate.sCldrDay.szValue,
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageDate.sCldrMonth.szValue,
               pConversationDetailListItem.at(u32Index).oMessageDateTime.oMessageDate.sCldrYear.szValue));

         oConversationDetailListItem[u32Counter].sMessageContent.bSet(
               pConversationDetailListItem.at(u32Index).sMessageContent.szValue);

         ETG_TRACE_USR4(("MessageContent : %s", pConversationDetailListItem.at(u32Index).sMessageContent.szValue));

         oConversationDetailListItem[u32Counter].u32AttachmentSize =
               pConversationDetailListItem.at(u32Index).u32AttachmentSize;

         ETG_TRACE_USR4(("AttachmentSize : %u", pConversationDetailListItem.at(u32Index).u32AttachmentSize));

         oConversationDetailListItem[u32Counter].e8MessageType.enType =
               pConversationDetailListItem.at(u32Index).e8MessageType.enType;

         ETG_TRACE_USR4(("MessageType : %u", ETG_ENUM(TR_MESSAGELIST, pConversationDetailListItem.at(u32Index).e8MessageType.enType)));

         oConversationDetailListItem[u32Counter].e8MessageStatus.enType =
                        pConversationDetailListItem.at(u32Index).e8MessageStatus.enType;

         ETG_TRACE_USR4(("MessageStatus : %u", ETG_ENUM(TR_MESSAGESTATUS, pConversationDetailListItem.at(u32Index).e8MessageStatus.enType)));

         oConversationDetailListItem[u32Counter].bReadStatusFlag =
               pConversationDetailListItem.at(u32Index).bReadStatusFlag;

         if(FALSE == pConversationDetailListItem.at(u32Index).bReadStatusFlag)
         {
            oUnreadMsgHandles.append(pConversationDetailListItem.at(u32Index).oMessageHandle);
         }

         ETG_TRACE_USR4(("ReadStatusFlag : %d", pConversationDetailListItem.at(u32Index).bReadStatusFlag));

         oConversationDetailListItem[u32Counter].oAddressField.e8AddressFieldType.enType =
                        pConversationDetailListItem.at(u32Index).oAddressField.e8AddressFieldType.enType;

         ETG_TRACE_USR4(("AddressFieldType : %d", pConversationDetailListItem.at(u32Index).oAddressField.e8AddressFieldType.enType));

         oConversationDetailListItem[u32Counter].oAddressField.sFirstName.bSet(
               pConversationDetailListItem.at(u32Index).oAddressField.sFirstName.szValue);

         ETG_TRACE_USR4(("FirstName : %s", pConversationDetailListItem.at(u32Index).oAddressField.sFirstName.szValue));

         oConversationDetailListItem[u32Counter].oAddressField.sLastName.bSet(
               pConversationDetailListItem.at(u32Index).oAddressField.sLastName.szValue);

         ETG_TRACE_USR4(("LastName : %s", pConversationDetailListItem.at(u32Index).oAddressField.sLastName.szValue));

         oConversationDetailListItem[u32Counter].oAddressField.sEmailAddress.bSet(
               pConversationDetailListItem.at(u32Index).oAddressField.sEmailAddress.szValue);

         ETG_TRACE_USR4(("EmailAddress : %s", pConversationDetailListItem.at(u32Index).oAddressField.sEmailAddress.szValue));

         oConversationDetailListItem[u32Counter].oAddressField.sPhoneNumber.bSet(
               pConversationDetailListItem.at(u32Index).oAddressField.sPhoneNumber.szValue);

         ETG_TRACE_USR4(("PhoneNumber : %s", pConversationDetailListItem.at(u32Index).oAddressField.sPhoneNumber.szValue));

         oConversationDetailListItem[u32Counter].u32ContactHandle =
               pConversationDetailListItem.at(u32Index).u32ContactHandle;

         ETG_TRACE_USR4(("ContactHandle : %u", pConversationDetailListItem.at(u32Index).u32ContactHandle));
      }

      oGetSliceConversationDetailsMethodResult.oConversationDetailList = oConversationDetailListItem;
   }

   fi_tclVisitorMessage oMRMsg(oGetSliceConversationDetailsMethodResult, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oGetSliceConversationDetails.u16SourceAppID,
         m_oGetSliceConversationDetails.u16RegisterID,
         m_oGetSliceConversationDetails.u16CmdCounter,
         m_oGetSliceConversationDetails.u16FunctionID,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_GetSliceConversationDetails_Return: PostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   if(0 < oUnreadMsgHandles.size() && fileSystemManager)
   {
      ETG_TRACE_USR4(("There are %d unread messages", oUnreadMsgHandles.size()));

      //set message read status to true
      fileSystemManager->vSetMsgUnreadMessageHandles(oUnreadMsgHandles);
   }
   else
   {
      MessagingMutex::messagingMutexUnlock();
   }

   oGetSliceConversationDetailsMethodResult.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetSliceConversationDetails_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of GetSliceConversationDetails to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetSliceConversationDetails_Error()
{
   ETG_TRACE_USR4(("vHandle_GetSliceConversationDetails_Error() called"));

   most_msgfi_tclMsgGetSliceConversationDetailsError oGetSliceConversationDetailsError;

   oGetSliceConversationDetailsError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(oGetSliceConversationDetailsError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,                              // ServiceDataMsg
          m_oGetSliceConversationDetails.u16SourceAppID,
          m_oGetSliceConversationDetails.u16RegisterID,  // RegisterID
          m_oGetSliceConversationDetails.u16CmdCounter,  // CmdCounter,
          m_oGetSliceConversationDetails.u16FunctionID,  // Function ID
          AMT_C_U8_CCAMSG_OPCODE_ERROR                   // Opcode
          );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_GetSliceConversationDetails_Error: PostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   MessagingMutex::messagingMutexUnlock();

   oGetSliceConversationDetailsError.vDestroy();
}
/*******************************************************************************
 ** FUNCTION:   vHandle_ReleaseMessageList_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for ReleaseMessageList method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReleaseMessageList_MethodStart(
        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ReleaseMessageList_MethodStart() called"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgReleaseMessageListMethodStart oReleaseMessageListMethodStart;

   if (oVisitorMsg.s32GetData(oReleaseMessageListMethodStart,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      m_oReleaseMessageList.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oReleaseMessageList.u16RegisterID = poMessage->u16GetRegisterID();
      m_oReleaseMessageList.u16CmdCounter = poMessage->u16GetCmdCounter();
      m_oReleaseMessageList.u16FunctionID = poMessage->u16GetFunctionID();

      FolderNode* l_poFolderNode = NULLPTR;
      QString serviceName;

      tU8 deviceHandle = static_cast<tU8>(oReleaseMessageListMethodStart.u16ListHandle >> 8);

      serviceName = DbusClientInterface::getInstance().getCurrentActiveService(deviceHandle);
      FileSystemManager* l_poFileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(deviceHandle, serviceName);

      if (l_poFileSystemManager)
      {
         l_poFolderNode = l_poFileSystemManager->poSearchNodeByListHandle(oReleaseMessageListMethodStart.u16ListHandle);

         if (!l_poFolderNode)
         {
            // SMS list not available for received ListHandle, check for EMAIL list
            serviceName = DbusClientInterface::getInstance().getCurrentActiveService(deviceHandle, most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_EMAIL);
            l_poFileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(deviceHandle, serviceName);

            if (l_poFileSystemManager)
            {
               l_poFolderNode = l_poFileSystemManager->poSearchNodeByListHandle(oReleaseMessageListMethodStart.u16ListHandle);
            }
         }
      }

      if (l_poFileSystemManager && l_poFolderNode)
      {
         //Bugfix for GMMY15-5992 Large empty gap is present in text messages list
         Phonebook_Detail::vDestroyPhonebookInstance(deviceHandle);

         vHandle_ReleaseMessageList_Return(oReleaseMessageListMethodStart.u16ListHandle);
      }
      else
      {
         vHandle_ReleaseMessageList_Error();
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_ReleaseMessageList_MethodStart");
   }

   oReleaseMessageListMethodStart.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ReleaseMessageList_Return
 *******************************************************************************/
/* DESCRIPTION:
 *  function is used to send the result of ReleaseMessageList to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReleaseMessageList_Return(tU16 u16ListHandle)
{
   ETG_TRACE_USR4(("vHandle_ReleaseMessageList_Return() ListHandle = %d", u16ListHandle));

   most_msgfi_tclMsgReleaseMessageListMethodResult oReleaseMessageListMethodResult;

   oReleaseMessageListMethodResult.u16ListHandle = u16ListHandle;

   fi_tclVisitorMessage oMRMsg(oReleaseMessageListMethodResult, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oReleaseMessageList.u16SourceAppID, // ServiceDataMsg
         m_oReleaseMessageList.u16RegisterID, // RegisterID
         m_oReleaseMessageList.u16CmdCounter, // CmdCounter,
         m_oReleaseMessageList.u16FunctionID, // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_ReleaseMessageList_Return: enPostMessage() failed"));

         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oReleaseMessageListMethodResult.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_ReleaseMessageList_Error()
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_ReleaseMessageList_Error
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReleaseMessageList_Error()
{
   ETG_TRACE_USR4(("vHandle_ReleaseMessageList_Error() called"));

   most_msgfi_tclMsgReleaseMessageListError oReleaseMessageListMethodError;

   oReleaseMessageListMethodError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(oReleaseMessageListMethodError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oReleaseMessageList.u16SourceAppID, // ServiceDataMsg
         m_oReleaseMessageList.u16RegisterID,  // RegisterID
         m_oReleaseMessageList.u16CmdCounter,  // CmdCounter
         m_oReleaseMessageList.u16FunctionID,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR          // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_ReleaseMessageList_Error: enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oReleaseMessageListMethodError.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetMessage_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetMessage method
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetMessage_MethodStart(
        amt_tclServiceData* poMessage, struct Job *f_prJob)
{
    ETG_TRACE_USR4(("vHandle_GetMessage_MethodStart : Called"));

    struct Job* l_prJob = NULLPTR;
    most_msgfi_tclMsgGetMessageMethodStart* l_poGetMessageMethodStart = NULLPTR;

    if(poMessage)
    {
        fi_tclVisitorMessage oVisitorMsg(poMessage);

        l_poGetMessageMethodStart = new most_msgfi_tclMsgGetMessageMethodStart;
        if(l_poGetMessageMethodStart)
        {
            if (oVisitorMsg.s32GetData(*l_poGetMessageMethodStart,
                                       MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
            {
                vHandleGetDataError(poMessage, "vHandle_GetMessage_MethodStart");
                delete l_poGetMessageMethodStart;
                return;
            }
            FileSystemManager* l_poFileSystemManager;
            l_poFileSystemManager = FileSystemManager::poFileSystemManager(
                                l_poGetMessageMethodStart->oMessageHandle);
            if (l_poFileSystemManager)
            {
               l_prJob = new Job;
               if(l_prJob)
               {
                   l_prJob->m_pfMethod  = &fc_messaging_tclService_Messaging::vHandle_GetMessage_MethodStart;
                   l_prJob->u8DeviceHandle = l_poGetMessageMethodStart->oMessageHandle.u8DeviceHandle;
                   l_prJob->u16CmdCounter = poMessage->u16GetCmdCounter();
                   l_prJob->u16FunctionID = poMessage->u16GetFunctionID();
                   l_prJob->u16RegisterID = poMessage->u16GetRegisterID();
                   l_prJob->u16SourceAppID = poMessage->u16GetSourceAppID();
                   l_prJob->m_pvMethodStart = l_poGetMessageMethodStart;
               }
            }
            else
            {
               ETG_TRACE_USR4(("vHandle_GetMessage_MethodStart : Invalid Message Handle"));
               vHandleInvalidDataError(poMessage);
               delete l_poGetMessageMethodStart;
               return;
            }
        }

    }
    else if(f_prJob)
    {
        l_prJob = f_prJob;
    }
    else
    {
        return; // Should not come here
    }

    if (MessagingMutex::messagingMutexTrylock(l_prJob) != 0)
    {
        tVoid*  pCustodialLose = l_prJob ;  //To remove LINT warning : Custodial Pointer
        (tVoid) pCustodialLose ;
        return; // Didn't get the lock :(
    }
    if(l_prJob)
    {
        m_oGetMessage.u16SourceAppID = l_prJob->u16SourceAppID;
        m_oGetMessage.u16RegisterID = l_prJob->u16RegisterID;
        m_oGetMessage.u16CmdCounter = l_prJob->u16CmdCounter;
        m_oGetMessage.u16FunctionID = l_prJob->u16FunctionID;

        if(l_poGetMessageMethodStart) //LINT
        {
            tVoid*  pCustodialLose = l_poGetMessageMethodStart ;  //To remove LINT warning
            (tVoid) pCustodialLose ;
        }
        l_poGetMessageMethodStart =
                (most_msgfi_tclMsgGetMessageMethodStart*)l_prJob->m_pvMethodStart;
    }
    FileSystemManager* l_poFileSystemManager = NULLPTR;
    if(l_poGetMessageMethodStart)
    {
        l_poFileSystemManager = FileSystemManager::poFileSystemManager(
                    l_poGetMessageMethodStart->oMessageHandle);
        if (l_poFileSystemManager)
        {
            l_poFileSystemManager->vGetMessage(
                        l_poGetMessageMethodStart->oMessageHandle,
                        e8GetMessageStart);
        }
        else
        {
            vHandle_GetMessage_Error();
        }

        l_poGetMessageMethodStart->vDestroy();
        delete l_poGetMessageMethodStart;
    }
    delete l_prJob;
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetAttachmentList_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetAttachmentList method
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetAttachmentList_MethodStart(
        amt_tclServiceData* poMessage)
{

    ETG_TRACE_USR4(("GetAttachmentList_MethodStart CALLED"));
    /* Extract the Call instance and other required info from poMessage */
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgGetAttachmentListMethodStart
            oGetAttachmentListMethodStart_MS;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.

    if (oVisitorMsg.s32GetData(oGetAttachmentListMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {

        ETG_TRACE_USR1(("DEVICE HANDLE: %u",oGetAttachmentListMethodStart_MS.oMessageHandle.u8DeviceHandle));
        ETG_TRACE_USR1(("MESSAGE HANDLER UPPER: %u",static_cast<unsigned int> (oGetAttachmentListMethodStart_MS.oMessageHandle.u32MsgHandleUpper)));
       ETG_TRACE_USR1(("MESSAGE HANDLER LOWER: %u",static_cast<unsigned int> (oGetAttachmentListMethodStart_MS.oMessageHandle.u32MsgHandleLower)));

        m_oGetAttachmentList.u16SourceAppID = poMessage->u16GetSourceAppID();
        m_oGetAttachmentList.u16RegisterID = poMessage->u16GetRegisterID();
        m_oGetAttachmentList.u16CmdCounter = poMessage->u16GetCmdCounter();
        m_oGetAttachmentList.u16FunctionID = poMessage->u16GetFunctionID();

        //        if (m_pMsgListInterface)
        //        {
        //            (void) m_pMsgListInterface->GetAttachmentList(
        //                        &oGetAttachmentListMethodStart_MS,
        //                        &m_oGetAttachmentListMethodResult_MR);
        //        }

    }
    else
    {
        // If we come here, we assume the client has sent an erroneous message,
        // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
        // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

        // Create AMT error message and set ErrorCode
        amt_tclServiceDataError oErrorMsg(*poMessage,
                                          CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
        // NOTE: The constructor of amt_tclServiceDataError is re-using the
        // received message '*poMessage' It will collect all necessary Service
        // Information (Source-, Target AppID, RegID, ..) and exchange the
        // Source- and Target AppID and even set the Opcode to ERROR.
        // Additionally the ErrorCode value is set to the second provided
        // parameter. For the new message new memory is allocated, i.e. the
        // received message 'poMessage' is not re-used in terms of memory and
        // the implicite call of 'poMessage->bDelete();' by the framework after
        // leaving this handler function does not harm.

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oErrorMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_GetAttachmentList_MethodStart:\
                         'ERROR', ail enPostMessage() failed"));

                         // We can come here e.g. if the client has changed his application
                         // state to OFF in the meantime. Therefore we dont throw an assert.
                         // NORMAL_M_ASSERT_ALWAYS();
        }

        ETG_TRACE_FATAL(
                    ("vHandle_GetAttachmentList_MethodStart: \
                     Received message invalid."));
                     // We never expect to come here.
                     NORMAL_M_ASSERT_ALWAYS();
    }

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)

    oGetAttachmentListMethodStart_MS.vDestroy();

}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetBodyPartList_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetBodyPartList method
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetBodyPartList_MethodStart(
        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_GetBodyPartList_MethodStart() called"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgGetBodyPartListMethodStart l_oGetBodyPartListMethodStart;
   most_msgfi_tclMsgGetBodyPartListMethodResult l_oGetBodyPartListMethodResult;

   if (oVisitorMsg.s32GetData(l_oGetBodyPartListMethodStart,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      QString l_sMessage;
      most_fi_tcl_MsgBodyPartListItem l_oBodyPartListItem;

      MsgRead_t* msgRead = DbusClientInterface::getInstance().getMsgReadObject(
            l_oGetBodyPartListMethodStart.oMessageHandle.u8DeviceHandle);

      if(msgRead)
      {
         l_sMessage = msgRead->sGetDataPayLoad();
      }

      l_oBodyPartListItem.sBodyText.bSet(l_sMessage.toUtf8().constData());
      l_sMessage.truncate(25);
      l_oBodyPartListItem.sBodyTitle.bSet(l_sMessage.toUtf8().constData());

      l_oGetBodyPartListMethodResult.oBodyPartList.oItems.push_back(
            l_oBodyPartListItem);

      //Get the coresponding Call Instance
      fi_tclVisitorMessage oMRMsg(l_oGetBodyPartListMethodResult,
            MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

      // Here the message related header information is added and by this the
      // creation of the message is completed.
      vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(),     // ServiceDataMsg
            poMessage->u16GetRegisterID(),      // RegisterID
            poMessage->u16GetCmdCounter(),      // CmdCounter,
            poMessage->u16GetFunctionID(),      // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
      );

      if (oMRMsg.bIsValid())
      {
         // Post message
         if(_poMainAppl->enPostMessage(&oMRMsg, TRUE) != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR(("vHandle_GetBodyPartList_MethodStart Result Message Post failed"));
         }
      }
   }
   else
   {
      // Create AMT error message and set ErrorCode
      amt_tclServiceDataError oErrorMsg(*poMessage, CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);

      // Post message
      if(_poMainAppl->enPostMessage(&oErrorMsg, TRUE) != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_GetBodyPartList_MethodStart Error Message Post failed"));
      }
   }

   l_oGetBodyPartListMethodStart.vDestroy();
   l_oGetBodyPartListMethodResult.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_UpdateInbox_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for UpdateInbox method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_UpdateInbox_MethodStart(
    amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("UpdateInbox_MethodStart CALLED"));

    if (MessagingMutex::messagingMutexTrylock(NULLPTR) != 0) // Do not queue it
    {
        return;
    }

    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgUpdateInboxMethodStart oUpdateInboxMethodStart_MS;

    if (oVisitorMsg.s32GetData(oUpdateInboxMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        m_oUpdateInbox.u16SourceAppID = poMessage->u16GetSourceAppID(); // ServiceDataMsg
        m_oUpdateInbox.u16RegisterID = poMessage->u16GetRegisterID(); // RegisterID
        m_oUpdateInbox.u16CmdCounter = poMessage->u16GetCmdCounter(); // CmdCounter,
        m_oUpdateInbox.u16FunctionID = poMessage->u16GetFunctionID(); // Function ID

        DbusClientInterface::getInstance().messagingMapUpdateInbox();
    }
    else
    {
        vHandleGetDataError(poMessage, "vHandle_UpdateInbox_MethodStart");
    }

    oUpdateInboxMethodStart_MS.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_UpdateInbox_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of updateInbox to client.
 *
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_UpdateInbox_Return()
{
	ETG_TRACE_USR4(("vHandle_UpdateInbox_Return : Called"));

    most_msgfi_tclMsgUpdateInboxMethodResult m_oUpdateInboxMethodResult_MR;

    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oMRMsg(m_oUpdateInboxMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oUpdateInbox.u16SourceAppID, // ServiceDataMsg
                     m_oUpdateInbox.u16RegisterID, // RegisterID
                     m_oUpdateInbox.u16CmdCounter, // CmdCounter,
                     m_oUpdateInbox.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_UpdateInbox_MethodStart:\
                         'MethodResult', ail enPostMessage() failed"));
                         NORMAL_M_ASSERT_ALWAYS();
        }
    }

    MessagingMutex::messagingMutexUnlock();

    m_oUpdateInboxMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_UpdateInbox_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of updateInbox to client.
 *
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_UpdateInbox_Error()
{
    ETG_TRACE_USR4(("vHandle_UpdateInbox_Error : Called"));
    most_msgfi_tclMsgUpdateInboxError oMsgUpdateInboxError_ER;

    oMsgUpdateInboxError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(oMsgUpdateInboxError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oUpdateInbox.u16SourceAppID, // ServiceDataMsg
                     m_oUpdateInbox.u16RegisterID, // RegisterID
                     m_oUpdateInbox.u16CmdCounter, // CmdCounter,
                     m_oUpdateInbox.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("'vHandle_UpdateInbox_Error', enPostMessage() failed"));

            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    MessagingMutex::messagingMutexUnlock();

    oMsgUpdateInboxError_ER.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_DeleteMessage_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for DeleteMessage method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_DeleteMessage_MethodStart(
        amt_tclServiceData* poMessage, struct Job* f_prJob)
{
   ETG_TRACE_USR4(("vHandle_DeleteMessage_MethodStart : Called"));
   struct Job* l_prJob = NULLPTR;
   most_msgfi_tclMsgDeleteMessageMethodStart* l_poDeleteMessageMethodStart =
         NULLPTR;

   if (poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);

      l_poDeleteMessageMethodStart
            = new most_msgfi_tclMsgDeleteMessageMethodStart;
      if (l_poDeleteMessageMethodStart)
      {
         if (oVisitorMsg.s32GetData(*l_poDeleteMessageMethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)
               == OSAL_ERROR)
         {
            vHandleGetDataError(poMessage, "vHandle_DeleteMessage_MethodStart");
            delete l_poDeleteMessageMethodStart;
            return;
         }
         tU8 u8DeviceHandle = 0;
         most_fi_tcl_ShortStream obj = l_poDeleteMessageMethodStart->oDeleteMessageHandles.oItems[0];
         obj.vGetData(&u8DeviceHandle, 1);
         if ((u8DeviceHandle>= MINIMUM_DEVICE_HANDLE) && (u8DeviceHandle<= MAXIMUM_DEVICE_HANDLE))
         {
            l_prJob = new Job;
            if (l_prJob)
            {
               l_prJob->m_pfMethod
                     = &fc_messaging_tclService_Messaging::vHandle_DeleteMessage_MethodStart;

               l_prJob->u8DeviceHandle = u8DeviceHandle;
               l_prJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               l_prJob->u16FunctionID = poMessage->u16GetFunctionID();
               l_prJob->u16RegisterID = poMessage->u16GetRegisterID();
               l_prJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               l_prJob->m_pvMethodStart = l_poDeleteMessageMethodStart;
            }
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_DeleteMessage_MethodStart : Invalid Device Handle"));
            vHandleInvalidDataError(poMessage);
            delete l_poDeleteMessageMethodStart;
            return;
         }
      }
   }
   else if (f_prJob)
   {
      l_prJob = f_prJob;
   }
   else
   {
      return; // Should not come here
   }

   if (MessagingMutex::messagingMutexTrylock(l_prJob) != 0)
   {
      tVoid* pCustodialLose = l_prJob; //To remove LINT warning : Custodial Pointer
      (tVoid) pCustodialLose;
      return; // Didn't get the lock :(
   }
   if (l_prJob)
   {
      m_oDeleteMessage.u16SourceAppID = l_prJob->u16SourceAppID;
      m_oDeleteMessage.u16RegisterID = l_prJob->u16RegisterID;
      m_oDeleteMessage.u16CmdCounter = l_prJob->u16CmdCounter;
      m_oDeleteMessage.u16FunctionID = l_prJob->u16FunctionID;

      if (l_poDeleteMessageMethodStart) //lint
      {
         tVoid* pCustodialLose = l_poDeleteMessageMethodStart; //To remove LINT warning
         (tVoid) pCustodialLose;
      }
      l_poDeleteMessageMethodStart
            = (most_msgfi_tclMsgDeleteMessageMethodStart*) l_prJob->m_pvMethodStart;
   }

   //CMG3G-10357 : handling of MSG_DELETE_FAIL_DEVICE_UNAVAILABLE case
   if (l_poDeleteMessageMethodStart)
   {
      tBool bReturnValue = false;
      tU8 u8DeviceHandle = 0;
      most_fi_tcl_ShortStream obj = l_poDeleteMessageMethodStart->oDeleteMessageHandles.oItems[0];
      obj.vGetData(&u8DeviceHandle, 1);

      ETG_TRACE_USR1(("vHandle_DeleteMessage_MethodStart::DeviceHandle: %u", u8DeviceHandle));

      std::map<tU8, std::string>::iterator it = fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.find(u8DeviceHandle);
      if (it != fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.end())
      {
         bReturnValue = FileSystemManager::bDeleteMessageMethodStart(*l_poDeleteMessageMethodStart);

         if (bReturnValue == false)
         {
            vHandle_Delete_Error();
         }
      }
      else
      {
         most_msgfi_tclMsgDeleteMessageMethodResult l_oDeleteMessageMethodResult_MR;

         l_oDeleteMessageMethodResult_MR.oDeleteMessageHandles = l_poDeleteMessageMethodStart->oDeleteMessageHandles;
         l_oDeleteMessageMethodResult_MR.e8DeletionStatus.enType
               = most_fi_tcl_e8_MsgDeletionStatus::FI_EN_E8MSG_DELETE_FAIL_DEVICE_UNAVAILABLE;

         vHandle_DeleteMessage_Return(l_oDeleteMessageMethodResult_MR);

         l_oDeleteMessageMethodResult_MR.vDestroy();        
      }
      l_poDeleteMessageMethodStart->vDestroy();
      delete l_poDeleteMessageMethodStart;
   }
   delete l_prJob;
}

/*******************************************************************************
 ** FUNCTION:   vHandle_DeleteMessage_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of DeleteMessage to client.
 *
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_DeleteMessage_Return(
        const most_msgfi_tclMsgDeleteMessageMethodResult& roDeleteMessageMethodResult_MR)
{
    ETG_TRACE_USR4(("vHandle_DeleteMessage_Return : Called"));

    fi_tclVisitorMessage oMRMsg(roDeleteMessageMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oDeleteMessage.u16SourceAppID, // ServiceDataMsg
                     m_oDeleteMessage.u16RegisterID, // RegisterID
                     m_oDeleteMessage.u16CmdCounter, // CmdCounter,
                     m_oDeleteMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
    	ETG_TRACE_USR1((" Deletion Status:: %u",ETG_ENUM(TR_DELETESTATUS,roDeleteMessageMethodResult_MR.e8DeletionStatus.enType)));
    	// Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_DeleteMessage_Return: enPostMessage() failed"));

            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    MessagingMutex::messagingMutexUnlock();
    // Not to destroy the delete object
}

/*******************************************************************************
 ** FUNCTION:   vHandle_Delete_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of DeleteMessage to client.
 *
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_Delete_Error()
{
    ETG_TRACE_USR4(("vHandle_Delete_Error : Called"));
    most_msgfi_tclMsgDeleteMessageError oMsgDeleteMessageError_ER;

    oMsgDeleteMessageError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(oMsgDeleteMessageError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oDeleteMessage.u16SourceAppID, // ServiceDataMsg
                     m_oDeleteMessage.u16RegisterID, // RegisterID
                     m_oDeleteMessage.u16CmdCounter, // CmdCounter,
                     m_oDeleteMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(("vHandle_Delete_Error: enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    MessagingMutex::messagingMutexUnlock();

    oMsgDeleteMessageError_ER.vDestroy();
}
/*******************************************************************************
 ** FUNCTION:   vHandle_ProvideMessageHeader_Method
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for ProvideMessageHeader method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ProvideMessageHeader_Method(
        amt_tclServiceData* poMessage)
{

    ETG_TRACE_USR4(("vHandle_ProvideMessageHeader_Method : Called"));

    //tBool bReturnValue = false; LINT
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    most_msgfi_tclMsgProvideMessageHeaderMethodStart
            oProvideMessageHeaderMethodStart_MS;

    if (oVisitorMsg.s32GetData(oProvideMessageHeaderMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        SendMessage* l_poSendMessage;
        l_poSendMessage = SendMessage::poGetSendMessageInstance();

        if (l_poSendMessage)
        {
            l_poSendMessage->vSetProvideMessageHeader(
                        oProvideMessageHeaderMethodStart_MS);
        }
    }
    else
    {
        vHandleGetDataError(poMessage, "vHandle_ProvideMessageHeader_Method");
    }

    most_msgfi_tclMsgProvideMessageHeaderMethodResult
            oProvideMessageHeaderMethodResult_MR;

    oProvideMessageHeaderMethodResult_MR.u8CreateMessageHandle
            = oProvideMessageHeaderMethodStart_MS.u8CreateMessageHandle;

    fi_tclVisitorMessage oMRMsg(oProvideMessageHeaderMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
                     poMessage->u16GetRegisterID(), // RegisterID
                     poMessage->u16GetCmdCounter(), // CmdCounter,
                     poMessage->u16GetFunctionID(), // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        (" vHandle_ProvideMessageHeader_Method: Post failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    oProvideMessageHeaderMethodResult_MR.vDestroy();
    oProvideMessageHeaderMethodStart_MS.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ReadMessageAloud_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for ReadMessageAloud method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReadMessageAloud_MethodStart(
        amt_tclServiceData* poMessage, struct Job* prJob)
{
   ETG_TRACE_USR4(("vHandle_ReadMessageAloud_MethodStart : Called"));

#ifdef READ_MESSAGE_ALOUD_SUPPORT

   struct Job* poTempJob = NULLPTR;
   most_msgfi_tclMsgReadMessageAloudMethodStart* poReadMessageAloudMethodStart = NULLPTR;

   if (poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);

      poReadMessageAloudMethodStart = new most_msgfi_tclMsgReadMessageAloudMethodStart;
      if (poReadMessageAloudMethodStart)
      {
         if (oVisitorMsg.s32GetData(*poReadMessageAloudMethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
         {
            vHandleGetDataError(poMessage, "vHandle_ReadMessageAloud_MethodStart");
            delete poReadMessageAloudMethodStart;
            return;
         }
         FileSystemManager* l_poFileSystemManager;
         l_poFileSystemManager = FileSystemManager::poFileSystemManager(
        		 poReadMessageAloudMethodStart->oMessageHandle);
         if (l_poFileSystemManager)
         {
            poTempJob = new Job;

            if (poTempJob)
            {
               poTempJob->m_pfMethod = &fc_messaging_tclService_Messaging::vHandle_ReadMessageAloud_MethodStart;
               poTempJob->u8DeviceHandle = poReadMessageAloudMethodStart->oMessageHandle.u8DeviceHandle;
               poTempJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               poTempJob->u16FunctionID = poMessage->u16GetFunctionID();
               poTempJob->u16RegisterID = poMessage->u16GetRegisterID();
               poTempJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               poTempJob->m_pvMethodStart = poReadMessageAloudMethodStart;
            }
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_ReadMessageAloud_MethodStart: Invalid Message Handle"));
            vHandleInvalidDataError(poMessage);
            delete poReadMessageAloudMethodStart;
            return;
         }
      }
   }
   else if (prJob)
   {
      poTempJob = prJob;
   }
   else
   {
      return; // Should not come here
   }

   if (MessagingMutex::messagingMutexTrylock(poTempJob) != 0)
   {

      tVoid* pCustodialLose = poTempJob; //To remove LINT warning
      (tVoid) pCustodialLose;
      return; // Didn't get the lock :(
   }
   if (poTempJob)
   {
      m_oReadMessageAloud.u16SourceAppID = poTempJob->u16SourceAppID;
      m_oReadMessageAloud.u16RegisterID = poTempJob->u16RegisterID;
      m_oReadMessageAloud.u16CmdCounter = poTempJob->u16CmdCounter;
      m_oReadMessageAloud.u16FunctionID = poTempJob->u16FunctionID;

      if (poReadMessageAloudMethodStart) //lint
      {
         tVoid* pCustodialLose = poReadMessageAloudMethodStart; //To remove LINT warning
         (tVoid) pCustodialLose;
      }

      poReadMessageAloudMethodStart = (most_msgfi_tclMsgReadMessageAloudMethodStart*) poTempJob->m_pvMethodStart;
   }

   ReadMessageAloud* oReadMessageAloud = ReadMessageAloud::poGetInstance();

   if (oReadMessageAloud)
   {
      if (poReadMessageAloudMethodStart)
      {
         ETG_TRACE_USR1(("Playback Action :: %u", ETG_ENUM(TR_MSG_PLAYBACKACTION, poReadMessageAloudMethodStart->e8PlaybackAction.enType)));

         if (poReadMessageAloudMethodStart->e8PlaybackAction.enType == most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_PLAY)
         {
            oReadMessageAloud->vGetMessageForReadAloud(*poReadMessageAloudMethodStart);
         }
         else
         {
            oReadMessageAloud->vStopReadAloud();
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("ReadMessageAloud Object is not present"));
      vHandle_ReadMessageAloud_Error();
   }

   if (poReadMessageAloudMethodStart)
   {
      poReadMessageAloudMethodStart->vDestroy();
      delete poReadMessageAloudMethodStart;
   }
   delete poTempJob;

#else
   (void)(prJob);
   if(poMessage)
   {
      m_oReadMessageAloud.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oReadMessageAloud.u16RegisterID = poMessage->u16GetRegisterID();
      m_oReadMessageAloud.u16CmdCounter = poMessage->u16GetCmdCounter();
      m_oReadMessageAloud.u16FunctionID = poMessage->u16GetFunctionID();
      ETG_TRACE_USR4(("ReadMessageAloud Interface not supported"));
      vHandle_ReadMessageAloud_Error();
   }
#endif
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ReadMessageAloud_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of ReadMessageAloud to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReadMessageAloud_Return(
        most_fi_tcl_e8_MsgPlaybackAction::tenType p_e8PlaybackAction)
{
    ETG_TRACE_USR4(("vHandle_ReadMessageAloud_Return : Called"));

    most_msgfi_tclMsgReadMessageAloudMethodResult
            l_oReadMessageAloudMethodResult_MR;
    l_oReadMessageAloudMethodResult_MR.e8PlaybackAction.enType
            = p_e8PlaybackAction;

    fi_tclVisitorMessage oMRMsg(l_oReadMessageAloudMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oReadMessageAloud.u16SourceAppID,
                     m_oReadMessageAloud.u16RegisterID,
                     m_oReadMessageAloud.u16CmdCounter,
                     m_oReadMessageAloud.u16FunctionID,
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);

    if (oMRMsg.bIsValid())
    {
    	ETG_TRACE_USR1((" Playback Action:: %u",ETG_ENUM(TR_MSG_PLAYBACKACTION,l_oReadMessageAloudMethodResult_MR.e8PlaybackAction.enType)));
    	// Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_ReadMessageAloud_Return: Messgae Post failed"));
        }
    }

    MessagingMutex::messagingMutexUnlock();

    l_oReadMessageAloudMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ReadMessageAloud_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of ReadMessageAloud to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReadMessageAloud_Error()
{
   ETG_TRACE_USR4(("vHandle_ReadMessageAloud_Error : Called"));

   most_msgfi_tclMsgReadMessageAloudError l_oMsgReadMessageAloudError_ER;
   l_oMsgReadMessageAloudError_ER.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;
   fi_tclVisitorMessage oMsg(l_oMsgReadMessageAloudError_ER, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMsg, m_oReadMessageAloud.u16SourceAppID, // ServiceDataMsg
   m_oReadMessageAloud.u16RegisterID, // RegisterID
   m_oReadMessageAloud.u16CmdCounter, // CmdCounter,
   m_oReadMessageAloud.u16FunctionID, // Function ID
   AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
   );

   if (oMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_ReadMessageAloud_Error: Error in message post"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   MessagingMutex::messagingMutexUnlock();

   l_oMsgReadMessageAloudError_ER.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ReadMessageAloudExtended_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for ReadMessageAloudExtended method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReadMessageAloudExtended_MethodStart(amt_tclServiceData* poMessage, struct Job* prJob)
{
   ETG_TRACE_USR4(("vHandle_ReadMessageAloudExtended_MethodStart() called"));

#ifdef READ_MESSAGE_ALOUD_SUPPORT
   struct Job* poTempJob = NULLPTR;

   most_msgfi_tclMsgReadMessageAloudExtendedMethodStart* poReadMessageAloudExtendedMethodStart = NULLPTR;

   if (poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);

      poReadMessageAloudExtendedMethodStart = new most_msgfi_tclMsgReadMessageAloudExtendedMethodStart;
      if (poReadMessageAloudExtendedMethodStart)
      {
         if (oVisitorMsg.s32GetData(*poReadMessageAloudExtendedMethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
         {
            vHandleGetDataError(poMessage, "vHandle_ReadMessageAloudExtended_MethodStart");
            delete poReadMessageAloudExtendedMethodStart;
            return;
         }

         poTempJob = new Job;

         if (poTempJob)
         {
            poTempJob->m_pfMethod = &fc_messaging_tclService_Messaging::vHandle_ReadMessageAloudExtended_MethodStart;
            poTempJob->u8DeviceHandle = 0; //handled this in loopback service
            poTempJob->u16CmdCounter = poMessage->u16GetCmdCounter();
            poTempJob->u16FunctionID = poMessage->u16GetFunctionID();
            poTempJob->u16RegisterID = poMessage->u16GetRegisterID();
            poTempJob->u16SourceAppID = poMessage->u16GetSourceAppID();
            poTempJob->m_pvMethodStart = poReadMessageAloudExtendedMethodStart;
         }
      }
   }
   else if (prJob)
   {
      poTempJob = prJob;
   }
   else
   {
      return; // Should not come here
   }

   if (MessagingMutex::messagingMutexTrylock(poTempJob) != 0)
   {
      tVoid* pCustodialLose = poTempJob; //To remove LINT warning
      (tVoid) pCustodialLose;
      return; // Didn't get the lock :(
   }
   if (poTempJob)
   {
      m_oReadMessageAloudExtended.u16SourceAppID = poTempJob->u16SourceAppID;
      m_oReadMessageAloudExtended.u16RegisterID = poTempJob->u16RegisterID;
      m_oReadMessageAloudExtended.u16CmdCounter = poTempJob->u16CmdCounter;
      m_oReadMessageAloudExtended.u16FunctionID = poTempJob->u16FunctionID;

      if (poReadMessageAloudExtendedMethodStart) //lint
      {
         tVoid* pCustodialLose = poReadMessageAloudExtendedMethodStart; //To remove LINT warning
         (tVoid) pCustodialLose;
      }

      poReadMessageAloudExtendedMethodStart = (most_msgfi_tclMsgReadMessageAloudExtendedMethodStart*) poTempJob->m_pvMethodStart;
   }

   ReadMessageAloud* oReadMessageAloud = ReadMessageAloud::poGetInstance();

   if (oReadMessageAloud)
   {
      if (poReadMessageAloudExtendedMethodStart)
      {
         ETG_TRACE_USR1(("Playback Action :: %u", ETG_ENUM(TR_MSG_PLAYBACKACTION, poReadMessageAloudExtendedMethodStart->e8PlaybackAction.enType)));

         if (poReadMessageAloudExtendedMethodStart->e8PlaybackAction.enType == most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_PLAY)
         {
            oReadMessageAloud->vReadAloudContentFromClient(poReadMessageAloudExtendedMethodStart->sReadAloudContent.szValue);
         }
         else
         {
            oReadMessageAloud->vStopReadAloud(e8ReadMessageAloudExtendedMethod);
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("ReadMessageAloud Object is not present"));
      vHandle_ReadMessageAloudExtended_Error();
   }

   if (poReadMessageAloudExtendedMethodStart)
   {
      poReadMessageAloudExtendedMethodStart->vDestroy();
      delete poReadMessageAloudExtendedMethodStart;
   }

   delete poTempJob;
#else
   (void)(prJob);
   if (poMessage)
   {
      m_oReadMessageAloudExtended.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oReadMessageAloudExtended.u16RegisterID = poMessage->u16GetRegisterID();
      m_oReadMessageAloudExtended.u16CmdCounter = poMessage->u16GetCmdCounter();
      m_oReadMessageAloudExtended.u16FunctionID = poMessage->u16GetFunctionID();
      ETG_TRACE_USR4(("ReadMessageAloudExtended Interface not supported"));
      vHandle_ReadMessageAloudExtended_Error();
   }

#endif
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ReadMessageAloudExtended_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of ReadMessageAloudExtended to client.
 *
 *  PARAMETERS:
 *     IN:       playbackAction : Result of PlaybackAction which is requested by client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReadMessageAloudExtended_Return(
        most_fi_tcl_e8_MsgPlaybackAction::tenType playbackAction)
{
   ETG_TRACE_USR4(("vHandle_ReadMessageAloudExtended_Return() called"));

   most_msgfi_tclMsgReadMessageAloudExtendedMethodResult poReadMessageAloudExtendedMethodResult;

   poReadMessageAloudExtendedMethodResult.e8PlaybackAction.enType = playbackAction;

   fi_tclVisitorMessage oMRMsg(poReadMessageAloudExtendedMethodResult,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oReadMessageAloudExtended.u16SourceAppID,
         m_oReadMessageAloudExtended.u16RegisterID,
         m_oReadMessageAloudExtended.u16CmdCounter,
         m_oReadMessageAloudExtended.u16FunctionID,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);

   if (oMRMsg.bIsValid())
   {
      ETG_TRACE_USR1((" Playback Action:: %u", ETG_ENUM(TR_MSG_PLAYBACKACTION, poReadMessageAloudExtendedMethodResult.e8PlaybackAction.enType)));

      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_ReadMessageAloudExtended_Return: Messgae Post failed"));
      }
   }

   MessagingMutex::messagingMutexUnlock();

   poReadMessageAloudExtendedMethodResult.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ReadMessageAloudExtended_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of ReadMessageAloudExtended to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ReadMessageAloudExtended_Error()
{
   ETG_TRACE_USR4(("vHandle_ReadMessageAloudExtended_Error() called"));

   most_msgfi_tclMsgReadMessageAloudExtendedError poReadMessageAloudExtendedError;
   poReadMessageAloudExtendedError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMsg(poReadMessageAloudExtendedError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMsg,                            // ServiceDataMsg
         m_oReadMessageAloudExtended.u16SourceAppID, // SourceAppID
         m_oReadMessageAloudExtended.u16RegisterID,  // RegisterID
         m_oReadMessageAloudExtended.u16CmdCounter,  // CmdCounter
         m_oReadMessageAloudExtended.u16FunctionID,  // FunctionID
         AMT_C_U8_CCAMSG_OPCODE_ERROR                // Opcode
   );

   if (oMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_ReadMessageAloudExtended_Error: Messgae Post failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   MessagingMutex::messagingMutexUnlock();

   poReadMessageAloudExtendedError.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::updateAttachmentAudioStateProperty(
        tU8 e8Playmode)
{
    ETG_TRACE_USR4(("updateAttachmentAudioStateProperty : Called"));

    m_oAudioAttachmentPlaybackState_Status.e8PlaybackState.enType
                = (most_fi_tcl_e8_MsgPlaybackState::tenType) e8Playmode;
    eUpdateClients( MOST_MSGFI_C_U16_AUDIOATTACHMENTPLAYBACKSTATE);
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetImageAttachment_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetImageAttachment method
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetImageAttachment_MethodStart(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandle_GetImageAttachment_MethodStart : Called"));
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgGetImageAttachmentMethodStart
            oGetImageAttachmentMethodStart_MS;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.

    if (oVisitorMsg.s32GetData(oGetImageAttachmentMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {

        ETG_TRACE_USR1(("ATTACHMENT HANDLE: %u",oGetImageAttachmentMethodStart_MS.u16AttachmentHandle));

        m_oGetImageAttachment.u16SourceAppID = poMessage->u16GetSourceAppID();
        m_oGetImageAttachment.u16RegisterID = poMessage->u16GetRegisterID();
        m_oGetImageAttachment.u16CmdCounter = poMessage->u16GetCmdCounter();
        m_oGetImageAttachment.u16FunctionID = poMessage->u16GetFunctionID();

        //        if (m_pMsgListInterface)
        //        {
        //            (void) m_pMsgListInterface->GetImageAttachment(
        //                        &oGetImageAttachmentMethodStart_MS,
        //                        &m_oGetImageAttachmentMethodResult_MR);
        //        }

    }
    else
    {
        // If we come here, we assume the client has sent an erroneous message,
        // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
        // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

        // Create AMT error message and set ErrorCode
        amt_tclServiceDataError oErrorMsg(*poMessage,
                                          CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
        // NOTE: The constructor of amt_tclServiceDataError is re-using the
        // received message '*poMessage' It will collect all necessary Service
        // Information (Source-, Target AppID, RegID, ..) and exchange the
        // Source- and Target AppID and even set the Opcode to ERROR.
        // Additionally the ErrorCode value is set to the second provided
        // parameter. For the new message new memory is allocated, i.e. the
        // received message 'poMessage' is not re-used in terms of memory and
        // the implicite call of 'poMessage->bDelete();' by the framework after
        // leaving this handler function does not harm.

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oErrorMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR((
                        "vHandle_GetImageAttachment_MethodStart:\
                        'ERROR', enPostMessage() failed"));

                        // We can come here e.g. if the client has changed his application
                        // state to OFF in the meantime. Therefore we dont throw an assert.
                        // NORMAL_M_ASSERT_ALWAYS();
        }

        ETG_TRACE_FATAL(
                    ("vHandle_GetImageAttachment_MethodStart: \
                     Received message invalid."));
                     // We never expect to come here.
                     NORMAL_M_ASSERT_ALWAYS();
    }

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)

    oGetImageAttachmentMethodStart_MS.vDestroy();

}

/*******************************************************************************
 ** FUNCTION:   vHandle_MessagingAudioPlayback_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for start the Playback of audio attachment
 *   of last got Email , MMS.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_MessagingAudioAttachmentPlayback_MethodStart(
        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_MessagingAudioAttachmentPlayback_MethodStart() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgAudioAttachmentPlaybackMethodStart p_start;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(p_start,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR1(("  ATTACHMENT HANDLE: %u", p_start.u16AttachmentHandle));
      ETG_TRACE_USR1((" Playback Action:: %u",ETG_ENUM(TR_MSG_PLAYBACKACTION,p_start.e8PlaybackAction.enType)));

      m_oMSGPlayAttachment.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oMSGPlayAttachment.u16RegisterID = poMessage->u16GetRegisterID();
      m_oMSGPlayAttachment.u16CmdCounter = poMessage->u16GetCmdCounter();
      m_oMSGPlayAttachment.u16FunctionID = poMessage->u16GetFunctionID();

      m_oAudioAttachmentPlaybackState_Status.u16AttachmentHandle = p_start.u16AttachmentHandle;

      switch (p_start.e8PlaybackAction.enType)
      {
         case most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_PLAY:
         {
            // Stroe the attachment handle to Play for the last Got Message
            //            FC_Messaging_MsgList::LastGotMessage()->setAttachMentHandleToPlay(
            //                        (int) p_start.u16AttachmentHandle);

            //Do Request AV Activation
            //fc_messaging_tclApp::m_poClientHandlerAVManager-> requestAvActivation(
            //            UCID_AttachAudioFileToMost);
            break;
         }
         case most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_PAUSE:
         {
            //if (FC_Messaging_Player::getCurrentPlayer() != NULL)
            //{
               //FC_Messaging_Player::getCurrentPlayer()->pause();
            //}

            updateAttachmentAudioStateProperty(
                  most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_PAUSED);

            vHandle_MessagingAudioAttachmentPlayback_Return(
                  most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_PAUSE);
            break;
         }
         case most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_STOP:
         {
            //if (FC_Messaging_Player::getCurrentPlayer() != NULL)
            //{
               //FC_Messaging_Player::getCurrentPlayer()->stop();
            //}

            updateAttachmentAudioStateProperty(
                  most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);

            //Do Request AV DeActivation
            //m_oAudioAttachmentPlaybackState_Status.u16AttachmentHandle = p_start.u16AttachmentHandle;

            //fc_messaging_tclApp::m_poClientHandlerAVManager->requestAVDeactivation(
            //      (tU8) UCID_AttachAudioFileToMost);
            break;
         }
      }
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      // Create AMT error message and set ErrorCode
      amt_tclServiceDataError oErrorMsg(*poMessage,
            CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
      // NOTE: The constructor of amt_tclServiceDataError is re-using the
      // received message '*poMessage' It will collect all necessary Service
      // Information (Source-, Target AppID, RegID, ..) and exchange the
      // Source- and Target AppID and even set the Opcode to ERROR.
      // Additionally the ErrorCode value is set to the second provided
      // parameter. For the new message new memory is allocated, i.e. the
      // received message 'poMessage' is not re-used in terms of memory and
      // the implicite call of 'poMessage->bDelete();' by the framework after
      // leaving this handler function does not harm.

      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
            &oErrorMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("'ERROR', enPostMessage() failed"));

         // We can come here e.g. if the client has changed his application
         // state to OFF in the meantime. Therefore we dont throw an assert.
         // NORMAL_M_ASSERT_ALWAYS();
      }
      // We never expect to come here.
      NORMAL_M_ASSERT_ALWAYS();
   }
   p_start.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vHandle_MessagingAudioAttachmentPlayback_Return(
        tU8 state)
{
	ETG_TRACE_USR4(("vHandle_MessagingAudioAttachmentPlayback_Return : Called"));
    most_msgfi_tclMsgAudioAttachmentPlaybackMethodResult oMethodResultObj;
    oMethodResultObj.e8PlaybackAction.enType
            = (most_fi_tcl_e8_MsgPlaybackAction::tenType) state;

    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oMethodResult(oMethodResultObj,
                                       MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData(oMethodResult, m_oMSGPlayAttachment.u16SourceAppID, // ServiceDataMsg
                     m_oMSGPlayAttachment.u16RegisterID, // RegisterID
                     m_oMSGPlayAttachment.u16CmdCounter, // CmdCounter,
                     m_oMSGPlayAttachment.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMethodResult.bIsValid())
    {
    	ETG_TRACE_USR1((" Playback Action:: %u",ETG_ENUM(TR_MSG_PLAYBACKACTION,oMethodResultObj.e8PlaybackAction.enType)));
    	// Post Message
        if (_poMainAppl->enPostMessage(&oMethodResult, TRUE)
                != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("fc_messaging_tclService_Messaging::vHandle_MessagingAudioAttachmentPlayback_Return: Message Post failed"));
        }
    }
    else
    {
        ETG_TRACE_ERR(
                    ("fc_messaging_tclService_Messaging::vHandle_MessagingAudioAttachmentPlayback_Return: Invaild Message"));
    }

    oMethodResultObj.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetImageAttachment_Return(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetImageAttachment_Return method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetImageAttachment_Return()
{
    ETG_TRACE_USR4(("vHandle_GetImageAttachment_Return : Called"));
    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oMRMsg(m_oGetImageAttachmentMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData(oMRMsg, m_oGetImageAttachment.u16SourceAppID, // ServiceDataMsg
                     m_oGetImageAttachment.u16RegisterID, // RegisterID
                     m_oGetImageAttachment.u16CmdCounter, // CmdCounter,
                     m_oGetImageAttachment.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR((
                        "vHandle_GetImageAttachment_MethodStart:\
                        'MethodResult', enPostMessage() failed"));

                        // We can come here e.g. if the client has changed his application
                        //state to OFF in the meantime. Therefore we dont throw an assert.
                        NORMAL_M_ASSERT_ALWAYS();
        }
    }

}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetImageAttachmentInfo_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetImageAttachmentInfo method
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetImageAttachmentInfo_MethodStart(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandle_GetImageAttachmentInfo_MethodStart : Called"));
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgGetImageAttachmentInfoMethodStart
            oGetImageAttachmentInfoMethodStart_MS;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.

    if (oVisitorMsg.s32GetData(oGetImageAttachmentInfoMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        ETG_TRACE_USR1(("ATTACHMENT HANDLE: %u",oGetImageAttachmentInfoMethodStart_MS.u16AttachmentHandle));

        m_oGetImageAttachmentInfo.u16SourceAppID
                = poMessage->u16GetSourceAppID();
        m_oGetImageAttachmentInfo.u16RegisterID = poMessage->u16GetRegisterID();
        m_oGetImageAttachmentInfo.u16CmdCounter = poMessage->u16GetCmdCounter();
        m_oGetImageAttachmentInfo.u16FunctionID = poMessage->u16GetFunctionID();

        //        if (m_pMsgListInterface)
        //        {
        //            (void) m_pMsgListInterface->GetImageAttachmentInfo(
        //                        &oGetImageAttachmentInfoMethodStart_MS,
        //                        &m_oGetImageAttachmentInfoMethodResult_MR);
        //        }

    }
    else
    {
        // If we come here, we assume the client has sent an erroneous message,
        // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
        // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

        // Create AMT error message and set ErrorCode
        amt_tclServiceDataError oErrorMsg(*poMessage,
                                          CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
        // NOTE: The constructor of amt_tclServiceDataError is re-using the
        // received message '*poMessage' It will collect all necessary Service
        // Information (Source-, Target AppID, RegID, ..) and exchange the
        // Source- and Target AppID and even set the Opcode to ERROR.
        // Additionally the ErrorCode value is set to the second provided
        // parameter. For the new message new memory is allocated, i.e. the
        // received message 'poMessage' is not re-used in terms of memory and
        // the implicite call of 'poMessage->bDelete();' by the framework after
        // leaving this handler function does not harm.

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oErrorMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR((
                        "vHandle_GetImageAttachmentInfo_MethodStart:\
                        'ERROR', enPostMessage() failed"));

                        // We can come here e.g. if the client has changed his application
                        // state to OFF in the meantime. Therefore we dont throw an assert.
                        // NORMAL_M_ASSERT_ALWAYS();
        }

        ETG_TRACE_FATAL(
                    ("vHandle_GetImageAttachmentInfo_MethodStart: \
                     Received message invalid."));
                     // We never expect to come here.
                     NORMAL_M_ASSERT_ALWAYS();
    }

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)

    oGetImageAttachmentInfoMethodStart_MS.vDestroy();

}
/*******************************************************************************
 ** FUNCTION:   vHandle_GetImageAttachmentInfo_Return(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetImageAttachmentInfo_Return method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetImageAttachmentInfo_Return()
{
    ETG_TRACE_USR4(("vHandle_GetImageAttachmentInfo_Return : Called"));
    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oMRMsg(m_oGetImageAttachmentInfoMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData(oMRMsg, m_oGetImageAttachmentInfo.u16SourceAppID, // ServiceDataMsg
                     m_oGetImageAttachmentInfo.u16RegisterID, // RegisterID
                     m_oGetImageAttachmentInfo.u16CmdCounter, // CmdCounter,
                     m_oGetImageAttachmentInfo.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR((
                        "vHandle_GetImageAttachmentInfo_MethodStart:\
                        'MethodResult', enPostMessage() failed"));

                        // We can come here e.g. if the client has changed his application
                        //state to OFF in the meantime. Therefore we dont throw an assert.
                        NORMAL_M_ASSERT_ALWAYS();
        }
    }

}

/*******************************************************************************
 ** FUNCTION:   vHandle_ProvideMessageBody_Method
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for ProvideMessageBody method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ProvideMessageBody_Method(
        amt_tclServiceData* poMessage)
{

    ETG_TRACE_USR4(("vHandle_ProvideMessageBody_Method : Called"));
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    most_msgfi_tclMsgProvideMessageBodyMethodStart
            oProvideMessageBodyMethodStart_MS;

    if (oVisitorMsg.s32GetData(oProvideMessageBodyMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        SendMessage* l_poSendMessage;
        l_poSendMessage = SendMessage::poGetSendMessageInstance();

        if (l_poSendMessage)
        {
            l_poSendMessage->vSetProvideMessageBody(
                        oProvideMessageBodyMethodStart_MS);
        }
    }
    else
    {
        vHandleGetDataError(poMessage, "vHandle_ProvideMessageBody_Method");
        return;
    }

    most_msgfi_tclMsgProvideMessageBodyMethodResult
            oProvideMessageBodyMethodResult_MR;

    oProvideMessageBodyMethodResult_MR.u8CreateMessageHandle
            = oProvideMessageBodyMethodStart_MS.u8CreateMessageHandle;

    fi_tclVisitorMessage oMRMsg(oProvideMessageBodyMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
                     poMessage->u16GetRegisterID(), // RegisterID
                     poMessage->u16GetCmdCounter(), // CmdCounter,
                     poMessage->u16GetFunctionID(), // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(("'MethodResult', enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    oProvideMessageBodyMethodResult_MR.vDestroy();
    oProvideMessageBodyMethodStart_MS.vDestroy();
}

//#ifdef FCAT1_6_1

tBool fc_messaging_tclService_Messaging::vHandle_MessagingDeviceConnection_ValetMode()
{
    ETG_TRACE_USR4(("vHandle_MessagingDeviceConnection_ValetMode: Entered"));
    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oStatusMsg(m_oMessagingDeviceConnection_Status,
                                    MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData(oStatusMsg, // ServiceDataMsg
                     CCA_C_U16_APP_UI, // Target AppID for HMI
                     0, // RegisterID
                     0, // CmdCounter,
                     MOST_MSGFI_C_U16_MESSAGINGDEVICECONNECTION, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_STATUS // Opcode
                     );

    // Post message
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                &oStatusMsg, TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        ETG_TRACE_ERR((
                    "vHandle_MessagingDeviceConnection_ValetMode: 'STATUS', enPostMessage() failed, 'ail_tenCommunicationError' = %u",
                    static_cast<unsigned int> ((tU32) enResult)));
        // We can come here e.g. if the client has changed his application
        // state to OFF in the meantime. Therefore we dont throw an assert.
        // NORMAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_USR4(("vHandle_MessagingDeviceConnection_ValetMode: 'STATUS Posting"));
    }
    return TRUE;

}

tBool fc_messaging_tclService_Messaging::vHandle_MessageListChange_ValetMode()
{
    ETG_TRACE_USR4(("vHandle_MessageListChange_ValetMode: Entered"));
    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oStatusMsg(m_oMessageListChange_Status,
                                    MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData(oStatusMsg, // ServiceDataMsg
                     CCA_C_U16_APP_UI, // Target AppID for HMI
                     0, // RegisterID
                     0, // CmdCounter,
                     MOST_MSGFI_C_U16_MESSAGELISTCHANGE, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_STATUS // Opcode
                     );

    // Post message
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                &oStatusMsg, TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        ETG_TRACE_ERR((
                    "vHandle_MessageListChange_ValetMode: 'STATUS', enPostMessage() failed, 'ail_tenCommunicationError' = %u",
                    static_cast<unsigned int> ((tU32) enResult)));
        // We can come here e.g. if the client has changed his application
        // state to OFF in the meantime. Therefore we dont throw an assert.
        // NORMAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_USR4(("vHandle_MessageListChange_ValetMode: 'STATUS Posting"));
    }
    return TRUE;

}

/*******************************************************************************
 ** FUNCTION:   vHandle_MessagingDeviceConnection_UpReg(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_MessagingDeviceConnection_UpReg method
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_MessagingDeviceConnection_UpReg(
        amt_tclServiceData* poMessage)
{
	ETG_TRACE_USR4(("vHandle_MessagingDeviceConnection_UpReg : Called"));
    /* Extract the required info from poMessage */
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgMessagingDeviceConnectionUpReg
            oMessagingDeviceConnection_UpReg;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.

    m_oMessagingDeviceConnection_Status.u8DeviceHandle = 2;
    m_oMessagingDeviceConnection_Status.bEmailSupport = 1;
    m_oMessagingDeviceConnection_Status.bSMSSupport = 1;
    m_oMessagingDeviceConnection_Status.bMMSSupport = 1;
    ETG_TRACE_USR4(("1"));
    m_oMessagingDeviceConnection_Status.bUpdateInboxSupport = 1;
    m_oMessagingDeviceConnection_Status.bMessagingMemoryStatus = 1;
    ETG_TRACE_USR4(("2"));
    m_oMessagingDeviceConnection_Status.oActiveAccount.u16AccountHandle = 1;
    m_oMessagingDeviceConnection_Status.oActiveAccount.sAccountName.bSet("");
    ETG_TRACE_USR4(("3"));

    if (oVisitorMsg.s32GetData(oMessagingDeviceConnection_UpReg,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {

        /* Get the coresponding Call Instance */
        fi_tclVisitorMessage oStatusMsg(m_oMessagingDeviceConnection_Status,
                                        MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

        // Here the message related header information is added and by this the
        // creation of the message is completed.
        vInitServiceData(oStatusMsg, // ServiceDataMsg
                         poMessage->u16GetSourceAppID(), // Target AppID
                         poMessage->u16GetRegisterID(), // RegisterID
                         poMessage->u16GetCmdCounter(), // CmdCounter,
                         poMessage->u16GetFunctionID(), // Function ID
                         AMT_C_U8_CCAMSG_OPCODE_UPREG // Opcode
                         );
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oStatusMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR((
                        "vHandle_MessagingDeviceConnection_UpReg: 'STATUS', enPostMessage() failed, 'ail_tenCommunicationError' = %u",
                        static_cast<unsigned int> ((tU32) enResult)));
            // We can come here e.g. if the client has changed his application
            // state to OFF in the meantime. Therefore we dont throw an assert.
            // NORMAL_M_ASSERT_ALWAYS();
        }
        else
        {
            ETG_TRACE_USR4((
                        "vHandle_MessagingDeviceConnection_UpReg: 'STATUS Posting"));
        }
    }
    else
    {
        // If we come here, we assume the client has sent an erroneous message,
        // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
        // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

        // Create AMT error message and set ErrorCode
        amt_tclServiceDataError oErrorMsg(*poMessage,
                                          CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
        // NOTE: The constructor of amt_tclServiceDataError is re-using the
        // received message '*poMessage' It will collect all necessary Service
        // Information (Source-, Target AppID, RegID, ..) and exchange the
        // Source- and Target AppID and even set the Opcode to ERROR.
        // Additionally the ErrorCode value is set to the second provided
        // parameter. For the new message new memory is allocated, i.e. the
        // received message 'poMessage' is not re-used in terms of memory and
        // the implicite call of 'poMessage->bDelete();' by the framework after
        // leaving this handler function does not harm.

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oErrorMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR((
                        "vHandle_MessagingDeviceConnection_UpReg: 'ERROR', enPostMessage() failed, 'ail_tenCommunicationError' = %u",
                        static_cast<unsigned int> ((tU32) enResult)));
            // We can come here e.g. if the client has changed his application
            // state to OFF in the meantime. Therefore we dont throw an assert.
            // NORMAL_M_ASSERT_ALWAYS();
        }

        ETG_TRACE_FATAL(
                    ("vHandle_MessagingDeviceConnection_UpReg: Received message invalid."));
        // We never expect to come here.
        NORMAL_M_ASSERT_ALWAYS();
    }

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)

    oMessagingDeviceConnection_UpReg.vDestroy();

}

/*******************************************************************************
 ** FUNCTION:   vHandle_MessagingDeviceConnection_RelUpReg(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_MessagingDeviceConnection_RelUpReg method
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_MessagingDeviceConnection_RelUpReg(
        amt_tclServiceData* poMessage)
{
	ETG_TRACE_USR4(("vHandle_MessagingDeviceConnection_RelUpReg : Called"));
    /* Extract the required info from poMessage */
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgMessagingDeviceConnectionRelUpReg
            oMessagingDeviceConnection_RelUpReg;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.

    if (oVisitorMsg.s32GetData(oMessagingDeviceConnection_RelUpReg,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        /* Get the coresponding Call Instance */
        fi_tclVisitorMessage oStatusMsg(m_oMessagingDeviceConnection_Status,
                                        MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

        // Here the message related header information is added and by this the
        // creation of the message is completed.
        vInitServiceData(oStatusMsg, // ServiceDataMsg
                         poMessage->u16GetSourceAppID(), // Target AppID
                         poMessage->u16GetRegisterID(), // RegisterID
                         poMessage->u16GetCmdCounter(), // CmdCounter,
                         poMessage->u16GetFunctionID(), // Function ID
                         AMT_C_U8_CCAMSG_OPCODE_RELUPREG // Opcode
                         );
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oStatusMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)

        {
            ETG_TRACE_ERR((
                        "vHandle_MessagingDeviceConnection_RelUpReg: 'STATUS', enPostMessage() failed, 'ail_tenCommunicationError' = %u",
                        static_cast<unsigned int> ((tU32) enResult)));
            // We can come here e.g. if the client has changed his application
            // state to OFF in the meantime. Therefore we dont throw an assert.
            // NORMAL_M_ASSERT_ALWAYS();
        }
        else
        {
            ETG_TRACE_USR4((
                        "vHandle_MessagingDeviceConnection_RelUpReg: 'STATUS Posting"));
        }
    }
    else
    {
        // If we come here, we assume the client has sent an erroneous message,
        // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
        // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

        // Create AMT error message and set ErrorCode
        amt_tclServiceDataError oErrorMsg(*poMessage,
                                          CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
        // NOTE: The constructor of amt_tclServiceDataError is re-using the
        // received message '*poMessage' It will collect all necessary Service
        // Information (Source-, Target AppID, RegID, ..) and exchange the
        // Source- and Target AppID and even set the Opcode to ERROR.
        // Additionally the ErrorCode value is set to the second provided
        // parameter. For the new message new memory is allocated, i.e. the
        // received message 'poMessage' is not re-used in terms of memory and
        // the implicite call of 'poMessage->bDelete();' by the framework after
        // leaving this handler function does not harm.

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oErrorMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR((
                        "vHandle_MessagingDeviceConnection_RelUpReg: 'ERROR', enPostMessage() failed, 'ail_tenCommunicationError' = %u",
                        static_cast<unsigned int> ((tU32) enResult)));
            // We can come here e.g. if the client has changed his application
            // state to OFF in the meantime. Therefore we dont throw an assert.
            // NORMAL_M_ASSERT_ALWAYS();
        }

        ETG_TRACE_FATAL(
                    ("vHandle_MessagingDeviceConnection_RelUpReg: Received message invalid."));
        // We never expect to come here.
        NORMAL_M_ASSERT_ALWAYS();
    }

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)

    oMessagingDeviceConnection_RelUpReg.vDestroy();

}
tVoid fc_messaging_tclService_Messaging::_MessagingDeviceConnection(
        DeviceConnection_t *objDeviceConnection_t)
{

    ETG_TRACE_USR4(("_MessagingDeviceConnection : Called"));

    m_oMessagingDeviceConnection_Status.u8DeviceHandle
            = objDeviceConnection_t->m_u8DeviceHandle;
    m_oMessagingDeviceConnection_Status.bEmailSupport
            = objDeviceConnection_t->m_bEmailSupport;
    m_oMessagingDeviceConnection_Status.bSMSSupport
            = objDeviceConnection_t->m_bSMSSupport;
    m_oMessagingDeviceConnection_Status.bMMSSupport
            = objDeviceConnection_t->m_bMMSSupport;
    m_oMessagingDeviceConnection_Status.bMessagingMemoryStatus
            = objDeviceConnection_t->m_bMessagingMemoryStatus;
    m_oMessagingDeviceConnection_Status.bUpdateInboxSupport = true;

    eUpdatePropertyToClients( MOST_MSGFI_C_U16_MESSAGINGDEVICECONNECTION);
}

tVoid fc_messaging_tclService_Messaging::vMessageListChange(
        most_msgfi_tclMsgMessageListChangeStatus f_roMessageListChangeStatus)
{
    ETG_TRACE_USR4(("vMessageListChange : Called"));
    m_oMessageListChange_Status.u16ListHandle
            = f_roMessageListChangeStatus.u16ListHandle;
    m_oMessageListChange_Status.e8ListChangeType
            = f_roMessageListChangeStatus.e8ListChangeType;
    m_oMessageListChange_Status.u16ListLength
            = f_roMessageListChangeStatus.u16ListLength;
    m_oMessageListChange_Status.u32NumItems
            = f_roMessageListChangeStatus.u32NumItems;

    most_fi_tcl_MsgChangeItems obj;

    obj.u32Items.push_back(f_roMessageListChangeStatus.oChangeItems.u32Items[0]);
    m_oMessageListChange_Status.oChangeItems = obj;

    m_oMessageListChange_Status = f_roMessageListChangeStatus;

    m_poMessagingService->eUpdatePropertyToClients(
                MOST_MSGFI_C_U16_MESSAGELISTCHANGE);
}

tVoid fc_messaging_tclService_Messaging::_MemoryStatusFromDbus(
        DeviceConnection_t *objDeviceConnection_t)
{
    ETG_TRACE_USR4(("_MemoryStatusFromDbus() called"));

    m_oMessagingDeviceConnection_Status.u8DeviceHandle
            = objDeviceConnection_t->m_u8DeviceHandle;
    m_oMessagingDeviceConnection_Status.bEmailSupport
            = objDeviceConnection_t->m_bEmailSupport;
    m_oMessagingDeviceConnection_Status.bSMSSupport
            = objDeviceConnection_t->m_bSMSSupport;
    m_oMessagingDeviceConnection_Status.bMMSSupport
            = objDeviceConnection_t->m_bMMSSupport;
    m_oMessagingDeviceConnection_Status.bUpdateInboxSupport
            = objDeviceConnection_t->m_bUpdateInboxSupport;
    m_oMessagingDeviceConnection_Status.bMessagingMemoryStatus
            = objDeviceConnection_t->m_bMessagingMemoryStatus;
    m_oMessagingDeviceConnection_Status.oActiveAccount.u16AccountHandle
            = objDeviceConnection_t->m_u16EmailAccountId;
    m_oMessagingDeviceConnection_Status.oActiveAccount.sAccountName.bSet(
                "SMSMMS");

    eUpdatePropertyToClients( MOST_MSGFI_C_U16_MESSAGINGDEVICECONNECTION);
}

tVoid fc_messaging_tclService_Messaging::_MapDeviceCapabilities(
      const trFC_Msg_MapDeviceCapabilities *prFC_Msg_MapDeviceCapabilities)
{
   QMutexLocker lock(&m_lockMapDeviceCapabilities);

   vResetMapDeviceCapabilities();

   most_fi_tcl_MapDeviceCapabilitiesListItem oMapDeviceCapabilitiesListItem;

   m_oMapDeviceCapabilitiesStatus.u8DeviceHandle = prFC_Msg_MapDeviceCapabilities->u8DeviceHandle;
   oMapDeviceCapabilitiesListItem.u8DeviceHandle = prFC_Msg_MapDeviceCapabilities->u8DeviceHandle;
   m_oMapDeviceCapabilitiesStatus.u16SupportedMessageTypes = prFC_Msg_MapDeviceCapabilities->u16SupportedMessageTypes;
   oMapDeviceCapabilitiesListItem.u16SupportedMessageTypes = prFC_Msg_MapDeviceCapabilities->u16SupportedMessageTypes;
   m_oMapDeviceCapabilitiesStatus.u16SupportedMapFeatures = prFC_Msg_MapDeviceCapabilities->u16SupportedMapFeatures;
   oMapDeviceCapabilitiesListItem.u16SupportedMapFeatures = prFC_Msg_MapDeviceCapabilities->u16SupportedMapFeatures;
   m_oMapDeviceCapabilitiesStatus.oActiveAccount.u16AccountHandle =
         prFC_Msg_MapDeviceCapabilities->rFC_Msg_ActiveEmailAccount.u16AccountHandle;
   oMapDeviceCapabilitiesListItem.oActiveAccount.u16AccountHandle =
         prFC_Msg_MapDeviceCapabilities->rFC_Msg_ActiveEmailAccount.u16AccountHandle;
   m_oMapDeviceCapabilitiesStatus.oActiveAccount.sAccountName.bSet(
         prFC_Msg_MapDeviceCapabilities->rFC_Msg_ActiveEmailAccount.sAccountName.toUtf8().constData());
   oMapDeviceCapabilitiesListItem.oActiveAccount.sAccountName.bSet(
         prFC_Msg_MapDeviceCapabilities->rFC_Msg_ActiveEmailAccount.sAccountName.toUtf8().constData());
   m_oMapDeviceCapabilitiesStatus.e8MapConnectionState.enType =
         (most_fi_tcl_e8_MapConnectionStatus::tenType)prFC_Msg_MapDeviceCapabilities->u8MapConnectionState;
   oMapDeviceCapabilitiesListItem.e8MapConnectionState.enType =
         (most_fi_tcl_e8_MapConnectionStatus::tenType)prFC_Msg_MapDeviceCapabilities->u8MapConnectionState;

   tBool isDeviceAvailable = false;
   for (tU8 i = 0; i < m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems.size(); i++)
   {
      if (prFC_Msg_MapDeviceCapabilities->u8DeviceHandle == m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems[i].u8DeviceHandle)
      {
         m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems[i] = oMapDeviceCapabilitiesListItem;
         isDeviceAvailable = true;
         break;
      }
   }
   if(!isDeviceAvailable)
   {
      m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems.push_back(oMapDeviceCapabilitiesListItem);
   }

   eUpdatePropertyToClients(MOST_MSGFI_C_U16_MAPDEVICECAPABILITIES);
   eUpdatePropertyToClients(MOST_MSGFI_C_U16_MAPDEVICECAPABILITIESEXTENDED);
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetMessage_Return(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetMessage_Return method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetMessage_Return(
        most_fi_tcl_MsgMessageDetailsResult& f_roMessageDetailsResult)
{
   ETG_TRACE_USR4(("vHandle_GetMessage_Return() called"));

   most_msgfi_tclMsgGetMessageMethodResult l_oGetMessageMethodResult;
   l_oGetMessageMethodResult.oMessageDetailsResult = f_roMessageDetailsResult;

   // Get the coresponding Call Instance
   fi_tclVisitorMessage oMRMsg(l_oGetMessageMethodResult, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData(oMRMsg, m_oGetMessage.u16SourceAppID, // ServiceDataMsg
         m_oGetMessage.u16RegisterID,                     // RegisterID
         m_oGetMessage.u16CmdCounter,                     // CmdCounter,
         m_oGetMessage.u16FunctionID,                     // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT              // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_GetMessage_Return: enPostMessage() failed"));

         // We can come here e.g. if the client has changed his application
         //state to OFF in the meantime. Therefore we dont throw an assert.
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   //Wait for ReadStatus set completion in evo stack
   //DbusClientIf client = DbusClientInterface::getInstance().getClientInterface();
   //if(DBUS_CLIENT_IF_EVOLUTION != client)
   //{
      //MessagingMutex::messagingMutexUnlock();
   //}

   l_oGetMessageMethodResult.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vHandle_GetMessage_Error()
{
    ETG_TRACE_USR4(("vHandle_GetMessage_Error : Called"));
    most_msgfi_tclMsgGetMessageError oMsgGetMessageError_ER;

    oMsgGetMessageError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(oMsgGetMessageError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oGetMessage.u16SourceAppID, // ServiceDataMsg
                     m_oGetMessage.u16RegisterID, // RegisterID
                     m_oGetMessage.u16CmdCounter, // CmdCounter,
                     m_oGetMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_GetMessage_Error: 'MethodResult', enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    MessagingMutex::messagingMutexUnlock();

    oMsgGetMessageError_ER.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetAttachmentList_Return(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetAttachmentList_Return method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetAttachmentList_Return()
{
	ETG_TRACE_USR4(("vHandle_GetAttachmentList_Return : Called"));
    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oMRMsg(m_oGetAttachmentListMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData(oMRMsg, m_oGetAttachmentList.u16SourceAppID, // ServiceDataMsg
                     m_oGetAttachmentList.u16RegisterID, // RegisterID
                     m_oGetAttachmentList.u16CmdCounter, // CmdCounter,
                     m_oGetAttachmentList.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_GetAttachmentList_MethodStart:\
                         'MethodResult',ail enPostMessage() failed"));

                         // We can come here e.g. if the client has changed his application
                         //state to OFF in the meantime. Therefore we dont throw an assert.
                         NORMAL_M_ASSERT_ALWAYS();
        }
    }

}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetBodyPartList_Return(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetBodyPartList_Return method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetBodyPartList_Return(most_msgfi_tclMsgGetBodyPartListMethodResult&
                                                                        f_roGetBodyPartListMethodResult)
{
	ETG_TRACE_USR4(("vHandle_GetBodyPartList_Return : Called"));
    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oMRMsg(f_roGetBodyPartListMethodResult,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData(oMRMsg, m_oGetBodyPartList.u16SourceAppID, // ServiceDataMsg
                     m_oGetBodyPartList.u16RegisterID, // RegisterID
                     m_oGetBodyPartList.u16CmdCounter, // CmdCounter,
                     m_oGetBodyPartList.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_GetBodyPartList_MethodStart:\
                         'MethodResult',ail enPostMessage() failed"));

                         // We can come here e.g. if the client has changed his application
                         //state to OFF in the meantime. Therefore we dont throw an assert.
                         NORMAL_M_ASSERT_ALWAYS();
        }

    }

}

tVoid fc_messaging_tclService_Messaging::vHandleGetDataError(
        amt_tclServiceData* poMessage, tCString poHandlerStr)
{
	ETG_TRACE_USR4(("vHandleGetDataError : Called"));
    // Create AMT error message and set ErrorCode
    amt_tclServiceDataError oErrorMsg(*poMessage,
                                      CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
    // Post message
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg,
                                                                    TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        ETG_TRACE_ERR(("ail_tenCommunicationError = %u", enResult));
        // We can come here e.g. if the client has changed his application
        // state to OFF in the meantime. Therefore we don't throw an assert.
        // NORMAL_M_ASSERT_ALWAYS();
    }
    else
    {
        ETG_TRACE_FATAL(("%s: Received message invalid.", poHandlerStr));
        // We never expect to come here.
        // NORMAL_M_ASSERT_ALWAYS();
    }
}

tVoid fc_messaging_tclService_Messaging::vHandleInvalidDataError(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandleInvalidDataError : Called"));
    // Create AMT error message and set ErrorCode
    amt_tclServiceDataError oErrorMsg(*poMessage,CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE);
    // Post message
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg,
                                                                    TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        ETG_TRACE_ERR(("ail_tenCommunicationError = %u", enResult));
        // We can come here e.g. if the client has changed his application
        // state to OFF in the meantime. Therefore we don't throw an assert.
        // NORMAL_M_ASSERT_ALWAYS();
    }
}

/*******************************************************************************
 ** FUNCTION:   isSourceNrValid
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a function to validate the SourceNr
 *   
 *  PARAMETERS:
 *     IN:   u8SourceNr       : This is the received through ISource interface
 *  RETURNVALUE: None
 *******************************************************************************/
tBool fc_messaging_tclService_Messaging::isSourceNrValid(tU8 u8SourceNr)
{
	tBool bValidSourceNr = false;
	if ((u8SourceNr == SOURCENR_NEW_MSG_ALERT_MESSAGING) || 
		(u8SourceNr == SOURCENR_DEVICE_MEMORY_FULL_MESSAGING))
	{
		bValidSourceNr = true;
	}

	return bValidSourceNr;
}

/*******************************************************************************
 ** FUNCTION:   vHandle_Allocate_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for Allocate method
 *   Thie function talks to FC_StreamRouter to get the required audio bandwidth
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_Allocate_MethodStart(
        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_Allocate_MethodStart() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   tU16 ucid = 0;

   bool hasNoError = true;

   //BugFix GMMY15-3986 - Methods with missing parameters are not responded.
   //Assign the member variables before the condition check since the Methoderror
   //will need the ID's
   m_MSGAllocateMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
   m_MSGAllocateMessage.u16RegisterID = poMessage->u16GetRegisterID();
   m_MSGAllocateMessage.u16CmdCounter = poMessage->u16GetCmdCounter();
   m_MSGAllocateMessage.u16FunctionID = poMessage->u16GetFunctionID();

   //Create the (message related) FI data object
   most_msgfi_tclMsgAllocateMethodStart oAllocate_MS;

   if (oVisitorMsg.s32GetData(oAllocate_MS,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR1(("SOURCE NR RECEIVED=%d", oAllocate_MS.u8SourceNr));

      if (!isSourceNrValid(oAllocate_MS.u8SourceNr))
      {
         ETG_TRACE_ERR(("Invalid SourceNr is passed by HMI"));
         vSendAllocateMethodError(
               most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE,
               oAllocate_MS.u8SourceNr);
         oAllocate_MS.vDestroy();

         return;
      }

      //fc_messaging_tclApp::m_poClientHandlerAVManager->m_oChannelAcquisitionState = e8ChannelActivated;

      //if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(FC_MESSAGING_ALERTAVACTIVATION_RETRY_TIMER_ID))
      //{
      //ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_ALERTAVACTIVATION_RETRY_TIMER_ID"));
      //}

      //fc_messaging_tclApp::m_poClientHandlerAVManager->m_bIsTimerRunning = false;

      tU8 l_u8CurrentAllocateStatus;
      l_u8CurrentAllocateStatus
      = AllocateStatusState::getCurrentAllocateState(
            oAllocate_MS.u8SourceNr);

      if (l_u8CurrentAllocateStatus == MSG_CHANNEL_ALLOCATING)
      {
         // Error condition.
         ETG_TRACE_USR3(("SourceNr is in allocating state already"));
         vSendAllocateMethodError(most_fi_tcl_e8_ErrorCode::FI_EN_BUSY,
               oAllocate_MS.u8SourceNr);
         hasNoError = false;
      }
      else if (l_u8CurrentAllocateStatus == MSG_CHANNEL_ALLOCATED)
      {
         // Return Method Result.
         ETG_TRACE_USR3(("SourceNr is in ALLOCATED state already"));

         tU16 l_u16BlockWidth (AllocateStatusState::getBlockWidth(
               oAllocate_MS.u8SourceNr));
         tU16 l_u16ConnectionLabel(AllocateStatusState::getConnectionLabel(
               oAllocate_MS.u8SourceNr));

         vSendAllocateMethodResult(oAllocate_MS.u8SourceNr, static_cast<tU8>(l_u16BlockWidth),
               l_u16ConnectionLabel);
         hasNoError = false;
      }

      if (hasNoError)
      {
         AllocateStatusState::setCurrentAllocateSourceNr(oAllocate_MS.u8SourceNr);
         AllocateStatusState::setCurrentAllocateState(oAllocate_MS.u8SourceNr,
               MSG_CHANNEL_ALLOCATING);

         MessagingTypes::getUcidFromSourceNr(oAllocate_MS.u8SourceNr, &ucid);

         if(ucid == INVALID_UCID_OR_SOURCENR)
         {
            //this->m_oAudioRecording->m_externalBeep = true;

            switch(oAllocate_MS.u8SourceNr)
            {
               case SOURCENR_NEW_MSG_ALERT_MESSAGING:
               case SOURCENR_DEVICE_MEMORY_FULL_MESSAGING:
               {
                  ucid = UCID_InMsgAlertToneToMost;
                  break;
               }
               case SOURCENR_BEEP_REC_MESSAGING:
               {
                  ucid = UCID_BeepRecMsgToMost;
                  break;
               }
               default:
               {
                  ETG_TRACE_USR3(("Invalid SourceNr"));
               }
            }
         }
         else
         {
            //this->m_oAudioRecording->m_externalBeep = false;
         }
      }
   }
   else
   {
      // paramter is missing error
      vSendAllocateMethodError(most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH);
   }

   oAllocate_MS.vDestroy();
}

/******************************************************************
 * *************
 ** FUNCTION:   vSendDeAllocateMethodResult
 *******************************************************************************/
/* DESCRIPTION:
 *   Send DeAllocate method result to AV_Manager
 *
 *  PARAMETERS:
 *          IN :None
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vSendDeAllocateMethodResult(
        tU16 u16UCID)
{
    ETG_TRACE_USR4(("vSendDeAllocateMethodResult : Called UCID = %d", u16UCID));
    bool error = false;
    most_msgfi_tclMsgDeAllocateMethodResult oDeAllocate_MR;

    error = !MessagingTypes::getSourceNrFromUcid(u16UCID,
                                                 &oDeAllocate_MR.u8SourceNr);

    if (error == false)
    {

        AllocateStatusState::setCurrentAllocateState(oDeAllocate_MR.u8SourceNr,
                                                     MSG_CHANNEL_NOT_ALLOCATED);
        AllocateStatusState::setBlockWidth(oDeAllocate_MR.u8SourceNr, 0);
        AllocateStatusState::setConnectionLabel(oDeAllocate_MR.u8SourceNr, 0);

        fi_tclVisitorMessage oMRMsg(oDeAllocate_MR,
                                    MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

        vInitServiceData(oMRMsg, // ServiceDataMsg
                         m_MSGDeAllocateMessage.u16SourceAppID, // Target AppID
                         m_MSGDeAllocateMessage.u16RegisterID, // RegisterID
                         m_MSGDeAllocateMessage.u16CmdCounter, // CmdCounter,
                         m_MSGDeAllocateMessage.u16FunctionID, // Function ID
                         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                         );

        // POST THE MESSAGE
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        (" enPostMessage() failed for 'Allocate - METHODRESULT' message, 'ail_tenCommunicationError' = %u", (tU32) enResult));
        }
    }
    else
    {
        vSendDeAllocateMethodError(
                    (most_fi_tcl_e8_ErrorCode::tenType) MSG_ERR_EN_PROCESSINGERR);
        ETG_TRACE_ERR(("vSendDeAllocateMethodResult: InvalidSourceNr"));
    }

    oDeAllocate_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vSendDeAllocateMethodError
 *******************************************************************************/
/* DESCRIPTION:
 *   Send DeAllocate method error.
 *
 *  PARAMETERS:
 *          IN :None
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vSendDeAllocateMethodError(
        most_fi_tcl_e8_ErrorCode::tenType eErrorCode)
{
   ETG_TRACE_USR4(("vSendDeAllocateMethodError : Called"));

   most_msgfi_tclMsgDeAllocateError oDeAllocate_MErr;
   tU8 au8ErrorInfo[1] = {0x00};

   switch (eErrorCode)
   {
      //Only these 4 error cases are set during DeAllocateMethod processing.

      case most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH:
      {
         au8ErrorInfo[0] = 0x00;//== Dummy zero (from document most_MPlay_fi)
      }
      break;

      case most_fi_tcl_e8_ErrorCode::FI_EN_BUSY:
      {
         au8ErrorInfo[0] = 0x02;//== Return Parameter (from document most_MPlay_fi)
      }
      break;

      case most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC:
      {
         au8ErrorInfo[0] = 0x04;//== Value Not Available (from document most_MPlay_fi)
      }
      break;

      case most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE:
      {
         au8ErrorInfo[0] = 0x02;//== Set the error parameter position (Second parameter is SourceNr) (from document most_MPlay_fi)
      }
      break;
      default:
         break;
   }

   // BugFix GMMY15-3986 - Methods with missing parameters are not responded
   // Assign only the error code and bypass other extra information
   if ((eErrorCode != MSG_ERR_EN_PROCESSINGERR) && (eErrorCode != most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH))
   {
      tU8 u8PositionOfErroneousParameter = 0x02;

      fi_tclOutContext oOut(au8ErrorInfo,
            au8ErrorInfo + sizeof(au8ErrorInfo),
            MOST_MSGFI_C_U16_SERVICE_MAJORVERSION,
            oDeAllocate_MErr.enGetByteOrder());

      // write values to buffer
      (tVoid)(oOut << u8PositionOfErroneousParameter);

      oDeAllocate_MErr.e8ErrorCode.enType = eErrorCode;
      oDeAllocate_MErr.oErrorInfo.vSetData(au8ErrorInfo, sizeof(au8ErrorInfo));
   }
   else
   {
      oDeAllocate_MErr.e8ErrorCode.enType = eErrorCode;
   }

   fi_tclVisitorMessage oMEMsg(oDeAllocate_MErr,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMEMsg, // ServiceDataMsg
         m_MSGDeAllocateMessage.u16SourceAppID, // Target AppID
         m_MSGDeAllocateMessage.u16RegisterID, // RegisterID
         m_MSGDeAllocateMessage.u16CmdCounter, // CmdCounter,
         m_MSGDeAllocateMessage.u16FunctionID, // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
   );

   // POST THE MESSAGE
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMEMsg, TRUE);
   if (enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR((" enPostMessage() failed for 'DeAllocate - METHODERROR' message, 'ail_tenCommunicationError' = %u", (tU32) enResult));
   }

   oDeAllocate_MErr.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vSendAllocateMethodResult(
        tU8 u8SourceNr, tU8 BlockWidth, tU16 ConnectionLabel)
{
    ETG_TRACE_USR4(("vSendAllocateMethodResult : Called  SourceNr = %d",u8SourceNr));

    most_msgfi_tclMsgAllocateMethodResult oAllocate_MR;

    oAllocate_MR.u8SourceNr = u8SourceNr;
    oAllocate_MR.u16BlockWidth = BlockWidth;
    oAllocate_MR.u16ConnectionLabel = ConnectionLabel;

    AllocateStatusState::setCurrentAllocateState(u8SourceNr,
                                                 MSG_CHANNEL_ALLOCATED);
    AllocateStatusState::setBlockWidth(u8SourceNr, BlockWidth);
    AllocateStatusState::setConnectionLabel(u8SourceNr, ConnectionLabel);

    fi_tclVisitorMessage oMRMsg(oAllocate_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, // ServiceDataMsg
                     m_MSGAllocateMessage.u16SourceAppID, // Target AppID
                     m_MSGAllocateMessage.u16RegisterID, // RegisterID
                     m_MSGAllocateMessage.u16CmdCounter, // CmdCounter,
                     MOST_MSGFI_C_U16_ALLOCATE, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT); // Opcode

    // POST THE MESSAGE
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg,
                                                                    TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        ETG_TRACE_ERR(
                    (" enPostMessage() failed for 'Allocate - METHODERROR' message, 'ail_tenCommunicationError' = %u", (tU32) enResult));
        
    }

    oAllocate_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vSendAllocateMethodError
 *******************************************************************************/
/* DESCRIPTION:
 *   Send Allocate method error.
 *
 *  PARAMETERS:
 *          IN :None
 *  RETURNVALUE: None
 *******************************************************************************/

tVoid fc_messaging_tclService_Messaging::vSendAllocateMethodError(
        most_fi_tcl_e8_ErrorCode::tenType eErrorCode, tU8 u8SourceNr,
        amt_tclServiceData* poMessage)
{
	ETG_TRACE_USR4(("vSendAllocateMethodError : Called SOURCE NR=%d", u8SourceNr));

    most_msgfi_tclMsgAllocateError oAllocate_MErr;

    if (poMessage)
    {
        amt_tclServiceDataError oError(poMessage);
        // Dispatch the 16 bit CCA error code number from the received message
        tU16 l_u16ErrorCode = oError.u16GetErrorData();

        ETG_TRACE_ERR(("ERROR CODE RECEIVED FROM STREAM ROUTER =%20X",
                  l_u16ErrorCode));
        ETG_TRACE_USR2((
                    "VALUE =%20X",
                    MIDW_STREAMROUTERFI_C_U16_ERROR_E_MOST_FAILURE_REQUIRED_CONNECTIONLABEL_NOT_ALLOCATED));

        if (MIDW_STREAMROUTERFI_C_U16_ERROR_E_MOST_FAILURE_REQUIRED_CONNECTIONLABEL_NOT_ALLOCATED
                == l_u16ErrorCode
                || MIDW_STREAMROUTERFI_C_U16_ERROR_E_ALLOCATE_FAILED
                == l_u16ErrorCode)
        {
            ETG_TRACE_USR4(("most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE"));
            eErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
        }
        else
        {
            ETG_TRACE_USR4(("most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC"));
            eErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

        }
    }

    if (isSourceNrValid(u8SourceNr))
    {
        AllocateStatusState::setCurrentAllocateState(u8SourceNr,
                                                     MSG_CHANNEL_NOT_ALLOCATED);
        AllocateStatusState::setBlockWidth(u8SourceNr, 0);
        AllocateStatusState::setConnectionLabel(u8SourceNr, 0);
    }

    switch(eErrorCode)
    {
    default:
    case most_fi_tcl_e8_ErrorCode::FI_EN_BUSY:
    case most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE:
    case most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC:

    {
        tU8 au8ErrorInfo[3];
        tU16 u16BlockWidth = 0; // Invalid BlockWidth

        if( (u8SourceNr == 1) || (u8SourceNr == 2) ||
                (u8SourceNr == 3)|| (u8SourceNr == 5))
        {
            u16BlockWidth = 2;
        }
        else if (u8SourceNr == 4)
        {
            u16BlockWidth = 4;
        }
        fi_tclOutContext oOut(au8ErrorInfo,
                              au8ErrorInfo + sizeof(au8ErrorInfo),
                              MOST_MSGFI_C_U16_SERVICE_MAJORVERSION,
                              oAllocate_MErr.enGetByteOrder());

        // write values to buffer
        (tVoid)(oOut << u8SourceNr << u16BlockWidth);

        ETG_TRACE_USR2(("FI_EN_FUNCTIONSPECIFIC SOURCE NR=%d , BLOCKWIDTH =%d",
                  u8SourceNr, u16BlockWidth));
        ETG_TRACE_USR2(("FI_EN_FUNCTIONSPECIFIC SOURCE NR=%d , BLOCKWIDTH =%d",
                  au8ErrorInfo[0], *(au8ErrorInfo + 1)));

        // write data to FI data object
        oAllocate_MErr.e8ErrorCode.enType
                = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;
        oAllocate_MErr.oErrorInfo.vSetData(au8ErrorInfo, sizeof(au8ErrorInfo));

    }
        break;
    /* BugFix GMMY15-3986 - Methods with missing parameters are not responded 
	   Assign only the error code and bypass other extra information */
    case most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH:
	{
	   oAllocate_MErr.e8ErrorCode.enType = eErrorCode;
	}
	   break;
    case most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERNOTAVAILABLE:
    case most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE:
    {
        ETG_TRACE_USR4(("OTHER ERROR"));
        tU8 au8ErrorInfo[1] = {0x02};
        tU8 u8PositionOfErroneousparameter = 0x02; // //the second parameter(SourceNr) is wrong. (First one is SenderHandle)

        fi_tclOutContext oOut(au8ErrorInfo,
                              au8ErrorInfo + sizeof(au8ErrorInfo),
                              MOST_MSGFI_C_U16_SERVICE_MAJORVERSION,
                              oAllocate_MErr.enGetByteOrder());

        // write values to buffer
        (tVoid)(oOut << u8PositionOfErroneousparameter);

        // write data to FI data object
        oAllocate_MErr.e8ErrorCode.enType = eErrorCode;
        oAllocate_MErr.oErrorInfo.vSetData(au8ErrorInfo, sizeof(au8ErrorInfo));

    }
        break;
    };

    fi_tclVisitorMessage oMEMsg(oAllocate_MErr,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMEMsg, // ServiceDataMsg
                     m_MSGAllocateMessage.u16SourceAppID, // Target AppID
                     m_MSGAllocateMessage.u16RegisterID, // RegisterID
                     m_MSGAllocateMessage.u16CmdCounter, // CmdCounter,
                     m_MSGAllocateMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    // POST THE MESSAGE
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMEMsg,
                                                                    TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        ETG_TRACE_ERR(
                    (" enPostMessage() failed for 'Allocate - METHODERROR' message, 'ail_tenCommunicationError' = %u", (tU32) enResult));
    }

    oAllocate_MErr.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_DeAllocate_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for DeAllocate method
 *   Thie function talks to FC_StreamRouter to get the required audio bandwidth
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/

tVoid fc_messaging_tclService_Messaging::vHandle_DeAllocate_MethodStart(
        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_DeAllocate_MethodStart : Called"));

   tU16 ucid = 0;

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // BugFix GMMY15-3986 - Methods with missing parameters are not responded
   // Assign the member variables before the condition check since the Methoderror will need the ID's
   m_MSGDeAllocateMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
   m_MSGDeAllocateMessage.u16RegisterID = poMessage->u16GetRegisterID();
   m_MSGDeAllocateMessage.u16CmdCounter = poMessage->u16GetCmdCounter();
   m_MSGDeAllocateMessage.u16FunctionID = poMessage->u16GetFunctionID();

   // Create the (message related) FI data object
   most_msgfi_tclMsgDeAllocateMethodStart oDeAllocate_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(oDeAllocate_MS,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      // Go and Deallocate the channel
      MessagingTypes::getUcidFromSourceNr(oDeAllocate_MS.u8SourceNr, &ucid);

      if(ucid == INVALID_UCID_OR_SOURCENR)
      {
         // this->m_oAudioRecording->m_externalBeep=true;
         switch(oDeAllocate_MS.u8SourceNr)
         {
            case SOURCENR_NEW_MSG_ALERT_MESSAGING:
            case SOURCENR_DEVICE_MEMORY_FULL_MESSAGING:
            {
               ucid=UCID_InMsgAlertToneToMost;

               break;
            }

            case SOURCENR_BEEP_REC_MESSAGING:
            {
               ucid=UCID_BeepRecMsgToMost;
               break;
            }
            default:
            {
               ETG_TRACE_USR3(("Invalid SourceNr"));
            }
         }
      }
      if (!isSourceNrValid(oDeAllocate_MS.u8SourceNr))
      {
         // Invalid SourceNr is passed by HMI
         ETG_TRACE_ERR(("Invalid SourceNr is passed by HMI"));

         vSendDeAllocateMethodError(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE);
      }

      tU8 l_u8CurrentAllocateStatus(AllocateStatusState::getCurrentAllocateState(oDeAllocate_MS.u8SourceNr));

      if (l_u8CurrentAllocateStatus == MSG_CHANNEL_NOT_ALLOCATED )
      {
         // Return Method Result.
         ETG_TRACE_USR3(("Deallocate a not allocated SourceNr"));

         vSendDeAllocateMethodResult(ucid);
      }
   }
   else
   {
      // Paramter is missing
      vSendDeAllocateMethodError(most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH);
   }

   oDeAllocate_MS.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vHandle_SourceActivity_MethodStart(
        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_SourceActivity_MethodStart : Called"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_msgfi_tclMsgSourceActivityMethodStart p_start;
   midw_streamrouterfi_tclMsgSetPlayMethodStart setplayStart;
   midw_streamrouterfi_tclMsgSetStopMethodStart setStopStart;

   // BugFix GMMY15-3986 - Methods with missing parameters are not responded
	// Assign the member variables before the condition check since the Methoderror will need the ID's
   m_MSGSourceActivityMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
   m_MSGSourceActivityMessage.u16RegisterID  = poMessage->u16GetRegisterID();
   m_MSGSourceActivityMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
   m_MSGSourceActivityMessage.u16FunctionID  = poMessage->u16GetFunctionID();

   if (oVisitorMsg.s32GetData(p_start, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      if (!isSourceNrValid(p_start.u8SourceNr))
      {
         // wrong parameter
         ETG_TRACE_USR2(("wrong parameter::%d %d", p_start.u8SourceNr, p_start.e8Activity.enType));
         m_u8SourceActivityErrorParameterPosition = 2;
         vSendSrcActivityMethodError(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE);

         return;
      }

      if ((p_start.e8Activity.enType
            == most_fi_tcl_e8_SrcActivity::FI_EN_E8SA_RESTRICTED_DO_NOT_USE))
      {
         // wrong parameter
         ETG_TRACE_USR2(("wrong parameter::%d %d", p_start.u8SourceNr, p_start.e8Activity.enType));
         m_u8SourceActivityErrorParameterPosition = 3;
         vSendSrcActivityMethodError(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE);

         return;
      }

      ETG_TRACE_USR2(("SOURCE NR TO START SOURCE ACTIVITY =%d ", p_start.u8SourceNr));

      switch (p_start.e8Activity.enType)
      {
         case most_fi_tcl_e8_SrcActivity::FI_EN_E8SA_ON:
         {
            MessagingTypes::getUcidFromSourceNr(p_start.u8SourceNr, &setplayStart.u16UCID);

            ETG_TRACE_USR2(("CALLING STREAM ROUTER SET_PLAY UCID =%d", setplayStart.u16UCID));

            setplayStart.eRampType.enType = midw_fi_tcl_e8_streamrouter_ramptype::FI_EN_MEDIUM;
         }
         break;
         case most_fi_tcl_e8_SrcActivity::FI_EN_E8SA_OFF:
         {
            ETG_TRACE_USR2(("vHandle_SourceActivity_MethodStart: FI_EN_E8SA_OFF"));

            MessagingTypes::getUcidFromSourceNr(p_start.u8SourceNr, &setStopStart.u16UCID);

            ETG_TRACE_USR2(("CALLING STREAM ROUTER SET_STOP UCID =%d", setStopStart.u16UCID));

            setStopStart.eRampType.enType = midw_fi_tcl_e8_streamrouter_ramptype::FI_EN_MEDIUM;
         }
         break;

         default:
         {
            ETG_TRACE_USR4(("vHandle_SourceActivity_MethodStart: WRONG SOURCE ACTIVITY CODE RECEIVED"));
            m_u8SourceActivityErrorParameterPosition = 3;
            vSendSrcActivityMethodError(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE);
         }
      }
   }
   else
   {
      // Parameter is missing
      ETG_TRACE_ERR(("Parameter is missing"));
      vSendSrcActivityMethodError(most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH);
      ETG_TRACE_ERR(("vHandle_SourceActivity_MethodStart OSAL ERROR"));
   }
}

/*******************************************************************************
 ** FUNCTION:   vSendSrcActivityMethodResultOn
 *******************************************************************************/
/* DESCRIPTION:
 *   Send This method sends Source Activity method result set to ON.
 *
 *  PARAMETERS:
 *          IN :None
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vSendSrcActivityMethodResultOn(
        tU16 ucid, tU16 cmdCounter)
{
    
    ETG_TRACE_USR4(("vSendSrcActivityMethodResultOn CALLED ucid =%d , cmdCounter = %d",ucid,  cmdCounter));
    most_msgfi_tclMsgSourceActivityMethodResult oSourceActivityMethodResult;

    bool error = !MessagingTypes::getSourceNrFromUcid(ucid,
                                                      &oSourceActivityMethodResult.u8SourceNr);

    //ETG_TRACE_USR4((" fc_messaging_tclService_Messaging::vSendSrcActivityMethodResultOn UCID: = %d", ucid));ETG_TRACE_USR4((" fc_messaging_tclService_Messaging::vSendSrcActivityMethodResultOn SOURCE NR: = %u", oSourceActivityMethodResult.u8SourceNr ));

    if (error == false)
    {
        oSourceActivityMethodResult.e8Activity.enType
                = most_fi_tcl_e8_SrcActivity::FI_EN_E8SA_ON;

        fi_tclVisitorMessage oMRMsg(oSourceActivityMethodResult,
                                    MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

        m_MSGSourceActivityMessage.u16CmdCounter = cmdCounter;

        vInitServiceData(oMRMsg, // ServiceDataMsg
                         m_MSGSourceActivityMessage.u16SourceAppID, // Target AppID
                         m_MSGSourceActivityMessage.u16RegisterID, // RegisterID
                         m_MSGSourceActivityMessage.u16CmdCounter, // CmdCounter,
                         m_MSGSourceActivityMessage.u16FunctionID, // Function ID
                         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                         );

        // POST THE MESSAGE
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            // Error situation:
            // oMsg1.bDelete();   // A '.bDelete()' is not necessary here as the
            // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care of this
            // implicitely.
            ETG_TRACE_ERR(
                        (" enPostMessage() failed for 'Allocate - METHODRESULT' message, 'ail_tenCommunicationError' = %u", (tU32) enResult));
            ETG_TRACE_ERR(
                        (" vSendSrcActivityMethodResultOn::AllocateRouteMethodResult "));
        }
    }
    else
    {
        error = true;
        ETG_TRACE_ERR((" wrong UCID : %d ,sending Method Error", ucid));
        ETG_TRACE_ERR(
                    ("vSendSrcActivityMethodResultOn: INVALID SOURCE NR : sending Method Error"));
        vSendSrcActivityMethodError(
                    most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);
    }

    oSourceActivityMethodResult.vDestroy();

}
/*******************************************************************************
 ** FUNCTION:   vSendSrcActivityMethodResultOff
 *******************************************************************************/
/* DESCRIPTION:
 *   Send This method sends Source Activity method result set to Off.
 *
 *  PARAMETERS:
 *          IN :None
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vSendSrcActivityMethodResultOff(
        tU16 ucid, tU16 cmdCounter)
{
    ETG_TRACE_USR4(("vSendSrcActivityMethodResultOff:CALLED ucid =%d , cmdCounter = %d", ucid,  cmdCounter));
    
    most_msgfi_tclMsgSourceActivityMethodResult oSourceActivityMethodResult;

    bool error = !MessagingTypes::getSourceNrFromUcid(ucid,
                                                      &oSourceActivityMethodResult.u8SourceNr);

    //ETG_TRACE_USR4((" fc_messaging_tclService_Messaging::vSendSrcActivityMethodResultOff UCID: = %d", ucid));ETG_TRACE_USR4((" fc_messaging_tclService_Messaging::vSendSrcActivityMethodResultOff SOURCE NR: = %u", oSourceActivityMethodResult.u8SourceNr ));

    if (error == false)
    {
        oSourceActivityMethodResult.e8Activity.enType
                = most_fi_tcl_e8_SrcActivity::FI_EN_E8SA_OFF;

        fi_tclVisitorMessage oMRMsg(oSourceActivityMethodResult,
                                    MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

        m_MSGSourceActivityMessage.u16CmdCounter = cmdCounter;

        vInitServiceData(oMRMsg, // ServiceDataMsg
                         m_MSGSourceActivityMessage.u16SourceAppID, // Target AppID
                         m_MSGSourceActivityMessage.u16RegisterID, // RegisterID
                         m_MSGSourceActivityMessage.u16CmdCounter, // CmdCounter,
                         m_MSGSourceActivityMessage.u16FunctionID, // Function ID
                         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                         );

        // POST THE MESSAGE
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            // Error situation:
            // oMsg1.bDelete();   // A '.bDelete()' is not necessary here as the
            // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care of this
            // implicitely.
            ETG_TRACE_ERR(
                        (" vSendSrcActivityMethodResultOff:enPostMessage() failed for 'Allocate - METHODRESULT' message, 'ail_tenCommunicationError' = %u", (tU32) enResult));
            ETG_TRACE_ERR(
                        (" vSendSrcActivityMethodResultOff. AllocateRouteMethodResult "));
        }
    }
    else
    {
        error = true;
        ETG_TRACE_ERR(("vSendSrcActivityMethodResultOff. InvalidSourceNr"));
        ETG_TRACE_ERR((" wrong UCID : %d ,sending Method Error", ucid));
        vSendSrcActivityMethodError(
                    most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);
    }

    oSourceActivityMethodResult.vDestroy();

}
/*******************************************************************************
 ** FUNCTION:   vSendSrcActivityMethodError
 *******************************************************************************/
/* DESCRIPTION:
 *   Send Allocate method error.
 *
 *  PARAMETERS:
 *          IN :None
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vSendSrcActivityMethodError(
        most_fi_tcl_e8_ErrorCode::tenType eErrorCode)
{

    ETG_TRACE_USR4(("vSendSrcActivityMethodError : Called"));

    most_msgfi_tclMsgSourceActivityError oSourceActivity_MErr;

    tU8 u8PositionOfErroneousParameter = 0x02;
    tU8 au8ErrorInfo[1] = {0x00};

    switch (eErrorCode)
    {
      //Only these 4 error cases are set during SourceActivityMethod processing.
    default:
    case most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE:
    {
      au8ErrorInfo[0] = m_u8SourceActivityErrorParameterPosition;
      u8PositionOfErroneousParameter = m_u8SourceActivityErrorParameterPosition;
    }
    break;

      case most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH:
      {
        au8ErrorInfo[0] = 0x00;//== Dummy zero (from document most_MPlay_fi)
        u8PositionOfErroneousParameter = 0x00;
      }
      break;

      case most_fi_tcl_e8_ErrorCode::FI_EN_BUSY:
      {
        au8ErrorInfo[0] = m_u8SourceActivityErrorParameterPosition;//== Return Parameter (from document most_MPlay_fi)
        u8PositionOfErroneousParameter = m_u8SourceActivityErrorParameterPosition;
      }
      break;

      case most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC:
      {
        au8ErrorInfo[0] = 0x04;//== Value Not Available (from document most_MPlay_fi)
        u8PositionOfErroneousParameter = 0x04;
      }
      break;

    case most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE:
    {
        au8ErrorInfo[0] = 0x00;
    }
        break;

    }
    
	/* BugFix GMMY15-3986 - Methods with missing parameters are not responded 
	   Assign only the error code and bypass other extra information */
	if(eErrorCode != most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH)
	{
       fi_tclOutContext oOut(au8ErrorInfo, au8ErrorInfo + sizeof(au8ErrorInfo),
                          MOST_MSGFI_C_U16_SERVICE_MAJORVERSION,
                          oSourceActivity_MErr.enGetByteOrder());

       // write values to buffer
       (tVoid)(oOut << u8PositionOfErroneousParameter);

       oSourceActivity_MErr.oErrorInfo.vSetData(au8ErrorInfo, sizeof(au8ErrorInfo));
	}

	oSourceActivity_MErr.e8ErrorCode.enType = eErrorCode;

    fi_tclVisitorMessage oMEMsg(oSourceActivity_MErr,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMEMsg, // ServiceDataMsg
                     m_MSGSourceActivityMessage.u16SourceAppID, // Target AppID
                     m_MSGSourceActivityMessage.u16RegisterID, // RegisterID
                     m_MSGSourceActivityMessage.u16CmdCounter, // CmdCounter,
                     m_MSGSourceActivityMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    // POST THE MESSAGE
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMEMsg,
                                                                    TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        ETG_TRACE_ERR(
                    ("vSendSrcActivityMethodError: oSourceActivity_MErr Sending Fail "));
    }

    oSourceActivity_MErr.vDestroy();

}

tVoid fc_messaging_tclService_Messaging::vHandle_SendMessageValet_MethodStart()
{
   ETG_TRACE_USR4(("vHandle_SendMessageValet_MethodStart : Called"));
   SendMessage* l_poSendMessage;
   l_poSendMessage = SendMessage::poGetSendMessageInstance();

   if (l_poSendMessage)
   {
      if(m_l_poSendMessageMethodStart)
      {
         l_poSendMessage->vSendMessage(*m_l_poSendMessageMethodStart);
      }
   }

   if(m_l_poSendMessageMethodStart)
   {
      m_l_poSendMessageMethodStart->vDestroy();
      m_l_poSendMessageMethodStart = NULLPTR;
   }
   if(m_l_poSendMessageMethodStart)
   {
      delete m_l_poSendMessageMethodStart;
      m_l_poSendMessageMethodStart = NULLPTR;
   }
   if(m_l_prJob)
   {
      delete m_l_prJob;
      m_l_prJob = NULLPTR;
   }
}

/*******************************************************************************
 ** FUNCTION:   vHandle_SendMessage_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for SendMessage method start.
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_SendMessage_MethodStart(
        amt_tclServiceData* poMessage, struct Job* f_prJob)
{
   ETG_TRACE_USR4(("vHandle_SendMessage_MethodStart() called"));

   struct Job* l_prJob = NULLPTR;
   most_msgfi_tclMsgSendMessageMethodStart* l_poSendMessageMethodStart = NULLPTR;

   if(poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);

      l_poSendMessageMethodStart = new most_msgfi_tclMsgSendMessageMethodStart;

      if(l_poSendMessageMethodStart)
      {
         if (oVisitorMsg.s32GetData(*l_poSendMessageMethodStart,
               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
         {
            vHandleGetDataError(poMessage, "vHandle_SendMessage_MethodStart");
            delete l_poSendMessageMethodStart;
            return;
         }
         ETG_TRACE_USR1((" Folder Type:: %u",ETG_ENUM(TR_FOLDER_TYPE,l_poSendMessageMethodStart->e8FolderType.enType)));
         SendMessage* l_poSendMessage = SendMessage::poGetSendMessageInstance();
         tU8 u8DeviceHandle = 0;
         if(l_poSendMessage)
         {
        	 u8DeviceHandle = l_poSendMessage->vGetDeviceHandle();
         }
         if ((u8DeviceHandle>= MINIMUM_DEVICE_HANDLE) && (u8DeviceHandle<= MAXIMUM_DEVICE_HANDLE))
         {
            l_prJob = new Job;
            if(l_prJob)
            {
               l_prJob->m_pfMethod  = &fc_messaging_tclService_Messaging::vHandle_SendMessage_MethodStart;
               l_prJob->u16CmdCounter = poMessage->u16GetCmdCounter();
               l_prJob->u16FunctionID = poMessage->u16GetFunctionID();
               l_prJob->u16RegisterID = poMessage->u16GetRegisterID();
               l_prJob->u16SourceAppID = poMessage->u16GetSourceAppID();
               l_prJob->m_pvMethodStart = l_poSendMessageMethodStart;
            }
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_SendMessage_MethodStart : Invalid Device Handle"));
            vHandleInvalidDataError(poMessage);
            delete l_poSendMessageMethodStart;
            return;
         }
      }
   }
   else if(f_prJob)
   {
      l_prJob = f_prJob;
   }
   else
   {
      return; // Should not come here
   }

   if (MessagingMutex::messagingMutexTrylock(l_prJob) != 0)
   {
      tVoid*  pCustodialLose = l_prJob ;  //To remove LINT warning : Custodial Pointer
      (tVoid) pCustodialLose ;
      return; // Didn't get the lock :(
   }
   if(l_prJob)
   {
      m_oSendMessage.u16SourceAppID = l_prJob->u16SourceAppID;
      m_oSendMessage.u16RegisterID = l_prJob->u16RegisterID;
      m_oSendMessage.u16CmdCounter = l_prJob->u16CmdCounter;
      m_oSendMessage.u16FunctionID = l_prJob->u16FunctionID;

      if(l_poSendMessageMethodStart)
      {
         //To remove LINT warning
         tVoid*  pCustodialLose = l_poSendMessageMethodStart;
         (tVoid) pCustodialLose;
      }

      l_poSendMessageMethodStart = (most_msgfi_tclMsgSendMessageMethodStart*) l_prJob->m_pvMethodStart;
      m_l_poSendMessageMethodStart = l_poSendMessageMethodStart;
      m_l_prJob = l_prJob;
   }

   vHandle_SendMessageValet_MethodStart();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_SendMessage_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of SendMessage to client.
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_SendMessage_Return(
        most_msgfi_tclMsgSendMessageMethodResult &poSendMessageMethodResult_MR)
{
    ETG_TRACE_USR4(("vHandle_SendMessage_Return : Called"));

    most_msgfi_tclMsgSendMessageMethodResult oSendMessageMethodResult_MR;
    oSendMessageMethodResult_MR = poSendMessageMethodResult_MR;

    fi_tclVisitorMessage oMsg(oSendMessageMethodResult_MR,
                              MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMsg, m_oSendMessage.u16SourceAppID, // Target ID
                     m_oSendMessage.u16RegisterID, // RegisterID
                     m_oSendMessage.u16CmdCounter, // CmdCounter,
                     m_oSendMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMsg.bIsValid())
    {
    	ETG_TRACE_USR1((" Sent Message Status:: %u",ETG_ENUM(TR_SENTMESSAGESTATUS,oSendMessageMethodResult_MR.e8SentMessageStatus.enType)));
    	// Post Message
        if (_poMainAppl->enPostMessage(&oMsg, TRUE) != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("fc_messaging_tclService_Messaging::vHandle_SendMessage_Return: Message Post failed"));
        }
    }
    else
    {
        ETG_TRACE_ERR(
                    ("fc_messaging_tclService_Messaging::vHandle_SendMessage_Return: Invaild Message"));
    }

    // unLock the mutex first
    MessagingMutex::messagingMutexUnlock();
    if(oSendMessageMethodResult_MR.e8SentMessageStatus.enType == most_fi_tcl_e8_MsgSentMessageStatus::FI_EN_E8MSG_SENT_STATUS_SUCCESS)
    {
       if(TRUE == fc_messaging_tclApp::m_poMessaging->bStartTimer(static_cast<tU16>(
              FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID + SendMessage::poGetSendMessageInstance()->vGetDeviceHandle()),
              FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_TIMEOUT,
              FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_TIMEOUT))
       {

          ETG_TRACE_ERR(("The timer FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID is started"));
       }
    }
    oSendMessageMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_SendMessage_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of SendMessage to client.
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_SendMessage_Error()
{
    ETG_TRACE_USR4(("vHandle_SendMessage_Error( : Called"));
    most_msgfi_tclMsgSendMessageError oSendMessageError_ER;

    oSendMessageError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(oSendMessageError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oSendMessage.u16SourceAppID, // ServiceDataMsg
                     m_oSendMessage.u16RegisterID, // RegisterID
                     m_oSendMessage.u16CmdCounter, // CmdCounter,
                     m_oSendMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(("vHandle_SendMessage_Error: enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    MessagingMutex::messagingMutexUnlock();

    oSendMessageError_ER.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestFolderList_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for RequestFolderList method start
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestFolderList_MethodStart(
        amt_tclServiceData* poMessage, struct Job* f_prJob)
{
   ETG_TRACE_USR4(("vHandle_RequestFolderList_MethodStart : Called"));

   // Bugfix for GMMY16-7568 and GMMY16-7536: Nullpointerexception in
   // /opt/bosch/gm/processes/fc_messaging_out.out in thread AE_60

   struct Job* prJob = NULLPTR;
   most_msgfi_tclMsgRequestFolderListMethodStart* poRequestFolderListMethodStart = NULLPTR;

   if (poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);
      poRequestFolderListMethodStart = new most_msgfi_tclMsgRequestFolderListMethodStart;

      if (poRequestFolderListMethodStart)
      {
         if (oVisitorMsg.s32GetData(*poRequestFolderListMethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
         {
            vHandleGetDataError(poMessage, "vHandle_RequestFolderList_MethodStart");
            delete poRequestFolderListMethodStart;
            return;
         }

         prJob = new Job;

         if (prJob)
         {
            prJob->m_pfMethod = &fc_messaging_tclService_Messaging::vHandle_RequestFolderList_MethodStart;
            prJob->u8DeviceHandle = 0; //TODO  retrieve deviceHandle
            prJob->u16CmdCounter = poMessage->u16GetCmdCounter();
            prJob->u16FunctionID = poMessage->u16GetFunctionID();
            prJob->u16RegisterID = poMessage->u16GetRegisterID();
            prJob->u16SourceAppID = poMessage->u16GetSourceAppID();
            prJob->m_pvMethodStart = poRequestFolderListMethodStart;
         }
      }
   }
   else if (f_prJob)
   {
      //Request(from queue) is processed after getting mutex lock
      prJob = f_prJob;
   }
   else
   {
      return; //Should not come here
   }

   if (MessagingMutex::messagingMutexTrylock(prJob) != 0)
   {
      tVoid* pCustodialLose = prJob; //To remove LINT warning : Custodial Pointer
      (tVoid) pCustodialLose;
      return; // Didn't get the lock :(
   }

   if (prJob)
   {
      m_oRequestFolderList.u16SourceAppID = prJob->u16SourceAppID;
      m_oRequestFolderList.u16RegisterID = prJob->u16RegisterID;
      m_oRequestFolderList.u16CmdCounter = prJob->u16CmdCounter;
      m_oRequestFolderList.u16FunctionID = prJob->u16FunctionID;

      if (poRequestFolderListMethodStart) //LINT
      {
         tVoid* pCustodialLose = poRequestFolderListMethodStart; //To remove LINT warning
         (tVoid) pCustodialLose;
      }
      poRequestFolderListMethodStart = (most_msgfi_tclMsgRequestFolderListMethodStart*) prJob->m_pvMethodStart;
   }

   if (poRequestFolderListMethodStart)
   {
      QString serviceName;

      tU8 deviceHandle = DbusClientInterface::getInstance().getCurrentOutgoingDeviceId();
      serviceName = DbusClientInterface::getInstance().getCurrentActiveService(deviceHandle, most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_EMAIL);
	  
      ETG_TRACE_USR2(("Current Active Service Name::%s", serviceName.toUtf8().constData()));
	  
      FileSystemManager* poFileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(deviceHandle, serviceName);

      if (poFileSystemManager)
      {
         poFileSystemManager->vRequestFolderList(poRequestFolderListMethodStart->u32FolderHandle, poRequestFolderListMethodStart->bRootFlag);
      }
      else
      {
         vHandle_RequestFolderList_Error();
      }

      poRequestFolderListMethodStart->vDestroy();
      delete poRequestFolderListMethodStart;
   }
   delete prJob;
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestFolderList_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of RequestFolderList to client.
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestFolderList_Return(
        most_msgfi_tclMsgRequestFolderListMethodResult& f_rRequestFolderListMethodResult_MR)
{
    ETG_TRACE_USR4(("vHandle_RequestFolderList_Return : Called"));
    tU8 l_u8Index = 0;
    (void)l_u8Index;

    //	for (l_u8Index = 0; l_u8Index
    //			< f_rRequestFolderListMethodResult_MR.oFolderListResult.oItems.size(); l_u8Index++)
    //	{
    //		MSG_DEBUG("Folder Name::%s",f_rRequestFolderListMethodResult_MR.oFolderListResult.oItems[l_u8Index].sFolderName.szValue);
    //		MSG_DEBUG("Folder Handle::%u",f_rRequestFolderListMethodResult_MR.oFolderListResult.oItems[l_u8Index].u32FolderHandle);
    //		MSG_DEBUG("SubFolders::%u",f_rRequestFolderListMethodResult_MR.oFolderListResult.oItems[l_u8Index].u8NumSubfolders);
    //		MSG_DEBUG("No of message::%u",f_rRequestFolderListMethodResult_MR.oFolderListResult.oItems[l_u8Index].u16NumMessages);
    //		MSG_DEBUG("No of unread messages::%u",f_rRequestFolderListMethodResult_MR.oFolderListResult.oItems[l_u8Index].u16NumUnreadMessages);
    //
    //	}

    fi_tclVisitorMessage oMsg(f_rRequestFolderListMethodResult_MR,
                              MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMsg, m_oRequestFolderList.u16SourceAppID,
                     m_oRequestFolderList.u16RegisterID,
                     m_oRequestFolderList.u16CmdCounter,
                     m_oRequestFolderList.u16FunctionID,
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);

    if (oMsg.bIsValid())
    {
        // Post Message
        if (_poMainAppl->enPostMessage(&oMsg, TRUE) != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_RequestFolderList_Return: Message Post failed"));
        }
    }
    else
    {
        ETG_TRACE_ERR(("vHandle_RequestFolderList_Return: Invalid Message"));
    }

    MessagingMutex::messagingMutexUnlock();

    f_rRequestFolderListMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestFolderList_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of RequestFolderList to client.
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestFolderList_Error()
{
    ETG_TRACE_USR4(("vHandle_RequestFolderList_Error : Called"));

    most_msgfi_tclMsgRequestFolderListError l_oMsgRequestFolderListError_ER;

    l_oMsgRequestFolderListError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(l_oMsgRequestFolderListError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oRequestFolderList.u16SourceAppID, // ServiceDataMsg
                     m_oRequestFolderList.u16RegisterID, // RegisterID
                     m_oRequestFolderList.u16CmdCounter, // CmdCounter,
                     m_oRequestFolderList.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_RequestFolderList_Error: enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }
    else
    {
        ETG_TRACE_ERR(("vHandle_RequestFolderList_Error: Invalid Message"));
    }

    MessagingMutex::messagingMutexUnlock();

    l_oMsgRequestFolderListError_ER.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_AddPredefinedMessage_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for Add predefined Message method
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddPredefinedMessage_MethodStart(
      amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_AddPredefinedMessage_MethodStart() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgAddPredefinedMessageMethodStart oAddPredefinedMessage_MethodStart;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(oAddPredefinedMessage_MethodStart,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      m_oAddpredefinedMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oAddpredefinedMessage.u16RegisterID  = poMessage->u16GetRegisterID();
      m_oAddpredefinedMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
      m_oAddpredefinedMessage.u16FunctionID  = poMessage->u16GetFunctionID();

      tU16 u16Handle = 0;
      tBool bSuccess = FALSE;
      MessagingSettings* l_poMessageSettings;
      FC_Messaging_PredefinedMsgList* l_poPredefinedMsgList;

      l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();

      if(l_poMessageSettings)
      {
         l_poPredefinedMsgList = l_poMessageSettings->poGetPredefinedMessageListInstance();
         if(l_poPredefinedMsgList)
         {
            bSuccess = l_poPredefinedMsgList->bAddPredefinedMessage(
                  0, //DeviceHandle is 0 -> SystemWideSettings
                  oAddPredefinedMessage_MethodStart.sPredefinedMessageText,
                  most_fi_tcl_e8_MsgPredefinedMessageType::FI_EN_E8PRE_MSG_STATIC,
                  FC_MSG_USER_PRE_DEF_MSG, //SourceType
                  u16Handle, FC_MSG_NO_SUB_CATEGORY);
         }
      }

      if (bSuccess == TRUE)
      {
         vHandle_AddPredefinedMessage_Return(u16Handle);
      }
      else
      {
         vHandle_AddPredefinedMessage_Error();
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_AddPredefinedMessage_MethodStart");
   }

   oAddPredefinedMessage_MethodStart.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_DeletePredefinedMessage_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for Delete predefined Message method
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_DeletePredefinedMessage_MethodStart(
      amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_DeletePredefinedMessage_MethodStart : Called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgDeletePredefinedMessageMethodStart oDeletePredefinedMessage_MethodStart;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(oDeletePredefinedMessage_MethodStart,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      m_oDeletepredefinedMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oDeletepredefinedMessage.u16RegisterID  = poMessage->u16GetRegisterID();
      m_oDeletepredefinedMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
      m_oDeletepredefinedMessage.u16FunctionID  = poMessage->u16GetFunctionID();

      tBool bSuccess = FALSE;
      MessagingSettings* l_poMessageSettings;
      FC_Messaging_PredefinedMsgList* l_poPredefinedMsgList;

      l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();

      if(l_poMessageSettings)
      {
         l_poPredefinedMsgList = l_poMessageSettings->poGetPredefinedMessageListInstance();
         if(l_poPredefinedMsgList)
         {
            bSuccess = l_poPredefinedMsgList->bDeletePredefinedMessage(
                        oDeletePredefinedMessage_MethodStart.u16PredefinedMessageHandle);
         }
      }

      if (bSuccess == TRUE)
      {
         vHandle_DeletePredefinedMessage_Return(
               oDeletePredefinedMessage_MethodStart.u16PredefinedMessageHandle);
      }
      else
      {
         vHandle_DeletePredefinedMessage_Error();
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_DeletePredefinedMessage_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving its creation scope)
   oDeletePredefinedMessage_MethodStart.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_EditPredefinedMessage_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_EditPredefinedMessage_MethodStart method
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_EditPredefinedMessage_MethodStart(
      amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_EditPredefinedMessage_MethodStart() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgEditPredefinedMessageMethodStart oEditPredefinedMessage_MethodStart;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(oEditPredefinedMessage_MethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("u16PredefinedMessageHandle : %d", oEditPredefinedMessage_MethodStart.u16PredefinedMessageHandle));
      ETG_TRACE_USR4(("sPredefinedMessageText : %s", oEditPredefinedMessage_MethodStart.sPredefinedMessageText.szValue));

      m_oEditPredefinedMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oEditPredefinedMessage.u16RegisterID = poMessage->u16GetRegisterID();
      m_oEditPredefinedMessage.u16CmdCounter = poMessage->u16GetCmdCounter();
      m_oEditPredefinedMessage.u16FunctionID = poMessage->u16GetFunctionID();

      tBool bSuccess = FALSE;
      MessagingSettings* l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();

      if (l_poMessageSettings)
      {
         FC_Messaging_PredefinedMsgList* l_poPredefinedMsgList =
               l_poMessageSettings->poGetPredefinedMessageListInstance();
         if (l_poPredefinedMsgList)
         {
            tBool bIsValidHandle = l_poPredefinedMsgList->bIsValidPredefMsgHandle(
                  oEditPredefinedMessage_MethodStart.u16PredefinedMessageHandle);

            ETG_TRACE_USR4(("bIsValidPredefMsgHandle : %d", bIsValidHandle));

            if (TRUE == bIsValidHandle)
            {
               bSuccess = l_poPredefinedMsgList->bEditPredefinedMessage(
                     oEditPredefinedMessage_MethodStart.u16PredefinedMessageHandle,
                     oEditPredefinedMessage_MethodStart.sPredefinedMessageText,
                     oEditPredefinedMessage_MethodStart.e8PredefinedMessageSubCategory.enType);
            }
         }
      }

      if (bSuccess == TRUE)
      {
         vHandle_EditPredefinedMessage_Return(oEditPredefinedMessage_MethodStart.u16PredefinedMessageHandle);
      }
      else
      {
         vHandle_EditPredefinedMessage_Error();
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_EditPredefinedMessage_MethodStart");
   }

   oEditPredefinedMessage_MethodStart.vDestroy();
}

//CMG3G-8730 PSA - Predefined Message Template handling Extension
/*******************************************************************************
 ** FUNCTION:   vHandle_AddPredefinedMessageExtended_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_AddPredefinedMessageExtended_MethodStart
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddPredefinedMessageExtended_MethodStart(
      amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_AddPredefinedMessageExtended_MethodStart() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgAddPredefinedMessageExtendedMethodStart oAddPredefinedMessageExtended_MethodStart;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.

   if (oVisitorMsg.s32GetData(oAddPredefinedMessageExtended_MethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("e8PredefinedMessageSubCategory : %d", oAddPredefinedMessageExtended_MethodStart.e8PredefinedMessageSubCategory.enType));
      ETG_TRACE_USR4(("sPredefinedMessageText : %s", oAddPredefinedMessageExtended_MethodStart.sPredefinedMessageText.szValue));

      m_oAddPredefinedMessageExtended.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oAddPredefinedMessageExtended.u16RegisterID  = poMessage->u16GetRegisterID();
      m_oAddPredefinedMessageExtended.u16CmdCounter  = poMessage->u16GetCmdCounter();
      m_oAddPredefinedMessageExtended.u16FunctionID  = poMessage->u16GetFunctionID();

      if (!((oAddPredefinedMessageExtended_MethodStart.e8PredefinedMessageSubCategory.enType == 0) ||
           ((oAddPredefinedMessageExtended_MethodStart.e8PredefinedMessageSubCategory.enType >= 1) &&
            (oAddPredefinedMessageExtended_MethodStart.e8PredefinedMessageSubCategory.enType <= 4))))
      {
         //send method error
         ETG_TRACE_USR4(("The input parameter e8PredefinedMessageSubCategory is not in range, Sending MethodError"));
         vHandle_AddPredefinedMessageExtended_Error();
      }
      else
      {
         tU16 u16PredefMsgHandle = 0;
         tBool bSuccess = FALSE;
         MessagingSettings* l_poMessageSettings;
         FC_Messaging_PredefinedMsgList* l_poPredefinedMsgList;

         l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();

         if (l_poMessageSettings)
         {
            l_poPredefinedMsgList = l_poMessageSettings->poGetPredefinedMessageListInstance();
            if (l_poPredefinedMsgList)
            {
               bSuccess = l_poPredefinedMsgList->bAddPredefinedMessage(
                     0, //DeviceHandle is 0 -> SystemWideSettings
                     oAddPredefinedMessageExtended_MethodStart.sPredefinedMessageText,
                     most_fi_tcl_e8_MsgPredefinedMessageType::FI_EN_E8PRE_MSG_STATIC,
                     FC_MSG_USER_PRE_DEF_MSG, //SourceType
                     u16PredefMsgHandle,
                     oAddPredefinedMessageExtended_MethodStart.e8PredefinedMessageSubCategory.enType);
            }
         }

         if (bSuccess == TRUE)
         {
            ETG_TRACE_USR4(("u16PredefMsgHandle to be returned:: %d", u16PredefMsgHandle));
            vHandle_AddPredefinedMessageExtended_Return(u16PredefMsgHandle);
         }
         else
         {
            //send method error
            vHandle_AddPredefinedMessageExtended_Error();
         }
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_AddPredefinedMessageExtended_MethodStart");
   }

   oAddPredefinedMessageExtended_MethodStart.vDestroy();
}
//CMG3G-8730 PSA - Predefined Message Template handling Extension
/*******************************************************************************
** FUNCTION:   vHandle_AddPredefinedMessageExtended_Return(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_AddPredefinedMessageExtended_MethodResult
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddPredefinedMessageExtended_Return(tU16 u16Handle)
{

   ETG_TRACE_USR4(("vHandle_AddPredefinedMessageExtended_Return : Called  Handle =%d", u16Handle));
   most_msgfi_tclMsgAddPredefinedMessageExtendedMethodResult oMethodResultObj;
   oMethodResultObj.u16PredefinedMessageHandle = u16Handle;

   /* Get the coresponding Call Instance */
   fi_tclVisitorMessage
         oMethodResult(oMethodResultObj, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData(oMethodResult, m_oAddPredefinedMessageExtended.u16SourceAppID, // ServiceDataMsg
   m_oAddPredefinedMessageExtended.u16RegisterID, // RegisterID
   m_oAddPredefinedMessageExtended.u16CmdCounter, // CmdCounter,
   m_oAddPredefinedMessageExtended.u16FunctionID, // Function ID
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
   );

   if (oMethodResult.bIsValid())
   {
      // Post Message
      if (_poMainAppl->enPostMessage(&oMethodResult, TRUE) != AIL_EN_N_NO_ERROR)
      {

         ETG_TRACE_ERR(("fc_messaging_tclService_Messaging::vHandle_AddPredefinedMessageExtended_Return: Message Post failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("fc_messaging_tclService_Messaging::vHandle_AddPredefinedMessageExtended_Return: Invaild Message"));
   }

   oMethodResultObj.vDestroy();

}
//CMG3G-8730 PSA - Predefined Message Template handling Extension
/*******************************************************************************
** FUNCTION:   vHandle_AddPredefinedMessageExtended_Error(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_AddPredefinedMessageExtended_Error
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddPredefinedMessageExtended_Error()
{
   most_msgfi_tclMsgAddPredefinedMessageExtendedError
         oMsgAddPredefMsgExtError_ER;

   oMsgAddPredefMsgExtError_ER.e8ErrorCode.enType
         = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage
         oMRMsg(oMsgAddPredefMsgExtError_ER, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg, m_oAddPredefinedMessageExtended.u16SourceAppID, // ServiceDataMsg
   m_oAddPredefinedMessageExtended.u16RegisterID, // RegisterID
   m_oAddPredefinedMessageExtended.u16CmdCounter, // CmdCounter,
   m_oAddPredefinedMessageExtended.u16FunctionID, // Function ID,
   AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult =
            _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_AddPredefinedMessageExtended_Error: enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
   oMsgAddPredefMsgExtError_ER.vDestroy();
}
/*******************************************************************************
 ** FUNCTION:   vHandle_AddPredefinedMessage_Return(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_AddPredefinedMessage_Return method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddPredefinedMessage_Return(
        tU16 u16Handle)
{
   ETG_TRACE_USR4(("vHandle_AddPredefinedMessage_Return : Called  Handle =%d", u16Handle));
   most_msgfi_tclMsgAddPredefinedMessageMethodResult oMethodResultObj;
   oMethodResultObj.u16PredefinedMessageHandle = u16Handle;

   // Get the coresponding Call Instance
   fi_tclVisitorMessage oMethodResult(oMethodResultObj,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData(oMethodResult,
         m_oAddpredefinedMessage.u16SourceAppID, // ServiceDataMsg
         m_oAddpredefinedMessage.u16RegisterID,  // RegisterID
         m_oAddpredefinedMessage.u16CmdCounter,  // CmdCounter
         m_oAddpredefinedMessage.u16FunctionID,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT     // Opcode
   );

   if (oMethodResult.bIsValid())
   {
      // Post Message
      if (_poMainAppl->enPostMessage(&oMethodResult, TRUE) != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_AddPredefinedMessage_Return: enPostMessage() failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("vHandle_AddPredefinedMessage_Return: Invaild Message"));
   }

   oMethodResultObj.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_AddPredefinedMessage_Error()
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_AddPredefinedMessage_Error
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddPredefinedMessage_Error()
{
   ETG_TRACE_USR4(("vHandle_AddPredefinedMessage_Error() called"));

   most_msgfi_tclMsgAddPredefinedMessageError oAddPredefinedMessageMethodError;

   oAddPredefinedMessageMethodError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(oAddPredefinedMessageMethodError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oAddpredefinedMessage.u16SourceAppID, // ServiceDataMsg
         m_oAddpredefinedMessage.u16RegisterID,  // RegisterID
         m_oAddpredefinedMessage.u16CmdCounter,  // CmdCounter
         m_oAddpredefinedMessage.u16FunctionID,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR            // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_AddPredefinedMessage_Error: enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oAddPredefinedMessageMethodError.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_DeletePredefinedMessage_Return(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for DeletePredefinedMessage_Return method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_DeletePredefinedMessage_Return(
        tU16 u16Handle)
{
    ETG_TRACE_USR4(("vHandle_DeletePredefinedMessage_Return : Called Handle =%d",u16Handle));

    most_msgfi_tclMsgDeletePredefinedMessageMethodResult oMethodResultObj;
    oMethodResultObj.u16PredefinedMessageHandle = u16Handle;

    // Get the coresponding Call Instance
    fi_tclVisitorMessage oMethodResult(oMethodResultObj,
                                       MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    // Here the message related header information is added and by this the
    // creation of the message is completed.
    vInitServiceData(oMethodResult,
          m_oDeletepredefinedMessage.u16SourceAppID, // ServiceDataMsg
          m_oDeletepredefinedMessage.u16RegisterID,  // RegisterID
          m_oDeletepredefinedMessage.u16CmdCounter,  // CmdCounter
          m_oDeletepredefinedMessage.u16FunctionID,  // Function ID
          AMT_C_U8_CCAMSG_OPCODE_METHODRESULT        // Opcode
    );

    if (oMethodResult.bIsValid())
    {
       // Post Message
       if (_poMainAppl->enPostMessage(&oMethodResult, TRUE) != AIL_EN_N_NO_ERROR)
       {
          ETG_TRACE_ERR(("vHandle_DeletePredefinedMessage_Return: enPostMessage() failed"));
       }
    }
    else
    {
        ETG_TRACE_ERR(("vHandle_DeletePredefinedMessage_Return: Invaild Message"));
    }

    oMethodResultObj.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_DeletePredefinedMessage_Error()
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_DeletePredefinedMessage_Error
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_DeletePredefinedMessage_Error()
{
   ETG_TRACE_USR4(("vHandle_DeletePredefinedMessage_Error() called"));

   most_msgfi_tclMsgDeletePredefinedMessageError oDeletePredefinedMessageMethodError;

   oDeletePredefinedMessageMethodError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(oDeletePredefinedMessageMethodError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oDeletepredefinedMessage.u16SourceAppID, // ServiceDataMsg
         m_oDeletepredefinedMessage.u16RegisterID,  // RegisterID
         m_oDeletepredefinedMessage.u16CmdCounter,  // CmdCounter
         m_oDeletepredefinedMessage.u16FunctionID,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR               // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_DeletePredefinedMessage_Error: enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oDeletePredefinedMessageMethodError.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_EditPredefinedMessage_Return(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for EditPredefinedMessage_Return method
 *
 *  PARAMETERS:
 *     IN:PredefinedMessageHandle - Handle of the predefined message to be edited
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_EditPredefinedMessage_Return(tU16 u16Handle)
{
   ETG_TRACE_USR4(("vHandle_EditPredefinedMessage_Return : Called Handle =%d", u16Handle));

   most_msgfi_tclMsgEditPredefinedMessageMethodResult oMethodResultObj;
   oMethodResultObj.u16PredefinedMessageHandle = u16Handle;

   // Get the coresponding Call Instance
   fi_tclVisitorMessage oMethodResult(oMethodResultObj, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData(oMethodResult,
         m_oEditPredefinedMessage.u16SourceAppID, // ServiceDataMsg
         m_oEditPredefinedMessage.u16RegisterID,  // RegisterID
         m_oEditPredefinedMessage.u16CmdCounter,  // CmdCounter
         m_oEditPredefinedMessage.u16FunctionID,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT      // Opcode
   );
   if (oMethodResult.bIsValid())
   {
      // Post Message
      if (_poMainAppl->enPostMessage(&oMethodResult, TRUE) != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("fc_messaging_tclService_Messaging::vHandle_EditPredefinedMessage_Return: Message Post failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("fc_messaging_tclService_Messaging::vHandle_EditPredefinedMessage_Return: Invaild Message"));
   }

   oMethodResultObj.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_EditPredefinedMessage_Error()
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_EditPredefinedMessage_Error
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_EditPredefinedMessage_Error()
{
   ETG_TRACE_USR4(("vHandle_EditPredefinedMessage_Error() called"));

   most_msgfi_tclMsgEditPredefinedMessageError oEditPredefinedMessageMethodError;

   oEditPredefinedMessageMethodError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(oEditPredefinedMessageMethodError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oEditPredefinedMessage.u16SourceAppID, // ServiceDataMsg
         m_oEditPredefinedMessage.u16RegisterID,  // RegisterID
         m_oEditPredefinedMessage.u16CmdCounter,  // CmdCounter
         m_oEditPredefinedMessage.u16FunctionID,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR             // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_EditPredefinedMessage_Error: enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oEditPredefinedMessageMethodError.vDestroy();
}

void fc_messaging_tclService_Messaging::messagingAudioPlaybackState(
        tU8 e8Playmode)
{
    ETG_TRACE_USR4(("messagingAudioPlaybackState : Called"));

    m_oMessagingAudioPlaybackState_Status.e8PlaybackState.enType
            = (most_fi_tcl_e8_MsgPlaybackState::tenType) e8Playmode;

    eUpdatePropertyToClients( MOST_MSGFI_C_U16_MESSAGINGAUDIOPLAYBACKSTATE);
}

/*******************************************************************************
 ** FUNCTION:   vHandle_SwitchMapOnOff_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_SwitchMapOnOff_MethodStart
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_SwitchMapOnOff_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_SwitchMapOnOff_MethodStart() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgSwitchMapOnOffMethodStart oSwitchMapOnOffMethodStart;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(oSwitchMapOnOffMethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("u8DeviceHandle : %d", oSwitchMapOnOffMethodStart.u8DeviceHandle));
      ETG_TRACE_USR4(("bOnOffState : %d", oSwitchMapOnOffMethodStart.bOnOffState));

      m_oSwitchMapOnOff.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oSwitchMapOnOff.u16RegisterID  = poMessage->u16GetRegisterID();
      m_oSwitchMapOnOff.u16CmdCounter  = poMessage->u16GetCmdCounter();
      m_oSwitchMapOnOff.u16FunctionID  = poMessage->u16GetFunctionID();

      tBool bRetValue = FALSE;

      //Check if the device handle is in the paired device list
      tBool isDeviceAvailable = FALSE;

      isDeviceAvailable = FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->bIsDeviceAvailableInDB(
            oSwitchMapOnOffMethodStart.u8DeviceHandle, FC_MSG_DEVICESETTINGS_TABLE_NAME);

      if(TRUE == isDeviceAvailable)
      {
         //Set MapOnOff for the specific device in DeviceSettings table in Database
         MessagingSettings* oMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
         if (oMessagingSettings)
         {
            bRetValue = oMessagingSettings->bSetDeviceSpecificSetting(oSwitchMapOnOffMethodStart.u8DeviceHandle, MAP_ON_OFF_SETTING, oSwitchMapOnOffMethodStart.bOnOffState);
         }

         if(bRetValue)
         {
            if((oSwitchMapOnOffMethodStart.u8DeviceHandle >= MINIMUM_DEVICE_HANDLE)
                  && (oSwitchMapOnOffMethodStart.u8DeviceHandle <= MAXIMUM_DEVICE_HANDLE))
            {
               //SwitchMapOnOff triggered for specific device
               DbusClientIf client = DbusClientInterface::getInstance().getClientInterface();
               if(DBUS_CLIENT_IF_EVOLUTION == client)
               {
                  if (TRUE == oSwitchMapOnOffMethodStart.bOnOffState)
                  {
                     EvolutionDbusClient::vConnectDeviceToMAPSession(oSwitchMapOnOffMethodStart.u8DeviceHandle);
                  }
                  else
                  {
                     EvolutionDbusClient::vDisconnectDeviceFromMAPSession(oSwitchMapOnOffMethodStart.u8DeviceHandle);
                  }
               }
            }
            else if(0 == oSwitchMapOnOffMethodStart.u8DeviceHandle)
            {
               //SwitchMapOnOff has been triggered for all devices
               vSwitchMapOnOff(oSwitchMapOnOffMethodStart.bOnOffState);

               //Update MessagingOnOff Property to Clients
               eUpdatePropertyToClients(MOST_MSGFI_C_U16_MESSAGINGONOFF);
            }
            else
            {
               ETG_TRACE_USR4(("SwitchMapOnOff called with invalid DeviceHandle"));
               bRetValue = FALSE;
            }
         }
      }

      if(bRetValue)
      {
         vHandle_SwitchMapOnOff_Return(oSwitchMapOnOffMethodStart.u8DeviceHandle, oSwitchMapOnOffMethodStart.bOnOffState);

         //Update MessagingOnOffList Property to Clients
         eUpdatePropertyToClients( MOST_MSGFI_C_U16_MESSAGINGONOFFLIST);
      }
      else
      {
         //If query result is failure, send method error
         vHandle_SwitchMapOnOff_Error();
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_SwitchMapOnOff_MethodStart");
   }

   oSwitchMapOnOffMethodStart.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_SwitchMapOnOff_Return(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_SwitchMapOnOff_Return
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_SwitchMapOnOff_Return(tU8 u8DeviceHandle, tBool bOnOffState)
{
   ETG_TRACE_USR4(("vHandle_SwitchMapOnOff_Return() called  OnOffState : %d", bOnOffState));

   most_msgfi_tclMsgSwitchMapOnOffMethodResult oSwitchMapOnOffMethodResult;
   oSwitchMapOnOffMethodResult.u8DeviceHandle = u8DeviceHandle;
   oSwitchMapOnOffMethodResult.bOnOffState = bOnOffState;

   /* Get the coresponding Call Instance */
   fi_tclVisitorMessage oMsg(oSwitchMapOnOffMethodResult, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData(oMsg,
         m_oSwitchMapOnOff.u16SourceAppID, // ServiceDataMsg
         m_oSwitchMapOnOff.u16RegisterID,  // RegisterID
         m_oSwitchMapOnOff.u16CmdCounter,  // CmdCounter,
         m_oSwitchMapOnOff.u16FunctionID,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
   );

   if (oMsg.bIsValid())
   {
      // Post Message
      if (_poMainAppl->enPostMessage(&oMsg, TRUE) != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_SwitchMapOnOff_Return() enPostMessage failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("vHandle_SwitchMapOnOff_Return() Invaild Message"));
   }

   oSwitchMapOnOffMethodResult.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_SwitchMapOnOff_Error()
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_SwitchMapOnOff_Error
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_SwitchMapOnOff_Error()
{
   ETG_TRACE_ERR(("vHandle_SwitchMapOnOff_Error() called"));

   most_msgfi_tclMsgSwitchMapOnOffError oSwitchMapOnOffMethodError;
   oSwitchMapOnOffMethodError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(oSwitchMapOnOffMethodError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oSwitchMapOnOff.u16SourceAppID, // ServiceDataMsg
         m_oSwitchMapOnOff.u16RegisterID,  // RegisterID
         m_oSwitchMapOnOff.u16CmdCounter,  // CmdCounter,
         m_oSwitchMapOnOff.u16FunctionID,  // Function ID,
         AMT_C_U8_CCAMSG_OPCODE_ERROR      // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_SwitchMapOnOff_Error: enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oSwitchMapOnOffMethodError.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_AddCustomMessage_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_AddCustomMessage_MethodStart
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddCustomMessage_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_AddCustomMessage_MethodStart() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_msgfi_tclMsgAddCustomMessageMethodStart oAddCustomMessageMethodStart;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(oAddCustomMessageMethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("u8DeviceHandle : %d", oAddCustomMessageMethodStart.u8DeviceHandle));
      ETG_TRACE_USR4(("e8PredefinedMessageSubCategory : %d", oAddCustomMessageMethodStart.e8PredefinedMessageSubCategory.enType));
      ETG_TRACE_USR4(("sPredefinedMessageText : %s", oAddCustomMessageMethodStart.sPredefinedMessageText.szValue));

      m_oAddCustomMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
      m_oAddCustomMessage.u16RegisterID  = poMessage->u16GetRegisterID();
      m_oAddCustomMessage.u16CmdCounter  = poMessage->u16GetCmdCounter();
      m_oAddCustomMessage.u16FunctionID  = poMessage->u16GetFunctionID();

      if (!((oAddCustomMessageMethodStart.e8PredefinedMessageSubCategory.enType == 0) ||
           ((oAddCustomMessageMethodStart.e8PredefinedMessageSubCategory.enType >= 1) &&
            (oAddCustomMessageMethodStart.e8PredefinedMessageSubCategory.enType <= 4))))
      {
         //send method error
         ETG_TRACE_USR4(("e8PredefinedMessageSubCategory is not in range. Sending MethodError"));
         vHandle_AddCustomMessage_Error();
      }
      else
      {
         tU16 u16PredefMsgHandle = 0;
         tBool bSuccess = FALSE;
         tBool isDeviceAvailable = FALSE;

         isDeviceAvailable = FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->bIsDeviceAvailableInDB(
               oAddCustomMessageMethodStart.u8DeviceHandle, FC_MSG_DEVICESETTINGS_TABLE_NAME);

         if(TRUE == isDeviceAvailable)
         {
            MessagingSettings* l_poMessageSettings;
            l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();

            if (l_poMessageSettings)
            {
               FC_Messaging_PredefinedMsgList* l_poPredefinedMsgList;
               l_poPredefinedMsgList = l_poMessageSettings->poGetPredefinedMessageListInstance();

               if (l_poPredefinedMsgList)
               {
                  bSuccess = l_poPredefinedMsgList->bAddPredefinedMessage(
                        oAddCustomMessageMethodStart.u8DeviceHandle,
                        oAddCustomMessageMethodStart.sPredefinedMessageText,
                        most_fi_tcl_e8_MsgPredefinedMessageType::FI_EN_E8PRE_MSG_STATIC,
                        FC_MSG_USER_PRE_DEF_MSG, //SourceType
                        u16PredefMsgHandle,
                        oAddCustomMessageMethodStart.e8PredefinedMessageSubCategory.enType);
               }
            }
         }

         if (bSuccess == TRUE)
         {
            vHandle_AddCustomMessage_Return(oAddCustomMessageMethodStart.u8DeviceHandle, u16PredefMsgHandle);
         }
         else
         {
            //adding predefined message failed, send method error
            vHandle_AddCustomMessage_Error();
         }
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_AddCustomMessage_MethodStart");
   }

   oAddCustomMessageMethodStart.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_AddCustomMessage_Return(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_AddCustomMessage_Return
*
*  PARAMETERS:
*     IN: u8DeviceHandle     : device handle
*     IN: u16PredefMsgHandle : generated handle for given predefined text
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddCustomMessage_Return(tU8 u8DeviceHandle, tU16 u16PredefMsgHandle)
{
   ETG_TRACE_USR4(("vHandle_AddCustomMessage_Return() called"));
   ETG_TRACE_USR4(("DeviceHandle : %d PredefMsgHandle : %d", u8DeviceHandle, u16PredefMsgHandle));

   most_msgfi_tclMsgAddCustomMessageMethodResult oAddCustomMessageMethodResult;
   oAddCustomMessageMethodResult.u8DeviceHandle = u8DeviceHandle;
   oAddCustomMessageMethodResult.u16PredefinedMessageHandle = u16PredefMsgHandle;

   // Get the coresponding Call Instance
   fi_tclVisitorMessage oMsg(oAddCustomMessageMethodResult, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData(oMsg,
         m_oAddCustomMessage.u16SourceAppID, // ServiceDataMsg
         m_oAddCustomMessage.u16RegisterID,  // RegisterID
         m_oAddCustomMessage.u16CmdCounter,  // CmdCounter,
         m_oAddCustomMessage.u16FunctionID,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
   );

   if (oMsg.bIsValid())
   {
      // Post Message
      if (_poMainAppl->enPostMessage(&oMsg, TRUE) != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_AddCustomMessage_Return() enPostMessage failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("vHandle_AddCustomMessage_Return() Invaild Message"));
   }

   oAddCustomMessageMethodResult.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_AddCustomMessage_Error()
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_AddCustomMessage_Error
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_AddCustomMessage_Error()
{
   ETG_TRACE_ERR(("vHandle_AddCustomMessage_Error() called"));

   most_msgfi_tclMsgAddCustomMessageError oAddCustomMessageMethodError;

   oAddCustomMessageMethodError.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(oAddCustomMessageMethodError, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg,
         m_oAddCustomMessage.u16SourceAppID, // ServiceDataMsg
         m_oAddCustomMessage.u16RegisterID,  // RegisterID
         m_oAddCustomMessage.u16CmdCounter,  // CmdCounter,
         m_oAddCustomMessage.u16FunctionID,  // Function ID,
         AMT_C_U8_CCAMSG_OPCODE_ERROR        // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_AddCustomMessage_Error: enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   oAddCustomMessageMethodError.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetConversationMessages_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for Get Conversation messages method start.
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetConversationMessages_MethodStart(
        amt_tclServiceData* poMessage, struct Job* f_prJob)
{
   ETG_TRACE_USR4(("vHandle_GetConversationMessages_MethodStart : Called"));

   // Bugfix for GMMY16-7568 and GMMY16-7536: Nullpointerexception in
   // /opt/bosch/gm/processes/fc_messaging_out.out" in thread AE_60

   struct Job* prJob = NULLPTR;
   most_msgfi_tclMsgGetConversationMessagesMethodStart* poGetConversationMessagesMethodStart = NULLPTR;

   if(poMessage)
   {
      fi_tclVisitorMessage oVisitorMsg(poMessage);
      poGetConversationMessagesMethodStart = new most_msgfi_tclMsgGetConversationMessagesMethodStart;
      if(poGetConversationMessagesMethodStart)
      {
         if(oVisitorMsg.s32GetData(*poGetConversationMessagesMethodStart, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
         {
            vHandleGetDataError(poMessage, "vHandle_GetConversationMessages_MethodStart");
            delete poGetConversationMessagesMethodStart;
            return;
         }

         prJob = new Job;
         if(prJob)
         {
            prJob->m_pfMethod = &fc_messaging_tclService_Messaging::vHandle_GetConversationMessages_MethodStart;
            prJob->u8DeviceHandle = poGetConversationMessagesMethodStart->oMessageHandle.u8DeviceHandle;
            prJob->u16CmdCounter = poMessage->u16GetCmdCounter();
            prJob->u16FunctionID = poMessage->u16GetFunctionID();
            prJob->u16RegisterID = poMessage->u16GetRegisterID();
            prJob->u16SourceAppID = poMessage->u16GetSourceAppID();
            prJob->m_pvMethodStart = poGetConversationMessagesMethodStart;
         }
      }
   }
   else if(f_prJob)
   {
      //Request(from queue) processed after getting mutex lock
      prJob = f_prJob;
   }
   else
   {
      return; // Should not come here
   }

   if(MessagingMutex::messagingMutexTrylock(prJob) != 0)
   {
      tVoid*  pCustodialLose = prJob; //To remove LINT warning : Custodial Pointer
      (tVoid) pCustodialLose;
      return; // Didn't get the lock, return
   }

   if(prJob)
   {
      m_oGetConversationMessages.u16CmdCounter = prJob->u16CmdCounter;
      m_oGetConversationMessages.u16FunctionID = prJob->u16FunctionID;
      m_oGetConversationMessages.u16RegisterID = prJob->u16RegisterID;
      m_oGetConversationMessages.u16SourceAppID = prJob->u16SourceAppID;

      if(poGetConversationMessagesMethodStart)
      {
         tVoid*  pCustodialLose = poGetConversationMessagesMethodStart;
         (tVoid) pCustodialLose;
      }

      poGetConversationMessagesMethodStart = (most_msgfi_tclMsgGetConversationMessagesMethodStart*) prJob->m_pvMethodStart;
   }

   if(poGetConversationMessagesMethodStart)
   {
      FileSystemManager* poFileSystemManager = FileSystemManager::poFileSystemManager(poGetConversationMessagesMethodStart->oMessageHandle);

      if(poFileSystemManager)
      {
         poFileSystemManager->bFindConversationMessages(*poGetConversationMessagesMethodStart);
      }
      else
      {
         ETG_TRACE_USR4(("vHandle_GetConversationMessages_MethodStart poFileSystemManager is null"));
         vHandle_GetConversationMessage_Error();
      }

      poGetConversationMessagesMethodStart->vDestroy();
      delete poGetConversationMessagesMethodStart;
   }
   delete prJob;
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetConversationMessages_MethodResult
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of GetConversation to client.
 *   method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetConversationMessages_MethodResult(
        QVector<most_fi_tcl_ShortStream> f_oMessageHandleList,
        tU32 f_u32MatchFound)
{
   ETG_TRACE_USR4(("vHandle_GetConversationMessages_MethodResult : Called  MatchFound =%d", f_u32MatchFound));
   tInt u32Index;

   most_msgfi_tclMsgGetConversationMessagesMethodResult oGetConversationMessagesMethodResult_MR;

   for (u32Index = 0;
        (u32Index < (tInt)f_u32MatchFound) && (u32Index < f_oMessageHandleList.size());
         u32Index++)
   {
      oGetConversationMessagesMethodResult_MR.oConversationMessages.oItems.push_back(f_oMessageHandleList.at(u32Index));
   }

   fi_tclVisitorMessage oMethodResult(oGetConversationMessagesMethodResult_MR, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   // Here the message related header information is added and by this the
   // creation of the message is completed.
   vInitServiceData(oMethodResult, m_oGetConversationMessages.u16SourceAppID, // ServiceDataMsg
         m_oGetConversationMessages.u16RegisterID, // RegisterID
         m_oGetConversationMessages.u16CmdCounter, // CmdCounter,
         m_oGetConversationMessages.u16FunctionID, // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
   );

   if (oMethodResult.bIsValid())
   {
      // Post Message
      if (_poMainAppl->enPostMessage(&oMethodResult, TRUE) != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_GetConversationMessages_MethodResult: Message Post failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("vHandle_GetConversationMessages_MethodResult: Invaild Message"));
   }

   MessagingMutex::messagingMutexUnlock(); //Bugfix for GMMY16-7536 & GMMY16-7568

   oGetConversationMessagesMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetConversationMessage_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of GetConversation to client.
 *   method
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetConversationMessage_Error()
{
   ETG_TRACE_USR4(("vHandle_GetConversationMessage_Error : Called"));
   most_msgfi_tclMsgGetConversationMessagesError oMsgGetConversationMessagesError_ER;

   oMsgGetConversationMessagesError_ER.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(oMsgGetConversationMessagesError_ER, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg, m_oGetConversationMessages.u16SourceAppID, // ServiceDataMsg
   m_oGetConversationMessages.u16RegisterID, // RegisterID
   m_oGetConversationMessages.u16CmdCounter, // CmdCounter,
   m_oGetConversationMessages.u16FunctionID, // Function ID
   AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
   );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("'MethodResult', ail enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
   }

   MessagingMutex::messagingMutexUnlock(); //Bugfix for GMMY16-7536 & GMMY16-7568

   oMsgGetConversationMessagesError_ER.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vHandle_SourceInfo_Error(
        most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandle_SourceInfo_Error : Called"));
    most_msgfi_tclMsgSourceInfoError oMsgSourceInfoError_ER;

    tU8 u8PositionOfErroneousParameter = 0x02;
    tU8 au8ErrorInfo[1] = {0x00};

    switch (e8ErrorCode)
    {
      //Only these 2 error cases are set during SourceInfoMethod processing.
    default:
    case most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE:
    {
      au8ErrorInfo[0] = 0x01;//== Set the error parameter position (First parameter is SourceNr) (from document most_MPlay_fi)
      u8PositionOfErroneousParameter = 0x01;
    }
    break;

      case most_fi_tcl_e8_ErrorCode::FI_EN_INVALIDLENGTH:
      {
        au8ErrorInfo[0] = 0x00;//== Dummy zero (from document most_MPlay_fi)
        u8PositionOfErroneousParameter = 0x00;
      }
      break;
    }

    fi_tclOutContext oOut(au8ErrorInfo, au8ErrorInfo + sizeof(au8ErrorInfo),
                          MOST_MSGFI_C_U16_SERVICE_MAJORVERSION,
                          oMsgSourceInfoError_ER.enGetByteOrder());

    // write values to buffer
    (tVoid)(oOut << u8PositionOfErroneousParameter);

    // write data to FI data object
    oMsgSourceInfoError_ER.oErrorInfo.vSetData(au8ErrorInfo,
                                               sizeof(au8ErrorInfo));
    oMsgSourceInfoError_ER.e8ErrorCode.enType = e8ErrorCode;

    fi_tclVisitorMessage oMRMsg(oMsgSourceInfoError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
                     poMessage->u16GetRegisterID(), // RegisterID
                     poMessage->u16GetCmdCounter(), // CmdCounter,
                     poMessage->u16GetFunctionID(), // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult =
                fc_messaging_tclApp::m_poMessaging->_poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(("'MethodResult', ail enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    oMsgSourceInfoError_ER.vDestroy();
}

tVoid fc_messaging_tclService_Messaging::vHandle_MessagingRecordAudio_MethodStart(
        amt_tclServiceData* poMessage)
{

    ETG_TRACE_USR4(("vHandle_MessagingRecordAudio_MethodStart( : Called"));
    /* Extract the required info from poMessage */
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgRecordAudioMessageMethodStart
            oRecordAudioMessage_MethodStart;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.

    if (oVisitorMsg.s32GetData(oRecordAudioMessage_MethodStart,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {

        m_oMSGRecordAudioMessage.u16SourceAppID
                = poMessage->u16GetSourceAppID();
        m_oMSGRecordAudioMessage.u16RegisterID = poMessage->u16GetRegisterID();
        m_oMSGRecordAudioMessage.u16CmdCounter = poMessage->u16GetCmdCounter();
        m_oMSGRecordAudioMessage.u16FunctionID = poMessage->u16GetFunctionID();

        ETG_TRACE_USR1((" Record Message Action:: %u",ETG_ENUM(TR_MSG_RECORDACTION,oRecordAudioMessage_MethodStart.e8RecordMessageAction.enType)));
        // Use the message handle to get the details of the message viz. sender and recipient addresses
        /*if (m_oAudioRecording)
        {

            m_oAudioRecording->vRequestRecordAudioMessage(
                        oRecordAudioMessage_MethodStart.e8RecordMessageAction.enType,
                        oRecordAudioMessage_MethodStart.u8LogicalAVChannel);
        }*/

    }
    else
    {
        vHandleGetDataError(poMessage,
                            "vHandle_MessagingRecordAudio_MethodStart");
    }

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)

    oRecordAudioMessage_MethodStart.vDestroy();

}

tVoid fc_messaging_tclService_Messaging::vHandle_MessagingRecordAudio_MethodResult(
        tU16 u16AudioRecordHandle)
{

	ETG_TRACE_USR4(("vHandle_MessagingRecordAudio_MethodResult : CALLED RECORDING HANDLE =%d", u16AudioRecordHandle));

    most_msgfi_tclMsgRecordAudioMessageMethodResult oRecordAudioMessage_MR;
    oRecordAudioMessage_MR.u16RecordedAudioMessageHandle = u16AudioRecordHandle;

    fi_tclVisitorMessage oMRMsg(oRecordAudioMessage_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, // ServiceDataMsg
                     m_oMSGRecordAudioMessage.u16SourceAppID, // Target AppID
                     m_oMSGRecordAudioMessage.u16RegisterID, // RegisterID
                     m_oMSGRecordAudioMessage.u16CmdCounter, // CmdCounter,
                     m_oMSGRecordAudioMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );
    if (oMRMsg.bIsValid())
    {
        // POST THE MESSAGE
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            // Error situation:
            // oMsg1.bDelete();   // A '.bDelete()' is not necessary here as the
            // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care of this
            // implicitely.
            ETG_TRACE_ERR(
                        (" enPostMessage() failed for 'vHandle_MessagingRecordAudio_MethodResult - METHODRESULT' message, 'ail_tenCommunicationError' = %u", (tU32) enResult));

        }
    }
    else
    {
        ETG_TRACE_ERR(
                    (" enPostMessage() failed for 'vHandle_MessagingRecordAudio_MethodResult - METHODRESULT oMRMsg is NOT VALID"));
    }

    oRecordAudioMessage_MR.vDestroy();


}

tVoid fc_messaging_tclService_Messaging::vHandle_MessagingPlaybackRecordedAudio_MethodStart(
        amt_tclServiceData* poMessage)
{

    ETG_TRACE_USR4(("vHandle_MessagingPlaybackRecordedAudio_MethodStart : Called"));
    /* Extract the required info from poMessage */
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgPlaybackAudioMessageMethodStart
            oPlaybackAudioMessage_MethodStart;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.

    if (oVisitorMsg.s32GetData(oPlaybackAudioMessage_MethodStart,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
    	ETG_TRACE_USR1((" Playback Action:: %u",ETG_ENUM(TR_MSG_PLAYBACKACTION,oPlaybackAudioMessage_MethodStart.e8PlaybackAction.enType)));
    	// Use the message handle to get the details of the message viz. sender and recipient addresses
        /*if (m_oAudioRecording)
        {
            m_oAudioRecording->vRequestPlaybackRecordedAudioMessage(
                        oPlaybackAudioMessage_MethodStart.u16RecordedAudioMessageHandle,
                        oPlaybackAudioMessage_MethodStart.e8PlaybackAction.enType);
        }*/

        m_oMSGPlaybackRecordAudioMessage.u16SourceAppID
                = poMessage->u16GetSourceAppID();
        m_oMSGPlaybackRecordAudioMessage.u16RegisterID
                = poMessage->u16GetRegisterID();
        m_oMSGPlaybackRecordAudioMessage.u16CmdCounter
                = poMessage->u16GetCmdCounter();
        m_oMSGPlaybackRecordAudioMessage.u16FunctionID
                = poMessage->u16GetFunctionID();
    }
    else
    {
        vHandleGetDataError(poMessage,
                            "vHandle_MessagingPlaybackRecordAudio_MethodStart");
    }

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)

    oPlaybackAudioMessage_MethodStart.vDestroy();

}

tVoid fc_messaging_tclService_Messaging::vHandle_MessagingPlaybackRecordAudio_MethodResult(
        most_fi_tcl_e8_MsgPlaybackState::tenType e8PlaybackState)
{

    ETG_TRACE_USR4(("vHandle_MessagingPlaybackRecordAudio_MethodResult : Called"));

    most_msgfi_tclMsgPlaybackAudioMessageMethodResult oPlaybackAudioMessage_MR;
    oPlaybackAudioMessage_MR.e8PlaybackState.enType = e8PlaybackState;

    fi_tclVisitorMessage oMRMsg(oPlaybackAudioMessage_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, // ServiceDataMsg
                     m_oMSGPlaybackRecordAudioMessage.u16SourceAppID, // Target AppID
                     m_oMSGPlaybackRecordAudioMessage.u16RegisterID, // RegisterID
                     m_oMSGPlaybackRecordAudioMessage.u16CmdCounter, // CmdCounter,
                     m_oMSGPlaybackRecordAudioMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    ETG_TRACE_USR1((" Playback State:: %u",ETG_ENUM(TR_MSG_PLAYBACKSTATE,oPlaybackAudioMessage_MR.e8PlaybackState.enType)));
    // POST THE MESSAGE
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg,
                                                                    TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        // Error situation:
        // oMsg1.bDelete();   // A '.bDelete()' is not necessary here as the
        // 'TRUE' in "enPostMessage(&oMsg, TRUE)" will take care of this
        // implicitely.
        ETG_TRACE_ERR(
                    (" enPostMessage() failed for 'MessagingPlaybackRecordAudio_MethodResult - METHODRESULT' message, 'ail_tenCommunicationError' = %u", (tU32) enResult));

    }

    messagingAudioPlaybackState((tU8) e8PlaybackState);
    oPlaybackAudioMessage_MR.vDestroy();


}

tVoid fc_messaging_tclService_Messaging::vHandle_MessagingRecordedAudioMessageInfoProp_Status(
        tU16 newHandle)
{
    ETG_TRACE_USR4(("vHandle_MessagingRecordedAudioMessageInfoProp_Status : Called  newHandle %d =", newHandle));

    m_oRecordedAudioMessageInfo_Status.u16RecordedAudioMessageHandle
            = newHandle;

    ail_tenCommunicationError enResult = eUpdatePropertyToClients(
                MOST_MSGFI_C_U16_RECORDEDAUDIOMESSAGEINFO);

    if (AIL_EN_N_NO_ERROR != enResult)
    {
        ETG_TRACE_ERR(
                    ("vHandle_MessagingRecordedAudioMessageInfoProp_Status : 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' =  %d", static_cast<unsigned int> ((tU32) enResult)));
    }
}

tVoid fc_messaging_tclService_Messaging::vHandle_MessagingRecordedAudioMessageState_Status(
        most_fi_tcl_e8_MsgRecordMessageState::tenType e8RecordMessageState)
{
    ETG_TRACE_USR4(("vHandle_MessagingRecordedAudioMessageState_Status : Called"));

    m_oAudioMessageRecordingState_Status.e8RecordMessageState.enType
            = e8RecordMessageState;

    ail_tenCommunicationError enResult = eUpdatePropertyToClients(
                MOST_MSGFI_C_U16_AUDIOMESSAGERECORDINGSTATE);

    if (AIL_EN_N_NO_ERROR != enResult)
    {
        ETG_TRACE_ERR(
                    ("RecordingSTATE_STATUS : 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' =  %d", static_cast<unsigned int> ((tU32) enResult)));
    }
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestAccountList_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for RequestAccountList method start.
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestAccountList_MethodStart(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandle_RequestAccountList_MethodStart : Called"));

    fi_tclVisitorMessage oVisitorMsg(poMessage);
    most_msgfi_tclMsgRequestAccountListMethodStart
            oRequestAccountListMethodStart_MS;

    if (oVisitorMsg.s32GetData(oRequestAccountListMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        m_oRequestAccountLlist.u16SourceAppID = poMessage->u16GetSourceAppID();
        m_oRequestAccountLlist.u16RegisterID = poMessage->u16GetRegisterID();
        m_oRequestAccountLlist.u16CmdCounter = poMessage->u16GetCmdCounter();
        m_oRequestAccountLlist.u16FunctionID = poMessage->u16GetFunctionID();

        vHandle_RequestAccountList_Return(
                    oRequestAccountListMethodStart_MS.u8DeviceHandle);
    }
    else
    {
        amt_tclServiceDataError oErrorMsg(*poMessage,
                                          CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oErrorMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("fc_messaging_tclService_Messaging::vHandle_RequestAccountList_MethodStart: Message Post Failed"));
        }
    }

    oRequestAccountListMethodStart_MS.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RequestAccountList_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the RequestAccountList result to client.
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RequestAccountList_Return(
        tU8 f_u8DeviceHandle)
{
    ETG_TRACE_USR4(("vHandle_RequestAccountList_Return : Called DeviceHandle = %d", f_u8DeviceHandle));

    tU8 l_u8NumberofService;
    tU8 l_u8Index;
    most_msgfi_tclMsgRequestAccountListMethodResult
            l_oRequestAccountListMethodResult_MR;

    most_fi_tcl_MsgAccountListResultItem
            l_oAccountListResultItem[MAX_INSTANCE_NUM];

    MsgCapability_t msgCapabilityInfo;
    if (false == DbusClientInterface::getInstance().getMapMsgCapabilityInfo(f_u8DeviceHandle, msgCapabilityInfo))
    {
        return;
    }

    l_u8NumberofService = msgCapabilityInfo.m_u8ServiceNum;

    if (l_u8NumberofService > MAX_INSTANCE_NUM)
    {
        l_u8NumberofService = MAX_INSTANCE_NUM;
    }

    for (l_u8Index = 0; l_u8Index < l_u8NumberofService; l_u8Index++)
    {
        l_oAccountListResultItem[l_u8Index].u16AccountHandle = msgCapabilityInfo.astServiceInfo[l_u8Index].m_u16AccountHandle;
        l_oAccountListResultItem[l_u8Index].sAccountName.bSet(msgCapabilityInfo.astServiceInfo[l_u8Index].m_sServiceName.toUtf8().constData());

        if ((l_u8Index == msgCapabilityInfo.m_u8ActiveSMSServiceNumber)
                || (l_u8Index == msgCapabilityInfo.m_u8ActiveEMAILServiceNumber))
        {
            l_oAccountListResultItem[l_u8Index].bActiveAccountStatus = true;
        }
        else
        {
            l_oAccountListResultItem[l_u8Index].bActiveAccountStatus = false;
        }

        l_oRequestAccountListMethodResult_MR.oAccountListResult.oItems.push_back(
                    l_oAccountListResultItem[l_u8Index]);
    }

    fi_tclVisitorMessage oMsg(l_oRequestAccountListMethodResult_MR,
                              MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMsg, m_oRequestAccountLlist.u16SourceAppID,
                     m_oRequestAccountLlist.u16RegisterID,
                     m_oRequestAccountLlist.u16CmdCounter,
                     m_oRequestAccountLlist.u16FunctionID,
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);

    if (oMsg.bIsValid())
    {
        // Post Message
        if (_poMainAppl->enPostMessage(&oMsg, TRUE) != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("fc_messaging_tclService_Messaging::vHandle_RequestAccountList_Return: Message Post failed"));
        }
    }
    else
    {
        ETG_TRACE_ERR(
                    ("fc_messaging_tclService_Messaging::vHandle_RequestAccountList_Return: Invaild Message"));
    }

    l_oRequestAccountListMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ChangeAccount_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for ChangeAccount method start
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ChangeAccount_MethodStart(
		amt_tclServiceData* poMessage, struct Job* f_prJob)
{
	ETG_TRACE_USR4(("vHandle_ChangeAccount_MethodStart : Called"));

	struct Job* l_prJob = NULLPTR;
	most_msgfi_tclMsgChangeAccountMethodStart* l_poChangeAccountMethodStart = NULLPTR;

	if(poMessage)
	{
		fi_tclVisitorMessage oVisitorMsg(poMessage);

		l_poChangeAccountMethodStart = new most_msgfi_tclMsgChangeAccountMethodStart;
		if(l_poChangeAccountMethodStart)
		{
			if (oVisitorMsg.s32GetData(*l_poChangeAccountMethodStart,
					MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) == OSAL_ERROR)
			{
				vHandleGetDataError(poMessage, "vHandle_ChangeAccount_MethodStart");
				delete l_poChangeAccountMethodStart;
				return;
			}

			l_prJob = new Job;
			if(l_prJob)
			{
				l_prJob->m_pfMethod  = &fc_messaging_tclService_Messaging::vHandle_ChangeAccount_MethodStart;
				l_prJob->u8DeviceHandle = 0; //TODO retrieve deviceHandle
				l_prJob->u16CmdCounter = poMessage->u16GetCmdCounter();
				l_prJob->u16FunctionID = poMessage->u16GetFunctionID();
				l_prJob->u16RegisterID = poMessage->u16GetRegisterID();
				l_prJob->u16SourceAppID = poMessage->u16GetSourceAppID();
				l_prJob->m_pvMethodStart = l_poChangeAccountMethodStart;
			}
		}
	}
	else if(f_prJob)
	{
		l_prJob = f_prJob;
	}
	else
	{
		return; // Should not come here
	}

	if (MessagingMutex::messagingMutexTrylock(l_prJob) != 0)
	{
		tVoid*  pCustodialLose = l_prJob ;  //To remove LINT warning : Custodial Pointer
		(tVoid) pCustodialLose ;
		return; // Didn't get the lock :(
	}

	if(l_prJob)
	{
		m_oChangeAccount.u16SourceAppID = l_prJob->u16SourceAppID;
		m_oChangeAccount.u16RegisterID = l_prJob->u16RegisterID;
		m_oChangeAccount.u16CmdCounter = l_prJob->u16CmdCounter;
		m_oChangeAccount.u16FunctionID = l_prJob->u16FunctionID;

		if(l_poChangeAccountMethodStart) //LINT
		{
			tVoid*  pCustodialLose = l_poChangeAccountMethodStart ;  //To remove LINT warning
			(tVoid) pCustodialLose ;
		}
		l_poChangeAccountMethodStart =
				(most_msgfi_tclMsgChangeAccountMethodStart*) l_prJob->m_pvMethodStart;
	}

	if(l_poChangeAccountMethodStart)
	{
		tBool bReturnValue = DbusClientInterface::getInstance().messagingMapChgInstance(l_poChangeAccountMethodStart->u16AccountHandle);

		if (bReturnValue == false)
		{
			vHandle_ChangeAccount_Error();
		}

		l_poChangeAccountMethodStart->vDestroy();
		delete l_poChangeAccountMethodStart;
	}
	delete l_prJob;
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ChangeAccount_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the ChangeAccount status to client.
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ChangeAccount_Return(
        tBool bActiveAccountStatus)
{
    ETG_TRACE_USR4(("vHandle_ChangeAccount_Return : Called"));

    most_msgfi_tclMsgChangeAccountMethodResult oChangeAccountMethodResult_MR;

    oChangeAccountMethodResult_MR.bAccountConnectionResult
            = bActiveAccountStatus;

    fi_tclVisitorMessage oMsg(oChangeAccountMethodResult_MR,
                              MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMsg, m_oChangeAccount.u16SourceAppID,
                     m_oChangeAccount.u16RegisterID, m_oChangeAccount.u16CmdCounter,
                     m_oChangeAccount.u16FunctionID, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);

    if (oMsg.bIsValid())
    {
        // Post Message
        if (_poMainAppl->enPostMessage(&oMsg, TRUE) != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(("vHandle_ChangeAccount_Return: Message Post failed"));
        }
    }
    else
    {
        ETG_TRACE_ERR(("vHandle_ChangeAccount_Return: Invalid Message"));
    }

    MessagingMutex::messagingMutexUnlock();

    oChangeAccountMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ChangeAccount_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the ChangeAccount Error to client.
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ChangeAccount_Error()
{
    ETG_TRACE_USR4(("vHandle_ChangeAccount_Error : Called"));

    most_msgfi_tclMsgChangeAccountError oMsgChangeAccountError_ER;

    oMsgChangeAccountError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(oMsgChangeAccountError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oChangeAccount.u16SourceAppID, // ServiceDataMsg
                     m_oChangeAccount.u16RegisterID, // RegisterID
                     m_oChangeAccount.u16CmdCounter, // CmdCounter,
                     m_oChangeAccount.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_ChangeAccount_Error: enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    MessagingMutex::messagingMutexUnlock();

    oMsgChangeAccountError_ER.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RecoverMessage_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for RecoverMessage method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RecoverMessage_MethodStart(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandle_RecoverMessage_MethodStart : Called"));
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgRecoverMessageMethodStart l_oRecoverMessageMethodStart_MS;

    if (oVisitorMsg.s32GetData(l_oRecoverMessageMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        m_oRecoverMessage.u16SourceAppID = poMessage->u16GetSourceAppID();
        m_oRecoverMessage.u16RegisterID = poMessage->u16GetRegisterID();
        m_oRecoverMessage.u16CmdCounter = poMessage->u16GetCmdCounter();
        m_oRecoverMessage.u16FunctionID = poMessage->u16GetFunctionID();

        tBool l_bRecoverMsg = false;
        FileSystemManager* l_poFileSystemManager;

        l_poFileSystemManager = FileSystemManager::poFileSystemManager(
                    l_oRecoverMessageMethodStart_MS.oMessageHandle);

        if (l_poFileSystemManager)
        {
            l_bRecoverMsg = l_poFileSystemManager->bRecoverMessageStart(
                        l_oRecoverMessageMethodStart_MS.oMessageHandle);
        }

        if (l_bRecoverMsg == true)
        {
            vHandle_RecoverMessage_Return(0,
                                          l_oRecoverMessageMethodStart_MS.oMessageHandle);
        }
        else
        {
            vHandle_RecoverMessage_Error();
        }
    }
    else
    {
        vHandleGetDataError(poMessage, "vHandle_RecoverMessage_MethodStart");
    }

    l_oRecoverMessageMethodStart_MS.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RecoverMessage_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of RecoverMessage to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RecoverMessage_Return(
        tBool bStatus, most_fi_tcl_MsgMessageHandle oMessageHandle)
{
    ETG_TRACE_USR4(("vHandle_RecoverMessage_Return : Called"));
    most_msgfi_tclMsgRecoverMessageMethodResult oRecoverMessageMethodResult_MR;

    if (bStatus == 0)
    {
        oRecoverMessageMethodResult_MR.e8RecoveryStatus.enType
                = most_fi_tcl_e8_MsgRecoveryStatus::FI_EN_E8MSG_RECOVER_SUCCESS;
    }
    {
        oRecoverMessageMethodResult_MR.e8RecoveryStatus.enType
                = most_fi_tcl_e8_MsgRecoveryStatus::FI_EN_E8MSG_RECOVER_UNSUCCESSFUL;
    }

    oRecoverMessageMethodResult_MR.oMessageHandle = oMessageHandle;

    fi_tclVisitorMessage oMRMsg(oRecoverMessageMethodResult_MR,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oRecoverMessage.u16SourceAppID, // ServiceDataMsg
                     m_oRecoverMessage.u16RegisterID, // RegisterID
                     m_oRecoverMessage.u16CmdCounter, // CmdCounter,
                     m_oRecoverMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
    	ETG_TRACE_USR1((" Recovery Status:: %u",ETG_ENUM(TR_RECOVERYSTATUS,oRecoverMessageMethodResult_MR.e8RecoveryStatus.enType)));
    	// Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_RecoverMessage_Return: enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    oRecoverMessageMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_RecoverMessage_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of RecoverMessage to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_RecoverMessage_Error()
{
    ETG_TRACE_USR4(("vHandle_RecoverMessage_Error : Called"));
    most_msgfi_tclMsgRecoverMessageError oRecoverMessageError_ER;

    oRecoverMessageError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    /* Get the coresponding Call Instance */
    fi_tclVisitorMessage oMRMsg(oRecoverMessageError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oRecoverMessage.u16SourceAppID, // ServiceDataMsg
                     m_oRecoverMessage.u16RegisterID, // RegisterID
                     m_oRecoverMessage.u16CmdCounter, // CmdCounter,
                     m_oRecoverMessage.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_RecoverMessage_Error: ail enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    oRecoverMessageError_ER.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_FolderListChange_RelUpReg(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_FolderListChange_RelUpReg method
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_FolderListChange_RelUpReg(
        amt_tclServiceData* poMessage)
{
	ETG_TRACE_USR4(("vHandle_FolderListChange_RelUpReg : Called"));
    // Create a FI visitor message for the received CCA message
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    // Create the (message related) FI data object
    most_msgfi_tclMsgFolderListChangeRelUpReg oFolderListChange_RelUpReg;

    // Unmarshal the FI visitor message to the given FI data object.
    // The parameters are copied individually from shared memory to the FI data
    // object.

    if (oVisitorMsg.s32GetData(oFolderListChange_RelUpReg,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {

        /* Get the coresponding Call Instance */
        fi_tclVisitorMessage oStatusMsg(m_oFolderListChange_Status,
                                        MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

        // Here the message related header information is added and by this the
        // creation of the message is completed.
        vInitServiceData(oStatusMsg, // ServiceDataMsg
                         poMessage->u16GetSourceAppID(), // Target AppID
                         poMessage->u16GetRegisterID(), // RegisterID
                         poMessage->u16GetCmdCounter(), // CmdCounter,
                         poMessage->u16GetFunctionID(), // Function ID
                         AMT_C_U8_CCAMSG_OPCODE_STATUS // Opcode
                         );
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oStatusMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_FolderListChange_RelUpReg: 'STATUS', enPostMessage() failed, 'ail_tenCommunicationError' = %u", static_cast<unsigned int> ((tU32) enResult)));
            // We can come here e.g. if the client has changed his application
            // state to OFF in the meantime. Therefore we dont throw an assert.
            // NORMAL_M_ASSERT_ALWAYS();
        }
        else
        {
            //MSG_DEBUG("vHandle_FolderListChange_RelUpReg: 'STATUS Posting");
        }
    }
    else
    {
        // If we come here, we assume the client has sent an erroneous message,
        // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
        // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

        // Create AMT error message and set ErrorCode
        amt_tclServiceDataError oErrorMsg(*poMessage,
                                          CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
        // NOTE: The constructor of amt_tclServiceDataError is re-using the
        // received message '*poMessage' It will collect all necessary Service
        // Information (Source-, Target AppID, RegID, ..) and exchange the
        // Source- and Target AppID and even set the Opcode to ERROR.
        // Additionally the ErrorCode value is set to the second provided
        // parameter. For the new message new memory is allocated, i.e. the
        // received message 'poMessage' is not re-used in terms of memory and
        // the implicite call of 'poMessage->bDelete();' by the framework after
        // leaving this handler function does not harm.

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oErrorMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_FolderListChange_RelUpReg: 'ERROR', enPostMessage() failed, 'ail_tenCommunicationError' = %u", static_cast<unsigned int> ((tU32) enResult)));
            // We can come here e.g. if the client has changed his application
            // state to OFF in the meantime. Therefore we dont throw an assert.
            // NORMAL_M_ASSERT_ALWAYS();
        }

        ETG_TRACE_FATAL(
                    ("vHandle_FolderListChange_RelUpReg: Received message invalid."));
        // We never expect to come here.
        NORMAL_M_ASSERT_ALWAYS();
    }

    // Always destroy the (message related) FI data object (before leaving
    // its creation scope)

    oFolderListChange_RelUpReg.vDestroy();

}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetMsgBodyActionElement_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for GetMsgBodyActionElement method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetMsgBodyActionElement_MethodStart(
        amt_tclServiceData* poMessage)
{
    ETG_TRACE_USR4(("vHandle_GetMsgBodyActionElement_MethodStart : Called"));
    fi_tclVisitorMessage oVisitorMsg(poMessage);

    most_msgfi_tclMsgGetMsgBodyActionElementMethodStart
            oGetMsgBodyActionElementMethodStart_MS;

    if (oVisitorMsg.s32GetData(oGetMsgBodyActionElementMethodStart_MS,
                               MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
    {
        m_oGetMsgBodyActionElement.u16SourceAppID
                = poMessage->u16GetSourceAppID();
        m_oGetMsgBodyActionElement.u16RegisterID
                = poMessage->u16GetRegisterID();
        m_oGetMsgBodyActionElement.u16CmdCounter
                = poMessage->u16GetCmdCounter();
        m_oGetMsgBodyActionElement.u16FunctionID
                = poMessage->u16GetFunctionID();

        vHandle_GetMsgBodyActionElement_Return(
                    oGetMsgBodyActionElementMethodStart_MS.u8MsgBodyActionElementTag);

    }
    else
    {
        amt_tclServiceDataError oErrorMsg(*poMessage,
                                          CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);

        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oErrorMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(
                        ("vHandle_GetMsgBodyActionElement_MethodStart:\
                         'ERROR', enPostMessage() failed"));
        }

        ETG_TRACE_FATAL(
                    ("vHandle_GetMsgBodyActionElement_MethodStart: \
                     Received message invalid."));
                     // We never expect to come here.
                     NORMAL_M_ASSERT_ALWAYS();
    }

    oGetMsgBodyActionElementMethodStart_MS.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_GetMsgBodyActionElement_Return
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the result of GetMsgBodyActionElement to client.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetMsgBodyActionElement_Return(
        tU8 u8MsgBodyActionElementTag)
{
    ETG_TRACE_USR4(("vHandle_GetMsgBodyActionElement_Return : Called MsgBodyActionElementTag = %d", u8MsgBodyActionElementTag));
    (void)u8MsgBodyActionElementTag;
//Lint Comment Start Here
//    char *result = NULL;
//    char email[10][100];
//    char phone[10][25];
//    int emailindex = 0;
//    int phoneindex = 0;
//    int flag;
//    int i;
//    tU32 u32DataLength;
//Lint Comment Ends Here

    //    u32DataLength = MessagingFunctionBlockDbusClient::getMessageDataLength();

    //    s8* pcMessagedata = new s8[u32DataLength + 1];
    //    memset(pcMessagedata, 0, u32DataLength + 1);
    //    strncpy(pcMessagedata, MessagingFunctionBlockDbusClient::getMessageData(),
    //            u32DataLength);

    //    memset(phone[0], 0, 25);
    //    memset(email[0], 0, 100);

    //    result = strtok((char*) pcMessagedata, " ");

    //    while (result != NULL)
    //    {
    //        if (strstr(result, "@") && strstr(result, "."))
    //        {
    //            memset(email[emailindex], 0, 100);
    //            strncpy(email[emailindex++], result, 100);
    //        }

    //        //if(strlen(result) > 8)
    //        {
    //            i = 0;
    //            flag = 1;

    //            while (result[i] != 0)
    //            {
    //                if (!isdigit(result[i]))
    //                    flag = 0;
    //                i++;
    //            }

    //            if (flag == 1)
    //            {
    //                memset(phone[phoneindex], 0, 25);
    //                strncpy(phone[phoneindex++], result, 25);
    //            }
    //        }
    //        result = strtok(NULL, " ");
    //    }

    //    most_msgfi_tclMsgGetMsgBodyActionElementMethodResult
    //            oGetMsgBodyActionElementMethodResult_MR;

    //    oGetMsgBodyActionElementMethodResult_MR.e8ActionElementType.enType
    //            = (most_fi_tcl_e8_MsgActionElementType::tenType) u8MsgBodyActionElementTag;

    //    if (u8MsgBodyActionElementTag
    //            == most_fi_tcl_e8_MsgActionElementType::FI_EN_E8ACTION_ELEMENT_PHONE_NUMBER)
    //    {
    //        oGetMsgBodyActionElementMethodResult_MR.sPhoneNumber.bSet(phone[0]);
    //        oGetMsgBodyActionElementMethodResult_MR.sEmailAddress.bSet("");
    //        oGetMsgBodyActionElementMethodResult_MR.sAddress.bSet("");
    //    }
    //    else if (u8MsgBodyActionElementTag
    //             == most_fi_tcl_e8_MsgActionElementType::FI_EN_E8ACTION_ELEMENT_EMAIL_ADDRESS)
    //    {
    //        oGetMsgBodyActionElementMethodResult_MR.sPhoneNumber.bSet("");
    //        oGetMsgBodyActionElementMethodResult_MR.sEmailAddress.bSet(email[0]);
    //        oGetMsgBodyActionElementMethodResult_MR.sAddress.bSet("");
    //    }
    //    else if (u8MsgBodyActionElementTag
    //             == most_fi_tcl_e8_MsgActionElementType::FI_EN_E8ACTION_ELEMENT_ADDRESS)
    //    {
    //        oGetMsgBodyActionElementMethodResult_MR.sPhoneNumber.bSet("");
    //        oGetMsgBodyActionElementMethodResult_MR.sEmailAddress.bSet("");
    //        oGetMsgBodyActionElementMethodResult_MR.sAddress.bSet("");
    //    }
    //    else
    //    {
    //        oGetMsgBodyActionElementMethodResult_MR.sPhoneNumber.bSet(phone[0]);
    //        oGetMsgBodyActionElementMethodResult_MR.sEmailAddress.bSet("");
    //        oGetMsgBodyActionElementMethodResult_MR.sAddress.bSet("");

    //        oGetMsgBodyActionElementMethodResult_MR .e8ActionElementType.enType
    //                = most_fi_tcl_e8_MsgActionElementType::FI_EN_E8ACTION_ELEMENT_PHONE_NUMBER;
    //    }

    //    /* Get the coresponding Call Instance */
    //    fi_tclVisitorMessage oMRMsg(oGetMsgBodyActionElementMethodResult_MR,
    //                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    //    // Here the message related header information is added and by this the
    //    // creation of the message is completed.
    //    vInitServiceData(oMRMsg, m_oGetMsgBodyActionElement.u16SourceAppID, // ServiceDataMsg
    //                     m_oGetMsgBodyActionElement.u16RegisterID, // RegisterID
    //                     m_oGetMsgBodyActionElement.u16CmdCounter, // CmdCounter,
    //                     m_oGetMsgBodyActionElement.u16FunctionID, // Function ID
    //                     AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
    //                     );

    //    if (oMRMsg.bIsValid())
    //    {
    //        // Post message
    //        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
    //                    &oMRMsg, TRUE);

    //        if (enResult != AIL_EN_N_NO_ERROR)
    //        {
    //            ETG_TRACE_ERR(
    //                        ("vHandle_GetMsgBodyActionElement_Return:
    //                         , enPostMessage() failed"));
    //                         NORMAL_M_ASSERT_ALWAYS();
    //        }
    //    }

    //    delete[] pcMessagedata;
    // oGetMsgBodyActionElementMethodResult_MR.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   FolderListChange
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the property update to clients.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_FolderListChange_Request(
        tU32 l_u32FolderHandle)

{
    ETG_TRACE_USR4(("vHandle_FolderListChange_Request : Called FolderHandle = %d", l_u32FolderHandle));

    m_oFolderListChange_Status.u32FolderHandle = l_u32FolderHandle;
    m_oFolderListChange_Status.bFolderStructureChangedFlag = true;

    eUpdatePropertyToClients( MOST_MSGFI_C_U16_FOLDERLISTCHANGE);

}

tVoid fc_messaging_tclService_Messaging::PredefinedListChangeStatus()

{
    ETG_TRACE_USR4(("PredefinedListChangeStatus : Called"));
    eUpdatePropertyToClients( MOST_MSGFI_C_U16_PREDEFINEDMESSAGELIST);
    // CMG3G-8602 - IS2543_PredefinedMessageListExtended@FC_Messaging
    eUpdatePropertyToClients( MOST_MSGFI_C_U16_PREDEFINEDMESSAGELISTEXTENDED);
}

/*******************************************************************************
 ** FUNCTION:   vPostLoopBackMessage
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the lookback Message to Messaging.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vPostLoopBackMessage(
        LoopBackMessageData oLoopBackMessageData)
{
    ETG_TRACE_USR4(("vPostLoopBackMessage : Called"));
    tU32 u32FunctionID = oLoopBackMessageData.u32FunctionId;

    QMutexLocker lock(&mLoopBackMessageDataVectorMutex);

    vLoopBackDataItems.push_back(oLoopBackMessageData);

    gm_tclU32Message oServiceReadyMsg(_poMainAppl->u16GetAppId(), // Source AppID
                                      _poMainAppl->u16GetAppId(), // Target AppID
                                      0, // RegisterID
                                      0, // CmdCounter
                                      CCA_C_U16_SRV_FB_MESSAGING, // ServiceID
                                      MESSAGING_C_U16_IFID, // Function ID
                                      AMT_C_U8_CCAMSG_OPCODE_STATUS // Opcode
                                      );

    oServiceReadyMsg.vSetDWord(u32FunctionID);

    // Post message
    ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                &oServiceReadyMsg, TRUE);

    if (enResult != AIL_EN_N_NO_ERROR)
    {
        ETG_TRACE_FATAL(
                    ("vProcessService: 'ServiceReady' Opcode STATUS: enPostMessage() failed, 'ail_tenCommunicationError' = %u", (tU32) enResult));
    }
}

/*******************************************************************************
 ** FUNCTION:   findMessagePosition
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to find the message data position in loopback vector.
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: None
 *******************************************************************************/
QVector<LoopBackMessageData>::Iterator fc_messaging_tclService_Messaging::findMessagePosition(
        tU32 u32FunctionId)
{
	ETG_TRACE_USR4(("findMessagePosition : Called FunctionId = %d ", u32FunctionId));
    QVector<LoopBackMessageData>::Iterator it;
    for (it = vLoopBackDataItems.begin(); it != vLoopBackDataItems.end(); it++)
    {
        if (it->u32FunctionId == u32FunctionId)
        {
            return it;
        }
    }

    return NULLPTR;
}

/*******************************************************************************
 ** FUNCTION:  s32ClearAllPrivateData
 *******************************************************************************/
/* DESCRIPTION:
 *
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: bool
 *******************************************************************************/

bool fc_messaging_tclService_Messaging::s32ClearAllPrivateData()
{
    ETG_TRACE_USR4(("s32ClearAllPrivateData entered"));
    //pthread_t WorkQueueThreadID; LINT

	/* GMMY15-1356 - Unclosed file handle due to messaging.out. Make the process of restoring the factory
       settings synchronous operation instead of spawning a thread to perform it */
    MessagingSettings::poGetMessagingSettingsInstance()->vRestoreTextSettingToDefault();

    // Handle predefined messages.
    FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->vDeleteAllPredefinedMessageBasedOnLanguage();


    //Add system predefined messages for all languages
    FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->bAddSystemPredefinedMessage();

    /* Bugfix for GMMY15-12513: Clear User Message Info does
     * not clear the user defined predefined messages */
    FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->newUserPredefinedMessageAdded();
    eUpdatePropertyToClients(MOST_MSGFI_C_U16_TEXTALERTENABLE);
    eUpdatePropertyToClients(MOST_MSGFI_C_U16_PREDEFINEDMESSAGELIST);
    // CMG3G-8602 - IS2543_PredefinedMessageListExtended@FC_Messaging
    eUpdatePropertyToClients(MOST_MSGFI_C_U16_PREDEFINEDMESSAGELISTEXTENDED);

    ETG_TRACE_USR4(("s32ClearAllPrivateData exit"));
    return true;
}

/*******************************************************************************
 ** FUNCTION:  IsMessagingServiceAvailable
 *******************************************************************************/
/* DESCRIPTION:
 *
 *
 *  PARAMETERS:
 *     IN:
 *  RETURNVALUE: bool
 *******************************************************************************/

bool fc_messaging_tclService_Messaging::IsMessagingServiceAvailable()
{
    return static_cast<bool> (m_bServerAvailable);
    //return true;
}

bool fc_messaging_tclService_Messaging::SendMessagingDeviceConnectionDefaultUpdate()
{
    most_msgfi_tclMsgMessagingDeviceConnectionStatus omessagingDeviceCOnnectionStatus;
    omessagingDeviceCOnnectionStatus.u8DeviceHandle = fc_messaging_tclService_Messaging::m_oMessagingDeviceConnection_Status.u8DeviceHandle;

    return true;
}
/*******************************************************************************
 ** FUNCTION:   vHandle_ClearUserMessagingInfo_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for vHandle_ClearUserMessagingInfo_MethodStart method start.
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ClearUserMessagingInfo_MethodStart(
        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ClearUserMessagingInfo_MethodStart : Called"));
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_msgfi_tclMsgClearUserMessagingInfoMethodStart
   oClearUserMessagingInfoMethodStart_MS;

   if (oVisitorMsg.s32GetData(oClearUserMessagingInfoMethodStart_MS,
         MOST_MSGFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      m_oClearUserMessagingInfo.u16SourceAppID
      = poMessage->u16GetSourceAppID();
      m_oClearUserMessagingInfo.u16RegisterID
      = poMessage->u16GetRegisterID();
      m_oClearUserMessagingInfo.u16CmdCounter
      = poMessage->u16GetCmdCounter();
      m_oClearUserMessagingInfo.u16FunctionID
      = poMessage->u16GetFunctionID();

      if(!s32ClearAllPrivateData())
      {
         vHandle_ClearUserMessagingInfo_Error();
      }
      //No Result object (most_msgfi_tclMsgClearUserMessagingInfoMethodResult) is defined.
   }
   else
   {
      amt_tclServiceDataError oErrorMsg(*poMessage,
            CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);

      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
            &oErrorMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(
               ("vHandle_ClearUserMessagingInfo_MethodStart:\
                     'ERROR', enPostMessage() failed"));
      }

      ETG_TRACE_FATAL(
            ("vHandle_ClearUserMessagingInfo_MethodStart: \
            Received message invalid."));
      // We never expect to come here.
      NORMAL_M_ASSERT_ALWAYS();
   }

   oClearUserMessagingInfoMethodStart_MS.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_ClearUserMessagingInfo_Error
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to send the error of DeleteMessage to client.
 *
 *  PARAMETERS:
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ClearUserMessagingInfo_Error()
{
    ETG_TRACE_USR4(("vHandle_ClearUserMessagingInfo_Error : Called"));
    most_msgfi_tclMsgClearUserMessagingInfoError oMsgClearUserMessagingInfoError_ER;

    oMsgClearUserMessagingInfoError_ER.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

    fi_tclVisitorMessage oMRMsg(oMsgClearUserMessagingInfoError_ER,
                                MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

    vInitServiceData(oMRMsg, m_oClearUserMessagingInfo.u16SourceAppID, // ServiceDataMsg
                     m_oClearUserMessagingInfo.u16RegisterID, // RegisterID
                     m_oClearUserMessagingInfo.u16CmdCounter, // CmdCounter,
                     m_oClearUserMessagingInfo.u16FunctionID, // Function ID
                     AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
                     );

    if (oMRMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
                    &oMRMsg, TRUE);

        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(("vHandle_ClearUserMessagingInfo_Error: enPostMessage() failed"));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }

    oMsgClearUserMessagingInfoError_ER.vDestroy();
}

//CMG3G-8287- IS2413_Connect to MAP profile manually@FC_Messaging
/*******************************************************************************
 ** FUNCTION:   vHandle_ConnectMapProfile_MethodStart
 *******************************************************************************/
/* DESCRIPTION:
 *   function is used to Handle the ConnectMapProfile Request from HMI
 *
 *  PARAMETERS: IN:   poMessage        : This is the received message from a client
 *
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ConnectMapProfile_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ConnectMapProfile_MethodStart : Called"));
   tBool bSuccess = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_msgfi_tclMsgConnectMapProfileMethodStart oConnectMapProfileMethodStart_MS;

   if (oVisitorMsg.s32GetData(oConnectMapProfileMethodStart_MS, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("vHandle_ConnectMapProfile_MethodStart : MapRetryCount :%d", oConnectMapProfileMethodStart_MS.RetryCount));

      //check if device handle is valid

      std::map< tU8, std::string >::iterator it = fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.find(oConnectMapProfileMethodStart_MS.DeviceHandle);

      if (((oConnectMapProfileMethodStart_MS.DeviceHandle >= MINIMUM_DEVICE_HANDLE)
            && (oConnectMapProfileMethodStart_MS.DeviceHandle <= MAXIMUM_DEVICE_HANDLE))
            && (it != fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.end()))
      {
         if (true == DbusClientInterface::getInstance().isDbusClientAvailable(oConnectMapProfileMethodStart_MS.DeviceHandle))
         {
            //Check device specific mapConnectionStatus
            for (tU8 i = 0; i < m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems.size(); i++)
            {
               if(m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems[i].u8DeviceHandle == oConnectMapProfileMethodStart_MS.DeviceHandle)
               {
                  if ((m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems[i].e8MapConnectionState.enType
                        != (most_fi_tcl_e8_MapConnectionStatus::tenType) FC_MSG_MAP_CONNECTED)
                        && (m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems[i].e8MapConnectionState.enType
                              != (most_fi_tcl_e8_MapConnectionStatus::tenType) FC_MSG_MAP_CONNECTING))
                  {
                     //set MapRetryCount
                     EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(oConnectMapProfileMethodStart_MS.DeviceHandle);
                     if(evolutionClient)
                     {
                        evolutionClient->vSetMapRetryCount(oConnectMapProfileMethodStart_MS.RetryCount);
                     }

                     //call mapstart
                     ETG_TRACE_USR4(("vHandle_ConnectMapProfile_MethodStart : DeviceHandle %d is valid",oConnectMapProfileMethodStart_MS.DeviceHandle));
                     fc_messaging_tclClientHandler_BluetoothSetings::addDeviceToRetryList(oConnectMapProfileMethodStart_MS.DeviceHandle);
                     DbusClientInterface::getInstance().messagingMapStart();
                  }
                  else
                  {
                     ETG_TRACE_USR4(("vHandle_ConnectMapProfile_MethodStart : MAP sesion already established for DeviceHandle %d ", oConnectMapProfileMethodStart_MS.DeviceHandle));
                  }
               }
            }

            most_msgfi_tclMsgConnectMapProfileMethodResult oMsgConnectMapProfileMethodResult_MR;

            oMsgConnectMapProfileMethodResult_MR.DeviceHandle = oConnectMapProfileMethodStart_MS.DeviceHandle;

            fi_tclVisitorMessage oMRMsg(oMsgConnectMapProfileMethodResult_MR, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

            vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(),
                  poMessage->u16GetRegisterID(),
                  poMessage->u16GetCmdCounter(),
                  poMessage->u16GetFunctionID(),
                  AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
            );

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_USR4(("vHandle_ConnectMapProfile_Return: Post failed"));
               }
               else
                  bSuccess = true;
            }
            else
            {
               ETG_TRACE_USR4(("vHandle_ConnectMapProfile_Return: Invalid message"));
            }

            oMsgConnectMapProfileMethodResult_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("vHandle_ConnectMapProfile_MethodStart: DbusClient is NULL"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("vHandle_ConnectMapProfile_MethodStart: Invalid device handle"));
      }

      if (FALSE == bSuccess)
      {
         most_msgfi_tclMsgConnectMapProfileError oMsgConnectMapProfileError_ER;

         oMsgConnectMapProfileError_ER.e8ErrorCode.enType = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

         fi_tclVisitorMessage oMRMsg(oMsgConnectMapProfileError_ER, MOST_MSGFI_C_U16_SERVICE_MAJORVERSION);

         vInitServiceData(oMRMsg,
               poMessage->u16GetSourceAppID(),
               poMessage->u16GetRegisterID(),
               poMessage->u16GetCmdCounter(),
               poMessage->u16GetFunctionID(),
               AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
         );

         if (oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult =
                  _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_ConnectMapProfile_Error: enPostMessage() failed"));
               NORMAL_M_ASSERT_ALWAYS();
            }
         }

         oMsgConnectMapProfileError_ER.vDestroy();
      }
   }
   else
   {
      amt_tclServiceDataError oErrorMsg(*poMessage, CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandle_ConnectMapProfile_MethodStart:\
                           'ERROR', enPostMessage() failed"));
      }

      ETG_TRACE_FATAL(("vHandle_ConnectMapProfile_MethodStart: \
                       Received message invalid."));
   }

   oConnectMapProfileMethodStart_MS.vDestroy();
}

// GMMY15-9119 FC_Messaging : logiscope code complexity value >= 20
tVoid fc_messaging_tclService_Messaging::vOnLoopbackServiceHelper(
		tU32 f_U32MsgContent, LoopBackMessageData* f_poLoopBackMessageData,
		FileSystemManager* f_poFileSystemManager)
{
   ETG_TRACE_USR4(("vOnLoopbackServiceHelper : Called"));

   switch (f_U32MsgContent)
   {
      case FC_MSG_LB_FID_MESSAGE_CREATE_MESSAGE_LIST:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_MESSAGE_CREATE_MESSAGE_LIST"));

         if(f_poLoopBackMessageData && Phonebook_Detail::pGetPhonebookInstance(f_poLoopBackMessageData->u8DeviceHandle))
         {
            // Bugfix for GMMY15-5992 Large empty gap is present in text messages list
            Phonebook_Detail::pGetPhonebookInstance(f_poLoopBackMessageData->u8DeviceHandle)->vPostCreateMessageListComplete(
                  f_poLoopBackMessageData->u8DeviceHandle);
         }
      }
      break;

      case FC_MSG_LB_FID_UNLOCK_DBUS_MUTEX:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_UNLOCK_DBUS_MUTEX"));
         MessagingMutex::messagingMutexUnlock();
      }
      break;

      // Request Folder List
      case FC_MSG_LB_FID_FOLDER_LIST:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_FOLDER_LIST"));

         if (f_poLoopBackMessageData && f_poLoopBackMessageData->vpData)
         {
            ETG_TRACE_USR2(("Folder name is::%s", (char*) f_poLoopBackMessageData->vpData));

            if(f_poFileSystemManager)
            {
               f_poFileSystemManager->vRequestFolderListResultFromDevice((const char*) f_poLoopBackMessageData->vpData, f_poLoopBackMessageData->u8DeviceHandle);
            }

            delete[] (gchar*) f_poLoopBackMessageData->vpData;
         }
      }
      break;

      case FC_MSG_LB_FID_FOLDER_LIST_COMPLETE:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_FOLDER_LIST_COMPLETE"));

         if (f_poLoopBackMessageData && f_poFileSystemManager)
         {
            f_poFileSystemManager->vRequestFolderListReturnToClient(static_cast<tU8>(f_poLoopBackMessageData->u16Private));
         }
      }
      break;

      case FC_MSG_LB_FID_GET_TOTAL_MESSAGE_COUNT_IN_FOLDER:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_GET_TOTAL_MESSAGE_COUNT_IN_FOLDER"));

         if (f_poLoopBackMessageData && f_poFileSystemManager)
         {
            f_poFileSystemManager->vRequestFolderListGetMessageCountReturn(static_cast<tU8>(f_poLoopBackMessageData->u16Private));
         }
      }
      break;

      case FC_MSG_LB_FID_GET_UNREAD_MESSAGE_COUNT_IN_FOLDER:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_GET_UNREAD_MESSAGE_COUNT_IN_FOLDER"));

         if (f_poLoopBackMessageData && f_poFileSystemManager)
         {
            f_poFileSystemManager->vRequestFolderListGetUnReadMessageCountReturn(static_cast<tU8>(f_poLoopBackMessageData->u16Private));
         }
      }
      break;

      case FC_MSG_LB_FID_GET_SUBFOLDERS_IN_FOLDER:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_GET_SUBFOLDERS_IN_FOLDER"));

         if (f_poLoopBackMessageData && f_poFileSystemManager)
         {
            f_poFileSystemManager->vRequestFolderListGetSubFolderReturn(static_cast<tU8>(f_poLoopBackMessageData->u16Private));
         }
      }
      break;

      //gmmy15_8172 : Text message remains in inbox when deleted from phone
      case FC_MSG_LB_FID_MESSAGE_DELETED_FROM_DEVICE:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_MESSAGE_DELETED_FROM_DEVICE"));
         if (f_poLoopBackMessageData && f_poFileSystemManager)
         {
            if (f_poLoopBackMessageData->vpData)
            {
               tU32 l_u32UpperHandle = 0;
               //CMG3G-1494 - Changes for Email support and sent items folder
               MapNotifySignalLoopBackData *pNotifyData = (MapNotifySignalLoopBackData*) f_poLoopBackMessageData->vpData;
               QString f_sObjectHandle(*(pNotifyData->pacMessageHandle));
               l_u32UpperHandle = f_poFileSystemManager->u32ConvertDbusHandleToUpperHandle(f_sObjectHandle);
               f_poFileSystemManager->bMessageDeletedFromPhoneDevice(l_u32UpperHandle, pNotifyData->u8FolderType, pNotifyData->u8MessageType);
               f_poFileSystemManager->vRemoveDbusMessageHandleonDeletion(l_u32UpperHandle);
               delete pNotifyData;
            }
         }

      }
      break; //lint fix

      case FC_MSG_LB_FID_READ_STATUS_CHANGED:
      {
              ETG_TRACE_USR3(("FC_MSG_LB_FID_READ_STATUS_CHANGED"));
              if (f_poLoopBackMessageData && f_poFileSystemManager)
              {
                 if (f_poLoopBackMessageData->vpData)
                 {
                    tU32 l_u32UpperHandle = 0;
                    //CMG3G-1494 - Changes for Email support and sent items folder
                    MapNotifySignalLoopBackData *pNotifyData = (MapNotifySignalLoopBackData*) f_poLoopBackMessageData->vpData;
                    QString f_sObjectHandle(*(pNotifyData->pacMessageHandle));
                    l_u32UpperHandle = f_poFileSystemManager->u32ConvertDbusHandleToUpperHandle(f_sObjectHandle);
                    f_poFileSystemManager->bMessageReadStatusChanged(l_u32UpperHandle, pNotifyData->u8FolderType);
                    delete pNotifyData;
                 }
              }

      }
      break; //lint fix

      //CMG3G-1494 - Changes for Email support and sent items folder
      case FC_MSG_LB_FID_RECENTLY_SENT_MESSAGE_INFO:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_RECENTLY_SENT_MESSAGE_INFO"));
         if (f_poLoopBackMessageData && f_poLoopBackMessageData->vpData)
         {
            MsgListing_t* pSendMsgList = (MsgListing_t*)f_poLoopBackMessageData->vpData;
            SendMessage::poGetSendMessageInstance()->vRecentlySentMessageInfo(f_poLoopBackMessageData->u8DeviceHandle,f_poLoopBackMessageData->sServiceName, pSendMsgList);
            delete[] pSendMsgList;
         }
      }
      break;

      // Bugfix for GMMY15-11135
      case FC_MSG_LB_FID_GET_MSG_ERROR:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_GET_MSG_ERROR"));

         if ((f_poLoopBackMessageData) && (f_poFileSystemManager))
         {
            //check who is the GetMessage request sender and post error accordingly
            e8GetMessageCaller l_eGetMessageCaller = f_poFileSystemManager->e8CurrentGetMessageCaller();
            f_poFileSystemManager->vSetGetMessageFailure(true);
            if (e8ReadMessageAloud == l_eGetMessageCaller)
            {
               ETG_TRACE_USR3(("FC_MSG_LB_FID_GET_MSG_ERROR caller e8ReadMessageAloud"));

               ReadMessageAloud* oReadMessageAloud = ReadMessageAloud::poGetInstance();
               oReadMessageAloud->vProcessEvent(FC_MESSAGING_SM_EVENT_GETMESSAGE_FAILURE);

               vHandle_ReadMessageAloud_Error();
            }
            else if (e8ConversationDetail == l_eGetMessageCaller)
            {
               ETG_TRACE_USR3(("FC_MSG_LB_FID_GET_MSG_ERROR caller e8ConversationDetail"));
               f_poFileSystemManager->vGetMessagesForConversation();
            }
            else if (e8GetMessageStart == l_eGetMessageCaller)
            {
               ETG_TRACE_USR3(("FC_MSG_LB_FID_GET_MSG_ERROR caller e8GetMessageStart"));
               vHandle_GetMessage_Error();
            }
            else
            {
               ETG_TRACE_USR3(("FC_MSG_LB_FID_GET_MSG_ERROR caller e8NewIncomingMessage or e8SendMessage"));

            }

         }
      }
      break;

      // Bugfix for GMMY16-8827
      case FC_MSG_LB_FID_DEVICE_MEMORY_STATUS:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_DEVICE_MEMORY_STATUS"));

         if (f_poLoopBackMessageData)
         {
            DeviceConnection_t* poDeviceConnection;
            poDeviceConnection = (DeviceConnection_t*) f_poLoopBackMessageData->vpData;
            if (poDeviceConnection)
               _MemoryStatusFromDbus(poDeviceConnection);
         }
      }
      break;

      case FC_MSG_LB_FID_JOB_FROM_WORKQUEUE:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_JOB_FROM_WORKQUEUE"));

         if (f_poLoopBackMessageData)
         {
            Job* prTempJob = (Job*) f_poLoopBackMessageData->vpData;

            if (prTempJob)
            {
               ETG_TRACE_USR3(("Calling using function pointer"));
               (fc_messaging_tclService_Messaging::poGetInstance()->*(prTempJob->m_pfMethod))(NULLPTR, prTempJob);
            }
         }
      }
      break;

      // Task CMG3G-7698: IS2201_API for device capabilities@FC_Messaging
      case FC_MSG_LB_FID_MAP_DEVICE_CAPABILITIES_STATUS:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_MAP_DEVICE_CAPABILITIES_STATUS"));

         vResetMapDeviceCapabilities();

         if (f_poLoopBackMessageData && f_poLoopBackMessageData->vpData)
         {
            trFC_Msg_MapDeviceCapabilities* prFC_Msg_MapDeviceCapabilities =
                  (trFC_Msg_MapDeviceCapabilities*) f_poLoopBackMessageData->vpData;

            m_oMapDeviceCapabilitiesStatus.u8DeviceHandle = prFC_Msg_MapDeviceCapabilities->u8DeviceHandle;
            m_oMapDeviceCapabilitiesStatus.u16SupportedMessageTypes =
                  prFC_Msg_MapDeviceCapabilities->u16SupportedMessageTypes;
            m_oMapDeviceCapabilitiesStatus.u16SupportedMapFeatures =
                  prFC_Msg_MapDeviceCapabilities->u16SupportedMapFeatures;
            m_oMapDeviceCapabilitiesStatus.oActiveAccount.u16AccountHandle =
                  prFC_Msg_MapDeviceCapabilities->rFC_Msg_ActiveEmailAccount.u16AccountHandle;
            m_oMapDeviceCapabilitiesStatus.oActiveAccount.sAccountName.bSet(
                  prFC_Msg_MapDeviceCapabilities->rFC_Msg_ActiveEmailAccount.sAccountName.toUtf8().constData());
            m_oMapDeviceCapabilitiesStatus.e8MapConnectionState.enType =
                  (most_fi_tcl_e8_MapConnectionStatus::tenType)prFC_Msg_MapDeviceCapabilities->u8MapConnectionState;
         }

         // TODO: Update the property only when there is really a change in the attributes
         eUpdatePropertyToClients(MOST_MSGFI_C_U16_MAPDEVICECAPABILITIES);
      }
      break;

      // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
      case FC_MSG_LB_FID_MSG_NOTIFICATION_STATUS:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_MSG_NOTIFICATION_STATUS"));

         if (f_poFileSystemManager && f_poLoopBackMessageData && f_poLoopBackMessageData->vpData)
         {
            trFC_Msg_Notification *l_prFC_Msg_Notification =
                  (trFC_Msg_Notification*) f_poLoopBackMessageData->vpData;

            QString f_sObjectHandle(*(l_prFC_Msg_Notification->rMapNoitfyData.pacMessageHandle));
            FolderNode* l_poFolderNode = NULLPTR;

            l_prFC_Msg_Notification->u32MsgHandleUpper = f_poFileSystemManager->u32ConvertDbusHandleToUpperHandle(f_sObjectHandle);

            if (l_prFC_Msg_Notification->rMapNoitfyData.u8FolderType == most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX)
            {
               l_poFolderNode = f_poFileSystemManager->poSearchNodeByFolderName(MSG_DBUS_FOLDER_TYPE_INBOX);
            }
            else if (l_prFC_Msg_Notification->rMapNoitfyData.u8FolderType  == most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT)
            {
               l_poFolderNode = f_poFileSystemManager->poSearchNodeByFolderName(MSG_DBUS_FOLDER_TYPE_SENT);
            }

            if(l_poFolderNode)
            {
               l_prFC_Msg_Notification->u32MsgHandleLower = l_poFolderNode->m_u32ListHandle;
            }
            else
            {
               l_prFC_Msg_Notification->u32MsgHandleLower = 0;
            }

            // send the notification status to HMI
            vMsgNotificationstatus(l_prFC_Msg_Notification);

            delete l_prFC_Msg_Notification;
         }
      }
      break;

      case FC_MSG_LB_FID_PUSH_MESSAGE_STATUS:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_PUSH_MESSAGE_STATUS"));

         if(f_poLoopBackMessageData)
         {
            SendMessage* sendMessage;
            sendMessage = SendMessage::poGetSendMessageInstance();

            if(sendMessage)
            {
               sendMessage->vSendMessageReturn(f_poLoopBackMessageData->u8DeviceHandle, 0, 0, f_poLoopBackMessageData->ucStatus);
            }
         }
      }
      break;

      case FC_MSG_LB_FID_SET_MSG_STATUS:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_SET_MSG_STATUS"));

         if(f_poLoopBackMessageData)
         {
            if(SET_MSG_STATUS_DELETED_CB == f_poLoopBackMessageData->u16Private)
            {
               if(SUCCESS == f_poLoopBackMessageData->ucStatus)
               {
                  if(f_poFileSystemManager)
                  {
                     f_poFileSystemManager->vDeleteMessageMethodReturn(SUCCESS);
                  }
               }
               else
               {
                  fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_Delete_Error();
               }
            }
            else
            {
               if(f_poFileSystemManager)
               {
                  f_poFileSystemManager->vSetMsgReadStatusReturn(f_poLoopBackMessageData->ucStatus);
               }
            }
         }
      }
      break;

      case FC_MSG_LB_FID_CONTINUE_MAP_SESSION:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_CONTINUE_MAP_SESSION"));

         if(f_poLoopBackMessageData)
         {
            EvolutionDbusClient* client = EvolutionDbusClient::getDeviceInstance(f_poLoopBackMessageData->u8DeviceHandle);
            if(client)
            {
               client->messagingMapSessionConnected();
            }
         }
      }
      break;

      case FC_MSG_LB_FID_DEVICE_MAP_CONNECTION_STATUS:
      {
         ETG_TRACE_USR3(("FC_MSG_LB_FID_DEVICE_MAP_CONNECTION_STATUS"));

         if(f_poLoopBackMessageData)
         {
            if(TRUE == f_poLoopBackMessageData->ucStatus)
            {
               //Device Connected
               if(f_poLoopBackMessageData->vpData)
               {
                  EvolutionDbusClient* client = EvolutionDbusClient::getDeviceInstance(f_poLoopBackMessageData->u8DeviceHandle);
                  if(client)
                  {
                     client->setClientSessionPath((tString)f_poLoopBackMessageData->vpData);

                     if(client->createMessageAccessProxy())
                     {
                        //client->messagingMapSessionConnected();
                     }
                  }
               }
            }
            else
            {
               //CMG3G-13910 Remove device specific signature text when device is disconnected
               //vEraseDeviceSignatureText(f_poLoopBackMessageData->u8DeviceHandle);

               //Device Disconnected
               EvolutionDbusClient::messagingDeviceDisconnected(f_poLoopBackMessageData->u8DeviceHandle);

               //Check and Stop Read Message Aloud if it is in progress
               ReadMessageAloud* oReadMessageAloud = ReadMessageAloud::poGetInstance();

               if (oReadMessageAloud)
               {
                  oReadMessageAloud->vHandleReadMessageAloudOnDeviceDisconnection(f_poLoopBackMessageData->u8DeviceHandle);
               }

               std::string deviceAddress;
               std::map< tU8, std::string >::iterator it2 = fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.find(f_poLoopBackMessageData->u8DeviceHandle);
               if(it2 != fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.end())
               {
                  deviceAddress = it2->second;
                  if(TRUE != deviceAddress.empty())
                  {
                     MessagingSettings* l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();
                     DbusClientIf client = DbusClientInterface::getInstance().getClientInterface();
                     tBool bMapOnOff = true;
                     if (l_poMessageSettings)
                     {
                        bMapOnOff = l_poMessageSettings->bGetDeviceSpecificSetting(f_poLoopBackMessageData->u8DeviceHandle, MAP_ON_OFF_SETTING);
                        ETG_TRACE_USR4(("Device Instance Available"));
                        if (bMapOnOff)
                        {
                           if(DBUS_CLIENT_IF_EVOLUTION == client)
                           {
                              ETG_TRACE_USR4(("Map Connection Sequence initiated again"));
                              EvolutionDbusClient::messagingDeviceConnectionStatus(f_poLoopBackMessageData->u8DeviceHandle, deviceAddress);
                           }
                        }
                        else
                        {
                           ETG_TRACE_USR4(("Text Message Capability Switched OFF in HMI"));
                        }
                     }
                  }
               }
            }
         }
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("vOnLoopbackServiceHelper(). Unknown u32MsgContent %u.", static_cast<unsigned int>(f_U32MsgContent)));
      }
   }
}

fc_messaging_tclService_Messaging* fc_messaging_tclService_Messaging::poGetInstance()
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging:: poGetInstance"));
   return (m_poMessagingService);
}

// Task CMG3G-7698: IS2201_API for device capabilities@FC_Messaging
/*******************************************************************************
 *
 * FUNCTION: vResetMapDeviceCapabilities
 *
 * DESCRIPTION: Resets the Map device capabilities to default values
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vResetMapDeviceCapabilities()
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vResetMapDeviceCapabilities"));

   m_oMapDeviceCapabilitiesStatus.u8DeviceHandle = 0;
   m_oMapDeviceCapabilitiesStatus.u16SupportedMessageTypes = FC_MSG_NO_MSG_TYPES_SUPPORTED;
   m_oMapDeviceCapabilitiesStatus.u16SupportedMapFeatures = FC_MSG_NO_FEATURES_SUPPORTED;
   m_oMapDeviceCapabilitiesStatus.oActiveAccount.u16AccountHandle = 0;
   m_oMapDeviceCapabilitiesStatus.oActiveAccount.sAccountName.bSet("");
   //CMG3G-8286, CMG3G-8160, CMG3G-8287- ConnectMapProfile and MapConnectionStatus
   m_oMapDeviceCapabilitiesStatus.e8MapConnectionState.enType = (most_fi_tcl_e8_MapConnectionStatus::tenType)FC_MSG_MAP_DISCONNECTED;
}

/*******************************************************************************
 *
 * FUNCTION: vResetMapDeviceCapabilitiesExtended
 *
 * DESCRIPTION: Resets the Map device capabilities list to empty
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vResetMapDeviceCapabilitiesExtended()
{
   ETG_TRACE_USR4(("vResetMapDeviceCapabilitiesExtended() called"));

   m_oMapDeviceCapabilitiesExtendedStatus.oMapDeviceCapabilitiesList.oItems.clear();
}

/*******************************************************************************
 *
 * FUNCTION: vResetMessagingDeviceConnectionStatus
 *
 * DESCRIPTION: Resets the Messaging Device Connection to default values
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vResetMessagingDeviceConnectionStatus()
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vResetMessagingDeviceConnectionStatus"));

   m_oMessagingDeviceConnection_Status.u8DeviceHandle = 0;
   m_oMessagingDeviceConnection_Status.bEmailSupport = FALSE;
   m_oMessagingDeviceConnection_Status.bSMSSupport = FALSE;
   m_oMessagingDeviceConnection_Status.bMMSSupport = FALSE;
   m_oMessagingDeviceConnection_Status.bMessagingMemoryStatus = FALSE;
   m_oMessagingDeviceConnection_Status.bUpdateInboxSupport = FALSE;
   m_oMessagingDeviceConnection_Status.oActiveAccount.u16AccountHandle = 0;
   m_oMessagingDeviceConnection_Status.oActiveAccount.sAccountName.bSet("");
}

/*******************************************************************************
 *
 * FUNCTION: vResetMessageNotificationStatus
 *
 * DESCRIPTION: Resets the Message Notification Status to default values
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vResetMessageNotificationStatus()
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vResetMessageNotificationStatus"));

   m_oMsgNotificationStatus.e8NotificationType.enType = most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_NEW_MSG;
   m_oMsgNotificationStatus.e8MessageType.enType = most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_SMS;
   m_oMsgNotificationStatus.oMessageHandle.u8DeviceHandle = 0;
   m_oMsgNotificationStatus.oMessageHandle.u32MsgHandleUpper = 0;
   m_oMsgNotificationStatus.oMessageHandle.u32MsgHandleLower = 0;
}

/*******************************************************************************
 *
 * FUNCTION: vResetMessageListChangeStatus
 *
 * DESCRIPTION: Resets the Message List Change Status to default values
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vResetMessageListChangeStatus()
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vResetMessageListChangeStatus"));

   m_oMessageListChange_Status.u16ListHandle = 0;
   m_oMessageListChange_Status.e8ListChangeType.enType = most_fi_tcl_e8_MsgListChangeType::FI_EN_E8LCH_CONTENT_CHANGED;
   m_oMessageListChange_Status.u16ListLength = 0;
   m_oMessageListChange_Status.u32NumItems = 0;
   m_oMessageListChange_Status.oChangeItems.u32Items.clear();
}

/*******************************************************************************
 *
 * FUNCTION: vResetSignatureTextStatus
 *
 * DESCRIPTION: Resets the Signature Text Status to default values
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vResetSignatureTextStatus()
{
   m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.clear();
   m_oSignatureSettingStatus.sSignatureText.bSet("");
}

/*******************************************************************************
 *
 * FUNCTION: vEraseDeviceSignatureText
 *
 * DESCRIPTION: Erases the device specific signature text when device is disconnected
 *
 * PARAMETER: IN : deviceHandle : disconnected device's deviceHandle
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
//This function is meant for device specific signature setting.
tVoid fc_messaging_tclService_Messaging::vEraseDeviceSignatureText(tU8 deviceHandle)
{
   tBool isSuccess = false;

   if(0 == deviceHandle)
   {
      m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.clear();
      isSuccess = true;
   }
   else
   {
      for (tU8 index = 0; index < m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.size(); index++)
      {
         if (deviceHandle == m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems[index].u8DeviceHandle)
         {
            m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.erase(m_oSignatureSettingExtendedStatus.oSignatureTextStream.oItems.begin() + index);
            isSuccess = true;
            break;
         }
      }
   }

   if(isSuccess)
   {
      eUpdatePropertyToClients(MOST_MSGFI_C_U16_SIGNATURESETTINGEXTENDED);
   }
}

//This function has to be invoked only for signature setting that is not device specific.
tString fc_messaging_tclService_Messaging::getSignatureText()
{
   return m_oSignatureSettingStatus.sSignatureText.szValue;
}

// Bugfix GMMY17-5280: The new text message received while reading previous
// messages is not shown in the INBOX for the SMS text application.
/*******************************************************************************
 *
 * FUNCTION: vHandle_NewMessageNotification
 *
 * DESCRIPTION: Handles MapNotification events from Work queue
 *
 * PARAMETER:
 *    IN:   prJob    : Job to be processed
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_NewMessageNotification(
      amt_tclServiceData* poMessage, struct Job* prJob)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vHandle_NewMessageNotification"));

   (tVoid) poMessage;

   if (prJob && prJob->m_pvMethodStart)
   {
      most_msgfi_tclMsgCreateMessageListMethodStart* poCreateMessageListMethodStart =
            (most_msgfi_tclMsgCreateMessageListMethodStart*) prJob->m_pvMethodStart;

      if (DbusClientInterface::getInstance().isDbusClientAvailable(poCreateMessageListMethodStart->u8DeviceHandle))
      {
         if (0 != MessagingMutex::messagingMutexTrylock(prJob))
         {
            tVoid* pCustodialLose = prJob; //To remove LINT warning : Custodial Pointer
            (tVoid) pCustodialLose;
            return; // Didn't get the lock :(
         }
         DbusClientInterface::getInstance().handleNewMessageNotification(poCreateMessageListMethodStart);
      }
   }
}

/*******************************************************************************
 *
 * FUNCTION: vHandle_GetMessageforSubject
 *
 * DESCRIPTION: Handles SubjectUpdate events from Work queue
 *
 * PARAMETER:
 *    IN:   prJob    : Job to be processed
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_GetMessageforSubject(
      amt_tclServiceData* poMessage, struct Job* prJob)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vHandle_GetMessageforSubject"));

   (tVoid) poMessage;

   if (prJob && prJob->m_pvMethodStart)
   {
      most_fi_tcl_MsgMessageHandle* l_messageHandle =
            (most_fi_tcl_MsgMessageHandle*) prJob->m_pvMethodStart;

      if (DbusClientInterface::getInstance().isDbusClientAvailable(l_messageHandle->u8DeviceHandle))
      {
         if (0 != MessagingMutex::messagingMutexTrylock(prJob))
         {
            tVoid* pCustodialLose = prJob; //To remove LINT warning : Custodial Pointer
            (tVoid) pCustodialLose;
            return; // Didn't get the lock :(
         }

         QString serviceName;
         serviceName = DbusClientInterface::getInstance().getCurrentActiveService(l_messageHandle->u8DeviceHandle);
         FileSystemManager* l_poFileSystemManager = DbusClientInterface::getInstance().getFileSystemManager(l_messageHandle->u8DeviceHandle,
                     serviceName);
         if(l_poFileSystemManager)
         {
            l_poFileSystemManager->vGetMessage(*l_messageHandle,e8SubjectUpdate);
         }
      }
   }
}

/*******************************************************************************
 *
 * FUNCTION: vHandle_ListFolders
 *
 * DESCRIPTION: Handles ListFolders from Work queue
 *
 * PARAMETER:
 *    IN:   prJob    : Job to be processed
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vHandle_ListFolders(
      amt_tclServiceData* poMessage, struct Job* prJob)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vHandle_ListFolders"));

   (tVoid) poMessage;

   if (prJob)
   {
      if (DbusClientInterface::getInstance().isDbusClientAvailable(prJob->u8DeviceHandle))
      {
         DbusClientInterface::getInstance().sendFolderListRequest(prJob->u8DeviceHandle, FC_MSG_FOLDER_LIST_REQUESTED_FOR_SMSMMS_SERVICE);

         //tVoid* pCustodialLose = prJob; //To remove LINT warning : Custodial Pointer
         //(tVoid) pCustodialLose;
      }
   }
}

// Task CMG3G-7697 - IS2200_Switching MAP ON/OFF@FC_Messaging
/*******************************************************************************
 *
 * FUNCTION: vSwitchMapOnOff
 *
 * DESCRIPTION:Directs to MAP ON/OFF based on the request from client
 *
 * PARAMETER: bMapInput- Specifies the input from the Client
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vSwitchMapOnOff(tBool bMapInput)
{
   DbusClientIf client = DbusClientInterface::getInstance().getClientInterface();
   if(DBUS_CLIENT_IF_EVOLUTION == client)
   {
      if (TRUE == bMapInput)
         EvolutionDbusClient::switchMapOn();
      else
         EvolutionDbusClient::switchMapOff();
   }
}

// Task CMG3G-7699: IS2202_message send notification (network level)@FC_Messaging
/*******************************************************************************
 *
 * FUNCTION: vMsgNotificationstatus
 *
 * DESCRIPTION: 
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclService_Messaging::vMsgNotificationstatus(trFC_Msg_Notification *prFC_Msg_Notification)
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vMsgNotificationstatus"));
   
   ETG_TRACE_USR4(("vMsgNotificationstatus - Notification Type: %d", prFC_Msg_Notification->u8MessgeNotificationType));
   ETG_TRACE_USR4(("vMsgNotificationstatus - Msg Type: %d", prFC_Msg_Notification->u8MessgeType));
   ETG_TRACE_USR4(("vMsgNotificationstatus - DeviceHandle: %d", prFC_Msg_Notification->u8DeviceHandle));
   ETG_TRACE_USR4(("vMsgNotificationstatus - MsgHandleUpper: %d", prFC_Msg_Notification->u32MsgHandleUpper));
   ETG_TRACE_USR4(("vMsgNotificationstatus - MsgHandleLower: %d", prFC_Msg_Notification->u32MsgHandleLower));
   
   m_oMsgNotificationStatus.e8NotificationType.enType = (most_fi_tcl_e8_MsgNotificationType::tenType)prFC_Msg_Notification->u8MessgeNotificationType;
   m_oMsgNotificationStatus.e8MessageType.enType = (most_fi_tcl_e8_MsgMessageType::tenType)prFC_Msg_Notification->u8MessgeType;
   m_oMsgNotificationStatus.oMessageHandle.u8DeviceHandle = prFC_Msg_Notification->u8DeviceHandle;
   m_oMsgNotificationStatus.oMessageHandle.u32MsgHandleUpper = prFC_Msg_Notification->u32MsgHandleUpper;
   m_oMsgNotificationStatus.oMessageHandle.u32MsgHandleLower = prFC_Msg_Notification->u32MsgHandleLower;

   eUpdatePropertyToClients(MOST_MSGFI_C_U16_MESSAGENOTIFICATION);
}



tVoid fc_messaging_tclService_Messaging::vUpdateDeviceSpecificSettingsProperties()
{
   eUpdatePropertyToClients(MOST_MSGFI_C_U16_MESSAGINGONOFFLIST); //Map OnOff
   eUpdatePropertyToClients(MOST_MSGFI_C_U16_TEXTRINGTONEONOFFLIST); //Text Ringtone OnOff
   eUpdatePropertyToClients(MOST_MSGFI_C_U16_SIGNATURESETTINGEXTENDED); //SignatureSettingExtended
   eUpdatePropertyToClients(MOST_MSGFI_C_U16_AUTOREPLYONOFFLIST); //AutoReply OnOff
   eUpdatePropertyToClients(MOST_MSGFI_C_U16_AUTOREPLYTEXTLIST); //AutoReplyText
}

