/*******************************************************************************
 *
 * FILE:          FC_Messaging_clienthandler_SDS.cpp
 *
 * SW-COMPONENT:  FC_Messaging application
 *
 * PROJECT:
 *
 * DESCRIPTION:   CCA client-handler for the service TTS_SDS.
 *
 * AUTHOR:        Sandeep Arjun
 *
 * COPYRIGHT:    (c) 2014  Robert Bosch
 *
 *******************************************************************************/
/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "FC_Messaging_main.h"
#include "FC_Messaging_service_Messaging.h"
#include "GenericWrapper/FC_Messaging_GenericWrapper.h"
#include "ReadMessageAloud/Gen3/FC_Messaging_ReadMessageAloud.h"
#include "FC_Messaging_clienthandler_SDS.h"

#define SDS_FI_S_IMPORT_INTERFACE_SDS_TTSFI_TYPES
#define SDS_FI_S_IMPORT_INTERFACE_SDS_TTSFI_FUNCTIONIDS
#define SDS_FI_S_IMPORT_INTERFACE_SDS_TTSFI_ERRORCODES
#define SDS_FI_S_IMPORT_INTERFACE_SDS_TTSFI_SERVICEINFO

#include "sds_fi_gen_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_clienthandler_SDS.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/

BEGIN_MSG_MAP(fc_messaging_ClientHandler_SDS, ahl_tclBaseWork)

// Add your ON_MESSAGE_SVCDATA() macros here to define which corresponding
// method should be called on receiving a specific message.

ON_MESSAGE(SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
      vHandle_SynthesizeSpeak)

ON_MESSAGE(SDS_TTSFI_C_U16_ABORTPROMPT,
      vHandle_AbortPrompt)

ON_MESSAGE(SDS_TTSFI_C_U16_PREPAREVOICEOUTPUT,
      vHandle_PrepareVoiceOutput)

END_MSG_MAP()

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
 *
 * FUNCTION: fc_messaging_ClientHandler_SDS::
 *             fc_messaging_ClientHandler_SDS(fc_messaging_tclApp* poMainAppl)
 *
 * DESCRIPTION: Constructor.
 *
 *              Create an object of the base class
 *              ahl_tclBaseOneThreadClientHandler with a pointer to this
 *              application, the to be used service identifier and the service
 *              version as parameters.
 *
 * PARAMETER: [IN] poMainAppl = Pointer to the object of this application.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_messaging_ClientHandler_SDS::fc_messaging_ClientHandler_SDS(fc_messaging_tclApp* poMainAppl) :
ahl_tclBaseOneThreadClientHandler(poMainAppl, CCA_C_U16_SRV_SDS_TTS,
      SDS_TTSFI_C_U16_SERVICE_MAJORVERSION, SDS_TTSFI_C_U16_SERVICE_MINORVERSION)
{
   m_poMainApp = (fc_messaging_tclApp*) poMainAppl;
}

/*******************************************************************************
 *
 * FUNCTION: fc_messaging_ClientHandler_SDS::
 *             ~fc_messaging_ClientHandler_SDS(tVoid)
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_messaging_ClientHandler_SDS::~fc_messaging_ClientHandler_SDS(tVoid)
{
   m_poMainApp = NULLPTR;
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_ClientHandler_SDS::vOnServiceAvailable()
 *
 * DESCRIPTION: This function is called by the CCA framework when the service
 *              this client-handler has registered for has become available.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable().
 *******************************************************************************/
tVoid fc_messaging_ClientHandler_SDS::vOnServiceAvailable()
{
   ETG_TRACE_USR4(("fc_messaging_ClientHandler_SDS::vOnServiceAvailable() entered : AppID = %u.", u16GetServerAppID()));
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_ClientHandler_SDS::vOnServiceUnavailable()
 *
 * DESCRIPTION: This function is called by the CCA framework when the service
 *              this client-handler has registered for has become unavailable.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable().
 *******************************************************************************/
tVoid fc_messaging_ClientHandler_SDS::vOnServiceUnavailable()
{
   ETG_TRACE_USR4(("fc_messaging_ClientHandler_SDS::vOnServiceUnavailable() entered : AppID = %u.", u16GetServerAppID()));
}


/*******************************************************************************
 *
 * FUNCTION:     tVoid  fc_messaging_ClientHandler_SDS::bPrepareVoiceOutput_MethodStart()
 *
 * DESCRIPTION:  Handles in Posting message to PrepareVoiceOutput Method Start
 *
 * PARAMETER:    None.
 *
 * RETURNVALUE:  None.
 *
 *******************************************************************************/

tBool fc_messaging_ClientHandler_SDS::bPrepareVoiceOutput_MethodStart()
{
   ETG_TRACE_USR4(("fc_messaging_ClientHandler_SDS::bPrepareVoiceOutput_MethodStart"));

   sds_ttsfi_tclMsgPrepareVoiceOutputMethodStart oNewFiDataObject;
   tBool bResult = FALSE;

   fi_tclVisitorMessage oVisitorMessage(oNewFiDataObject, SDS_TTSFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oVisitorMessage, 0, SDS_TTSFI_C_U16_PREPAREVOICEOUTPUT, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

   if (bIfServiceAvailable())
   {
      enResult = m_poMainApp->enPostMessage(&oVisitorMessage, TRUE);
   }
   else
   {
      ETG_TRACE_ERR(("bPrepareVoiceOutput_MethodStart() SDS service unavailable"));

      enResult = AIL_EN_N_REGISTRATION_DENIED_SERVICE_NOT_FOUND;
   }

   if (enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("enPostMessage failed for bPrepareVoiceOutput_MethodStart()"));
      if (!oVisitorMessage.bDelete())
      {
         ETG_TRACE_ERR(("bPrepareVoiceOutput_MethodStart() Message deletion failed"));
      }
   }
   else
   {
      bResult = TRUE;
   }
   oNewFiDataObject.vDestroy();

   return bResult;
}

/*******************************************************************************
 *
 * FUNCTION:     tVoid  fc_messaging_ClientHandler_SDS::bSynthesizeSpeak_MethodStart
 *              (poInternalDataObj)
 *
 * DESCRIPTION:  Handles in Posting message to SynthesizeSpeak Method Start
 *
 * PARAMETER:    Pointer to the Message
 *
 * RETURNVALUE:  None.
 *
 *******************************************************************************/

tBool fc_messaging_ClientHandler_SDS::bSynthesizeSpeak_MethodStart(most_fi_tcl_String sText)
{
   ETG_TRACE_USR4(("fc_messaging_ClientHandler_SDS::bSynthesizeSpeak_MethodStart sText :%s:", sText.szValue));

   tBool bResult = FALSE;

   sds_ttsfi_tclMsgSynthesizeSpeakMethodStart oNewFiDataObject;
   sds_fi_tcl_e8_TextStyleOption e8TextStyleOption;

   e8TextStyleOption.enType =  sds_fi_tcl_e8_TextStyleOption::FI_EN_SMS;

   oNewFiDataObject.TextAndTags.bSet(sText.szValue, sds_fi_tclString::FI_EN_UTF8);
   oNewFiDataObject.TextStyle = e8TextStyleOption; //Plain/SMS?
   oNewFiDataObject.AudioOutputLocation.u8Value = 255; //Same as in SDS, check which speaker should be used
   oNewFiDataObject.ExpDate = 0x0000; //Same as in SDS
   oNewFiDataObject.StatusInfo.enType = sds_fi_tcl_e8_StatusInfoOption::FI_EN_NOINFO;
   oNewFiDataObject.LastPackage = TRUE; //This is the only job

   fi_tclVisitorMessage oVisitorMessage(oNewFiDataObject, SDS_TTSFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oVisitorMessage, 0, SDS_TTSFI_C_U16_SYNTHESIZESPEAK, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

   if (bIfServiceAvailable())
   {
      enResult = m_poMainApp->enPostMessage(&oVisitorMessage, TRUE);
   }
   else
   {
      ETG_TRACE_ERR(("bSynthesizeSpeak_MethodStart() SDS service unavailable"));

      enResult = AIL_EN_N_REGISTRATION_DENIED_SERVICE_NOT_FOUND;
   }

   if (enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("enPostMessage failed for bSynthesizeSpeak_MethodStart()"));
      if (!oVisitorMessage.bDelete())
      {
         ETG_TRACE_ERR(("bSynthesizeSpeak_MethodStart() Message deletion failed"));
      }
   }
   else
   {
      bResult = TRUE;
   }
   oNewFiDataObject.vDestroy();

   return bResult;
}


/*******************************************************************************
 *
 * FUNCTION:     tVoid  fc_messaging_ClientHandler_SDS::bAbortPrompt_MethodStart
 *               (poInternalDataObj)
 *
 * DESCRIPTION:  Handles in Posting message to AbortPrompt or
 *           PausePrompt or ContinousPrompt Method Start based on the PlayMode.
 *
 * PARAMETER:    Pointer to the Message
 *
 * RETURNVALUE:  None.
 *
 *******************************************************************************/
tBool fc_messaging_ClientHandler_SDS::bAbortPrompt_MethodStart()
{
   ETG_TRACE_USR4(("fc_messaging_ClientHandler_SDS::bAbortPrompt_MethodStart"));

   sds_ttsfi_tclMsgAbortPromptMethodStart oNewFiDataObject;
   tBool bResult = FALSE;

   oNewFiDataObject.AbortPromptOption.enType = sds_fi_tcl_e8_PromptAbortOption::FI_EN_IMMEDIATE;

   fi_tclVisitorMessage oVisitorMessage(oNewFiDataObject, SDS_TTSFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oVisitorMessage, 0, SDS_TTSFI_C_U16_ABORTPROMPT, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);

   ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

   if (bIfServiceAvailable())
   {
      enResult = m_poMainApp->enPostMessage(&oVisitorMessage, TRUE);
   }
   else
   {
      ETG_TRACE_ERR(("bAbortPrompt_MethodStart() SDS service unavailable"));

      enResult = AIL_EN_N_REGISTRATION_DENIED_SERVICE_NOT_FOUND;
   }

   if (enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("enPostMessage failed for bAbortPrompt_MethodStart()"));
      if (!oVisitorMessage.bDelete())
      {
         ETG_TRACE_ERR(("bAbortPrompt_MethodStart() Message deletion failed"));
      }
   }
   else
   {
      bResult = TRUE;
   }
   oNewFiDataObject.vDestroy();

   return bResult;
}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_ClientHandler_SDS:: vHandle_PrepareVoiceOutput
 *        (amt_tclServiceData* poMessage)
 *
 * DESCRIPTION: Handler function to process the arrival of PrepareVoiceOutput Method Result.
 *
 * PARAMETER: Pointer to the Message.
 *
 * RETURNVALUE: -
 *
 *******************************************************************************/
tVoid fc_messaging_ClientHandler_SDS::vHandle_PrepareVoiceOutput(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_messaging_ClientHandler_SDS::vHandle_PrepareVoiceOutput"));

   tU8 u8OpCode = poMessage->u8GetOpCode();

   switch (u8OpCode)
   {
      case AMT_C_U8_CCAMSG_OPCODE_ERROR:
      {
         amt_tclServiceDataError oErrorMsg(poMessage);
         tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
         ETG_TRACE_ERR(("vHandle_PrepareVoiceOutput - PrepareVoiceOutput Failed!! Error :%d:", u16ErrorCode));
         m_poMainApp->poGetGenericWrapper()->vHandle_PrepareVoiceOutput_Reponse(FALSE);
      }
      break;

      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
      {
         ETG_TRACE_ERR(("vHandle_PrepareVoiceOutput - PrepareVoiceOutput Success!!"));
         m_poMainApp->poGetGenericWrapper()->vHandle_PrepareVoiceOutput_Reponse(TRUE);
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("vHandle_PrepareVoiceOutput() : Invalid OpCode = %d received", poMessage->u8GetOpCode()));
      }
   }
}



/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_ClientHandler_SDS:: vHandle_SynthesizeSpeak_First
 *        (amt_tclServiceData* poMessage)
 *
 * DESCRIPTION: Handler function to process the arrival of SynthesizeSpeak Method Result First.
 *
 * PARAMETER: Pointer to the Message.
 *
 * RETURNVALUE: -
 *
 *******************************************************************************/
tVoid fc_messaging_ClientHandler_SDS::vHandle_SynthesizeSpeak(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_messaging_ClientHandler_SDS::vHandle_SynthesizeSpeak"));

   tU8 u8OpCode = poMessage->u8GetOpCode();

   switch (u8OpCode)
   {
      case AMT_C_U8_CCAMSG_OPCODE_ERROR :
      {
         amt_tclServiceDataError oErrorMsg(poMessage);
         tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
         ETG_TRACE_ERR(("vHandle_SynthesizeSpeak - SynthesizeSpeak Failed!! Error Code :%d:", u16ErrorCode));
         m_poMainApp->poGetGenericWrapper()->vHandle_SynthesizeSpeak_Reponse(FC_MSG_SYNTHESIZESPEAKRESULT_ERROR);
      }
      break;

      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTFIRST :
      {
         sds_ttsfi_tclMsgSynthesizeSpeakMethodResultFirst oFiDataObject;
         fi_tclVisitorMessage oVisitorMsg(poMessage);

         if (oVisitorMsg.s32GetData(oFiDataObject, SDS_TTSFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
         {
            //m_oJobProgress = oFiDataObject.JobProgress;
            sds_fi_tcl_e8_JobStatus oJobStatus = oFiDataObject.JobProgress.JobStatus;
            ETG_TRACE_ERR(("vHandle_SynthesizeSpeak JobStatus :%d:", oJobStatus.enType));
            m_poMainApp->poGetGenericWrapper()->vHandle_SynthesizeSpeak_Reponse(FC_MSG_SYNTHESIZESPEAKRESULT_FIRST);
         }
         oFiDataObject.vDestroy();
      }
      break;

      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTMIDDLE :
      {
         sds_ttsfi_tclMsgSynthesizeSpeakMethodResultMiddle oFiDataObject;
         fi_tclVisitorMessage oVisitorMsg(poMessage);

         if (oVisitorMsg.s32GetData(oFiDataObject, SDS_TTSFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
         {
            sds_fi_tcl_e8_JobStatus oJobStatus = oFiDataObject.JobProgress.JobStatus;
            ETG_TRACE_ERR(("vHandle_SynthesizeSpeak JobStatus :%d:", oJobStatus.enType));
            m_poMainApp->poGetGenericWrapper()->vHandle_SynthesizeSpeak_Reponse(FC_MSG_SYNTHESIZESPEAKRESULT_MIDDLE);
         }
         oFiDataObject.vDestroy();
      }
      break;

      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULTLAST :
      {
         sds_ttsfi_tclMsgSynthesizeSpeakMethodResultLast oFiDataObject;
         fi_tclVisitorMessage oVisitorMsg(poMessage);

         if (oVisitorMsg.s32GetData(oFiDataObject, SDS_TTSFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
         {
            sds_fi_tcl_e8_JobStatus oJobStatus = oFiDataObject.JobProgress.JobStatus;
            ETG_TRACE_ERR(("vHandle_SynthesizeSpeak JobStatus :%d:", oJobStatus.enType));
            m_poMainApp->poGetGenericWrapper()->vHandle_SynthesizeSpeak_Reponse(FC_MSG_SYNTHESIZESPEAKRESULT_LAST);
         }
         oFiDataObject.vDestroy();
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("vHandle_SynthesizeSpeak Invalid OpCode = %d received", poMessage->u8GetOpCode()));
      }
   }
}
/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_ClientHandler_SDS::vHandle_AbortPrompt
 *        (amt_tclServiceData* poMessage)
 *
 * DESCRIPTION: Handler function to process the arrival of AbortPrompt Method Result.
 *
 * PARAMETER: Pointer to the Message.
 *
 * RETURNVALUE: -
 *
 *******************************************************************************/

tVoid fc_messaging_ClientHandler_SDS::vHandle_AbortPrompt(amt_tclServiceData* poMessage)
{
   tU8 u8OpCode = poMessage->u8GetOpCode();

   switch (u8OpCode)
   {
      case AMT_C_U8_CCAMSG_OPCODE_ERROR:
      {
         ETG_TRACE_ERR(("vHandle_AbortPrompt:: AbortPrompt Failed!!"));

         //Should be same as SynthesizeSpeak_MethodResult_Last
         amt_tclServiceDataError oErrorMsg(poMessage);
         tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();
         ETG_TRACE_ERR(("vHandle_AbortPrompt - AbortPrompt Failed!! Error :%d:", u16ErrorCode));
         m_poMainApp->poGetGenericWrapper()->vHandle_AbortPrompt_Response(FALSE);
      }
      break;

      case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
      {
         ETG_TRACE_USR4(("vHandle_AbortPrompt - AbortPrompt Success!!"));
         m_poMainApp->poGetGenericWrapper()->vHandle_AbortPrompt_Response(TRUE);
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("vHandle_AbortPrompt() Invalid  OpCode = %d received", poMessage->u8GetOpCode()));
      }
   }
}
