/************************************************************************
* FILE:         Messaging_DefSetHandler.h
* PROJECT:      
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION: Example implementation of a system set listener.
* This handler sets the demodata value used for IncreaseDecrease demo
* back to default on a DefSet request.
* Take this class as a template for your own listener classes and
* adjust the parts marked with
* // +++ comment +++
*
* - ccademo_diaghandler.x              => // +++ Comment +++
* - ccademo_IncreaseDecreaseHandler.x  => // +++ Comment +++
* - Bluetooth_DefSetHandler.x            => // +++ Comment +++
* - ccademo_main.x                     => // +++diaglib+++ comment
* - ccademo_msgtypes.h                 => // +++diaglib+++ comment
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author                 | Modification
* 02.10.08  | CM-AI/PJ-FO45 Evers    | Inital version
* 06.06.11  | TMS Plischke           | Demoserver for GMG2
*
*************************************************************************/
// +++ Remove feature switch when not needed. Used to turn diaglib support on/off +++
// +++ Specify what part of diaglib you need in this file +++

#include "FC_Messaging_Debug.h"
#include "FC_Messaging_main.h"
#include "FC_Messaging_clienthandler_DefSetHandler.h"
#include "FC_Messaging_service_Messaging.h"

#define DIAGLIB_INCLUDE_SYSTEM_SET
#include "diaglib_if.h"

/* Needed for Trace */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_clienthandler_DefSetHandler.cpp.trc.h"
#endif

Messaging_DefSetHandler::Messaging_DefSetHandler(diaglib::tclServiceDiaglib* pDiaglibService) :
      mpDiaglibService(pDiaglibService)
{
   if (mpDiaglibService)
   {
      ETG_TRACE_COMP_THR(("Messaging_DefSetHandler::Messaging_DefSetHandler => "
            "vRegisterListener(SYSSETID_MESSAGING)"));
      mpDiaglibService->poGetSysSet()->vRegisterListener(SYSSETID_MESSAGING, this);
      ETG_TRACE_COMP_THR(("Messaging_DefSetHandler::Messaging_DefSetHandler => "
            "vRegisterListener(SYSSETID_ALL_COMP)"));
      mpDiaglibService->poGetSysSet()->vRegisterListener(SYSSETID_ALL_COMP, this);
      ETG_TRACE_COMP_THR(("Messaging_DefSetHandler::Messaging_DefSetHandler => "
            "vRegisterListener(SYSSETID_GROUP_HMI_CLEAR_ALL_PRIVATE_DATA)"));
      mpDiaglibService->poGetSysSet()->vRegisterListener(SYSSETID_GROUP_HMI_CLEAR_ALL_PRIVATE_DATA, this);
   }
   else
   {
      ETG_TRACE_USR4(("mpDiaglibService is NULL "));
   }
}

Messaging_DefSetHandler::~Messaging_DefSetHandler()
{
   mpDiaglibService = NULLPTR;
}

bool Messaging_DefSetHandler::ClearMessagingPrivateData()
{
   ETG_TRACE_USR4(("Messaging_DefSetHandler::ClearMessagingPrivateData"));
   fc_messaging_tclApp* poMainApp = fc_messaging_tclApp::poGetInstance();
   if (poMainApp)
   {
      fc_messaging_tclService_Messaging* poMessagingServiceInstance;
      poMessagingServiceInstance = poMainApp->m_poMessaging;
      if (poMessagingServiceInstance)
      {
         ETG_TRACE_USR4(("calling vHandle_ClearAllPrivateData"));
         return poMessagingServiceInstance->s32ClearAllPrivateData();
      }
      else
      {
         ETG_TRACE_USR4(("service Instance NULL"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("Application Instance NULL"));
   }
   return false;
}

// Fix for the task CMG3G-5152: "Generic behavior" for DefSet in Connectivity component - Messaging
tU32 Messaging_DefSetHandler::vOnSystemSet(tU32 u32SystemSetID, diaglib::tenSystemSetType u32SystemSetType, diaglib::tContext MsgContext)
{
   ETG_TRACE_USR3_THR(("Messaging_DefSetHandler::vOnSystemSet u32SystemSetID : %d, u32SystemSetType : %d", u32SystemSetID, u32SystemSetType));

   diaglib::tclParameterVector oTmpVec;
   diaglib::tenSystemSetResult oSysSetResult = diaglib::EN_SYSTEMSET_OK; // default response: success

   // Set values back to default, then send positive response

   switch (u32SystemSetID)
   {
      case SYSSETID_ALL_COMP:
      case SYSSETID_MESSAGING:
      {
         ETG_TRACE_COMP_THR(("vOnSystemSet => SYSSETID_ALL_COMP/SYSSETID_MESSAGING"));

         switch (u32SystemSetType)
         {
            case diaglib::EN_TYPE_DEFAULT_TEF:
            case diaglib::EN_TYPE_DEFAULT_HMI:
            {
               ETG_TRACE_COMP_THR(("vOnSystemSet => EN_TYPE_DEFAULT_TEF/EN_TYPE_DEFAULT_HMI"));

#ifdef VARIANT_S_FTR_ENABLE_GM_FEATURES
               // Since Predefined Message List and Message settings table
               // both are used only in GM projects no need to reset them
               ClearMessagingPrivateData();
#endif // VARIANT_S_FTR_ENABLE_GM_FEATURES

               break;
            }

            case diaglib::EN_TYPE_CODING: // Customer EOL
            {
               // No need to clear the data
               ETG_TRACE_COMP_THR(("vOnSystemSet => EN_TYPE_CODING"));

               break;
            }

            default:
            {
               ETG_TRACE_ERR_THR(("Messaging_DefSetHandler::vOnSystemSet => Not supported SystemSet request"));

               oSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;
               diaglib::trParameter paramError;
               paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
               paramError.u8Value = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
               oTmpVec.push_back(paramError);

               break;
            }
         }

         break;
      }

      case SYSSETID_GROUP_HMI_CLEAR_ALL_PRIVATE_DATA:
      {
         ETG_TRACE_COMP_THR(("vOnSystemSet => SYSSETID_GROUP_HMI_CLEAR_ALL_PRIVATE_DATA"));

         switch (u32SystemSetType)
         {
            case diaglib::EN_TYPE_DEFAULT_TEF:
            case diaglib::EN_TYPE_DEFAULT_HMI:
            case diaglib::EN_TYPE_CODING:
            {
               ETG_TRACE_COMP_THR(("vOnSystemSet => EN_TYPE_DEFAULT_TEF/EN_TYPE_DEFAULT_HMI/EN_TYPE_CODING"));

#ifdef VARIANT_S_FTR_ENABLE_GM_FEATURES
               // Since Predefined Message List and Message settings table
               // both are used only in GM projects no need to reset them
               ClearMessagingPrivateData();
#endif // VARIANT_S_FTR_ENABLE_GM_FEATURES

               break;
            }

            default:
            {
               ETG_TRACE_ERR_THR(("Messaging_DefSetHandler::vOnSystemSet => Not supported SystemSet request"));

               oSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;
               diaglib::trParameter paramError;
               paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
               paramError.u8Value = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
               oTmpVec.push_back(paramError);

               break;
            }
         }

         break;
      }

      default:
      {
         ETG_TRACE_ERR_THR(("Messaging_DefSetHandler::vOnSystemSet => Not supported ID request"));

         oSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;
         diaglib::trParameter paramError;
         paramError.enType = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
         paramError.u8Value = diaglib::EN_ERROR_ID_NOT_SUPPORTED;
         oTmpVec.push_back(paramError);

         break;
      }
   }

   // send Data back to Server
   if (mpDiaglibService)
   {
      if (TRUE == mpDiaglibService->poGetSysSet()->bSendSystemSetResult(oSysSetResult, oTmpVec, MsgContext))
      {
         ETG_TRACE_USR3_THR(("Messaging_DefSetHandler::vOnSystemSet => bSendSystemSetResult successful"));
      }
      else
      {
         ETG_TRACE_ERR_THR(("Messaging_DefSetHandler::vOnSystemSet => ERROR : bSendSystemSetResult failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("Messaging_DefSetHandler::vOnSystemSet => ERROR : mpDiaglibService == OSAL_NULL"));
   }

   return diaglib::U32_DIAGLIB_RETURN_OK;
}

tU32 Messaging_DefSetHandler::vOnSystemSetFinished(tU32 u32SystemSetID,
      diaglib::tenSystemSetType u32SystemSetType, diaglib::tContext MsgContext)
{
   ETG_TRACE_USR3_THR(("--> Messaging_DefSetHandler::vOnSystemSetFinished"));

   // +++ Reallow access to values that needed default setting, then send positive response +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32SystemSetID);
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32SystemSetType);
   /*
    Respond does not contain any data and is just an acknowledgement.
    */

   if (mpDiaglibService)
   {
      (void)mpDiaglibService->poGetSysSet()->bAcknowledgeSystemSetFinished(MsgContext);
   }

   ETG_TRACE_USR3_THR(("<-- Messaging_DefSetHandler::vOnSystemSetFinished"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}

tU32 Messaging_DefSetHandler::vOnSystemSetPrepare(tU32 u32SystemSetID,
      diaglib::tenSystemSetType u32SystemSetType, diaglib::tContext MsgContext)
{
   ETG_TRACE_USR3_THR(("--> Messaging_DefSetHandler::vOnSystemSetPrepare"));

   // +++ Stop access to values that need default setting, then send positive response +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32SystemSetID);
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32SystemSetType);

   /*
    Response can be one of the following:
    EN_SYSTEMSET_OK            -> Empty error vector

    Actually the diaglib already checks if ID is supported, so we don't need
    to do it here again.
    */
   diaglib::tclParameterVector oTmpVec;

   if (mpDiaglibService)
   {
      (void)mpDiaglibService->poGetSysSet()->bSendSystemSetResult((
            diaglib::EN_SYSTEMSET_OK), oTmpVec, MsgContext);
   }

   ETG_TRACE_USR3_THR(("<-- Messaging_DefSetHandler::vOnSystemSetPrepare"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}

tU32 Messaging_DefSetHandler::vOnSystemSetCheck(tU32 u32SystemSetID,
      diaglib::tenSystemSetType u32SystemSetType, diaglib::tContext MsgContext)
{
   ETG_TRACE_USR3_THR(("--> Messaging_DefSetHandler::vOnSystemSetCheck"));

   // +++ Check if all values are set to default. +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32SystemSetID);
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32SystemSetType);
   /*
    Response can be one of the following:
    EN_SYSTEMSET_OK            -> Empty error vector
    EN_SYSTEMSET_NOT_OK        -> List of wrong values

    diaglib already checks if ID is supported, so we don't need
    to do it here again.
    */
   diaglib::tclParameterVector oTmpVec;

   if (mpDiaglibService)
   {
      (void)mpDiaglibService->poGetSysSet()->bSendSystemSetCheckResult((
            diaglib::EN_SYSTEMSET_OK), oTmpVec, MsgContext);
   }

   ETG_TRACE_USR3_THR(("<-- Messaging_DefSetHandler::vOnSystemSetCheck"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}
