/*******************************************************************************
 *
 * FILE:          FC_Messaging_clienthandler_BluetoothSetings.cpp
 *
 * SW-COMPONENT:  FC_Messaging application
 *
 * PROJECT:       Bosch
 *
 * DESCRIPTION:   CCA client-handler for the service BluetoothSetings.
 *
 * AUTHOR:        Arun Choudhary
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "FC_Messaging_Debug.h"
#include "FC_Messaging_main.h"
#include "FC_Messaging_clienthandler_BluetoothSetings.h"
#include "DbusClient/FC_Messaging_EvoDbusClient.h"
#include "FC_Messaging_phonebook.h"
#include "FC_Messaging_StateMachine.h"
#include "FC_Messaging_WorkQueue.h"
#include "MsgList/FC_Messaging_Settings.h"
#include "Database/FC_Messaging_SQLite.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_clienthandler_BluetoothSetings.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

most_fi_tcl_String fc_messaging_tclClientHandler_BluetoothSetings::m_VehicleBTFriendlyName;
QList<tU8> fc_messaging_tclClientHandler_BluetoothSetings::m_u8DiscDeviceHandleList;
QMutex fc_messaging_tclClientHandler_BluetoothSetings::m_DeviceListmutex;
std::map< tU8, std::string > fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList;
std::vector< tU8 > fc_messaging_tclClientHandler_BluetoothSetings::m_u8ConnectedDevicesOrder;
std::vector< tU8 > fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList;
std::vector< tU8 > fc_messaging_tclClientHandler_BluetoothSetings::m_u8MAPConnRetryReqForMNSList;
/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/

BEGIN_MSG_MAP(fc_messaging_tclClientHandler_BluetoothSetings, ahl_tclBaseWork)

// Add your ON_MESSAGE_SVCDATA() macros here to define which corresponding
// method should be called on receiving a specific message.

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_DEVICELISTEXTENDED,
                  AMT_C_U8_CCAMSG_OPCODE_STATUS ,
                  vHandle_DeviceListExtended_Status )

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_VEHICLEBTFRIENDLYNAME,
                  AMT_C_U8_CCAMSG_OPCODE_STATUS,
                  vHandle_VehicleBTFriendlyName_Status )

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_VEHICLEBTADDRESS,
                  AMT_C_U8_CCAMSG_OPCODE_STATUS,
                  vHandle_VehicleBTAddress_Status )

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_CONNECTBLUETOOTHPROFILE,
                  AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
                  vHandle_ConnectBluetoothProfile_MethodResult )

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_CONNECTBLUETOOTHPROFILE,
                  AMT_C_U8_CCAMSG_OPCODE_ERROR,
                  vHandle_ConnectBluetoothProfile_MethodError )

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_DISCONNECTBLUETOOTHPROFILE,
                  AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,
                  vHandle_DisconnectBluetoothProfile_MethodResult )

         ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_DISCONNECTBLUETOOTHPROFILE,
                  AMT_C_U8_CCAMSG_OPCODE_ERROR,
                  vHandle_DisconnectBluetoothProfile_MethodError )

END_MSG_MAP()

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
 *
 * FUNCTION: fc_messaging_tclClientHandler_BluetoothSetings::
 *     fc_messaging_tclClientHandler_BluetoothSetings(fc_messaging_tclApp* poMainAppl)
 *
 * DESCRIPTION: Constructor.
 *
 *              Create an object of the base class
 *              ahl_tclBaseOneThreadClientHandler with a pointer to this
 *              application, the to be used service identifier and the service
 *              version as parameters.
 *
 * PARAMETER: [IN] poMainAppl = Pointer to the object of this application.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_messaging_tclClientHandler_BluetoothSetings::fc_messaging_tclClientHandler_BluetoothSetings(	fc_messaging_tclApp* poMainAppl) :
      ahl_tclBaseOneThreadClientHandler(
            /* Application Pointer          */poMainAppl,
            /* ID of used Service           */CCA_C_U16_SRV_FB_BLUETOOTHSETTINGS,
            /* MajorVersion of used Service */MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION,
            /* MinorVersion of used Service */MOST_BTSETFI_C_U16_SERVICE_MINORVERSION)
{
   ETG_TRACE_USR4(("fc_messaging_tclClientHandler_BluetoothSetings() entered"));

   vAddAutoRegisterForProperty( MOST_BTSETFI_C_U16_VEHICLEBTADDRESS);
   vAddAutoRegisterForProperty( MOST_BTSETFI_C_U16_DEVICELISTEXTENDED);
   vAddAutoRegisterForProperty( MOST_BTSETFI_C_U16_VEHICLEBTFRIENDLYNAME);
}

/*******************************************************************************
 *
 * FUNCTION: fc_messaging_tclClientHandler_BluetoothSetings::
 *             ~fc_messaging_tclClientHandler_BluetoothSetings(tVoid)
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_messaging_tclClientHandler_BluetoothSetings::~fc_messaging_tclClientHandler_BluetoothSetings(tVoid)
{
	ETG_TRACE_USR4(("~fc_messaging_tclClientHandler_BluetoothSetings() entered"));
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_tclClientHandler_BluetoothSetings::
 vOnServiceAvailable()
 *
 * DESCRIPTION: This function is called by the CCA framework when the service
 *              this client-handler has registered for has become available.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable().
 *******************************************************************************/
tVoid fc_messaging_tclClientHandler_BluetoothSetings::vOnServiceAvailable()
{
   ETG_TRACE_USR4(("fc_messaging_tclClientHandler_BluetoothSetings::vOnServiceAvailable() called"));
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_messaging_tclClientHandler_BluetoothSetings::
 vOnServiceUnavailable()
 *
 * DESCRIPTION: This function is called by the CCA framework when the service
 *              this client-handler has registered for has become unavailable.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable().
 *******************************************************************************/
tVoid fc_messaging_tclClientHandler_BluetoothSetings::vOnServiceUnavailable()
{
   ETG_TRACE_USR4(("fc_messaging_tclClientHandler_BluetoothSetings::vOnServiceUnavailable() called"));
}

/*******************************************************************************
 ** FUNCTION:   vHandle_DeviceListExtended_Status(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for DeviceListExtended property status
 *   
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_messaging_tclClientHandler_BluetoothSetings::vHandle_DeviceListExtended_Status(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgDeviceListExtendedStatus oDeviceListExtendedStatus;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(oDeviceListExtendedStatus, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->szQueryGetDevicesList(m_mapPairedDeviceList);
      vPrintPairedDeviceList();

      if (most_fi_tcl_e8_BTSetDeviceStatus::FI_EN_E8DEVICE_DELETED == oDeviceListExtendedStatus.oDeviceListChange.e8DeviceStatus.enType)
      {
         tU8 u8DeletedDeviceHandle = oDeviceListExtendedStatus.oDeviceListChange.u8DeviceHandle;
         ETG_TRACE_USR4(("Device Handle of the deleted Device = %d", u8DeletedDeviceHandle));
         vCheckAndUpdatePairedDeviceListinDB(false, u8DeletedDeviceHandle);

         ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status: Ignore Device Deleted"));
         oDeviceListExtendedStatus.vDestroy();
         return;
      }

      //Fix for NCG3D-10517: MAP profile not connected on startup when auto connection happens
      //On Startup, Device is connected & DeviceStatus updated to NO_CHANGE due to auto connection and the same status ignored. Hence MAP profile is not connected.
      //So process the NO_CHANGE DeviceStatus, when m_oConnectedDeviceList is empty (means that MAP profile is not connected).
     //else if ((oDeviceListExtendedStatus.oDeviceListChange.e8DeviceStatus.enType == most_fi_tcl_e8_BTSetDeviceStatus::FI_EN_E8DEVICE_NO_CHANGE) &&
     //       (TRUE != m_oConnectedDeviceList.empty()))
      //{
         //ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status: Ignore No Change"));
         //oDeviceListExtendedStatus.vDestroy();
         //return;
      //}

      ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status::Previous_Active_Handle: %d", DbusClientInterface::getInstance().getCurrentOutgoingDeviceId()));

      tU8 deviceListExtendedSize = static_cast <tU8>(oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems.size());
      if (0 != deviceListExtendedSize)
      {
         for (tU8 index = 0; index < deviceListExtendedSize; index++)
         {
            most_fi_tcl_BTSetDeviceListExtendedResultItem deviceListExtendedItem = oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems[index];

            vDeviceStatus(deviceListExtendedItem);
         }
      }
      else
      {
         vCheckAndUpdatePairedDeviceListinDB(false, 0);
      }
   }
   else
   {
      ETG_TRACE_ERR(("Unmarshaling failed for DeviceListExtended"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   oDeviceListExtendedStatus.vDestroy();
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vDeviceStatus(const most_fi_tcl_BTSetDeviceListExtendedResultItem& deviceListExtendedItem)
{
   ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status::sDeviceAddress: %s", deviceListExtendedItem.sDeviceAddress.szValue));
   ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status::bHFP: %d", deviceListExtendedItem.oDeviceProfileStatus.bHFP));
   ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status::bDeviceConnectedStatus: %d", deviceListExtendedItem.bDeviceConnectedStatus));
   ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status::bHFPSupport: %d", deviceListExtendedItem.oDeviceSupportedProfiles.bHFPSupport));
   ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status::bMAPSupport: %d", deviceListExtendedItem.oDeviceSupportedProfiles.bMAPSupport));

   tU8 currentDeviceHandle = deviceListExtendedItem.u8DeviceHandle;

   std::map<tU8, std::string>::iterator it = m_oConnectedDeviceList.find(currentDeviceHandle);

   if((true == deviceListExtendedItem.oDeviceProfileStatus.bHFP) && (true == deviceListExtendedItem.bDeviceConnectedStatus))
   {
      ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status DeviceConnectedStatus true"));

      vCheckAndUpdatePairedDeviceListinDB(true, deviceListExtendedItem.u8DeviceHandle, deviceListExtendedItem.sDeviceAddress.szValue);

      tBool deviceAvailable = DbusClientInterface::getInstance().isDbusClientAvailable(currentDeviceHandle);
      if ((it != m_oConnectedDeviceList.end()) && (true == deviceAvailable))
      {
         //No action required as device is already connected
         ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status:Device_Already_Connected is %d", currentDeviceHandle));
      }
      else
      {
         ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status:Current_Active_Handle is %d", currentDeviceHandle));

         DbusClientInterface::getInstance().setCurrentOutgoingDeviceId(currentDeviceHandle);

         if(TRUE == m_u8RetryDeviceList.empty())
         {
            vDeviceConnected(currentDeviceHandle, deviceListExtendedItem.sDeviceAddress.szValue);
         }
         else
         {
            addDeviceToRetryList(currentDeviceHandle);
         }

         if (it == m_oConnectedDeviceList.end())
         {
            m_oConnectedDeviceList.insert(std::pair<tU8, std::string>(currentDeviceHandle, deviceListExtendedItem.sDeviceAddress.szValue));
            m_u8ConnectedDevicesOrder.push_back(currentDeviceHandle);
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("vHandle_DeviceListExtended_Status DeviceConnectedStatus false"));

      if (it != m_oConnectedDeviceList.end())
      {
         m_oConnectedDeviceList.erase(it);

         //C++11 Semantic Changes - Rule 11: Do not rely on unspecified states of the algorithms defined in the <algorithm> standard header.
         std::vector< tU8 >::iterator it1 = std::find(m_u8ConnectedDevicesOrder.begin(), m_u8ConnectedDevicesOrder.end(), currentDeviceHandle);
         if(it1 != m_u8ConnectedDevicesOrder.end())
         {
            m_u8ConnectedDevicesOrder.erase(it1);
         }

         removeDeviceFromRetryList(currentDeviceHandle);
         removeDeviceFromMAPConnRetryForMNSList(currentDeviceHandle);

         vDeviceDisconnected(currentDeviceHandle);
      }
   }
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vDeviceConnected(tU8 deviceHandle, tString deviceAddress)
{
   ETG_TRACE_USR4(("vDeviceConnected() called"));

   vResetMessagingQueuesAndValues();
   // Bugfix for GMMY15-9649 Exception in "messaging.out"
   Phonebook_Detail::vDestroyPhonebookInstance(deviceHandle);

   // Task CMG3G-7697 - IS2200_Switching MAP ON/OFF@FC_Messaging
   // retrieve MapOnOff value from db
   MessagingSettings* l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();
   tBool l_bMapOnOff = true;

   if (l_poMessageSettings)
   {
      l_bMapOnOff = l_poMessageSettings->bGetDeviceSpecificSetting(deviceHandle, MAP_ON_OFF_SETTING);
   }

   if (l_bMapOnOff)
   {
      DbusClientIf client = DbusClientInterface::getInstance().getClientInterface();

      if(DBUS_CLIENT_IF_EVOLUTION == client)
      {
         EvolutionDbusClient::messagingDeviceConnectionStatus(deviceHandle, deviceAddress);
      }
   }
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vDeviceDisconnected(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("vDeviceDisconnected() called"));

   // GMMY15-255 Bugfix - messaging inbox is not available when phone reconnects after disconnection.
   // Append the Disconnected BT devices to a list, when DeviceList is updated
   QMutexLocker lock(&m_DeviceListmutex);
   m_u8DiscDeviceHandleList.append(deviceHandle);

   DbusClientIf client = DbusClientInterface::getInstance().getClientInterface();
   if(DBUS_CLIENT_IF_EVOLUTION == client)
   {
      EvolutionDbusClient::handleDeviceDisconnection();
   }

   // Clear queues, instances and release mutex lock
   vResetMessagingQueuesAndValues(TRUE);
   // Bugfix for GMMY15-9649 Exception in "messaging.out"
   Phonebook_Detail::vDestroyPhonebookInstance(deviceHandle);
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vResetMessagingQueuesAndValues(tBool disconnect)
{
   ETG_TRACE_USR4(("vResetMessagingQueuesAndValues() called"));

   //if any device connected then don't do this.
   if(TRUE == m_oConnectedDeviceList.empty())
   {
      WorkQueue* poWorkQueue = WorkQueue::poGetWorkQueue();
      if (poWorkQueue)
      {
         poWorkQueue->vClearWorkQueue();
      }

      fc_messaging_tclService_Messaging::poGetInstance()->vResetMapDeviceCapabilities(); // Task CMG3G-7698
      fc_messaging_tclService_Messaging::poGetInstance()->vResetMapDeviceCapabilitiesExtended();
      fc_messaging_tclService_Messaging::poGetInstance()->vResetMessagingDeviceConnectionStatus();
      fc_messaging_tclService_Messaging::poGetInstance()->vResetMessageNotificationStatus();
      fc_messaging_tclService_Messaging::poGetInstance()->vResetMessageListChangeStatus();

      if(TRUE == disconnect)
      {
         FileSystemManager::vResetNewMsgHandleListAndFolderHandle();

         MessagingMutex::messagingMutexUnlock();
      }
   }
}

bool fc_messaging_tclClientHandler_BluetoothSetings::bConnectMAPProfile(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("bConnectMAPProfile() called for device: %d", deviceHandle));
   most_btsetfi_tclMsgConnectBluetoothProfileMethodStart connectBluetoothProfileMethodStart;

   connectBluetoothProfileMethodStart.u8DeviceHandle = deviceHandle;
   connectBluetoothProfileMethodStart.oServiceType.bMAP = TRUE;

   fi_tclVisitorMessage oMsg(connectBluetoothProfileMethodStart,
         MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMsg,
         (tU16)deviceHandle,                          // CmdCounter (deviceHandle will be used in Error Result)
         MOST_BTSETFI_C_U16_CONNECTBLUETOOTHPROFILE,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART           // Opcode
   );

   if (oMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

      if (bIfServiceAvailable())
      {
         // Post message
         enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);
      }
      else
      {
         ETG_TRACE_ERR(("bConnectMAPProfile() BluetoothSettings service unavailable"));
         return false;
      }

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("enPostMessage() failed for  ConnectBluetoothProfile"));

         if (!oMsg.bDelete())
         {
            ETG_TRACE_ERR(("bConnectMAPProfile() Message deletion failed."));
         }
      }
      else
      {
#ifdef RETRY_PBAP_MAP_VIA_BM_APPL
         EvolutionDbusClient::setRetryTimerRunningStatus(TRUE);
         addDeviceToRetryList(deviceHandle);
#endif
      }
   }
   else
   {
      ETG_TRACE_ERR(("ConnectBluetoothProfile: Invalid Send Message"));
   }

   connectBluetoothProfileMethodStart.vDestroy();
   return true;
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vHandle_ConnectBluetoothProfile_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ConnectBluetoothProfile_MethodResult() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgConnectBluetoothProfileMethodResult connectBluetoothProfileMethodResult;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(connectBluetoothProfileMethodResult, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("DeviceHandle: %d", connectBluetoothProfileMethodResult.u8DeviceHandle));
      ETG_TRACE_USR4(("MAP Connection Status: %d", connectBluetoothProfileMethodResult.oProfileConnectionStatus.bMAP));

      MessagingSettings* l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();
      tBool l_bMapOnOff = true;

      if (l_poMessageSettings)
      {
         l_bMapOnOff = l_poMessageSettings->bGetDeviceSpecificSetting(connectBluetoothProfileMethodResult.u8DeviceHandle, MAP_ON_OFF_SETTING);
      }

      if (l_bMapOnOff)
      {
#ifdef RETRY_PBAP_MAP_VIA_BM_APPL
         EvolutionDbusClient* client = EvolutionDbusClient::getDeviceInstance(connectBluetoothProfileMethodResult.u8DeviceHandle);
         if(client)
         {
            if(FALSE == connectBluetoothProfileMethodResult.oProfileConnectionStatus.bMAP)
            {
               client->doMapRetry();
            }
            else
            {
               EvolutionDbusClient::setRetryTimerRunningStatus(FALSE);
               removeDeviceFromRetryList(connectBluetoothProfileMethodResult.u8DeviceHandle);
            }
         }

         else
         {
        	 EvolutionDbusClient::setRetryTimerRunningStatus(FALSE);
        	 removeDeviceFromRetryList(connectBluetoothProfileMethodResult.u8DeviceHandle);
         }
         
         if((TRUE != fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.empty()) &&
                  (TRUE != EvolutionDbusClient::getRetryTimerRunningStatus()))
         {
             ETG_TRACE_USR4(("RetryDeviceList is not empty"));
             DbusClientInterface::getInstance().messagingMapStart();
         }
#endif
         //Note: further MAP connection actions will be done via InterfacesAdded callback
      }
      else
      {
         EvolutionDbusClient::setRetryTimerRunningStatus(FALSE);
         removeDeviceFromRetryList(connectBluetoothProfileMethodResult.u8DeviceHandle);
         removeDeviceFromMAPConnRetryForMNSList(connectBluetoothProfileMethodResult.u8DeviceHandle);

         if(TRUE != fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.empty())
         {
            ETG_TRACE_USR4(("RetryDeviceList is not empty"));

            DbusClientInterface::getInstance().messagingMapStart();
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Unmarshaling failed for ConnectBluetoothProfile MethodResult"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   connectBluetoothProfileMethodResult.vDestroy();
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vHandle_ConnectBluetoothProfile_MethodError(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ConnectBluetoothProfile_MethodError() called"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_btsetfi_tclMsgConnectBluetoothProfileError connectBluetoothProfileError;

   if (oVisitorMsg.s32GetData(connectBluetoothProfileError, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR4((" Error Code: %u", connectBluetoothProfileError.e8ErrorCode.enType));

      MessagingSettings* l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();
      tBool l_bMapOnOff = true;
      tU8 u8DeviceHandle = (tU8)poMessage->u16GetCmdCounter();

      if (l_poMessageSettings)
      {
         l_bMapOnOff = l_poMessageSettings->bGetDeviceSpecificSetting(u8DeviceHandle, MAP_ON_OFF_SETTING);
      }

      if (l_bMapOnOff)
      {
         EvolutionDbusClient* client = EvolutionDbusClient::getDeviceInstance(u8DeviceHandle);
         if(client)
         {
#ifdef RETRY_PBAP_MAP_VIA_BM_APPL
            client->doMapRetry();

            if((TRUE != fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.empty()) &&
                  (TRUE != EvolutionDbusClient::getRetryTimerRunningStatus()))
            {
               ETG_TRACE_USR4(("RetryDeviceList is not empty"));

               DbusClientInterface::getInstance().messagingMapStart();
            }
#else
            client->updateMapDeviceConnectionStatus(FC_MSG_MAP_CONNECTION_FAILED);
#endif
         }
      }
      else
      {
         EvolutionDbusClient::setRetryTimerRunningStatus(FALSE);
         removeDeviceFromRetryList(u8DeviceHandle);
         removeDeviceFromMAPConnRetryForMNSList(u8DeviceHandle);

         if(TRUE != fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.empty())
         {
            ETG_TRACE_USR4(("RetryDeviceList is not empty"));

            DbusClientInterface::getInstance().messagingMapStart();
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Unmarshaling failed for ConnectBluetoothProfile MethodError"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   connectBluetoothProfileError.vDestroy();
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vDisconnectMAPProfile(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("vDisconnectMAPProfile() called for device: %d", deviceHandle));

   most_btsetfi_tclMsgDisconnectBluetoothProfileMethodStart disconnectBluetoothProfileMethodStart;

   disconnectBluetoothProfileMethodStart.u8DeviceHandle = deviceHandle;
   disconnectBluetoothProfileMethodStart.oServiceType.bMAP = TRUE;

   fi_tclVisitorMessage oMsg(disconnectBluetoothProfileMethodStart,
         MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMsg, 0,                             // CmdCounter
         MOST_BTSETFI_C_U16_DISCONNECTBLUETOOTHPROFILE,  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART              // Opcode
   );

   if (oMsg.bIsValid())
   {
      ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

      if (bIfServiceAvailable())
      {
         // Post message
         enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);
      }
      else
      {
         ETG_TRACE_ERR(("vDisconnectMAPProfile() BluetoothSettings service unavailable"));

         enResult = AIL_EN_N_REGISTRATION_DENIED_SERVICE_NOT_FOUND;
      }

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("enPostMessage() failed for  DisconnectBluetoothProfile"));

         if (!oMsg.bDelete())
         {
            ETG_TRACE_ERR(("vDisconnectMAPProfile() Message deletion failed."));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("DisconnectBluetoothProfile: Invalid Send Message"));
   }

   disconnectBluetoothProfileMethodStart.vDestroy();
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vHandle_DisconnectBluetoothProfile_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_DisconnectBluetoothProfile_MethodResult() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgDisconnectBluetoothProfileMethodResult disconnectBluetoothProfileMethodResult;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(disconnectBluetoothProfileMethodResult, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("DeviceHandle: %d", disconnectBluetoothProfileMethodResult.u8DeviceHandle));
      ETG_TRACE_USR4(("MAP Disconnection Status: %d", disconnectBluetoothProfileMethodResult.oProfileDisconnectionStatus.bMAP));

      //Note: further MAP disconnection actions will be done via InterfacesRemoved callback

      //NCG3D-68657: Workaround Start: Incomplete MNS Connection
      std::vector< tU8 >::iterator it = std::find(m_u8MAPConnRetryReqForMNSList.begin(), m_u8MAPConnRetryReqForMNSList.end(),
            disconnectBluetoothProfileMethodResult.u8DeviceHandle);

      if(it != m_u8MAPConnRetryReqForMNSList.end())
      {
         addDeviceToRetryList(disconnectBluetoothProfileMethodResult.u8DeviceHandle);

         DbusClientInterface::getInstance().messagingMapStart();

         EvolutionDbusClient* client = EvolutionDbusClient::getDeviceInstance(disconnectBluetoothProfileMethodResult.u8DeviceHandle);
         if(client)
         {
            client->vSetMapRetryCount(MAP_CONNECTION_RETRY_FOR_MNS);
         }
      }
      //Workaround End
   }
   else
   {
      ETG_TRACE_ERR(("Unmarshaling failed for DisconnectBluetoothProfile"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   disconnectBluetoothProfileMethodResult.vDestroy();
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vHandle_DisconnectBluetoothProfile_MethodError(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_DisconnectBluetoothProfile_MethodError() called"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_btsetfi_tclMsgConnectBluetoothProfileError disconnectBluetoothProfileError;

   if (oVisitorMsg.s32GetData(disconnectBluetoothProfileError, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR4((" Error Code: %u", disconnectBluetoothProfileError.e8ErrorCode.enType));
   }
   else
   {
      ETG_TRACE_ERR(("Unmarshaling failed for DisconnectBluetoothProfile MethodError"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   disconnectBluetoothProfileError.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_VehicleBTFriendlyName_Status(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for VehicleBTFriendlyName property status
 *   
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_messaging_tclClientHandler_BluetoothSetings::vHandle_VehicleBTFriendlyName_Status(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_VehicleBTFriendlyName_Status() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgVehicleBTFriendlyNameStatus oVehicleBTFriendlyNameStatus;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(oVehicleBTFriendlyNameStatus, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      if (oVehicleBTFriendlyNameStatus.sVehicleBTFriendlyName.szValue)
      {
         ETG_TRACE_USR1(("Vehicle BT Friendly Name : %s",oVehicleBTFriendlyNameStatus.sVehicleBTFriendlyName.szValue));

         if (isalpha(oVehicleBTFriendlyNameStatus.sVehicleBTFriendlyName.szValue[0]))
         {
            m_VehicleBTFriendlyName.bSet(oVehicleBTFriendlyNameStatus.sVehicleBTFriendlyName.szValue);
         }
         else
         {
            m_VehicleBTFriendlyName.bSet("GM Vehicle");
         }

         ETG_TRACE_USR1(("m_VehicleBTFriendlyName : %s",m_VehicleBTFriendlyName.szValue));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Unmarshaling failed for VehicleFriendlyName"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   oVehicleBTFriendlyNameStatus.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_VehicleBTAddress_Status(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for VehicleBTAddress property status
 *   
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_messaging_tclClientHandler_BluetoothSetings::vHandle_VehicleBTAddress_Status(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_VehicleBTAddress_Status() called"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgVehicleBTAddressStatus vehicleBTAddressStatus;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if (oVisitorMsg.s32GetData(vehicleBTAddressStatus, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      if (vehicleBTAddressStatus.sVehicleBTAddress.szValue)
      {
         ETG_TRACE_USR1(("Vehicle BT Address : %s",vehicleBTAddressStatus.sVehicleBTAddress.szValue));

         m_VehicleBTAddress.bSet(vehicleBTAddressStatus.sVehicleBTAddress.szValue);

         ETG_TRACE_USR1(("m_VehicleBTAddress : %s",m_VehicleBTAddress.szValue));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Unmarshaling failed for VehicleBTAddress"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   vehicleBTAddressStatus.vDestroy();
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::vReconnectMAPProfile(tU8 deviceHandle)
{
   addDeviceToMAPConnRetryForMNSList(deviceHandle);

   vDisconnectMAPProfile(deviceHandle);
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::addDeviceToRetryList(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("addDeviceToRetryList() deviceHandle : %d", deviceHandle));

   std::vector< tU8 >::iterator it = std::find(m_u8RetryDeviceList.begin(), m_u8RetryDeviceList.end(), deviceHandle);

   if(it != m_u8RetryDeviceList.end())
   {
      ETG_TRACE_USR4(("Device already available in the retry list"));
   }
   else
   {
      m_u8RetryDeviceList.push_back(deviceHandle);
   }
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromRetryList(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("removeDeviceFromRetryList() deviceHandle : %d", deviceHandle));

   std::vector< tU8 >::iterator it = std::find(m_u8RetryDeviceList.begin(), m_u8RetryDeviceList.end(), deviceHandle);

   if(it != m_u8RetryDeviceList.end())
   {
      m_u8RetryDeviceList.erase(it);
   }
   else
   {
      ETG_TRACE_USR4(("Device not available in the retry list"));
   }
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::addDeviceToMAPConnRetryForMNSList(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("addDeviceToMAPConnRetryForMNSList() deviceHandle : %d", deviceHandle));

   std::vector< tU8 >::iterator it = std::find(m_u8MAPConnRetryReqForMNSList.begin(), m_u8MAPConnRetryReqForMNSList.end(), deviceHandle);

   if(it != m_u8MAPConnRetryReqForMNSList.end())
   {
      ETG_TRACE_USR4(("Device already available in the MAP Connection Retry for MNS list"));
   }
   else
   {
      m_u8MAPConnRetryReqForMNSList.push_back(deviceHandle);
   }
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromMAPConnRetryForMNSList(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("removeDeviceFromMAPConnRetryForMNSList() deviceHandle : %d", deviceHandle));

   std::vector< tU8 >::iterator it = std::find(m_u8MAPConnRetryReqForMNSList.begin(), m_u8MAPConnRetryReqForMNSList.end(), deviceHandle);

   if(it != m_u8MAPConnRetryReqForMNSList.end())
   {
      m_u8MAPConnRetryReqForMNSList.erase(it);
   }
   else
   {
      ETG_TRACE_USR4(("Device not available in the MAP Connection Retry for MNS list"));
   }
}

tVoid fc_messaging_tclClientHandler_BluetoothSetings::clearDevicesFromRetryList()
{
   ETG_TRACE_USR4(("clearDevicesFromRetryList() called"));

   if(TRUE != m_u8RetryDeviceList.empty())
   {
      m_u8RetryDeviceList.clear();
   }
   else
   {
      ETG_TRACE_USR4(("RetryList is empty"));
   }
}

tString fc_messaging_tclClientHandler_BluetoothSetings::getVehicleBTAddress()
{
   return m_VehicleBTAddress.szValue;
}


/*******************************************************************************
 *
 * FUNCTION: vCheckAndUpdatePairedDeviceList
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclClientHandler_BluetoothSetings::vCheckAndUpdatePairedDeviceListinDB(tBool isDevPaired, tU8 DevHandle, std::string DevAddress)
{
   ETG_TRACE_USR4(("vCheckAndUpdatePairedDeviceListinDB entered with DevHandle: %d and DevAddress: %s", DevHandle, DevAddress.c_str()));

   MessagingSettings* l_poMessagingSettings = MessagingSettings::poGetMessagingSettingsInstance();
   if (!l_poMessagingSettings)
   {
      ETG_TRACE_USR4(("l_poMessagingSettings is NULL"));
      return;
   }

   tBool bIsUpdatedRequired = TRUE;
   tMap_DevHandle_DevAddress::const_iterator MapIterator = m_mapPairedDeviceList.find(DevHandle);

   if (isDevPaired)
   {
      ETG_TRACE_USR4(("Device PAIRED"));

      if (MapIterator != m_mapPairedDeviceList.end())
      {
         if (0 == strcmp(MapIterator->second.c_str(), DevAddress.c_str()))
         {
            ETG_TRACE_USR4(("Address also matches. Hence Device is already in the paired List"));
            bIsUpdatedRequired = FALSE;
         }
         else
         {
            ETG_TRACE_USR4(("Address is not matching but the Device Handle is present. Need to reset the corresponding DeviceSpecificSettings"));

            //Reset all device specific settings to default
            FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->bQueryResetDeviceSettingsInDeviceSettingsTable(DevHandle, DevAddress);

            //Remove all device specific custom messages
            FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->bQueryRemoveCustomMessagesFromPreDefMsgTable(DevHandle);
         }
      }
      else
      {
         ETG_TRACE_USR4(("New device paired and connected. Hence adding it to the DB table"));
         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->bQueryAddDevicetoDeviceSettingsTable(DevHandle, DevAddress);
      }
   }
   else
   {
      ETG_TRACE_USR4(("Device DELETED"));

      if ((0 == DevHandle) || (MapIterator != m_mapPairedDeviceList.end()))
      {
         ETG_TRACE_USR4(("Deleting entries from the DB table"));

         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->bQueryRemoveDeviceFromDeviceSettingsTable(DevHandle);

         //Remove device specific custom messages
         FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->bQueryRemoveCustomMessagesFromPreDefMsgTable(DevHandle);

         //Remove device specific signature text
         fc_messaging_tclService_Messaging::poGetInstance()->vEraseDeviceSignatureText(DevHandle);
      }
      else
      {
         ETG_TRACE_USR4(("Device already removed or not present"));
         bIsUpdatedRequired = FALSE;
      }
   }

   if(bIsUpdatedRequired)
   {
      //Checking the updated list
      FC_Messaging_SQLite::poGetFC_Messaging_SQLite()->szQueryGetDevicesList(m_mapPairedDeviceList);
      vPrintPairedDeviceList();

      //Update required for all device specific settings
      fc_messaging_tclService_Messaging::poGetInstance()->vUpdateDeviceSpecificSettingsProperties();
   }
}

/*******************************************************************************
 *
 * FUNCTION: vPrintPairedDeviceList
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_messaging_tclClientHandler_BluetoothSetings::vPrintPairedDeviceList()
{
   ETG_TRACE_USR4(("fc_messaging_tclService_Messaging::vPrintPairedDeviceList"));

   tMap_DevHandle_DevAddress::const_iterator MapIterator;
   for (MapIterator = m_mapPairedDeviceList.begin(); MapIterator != m_mapPairedDeviceList.end(); ++MapIterator)
   {
      ETG_TRACE_USR1(("m_mapPairedDeviceList[].DeviceHandle: %d", MapIterator->first));
      ETG_TRACE_USR1(("m_mapPairedDeviceList[].DeviceAddress: %s", MapIterator->second.c_str()));
   }
}

