/*******************************************************************************
 *
 * FILE:        	FC_Messaging_WorkQueue.cpp
 *
 * SW-COMPONENT:    FC_Messaging application
 *
 * PROJECT:
 *
 * DESCRIPTION:
 *
 * AUTHOR:		Rakesh Kumar
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

#include "FC_Messaging_Debug.h"
#include "FC_Messaging_service_Messaging.h"
#include "DbusClient/FC_Messaging_DbusClientInterface.h"
#include "FC_Messaging_WorkQueue.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_WorkQueue.cpp.trc.h"
#endif

WorkQueue* WorkQueue::s_poWorkQueue = NULLPTR;
QMutex WorkQueue::s_oMutexLock;

/*******************************************************************************
 * FUNCTION:  	vCreateWorkQueue
 * DESCRIPTION:  function is used to create workqueue object.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
tVoid WorkQueue::vCreateWorkQueue()
{
   ETG_TRACE_USR4(("WorkQueue::vCreateWorkQueue"));

   QMutexLocker lock(&s_oMutexLock);

   if (!s_poWorkQueue)
   {
      s_poWorkQueue = OSAL_NEW WorkQueue;
   }
}

WorkQueue* WorkQueue::poGetWorkQueue()
{
   ETG_TRACE_USR4(("WorkQueue::poGetWorkQueue"));

   return s_poWorkQueue;
}

/*******************************************************************************
 * FUNCTION:  	vDestroyWorkQueue
 * DESCRIPTION:  function is used to destroy workqueue object.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
tVoid WorkQueue::vDestroyWorkQueue()
{
   ETG_TRACE_USR4(("WorkQueue::vDestroyWorkQueue"));

   QMutexLocker lock(&s_oMutexLock);

   if (s_poWorkQueue)
   {
      OSAL_DELETE s_poWorkQueue;
      s_poWorkQueue = NULLPTR;
   }
}

/*******************************************************************************
 * FUNCTION:  	WorkQueue
 * DESCRIPTION:  constructor
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
WorkQueue::WorkQueue()
{
   ETG_TRACE_USR4(("WorkQueue::WorkQueue"));

   m_u8Size = 0;
   m_prHead = NULLPTR;
   m_prTail = NULLPTR;
}

/*******************************************************************************
 * FUNCTION:  	~WorkQueue
 * DESCRIPTION:  destructor
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
WorkQueue::~WorkQueue()
{
   ETG_TRACE_USR4(("WorkQueue::~WorkQueue"));

   jobNode* prTempNode;

   while (m_prHead)
   {
      ETG_TRACE_USR4(("Deleting the node from work queue"));
      prTempNode = m_prHead;
      m_prHead = m_prHead->prNextJobNode;
      delete prTempNode;
   }

   m_u8Size = 0;
   m_prHead = NULLPTR;
   m_prTail = NULLPTR;
}

/*******************************************************************************
 * FUNCTION:  	vInsertNewJob
 * DESCRIPTION:  function is used to insert new job in queue.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
tVoid WorkQueue::vInsertNewJob(struct Job *f_prJob)
{
   ETG_TRACE_USR4(("WorkQueue::vInsertNewJob"));

   QMutexLocker lock(&s_oMutexLock);

   jobNode* prTempJob = new jobNode;

   if (prTempJob)
   {
      prTempJob->prNextJobNode = NULLPTR;

      prTempJob->prJob = f_prJob;

      if (!m_prHead)
      {
         m_prHead = prTempJob;
         m_prTail = m_prHead;
      }
      else
      {
         if(m_prTail)
         {
            m_prTail->prNextJobNode = prTempJob;
         }
         m_prTail = prTempJob;
      }

      m_u8Size++;

      ETG_TRACE_USR2(("Job(s) waiting in work queue : '%u'", m_u8Size));
   }
}

/*******************************************************************************
 * FUNCTION:  	vFetchNextJob
 * DESCRIPTION:  function is used to fetch job in queue.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
tVoid WorkQueue::vFetchNextJob()
{
   ETG_TRACE_USR4(("WorkQueue::vFetchNextJob"));

   QMutexLocker lock(&s_oMutexLock);

   if (!m_prHead)
   {
      ETG_TRACE_USR4(("WorkQueue is empty"));
      return;
   }

   jobNode* prTempJob = m_prHead;
   m_prHead = m_prHead->prNextJobNode;

   if (prTempJob->prJob)
   {
      // Post the job to CCA Thread
      vSendToCCAThread(prTempJob->prJob);
   }

   m_u8Size--;
   delete prTempJob;

   ETG_TRACE_USR2(("Job(s) waiting in work queue : '%u'", m_u8Size));
}

// clear workqueue on device disconnect
tVoid WorkQueue::vClearWorkQueue()
{
   ETG_TRACE_USR4(("WorkQueue::vClearWorkQueue Called"));

   QMutexLocker lock(&s_oMutexLock);

   jobNode* prTempJob;

   while (m_prHead)
   {
      ETG_TRACE_USR4(("Deleting the node from work queue"));
      prTempJob = m_prHead;
      m_prHead = m_prHead->prNextJobNode;
      delete prTempJob;
   }
}

/*******************************************************************************
 * FUNCTION:   vSendToCCAThread
 * DESCRIPTION:  function is used to send a job from worker thread to CCA thread.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
tVoid WorkQueue::vSendToCCAThread(Job* jobToBeSent)
{
   ETG_TRACE_USR4(("WorkQueue::vSendToCCAThread"));

   fc_messaging_tclService_Messaging* poMessagingService = fc_messaging_tclService_Messaging::poGetInstance();

   if (poMessagingService)
   {
      if (TRUE == poMessagingService->m_bServerAvailable)
      {
         Job *prTempJob = new Job;

         if (prTempJob)
         {
            tU8 u8DeviceHandle = jobToBeSent->u8DeviceHandle;

            prTempJob->m_pfMethod = jobToBeSent->m_pfMethod;
            prTempJob->u8DeviceHandle = u8DeviceHandle;
            prTempJob->u16SourceAppID = jobToBeSent->u16SourceAppID;
            prTempJob->u16RegisterID = jobToBeSent->u16RegisterID;
            prTempJob->u16CmdCounter = jobToBeSent->u16CmdCounter;
            prTempJob->u16FunctionID = jobToBeSent->u16FunctionID;
            prTempJob->m_pvMethodStart = (tVoid*) jobToBeSent->m_pvMethodStart;

            LoopBackMessageData rLoopBackMessageData;
            rLoopBackMessageData.vpData = prTempJob;

            ETG_TRACE_USR4(("WorkQueue::vSendToCCAThread DeviceHandle: %d", u8DeviceHandle));

            QString sServiceName = DbusClientInterface::getInstance().getCurrentActiveService(u8DeviceHandle);

            rLoopBackMessageData.u8DeviceHandle = u8DeviceHandle;
            rLoopBackMessageData.sServiceName = sServiceName;
            rLoopBackMessageData.u32FunctionId = FC_MSG_LB_FID_JOB_FROM_WORKQUEUE;
            poMessagingService->vPostLoopBackMessage(rLoopBackMessageData);
         }
      }
      else
      {
         ETG_TRACE_ERR(("Error: Messaging Service not available!!"));
      }
   }
}
