/*******************************************************************************
 *
 * FILE:        	FC_Messaging_StateMachine.cpp
 *
 * SW-COMPONENT:  FC_Messaging application
 *
 * PROJECT:
 *
 * DESCRIPTION:
 *
 * AUTHOR:		Rakesh Kumar
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

#ifndef __PLACEMENT_NEW_INLINE
#define __PLACEMENT_NEW_INLINE 1
#endif

#include "FC_Messaging_main.h"
#include "FC_Messaging_StateMachine.h"
#include "FC_Messaging_Debug.h"
#include "FC_Messaging_service_Messaging.h"
#include "unistd.h"
#include "signal.h"
#include "FC_Messaging_WorkQueue.h"
#include "SendMessage/FC_Messaging_SendMessage.h"
#include "DbusClient/FC_Messaging_DbusClientInterface.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_StateMachine.cpp.trc.h"
#endif

MessagingMutex* MessagingMutex::poMessagingMutexObject = NULLPTR;
pthread_t MessagingMutex::WorkQueueThreadID = 0;

/*******************************************************************************
 * FUNCTION:  	unlockMessagingDbusMutex
 * DESCRIPTION:  function is used to unlock mutex on timer expiry.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
void unlockMessagingDbusMutex(int signalNumber)
{
   ETG_TRACE_USR4(("unlockMessagingDbusMutex : Called"));

   (void) signalNumber;

   ETG_TRACE_USR4(("Timer is expired"));

   // Bugfix for GMMY16-5367 : Selected view on new message alert and nothing happened
   // and GMMY16-5368 : Message inbox was missing many of the newest messages
   if (false == DbusClientInterface::getInstance().waitForMapListMsgCompSignal())
   {
      //Release lock if not waiting for MapListMsgComp signal
      MessagingMutex::messagingMutexUnlock();
   }
}

/*******************************************************************************
 * FUNCTION:  	messagingMutexInit
 * DESCRIPTION:  function is used to create MessageMutex object.
 *
 * PARAMETER:
 *
 * RETURNVALUE: object address
 ********************************************************************************/
MessagingMutex* MessagingMutex::messagingMutexInit()
{
   ETG_TRACE_USR4(("messagingMutexInit : Called"));

   if (!poMessagingMutexObject)
   {
      poMessagingMutexObject = new MessagingMutex;
   }

   return poMessagingMutexObject;
}

/*******************************************************************************
 * FUNCTION:  	messagingMutexDestroy
 * DESCRIPTION:  function is used to destroy MessageMutex object.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
void MessagingMutex::messagingMutexDestroy()
{
   ETG_TRACE_USR4(("messagingMutexDestroy : Called"));

   delete poMessagingMutexObject;
   poMessagingMutexObject = NULLPTR;
}

/*******************************************************************************
 * FUNCTION:  	MessagingMutex
 * DESCRIPTION:  constructor.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
MessagingMutex::MessagingMutex()
{
   ETG_TRACE_USR4(("MessagingMutex : Called"));

   pthread_mutex_init(&messagingDbusMutex, NULLPTR);

   pthread_create(&WorkQueueThreadID, NULLPTR, MessagingMutex::WorkQueueScheduler, NULLPTR);
}

/*******************************************************************************
 * FUNCTION:  	~MessagingMutex
 * DESCRIPTION:  destructor.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
MessagingMutex::~MessagingMutex()
{
   ETG_TRACE_USR4(("~MessagingMutex : Called"));

   pthread_mutex_destroy(&messagingDbusMutex);
}

/*******************************************************************************
 * FUNCTION:  	messagingMutexTrylock
 * DESCRIPTION:  function to lock the mutex.
 *
 * PARAMETER:
 *
 * RETURNVALUE: success or failure
 ********************************************************************************/
int MessagingMutex::messagingMutexTrylock(struct Job *prJob)
{
   ETG_TRACE_USR4(("MessagingMutex::messagingMutexTrylock"));

   int nReturn = pthread_mutex_trylock(&poMessagingMutexObject->messagingDbusMutex);

   if (0 != nReturn)
   {
      ETG_TRACE_ERR(("Unable to acquire lock, Inserting the job in WorkQueue"));

      // Insert the job in the queue
      if (prJob)
      {
         WorkQueue* poWorkQueue = WorkQueue::poGetWorkQueue();
         if (poWorkQueue)
         {
            poWorkQueue->vInsertNewJob(prJob);
         }
      }
   }
   else
   {
      //Start the timer
      ETG_TRACE_USR4(("Timer started"));

      alarm(30);
      signal(SIGALRM, unlockMessagingDbusMutex);
   }

   return nReturn;
}

/*******************************************************************************
 * FUNCTION:  	messagingMutexUnlock
 * DESCRIPTION:  function to unlock the mutex.
 *
 * PARAMETER:
 *
 * RETURNVALUE: success or failure
 ********************************************************************************/
int MessagingMutex::messagingMutexUnlock()
{
   ETG_TRACE_USR4(("messagingMutexUnlock : Called"));

   // Stop the timer
   // alarm(0); //Don't stop alarm until unlock signal received by WorkQueueScheduler

   if (WorkQueueThreadID == pthread_self())
   {
      pthread_mutex_unlock(&poMessagingMutexObject->messagingDbusMutex);
   }

   pthread_kill(WorkQueueThreadID, SIGUSR1);

   ETG_TRACE_USR4(("Signal sent"));

   return 0;
}

/*******************************************************************************
 * FUNCTION:  	dummy
 * DESCRIPTION:  dummy function.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
void dummy(int signalNumber)
{
   (void) signalNumber;
   ETG_TRACE_USR4(("dummy : Called"));
}

/*******************************************************************************
 * FUNCTION:  	WorkQueueScheduler
 * DESCRIPTION:  function run by workqueue thread.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
void *MessagingMutex::WorkQueueScheduler(void *ptr)
{
   ETG_TRACE_USR4(("MessagingMutex::WorkQueueScheduler"));
   (void) ptr;

   signal(SIGUSR1, dummy);

   while (true)
   {
      // Wait for the signal
      pause();

      alarm(0); //Set the alarm to 0 when signal received

      pthread_mutex_unlock(&poMessagingMutexObject->messagingDbusMutex);

      ETG_TRACE_USR4(("Got the signal"));

      // Fetch the next job
      WorkQueue* poWorkQueue = WorkQueue::poGetWorkQueue();
      if (poWorkQueue)
      {
         poWorkQueue->vFetchNextJob();
      }
   }

   return NULLPTR;
}
