/**
 * @file FC_Messaging_DbusClientInterface.cpp
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */
#include "FC_Messaging_DbusClientInterface.h"
#include "FC_Messaging_EvoDbusClient.h"
#include "../FC_Messaging_main.h"
#include "../SendMessage/FC_Messaging_SendMessage.h"
#include "../FC_Messaging_clienthandler_BluetoothSetings.h"
#include "../MsgList/FC_Messaging_Settings.h"
#include "FC_Messaging_Macro.h"
#include <unistd.h>
#include <fcntl.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_DbusClientInterface.cpp.trc.h"
#endif

DbusClientInterface* DbusClientInterface::_dbusClientIf = NULLPTR;

DbusClientInterface::DbusClientInterface()
{
	_dbusClient = DBUS_CLIENT_IF_EVOLUTION;
}

DbusClientInterface::~DbusClientInterface()
{
}

DbusClientInterface& DbusClientInterface::getInstance(void)
{
   if (!_dbusClientIf)
   {
      _dbusClientIf = new DbusClientInterface;
   }
   return *_dbusClientIf;
}

void DbusClientInterface::destroyInstance()
{
   delete _dbusClientIf;
   _dbusClientIf = NULLPTR;
}

DbusClientIf DbusClientInterface::getClientInterface(void)
{
   return _dbusClient;
}

void DbusClientInterface::messagingMapGetCapabilities(tU8 deviceHandle)
{
   (void)deviceHandle;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      //TODO: No direct interface!!!
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::messagingMapStart()
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      /**
       * CreateSession():
       * detination String Is the Bluetooth address of the remote device
       * args       Dict
       *
       * dictionary include the following:
       * String  Target: type of session to be created. (PBAP or MAP)
       * String  Source: local address to be used.
       * byte   Channel: The parameter will be ignored by module, host neednt set it.
       */

      do
      {
         //Retrieve the device address from retry device detail.
         tU8 deviceHandle = 0;
         std::string deviceAddress;
         bool btServiceAvailable = true;
         std::vector< tU8 >::iterator it1 = fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.begin();
         EvolutionDbusClient::setRetryTimerRunningStatus(FALSE);

         if(it1 != fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.end())
         {
            deviceHandle = *it1;

            //fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.erase(it1);

            std::map< tU8, std::string >::iterator it2 = fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.find(deviceHandle);
            if(it2 != fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.end())
            {
               deviceAddress = it2->second;
            }
            else
            {
               ETG_TRACE_ERR(("Device not available"));
               fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromRetryList(deviceHandle);
               fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromMAPConnRetryForMNSList(deviceHandle);
            }
         }

         if(TRUE != deviceAddress.empty())
         {
            MessagingSettings* l_poMessageSettings = MessagingSettings::poGetMessagingSettingsInstance();
            tBool bMapOnOff = true;

            if (l_poMessageSettings)
            {
               bMapOnOff = l_poMessageSettings->bGetDeviceSpecificSetting(deviceHandle, MAP_ON_OFF_SETTING);
            }

            if (bMapOnOff)
            {
               EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);
               if(evolutionClient)
               {
                  ETG_TRACE_USR4(("MAP Session Retry for Device: %s", deviceAddress.c_str()));

#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
                  if(evolutionClient->messagingMapCreateSession(deviceAddress))
                  {
                     evolutionClient->messagingMapSessionConnected();
                  }
#else
                  if((fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings)
                  {
                     btServiceAvailable = (fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings->bConnectMAPProfile(deviceHandle);
                     if(false == btServiceAvailable)
                     {
                         ETG_TRACE_ERR(("Bluetooth Service not available : Exiting from Retry Loop"));
                         break;
                     }
                  }
#endif
               }
               else
               {
                  EvolutionDbusClient::messagingDeviceConnectionStatus(deviceHandle, deviceAddress);
               }
            }
            else
            {
               ETG_TRACE_ERR(("MAP is OFF"));
               fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromRetryList(deviceHandle);
               fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromMAPConnRetryForMNSList(deviceHandle);
            }
         }
      }
      while((TRUE != fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.empty()) &&
            (TRUE != EvolutionDbusClient::getRetryTimerRunningStatus()));
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::messagingMapStop()
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      /**
       * RemoveSession():
       * session ObjectPath -> which is created when CreateSession() completed.
       */
      EvolutionDbusClient* evolutionClient = getEvolutionDBusClient();
      if(evolutionClient)
      {
         evolutionClient->messagingMapRemoveSession();
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

tBool DbusClientInterface::messagingMapChgInstance(tU16 accountHandle)
{
   (void)accountHandle;

   tBool retValue = false;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = getEvolutionDBusClient();
      if(evolutionClient)
      {
         //TODO: No direct interface!!!
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return retValue;
}

void DbusClientInterface::messagingMapUpdateInbox()
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      /**
       * UpdateInbox():
       * No parameter required as of now.
       */
      EvolutionDbusClient* evolutionClient = getEvolutionDBusClient();
      if(evolutionClient)
      {
         evolutionClient->messagingMapUpdateInbox();
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::messagingMapListFolder(QString folderName)
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      const QByteArray tmp = folderName.toUtf8();
      ::std::string folder = tmp.constData();
      /**
       * ListFolders():
       * uint16 Offset: Offset of the first item, default is 0.
       * uint16 MaxCount: Maximum number of items, default is 1024.
       *
       * Note: setFolder should be called first!!!
       */

      EvolutionDbusClient* evolutionClient = getEvolutionDBusClient();
      if(evolutionClient)
      {
         evolutionClient->messagingMapListFolder(folder);
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::messagingMapGetSubFolderCount(tString folderName)
{
   (void)folderName;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = getEvolutionDBusClient();
      if(evolutionClient)
      {
         //TODO No direct interface!!!
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::messagingMapReadMsg(tU8 deviceHandle, MessageTypeDbusMessageHandleInfo& messageTypeDbusMessageHandleInfo)
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      /**
       * Get():
       * targetfile String  The name of the file to store the downloaded message in bMessage format.
       *                    Format:
       *                    /home/user/genivi/bmessage
       *                    If an empty target file is given, a temporary file will be automatically generated.
       * attachment Boolean Set to True to download the attachment, else False.
       */
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         evolutionClient->messagingMapGet(messageTypeDbusMessageHandleInfo);
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

tBool DbusClientInterface::messagingMapSetMsgStatus(tU8 deviceHandle, QString messageHandle, most_fi_tcl_e8_MsgMessageListType messageListType, tU8 operation)
{
   tBool retValue = false;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      const QByteArray tmp = messageHandle.toUtf8();
      ::std::string handle = tmp.constData();
      /**
       * Read [readwrite] Boolean : Keep the value False, if message was not read, else set it to True.
       * Deleted [writeonly] Boolean : Keep the value False, if message shall not be deleted, else set it to True.
       */
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         retValue = evolutionClient->messagingMapSetMsgStatus(handle, messageListType, operation);
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return retValue;
}

void DbusClientInterface::messagingMapAbort(tU8 deviceHandle)
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      /**
       * Cancel():
       * No parameter required as of now!!!
       */
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         evolutionClient->messagingMapCancel();
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

tVoid DbusClientInterface::sendFolderListRequest(tU8 deviceHandle, tenFC_Msg_FolderListRequestStatus folderListRequestStatus)
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         evolutionClient->sendFolderListRequest(folderListRequestStatus);
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::messagingMapListMsg(most_msgfi_tclMsgCreateMessageListMethodStart createMessageListMethodStart,
      tU16 listMaxCount, e8MapListMsgCaller mapListMsgCaller, tU16 listStartOffset)
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      /**
       * ListMessages():
       * uint16 Offset: Offset of the first item, default is 0.
       * uint16 MaxCount: Maximum number of items, default is 1024
       * byte SubjectLength: Maximum length of the Subject property in the message, default is 256.
       * array{String} Fields: Message fields, default is all values. Possible values can be query with ListFilterFields.
       * array{String} Types: Filter messages by type.
       * String PeriodBegin: Filter messages by starting period. Possible values: Date in "YYYYMMDDTHHMMSS" format.
       * String PeriodEnd: Filter messages by ending period. Possible values: Date in "YYYYMMDDTHHMMSS" format.
       * boolean Read: Filter messages by read flag. Possible values: True for read or False for unread.
       * String Recipient: Filter messages by recipient address.
       * String Sender: Filter messages by sender address.
       * boolean Priority: Filter messages by priority flag. Possible values: True for high priority or False for non-high priority.
       */
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(createMessageListMethodStart.u8DeviceHandle);

      if(evolutionClient)
      {
         evolutionClient->messagingMapListMsg(createMessageListMethodStart, listMaxCount, mapListMsgCaller, listStartOffset);
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::messagingMapGetMessageCount(tCString folderName, tU16 listMaxCount, tU8 statusFilter)
{
   (void)folderName;
   (void)listMaxCount;
   (void)statusFilter;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      //TODO No direct interface!!!
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

tBool DbusClientInterface::messagingMapUpload(tU8 deviceHandle, SendMessage* sendMessage)
{
   tBool retVal = false;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      /**
       * PushMessage():
       * sourcefile String The path of the message in bMessage format.
       * folder     String The name of the folder to push the new message to.
       * args       Dict
       * boolean Transparent: It may be used to indicate to the MSE that no copy of the message shall be kept in the 'Sent' folder after the message was sent.
       *                        The possible values of this parameter are 0x00 (keep messages in 'Sent' folder) and 0x01 (don't keep messages in Sent' folder).
       * boolean Retry: This parameter may be used to indicate whether successive attempts of sending a message shall be carried out in case that the cellular network is not accessible when the message is sent from the MCE to the outbox of the MSE.
       *                The possible values of this parameter are 0x00 (don't retry) and 0x01 (retry).
       * String Charset: This application parameter shall be used to indicate which kind of content is delivered in the bMessage.
       *                 The value shall be set as native or utf8.
       */
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         retVal = evolutionClient->messagingMapPushMessage(sendMessage);
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return retVal;
}

void DbusClientInterface::messagingMapUploadData(tU8 deviceHandle, SendMessage* sendMessage)
{
   (void)deviceHandle;
   (void)sendMessage;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      /**
       * PushMessage():
       * sourcefile String The path of the message in bMessage format.
       * folder     String The name of the folder to push the new message to.
       * args       Dict
       * boolean Transparent: It may be used to indicate to the MSE that no copy of the message shall be kept in the 'Sent' folder after the message was sent.
       *                        The possible values of this parameter are 0x00 (keep messages in 'Sent' folder) and 0x01 (don't keep messages in Sent' folder).
       * boolean Retry: This parameter may be used to indicate whether successive attempts of sending a message shall be carried out in case that the cellular network is not accessible when the message is sent from the MCE to the outbox of the MSE.
       *                The possible values of this parameter are 0x00 (don't retry) and 0x01 (retry).
       * String Charset: This application parameter shall be used to indicate which kind of content is delivered in the bMessage.
       *                 The value shall be set as native or utf8.
       */

      //This is not required for evolution stack.
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::handleNewMessageNotification(most_msgfi_tclMsgCreateMessageListMethodStart* createMessageListMethodStart)
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(createMessageListMethodStart->u8DeviceHandle);

      if(evolutionClient)
      {
         evolutionClient->handleNewMessageNotification(createMessageListMethodStart);
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

void DbusClientInterface::setCurrentOutgoingDeviceId(tU8 deviceHandle)
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient::setCurrentOutgoingDeviceId(deviceHandle);
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }
}

tU8 DbusClientInterface::getCurrentOutgoingDeviceId()
{
   tU8 currentOutgoingDeviceId = 0;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      currentOutgoingDeviceId = EvolutionDbusClient::getCurrentOutgoingDeviceId();
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return currentOutgoingDeviceId;
}

QString DbusClientInterface::getCurrentActiveService(tU8 deviceHandle, most_fi_tcl_e8_MsgMessageListType::tenType MsgListType)
{
   QString currentActiveAccount;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         if (most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS == MsgListType)
         {
            currentActiveAccount = evolutionClient->_serviceNameAccountManager.sGetCurrentActiveSMSMMSService();
         }
         else
         {
            currentActiveAccount = evolutionClient->_serviceNameAccountManager.sGetCurrentActiveEmailService();
         }
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return currentActiveAccount;
}

FileSystemManager* DbusClientInterface::getFileSystemManager(tU8 deviceHandle, QString serviceName)
{
   FileSystemManager* fileSystemManager = NULLPTR;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         fileSystemManager = evolutionClient->_serviceNameAccountManager.poGetFileSystemManager(serviceName);
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return fileSystemManager;
}

MsgRead_t* DbusClientInterface::getMsgReadObject(tU8 deviceHandle)
{
   MsgRead_t* msgRead = NULLPTR;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         msgRead = evolutionClient->getMsgReadObject();
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return msgRead;
}

MsgListing_t* DbusClientInterface::getMsgListObject(tU8 deviceHandle)
{
   MsgListing_t* msgListing = NULLPTR;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         msgListing = evolutionClient->getMsgListObject();
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return msgListing;
}

tBool DbusClientInterface::getMapNotifyStatus(tU8 deviceHandle)
{
   tBool bRet = FALSE;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         bRet = evolutionClient->getMapNotifyStatus();
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return bRet;
}

tBool DbusClientInterface::getMapMsgCapabilityInfo(tU8 deviceHandle, MsgCapability_t& msgCapabilityInfo)
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::_bluetoothDeviceObject[deviceHandle];

      if(evolutionClient)
      {
         msgCapabilityInfo = evolutionClient->_msgCapabilityInfo;
         return true;
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return false;
}

EvolutionDbusClient* DbusClientInterface::getEvolutionDBusClient()
{
   return EvolutionDbusClient::getDeviceInstance(EvolutionDbusClient::getCurrentOutgoingDeviceId()); //TODO try removing this concept
}

tBool DbusClientInterface::waitForMapListMsgCompSignal()
{
   tBool bRet = false;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      std::map< tU8, std::string >::iterator it;
      for( it = fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.begin();
            it != fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.end(); it++)
      {
         EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(it->first);
         if(evolutionClient)
         {
            bRet = static_cast<tBool>(bRet | evolutionClient->getWaitForMapListMsgCompSignal());
         }
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return bRet;
}

tBool DbusClientInterface::isDbusClientAvailable(tU8 deviceHandle)
{
   tBool clientAvailable = false;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         clientAvailable = true;
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return clientAvailable;
}

tBool DbusClientInterface::setSignatureText(tU8 deviceHandle, ::std::string signatureText)
{
   tBool isTextSet = false;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
         evolutionClient->setSignatureText(signatureText);
         isTextSet = true;
      }
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   return isTextSet;
}

void DbusClientInterface::getSystemBus()
{
   GError *error = NULLPTR;

   if (DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
      EvolutionDbusClient::_evoMsgBus = g_bus_get_sync(G_BUS_TYPE_SESSION, NULLPTR, &error);
   }
   else
   {
      ETG_TRACE_FATAL(("FATAL: Interface error"));
   }

   if(error)
   {
      ETG_TRACE_ERR(("Failed to connect to session bus: error=%d %s", error->code, (error->message)));
      ETG_TRACE_ERRMEM(("Failed to connect to session bus: error=%d %s", error->code, (error->message)));
      //FATAL_M_ASSERT_ALWAYS(); // This option with the fatal assert would issue a direct reset and could be used to avoid
   }
}

tBool DbusClientInterface::messagingCreateEvoProxies()
{
   tBool bRet = false;

   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
      bRet = messagingCreateClientProxy();
#endif
      bRet = messagingCreateObjectManagerProxy();
   }

   return bRet;
}

tBool DbusClientInterface::messagingCreateClientProxy()
{
   ETG_TRACE_USR4(("messagingCreateClientProxy() called "));

   GError *error = NULLPTR;

   EvolutionDbusClient::_proxyClient = org_bluez_obex_client_proxy_new_sync(EvolutionDbusClient::_evoMsgBus,
               G_DBUS_PROXY_FLAGS_NONE, BASE_EVO_SERVICE_NAME,
               BASE_EVO_OBJECT_PATH, NULLPTR, &error);

   if (!(EvolutionDbusClient::_proxyClient) && error)
   {
      ETG_TRACE_ERR(("ClientProxyError: %s", g_dbus_error_get_remote_error(error)));
      g_error_free(error);

      return false;
   }
   else
   {
      ETG_TRACE_USR4(("ClientProxy created"));

      //Connect signals
      EvolutionDbusClient::registerClientSignals();
   }

   //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
   // this removes the lint warning for this block only

   OrgBluezObexClientProxy *clientProxy = ORG_BLUEZ_OBEX_CLIENT_PROXY(EvolutionDbusClient::_proxyClient);
   if(clientProxy)
   {
      //The default timeout value for DBus call in G3g is set to 300s - CMG3GB-1364
      g_dbus_proxy_set_default_timeout(&clientProxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
   }
   else
   {
      ETG_TRACE_ERR(("set timeout failed for Client proxy"));
   }

   return true;
}

tBool DbusClientInterface::messagingCreateObjectManagerProxy()
{
   ETG_TRACE_USR4(("messagingCreateObjectManagerProxy() called "));

   GError *error = NULLPTR;

   //Object Manager Proxy Creation
   EvolutionDbusClient::_proxyObjectManager = org_freedesktop_dbus_object_manager_proxy_new_sync(EvolutionDbusClient::_evoMsgBus,
         G_DBUS_PROXY_FLAGS_NONE, BASE_EVO_SERVICE_NAME,
         "/", NULLPTR, &error);

   if (!(EvolutionDbusClient::_proxyObjectManager) && error)
   {
      ETG_TRACE_ERR(("ObjectManagerProxyError: %s", g_dbus_error_get_remote_error(error)));
      g_error_free(error);

      return false;
   }
   else
   {
      ETG_TRACE_USR4(("ObjectManagerProxy created"));

      //Connect signals
      EvolutionDbusClient::registerObjectManagerSignals();
   }

   //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
   // this removes the lint warning for this block only

   OrgFreedesktopDBusObjectManagerProxy* objectManagerProxy = ORG_FREEDESKTOP_DBUS_OBJECT_MANAGER_PROXY(EvolutionDbusClient::_proxyObjectManager);
   if(objectManagerProxy)
   {
      g_dbus_proxy_set_default_timeout(&objectManagerProxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
   }
   else
   {
      ETG_TRACE_ERR(("set timeout failed for ObjectManager proxy"));
   }

   return true;
}

tVoid DbusClientInterface::messagingRemoveEvoProxies()
{
   if(DBUS_CLIENT_IF_EVOLUTION == _dbusClient)
   {
#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
      //Unregister Client Signals
      EvolutionDbusClient::unregisterClientSignals();
      //Unref Client Proxy
      EvolutionDbusClient::unrefClientProxy();
#endif

      //Unregister ObjectManager Signals
      EvolutionDbusClient::unregisterObjectManagerSignals();
      //Unref ObjectManager Proxy
      EvolutionDbusClient::unrefObjectManagerProxy();
   }
}

/** @} */
