/*******************************************************************************
 *
 * FILE:          FC_Messaging_SQLite.cpp
 *
 * SW-COMPONENT:  FC_Messaging application
 *
 * PROJECT:       BOSCH-GM-NEXTGEN
 *
 * DESCRIPTION:   Handles the sqlite database using QtSql module. This is the
 *                real class where all the Qt/QSql work is done.
 *
 * AUTHOR:        Nishant Parashar
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

/*****************************************************************
 | includes
 |----------------------------------------------------------------*/

#include "FC_Messaging_SQLite.h"
#include "../FC_Messaging_Debug.h"

#include "../FC_Messaging_service_Messaging.h"
#include "../MsgList/FC_Messaging_PredefinedMsgList.h" 
#include "../../Configuration/Messaging/Msg_FeatureConfig.h"

/******** BOSCH CORRECTION  ************	

Generate the FC_Messaging_PredefinedMsgList.h header file from the Excel document found in :
https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-482206/PredefinedMessages_All_Translations_20120907_Macro.xls

******** BOSCH CORRECTION  ************/

#if defined (DEFAULT_PREDEFINEDMESSAGES_PSA)
#include "DefaultPredefMsgs/PSA/FC_Messaging_DefaultPreDefMsg.h"
#else
#include "DefaultPredefMsgs/AIVI/FC_Messaging_DefaultPreDefMsg.h"
#endif

#include <time.h>
#include <sys/stat.h>   // needed for "chmod(..)", "stat(..)" and enums like: S_IRUSR, S_IWUSR, S_IRGRP, ..
#include <pwd.h>
#include <grp.h>
#include <unistd.h>    // needed for "chown(..)" or 'syncfs(fd)'
#include "sqlite3.h"   // for SQLite error codes, e.g. SQLITE_CORRUPT
//#define MSG_CHECK_QUERY_EXEC__TESTMODE  // ONLY for test of: "-MSG-QUERY-ERR". This complier switch shall not be active in released code. It shall be active only in developer builds for testing.

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_SQLITE
#include "trcGenProj/Header/FC_Messaging_SQLite.cpp.trc.h"
#endif

/*****************************************************************
 | Initialisation of static data members
 |----------------------------------------------------------------*/

//The static FC_Messaging_SQLite interface
FC_Messaging_SQLite* FC_Messaging_SQLite::m_poInstance = NULLPTR;

/*****************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------*/

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Constructor
//!
//
//******************************************************************************
FC_Messaging_SQLite::FC_Messaging_SQLite()
{
   ETG_TRACE_USR4(("FC_Messaging_SQLite()"));

   // GMMY17-7826 - Unable to turn on/off text alerts
   // Initialise the DB every time when system switches from OFF -> NORMAL state
   #if 0
   if (!bInitSQLDatabase())
   {
      ETG_TRACE_ERR(("%s :-ERROR - Cannot initialise database", pszGetClassName()));
      return;
   }
   #endif
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Get the unique one and only instance of this class (singleton design pattern)
//!
//! \return
//!   The only existing instance of this class
//
//******************************************************************************
FC_Messaging_SQLite* FC_Messaging_SQLite::poGetFC_Messaging_SQLite()
{
   if (!m_poInstance)
   {
      ETG_TRACE_USR4(("CREATING DATABASE"));
      m_poInstance = new FC_Messaging_SQLite();
   }

   return m_poInstance;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Delete one and only instance of this class (singleton design pattern)
//!
//! \return
//!   -
//
//******************************************************************************
void FC_Messaging_SQLite::vDeleteInstance()
{
   ETG_TRACE_USR4(("vDeleteInstance() called"));

   if (m_poInstance)
   {
      delete m_poInstance;
   }
   m_poInstance = NULLPTR;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Destructor
//!
//
//******************************************************************************
FC_Messaging_SQLite::~FC_Messaging_SQLite()
{
   ETG_TRACE_USR4(("~FC_Messaging_SQLite() called"));

   QSqlDatabase::removeDatabase(FC_MSG_DB_CONNECTION_NAME);
   m_poInstance = NULLPTR;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Helper function for QSqlQuery to get the current database.
//!  We should never store the QSqlDatabase as it will cause
//!  warnings while closing it.
//!
//!  \return
//!         QSqlDatabase for the current connection
//
//******************************************************************************
QSqlDatabase FC_Messaging_SQLite::GetSQLDatabase()
{
   // GMNGA-53836 - Unclosed file handle due to messaging.out. Invoking the Qt API
   // QSqlDatabase::database() with the connection name alone will open the DB even
   // if it is closed. As the default parameter of this API is true

   return QSqlDatabase::database(FC_MSG_DB_CONNECTION_NAME, false);
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Initialises the database connection
//!
//!  \return
//!        bool  - indicates success or failure
//
//******************************************************************************
bool FC_Messaging_SQLite::bInitSQLDatabase()
{
   ETG_TRACE_USR4(("FC_Messaging_SQLite::bInitSQLDatabase Called"));

   // Security: BGN: Check if database file exists and which settings are active for "permissions", "owner" and "group" (multi-user target)
   int iResult = 0;
   struct stat oStatBuf;

   // Retrieve current "owner", "group" and "permissions" of database file
   iResult = stat(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME, &oStatBuf);
   if( iResult != 0 )
   {
      ETG_TRACE_ERR(( "-MSG.SECURITY- Startup: 'stat(file, &oStatBuf)' has failed with error=%d, the file probably does not exist. [file='%s'] ",
            iResult, FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME ));
   }
   else
   {
      // Print out "owner", "group" and "permissions" of database file
      ETG_TRACE_USR1(( "-MSG.SECURITY- Startup: 'stat' database file: owner_number= '%d' ", oStatBuf.st_uid ));
      ETG_TRACE_USR1(( "-MSG.SECURITY- Startup: 'stat' database file: group_number= '%d' ", oStatBuf.st_gid ));
      ETG_TRACE_USR1(( "-MSG.SECURITY- Startup: 'stat' database file:  permissions= '%d' (octal) ",
            ((oStatBuf.st_mode>>6)&0x07)*100 + ((oStatBuf.st_mode>>3)&0x07)*10 + (oStatBuf.st_mode&0x07) ));
   }
   // Security: END: Check if database file exists and which settings are active for "permissions", "owner" and "group" (multi-user target)

   QDir dir(FC_MSG_DB_DIR_PATH);
   if (!dir.exists())
   {
      if (!QDir().mkpath(FC_MSG_DB_DIR_PATH))
      {
         //try to use the old path.
         ETG_TRACE_ERR(("%s :- ERROR - Cannot create database ", pszGetClassName()));

         dir.setPath(FC_MSG_DB_OLD_DIR_PATH);
         if (!dir.exists())
         {
            ETG_TRACE_ERR(( "%s :- ERROR - Cannot create database", pszGetClassName() ));
            return false;
         }
      }
   }

   QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE", FC_MSG_DB_CONNECTION_NAME);
   db.setDatabaseName(dir.absoluteFilePath(FC_MSG_DB_FILE_NAME));

   if (!db.open())
   {
      ETG_TRACE_ERR(("%s :- ERROR - Cannot open database", pszGetClassName()));
      return false;
   }

   // Security: BEGIN: Check and (if necessary) adapt database file to the wanted "permissions", "owner" and "group" settings (multi-user target),
   // which are as follows: permission= 640 ("-rw-r-----"), user= aid_messaging, group= aid_messaging
   struct passwd *poWantedPasswd = NULLPTR;
   struct group  *poWantedGroup  = NULLPTR;

   // Retrieve current "owner", "group" and "permissions" of database file - after 'db.open()'
   iResult = 0;
   iResult = stat(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME, &oStatBuf);
   if( iResult != 0 )
   {
      // We never expect to come here .. (as "db.open()" should create a new database if there exists currently no one)
      ETG_TRACE_FATAL(( "-MSG.SECURITY- After 'db.open()': 'stat(file, &oStatBuf)' has FAILED with error=%d, the file probably does not exist. [file='%s'] ",
            iResult, FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME ));
   }
   else
   {
      // Check if current "permissions" of the database file matches to the wanted ones, if not try to adapt accordingly
      if( (oStatBuf.st_mode & 0x1FF) != (S_IRUSR | S_IWUSR | S_IRGRP) )
      {
         // Set "-rw-r-----" access for the database file
         ETG_TRACE_ERR(( "-MSG.SECURITY- After Security Check: Correcting the database 'permissions' to '-rw-r-----' " ));
         iResult = chmod(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME, S_IRUSR | S_IWUSR | S_IRGRP);
         if( iResult != 0 )
         {
            ETG_TRACE_ERR(( "-MSG.SECURITY- After Security Check: 'chmod(file, -rw-r-----)' has FAILED with error=%d, [file='%s'] ",
                  iResult, FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME ));
         }
      }

      // Check if current "owner" and "group" matches to the wanted ones, if not try to adapt accordingly
      poWantedPasswd = getpwnam(FC_MSG_DB_USER_NAME);
      poWantedGroup  = getgrnam(FC_MSG_DB_GROUP_NAME);

      if(poWantedPasswd && poWantedGroup)
      {
         // Check if current "owner" and "group" for database are correct. If this not o.k., do the needed adaptations.
         if( (poWantedPasswd->pw_uid != oStatBuf.st_uid) || (poWantedGroup->gr_gid != oStatBuf.st_gid) )
         {
            // Change "owner" and "group" settings if current "owner/group" settings are not correct.
            ETG_TRACE_ERR(( "-MSG.SECURITY- After Security Check: Correcting the database 'owner' and 'group' settings. " ));
            iResult = chown(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME, poWantedPasswd->pw_uid, poWantedGroup->gr_gid);
            if( iResult != 0 )
            {
               ETG_TRACE_ERR(( "-MSG.SECURITY- After Security Check: 'chown(file, %d, %d)' has FAILED with error=%d, [file='%s'] ",
                     poWantedPasswd->pw_uid, poWantedGroup->gr_gid, iResult, FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME ));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(( "-MSG.SECURITY- After Security Check: Cannot resolve the 'wanted' user-name='%s' ", FC_MSG_DB_USER_NAME ));
         ETG_TRACE_ERR(( "-MSG.SECURITY- After Security Check:   or the 'wanted' group-name='%s' ", FC_MSG_DB_GROUP_NAME ));
      }

      // Print out "owner", "group" and "permissions" of database file (after security check) and related changes might have been done.
      iResult = 0;
      iResult = stat(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME, &oStatBuf);
      if( iResult != 0 )
      {
         ETG_TRACE_ERR(( "-MSG.SECURITY- After Security Check: 'stat(file, &oStatBuf)' has FAILED with error=%d, the file probably does not exist. [file='%s'] ",
               iResult, FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_FILE_NAME ));
      }
      else
      {
         ETG_TRACE_USR1(( "-MSG.SECURITY- After Security Check:  'stat' database file: owner_number= '%d' ", oStatBuf.st_uid ));
         ETG_TRACE_USR1(( "-MSG.SECURITY- After Security Check:  'stat' database file: group_number= '%d' ", oStatBuf.st_gid ));
         ETG_TRACE_USR1(( "-MSG.SECURITY- After Security Check:  'stat' database file:  permissions= '%d' (octal) ",
               ((oStatBuf.st_mode>>6)&0x07)*100 + ((oStatBuf.st_mode>>3)&0x07)*10 + (oStatBuf.st_mode&0x07) ));
      }
   }
   // Security: END: Check and (if necessary) adapt database file to the wanted "permissions", "owner" and "group" settings (multi-user target)

   bCreateSystemWideSettingsDatabase();
   bCreateDeviceSettingsDatabase();
   bCreatePredefinedMessageDatabase();

   return true;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Initialises the SystemWideSettings Database
//!
//!  \return
//!        bool  - indicates success or failure
//
//******************************************************************************
bool FC_Messaging_SQLite::bCreateSystemWideSettingsDatabase()
{
    return bCheckSystemWideSettingsTableSchema();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Validate the sql Message Setting Table Schema
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//******************************************************************************
bool FC_Messaging_SQLite::bCheckSystemWideSettingsTableSchema()
{
   ETG_TRACE_USR4(("bCheckSystemWideSettingsTableSchema entered"));

   // Bugfix for GMMY16-7867
   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   QStringList list = db.tables();

   if (list.contains(FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME))
   {
      if (!query.exec("PRAGMA table_info(" FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME ")"))
      {
         MSG_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR(("ERROR - SQL query failed in CheckSystemWideSettingsTableSchema - 1"));
         return false;
      }

      list.clear();

      while (query.next())
         list << query.value(1).toString();

      //database consistency check for SystemWideSettings table
      if ((list.contains(FC_MSG_COL_EMAIL_ALERT)) && (list.contains(FC_MSG_COL_TEXT_ALERT))&&
            (list.contains(FC_MSG_COL_SAVE_SENT_EMAIL)) && (list.contains(FC_MSG_COL_SAVE_SENT_TEXT)) &&
            (list.contains(FC_MSG_COL_AUTO_UPDATE_INBOX)) && (list.contains(FC_MSG_COL_UPDATE_INBOX_PERIOD)) &&
            (list.contains(FC_MSG_COL_MSG_LANG )))
      {
         ETG_TRACE_USR4(("bCheckSystemWideSettingsTableSchema - SystemWideSettings table exists"));
         return true;
      }
      else
      {
         if (!query.exec("DROP TABLE " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME))
         {
            MSG_CHECK_QUERY_ERROR(query);
            ETG_TRACE_ERR(("ERROR - SQL query failed in bCheckSystemWideSettingsTableSchema - 2"));
            return false;
         }
      }
   }
   return bCreateSystemWideSettingsTable();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    QtSql specific function to Create Message Setting table
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//******************************************************************************
bool FC_Messaging_SQLite::bCreateSystemWideSettingsTable()
{
   ETG_TRACE_USR4(("FC_Messaging_SQLite::bCreateSystemWideSettingsTable entered"));

   // Bugfix for GMMY16-7867
   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   static const QString sql = "CREATE TABLE IF NOT EXISTS " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME " "
   "(" FC_MSG_COL_EMAIL_ALERT " BOOLEAN,"
   " " FC_MSG_COL_TEXT_ALERT " BOOLEAN,"
   " " FC_MSG_COL_SAVE_SENT_EMAIL " BOOLEAN,"
   " " FC_MSG_COL_SAVE_SENT_TEXT " BOOLEAN,"
   " " FC_MSG_COL_AUTO_UPDATE_INBOX " BOOLEAN,"
   " " FC_MSG_COL_UPDATE_INBOX_PERIOD " INTEGER,"
   " " FC_MSG_COL_MSG_LANG " INTEGER);";

   ETG_TRACE_USR4(("bCreateSystemWideSettingsTable entered %s", sql.toLatin1().constData()));

   QSqlQuery query(db);

   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in SystemWideSettings Create Table ", pszGetClassName()));
      return false;
   }

   // Insert the default settings in SystemWideSettings Table
   bQuerySetDefaultSystemWideSettings();

   return true;
}

// ***************** F U N C T I O N  H E A D E R *********************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to Enable/Disable System wide Settings in database
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//*********************************************************************************
bool FC_Messaging_SQLite::bQuerySetDefaultSystemWideSettings()
{
   ETG_TRACE_USR4(("bQuerySetDefaultSystemWideSettings() called"));

   // Bugfix for GMMY16-7867
   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   bool enableSetting = TRUE;

   QSqlQuery queryTrunc(db);

   if (!queryTrunc.exec("DELETE FROM " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME))
   {
      MSG_CHECK_QUERY_ERROR(queryTrunc);
      QByteArray ba = queryTrunc.lastError().text().toLatin1(); //Fix for CMG3G-5593
      ETG_TRACE_ERR(("ERROR - %s SQL query failed in SetSystemWideSettings ", ba.constData()));
      return false;
   }

   static const QString sql = "INSERT INTO " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME " "
   "(" FC_MSG_COL_EMAIL_ALERT ", "
   " " FC_MSG_COL_TEXT_ALERT ", "
   " " FC_MSG_COL_SAVE_SENT_EMAIL ", "
   " " FC_MSG_COL_SAVE_SENT_TEXT ", "
   " " FC_MSG_COL_AUTO_UPDATE_INBOX ", "
   " " FC_MSG_COL_UPDATE_INBOX_PERIOD ", "
   " " FC_MSG_COL_MSG_LANG ") "
   "VALUES (?, ?, ?, ?, ?, ?, ?);";

   ETG_TRACE_USR4(("FC_Messaging_SQLite::bQuerySetSystemWideSettings entered %s", sql.toLatin1().constData()));

   QSqlQuery query(db);
   query.prepare(sql);
   query.addBindValue(enableSetting);
   query.addBindValue(enableSetting);
   query.addBindValue(enableSetting);
   query.addBindValue(enableSetting);
   query.addBindValue(enableSetting);
   query.addBindValue(FC_MSG_AUTO_UPDATE_INBOX_PERIOD);
   query.addBindValue(FC_MSG_DEFAULT_MSG_LANG);

   if (!query.exec())
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1(); //Fix for CMG3G-5593
      ETG_TRACE_ERR(("ERROR - %s SQL query failed in inserting SystemWideSettings", ba.constData()));
      return false;
   }
   return true;
}

// ***************** F U N C T I O N  H E A D E R *********************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to Update Inbox Setting in MsgSetting table
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//*********************************************************************************
bool FC_Messaging_SQLite::bQueryUpdateInboxSetting
		          (unsigned int uiUpdateInboxPeriod, bool autoUpdate)
{
   ETG_TRACE_USR4(("bQueryUpdateInboxSetting : Called"));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   static const QString sqlUpdate = "UPDATE " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME " "
         " SET " FC_MSG_COL_AUTO_UPDATE_INBOX " = ? , "
         " " FC_MSG_COL_UPDATE_INBOX_PERIOD " = ? ";

   QSqlQuery query(db);
   query.prepare(sqlUpdate);
   query.addBindValue(autoUpdate);
   query.addBindValue(uiUpdateInboxPeriod);

   if (!query.exec())
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1(); //Fix for CMG3G-5593
      ETG_TRACE_ERR(( "ERROR - %s SQL query failed in Updating Inbox Setting", ba.constData()));
      return false;
   }

   return true;
}

bool FC_Messaging_SQLite::bQueryUpdateMessagingLanguage(unsigned int uiMsgLang)
 {
   ETG_TRACE_USR4(("bQueryUpdateMessagingLanguage : Called"));
   ETG_TRACE_USR4(("bQueryUpdateMessagingLanguage : uiMsgLang= %d", uiMsgLang));

   QSqlDatabase db = GetSQLDatabase();

   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   static const QString sqlUpdateLang = "UPDATE " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME " "
   " SET " FC_MSG_COL_MSG_LANG " = ? ";

   QSqlQuery query(db);
   query.prepare(sqlUpdateLang);
   query.addBindValue(uiMsgLang);

   ETG_TRACE_USR4(("bQueryUpdateMessagingLanguage:: query to be executed::sqlUpdateLang = %s", sqlUpdateLang.toLatin1().constData()));

   if (!query.exec())
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1();
      ETG_TRACE_ERR(("ERROR - %s SQL query failed in Updating Msg Lang", ba.constData()));
      return false;
   }

   FC_Messaging_PredefinedMsgList::setLanguageSetting(static_cast <tU8> (uiMsgLang));
   newUserPredefinedMessageAdded();
   
   return true;
}
// ***************** F U N C T I O N  H E A D E R *********************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to update SystemWideSettings in database
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//*********************************************************************************
bool FC_Messaging_SQLite::bQueryUpdateSystemWideSettings(SystemWideSettings enSystemWideSetting, bool bSettingOn)
{
   ETG_TRACE_USR4(("bQueryUpdateSystemWideSettings() called"));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   static QString sqlUpdate;

   switch (enSystemWideSetting)
   {
      case EMAIL_ALERT_SETTING:
            sqlUpdate = "UPDATE " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME " "
            " SET " FC_MSG_COL_EMAIL_ALERT " = ? ";
            break;

      case TEXT_ALERT_SETTING:
            sqlUpdate = "UPDATE " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME " "
            " SET " FC_MSG_COL_TEXT_ALERT " = ? ";
            break;

      case SAVE_SENT_EMAIL_SETTING:
            sqlUpdate = "UPDATE " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME " "
            " SET " FC_MSG_COL_SAVE_SENT_EMAIL " = ? ";
            break;

     case SAVE_SENT_TEXT_SETTING:
            sqlUpdate = "UPDATE " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME " "
            " SET " FC_MSG_COL_SAVE_SENT_TEXT " = ? ";
            break;

      default:
            sqlUpdate = "";
            break;
   }
   if (sqlUpdate.isEmpty())
   {
      ETG_TRACE_ERR(("ERROR - Trying to Update Incorrect Setting "));
      return false;
   }

   QSqlQuery query(db);
   query.prepare(sqlUpdate);
   query.addBindValue(bSettingOn);

   if (!query.exec())
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1(); //Fix for CMG3G-5593
      ETG_TRACE_ERR(("ERROR - %s  SQL query failed in Updating SystemWideSettings", ba.constData()));
      return false;
   }
#ifdef MSG_CHECK_QUERY_EXEC__TESTMODE
   MSG_CHECK_QUERY_ERROR(query); // This is only for TEST (Error Injection) of: -MSG-QUERY-ERR-
#endif
   return true;
}

// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get the AutoUpdateInbox period from the database
//!
//!  \return
//!        unsigned int - Auto Update Inbox period
//
//*******************************************************************************
unsigned int FC_Messaging_SQLite::szQueryGetUpdateInboxPeriod()
{
   ETG_TRACE_USR4(("szQueryGetUpdateInboxPeriod Entered"));

   unsigned int autoUpdateInboxPeriod = FC_MSG_AUTO_UPDATE_INBOX_PERIOD;

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return autoUpdateInboxPeriod;
   }

   static const QString sql = "SELECT " FC_MSG_COL_UPDATE_INBOX_PERIOD " FROM " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME;

   QSqlQuery query(db);

   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR (( "%s :- ERROR - SQL query failed in updateInboxPeriod ", pszGetClassName() ));
      return autoUpdateInboxPeriod;
   }

   if (query.next())
   {
      autoUpdateInboxPeriod = query.value(0).toUInt();
   }
   else
   {
      ETG_TRACE_USR4 (( "%s :- WARNING - Empty list of updateInboxPeriod",pszGetClassName()));
   }

   return autoUpdateInboxPeriod;
}

unsigned int FC_Messaging_SQLite::szQueryGetMessagingLanguage()
{
   ETG_TRACE_USR4(("szQueryGetMessagingLanguage Entered"));

   unsigned int MsgLang = FC_MSG_DEFAULT_MSG_LANG;

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return MsgLang;
   }

   static const QString sql =
         "SELECT " FC_MSG_COL_MSG_LANG " FROM " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME;

   QSqlQuery query(db);

   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in getting MsgLang ", pszGetClassName()));
      return MsgLang;
   }

   if (query.next())
   {
      MsgLang = query.value(0).toUInt();
   }
   else
   {
      ETG_TRACE_USR4(("%s :- WARNING - Empty list returned while fetching MsgLang", pszGetClassName()));
   }

   return MsgLang;
}

// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get the SystemWideSettings from the database
//!
//!  \return
//!        bool  - Setting
//
//*******************************************************************************
bool FC_Messaging_SQLite::szQueryGetSystemWideSettings(SystemWideSettings enSystemWideSetting)
{
   ETG_TRACE_USR4(("szQueryGetSystemWideSettings() called"));
   bool isSettingEnable = TRUE;

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return isSettingEnable;
   }

   QString sqlGetValue;

   switch (enSystemWideSetting)
   {
      case EMAIL_ALERT_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_EMAIL_ALERT " FROM " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME;
         break;

      case TEXT_ALERT_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_TEXT_ALERT " FROM " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME;
         break;

      case SAVE_SENT_EMAIL_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_SAVE_SENT_EMAIL " FROM " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME;
         break;

      case SAVE_SENT_TEXT_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_SAVE_SENT_TEXT " FROM " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME;
         break;

      case AUTO_UPDATE_INBOX_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_AUTO_UPDATE_INBOX " FROM " FC_MSG_SYSTEMWIDESETTINGS_TABLE_NAME;
         break;

      default:
         sqlGetValue = "";
         break;
   }
   if (sqlGetValue.isEmpty())
   {
      ETG_TRACE_ERR(("ERROR - Trying to Read Incorrect SystemWideSettings "));
      return false;
   }

   QSqlQuery query(db);

   if (!query.exec(sqlGetValue))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in Getting SystemWideSettings ", pszGetClassName()));
      return isSettingEnable;
   }

   if (query.next())
   {
      isSettingEnable = query.value(0).toBool();
   }
   else
   {
      ETG_TRACE_USR4(("%s :- WARNING - Empty list of SystemWideSettings", pszGetClassName()));
   }

   return isSettingEnable;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    QtSql specific function to create a predefined messages database
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//******************************************************************************
bool FC_Messaging_SQLite::bCreatePredefinedMessageDatabase()
{
    return bCheckPredefinedMessageTableSchema();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    QtSql specific function to create a predefined messages table
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//******************************************************************************
bool FC_Messaging_SQLite::bCreateTablePreDefinedTable()
{
   ETG_TRACE_USR4(("bCreateTablePreDefinedTable() called"));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
   static const QString sqlPredefined = "CREATE TABLE IF NOT EXISTS " FC_MSG_PREDEFMSG_TABLE_NAME " "
   "(PID INTEGER PRIMARY KEY, "
   " " FC_MSG_COL_PRE_MSG " TEXT,"
   " " FC_MSG_COL_PRE_MSG_TYPE " INTEGER,"
   " " FC_MSG_COL_PRE_MSG_CREATE_TYPE " INTEGER,"
   " " FC_MSG_COL_PRE_MSG_LAN_TYPE " INTEGER,"
   " " FC_MSG_COL_PRE_MSG_MOD_TIME " INTEGER,"
   " " FC_MSG_COL_PRE_MSG_SUB_CAT " INTEGER,"
   " " FC_MSG_COL_DEVICE_HANDLE " INTEGER);";
#else
   static const QString sqlPredefined = "CREATE TABLE IF NOT EXISTS " FC_MSG_PREDEFMSG_TABLE_NAME " "
      "(PID INTEGER PRIMARY KEY, "
      " " FC_MSG_COL_PRE_MSG " TEXT,"
      " " FC_MSG_COL_PRE_MSG_TYPE " INTEGER,"
      " " FC_MSG_COL_PRE_MSG_CREATE_TYPE " INTEGER,"
      " " FC_MSG_COL_PRE_MSG_LAN_TYPE " INTEGER,"
      " " FC_MSG_COL_PRE_MSG_MOD_TIME " INTEGER,"
      " " FC_MSG_COL_DEVICE_HANDLE " INTEGER);";
#endif

   QSqlQuery query(db);

   if (!query.exec(sqlPredefined))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(( "%s :- ERROR - SQL query failed in CreateTable for Predefined Messages ",pszGetClassName() ));
      return false;
   }

   return bAddSystemPredefinedMessage();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Function to add system predefined messages to predefined msg database
//!
//!  \return
//!        bool  - indicates success or failure of adding system predefined msgs
//
//******************************************************************************

/******** BOSCH CORRECTION  ************/
 bool FC_Messaging_SQLite::bAddSystemPredefinedMessage()
{
   ETG_TRACE_USR2(("FC_Messaging_SQLite::bAddSystemPredefinedMessage entered"));

   /* Bugfix for GMMY16-5730: SHUTDOWN_AFTER_NO_STATE_REQUEST state
    * change failed of CCA_C_U16_APP_MESSAGING (OFF -> NORMAL) */
   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
	   ETG_TRACE_ERR(("Database is not open"));
	   return false;
   }

   quint16 u16Handle = 0;
   QMutexLocker lock(&m_mutex);
   tBool bNewQuery = true;
   QString qsSqlQueryString;

   for (tU8 u8LangIndex = 0; u8LangIndex < LANG_END; ++u8LangIndex)
   {
      for (tU8 u8TextMsgIndex = 0; u8TextMsgIndex < txtTEXT_MSG_END; ++u8TextMsgIndex)
      {
         if(true == bNewQuery)
         {
#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
            qsSqlQueryString = "INSERT INTO " FC_MSG_PREDEFMSG_TABLE_NAME " "
                  "( PID, "
                  " " FC_MSG_COL_PRE_MSG " , "
                  " " FC_MSG_COL_PRE_MSG_TYPE " , "
                  " " FC_MSG_COL_PRE_MSG_CREATE_TYPE " , "
                  " " FC_MSG_COL_PRE_MSG_LAN_TYPE " , "
                  " " FC_MSG_COL_PRE_MSG_MOD_TIME " , "
                  " " FC_MSG_COL_PRE_MSG_SUB_CAT " , "
                  " " FC_MSG_COL_DEVICE_HANDLE " ) ";
#else
            qsSqlQueryString = "INSERT INTO " FC_MSG_PREDEFMSG_TABLE_NAME " "
                  "( PID, "
                  " " FC_MSG_COL_PRE_MSG " , "
                  " " FC_MSG_COL_PRE_MSG_TYPE " , "
                  " " FC_MSG_COL_PRE_MSG_CREATE_TYPE " , "
                  " " FC_MSG_COL_PRE_MSG_LAN_TYPE " , "
                  " " FC_MSG_COL_PRE_MSG_MOD_TIME " , "
                  " " FC_MSG_COL_DEVICE_HANDLE " ) ";
#endif

            bNewQuery = false;
         }
         else
         {
            qsSqlQueryString += " UNION ";
         }

         /* Constructing two different QStrings and concatenating the same at the end, since
            the dynamic predefined messages had a PERCENTAGE sign and it is resolved as
            format specifier by QT, resulting in formation of invalid query */

         qsSqlQueryString += QString("SELECT %1, \"%2\", ").arg(++u16Handle). \
            arg(QString(strlistPreDefMsg[u16MostLangArray[u8LangIndex]][u8TextMsgIndex]));
#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
         QString qsSqlTemp = QString("%1, %2, %3, %4, %5, %6").arg(arrMsgType[u8TextMsgIndex]). \
            arg(FC_MSG_SYS_PRE_DEF_MSG).arg(u8LangIndex).arg(1).arg(arrMsgSubCategory[u8TextMsgIndex+1]).arg(0);
#else
         
         QString qsSqlTemp = QString("%1, %2, %3, %4, %5").arg(arrMsgType[u8TextMsgIndex]). \
            arg(FC_MSG_SYS_PRE_DEF_MSG).arg(u8LangIndex).arg(1).arg(0);
#endif
         qsSqlQueryString += qsSqlTemp;
      }

      // Insert only 48 Messages i.e. Predefined Messages of 4 languages alone at a time.
      // More than this would result in a stack overflow.

      if (((u8LangIndex != 0) && ((u8LangIndex % 4) == 0)) || (u8LangIndex == (LANG_END - 1)))
      {
         ETG_TRACE_USR4(("bAddSystemPredefinedMessage index: %d", u8LangIndex));
         QSqlQuery query(db);
         if (db.isOpen())
         {
            if (!query.exec(qsSqlQueryString))
            {
               MSG_CHECK_QUERY_ERROR(query);
               QByteArray ba = query.lastError().text().toLatin1(); //Fix for CMG3G-5593
               ETG_TRACE_ERR(("ERROR - %s SQL query failed in Adding predef msg", ba.constData()));
            }
         }
         else
         {
            ETG_TRACE_ERR(("Database is not open"));
            return false; // Bugfix for GMMY16-7867
         }

         bNewQuery = true;
         qsSqlQueryString.clear();
      }
   }

   ETG_TRACE_USR2(("Completed Adding Predefined Messages"));
   return true;
}
/******** BOSCH CORRECTION  ************/


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Helper function to validate the sql table schema
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//******************************************************************************
bool FC_Messaging_SQLite::bCheckPredefinedMessageTableSchema()
{
   ETG_TRACE_USR4(("bCheckPredefinedMessageTableSchema: Called"));

   // Bugfix for GMMY16-7867
   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   QStringList list = db.tables();

   //Predefined Message Table Check Schema Table
   if (list.contains(FC_MSG_PREDEFMSG_TABLE_NAME))
   {
      if (!query.exec("PRAGMA table_info(" FC_MSG_PREDEFMSG_TABLE_NAME ")"))
      {
         MSG_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR(("ERROR - SQL query failed in CreateTable for Predefined Messages"));
         return false;
      }

      list.clear();

      while (query.next())
         list << query.value(1).toString();

#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
      if (list.contains("PID") && list.contains(FC_MSG_COL_PRE_MSG)
            && list.contains(FC_MSG_COL_PRE_MSG_TYPE) && list.contains(FC_MSG_COL_PRE_MSG_CREATE_TYPE)
            && list.contains(FC_MSG_COL_PRE_MSG_MOD_TIME) && list.contains(FC_MSG_COL_PRE_MSG_LAN_TYPE)
            && list.contains(FC_MSG_COL_PRE_MSG_SUB_CAT) && list.contains(FC_MSG_COL_DEVICE_HANDLE))
#else
      if (list.contains("PID") && list.contains(FC_MSG_COL_PRE_MSG)
            && list.contains(FC_MSG_COL_PRE_MSG_TYPE) && list.contains(FC_MSG_COL_PRE_MSG_CREATE_TYPE)
            && list.contains(FC_MSG_COL_PRE_MSG_MOD_TIME) && list.contains(FC_MSG_COL_PRE_MSG_LAN_TYPE)
            && list.contains(FC_MSG_COL_DEVICE_HANDLE))
#endif
      {
         ETG_TRACE_USR4(("bCheckPredefinedMessageTableSchema - Predef table exists"));
         return true;
      }
      else
      {
         if (!query.exec("DROP TABLE " FC_MSG_PREDEFMSG_TABLE_NAME))
         {
            MSG_CHECK_QUERY_ERROR(query);
            ETG_TRACE_ERR(("ERROR - SQL query failed in CheckTableSchema 2"));
         }
      }
   }
   return bCreateTablePreDefinedTable();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to insert the predefined messages into database
//!
//!  \return
//!        bool  - indicating success or failure of this query
//
//******************************************************************************
bool FC_Messaging_SQLite::bQueryInsertIntoPredefinedMessage(
      //! [IN] DeviceHandle
      tU8 u8DeviceHandle,

      //! [IN] The new predefined message
      const QString& strPreMessage,

      //! [IN] Predefined Message type:
      //! 1 : Static
      //! 2 : Intersection
      //! 3 : ETA
      quint16 u16PredefMsgType,

      //! [IN] CreationType
      //! 1 : System created messages
      //! 2 : User created messages
      quint16 u16PredefMsgCreateType,

      //! [OUT] Unique Handle for inserted message
      quint16& u16Handle,

      //! [IN] LanguageType
      tU16 u16LanguageType,

      //! [IN] SubCategory (0-4)
      quint16 u16SubCategory
)
{
   ETG_TRACE_USR4(("bQueryInsertIntoPredefinedMessage: Entered"));
   (void) u16SubCategory;

   /* Bugfix for GMMY16-5730: SHUTDOWN_AFTER_NO_STATE_REQUEST state
    * change failed of CCA_C_U16_APP_MESSAGING (OFF -> NORMAL) */
   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QMutexLocker lock(&m_mutex);

   //check the total number of predefined messages is not more than 50

   quint16 u16PredefMsgCnt = 0;
   if (u16PredefMsgCreateType == FC_MSG_USER_PRE_DEF_MSG)
   {
      u16GetTotalPredefMsgCount(u16PredefMsgCnt, u16LanguageType);

      if (u16PredefMsgCnt > FC_MSG_MAX_PREDEF_MSG_COUNT)
      {
         ETG_TRACE_ERR(("ERROR - Maximum number of predefined messages reached "));
         return false;
      }
   }

   qint32 l_s32LatestMessageTime;
   if (u16PredefMsgCnt > 0)
   {
      l_s32LatestMessageTime = s32GetLatestMessageTime(u16LanguageType);
   }
   else
   {
      l_s32LatestMessageTime = 0;
   }

   l_s32LatestMessageTime = l_s32LatestMessageTime + 1;

   quint16 u16PreDefMsghandle = u16GetNewUniquePredefMsgHandle();

   ETG_TRACE_USR2(("DeviceHandle: %d, PreDefMsghandle: %u, MsgType %u, CreateType: %u, PredefMsg: %s ",
         u8DeviceHandle, u16PreDefMsghandle, u16PredefMsgType, u16PredefMsgCreateType, strPreMessage.toLatin1().constData())); //Fix for CMG3G-5593

#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
   static const QString sql = "INSERT INTO " FC_MSG_PREDEFMSG_TABLE_NAME " "
         "( PID, "
         " " FC_MSG_COL_PRE_MSG " , "
         " " FC_MSG_COL_PRE_MSG_TYPE " , "
         " " FC_MSG_COL_PRE_MSG_CREATE_TYPE " , "
         " " FC_MSG_COL_PRE_MSG_LAN_TYPE " , "
         " " FC_MSG_COL_PRE_MSG_MOD_TIME " , "
         " " FC_MSG_COL_PRE_MSG_SUB_CAT " , "
         " " FC_MSG_COL_DEVICE_HANDLE " ) "
         "VALUES (?, ?, ?, ?, ?, ?, ?, ?);";
#else
   static const QString sql = "INSERT INTO " FC_MSG_PREDEFMSG_TABLE_NAME " "
         "( PID, "
         " " FC_MSG_COL_PRE_MSG " , "
         " " FC_MSG_COL_PRE_MSG_TYPE " , "
         " " FC_MSG_COL_PRE_MSG_CREATE_TYPE " , "
         " " FC_MSG_COL_PRE_MSG_LAN_TYPE " , "
         " " FC_MSG_COL_PRE_MSG_MOD_TIME " , "
         " " FC_MSG_COL_DEVICE_HANDLE " ) "
         "VALUES (?, ?, ?, ?, ?, ?, ?);";
#endif

   QSqlQuery query(db);
   query.prepare(sql);
   query.addBindValue(u16PreDefMsghandle);
   query.addBindValue(strPreMessage);
   query.addBindValue(u16PredefMsgType);
   query.addBindValue(u16PredefMsgCreateType);
   query.addBindValue(u16LanguageType);
   query.addBindValue(l_s32LatestMessageTime);
#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
   query.addBindValue(u16SubCategory);
#endif
   query.addBindValue(u8DeviceHandle);

   if (db.isOpen())
   {
      if (!query.exec())
      {
         MSG_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR(("ERROR - SQL query failed in SetSign "));
         QByteArray ba = query.lastError().text().toLatin1(); //Fix for CMG3G-5593
         ETG_TRACE_ERR(("ERROR - %s", ba.constData()));
         return false;
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   u16Handle = u16PreDefMsghandle;

   if (u16PredefMsgCreateType == FC_MSG_USER_PRE_DEF_MSG)
   {
      newUserPredefinedMessageAdded();
      fc_messaging_tclService_Messaging::m_poMessagingService->PredefinedListChangeStatus();
   }

   return true;
}

void FC_Messaging_SQLite::newUserPredefinedMessageAdded()
{
	ETG_TRACE_USR4(("newUserPredefinedMessageAdded : Called"));

	/* Bugfix for GMMY16-5730: SHUTDOWN_AFTER_NO_STATE_REQUEST state
	 * change failed of CCA_C_U16_APP_MESSAGING (OFF -> NORMAL) */
	QSqlDatabase db = GetSQLDatabase();
	if (!db.isOpen())
	{
		ETG_TRACE_ERR(("Database is not open"));
		return;
	}

    //QMutexLocker lock(&m_mutex);
    tU8 u8CurrentLanguageSetting = 0;
    quint16 temp;

   u8CurrentLanguageSetting = FC_Messaging_PredefinedMsgList::getLanguageSetting();
   ETG_TRACE_USR4(("newUserPredefinedMessageAdded : u8CurrentLanguageSetting =%d",u8CurrentLanguageSetting));

    temp = u8CurrentLanguageSetting;

    QString str = QString::number(temp);

	/*FixGMMY15-5294:After adding a new user Pre-Defined message, the complete Pre-Defined message list were selected from the Messaging-DB in descending 
        order without mentioning the PID column, due to which the message list were sorted & retrieved alphabetically and the same list was sent to HMI.*/
#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
    QString
            sql =
            "SELECT PID, " FC_MSG_COL_PRE_MSG_TYPE " ," FC_MSG_COL_PRE_MSG " ," FC_MSG_COL_PRE_MSG_CREATE_TYPE "," FC_MSG_COL_PRE_MSG_SUB_CAT "," FC_MSG_COL_DEVICE_HANDLE " FROM " FC_MSG_PREDEFMSG_TABLE_NAME "  WHERE " FC_MSG_COL_PRE_MSG_LAN_TYPE " = "
            + str
            + " ORDER BY " FC_MSG_COL_PRE_MSG_MOD_TIME " DESC, PID";
#else
    QString
                sql =
                "SELECT PID, " FC_MSG_COL_PRE_MSG_TYPE " ," FC_MSG_COL_PRE_MSG " ," FC_MSG_COL_PRE_MSG_CREATE_TYPE "," FC_MSG_COL_DEVICE_HANDLE " FROM " FC_MSG_PREDEFMSG_TABLE_NAME "  WHERE " FC_MSG_COL_PRE_MSG_LAN_TYPE " = "
                + str
                + " ORDER BY " FC_MSG_COL_PRE_MSG_MOD_TIME " DESC, PID";
#endif
    QSqlQuery query(db);

    if (!query.exec(sql))
    {
        MSG_CHECK_QUERY_ERROR(query);
        ETG_TRACE_ERR(("SQL query failed in newUserPredefinedMessageAdded"));
        return;
    }

    fc_messaging_tclService_Messaging::m_poMessagingService->m_oPredefinedMessageList_Status.oPredefinedMessageListResult.oItems.clear();
   fc_messaging_tclService_Messaging::m_poMessagingService->m_oPredefinedMessageListExtended_Status.oPredefinedMessageListExtendedResult.oItems.clear();
    most_fi_tcl_MsgPredefinedMessageListResultItem
            oPredefinedMessageListResultItem[FC_MSG_MAX_PREDEF_MSG_COUNT];
    most_fi_tcl_MsgPredefinedMessageListExtendedResultItem
    		oPredefinedMessageListExtendedResultItem[FC_MSG_MAX_PREDEF_MSG_COUNT];

    int i = 0;
    while (db.isOpen() && (query.next()) && (i < FC_MSG_MAX_PREDEF_MSG_COUNT))
    {
      tU8 u8MessageCategory = 0;
      oPredefinedMessageListResultItem[i].u16PredefinedMessageHandle
      = oPredefinedMessageListExtendedResultItem[i].u16PredefinedMessageHandle
                = (tU16)query.value(0).toUInt();
      oPredefinedMessageListResultItem[i].e8PredefinedMessageType.enType
      = oPredefinedMessageListExtendedResultItem[i].e8PredefinedMessageType.enType
      = (most_fi_tcl_e8_MsgPredefinedMessageType::tenType) query.value(1).toUInt();

	  oPredefinedMessageListResultItem[i].sPredefinedMessageText.bSet(query.value(2).toString().toUtf8().constData()); //Fix for CMG3G-5593
	  oPredefinedMessageListExtendedResultItem[i].sPredefinedMessageText.bSet(query.value(2).toString().toUtf8().constData());
        ETG_TRACE_USR4(("newUserPredefinedMessageAdded : FC_MSG_COL_PRE_MSG_CREATE_TYPE value %d::",query.value(3).toUInt()));
        if(1 == query.value(3).toUInt())
        	u8MessageCategory = 1;

      ETG_TRACE_USR4(("newUserPredefinedMessageAdded : u8MessageCategory value %d::", u8MessageCategory));
        oPredefinedMessageListExtendedResultItem[i].e8PredefinedMessageCategory.enType
        =(most_fi_tcl_e8_MsgPredefinedMessageCategory::tenType)u8MessageCategory ;

#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
        ETG_TRACE_USR4(("newUserPredefinedMessageAdded : u8MessageSubCategory value %d::",query.value(4).toUInt()));
        oPredefinedMessageListExtendedResultItem[i].e8PredefinedMessageSubCategory.enType
                =(most_fi_tcl_e8_MsgPredefinedMessageSubCategory::tenType)query.value(4).toUInt();
        ETG_TRACE_USR4(("newUserPredefinedMessageAdded : DeviceHandle value %d::",query.value(5).toUInt()));
        oPredefinedMessageListExtendedResultItem[i].u8DeviceHandle = (tU8)query.value(5).toUInt();
#else
        //ETG_TRACE_USR4(("newUserPredefinedMessageAdded : u8MessageSubCategory value %d::",query.value(4).toUInt()));
        oPredefinedMessageListExtendedResultItem[i].e8PredefinedMessageSubCategory.enType
                = most_fi_tcl_e8_MsgPredefinedMessageSubCategory::FI_EN_E8NO_SUB_CATEGORY;
        ETG_TRACE_USR4(("newUserPredefinedMessageAdded : DeviceHandle value %d::",query.value(4).toUInt()));
        oPredefinedMessageListExtendedResultItem[i].u8DeviceHandle = (tU8)query.value(4).toUInt();
#endif

        fc_messaging_tclService_Messaging::m_poMessagingService->m_oPredefinedMessageList_Status.oPredefinedMessageListResult.oItems.push_back(
                    oPredefinedMessageListResultItem[i]);
        fc_messaging_tclService_Messaging::m_poMessagingService->m_oPredefinedMessageListExtended_Status.oPredefinedMessageListExtendedResult.oItems.push_back(
        		    oPredefinedMessageListExtendedResultItem[i]);
        i++;
    }

   fc_messaging_tclService_Messaging::m_poMessagingService->m_oPredefinedMessageList_Status.u8NumPredefinedMessages = (tU8)i;
   fc_messaging_tclService_Messaging::m_poMessagingService->m_oPredefinedMessageListExtended_Status.u8NumPredefinedMessages = (tU8)i;
}

void FC_Messaging_SQLite::setDataBaseState(bool stateOn)
{
   ETG_TRACE_USR4(("FC_Messaging_SQLite::setDataBaseState Entered"));

   QSqlDatabase db = GetSQLDatabase();

   if (db.isOpen() && !stateOn)
   {
      ETG_TRACE_USR4(("setDataBaseState: DATABASE CLOSED"));
      db.close();
      // Bugfix GMMY15-8509 - Reset ticket - Remove the DB connection also after closing DB
      QSqlDatabase::removeDatabase(FC_MSG_DB_CONNECTION_NAME);
   }
   else if (!db.isOpen() && stateOn)
   {
      ETG_TRACE_USR4(("setDataBaseState: DATABASE OPEN"));
      if (!db.open())
      {
         ETG_TRACE_ERR(("%s :- ERROR - Cannot open database", pszGetClassName()));
      }
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to select/fetch the predefined messages list
//!
//!  \return
//!       -
//
//******************************************************************************
tVoid FC_Messaging_SQLite::vQuerySelectPredefinedMessageList()
{
   ETG_TRACE_USR4(("vQuerySelectPredefinedMessageList: Entered"));

   /* Bugfix for GMMY16-5730: SHUTDOWN_AFTER_NO_STATE_REQUEST state
    * change failed of CCA_C_U16_APP_MESSAGING (OFF -> NORMAL) */
   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return;
   }

   QMutexLocker lock(&m_mutex);
   tU8 u8CurrentLanguageSetting = 0;
   quint16 temp;
   u8CurrentLanguageSetting = FC_Messaging_PredefinedMsgList::getLanguageSetting();

   ETG_TRACE_USR2(("Current language is %u", u8CurrentLanguageSetting));
   temp = u8CurrentLanguageSetting;
   ETG_TRACE_USR2(("Current language is passed to query %u", temp));

   QString str = QString::number(temp);

   QString sql = "SELECT PID, " FC_MSG_COL_PRE_MSG_TYPE " ," FC_MSG_COL_PRE_MSG " FROM " FC_MSG_PREDEFMSG_TABLE_NAME "  WHERE " FC_MSG_COL_PRE_MSG_LAN_TYPE " = "
         + str
         + " ORDER BY " FC_MSG_COL_PRE_MSG_MOD_TIME " DESC, PID";

   QSqlQuery query(db);

   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in PredefinedMessageList ", pszGetClassName()));
      return;
   }

   while (db.isOpen() && query.next())
   {

      m_oPredefinedMessageListResult.oItems.push_back(most_fi_tcl_MsgPredefinedMessageListResultItem());

      ETG_TRACE_USR2(("PredefinedMessageList handle : %u", query.value(0).toUInt()));
      ETG_TRACE_USR2(("PredefinedMessageList Msg : %s", query.value(2).toString().toLatin1().constData())); //Fix for CMG3G-5593
      m_oPredefinedMessageListResult.oItems.back().u16PredefinedMessageHandle = (tU16) query.value(0).toUInt();
      m_oPredefinedMessageListResult.oItems.back().e8PredefinedMessageType.enType
            = most_fi_tcl_e8_MsgPredefinedMessageType::tenType(query.value(1).toUInt());
      m_oPredefinedMessageListResult.oItems.back().sPredefinedMessageText.bSet(
            const_cast<char*>(query.value(2).toString().toLatin1().constData())); //Fix for CMG3G-5593
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to Generated an unique
//!    predefined message not found in the database
//!
//!  \return
//!       quint16 - unique predefinedMessage handle
//
//******************************************************************************

/******** BOSCH CORRECTION  ************/

quint16 FC_Messaging_SQLite::u16GetNewUniquePredefMsgHandle()
{
   ETG_TRACE_USR4(("u16GetNewUniquePredefMsgHandle: Entered"));
   static quint16 u16PredefinedMessageHandle = 1;
   //    quint16 u16tempPredefMsgHandle = 0;

   //    while (0 == u16PredefinedMessageHandle)
   //    {
   //        srand(time(0));
   //
   //        u16PredefinedMessageHandle = (quint16) rand();// % FC_MSG_INT16_MAX;
   //
   //        if (0 == u16PredefinedMessageHandle)
   //        {
   //            MSG_DEBUG ( "%s:- generated handle is 0. continuing.. ", pszGetClassName());
   //            continue;
   //        }

   /******** BOSCH CORRECTION  ************

    Random-functions were not generating 'random numbers', instead created numbers in a sequence every time the function was called.
    Since already-assigned-handles were generated, it executed comparison loops over a long time, after which the process was terminated.
    This is replaced by a sequentially increasing static variable, which minimized comparisons.
    When a message is deleted, its handle becomes free, and will be alloted to a newly added message.

    ******** BOSCH CORRECTION  ************/

   /* check for uniqueness of the generated handle */
   static const QString sql = "SELECT PID FROM " FC_MSG_PREDEFMSG_TABLE_NAME "";
   QSqlQuery query(GetSQLDatabase());

   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s:- ERROR - SQL query failed in select PID ", pszGetClassName()));
      return false;
   }

   //Check first time after start/restart
   if (u16PredefinedMessageHandle == 1)
   {
      while (query.next())
      {
         if (u16PredefinedMessageHandle == (quint16)(query.value(0).toUInt()))
         {
            u16PredefinedMessageHandle++;
         }
         else//Take unused/deleted message id
         {
            break;
         }
      }
   }
   else
   {
      if (query.last())
      {
         ETG_TRACE_USR2(("Last unique Predefined Message Handle :%d", query.value(0).toUInt()));
         u16PredefinedMessageHandle = (quint16) (query.value(0).toUInt() + 1);
      }
      else//This case should not come
      {
         ETG_TRACE_ERR(("%s:- ERROR - SQL query failed to get last PID ", pszGetClassName()));
         u16PredefinedMessageHandle++;
      }
   }
   ETG_TRACE_USR4(("The unique Predefined Message Handle :%d", u16PredefinedMessageHandle));

   //    }
   return u16PredefinedMessageHandle;
}

/******** BOSCH CORRECTION  ************/


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to Delete a predefined message from the database
//!
//!  \return
//!       bool - indicating success or failure of this function
//
//******************************************************************************
bool FC_Messaging_SQLite::bQueryDeleteFromPredefinedMessage(
        //! Handle of the message to be deleted
        quint16 PredefMsgHandle)
{
   ETG_TRACE_USR4(("bQueryDeleteFromPredefinedMessage : Called"));

   //Bugfix for GMMY16-5730: SHUTDOWN_AFTER_NO_STATE_REQUEST state
   //change failed of CCA_C_U16_APP_MESSAGING (OFF -> NORMAL)
   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   quint16 u16tempPredefMsgHandle = 0;

   QMutexLocker lock(&m_mutex);

   // check for uniqueness of the generated handle
   QString sql = "SELECT PID, " FC_MSG_COL_PRE_MSG_CREATE_TYPE " FROM " FC_MSG_PREDEFMSG_TABLE_NAME " WHERE PID = "
         + QString::number(PredefMsgHandle);

   QSqlQuery query(db);

   if (db.isOpen())
   {
      if (!query.exec(sql))
      {
         MSG_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in select PID ", pszGetClassName()));
         return false;
      }

      if(query.next())
      {
         u16tempPredefMsgHandle = (quint16)(query.value(0).toUInt());
         quint32 PredefMsgType = query.value(1).toUInt();
         ETG_TRACE_USR2((" predefined message type is %u", PredefMsgType));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   if (u16tempPredefMsgHandle == PredefMsgHandle)
   {
      QString delQuery = "DELETE FROM " FC_MSG_PREDEFMSG_TABLE_NAME " WHERE PID = "
            + QString::number(PredefMsgHandle);

      if (db.isOpen())
      {
         if (!query.exec(delQuery))
         {
            MSG_CHECK_QUERY_ERROR(query);
            ETG_TRACE_ERR((" could not delete "));
            return false;
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }

      ETG_TRACE_USR4((" Predefined message deleted successfully "));

      newUserPredefinedMessageAdded();
      fc_messaging_tclService_Messaging::m_poMessagingService->PredefinedListChangeStatus();

      return true;
   }
   else
   {
      ETG_TRACE_ERR(("delete unsuccessful "));
      return false;
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to check if the given PredefinedMessageHandle exists in database
//!
//!  \return
//!       bool - indicating success or failure of this function
//
//******************************************************************************
bool FC_Messaging_SQLite::bIsValidPredefMsgHandle(quint16 PredefMsgHandle)
{
   ETG_TRACE_USR4(("bIsValidPredefMsgHandle :: Handle= %d", PredefMsgHandle));

   tU8 u8CurrentLanguageSetting = FC_Messaging_PredefinedMsgList::getLanguageSetting();

   ETG_TRACE_USR2(("Current language is %d", u8CurrentLanguageSetting));

   QString str = QString::number(u8CurrentLanguageSetting);

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QString sql = "SELECT PID FROM " FC_MSG_PREDEFMSG_TABLE_NAME " WHERE PID = " + QString::number(PredefMsgHandle)
   + " AND " FC_MSG_COL_PRE_MSG_LAN_TYPE " =" + str;

   ETG_TRACE_USR4(("bIsValidPredefMsgHandle query to be executed = %s", sql.toLatin1().constData()));
   QSqlQuery query(db);

   if (db.isOpen())
   {
      if (!query.exec(sql))
      {
         MSG_CHECK_QUERY_ERROR(query);
         QByteArray ba = query.lastError().text().toLatin1();
         ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in select PID ", pszGetClassName()));
         ETG_TRACE_ERR(("Error Details - %s ", ba.constData()));
         return false;
      }

      if(query.next())
      {
         ETG_TRACE_USR4(("bIsValidPredefMsgHandle :PID = %d", query.value(0).toUInt()));

         quint16 u16tempPredefMsgHandle = (quint16)(query.value(0).toUInt());
         ETG_TRACE_USR4(("bIsValidPredefMsgHandle :u16tempPredefMsgHandle = %d", u16tempPredefMsgHandle));

         if (u16tempPredefMsgHandle == PredefMsgHandle)
         {
            return true;
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
   }

   return false;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to update the PredefinedMessageText for the given PredefindMessageHandle
//!
//!  \return
//!       bool - indicating success or failure of this function
//
//******************************************************************************
bool FC_Messaging_SQLite::bQueryUpdatePredefinedMessage
(
         quint16 PredefMsgHandle,
         const QString& strPreMessage,
         quint8 u8SubCategory
)
{
   ETG_TRACE_USR4(("bQueryUpdatePredefinedMessage : Handle= %d", PredefMsgHandle));
   (void) u8SubCategory;
   QSqlDatabase db = GetSQLDatabase();

   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QMutexLocker lock(&m_mutex);

#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
   QString sql = "UPDATE " FC_MSG_PREDEFMSG_TABLE_NAME " SET "
   FC_MSG_COL_PRE_MSG " = ?" ","
   FC_MSG_COL_PRE_MSG_SUB_CAT " = ?"
   " WHERE PID = ?";
#else
   QString sql = "UPDATE " FC_MSG_PREDEFMSG_TABLE_NAME " SET "
   FC_MSG_COL_PRE_MSG " = ?"
   " WHERE PID = ?";
#endif

   QSqlQuery query(db);
   query.prepare(sql);
   query.addBindValue(strPreMessage);
#ifdef DEFAULT_PREDEFINEDMESSAGES_PSA
   query.addBindValue(u8SubCategory);
#endif
   query.addBindValue(PredefMsgHandle);

   ETG_TRACE_USR4(("bQueryUpdatePredefinedMessage : Query to be executed: %s", sql.toLatin1().constData()));
   if (db.isOpen())
   {
      if (!query.exec())
      {
         MSG_CHECK_QUERY_ERROR(query);
         QByteArray ba = query.lastError().text().toLatin1();
         ETG_TRACE_ERR(("ERROR - %s SQL query failed in Updating PredefinedMessage Text", ba.constData()));
         return false;
      }
      ETG_TRACE_USR4((" Predefined message Edited successfully "));
      newUserPredefinedMessageAdded();
      fc_messaging_tclService_Messaging::m_poMessagingService->PredefinedListChangeStatus();
      return true;
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get total number of Predefined Messages in the table
//!
//!  \return
//!       bool - indicating success or failure of this function
//
//******************************************************************************
bool FC_Messaging_SQLite::u16GetTotalPredefMsgCount(
        //! [OUT] Total Number of Predefined Messages
        quint16& u16PredefMsgCount, quint16 u16LanguageType)
{
   ETG_TRACE_USR4(("u16GetTotalPredefMsgCount: Entered"));
   u16PredefMsgCount = 0;

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QString sql = "SELECT COUNT(PID) FROM " FC_MSG_PREDEFMSG_TABLE_NAME " WHERE "
         FC_MSG_COL_PRE_MSG_LAN_TYPE " = " + QString::number(u16LanguageType);

   ETG_TRACE_ERR(("%s",sql.toLatin1().constData())); //Fix for CMG3G-5593

   QSqlQuery query(db);

   //query.prepare(sql);
   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(( "u16GetTotalPredefMsgCount: ERROR - SQL query failed in  DisplayDB"));
      QByteArray ba = query.lastError().text().toLatin1(); //Fix for CMG3G-5593
      ETG_TRACE_ERR(( "u16GetTotalPredefMsgCount: ERROR - %s", ba.constData()));
      return false;
   }

   while (query.next())
   {
      u16PredefMsgCount = (quint16) (query.value(0).toUInt());
   }

   return true;
}

tVoid FC_Messaging_SQLite::vDeleteAllPredefinedMessageBasedOnLanguage()
{
	ETG_TRACE_USR4(("vDeleteAllPredefinedMessageBasedOnLanguage: Entered"));

	/* Bugfix for GMMY16-5730: SHUTDOWN_AFTER_NO_STATE_REQUEST state
	 * change failed of CCA_C_U16_APP_MESSAGING (OFF -> NORMAL) */
	QSqlDatabase db = GetSQLDatabase();
	if (!db.isOpen())
	{
		ETG_TRACE_ERR(("Database is not open"));
		return;
	}

    QMutexLocker lock(&m_mutex);

//    tU16 u16LanguageType
//            = FC_Messaging_PredefinedMsgList::getLanguageSetting();
//
//    QString str = QString::number(u16LanguageType);

    //Query to delete complete predefined message table
    QString sql = "DELETE FROM " FC_MSG_PREDEFMSG_TABLE_NAME;

    QSqlQuery query(db);

    if (!query.exec(sql))
    {
       MSG_CHECK_QUERY_ERROR(query);
        ETG_TRACE_ERR(("Could not delete all Predefined message"));
        return;
    }
}

qint32 FC_Messaging_SQLite::s32GetLatestMessageTime(quint16 u16LanguageType)
{
   ETG_TRACE_USR4(("s32GetLatestMessageTime() called"));

   const QString sql = "SELECT MAX(" FC_MSG_COL_PRE_MSG_MOD_TIME ") FROM "
         FC_MSG_PREDEFMSG_TABLE_NAME " WHERE " FC_MSG_COL_PRE_MSG_LAN_TYPE
         " = " + QString::number(u16LanguageType);

   ETG_TRACE_ERR(("Executing Query : %s", sql.toLatin1().constData())); //Fix for CMG3G-5593

   QSqlQuery query(GetSQLDatabase());

   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1(); //Fix for CMG3G-5593
      ETG_TRACE_ERR(("u32GetLatestMessageTime: ERROR - %s", ba.constData()));
      return -1;
   }

   qint32 l_s32LatestMessageTime = -1;
   while (query.next())
   {
      l_s32LatestMessageTime = (qint32)(query.value(0).toUInt());
   }

   return l_s32LatestMessageTime;
}

/*FUNCTION: FC_Messaging_SQLite::vCheckQueryError
*
* DESCRIPTION: In case that a query has failed, this method should be called to
*              check the returned SQLite error in more detail.
*              Based on the severity of the related error code, we will decide
*              here if the database has to be DELETED, so that it can be recreated
*              during the next start-up.
*

*******************************************************************************/
void FC_Messaging_SQLite::vCheckQueryError(QSqlQuery &query, quint32 u32LineNr, QString qsFileName, QString qsFunctionName)
{
   // Strip path name from 'qsFileName'
   int iLastIndex = qsFileName.lastIndexOf("/"); // returns -1, if "/" is not found, otherwise the position
   qsFileName = qsFileName.mid(iLastIndex + 1); // (iLastIndex +1) to skip the "/" resp. to start with position 0

   QSqlError qsqlError = query.lastError();
   // Hints regarding QSqlError:
   // QSqlError::type()    Returns the error type, or -1 if the type cannot be determined.
   // QSqlError::number()  Returns the database-specific error number, or -1 if it cannot be determined.
   // QSqlError::text()    This is a convenience function that returns databaseText() and driverText() concatenated into a single string.
   // Note: Note that the last error for this query is reset when exec() is called.

   bool bFatalTrace = false;
   bool bErrmemEntry = false;
   bool bDbRecreate = false;

   if (qsqlError.type() != QSqlError::NoError) // Note: QSqlError::NoError = 0
   { // Query error has occurred.

      // Based on available error details, the resulting measures will be taken appropriately:
      switch (qsqlError.number())
      {
         case -1:
         {
            if (qsqlError.type() == QSqlError::ConnectionError)
            {
               // QSqlError.type = 1, .number= -1, .text= 'Driver not loaded Driver not loaded' has been
               bFatalTrace = true;
            }
            else
            {
               // This error combination [(qsqlError.number() == -1) && (qsqlError.type() != QSqlError::ConnectionError)]
               // is not known so far. We will do a FATAL trace and ERRMEM entry.
               bFatalTrace = true;
               bErrmemEntry = true;
            }

            break;
         }
         case SQLITE_CORRUPT:
         {
            // This error code is related to  QSqlError.type == 2, .number == 11 (SQLITE_CORRUPT),
            // .text= 'malformed database schema (table_devices) - unrecognised token: "." Unable to execute statement'
            // A malformed database schema typically is created by:
            //  --  a memory writer (by our component or a kernel module),
            //  --  or a HW defect in the eMMC circuit
            // as the file system itself (as used by SQlite) "can be assumed" to be transaction safe.
            //
            // Anyhow, this error case is a most severe error, indicating that the database was corrupted.
            // Therefore the database shall be recreated with next transition to NORMAL.
            bFatalTrace = true;
            bErrmemEntry = true;
            bDbRecreate = true;

            break;
         }
         default:
         {
            // Any error in this section is not known so far. So if such errors occur,
            // We will do an FATAL trace and an ERMMEM entry (but do not trigger the database to be recreated).
            bFatalTrace = true;
            bErrmemEntry = true;

            break;
         }
      }
   }

#ifdef MSG_CHECK_QUERY_EXEC__TESTMODE
   // - If 'MSG_CHECK_QUERY_EXEC__TESTMODE is defined' then a special Testmode is active. This shall be
   //   done only for developer builds for dedicated testing.
   // - If this dedicated switch is not defined: Normal operation mode (product code delivery)
   //
   // In "Testmode active" we simulate a severe error (e.g. corrupt database). The database recreation
   // trigger will be set (with each call of this method [vCheckQueryError(..)]), independent if a severe
   // error has occurred in a query or not.


   // Here we simulate an fatal database error
   ETG_TRACE_FATAL(( " -MSG-QUERY-ERR-: TESTMODE is active. Now we simulate a fatal DB error. " ));
   ETG_TRACE_ERRMEM(( " -MSG-QUERY-ERR-: TESTMODE is active. Now we simulate a fatal DB error. " ));
   bFatalTrace = true;
   bErrmemEntry = true;
   bDbRecreate = true;
#endif

   // Get time via 'clock_gettime(CLOCK_MONOTONIC,.)'
   struct timespec ts; // for clock_gettime(CLOCK_MONOTONIC,.)
   char sz_index_ts[MSG_CHECK_QUERY_TIMESTAMP_BYTE_LEN + 1] =
   { 0 };

   clock_gettime(CLOCK_MONOTONIC, &ts);

   snprintf(sz_index_ts, MIN(sizeof(sz_index_ts), MSG_CHECK_QUERY_TIMESTAMP_BYTE_LEN), "up-time= [%7lu.%03lu] sec, ", ts.tv_sec, ts.tv_nsec
         / 1000000);
   sz_index_ts[MSG_CHECK_QUERY_TIMESTAMP_BYTE_LEN] = 0; // The string should be terminated with \0

   // Generate "Main Trace line": ['up-time'] + '<::function_name> ' + '[Filename.cpp (LineNr)] - '
   QString qsMainTraceLine = "-MSG-QUERY-ERR-MAIN-: " + QString(sz_index_ts)
         + "<::" + qsFunctionName + "> " + "[" + qsFileName + " ("
         + QString::number(u32LineNr) + ")] -";

   ETG_TRACE_FATAL((" %s ", qsMainTraceLine.toLatin1().constData()));
   if (bErrmemEntry == true)
   {
      ETG_TRACE_ERRMEM((" %s ", qsMainTraceLine.toLatin1().constData()));
   }

   if (bFatalTrace == true)
   {
      ETG_TRACE_FATAL((" -MSG-QUERY-ERR-: lastError: QSqlError.type = %d, .number= %d, .text= '%s' ", qsqlError.type(), qsqlError.number(), qsqlError.text().toLatin1().constData()));
   }
   if (bErrmemEntry == true)
   {
      ETG_TRACE_ERRMEM((" -MSG-QUERY-ERR-: lastError: QSqlError.type = %d, .number= %d, .text= '%s' ", qsqlError.type(), qsqlError.number(), qsqlError.text().toLatin1().constData()));
   }

   if (bDbRecreate)
   {
      // Set the database "recreation trigger file" to the FFS
      vSetDbRecreationTrigger();

      NORMAL_M_ASSERT_ALWAYS(); // With the normal assert we can see from the callstack , from which method we have been called
      // FATAL_M_ASSERT_ALWAYS();  // This option with the fatal assert would issue a direct reset and could be used to avoid
      // bad user experience with a corrupted database.
   }
}
/*******************************************************************************
*
* FUNCTION: FC_Messaging_SQLite::vSetDbRecreationTrigger
*
* DESCRIPTION: Sets DB recreation trigger in FFS.
*              The used sequence of operations guarantees that the recreation
*              trigger is written into the FFS and the file system is synced
*              after returning from the method.
*              This allows to issue directly afterwards ,a FATAL_M_ASSERT_ALWAYS()
*              if required (e.g. to avoid bad user experience with a corrupted
*              database in the currently active power cycle).
*
*
*******************************************************************************/
void FC_Messaging_SQLite::vSetDbRecreationTrigger()
{
   bool bRes = false;

   FILE *fp_DbRecreateTriggerFile = NULLPTR;
   fp_DbRecreateTriggerFile = fopen(FC_MSG_DB_DIR_PATH "/" FC_MSG_DB_RECREATE_TRIGGER_FILE_NAME, "w");

   if (fp_DbRecreateTriggerFile)
   {
      // The following fflush(.) is not necessary here, as we have not added any content to the file, but we do it for generality.
      if (0 == fflush(fp_DbRecreateTriggerFile))
      { // fflush SUCCESS

         // Get file descriptor 'fd' from an open FILE* object 'fp'
         int fd = fileno(fp_DbRecreateTriggerFile);
         if (-1 != fd)
         { // fileno(.) SUCCESS

            // Sync the file system (partition), in which the file descriptor resides, to the FFS (eMMC)
            if (0 == syncfs(fd))
            { // syncfs(.) SUCCESS
               bRes = true;

               // The database recreation trigger file has been written successfully to the FFS.
               ETG_TRACE_FATAL((" -MSG-QUERY-ERR-: Severe database error detected. Database RE-CREATION trigger has been set!! "));
               ETG_TRACE_ERRMEM((" -MSG-QUERY-ERR-: Severe database error detected. Database RE-CREATION trigger has been set!! "));
            }
         }
      }

      if (false == bRes)
      {
         // We wanted to set the database recreation trigger file, but have failed on this. So we will do (as last resort) a related EM-Trace entry.
         ETG_TRACE_FATAL((" -MSG-QUERY-ERR-: Database RE-CREATION trigger 'msg.db.recreate.trigger' cannot be created. "));
         ETG_TRACE_ERRMEM((" -MSG-QUERY-ERR-: Database RE-CREATION trigger 'msg.db.recreate.trigger' cannot be created. "));
      }

      // Close FILE*
      if (0 == fclose(fp_DbRecreateTriggerFile))
      { // fclose(.) SUCCESS

         ETG_TRACE_USR4((" -MSG-QUERY-ERR-: Database RE-CREATE trigger file closed successfully. "));
         // Note: If we would have added any content to the "trigger file", we would have the need to do a 2nd 'syncfs' here.
         //       For this purpose we would first have to get a fresh valid file descriptor, e.g. from the file or its directory.
         //       As we have not added data to the file we skip these steps.
      }
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    QtSql specific function to create a Device settings database
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//******************************************************************************
bool FC_Messaging_SQLite::bCreateDeviceSettingsDatabase()
{
   return bCheckDeviceSettingsTableSchema();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Validate the sql Device Settings Table Schema
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//******************************************************************************
bool FC_Messaging_SQLite::bCheckDeviceSettingsTableSchema()
{
   ETG_TRACE_USR4(("FC_Messaging_SQLite::bCheckDeviceSettingsTableSchema entered"));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   QStringList list = db.tables();

   if (list.contains(FC_MSG_DEVICESETTINGS_TABLE_NAME))
   {
      if (!query.exec("PRAGMA table_info(" FC_MSG_DEVICESETTINGS_TABLE_NAME ")"))
      {
         MSG_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR(("ERROR - SQL query failed in bCheckDeviceSettingsTableSchema - 1"));
         return false;
      }

      list.clear();

      while (query.next())
         list << query.value(1).toString();

      //database consistency check for DeviceSettings table
      if ((list.contains(FC_MSG_COL_DEVICE_HANDLE)) && (list.contains(FC_MSG_COL_DEVICE_ADDRESS)) &&
            (list.contains(FC_MSG_COL_MAP_ON_OFF)) && (list.contains(FC_MSG_COL_TEXT_RINGTONE)) &&
            (list.contains(FC_MSG_COL_SIGN_ON_OFF)) && (list.contains(FC_MSG_COL_AUTOREPLY_ON_OFF)) &&
            (list.contains(FC_MSG_COL_AUTOREPLY_TEXT)))
      {
         ETG_TRACE_USR4(("bCheckDeviceSettingsTableSchema - DeviceSettings table exists"));
         return true;
      }
      else
      {
         if (!query.exec("DROP TABLE " FC_MSG_DEVICESETTINGS_TABLE_NAME))
         {
            MSG_CHECK_QUERY_ERROR(query);
            ETG_TRACE_ERR(("ERROR - SQL query failed in bCheckDeviceSettingsTableSchema - 2"));
            return false;
         }
      }
   }

   return bCreateDeviceSettingsTable();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    QtSql specific function to Device Settings table
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//******************************************************************************
bool FC_Messaging_SQLite::bCreateDeviceSettingsTable()
{
   ETG_TRACE_USR4(("bCreateDeviceSettingsTable() called"));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   static const QString sql = "CREATE TABLE IF NOT EXISTS " FC_MSG_DEVICESETTINGS_TABLE_NAME " "
   "(" FC_MSG_COL_DEVICE_HANDLE " INTEGER,"
   " " FC_MSG_COL_DEVICE_ADDRESS " TEXT,"
   " " FC_MSG_COL_MAP_ON_OFF " BOOLEAN,"
   " " FC_MSG_COL_TEXT_RINGTONE " BOOLEAN,"
   " " FC_MSG_COL_SIGN_ON_OFF " BOOLEAN,"
   " " FC_MSG_COL_AUTOREPLY_ON_OFF " BOOLEAN,"
   " " FC_MSG_COL_AUTOREPLY_TEXT " INTEGER);";

   ETG_TRACE_USR4(("Executing Query : %s", sql.toLatin1().constData()));

   QSqlQuery query(db);

   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in deviceSettings Create Table ", pszGetClassName()));
      return false;
   }

   return true;
}

// ***************** F U N C T I O N  H E A D E R *********************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt Specific function to Add device in DeviceSettings table
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//*********************************************************************************
tBool FC_Messaging_SQLite::bQueryAddDevicetoDeviceSettingsTable(tU8 u8DeviceHandle, std::string sDeviceAddress)
{
   ETG_TRACE_USR4(("bQueryAddDevicetoDeviceSettingsTable() called with u8DeviceHandle: %d and sDeviceAddress: %s", u8DeviceHandle, sDeviceAddress));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   bool enableSetting = TRUE;
   bool bAutoreplyOnOff = FALSE;
   tU16 u16AutoReplyText = FC_MSG_DEFAULT_AUTO_REPLY_TEXT;

   static const QString sql = "INSERT INTO " FC_MSG_DEVICESETTINGS_TABLE_NAME " "
   "( " FC_MSG_COL_DEVICE_HANDLE ", "
   " " FC_MSG_COL_DEVICE_ADDRESS ", "
   " " FC_MSG_COL_MAP_ON_OFF ", "
   " " FC_MSG_COL_TEXT_RINGTONE ", "
   " " FC_MSG_COL_SIGN_ON_OFF ", "
   " " FC_MSG_COL_AUTOREPLY_ON_OFF ", "
   " " FC_MSG_COL_AUTOREPLY_TEXT ") "
   "VALUES (?, ?, ?, ?, ?, ?, ?);";

   ETG_TRACE_USR4(("Executing Query : %s", sql.toLatin1().constData()));

   QSqlQuery query(db);
   query.prepare(sql);
   query.addBindValue(u8DeviceHandle);
   query.addBindValue(sDeviceAddress.c_str());
   query.addBindValue(enableSetting);     //MapOnOff
   query.addBindValue(enableSetting);     //TextRingtoneOnOff
   query.addBindValue(enableSetting);     //SignatureOnOff
   query.addBindValue(bAutoreplyOnOff);   //AutoReplyOnOff
   query.addBindValue(u16AutoReplyText);  //AutoReplyText

   if (!query.exec())
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1();
      ETG_TRACE_ERR(("ERROR - %s SQL query failed in inserting SetDefaultDeviceSettings", ba.constData()));
      return false;
   }

   return true;
}

// ***************** F U N C T I O N  H E A D E R *********************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt Specific function to Reset device settings to default in DeviceSettings table
//!
//!  \return
//!        bool  - indicates success or failure of query
//
//*********************************************************************************
tBool FC_Messaging_SQLite::bQueryResetDeviceSettingsInDeviceSettingsTable(tU8 u8DeviceHandle, std::string sDeviceAddress)
{
   ETG_TRACE_USR4(("bQueryResetDeviceSettingsInDeviceSettingsTable() called with u8DeviceHandle: %d sDeviceAddress: %s", u8DeviceHandle, sDeviceAddress));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   bool enableSetting = TRUE;
   bool bAutoreplyOnOff = FALSE;
   tU16 u16AutoReplyText = FC_MSG_DEFAULT_AUTO_REPLY_TEXT;

   static const QString sql = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME " SET "
         FC_MSG_COL_DEVICE_ADDRESS " = ? " ","
         FC_MSG_COL_MAP_ON_OFF " = ? " ","
         FC_MSG_COL_TEXT_RINGTONE " = ? " ","
         FC_MSG_COL_SIGN_ON_OFF " = ? " ","
         FC_MSG_COL_AUTOREPLY_ON_OFF " = ? " ","
         FC_MSG_COL_AUTOREPLY_TEXT " = ? "
         " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);

   ETG_TRACE_USR4(("Executing Query : %s", sql.toLatin1().constData()));

   QSqlQuery query(db);
   query.prepare(sql);
   query.addBindValue(sDeviceAddress.c_str());
   query.addBindValue(enableSetting);     //MapOnOff
   query.addBindValue(enableSetting);     //TextRingtoneOnOff
   query.addBindValue(enableSetting);     //SignatureOnOff
   query.addBindValue(bAutoreplyOnOff);   //AutoReplyOnOff
   query.addBindValue(u16AutoReplyText);  //AutoReplyText

   if (!query.exec())
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1();
      ETG_TRACE_ERR(("ERROR - %s SQL query failed in ResetDeviceSettings", ba.constData()));
      return false;
   }

   return true;
}

// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get the Device Specific Setting from the database
//!
//!  \return
//
//*******************************************************************************
tBool FC_Messaging_SQLite::bQueryRemoveDeviceFromDeviceSettingsTable(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("bQueryRemoveDeviceFromDeviceSettingsTable() called with u8DeviceHandle: %d", u8DeviceHandle));

   QSqlDatabase db = GetSQLDatabase();

   QString delQuery;
   if (0 == u8DeviceHandle)
   {
      delQuery = "DELETE FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME;
   }
   else
   {
      delQuery = "DELETE FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
   }

   ETG_TRACE_USR4(("Executing Query : %s", delQuery.toLatin1().constData()));

   QSqlQuery query(db);
   if (db.isOpen())
   {
      if (!query.exec(delQuery))
      {
         MSG_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR((" could not delete device from DeviceSettings Table "));
         return false;
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   ETG_TRACE_USR4((" Device deleted successfully from DeviceSettings table "));
   return true;
}
// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to remove user defined messages based on DeviceHandle
//!
//!  \return
//
//*******************************************************************************
tBool FC_Messaging_SQLite::bQueryRemoveCustomMessagesFromPreDefMsgTable(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("bQueryRemoveCustomMessagesFromPreDefMsgTable() called with u8DeviceHandle: %d", u8DeviceHandle));

   QSqlDatabase db = GetSQLDatabase();

   QString delQuery;
   if (0 == u8DeviceHandle)
   {
      delQuery = "DELETE FROM " FC_MSG_PREDEFMSG_TABLE_NAME " WHERE " FC_MSG_COL_PRE_MSG_CREATE_TYPE "=" + QString::number(FC_MSG_USER_PRE_DEF_MSG);
   }
   else
   {
      delQuery = "DELETE FROM " FC_MSG_PREDEFMSG_TABLE_NAME " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
   }

   ETG_TRACE_USR4(("Executing Query : %s", delQuery.toLatin1().constData()));

   QSqlQuery query(db);
   if (db.isOpen())
   {
      if (!query.exec(delQuery))
      {
         MSG_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR((" could not delete Custom messages from PredefMsg Table "));
         return false;
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   ETG_TRACE_USR4((" Custom message deleted successfully from PredefMsg table "));

   //Update the status to clients
   newUserPredefinedMessageAdded();
   fc_messaging_tclService_Messaging::m_poMessagingService->PredefinedListChangeStatus();

   return true;
}
// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get the Device Specific Setting from the database
//!
//!  \return
//
//*******************************************************************************
tVoid FC_Messaging_SQLite::szQueryGetDeviceSpecificSettingList(DeviceSettings enDeviceSettings, tMap_DevHandle_SettingState &mapPairedDevicesSettingList)
{
   ETG_TRACE_USR4(("szQueryGetDeviceSpecificSettingList() called with enDeviceSettings: %d", enDeviceSettings));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return;
   }

   QString sqlGetString;

   switch (enDeviceSettings)
   {
      case MAP_ON_OFF_SETTING:
         sqlGetString = " SELECT " FC_MSG_COL_DEVICE_HANDLE " , " FC_MSG_COL_MAP_ON_OFF " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME;
         break;

      case TEXT_RINGTONE_ON_OFF_SETTING:
         sqlGetString = " SELECT " FC_MSG_COL_DEVICE_HANDLE " , " FC_MSG_COL_TEXT_RINGTONE " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME;
         break;

      case SIGNATURE_ON_OFF_SETTING:
         sqlGetString = " SELECT " FC_MSG_COL_DEVICE_HANDLE " , " FC_MSG_COL_SIGN_ON_OFF " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME;
         break;

      case AUTO_REPLY_ON_OFF_SETTING:
         sqlGetString = " SELECT " FC_MSG_COL_DEVICE_HANDLE " , " FC_MSG_COL_AUTOREPLY_ON_OFF " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME;
         break;

      default:
         ETG_TRACE_USR4(("enDeviceSettings- %d is invalid", enDeviceSettings));
  }

   if (sqlGetString.isEmpty())
   {
      ETG_TRACE_ERR(("ERROR - Trying to Read Incorrect DeviceSettings"));
      return;
   }

   ETG_TRACE_USR4(("Executing Query : %s", sqlGetString.toLatin1().constData()));

   QSqlQuery query(db);

   if (!query.exec(sqlGetString))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in Getting DeviceSettings", pszGetClassName()));
      return;
   }

   while (query.next())
   {
      ETG_TRACE_USR4(("DeviceHandle: %d, SettingValue : %d", query.value(0).toUInt(), query.value(1).toBool()));
      mapPairedDevicesSettingList.insert(std::pair<tU8, tBool>((query.value(0)).value<tU8>(), query.value(1).toBool()));
   }

}

// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get the Device Specific Setting from the database
//!
//!  \return
//
//*******************************************************************************
tVoid FC_Messaging_SQLite::szQueryGetDeviceSpecificAutoReplyMsgHandleList(tMap_DevHandle_MsgHandle &mapPairedDevicesMsgHandleList)
{
   ETG_TRACE_USR4(("szQueryGetDeviceSpecificAutoReplyMsgHandleList() called"));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return;
   }

   QString sqlGetString = " SELECT " FC_MSG_COL_DEVICE_HANDLE " , " FC_MSG_COL_AUTOREPLY_TEXT " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME;

   ETG_TRACE_USR4(("Executing Query : %s", sqlGetString.toLatin1().constData()));

   QSqlQuery query(db);

   if (!query.exec(sqlGetString))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in Getting DeviceSettings", pszGetClassName()));
      return;
   }

   while (query.next())
   {
      ETG_TRACE_USR4(("DeviceHandle: %d, AutoReply-MsgHandle : %d", query.value(0).toUInt(), query.value(1).toUInt()));
      mapPairedDevicesMsgHandleList.insert(std::pair<tU8, tU16>((query.value(0)).value<tU8>(), (query.value(1)).value<tU16>()));
   }
}

// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get the Device Specific Setting from the database
//!
//!  \return
//
//*******************************************************************************
tVoid FC_Messaging_SQLite::szQueryGetDevicesList(tMap_DevHandle_DevAddress &mapDevicesList)
{
   ETG_TRACE_USR4(("szQueryGetDevicesList() called"));

   //Clear the list and update the devices.
   mapDevicesList.clear();

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return;
   }

   QString sqlGetString;

   sqlGetString = " SELECT " FC_MSG_COL_DEVICE_HANDLE " , " FC_MSG_COL_DEVICE_ADDRESS " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME;

   ETG_TRACE_USR4(("Executing Query : %s", sqlGetString.toLatin1().constData()));

   QSqlQuery query(db);

   if (!query.exec(sqlGetString))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in Getting DeviceSettings", pszGetClassName()));
      return;
   }

   while (query.next())
   {
      ETG_TRACE_USR4(("DeviceHandle: %d, DevAddress : %s", query.value(0).toUInt(), query.value(1).toString().toStdString().c_str()));
      mapDevicesList.insert(std::pair<tU8, std::string>((query.value(0)).value<tU8>(), query.value(1).toString().toStdString().c_str()));
   }
}

// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get the Device Specific Setting from the database
//!
//!  \return
//
//*******************************************************************************
tBool FC_Messaging_SQLite::bIsDeviceAvailableInDB(tU8 u8DeviceHandle, QString strTableName)
{
   ETG_TRACE_USR4(("bIsDeviceAvailableInDB() called"));

   tBool bRetValue = FALSE;

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else if(0 == u8DeviceHandle)
   {
      bRetValue = TRUE;
   }
   else
   {
      QString sqlGetString = " SELECT " FC_MSG_COL_DEVICE_HANDLE " FROM " + strTableName + " WHERE " FC_MSG_COL_DEVICE_HANDLE "="
            + QString::number(u8DeviceHandle) + " LIMIT 1";

      ETG_TRACE_USR4(("Executing Query : %s", sqlGetString.toLatin1().constData()));

      QSqlQuery query(db);

      if (!query.exec(sqlGetString))
      {
         MSG_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in Getting DeviceSettings", pszGetClassName()));
      }
      else
      {
         if(query.next())
         {
            if(query.value(0).toUInt() == u8DeviceHandle)
            {
               bRetValue = TRUE;
            }
         }
      }
   }

   return bRetValue;
}

// ******************* F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!    Qt specific function to get the DeviceSettings from the database
//!
//!  \return
//!        bool  - Setting
//
//*******************************************************************************
bool FC_Messaging_SQLite::szQueryGetDeviceSettings(tU8 u8DeviceHandle, DeviceSettings enDeviceSettings)
{
   ETG_TRACE_USR4(("szQueryGetDeviceSettings() called"));
   bool isSettingEnable = TRUE;

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return isSettingEnable;
   }

   QString sqlGetValue;

   switch (enDeviceSettings)
   {
      case MAP_ON_OFF_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_MAP_ON_OFF " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
         break;

      case TEXT_RINGTONE_ON_OFF_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_TEXT_RINGTONE " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
         break;

      case SIGNATURE_ON_OFF_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_SIGN_ON_OFF " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
         break;

      case AUTO_REPLY_ON_OFF_SETTING:
         sqlGetValue = "SELECT " FC_MSG_COL_AUTOREPLY_ON_OFF " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
         break;

      default:
         sqlGetValue = "";
         break;
   }

   if (sqlGetValue.isEmpty())
   {
      ETG_TRACE_ERR(("ERROR - Trying to Read Incorrect DeviceSettings"));
      return false;
   }

   ETG_TRACE_USR4(("Executing Query : %s", sqlGetValue.toLatin1().constData()));

   QSqlQuery query(db);

   if (!query.exec(sqlGetValue))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR(("%s :- ERROR - SQL query failed in Getting DeviceSettings", pszGetClassName()));
      return isSettingEnable;
   }

   if (query.next())
   {
      isSettingEnable = query.value(0).toBool();
   }
   else
   {
      ETG_TRACE_USR4(("%s :- WARNING - Empty list of DeviceSettings", pszGetClassName()));
   }

   return isSettingEnable;
}

bool FC_Messaging_SQLite::bQueryUpdateDeviceSettings(tU8 u8DeviceHandle, DeviceSettings enDeviceSettings, tBool bOnOffState)
{
   ETG_TRACE_USR4(("bQueryUpdateDeviceSettings() called"));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   static QString sqlUpdate;

   switch (enDeviceSettings)
   {
      case MAP_ON_OFF_SETTING:
      {
         if (0 == u8DeviceHandle)
         {
            sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
                  " SET " FC_MSG_COL_MAP_ON_OFF " = ? ";
         }
         else
         {
            sqlUpdate = " UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
                  " SET " FC_MSG_COL_MAP_ON_OFF " = ? "
                  " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
         }
      }
      break;

      case TEXT_RINGTONE_ON_OFF_SETTING:
      {
         if (0 == u8DeviceHandle)
         {
            sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
                  " SET " FC_MSG_COL_TEXT_RINGTONE " = ? ";
         }
         else
         {
            sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
                  " SET " FC_MSG_COL_TEXT_RINGTONE " = ? "
                  " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
         }
      }
      break;

      case SIGNATURE_ON_OFF_SETTING:
      {
         if (0 == u8DeviceHandle)
         {
            sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
                  " SET " FC_MSG_COL_SIGN_ON_OFF " = ? ";
         }
         else
         {
            sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
                  " SET " FC_MSG_COL_SIGN_ON_OFF " = ? "
                  " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
         }
      }
      break;

      case AUTO_REPLY_ON_OFF_SETTING:
      {
         if (0 == u8DeviceHandle)
         {
            sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
                  " SET " FC_MSG_COL_AUTOREPLY_ON_OFF " = ? ";
         }
         else
         {
            sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
                  " SET " FC_MSG_COL_AUTOREPLY_ON_OFF " = ? "
                  " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
         }
      }
      break;

      default:
         sqlUpdate = "";
         break;
   }

   if (sqlUpdate.isEmpty())
   {
      ETG_TRACE_ERR(("ERROR - Trying to Update Incorrect Setting"));
      return false;
   }

   ETG_TRACE_USR4(("Executing Query : %s", sqlUpdate.toLatin1().constData()));

   QSqlQuery query(db);
   query.prepare(sqlUpdate);
   query.addBindValue(bOnOffState);

   if (!query.exec())
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1();
      ETG_TRACE_ERR(("ERROR - %s  SQL query failed in Updating DeviceSettings", ba.constData()));
      return false;
   }

#ifdef MSG_CHECK_QUERY_EXEC__TESTMODE
   MSG_CHECK_QUERY_ERROR(query); // This is only for TEST (Error Injection) of: -MSG-QUERY-ERR-
#endif
   return true;
}

unsigned int FC_Messaging_SQLite::szQueryGetAutoReplyText(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("szQueryGetAutoReplyText() called"));

   unsigned int u16AutoReplyText = FC_MSG_DEFAULT_AUTO_REPLY_TEXT;

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return u16AutoReplyText;
   }

   QString sql = "SELECT " FC_MSG_COL_AUTOREPLY_TEXT " FROM " FC_MSG_DEVICESETTINGS_TABLE_NAME
         " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);

   ETG_TRACE_USR4(("Executing Query : %s", sql.toLatin1().constData()));

   QSqlQuery query(db);

   if (!query.exec(sql))
   {
      MSG_CHECK_QUERY_ERROR(query);
      ETG_TRACE_ERR (( "%s :- ERROR - SQL query failed in GetAutoReplyText", pszGetClassName() ));
      return u16AutoReplyText;
   }

   if (query.next())
   {
      u16AutoReplyText = query.value(0).toUInt();
   }
   else
   {
      ETG_TRACE_USR4 (( "%s :- WARNING - Empty list of GetAutoReplyText", pszGetClassName()));
   }

   return u16AutoReplyText;
}

bool FC_Messaging_SQLite::bQueryUpdateAutoReplyText(tU8 u8DeviceHandle, tU16 u16AutoReplyText)
{
   ETG_TRACE_USR4(("bQueryUpdateAutoReplyText() called"));

   QSqlDatabase db = GetSQLDatabase();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QString sqlUpdate;

   if (0 == u8DeviceHandle)
   {
      sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
            " SET " FC_MSG_COL_AUTOREPLY_TEXT " = ? ";
   }
   else
   {
      sqlUpdate = "UPDATE " FC_MSG_DEVICESETTINGS_TABLE_NAME
            " SET " FC_MSG_COL_AUTOREPLY_TEXT " = ?  "
            " WHERE " FC_MSG_COL_DEVICE_HANDLE "=" + QString::number(u8DeviceHandle);
   }

   ETG_TRACE_USR4(("Executing Query : %s", sqlUpdate.toLatin1().constData()));

   QSqlQuery query(db);
   query.prepare(sqlUpdate);
   query.addBindValue(u16AutoReplyText);

   if (!query.exec())
   {
      MSG_CHECK_QUERY_ERROR(query);
      QByteArray ba = query.lastError().text().toLatin1();
      ETG_TRACE_ERR(( "ERROR - %s SQL query failed in Updating AutoReplyText Setting", ba.constData()));
      return false;
   }

   return true;
}



