/**
 * @file LcmRequestIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the LcmRequestIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file holds the class definition for LcmRequestIf class
 *
 * @ingroup IpcWrapper
 */

#include "LcmRequestIf.h"

namespace com
{
namespace bosch
{
namespace pmapp
{

DEFINE_CLASS_LOGGER_AND_LEVEL("phone_call_mgr/clienthandlers", LcmRequestIf, Info);

LcmRequestIf::LcmRequestIf()
:ILcmRequestIf(false, false), _lcmCallBackIf(nullptr), _bdAddressListOfPhoneSubState()
{
   LOG_INFO("LcmRequestIf::LcmRequestIf() entered");

   createProxy("lcmPort");
}

LcmRequestIf::~LcmRequestIf()
{
   destroyProxy();
}

void LcmRequestIf::createProxy(const std::string portName)
{
   LOG_INFO("LcmRequestIf::createProxy entered");

   if (!(portName.empty()))
   {
      _lcmFiProxy = Lcm_dbus_fi_serviceProxy::createProxy(portName, *this);
   }
}

void LcmRequestIf::destroyProxy()
{
   if (_lcmFiProxy.get())
   {
      _lcmFiProxy.reset();
   }
}

void LcmRequestIf::setLcmCallBackIf(ILcmCallBackIf* lcmCallBackIf)
{
   LOG_INFO("LcmRequestIf::setILcmCallBackIf entered");
   if (lcmCallBackIf)
   {
      _lcmCallBackIf = lcmCallBackIf;
      _lcmCallBackIf->setLcmFiProxy(_lcmFiProxy);
   }
}

void LcmRequestIf::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   LOG_INFO("LcmRequestIf::onAvailable() entered");

   if ((_lcmFiProxy.get()) && (proxy == _lcmFiProxy))
   {
      registerProperties();
      _serviceAvailability = true;
   }
   else
   {
      LOG_ERROR("Undesirable proxy");
   }
}

void LcmRequestIf::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   LOG_INFO("LcmRequestIf::onUnavailable() entered");

   if ((_lcmFiProxy.get()) && (proxy == _lcmFiProxy))
   {
      deregisterProperties();
      _serviceAvailability = false;
      _bdAddressListOfPhoneSubState.clear();
   }
   else
   {
      LOG_ERROR("Undesirable proxy");
   }
}

void LcmRequestIf::registerProperties()
{
   LOG_INFO("LcmRequestIf::registerProperties() entered");

   if (_lcmCallBackIf)
   {
      _lcmFiProxy->sendSystemStateRegister(*_lcmCallBackIf);
      _lcmFiProxy->sendSubStatesRegister(*_lcmCallBackIf);
      _lcmFiProxy->sendBluetoothStateRegister(*_lcmCallBackIf);

      _lcmFiProxy->sendSystemStateGet(*_lcmCallBackIf);
      _lcmFiProxy->sendSubStatesGet(*_lcmCallBackIf);
      _lcmFiProxy->sendBluetoothStateGet(*_lcmCallBackIf);
   }
}

void LcmRequestIf::deregisterProperties()
{
   LOG_INFO("LcmRequestIf::deregisterProperties() entered");

   _lcmFiProxy->sendSystemStateDeregisterAll();
   _lcmFiProxy->sendSubStatesDeregisterAll();
}

void LcmRequestIf::setSpmPhoneSubState(const pmcore::BdAddress deviceAddress, const bool phoneSubState)
{
   LOG_INFO("setSpmPhoneSubState() called with phoneSubState %u and deviceAddress: %s", phoneSubState, deviceAddress.c_str());

   bool updatePropertyChange = false;
   auto iter = std::find_if(_bdAddressListOfPhoneSubState.begin(), _bdAddressListOfPhoneSubState.end(),
         [&deviceAddress](pmcore::BdAddress const& obj){return obj == deviceAddress;});

   if (phoneSubState)
   {
      if (_bdAddressListOfPhoneSubState.end() == iter)
      {
         // Device not present. Add to the list.
         _bdAddressListOfPhoneSubState.emplace_back(deviceAddress);
         if (!_phoneSubState)
         {
            updatePropertyChange = true;
         }
      }
   }
   else
   {
      if (_bdAddressListOfPhoneSubState.end() != iter)
      {
         _bdAddressListOfPhoneSubState.erase(iter);

         if (_bdAddressListOfPhoneSubState.empty() && _phoneSubState)
         {
            updatePropertyChange = true;
         }
      }
   }

   if (_serviceAvailability && (_lcmFiProxy.get()) && (_lcmCallBackIf) && (updatePropertyChange))
   {
      _phoneSubState = phoneSubState;

      LOG_INFO("setSpmPhoneSubState() phoneSubState change updated");

      act_t act = _lcmFiProxy->sendOnOffTriggerRequest(*_lcmCallBackIf,
            T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_PHONE, _phoneSubState);

      if (DEFAULT_ACT == act)
      {
         LOG_INFO("sendSetPrimaryStart posting FAILED");
      }
   }
}

} // namespace pmapp
} // namespace bosch
} // namespace com
