/**
 * @file DiagSystemSettings.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the DiagSystemSettings class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "DiagSystemSettings.h"
#include "IpcWrapper.h"

namespace com
{
namespace bosch
{
namespace pmapp
{

DEFINE_CLASS_LOGGER_AND_LEVEL("phone_call_mgr/clienthandlers", DiagSystemSettings, Info);

ExecuteSettingsInfo::ExecuteSettingsInfo() : _sysSetID(""), _sysSetType(""), _cookie(0)
{
}

ExecuteSettingsInfo::ExecuteSettingsInfo(const SystemSetID sysSetID, const SystemSetType sysSetType,
      const uint32 cookie) : _sysSetID(sysSetID), _sysSetType(sysSetType), _cookie(cookie)
{
}

ExecuteSettingsInfo::ExecuteSettingsInfo(const ExecuteSettingsInfo& other) :
               _sysSetID(other._sysSetID), _sysSetType(other._sysSetType), _cookie(other._cookie)
{
}

ExecuteSettingsInfo::~ExecuteSettingsInfo()
{
}

void swap(ExecuteSettingsInfo& first, ExecuteSettingsInfo& second)
{
   using std::swap;

   swap(first._sysSetID, second._sysSetID);
   swap(first._sysSetType, second._sysSetType);
   swap(first._cookie, second._cookie);
}

//lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
ExecuteSettingsInfo& ExecuteSettingsInfo::operator=(ExecuteSettingsInfo other)
{
   swap(*this, other);

   return *this;
}

bool ExecuteSettingsInfo::operator==(const ExecuteSettingsInfo& rhs) const
{
   return ((this->_sysSetID == rhs._sysSetID)
         && (this->_sysSetType == rhs._sysSetType) && (this->_cookie == rhs._cookie));
}

bool ExecuteSettingsInfo::operator!=(const ExecuteSettingsInfo& rhs) const
{
   return (!((this->_sysSetID == rhs._sysSetID)
         && (this->_sysSetType == rhs._sysSetType)
         && (this->_cookie == rhs._cookie)));
}

DiagSystemSettings::DiagSystemSettings() :
      IDiagSystemSettings("pmSystemSettingsPort") /* Port name specified in the component manifest */
{
   LOG_INFO("DiagSystemSettings::DiagSystemSettings() entered");
}

DiagSystemSettings::~DiagSystemSettings()
{
   _actToExecuteSettingsInfoMap.clear();
}

void DiagSystemSettings::onPrepareSystemSettingRequest(
      const ::boost::shared_ptr< PrepareSystemSettingRequest >& request)
{
   LOG_INFO("DiagSystemSettings::onPrepareSystemSettingRequest() entered");

   SystemSetID sysSetID = request->getSysSetID();
   SystemSetType sysSetType = request->getSysSetType();
   uint32 cookie = request->getCookie();

   ::std::vector< PrepareSystemSettingResponseExtendedDataStruct > responseData;

   sendPrepareSystemSettingResponse(sysSetID, sysSetType, responseData, 0, cookie, request->getAct());
}

void DiagSystemSettings::onExecuteSystemSettingRequest(
      const ::boost::shared_ptr< ExecuteSystemSettingRequest >& request)
{
   LOG_INFO("DiagSystemSettings::onExecuteSystemSettingRequest() entered");

   // Note: Two actions to be performed in Defset handling, based on the request received from Diagnosis component.
   //       a. Resetting the default setting for the user data. E.X. Reset the Ringtone-ID for all devices in PM.db,
   //          Reset the AutoWaitingModeSetting, SuppressRingtoneSetting etc. in PM.db
   //       This will be achieved by invoking the PM-Core API - restoreDefaultSetting with Device Address is "ALL"
   //       b. Clear the complete user / private data. E.X. delete all the devices and their info from PM.db
   //       This will be achieved by invoking the PM-Core API - restoreDefaultSetting with Device Address is "DELETE_ALL"
   // @Attention: Relevant properties - DeviceRingtoneList, SystemWideRingtoneList, AutoWaitingModeOnOffList etc.
   //             has to be updated accordingly to the clients.

   //Currently "DELETE_ALL" alone is used when we are getting method invocation from Diagnostic component.
   //As of now, Device address "ALL" is not used by any of the component including diagnostics.

   ExecuteSettingsInfo settingsInfo;
   settingsInfo._sysSetID = request->getSysSetID();
   settingsInfo._sysSetType = request->getSysSetType();
   settingsInfo._cookie = request->getCookie();

   LOG_INFO("DiagSystemSettings::onExecuteSystemSettingRequest:_sysSetID:%s",settingsInfo._sysSetID.c_str());
   LOG_INFO("DiagSystemSettings::onExecuteSystemSettingRequest:_sysSetID:%s",settingsInfo._sysSetType.c_str());

   _actToExecuteSettingsInfoMap.emplace_hint(_actToExecuteSettingsInfoMap.end(), request->getAct(), settingsInfo);

   if ((std::string("DIA_SYSSET_ID_PmApp") == settingsInfo._sysSetID) && ((std::string("DIA_SYSSET_TYPE_HMI") == settingsInfo._sysSetType)
         || (std::string("DIA_SYSSET_TYPE_PRODUCTION") == settingsInfo._sysSetType)))
   {
      pm_ipc_wrapper::IpcWrapper::getInstance().postRestoreDefaultSettingRequest2PmCore(DEVICE_ADDRESS_DELETE_ALL, request->getAct());
   }
   else
   {
      //Received System set Type other than DIA_SYSSET_TYPE_HMI and DIA_SYSSET_TYPE_PRODUCTION
      //Hence sending method error.
      DiagSystemSettings::onRestoreDefaultSettingResponse(false, request->getAct());
   }
}

void DiagSystemSettings::onFinalizeSystemSettingRequest(
      const ::boost::shared_ptr< FinalizeSystemSettingRequest >& request)
{
   LOG_INFO("DiagSystemSettings::onFinalizeSystemSettingRequest() entered");

   SystemSetID sysSetID = request->getSysSetID();
   SystemSetType sysSetType = request->getSysSetType();
   uint32 cookie = request->getCookie();

   ::std::vector< FinalizeSystemSettingResponseExtendedDataStruct > responseData;

   sendFinalizeSystemSettingResponse(sysSetID, sysSetType, responseData, 0, cookie, request->getAct());
}

void DiagSystemSettings::onRestoreDefaultSettingResponse(const bool operationSuccess, const act_t act)
{
   LOG_INFO("DiagSystemSettings::onRestoreDefaultSettingResponse() entered");

   auto iterSettingsInfoMap = _actToExecuteSettingsInfoMap.find(act);

   if (_actToExecuteSettingsInfoMap.end() != iterSettingsInfoMap)
   {
      SystemSetID sysSetID = iterSettingsInfoMap->second._sysSetID;
      SystemSetType sysSetType = iterSettingsInfoMap->second._sysSetType;
      uint32 cookie = iterSettingsInfoMap->second._cookie;

      std::vector< ExecuteSystemSettingResponseExtendedDataStruct > extendedData;

      if(operationSuccess)
      {
         sendExecuteSystemSettingResponse(sysSetID, sysSetType, extendedData, 0, cookie, act);
      }
      else
      {
         // TODO: Check whether "sendExecuteSystemSettingError" should be called or "sendExecuteSystemSettingResponse"
         sendExecuteSystemSettingResponse(sysSetID, sysSetType, extendedData, 1, cookie, act);
      }

      _actToExecuteSettingsInfoMap.erase(iterSettingsInfoMap);
   }
}

} // namespace pmapp
} // namespace bosch
} // namespace com
