/**
 * @file IpcMsgTypes.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the MsgTypes between IpcWrapper and CPP layer
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef IpcMsgTypes_h
#define IpcMsgTypes_h

#define PM_DEVICEHANDLE_ZERO (unsigned char)0x00
#define PM_CALLINSTANCE_INVALID 0xFF

#include "CallsReport.h"

#define SIGNAL_STRENGTH_MAX_VALUE 0x05
#define BATTERY_CHARGE_MAX_VALUE 0x05

#define CCA_OPCODE_STATUS (unsigned char)0x06

#define PROPERTY_ID_INVALID 0x00

enum updatedFieldMarkerEnumType
{
   VOICERECOGNITION_STATUS= 0x00,
   VR_AVAILABILITY_STATUS,
   ENHANCEDVOICERECOGNITION_FEATURE_STATUS,
};

/**
 * Defines the Profile usage types from Phone call Manager's perspective.
 */
enum BTProfileUsageType {
   PROFILE_USAGE_NONE = 0u,
   PROFILE_USAGE_GENERIC,
   PROFILE_USAGE_ACTIVE_CALL
};

enum IncomingCallEventType
{
   CALL_IDLE = 0u,
   CALL_INCOMING,
   CALL_WAITING
};

/**
 * Default value for asynchronous completion token (ACT).
 */
enum
{
   DEFAULT_ACT = 0
};

/**
 * Default value for asynchronous completion token (ACT).
 */
enum PhonebookErrorResponseEnum
{
   NO_ERROR = 0,
   COMPONENT_UNAVAILABLE,
   COMPONENT_BUSY,
   COMPONENT_ERROR,
   INVALID_PARAMETER
};

/**
 * Property IDs are the same as CCA property IDs.
 * This is because the Property IDs are checked and updated in Property Notification table in CcaProperty class.
 */
enum
{
   PM_PROPERTYID_SETACTIVEPHONE = 0xF07,
   PM_PROPERTYID_ACTIVECALLINHANDSET = 0x201,
   PM_PROPERTYID_CALLSTATUSNOTICE = 0x202,
   PM_PROPERTYID_CELLDEVICENAME = 0x205,
   PM_PROPERTYID_CELLNETWORKSTATUS = 0x206,
   PM_PROPERTYID_CELLSIGNALSTRENGTH = 0x207,
   PM_PROPERTYID_CELLSTATEOFCHARGE = 0x208,
   PM_PROPERTYID_INCOMINGCALLEVENT = 0x20D,
   PM_PROPERTYID_NETWORKOPERATOR = 0x20F,
   PM_PROPERTYID_SLCSTATUS = 0x212,
   PM_PROPERTYID_VEHICLEAUDIO = 0x21B,
   PM_PROPERTYID_MICROPHONEMUTESTATE = 0x220,
   PM_PROPERTYID_RINGTONEMUTESTATE = 0x221,
   PM_PROPERTYID_HFSERVICE = 0x222,
   PM_PROPERTYID_BTDEVICEVOICERECOGNITION = 0x225,
   PM_PROPERTYID_HANDSFREEENHANCEDCALLCONTROL = 0x226,
   PM_PROPERTYID_WAITINGMODEFILEPATH = 0xF01,
   PM_PROPERTYID_BTDEVICEVOICERECOGNITIONEXTENDED = 0xF03,
   PM_PROPERTYID_CALLSTATUSNOTICEEXTENDED = 0xF05,
   PM_PROPERTYID_ACTIVEPHONEDEVICE = 0xF06,
   PM_PROPERTYID_OUTGOINGCALLFAILEDEVENT = 0xF08,
   PM_PROPERTYID_INCOMINGCALLREJECTEDEVENT = 0xF09,
   PM_PROPERTYID_WAITINGMODESTATE = 0xF0A,
   PM_PROPERTYID_EXTERNALVRSTATE = 0xF0B,
   PM_PROPERTYID_SYSTEMWIDERINGTONESLIST = 0xF0E,
   PM_PROPERTYID_DEVICERINGTONELIST = 0xF0F,
   PM_PROPERTYID_SUPPRESSRINGTONEONOFFLIST = 0xF10,
   PM_PROPERTYID_AUTOWAITINGMODEONOFFLIST = 0xF11,
   PM_PROPERTYID_CALLOPERATIONFAILEDEVENT = 0xF17,
   PM_PROPERTYID_SENDDTMFFAILEDEVENT = 0xF18,
   PM_PROPERTYID_AUDIOTRANSFERFAILEDEVENT = 0xF19,
   PM_PROPERTYID_RINGTONEPLAYBACKSTATUS = 0xF1A,

   // The below property ID is not used in CCA but for DBus. Hence the value "0x001" is used
   //which will not conflict with the function ids of CCA based Phone Application
   PM_PROPERTYID_NOTIFICATIONEVENT = 0x001
};

/**
 * IPC type notification value
 */
enum IpcRequestType
{
   CCA = 1,
   ASF_DBUS
};

class DialInfo
{
public:
   DialInfo() : _telephoneNumber(""), _firstName(""), _lastName(""), _hideCallerId(pmcore::DEFAULT_CALLERID)
   {

   }

   DialInfo(pmcore::TelephoneNumber telephoneNumber, pmcore::FirstName firstName,
         pmcore::LastName lastName, pmcore::HideCallerId hideCallerId)
   : _telephoneNumber(telephoneNumber), _firstName(firstName), _lastName(lastName), _hideCallerId(hideCallerId)
   {

   }

   DialInfo(const DialInfo& ref)
   : _telephoneNumber(ref._telephoneNumber), _firstName(ref._firstName), _lastName(ref._lastName),
     _hideCallerId(ref._hideCallerId)
   {

   }

   friend void swap(DialInfo& first, DialInfo& second)
   {
      using std::swap;

      swap(first._telephoneNumber, second._telephoneNumber);
      swap(first._firstName, second._firstName);
      swap(first._lastName, second._lastName);
      swap(first._hideCallerId, second._hideCallerId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   DialInfo& operator=(DialInfo ref)
   {
      swap(*this, ref);
      return *this;
   }

   bool operator==(const DialInfo& rhs) const
   {
      return ((this->_telephoneNumber == rhs._telephoneNumber)
            && (this->_firstName == rhs._firstName)
            && (this->_lastName == rhs._lastName)
            && (this->_hideCallerId == rhs._hideCallerId));
   }

   pmcore::TelephoneNumber _telephoneNumber;
   pmcore::FirstName _firstName;
   pmcore::LastName _lastName;
   pmcore::HideCallerId _hideCallerId;
};

/**
 * The classes defined here are majority - a collection of types. Hence the member variables shall be in public scope:
 * @Refer: Rule 59: Class variables should never be declared public. Source: Docupedia - C++ Best Practices
 * The concept of C++ information hiding and encapsulation is violated by public variables. Use private variables
 * and access functions instead. One exception to this rule is when the class is essentially a data structure,
 * with no behavior (equivalent to a C struct). In this case it is appropriate to make the class'instance
 * variables public.
 */

// TODO: Generic types are used -unsigned int, unsigned long.. typedef shall be used once they are collectively defined
/*
 * This structure defines the CCA message header information
 */
class CcaMsgHeader
{
public:

   /**
    * Constructor of CcaMsgHeader class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CcaMsgHeader()
   {
      _u16DestAppID = 0;
      _u16CmdCtr = 0;
      _u16RegisterId = 0;
      _u16Fid = 0;
      _u8OpCode = CCA_OPCODE_STATUS;
   }

   unsigned short int _u16DestAppID;   /**< Destination Application ID */

   unsigned short int _u16CmdCtr;   /**< Command counter of the Message set by client*/

   unsigned short int _u16RegisterId;   /**< Register ID of the Message */

   unsigned short int _u16Fid;   /**< Function ID of the Message */

   unsigned char _u8OpCode;   /**< OPCODE of the Message  - SET, GET, PURESET, STATUS, METHODSTART.. */
};

/*
 * class that contains the format of response from CPP to the IpcMessageHandler's
 */
class PmCoreResponseData
{
public:
   /**
    * Constructor of PmCoreResponseData class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   PmCoreResponseData() : _pmResult(), _u64TokenId(DEFAULT_ACT)
   {

   }

   PmCoreResponseData(pmcore::PmResult pmResult, unsigned long long int u64TokenId)
   : _pmResult(pmResult), _u64TokenId(u64TokenId)
   {

   }

   PmCoreResponseData(const PmCoreResponseData& ref)
   : _pmResult(ref._pmResult), _u64TokenId(ref._u64TokenId)
   {

   }

   friend void swap(PmCoreResponseData& first, PmCoreResponseData& second)
   {
      using std::swap;

      swap(first._pmResult, second._pmResult);
      swap(first._u64TokenId, second._u64TokenId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreResponseData& operator=(PmCoreResponseData ref)
   {
      swap(*this, ref);
      return *this;
   }

   bool operator==(const PmCoreResponseData& rhs) const
   {
      return ((this->_pmResult == rhs._pmResult)
            && (this->_u64TokenId == rhs._u64TokenId));
   }

   // Invoked to get the Token ID
   unsigned long long int getTokenId()
   {
      return _u64TokenId;
   }

   pmcore::PmResult _pmResult; /**<Message response */
   unsigned long long int _u64TokenId; /**< Asynchronous token to match the request and response */
};

// Class to handle the Accept call response from PMCore
class AcceptCallResponseMsg : public PmCoreResponseData
{
public:
   AcceptCallResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress(""),
   _callInstance(PM_CALLINSTANCE_INVALID)
   {

   }

   AcceptCallResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress, pmcore::CallInstance callInstance)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress), _callInstance(callInstance)
   {

   }

   AcceptCallResponseMsg(const AcceptCallResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress), _callInstance(ref._callInstance)
   {

   }

   friend void swap(AcceptCallResponseMsg& first, AcceptCallResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AcceptCallResponseMsg& operator=(AcceptCallResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const AcceptCallResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   pmcore::BdAddress _deviceAddress;
   pmcore::CallInstance _callInstance;
};

// Class to handle the Re-dial response from PMCore
class RedialResponseMsg : public PmCoreResponseData
{
public:
   RedialResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress(""),
   _callInstance(PM_CALLINSTANCE_INVALID)
   {

   }

   RedialResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress, pmcore::CallInstance callInstance)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress), _callInstance(callInstance)
   {

   }

   RedialResponseMsg(const RedialResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress), _callInstance(ref._callInstance)
   {

   }

   friend void swap(RedialResponseMsg& first, RedialResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   RedialResponseMsg& operator=(RedialResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const RedialResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   pmcore::BdAddress _deviceAddress;
   pmcore::CallInstance _callInstance;
};

// Class to handle the speed dial response from PMCore
class SpeedDialResponseMsg : public PmCoreResponseData
{
public:
   SpeedDialResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress(""),
   _callInstance(PM_CALLINSTANCE_INVALID)
   {

   }

   SpeedDialResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress, pmcore::CallInstance callInstance)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress), _callInstance(callInstance)
   {

   }

   SpeedDialResponseMsg(const SpeedDialResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress), _callInstance(ref._callInstance)
   {

   }

   friend void swap(SpeedDialResponseMsg& first, SpeedDialResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   SpeedDialResponseMsg& operator=(SpeedDialResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const SpeedDialResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   pmcore::BdAddress _deviceAddress;
   pmcore::CallInstance _callInstance;
};

// Class to handle the dial response from PMCore
class DialResponseMsg : public PmCoreResponseData
{
public:
   DialResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress(""),
   _callInstance(PM_CALLINSTANCE_INVALID)
   {

   }

   DialResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress, pmcore::CallInstance callInstance)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress), _callInstance(callInstance)
   {

   }

   DialResponseMsg(const DialResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress), _callInstance(ref._callInstance)
   {

   }

   friend void swap(DialResponseMsg& first, DialResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   DialResponseMsg& operator=(DialResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const DialResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   pmcore::BdAddress _deviceAddress;
   pmcore::CallInstance _callInstance;
};

// Class to handle the swap call response from PMCore.
class SwapCallResponseMsg : public PmCoreResponseData
{
public:
   SwapCallResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress("")
   {

   }

   SwapCallResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId, pmcore::BdAddress deviceAddress)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress)
   {

   }

   SwapCallResponseMsg(const SwapCallResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress)
   {

   }

   friend void swap(SwapCallResponseMsg& first, SwapCallResponseMsg& second)
   {
      using std::swap;
      swap(first._deviceAddress, second._deviceAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   SwapCallResponseMsg& operator=(SwapCallResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const SwapCallResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress));
   }

   pmcore::BdAddress _deviceAddress;
};

// Class to handle the Start-stop waiting mode response from PMCore.
class StartStopwaitingModeResponseMsg : public PmCoreResponseData
{
public:
   StartStopwaitingModeResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress("")
   {

   }

   StartStopwaitingModeResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress)
   {

   }

   StartStopwaitingModeResponseMsg(const StartStopwaitingModeResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress)
   {

   }

   friend void swap(StartStopwaitingModeResponseMsg& first, StartStopwaitingModeResponseMsg& second)
   {
      using std::swap;
      swap(first._deviceAddress, second._deviceAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   StartStopwaitingModeResponseMsg& operator=(StartStopwaitingModeResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const StartStopwaitingModeResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress));
   }

   pmcore::BdAddress _deviceAddress;
};

// Class to handle the sendDTMF call response from PMCore.
class SendDTMFResponseMsg : public PmCoreResponseData
{
public:
   SendDTMFResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress("")
   {

   }

   SendDTMFResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId, pmcore::BdAddress deviceAddress)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress)
   {

   }

   SendDTMFResponseMsg(const SendDTMFResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress)
   {

   }

   friend void swap(SendDTMFResponseMsg& first, SendDTMFResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   SendDTMFResponseMsg& operator=(SendDTMFResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const SendDTMFResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress));
   }

   pmcore::BdAddress _deviceAddress;
};

// Class to handle the Transfer audio response from PMCore.
class TransferAudioResponseMsg : public PmCoreResponseData
{
public:
   TransferAudioResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress("")
   {

   }

   TransferAudioResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress)
   {

   }

   TransferAudioResponseMsg(const TransferAudioResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress)
   {

   }

   friend void swap(TransferAudioResponseMsg& first, TransferAudioResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   TransferAudioResponseMsg& operator=(TransferAudioResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const TransferAudioResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress));
   }

   pmcore::BdAddress _deviceAddress;
};

// Class to handle the VoiceMail list response from PMCore
class VoiceMailListResponseMsg : public PmCoreResponseData
{
public:
   VoiceMailListResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress("")
   {

   }

   VoiceMailListResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress, pmcore::VoiceMailList voiceMailList = pmcore::VoiceMailList())
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress),_voiceMailList(voiceMailList)
   {

   }

   VoiceMailListResponseMsg(const VoiceMailListResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress), _voiceMailList(ref._voiceMailList)
   {

   }

   friend void swap(VoiceMailListResponseMsg& first, VoiceMailListResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
      swap(first._voiceMailList, second._voiceMailList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   VoiceMailListResponseMsg& operator=(VoiceMailListResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const VoiceMailListResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_voiceMailList == rhs._voiceMailList));
   }

   pmcore::BdAddress _deviceAddress;
   pmcore::VoiceMailList _voiceMailList;
};

class PropertyUpdate
{
public:
   /**
    * Constructor of PropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   PropertyUpdate() : _u16PropertyId(PROPERTY_ID_INVALID)
   {

   }

   PropertyUpdate(unsigned short int u16PropertyId) : _u16PropertyId(u16PropertyId)
   {

   }

   PropertyUpdate(const PropertyUpdate& ref) : _u16PropertyId(ref._u16PropertyId)
   {

   }

   friend void swap(PropertyUpdate& first, PropertyUpdate& second)
   {
      using std::swap;
      swap(first._u16PropertyId, second._u16PropertyId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PropertyUpdate& operator=(PropertyUpdate ref)
   {
      swap(*this, ref);
      return *this;
   }

   bool operator==(const PropertyUpdate& rhs) const
   {
      return (this->_u16PropertyId == rhs._u16PropertyId);
   }

   // Invoked to get the property ID
   unsigned short int u16GetFunctionID()
   {
      return _u16PropertyId;
   }

   unsigned short int _u16PropertyId;
};

class RingtoneMuteStatePropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of RingtoneMuteStatePropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   RingtoneMuteStatePropertyUpdate()
   : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _deviceAddress(""), _ringtoneMuteStateObj(false)
   {

   }

   RingtoneMuteStatePropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId,
         pmcore::BdAddress deviceAddress, pmcore::RingtoneMuteState ringtoneMuteState)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _deviceAddress(deviceAddress), _ringtoneMuteStateObj(ringtoneMuteState)
   {

   }

   RingtoneMuteStatePropertyUpdate(const RingtoneMuteStatePropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _deviceAddress(ref._deviceAddress),
     _ringtoneMuteStateObj(ref._ringtoneMuteStateObj)
   {

   }

   friend void swap(RingtoneMuteStatePropertyUpdate& first, RingtoneMuteStatePropertyUpdate& second)
   {
      using std::swap;
      swap(first._deviceAddress, second._deviceAddress);
      swap(first._ringtoneMuteStateObj, second._ringtoneMuteStateObj);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   RingtoneMuteStatePropertyUpdate& operator=(RingtoneMuteStatePropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const RingtoneMuteStatePropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs))
            && (PropertyUpdate::operator==(rhs))
            && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_ringtoneMuteStateObj == rhs._ringtoneMuteStateObj));
   }

   pmcore::BdAddress _deviceAddress;
   pmcore::RingtoneMuteState _ringtoneMuteStateObj;
};

class PhoneCallAudioPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of PhoneCallAudioPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   PhoneCallAudioPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _phoneCallAudioActive(false)
   {

   }

   PhoneCallAudioPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId, pmcore::PhoneCallAudioActive phoneCallAudioActive)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _phoneCallAudioActive(phoneCallAudioActive)
   {

   }

   PhoneCallAudioPropertyUpdate(const PhoneCallAudioPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _phoneCallAudioActive(ref._phoneCallAudioActive)
   {

   }

   friend void swap(PhoneCallAudioPropertyUpdate& first, PhoneCallAudioPropertyUpdate& second)
   {
      using std::swap;
      swap(first._phoneCallAudioActive, second._phoneCallAudioActive);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PhoneCallAudioPropertyUpdate& operator=(PhoneCallAudioPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const PhoneCallAudioPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs))
            && (PropertyUpdate::operator==(rhs))
            && (this->_phoneCallAudioActive == rhs._phoneCallAudioActive));
   }

   pmcore::PhoneCallAudioActive _phoneCallAudioActive;
};

class WaitingModeStatePropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of WaitingModeStatePropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   WaitingModeStatePropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _deviceAddress(""), _waitingModeStateObj(0)
   {

   }

   WaitingModeStatePropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId, pmcore::BdAddress deviceAddress,
         pmcore::WaitingModeState waitingModeStateObj)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId), _deviceAddress(deviceAddress),
     _waitingModeStateObj(waitingModeStateObj)
   {

   }

   WaitingModeStatePropertyUpdate(const WaitingModeStatePropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _deviceAddress(ref._deviceAddress),
     _waitingModeStateObj(ref._waitingModeStateObj)
   {

   }

   friend void swap(WaitingModeStatePropertyUpdate& first, WaitingModeStatePropertyUpdate& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
      swap(first._waitingModeStateObj, second._waitingModeStateObj);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   WaitingModeStatePropertyUpdate& operator=(WaitingModeStatePropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const WaitingModeStatePropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_waitingModeStateObj == rhs._waitingModeStateObj));
   }

   pmcore::BdAddress _deviceAddress;
   pmcore::WaitingModeState _waitingModeStateObj;
};

class RingtonePlaybackStatusPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of RingtonePlaybackStatusPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   RingtonePlaybackStatusPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _ringtonePlaybackState(pmcore::RT_NOT_PLAYING)
   {

   }

   RingtonePlaybackStatusPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId,
         pmcore::RingtonePlaybackState ringtonePlaybackState)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _ringtonePlaybackState(ringtonePlaybackState)
   {

   }

   RingtonePlaybackStatusPropertyUpdate(const RingtonePlaybackStatusPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref),
     _ringtonePlaybackState(ref._ringtonePlaybackState)
   {

   }

   friend void swap(RingtonePlaybackStatusPropertyUpdate& first, RingtonePlaybackStatusPropertyUpdate& second)
   {
      using std::swap;
      swap(first._ringtonePlaybackState, second._ringtonePlaybackState);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   RingtonePlaybackStatusPropertyUpdate& operator=(RingtonePlaybackStatusPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const RingtonePlaybackStatusPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_ringtonePlaybackState == rhs._ringtonePlaybackState));
   }

   pmcore::RingtonePlaybackState _ringtonePlaybackState;
};

class WaitingModeFilePathPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of WaitingModeFilePathPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   WaitingModeFilePathPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID), _waitingModeFilePath()
   {

   }

   WaitingModeFilePathPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId, pmcore::WaitingModeFilePath waitingModeFilePath)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId), _waitingModeFilePath(waitingModeFilePath)
   {

   }

   WaitingModeFilePathPropertyUpdate(const WaitingModeFilePathPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _waitingModeFilePath(ref._waitingModeFilePath)
   {

   }

   friend void swap(WaitingModeFilePathPropertyUpdate& first, WaitingModeFilePathPropertyUpdate& second)
   {
      using std::swap;

      swap(first._waitingModeFilePath, second._waitingModeFilePath);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   WaitingModeFilePathPropertyUpdate& operator=(WaitingModeFilePathPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const WaitingModeFilePathPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs)) &&
            (this->_waitingModeFilePath == rhs._waitingModeFilePath));
   }

   pmcore::WaitingModeFilePath _waitingModeFilePath;
};

class ActivePassiveDeviceListPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of ActivePassiveDeviceListPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ActivePassiveDeviceListPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _activePassiveDeviceListObj()
   {

   }

   ActivePassiveDeviceListPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId, pmcore::ActivePassiveDeviceList activePassiveDeviceListObj)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _activePassiveDeviceListObj(activePassiveDeviceListObj)
   {

   }

   ActivePassiveDeviceListPropertyUpdate(const ActivePassiveDeviceListPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _activePassiveDeviceListObj(ref._activePassiveDeviceListObj)
   {

   }

   friend void swap(ActivePassiveDeviceListPropertyUpdate& first, ActivePassiveDeviceListPropertyUpdate& second)
   {
      using std::swap;

      swap(first._activePassiveDeviceListObj, second._activePassiveDeviceListObj);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   ActivePassiveDeviceListPropertyUpdate& operator=(ActivePassiveDeviceListPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const ActivePassiveDeviceListPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs)) &&
            (this->_activePassiveDeviceListObj == rhs._activePassiveDeviceListObj));
   }

   pmcore::ActivePassiveDeviceList _activePassiveDeviceListObj;
};

class CallsReportListPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of CallsReportListPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CallsReportListPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID), _callsReportList()
   {

   }

   CallsReportListPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId, com::bosch::pmapp::CallsReportList callsReportList)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId), _callsReportList(callsReportList)
   {

   }

   CallsReportListPropertyUpdate(const CallsReportListPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _callsReportList(ref._callsReportList)
   {

   }

   friend void swap(CallsReportListPropertyUpdate& first, CallsReportListPropertyUpdate& second)
   {
      using std::swap;

      swap(first._callsReportList, second._callsReportList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   CallsReportListPropertyUpdate& operator=(CallsReportListPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const CallsReportListPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_callsReportList == rhs._callsReportList));
   }

   com::bosch::pmapp::CallsReportList _callsReportList;
};

// The below template class shall be used for AG device specific properties such as CellNetworkStatus,
// CellSignalStrength, CellStateOfCharge, DeviceFriendlyName, NetworkOperator,
// HandsFreeEnhancedCallControl(SupportedFeatures)
template<typename TPropertyType>
class AgInfoPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of AgInfoPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   AgInfoPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _deviceAddress(""), _value()
   {

   }

   AgInfoPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId, unsigned short int u16PropertyId,
         pmcore::BdAddress deviceAddress, TPropertyType value)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _deviceAddress(deviceAddress), _value(value)
   {

   }

   AgInfoPropertyUpdate(const AgInfoPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _deviceAddress(ref._deviceAddress), _value(ref._value)
   {

   }

   friend void swap(AgInfoPropertyUpdate& first, AgInfoPropertyUpdate& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
      swap(first._value, second._value);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AgInfoPropertyUpdate& operator=(AgInfoPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const AgInfoPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_value == rhs._value));
   }

   pmcore::BdAddress _deviceAddress;
   TPropertyType _value;
};

class IncomingCallPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{

public:

   // Class containing information about the incoming call.
   class IncomingCallInfo
   {
   public:
      IncomingCallInfo() : _deviceAddress(""), _firstName(""), _lastName("")
      {

      }

      IncomingCallInfo(pmcore::BdAddress deviceAddress, pmcore::FirstName firstName, pmcore::LastName lastName)
      : _deviceAddress(deviceAddress), _firstName(firstName), _lastName(lastName)
      {

      }

      IncomingCallInfo(const IncomingCallInfo& ref)
      : _deviceAddress(ref._deviceAddress), _firstName(ref._firstName), _lastName(ref._lastName)
      {

      }

      friend void swap(IncomingCallInfo& first, IncomingCallInfo& second)
      {
         using std::swap;

         swap(first._deviceAddress, second._deviceAddress);
         swap(first._firstName, second._firstName);
         swap(first._lastName, second._lastName);
      }

      //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
      IncomingCallInfo& operator=(IncomingCallInfo ref)
      {
         swap(*this, ref);
         return *this;
      }

      bool operator==(const IncomingCallInfo& rhs) const
      {
         return ((this->_deviceAddress == rhs._deviceAddress)
               && (this->_firstName == rhs._firstName)
               && (this->_lastName == rhs._lastName));
      }

      pmcore::BdAddress _deviceAddress;
      pmcore::FirstName _firstName;
      pmcore::LastName _lastName;
   };

   /**
    * Constructor of IncomingCallPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   IncomingCallPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _callInfo(), _incomingCallInfo()
   {

   }

   IncomingCallPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId, pmcore::CallInfo callInfo, IncomingCallInfo incomingCallInfoMap)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _callInfo(callInfo), _incomingCallInfo(incomingCallInfoMap)
   {

   }

   IncomingCallPropertyUpdate(const IncomingCallPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _callInfo(ref._callInfo), _incomingCallInfo(ref._incomingCallInfo)
   {

   }

   friend void swap(IncomingCallPropertyUpdate& first, IncomingCallPropertyUpdate& second)
   {
      using std::swap;

      swap(first._callInfo, second._callInfo);
      swap(first._incomingCallInfo, second._incomingCallInfo);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   IncomingCallPropertyUpdate& operator=(IncomingCallPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const IncomingCallPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_callInfo == rhs._callInfo)
            && (this->_incomingCallInfo == rhs._incomingCallInfo));
   }

   pmcore::CallInfo _callInfo;
   IncomingCallInfo _incomingCallInfo;
};

class PmNotificationEventPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of PmNotificationEventPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   PmNotificationEventPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID), _eventInfoMap(),
      _callInstanceList()
   {

   }

   PmNotificationEventPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId, std::map<std::string, std::string> eventInfoMap,
         pmcore::CallInstanceList callInstanceList)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId), _eventInfoMap(eventInfoMap),
     _callInstanceList(callInstanceList)
   {

   }

   PmNotificationEventPropertyUpdate(const PmNotificationEventPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _eventInfoMap(ref._eventInfoMap),
     _callInstanceList(ref._callInstanceList)
   {

   }

   friend void swap(PmNotificationEventPropertyUpdate& first, PmNotificationEventPropertyUpdate& second)
   {
      using std::swap;

      swap(first._eventInfoMap, second._eventInfoMap);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmNotificationEventPropertyUpdate& operator=(PmNotificationEventPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const PmNotificationEventPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_eventInfoMap == rhs._eventInfoMap));
   }

   // _eventInfoMap should have details such as DeviceAddress, pmcore::FirstName, pmcore::LastName,
   // TelephoneNumber
   std::map<std::string, std::string> _eventInfoMap;
   pmcore::CallInstanceList _callInstanceList;
};

class MicMuteStatePropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of MicMuteStatePropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   MicMuteStatePropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _microphoneMuteStateObj(false)
   {

   }

   MicMuteStatePropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId, pmcore::MicrophoneMuteState microphoneMuteStateObj)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _microphoneMuteStateObj(microphoneMuteStateObj)
   {

   }

   MicMuteStatePropertyUpdate(const MicMuteStatePropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _microphoneMuteStateObj(ref._microphoneMuteStateObj)
   {

   }

   friend void swap(MicMuteStatePropertyUpdate& first, MicMuteStatePropertyUpdate& second)
   {
      using std::swap;

      swap(first._microphoneMuteStateObj, second._microphoneMuteStateObj);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   MicMuteStatePropertyUpdate& operator=(MicMuteStatePropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const MicMuteStatePropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs)) &&
            (this->_microphoneMuteStateObj == rhs._microphoneMuteStateObj));
   }

   pmcore::MicrophoneMuteState _microphoneMuteStateObj;
};

class VoiceRecognitionPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of VoiceRecognitionPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   VoiceRecognitionPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _deviceAddress(""), _vrAvailabilityStatus(false), _voiceRecognitionStatus(), _enhancedVoiceRecognitionFeature(),
      _updatedFieldMarker(VOICERECOGNITION_STATUS)
   {

   }

   VoiceRecognitionPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId,
         pmcore::BdAddress deviceAddress, bool vrAvailabilityStatus,
         pmcore::VoiceRecognitionStatus voiceRecognitionStatus,
         pmcore::EnhancedVoiceRecognitionFeature enhancedVoiceRecognitionFeature,
         updatedFieldMarkerEnumType updatedFieldMarker = VOICERECOGNITION_STATUS)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId), _deviceAddress(deviceAddress),
     _vrAvailabilityStatus(vrAvailabilityStatus),
     _voiceRecognitionStatus(voiceRecognitionStatus), _enhancedVoiceRecognitionFeature(enhancedVoiceRecognitionFeature),
     _updatedFieldMarker(updatedFieldMarker)
   {

   }

   VoiceRecognitionPropertyUpdate(const VoiceRecognitionPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _deviceAddress(ref._deviceAddress),
     _vrAvailabilityStatus(ref._vrAvailabilityStatus),
     _voiceRecognitionStatus(ref._voiceRecognitionStatus),
     _enhancedVoiceRecognitionFeature(ref._enhancedVoiceRecognitionFeature),
     _updatedFieldMarker(ref._updatedFieldMarker)
   {

   }

   friend void swap(VoiceRecognitionPropertyUpdate& first, VoiceRecognitionPropertyUpdate& second)
   {
      using std::swap;
      swap(first._deviceAddress, second._deviceAddress);
      swap(first._vrAvailabilityStatus, second._vrAvailabilityStatus);
      swap(first._voiceRecognitionStatus, second._voiceRecognitionStatus);
      swap(first._enhancedVoiceRecognitionFeature, second._enhancedVoiceRecognitionFeature);
      swap(first._updatedFieldMarker, second._updatedFieldMarker);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   VoiceRecognitionPropertyUpdate& operator=(VoiceRecognitionPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const VoiceRecognitionPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_vrAvailabilityStatus == rhs._vrAvailabilityStatus)
            && (this->_voiceRecognitionStatus == rhs._voiceRecognitionStatus)
            && (this->_enhancedVoiceRecognitionFeature == rhs._enhancedVoiceRecognitionFeature)
            && (this->_updatedFieldMarker == rhs._updatedFieldMarker));
   }

   pmcore::BdAddress _deviceAddress;
   bool _vrAvailabilityStatus;
   pmcore::VoiceRecognitionStatus _voiceRecognitionStatus;
   pmcore::EnhancedVoiceRecognitionFeature _enhancedVoiceRecognitionFeature;

   // The field "_updatedFieldMarker" is used to notify as of which parameter is updated.
   updatedFieldMarkerEnumType _updatedFieldMarker;
};

class DeviceRingtoneListPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of DeviceRingtoneListPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   DeviceRingtoneListPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _deviceRingtoneList()
   {

   }

   DeviceRingtoneListPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId,
         pmcore::DeviceRingtoneList deviceRingtoneList)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _deviceRingtoneList(deviceRingtoneList)
   {

   }

   DeviceRingtoneListPropertyUpdate(const DeviceRingtoneListPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref),
     _deviceRingtoneList(ref._deviceRingtoneList)
   {

   }

   friend void swap(DeviceRingtoneListPropertyUpdate& first, DeviceRingtoneListPropertyUpdate& second)
   {
      using std::swap;
      swap(first._deviceRingtoneList, second._deviceRingtoneList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   DeviceRingtoneListPropertyUpdate& operator=(DeviceRingtoneListPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const DeviceRingtoneListPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_deviceRingtoneList == rhs._deviceRingtoneList));
   }

   pmcore::DeviceRingtoneList _deviceRingtoneList;
};

class SystemWideRingtonesListPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of SystemWideRingtonesListPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   SystemWideRingtonesListPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _systemwideRingtonesList()
   {

   }

   SystemWideRingtonesListPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId,
         pmcore::SystemWideRingtonesList systemwideRingtonesList)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _systemwideRingtonesList(systemwideRingtonesList)
   {

   }

   SystemWideRingtonesListPropertyUpdate(const SystemWideRingtonesListPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref),
     _systemwideRingtonesList(ref._systemwideRingtonesList)
   {

   }

   friend void swap(SystemWideRingtonesListPropertyUpdate& first, SystemWideRingtonesListPropertyUpdate& second)
   {
      using std::swap;
      swap(first._systemwideRingtonesList, second._systemwideRingtonesList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   SystemWideRingtonesListPropertyUpdate& operator=(SystemWideRingtonesListPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const SystemWideRingtonesListPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_systemwideRingtonesList == rhs._systemwideRingtonesList));
   }

   pmcore::SystemWideRingtonesList _systemwideRingtonesList;
};

class AutoWaitingModeOnOffListPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of AutoWaitingModeOnOffListPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   AutoWaitingModeOnOffListPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _autoWaitingModeOnOffList()
   {
   }

   AutoWaitingModeOnOffListPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId,
         pmcore::AutoWaitingModeOnOffList autoWaitingModeOnOffList)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _autoWaitingModeOnOffList(autoWaitingModeOnOffList)
   {
   }

   AutoWaitingModeOnOffListPropertyUpdate(const AutoWaitingModeOnOffListPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref),
     _autoWaitingModeOnOffList(ref._autoWaitingModeOnOffList)
   {
   }

   friend void swap(AutoWaitingModeOnOffListPropertyUpdate& first, AutoWaitingModeOnOffListPropertyUpdate& second)
   {
      using std::swap;
      swap(first._autoWaitingModeOnOffList, second._autoWaitingModeOnOffList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   AutoWaitingModeOnOffListPropertyUpdate& operator=(AutoWaitingModeOnOffListPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const AutoWaitingModeOnOffListPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_autoWaitingModeOnOffList == rhs._autoWaitingModeOnOffList));
   }

   pmcore::AutoWaitingModeOnOffList _autoWaitingModeOnOffList;
};

class SuppressRingtoneOnOffListPropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of SuppressRingtoneOnOffListPropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   SuppressRingtoneOnOffListPropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID),
      _suppressRingtoneOnOffList()
   {
   }

   SuppressRingtoneOnOffListPropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId,
         unsigned short int u16PropertyId,
         pmcore::SuppressRingtoneOnOffList suppressRingtoneOnOffList)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId),
     _suppressRingtoneOnOffList(suppressRingtoneOnOffList)
   {
   }

   SuppressRingtoneOnOffListPropertyUpdate(const SuppressRingtoneOnOffListPropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref),
     _suppressRingtoneOnOffList(ref._suppressRingtoneOnOffList)
   {
   }

   friend void swap(SuppressRingtoneOnOffListPropertyUpdate& first, SuppressRingtoneOnOffListPropertyUpdate& second)
   {
      using std::swap;
      swap(first._suppressRingtoneOnOffList, second._suppressRingtoneOnOffList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   SuppressRingtoneOnOffListPropertyUpdate& operator=(SuppressRingtoneOnOffListPropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const SuppressRingtoneOnOffListPropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_suppressRingtoneOnOffList == rhs._suppressRingtoneOnOffList));
   }

   pmcore::SuppressRingtoneOnOffList _suppressRingtoneOnOffList;
};

class ExternalVRStatePropertyUpdate : public PmCoreResponseData, public PropertyUpdate
{
public:
   /**
    * Constructor of ExternalVRStatePropertyUpdate class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ExternalVRStatePropertyUpdate() :
      PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), PropertyUpdate(PROPERTY_ID_INVALID), _deviceAddress(""),
      _externalVRStatus(pmcore::EXT_VR_SESSION_IDLE), _vrInterruptedReason(pmcore::AUDIO_CHANNEL_RELEASED)
   {
   }

   ExternalVRStatePropertyUpdate(pmcore::PmResult pmResult, pmcore::ActType u64TokenId, unsigned short int u16PropertyId,
         pmcore::BdAddress deviceAddress, pmcore::ExternalVRStatusEnumType externalVRStatus,
         pmcore::VRInterruptedReason vrInterruptedReason)
   : PmCoreResponseData(pmResult, u64TokenId), PropertyUpdate(u16PropertyId), _deviceAddress(deviceAddress),
     _externalVRStatus(externalVRStatus), _vrInterruptedReason(vrInterruptedReason)
   {
   }

   ExternalVRStatePropertyUpdate(const ExternalVRStatePropertyUpdate& ref)
   : PmCoreResponseData(ref), PropertyUpdate(ref), _deviceAddress(ref._deviceAddress),
     _externalVRStatus(ref._externalVRStatus), _vrInterruptedReason(ref._vrInterruptedReason)
   {
   }

   friend void swap(ExternalVRStatePropertyUpdate& first, ExternalVRStatePropertyUpdate& second)
   {
      using std::swap;
      swap(first._deviceAddress, second._deviceAddress);
      swap(first._externalVRStatus, second._externalVRStatus);
      swap(first._vrInterruptedReason, second._vrInterruptedReason);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   ExternalVRStatePropertyUpdate& operator=(ExternalVRStatePropertyUpdate ref)
   {
      PmCoreResponseData::operator=(ref);
      PropertyUpdate::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const ExternalVRStatePropertyUpdate& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (PropertyUpdate::operator==(rhs))
            && (this->_deviceAddress == rhs._deviceAddress)
            && (this->_externalVRStatus == rhs._externalVRStatus)
            && (this->_vrInterruptedReason == rhs._vrInterruptedReason));
   }

   pmcore::BdAddress                _deviceAddress;
   pmcore::ExternalVRStatusEnumType _externalVRStatus;
   pmcore::VRInterruptedReason      _vrInterruptedReason;
};

// Class to handle the activateExternalVR response from PMCore
class ActivateExternalVRResponseMsg : public PmCoreResponseData
{
public:
   ActivateExternalVRResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress("")
   {
   }

   ActivateExternalVRResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress)
   {
   }

   ActivateExternalVRResponseMsg(const ActivateExternalVRResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress)
   {
   }

   friend void swap(ActivateExternalVRResponseMsg& first, ActivateExternalVRResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   ActivateExternalVRResponseMsg& operator=(ActivateExternalVRResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const ActivateExternalVRResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress));
   }

   pmcore::BdAddress _deviceAddress;
};

// Class to handle the activateExternalVR response from PMCore
class DeActivateExternalVRResponseMsg : public PmCoreResponseData
{
public:
   DeActivateExternalVRResponseMsg() : PmCoreResponseData(pmcore::PmResult(), DEFAULT_ACT), _deviceAddress("")
   {
   }

   DeActivateExternalVRResponseMsg(pmcore::PmResult pmResult, unsigned long long int u64TokenId,
         pmcore::BdAddress deviceAddress)
   : PmCoreResponseData(pmResult, u64TokenId), _deviceAddress(deviceAddress)
   {
   }

   DeActivateExternalVRResponseMsg(const DeActivateExternalVRResponseMsg& ref)
   : PmCoreResponseData(ref), _deviceAddress(ref._deviceAddress)
   {
   }

   friend void swap(DeActivateExternalVRResponseMsg& first, DeActivateExternalVRResponseMsg& second)
   {
      using std::swap;

      swap(first._deviceAddress, second._deviceAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   DeActivateExternalVRResponseMsg& operator=(DeActivateExternalVRResponseMsg ref)
   {
      PmCoreResponseData::operator=(ref);
      swap(*this, ref);
      return *this;
   }

   bool operator==(const DeActivateExternalVRResponseMsg& rhs) const
   {
      return ((PmCoreResponseData::operator==(rhs)) && (this->_deviceAddress == rhs._deviceAddress));
   }

   pmcore::BdAddress _deviceAddress;
};

#endif // #define IpcMsgTypes_h
