/**
 * @file IBmAppRequestIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the IBmAppRequestIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file holds the class definition for IBmAppRequestIf class
 *
 * @ingroup IpcWrapper
 */

#ifndef IBmAppRequestIf_h
#define IBmAppRequestIf_h

namespace com
{
namespace bosch
{
namespace pmapp
{

class IBmAppCallBackIf;

class IBmAppRequestIf
{
public:
   /**
    * Virtual destructor of the interface class IBmAppRequestIf
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual ~IBmAppRequestIf() {}

   /**
    * This function is called to set the callback interface object of the BM Application.
    *
    * @param[in] IBmAppCallBackIf* - Pointer to the Callback Interface of BM Application client.
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void setBmAppCallBackIf(IBmAppCallBackIf*) = 0;

   /**
    * This function is used to call the SetPrimary method of the BM Application interface
    * in order to set a device as a Primary(Active) device(s).
    *
    * @param[in]  activeDevicesList- List of Active devices
    *
    * @param[out]
    *
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void setPrimaryDevice(const pmcore::BdAddressList& activeDevicesList) = 0;

   /**
    * This function is used to call the SetBTProfilesUsage method of the BM Application interface
    * in order to set the HFP profile usage.
    *
    * @param[in]
    *            std::string deviceAddress
    *             - Device Address of the device for which the HFP profile usage needs to be set.
    *            bool profileUsageInfo
    *             - Value to be set for the HFP profile usage. If TRUE- HFP profile usage is done, else not.
    *            BTProfileUsageType profileUsageType
    *             - HFP profile usage type and it is of ENUM type.
    *
    * @param[out] act_t - Asynchronous Callback Token received from the ASF FW.
    *
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void setBTProfilesUsage(const std::string deviceAddress, const bool profileUsageInfo,
         const BTProfileUsageType profileUsageType = PROFILE_USAGE_ACTIVE_CALL) = 0;

   /**
    * This function is used to call the SetBTProfilesUsage method of the BM Application interface
    * in order to set the HFP profile usage.
    *
    * @param[in]
    *            uint8 deviceHandle
    *             - Device handle of the device for which the HFP profile usage needs to be set.
    *            bool profileUsageInfo
    *             - Value to be set for the HFP profile usage. If TRUE- HFP profile usage is done, else not.
    *            BTProfileUsageType profileUsageType
    *             - HFP profile usage type and it is of ENUM type.
    *
    * @param[out] act_t - Asynchronous Callback Token received from the ASF FW.
    *
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void setBTProfilesUsage(const uint8 deviceHandle, const bool profileUsageInfo,
         const BTProfileUsageType profileUsageType = PROFILE_USAGE_ACTIVE_CALL) = 0;

   /**
    * On receiving Vehicle BT address status, this function is called to register DeviceListExtendedProperty.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void registerDeviceListExtendedProperty() = 0;

protected:
   /**
    * Parameterized constructor of IBmAppRequestIf
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   IBmAppRequestIf(bool serviceAvailability){_serviceAvailability = serviceAvailability;}

   /**
    * Once the service is available, this function is called to register the required properties of BM Application.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void registerProperties() = 0;

   /**
    * Once the service becomes unavailable, this function is called to de-register
    * the registered properties of BM Application.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void deregisterProperties() = 0;

   bool _serviceAvailability;
};

} // namespace pmapp
} // namespace bosch
} // namespace com

#endif //IBmAppRequestIf_h
