/**
 * @file DeviceDetailsListHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the MsgTypes between IpcWrapper and CPP layer
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef DeviceDetailsListHandler_h
#define DeviceDetailsListHandler_h

#include "DeviceDetails.h"
#include "ActivePassiveDeviceList.h"
#include "QueryRequestedContact.h"
#include "QueriedContactInfo.h"
#include <map>
#include "PmSingleton.h"
#include "Lock.h"
#include "DialRequestInfo.h"

namespace com
{
namespace bosch
{
namespace pmapp
{

class DeviceDetailsListHandler : public PmSingleton<DeviceDetailsListHandler>
{
public:

   /**
    * This function is used to update Device Details in the vector.
    *
    * @param[in] DeviceDetails& - Reference to DeviceDetails
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void updateDeviceDetails(const DeviceDetails& deviceDetails);

   /**
    * This function is used to update Device Details in the vector.
    *
    * @param[in] BdAddress& deviceAddress - Reference to device address
    *            DeviceConnectionStatus deviceConnectionStatus - device connection status to be updated
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool updateDeviceConnectionStatus(const pmcore::BdAddress& deviceAddress,
         const DeviceConnectionStatus deviceConnectionStatus);

   /**
    * This function is used to update Device Identification detail in the vector.
    *
    * @param[in] BdAddress& deviceAddress - Reference to device address
    *            DeviceIdentification& deviceIdentification - device Identification info to be updated
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool updateDeviceIdentification(const pmcore::BdAddress& deviceAddress,
         const pmcore::DeviceIdentification& deviceIdentification);

   /**
    * This function is used to set the device role for the specific device from the ActivePassiveDeviceList map
    *
    * @param[in] ActivePassiveDeviceListMap& activePassiveDeviceList - Reference to ActivePassiveDeviceListMap
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void setDeviceRole(const pmcore::ActivePassiveDeviceListMap& activePassiveDeviceList);

   /**
    * This function is used to remove the device and the corresponding Details from the vector and fill the
    * deleted device address- "deletedDeviceAddress"
    *
    * @param[in] DeviceHandle& deletedDeviceHandle - Reference to Device handle of the deleted device
    *            BdAddress& deletedDeviceAddress - Device address of the deleted device to be filled
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool removeDevice(const pmcore::DeviceHandle deletedDeviceHandle, pmcore::BdAddress& deletedDeviceAddress);

   /**
    * This function is used to remove the device and the corresponding Details from the vector.
    *
    * @param[in] std::string& deletedDeviceAddress - Reference to Device address of the deleted device
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool removeDevice(const pmcore::BdAddress& deletedDeviceAddress);

   /**
    * This function is used to get the Device details from the vector
    *
    * @param[in] BdAddress& deviceAddress - BT address of the device for which the Device details is needed
    * @param[out]
    * @param[in,out] DeviceDetails& deviceDetails - Reference to DeviceDetails
    *
    * @return True if the device is present. Else false.
    *
    */
   bool getDeviceDetails(const pmcore::BdAddress& deviceAddress, DeviceDetails& deviceDetails);

   /**
    * This function is used to get the Device details from the vector
    *
    * @param[in] DeviceHandle& deviceHandle - Device handle of the device for which the Device details is needed
    * @param[out]
    * @param[in,out] DeviceDetails& deviceDetails - Reference to DeviceDetails
    *
    * @return True if the device is present. Else false.
    *
    */
   bool getDeviceDetails(const pmcore::DeviceHandle& deviceHandle, DeviceDetails& deviceDetails);

   /**
    * This function is used to get the Device handle of the given device address
    *
    * @param[in] std::string deviceAddress - BT-Address of the device
    *            DeviceHandle& deviceHandle - Reference variable to be filled
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void getDeviceHandle(const pmcore::BdAddress& deviceAddress, pmcore::DeviceHandle& deviceHandle);

   /**
    * This function is used to get the Device address of the given device handle
    *
    * @param[in] uint8 deviceHandle - deviceHandle of the device
    *            BdAddress& deviceAddress - Reference variable to filled
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void getDeviceAddress(const pmcore::DeviceHandle deviceHandle, pmcore::BdAddress& deviceAddress);

   /**
    * This function is used to print all the information from DeviceDetailsList.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void printDeviceDetailsList();

   /**
    * This function is used to set the given device as the Active device.
    *
    * @param[in] std::vector<std::string>& deviceAddressList - List of device addresses which should be made Active
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void setActiveDevices(const pmcore::BdAddressList& deviceAddressList);

   /**
    * This function is used to get the list of Active devices
    *
    * @param[in]
    * @param[out]  std::vector<std::string>& deviceAddressList - List of device addresses which are Active
    * @param[in,out]
    *
    * @return
    */
   void getActiveDevices(pmcore::BdAddressList& deviceAddressList);

   /**
    * This function is used to get the address list of all the Active devices.
    *
    * @param[in] BdAddressList& deviceAddressList - List of device addresses which are Active
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void getActiveDeviceAddressList(pmcore::BdAddressList& deviceAddressList);

   /**
    * This function is used to get the device role of a device from the list.
    *
    * @param[in] pmcore::BdAddress& deviceAddress - Device address
    *            DeviceRole& deviceRole - Reference to "DeviceRole"- Role of the device
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void getDeviceRole(const pmcore::BdAddress& deviceAddress, pmcore::DeviceRole& deviceRole);

   /**
    * This function is used to get the first Active device from the list of Active devices in the list
    *
    * @param[in]
    * @param[out] std::map<std::string, unsigned char>& activePassiveDeviceListMap -
    *             Reference to the list of Active and passive devices.
    * @param[in,out]
    *
    * @return
    */
   void getActiveDeviceAddress(pmcore::BdAddress& deviceAddress);

   /**
    * This function is used to update the call status list received from PM Core.
    * This function requests for searchPhonebook() from Phonebook Client handler.
    *
    * And finally on receiving the response for all searchPhonebook requests,
    * the call status is updated to the clients along with the contact details.
    *
    * @param[in]  pmcore::CallStatusList& callStatusList - call status list received from PM core
    *             pmcore::ActType act - Act should be used while updating the calls Report
    *                   after search phonebook response
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void updateCallStatusListFromPmCore(const pmcore::CallStatusList& callStatusList,
         const pmcore::ActType act = pmcore::PM_DEFAULT_ACT);

   /**
    * This function is called from the Phonebook client handler when the searchPhonebook response
    * is received.
    *
    * @param[in]  act_t act - act which was received when the request was posted
    *             QueriedContactInfo& queriedContactInfo - received contact details
    *             PhonebookErrorResponseEnum phonebookErrorResponse - Response enum from Phonebook
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void queryContactInfoResponse(const act_t act,
         const QueriedContactInfo& queriedContactInfo, const PhonebookErrorResponseEnum phonebookErrorResponse);

   /**
    * This function is used to set the contact info which are received from the clients.
    * eg: when dial, HMI shall set the contact details.
    *
    * @param[in]  BdAddress& deviceAddress - Device address
    *             TelephoneNumber& telephoneNumber - Telephone number for which the contact is set
    *             pmcore::ActType act - Act that was set for the request by PM Application
    * @param[out]
    * @param[in,out]    QueriedContactInfo queriedContactInfo - Contact info set by the client when setInfo is true
    *                                                         - Contact info to be set when setInfo is false
    *
    * @return
    */
   void setContactInfoFromClients(const pmcore::BdAddress& deviceAddress,
         const pmcore::TelephoneNumber& telephoneNumber, const pmcore::ActType act,
         QueriedContactInfo& queriedContactInfo);

   /**
    * This function is used to reset the contact info in the STL "_contactInfoFromClients" by act
    * which was received from the clients
    * eg: when dial, HMI shall set the contact details.
    *
    * @param[in]  BdAddress& deviceAddress - Device address
    *             pmcore::ActType act - Act that was set during the request by PM Application
    * @param[out] QueriedContactInfo queriedContactInfo - Contact info set by the client
    *             TelephoneNumber& telephoneNumber - Telephone number for which the contact is set
    * @param[in,out]
    * @return  true- operation is success; false- the operation is failure
    */
   bool resetContactInfoFromClientsByAct(const pmcore::BdAddress& deviceAddress,
         const pmcore::ActType act, pmcore::TelephoneNumber& telephoneNumber,
         QueriedContactInfo& queriedContactInfo);

   /**
    * This function is used to reset the contact info in the STL "_contactInfoFromClients" by telephoneNumber
    * which was received from the clients
    * eg: when dial, HMI shall set the contact details.
    *
    * @param[in]  BdAddress& deviceAddress - Device address
    *             TelephoneNumber& telephoneNumber - Telephone number for which the contact is set
    * @param[out] QueriedContactInfo queriedContactInfo - Contact info set by the client
    *             ActType act - Act that was set during the request by PM Application
    * @param[in,out]
    * @return  true- operation is success; false- the operation is failure
    */
   bool resetContactInfoFromClientsByTelephoneNumber(const pmcore::BdAddress& deviceAddress,
         const pmcore::TelephoneNumber& telephoneNumber, pmcore::ActType& act,
         QueriedContactInfo& queriedContactInfo);

   /**
    * This function is used to post the searchPhoneBook once PhoneBook component available.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void checkPendingQueryContactsList();

private:

   friend class PmSingleton<DeviceDetailsListHandler>;

   /**
    * Constructor of DeviceDetailsListHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   DeviceDetailsListHandler();

   /**
    * Destructor of DeviceDetailsListHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ~DeviceDetailsListHandler();

   /**
    * This function is used to insert the Device and the corresponding Details into the vector.
    *
    * @param[in] DeviceDetails& - Reference to DeviceDetails
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void insertDeviceDetails(const DeviceDetails& deviceDetails);

   /**
    * This function is used to update the call status of a specific device
    *
    * @param[in]  pmcore::BdAddress& deviceAddress - Device address for which the call status
    *             need to be updated
    *             pmcore::CallStatus& callStatus - updated call status from PM core
    * @param[out] bool &isQueryContactRequested  -  true: if a query contact is requested;
    *                                               false: query contact is not requested
    * @param[in,out]
    *
    * @return
    */
   void updateDeviceCallStatus(const pmcore::BdAddress& deviceAddress,
         const pmcore::CallStatus& callStatus, bool &isQueryContactRequested);

   /**
    * This function is used to clear the call report of the removed device in the call status list received from
    * PM Core
    *
    * @param[in]  callStatusList - Call status list received from PM Core
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void clearCallReportOfRemovedDevice(const pmcore::CallStatusList& callStatusList);

   /**
    * This function is used to send the call status to the clients
    *
    * @param[in]  pmcore::ActType act - The act for the Get call status request
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void sendCallsReportList(const pmcore::ActType act = pmcore::PM_DEFAULT_ACT);

   /**
    * This function is used to request for contact details from Phonebook component
    * and add the corresponding key and value into "_pendingQueryContactsList"
    *
    * @param[in]  pmcore::BdAddress& deviceAddress - Device address
    *             TelephoneNumber& telephoneNumber - TelephoneNumber for which the contact info is needed
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void requestQueryContactInfo(const pmcore::BdAddress& deviceAddress,
         const pmcore::TelephoneNumber& telephoneNumber, bool& isQueryContactInfoSuccess);

   /**
    * This function is called to update the received contact details in the list "_deviceDetailsList"
    *
    * @param[in]  act_t act - act which was received when the request was posted
    *             QueriedContactInfo& queriedContactInfo - received contact details
    * @param[out] bool &sendUpdateToClients - If some other contact is also waiting for the queried info from PB,
    *             then the recently updated contact info should not be sent to the clients. And wait for the
    *             rest of the contact info to be retrieved from PB for the same device
    * @param[in,out]
    *
    * @return true - the queried contact is updated in the "_deviceDetailsList";false- the update is not done
    */
   bool updateQueriedContactInfo(const act_t act, const QueriedContactInfo& queriedContactInfo,
         bool &sendUpdateToClients);

   /**
    * This function is used to clear the device specific details in all the resources(STL) that are maintained
    * by this class.
    *
    * @param[in]  BdAddress& deviceAddress - Device address for which the resources should be cleared
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void clearDeviceInfo(const pmcore::BdAddress& deviceAddress);

   /**
    * This function is used to print the call attributes list
    *
    * @param[in] CallAttributesList& callAttributesList - Reference to CallAttributesList
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void printCallAttributesList(const CallAttributesList& callAttributesList);

   /**
    * This function is used to print the "_pendingQueryContactsList"
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void printPendingQueryContactsList();

   LockForever _deviceDetailsListLock;

   // This vector is used to keep track of active devices list. So that only the change shall be updated
   pmcore::BdAddressList _activeDevicesList;

   static DeviceDetailsList _deviceDetailsList;

   /*
    * The telephone number and its corresponding info are added to this list after posting
    * searchPhonebook request
    */
   static std::map<act_t, QueryRequestedContact> _pendingQueryContactsList;

   /*
    * A pair is pushed into this map when the clients such as HMI itself provides contact details
    */
   static std::map<DialRequestInfo, QueriedContactInfo> _contactInfoFromClients;
};

} // namespace pmapp
} // namespace bosch
} // namespace com

#endif // #define DeviceDetailsListHandler_h
