/**
 * @file CcaFunction.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaFunction class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This class is the base class for all the CCA related methods
 * and properties. The CCA message handlers has to inherit this class
 *
 */

#include "CcaFunction.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaFunction.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

#include "generic_msgs_fw.h"

#define SOURCE_APP_ID_OFFSET_POS 68719476736
#define COMMAND_COUNTER_OFFSET_POS 1048576
#define FUNCTION_ID_OFFSET_POS 16

std::map<tU64, CcaMsgHeader> CcaFunction::_ccaRequestMap;

CcaFunction::CcaFunction(ahl_tclBaseOneThreadService* pService)
:  _pService(pService)
{
   ETG_TRACE_USR4(("CcaFunction::CcaFunction entered"));
}

void CcaFunction::sendMessage(const CcaMsgHeader& msgHeaderInfo, const fi_tclTypeBase& oOutData)
{
   ETG_TRACE_USR4(("CcaFunction::sendMessage entered"));

   fi_tclVisitorMessage oOutVisitorMsg(oOutData, getMajorVersion());

   /* Set the CCA message information */
   oOutVisitorMsg.vInitServiceData(
         _pService->u16GetAppID(),           /**< Source app-ID */
         msgHeaderInfo._u16DestAppID,         /**< Dest. app-ID */
         AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  /**< stream type*/
         0,                                  /**< stream counter*/
         msgHeaderInfo._u16RegisterId,        /**< Registry ID */
         msgHeaderInfo._u16CmdCtr,            /**< Command counter */
         _pService->u16GetServiceID(),       /**< Service ID */
         msgHeaderInfo._u16Fid,               /**< Function-ID */
         msgHeaderInfo._u8OpCode,             /**< OpCode */
         0,                                  /**< ACT */
         0,                                  /**< Source sub-ID */
         0);                                 /**< Dest. sub-ID */

   ail_tenCommunicationError communicationResult = _pService->ePostMessage(&oOutVisitorMsg);

   ETG_TRACE_USR4(("CcaFunction::sendMessage status: %u", ETG_CENUM(ail_tenCommunicationError, communicationResult)));
}

void CcaFunction::sendEmptyMessage(const CcaMsgHeader& msgHeaderInfo)
{
   ETG_TRACE_USR4(("CcaFunction::sendEmptyMessage entered"));

   gm_tclEmptyMessage oOutMessage(
         _pService->u16GetAppID(),
         msgHeaderInfo._u16DestAppID,
         msgHeaderInfo._u16RegisterId,
         msgHeaderInfo._u16CmdCtr,
         _pService->u16GetServiceID(),
         msgHeaderInfo._u16Fid,
         msgHeaderInfo._u8OpCode);

   _pService->ePostMessage(&oOutMessage);
}

void CcaFunction::getDataFromAmt(amt_tclServiceData* pFIMsg, fi_tclTypeBase& oFIData)
{
   ETG_TRACE_USR4(("CcaFunction::getDataFromAmt entered"));

   fi_tclVisitorMessage oInVisitorMsg(pFIMsg);

   //FIXME:OSAL-Generic Macro has to defined later for -1
   if (oInVisitorMsg.s32GetData(oFIData, getMajorVersion()) == -1)
   {
      ETG_TRACE_ERR(("CcaFunction::getDataFromAmt error in s32GetData"));
   }
}

void CcaFunction::sendErrorMessage(const CcaMsgHeader& msgHeaderInfo, tU16 u16ErrorCode)
{
   ETG_TRACE_USR4(("CcaFunction::sendErrorMessage entered"));

   gm_tclU16Message oSendErrorMessage(
         _pService->u16GetAppID(),
         msgHeaderInfo._u16DestAppID,
         msgHeaderInfo._u16RegisterId,
         msgHeaderInfo._u16CmdCtr,
         _pService->u16GetServiceID(),
         msgHeaderInfo._u16Fid,
         AMT_C_U8_CCAMSG_OPCODE_ERROR);

   oSendErrorMessage.vSetWord(u16ErrorCode);

   _pService->ePostMessage(&oSendErrorMessage);
}

tU16 CcaFunction::getMajorVersion()
{
   ETG_TRACE_USR4(("CcaFunction::getMajorVersion entered"));

   tU16 major = 0;
   tU16 minor = 0;
   tU16 patch = 0;

   _pService->bGetServiceVersion(_pService->u16GetServiceID(), major, minor, patch);

   return major;
}

void CcaFunction::retrieveMessageHeader(const amt_tclServiceData* pFIMsg, CcaMsgHeader& msgHeaderInfo)
{
   ETG_TRACE_USR4(("CcaFunction::retrieveMessageHeader entered"));

   msgHeaderInfo._u16CmdCtr = pFIMsg->u16GetCmdCounter();
   msgHeaderInfo._u16DestAppID = pFIMsg->u16GetSourceAppID();
   msgHeaderInfo._u16Fid = pFIMsg->u16GetFunctionID();
   msgHeaderInfo._u16RegisterId = pFIMsg->u16GetRegisterID();
   msgHeaderInfo._u8OpCode = pFIMsg->u8GetOpCode();
}

bool CcaFunction::retrieveMessageHeader(const tU64 ccaToken, CcaMsgHeader& msgHeaderInfo)
{
   ETG_TRACE_USR4(("CcaFunction::retrieveMessageHeader entered"));
   bool entryFoundInRequestMap = false;

   std::map<tU64, CcaMsgHeader>::const_iterator ccaReqMapIter = _ccaRequestMap.find(ccaToken);

   if(ccaReqMapIter != _ccaRequestMap.end())
   {
      msgHeaderInfo = ccaReqMapIter->second;
      entryFoundInRequestMap = true;
   }

   return entryFoundInRequestMap;
}

void CcaFunction::generateToken(amt_tclServiceData* inMsg, tU64& ipcToken, CcaMsgHeader& msgHeader)
{
   /**
    * @details: The different parameters that shall be considered for generating the token for a CCA request shall be:
    * SourceAppID, CmdCounter, FunctionID, RequestType [CCA].
    *
    * Consider a CCA request of the input parameters as:
    * SourceAppID = 0xAAAA, CmdCounter = 0xFFFF, FunctionID = 0xF0A, RequestType = CCA = 1, then the generated token is
    * ==>000 AAAA FFFF 0F0A 1
    *
    * Below briefed the significance of each bit:
    *
    * Bit 0-3:
    * This range represents the Request Type, this represents the IPC message request type like CCA, ASF-Dbus, etc.
    * Each request type is assigned with a unique value: 1 - CCA, 2 - ASF-DBus.
    *
    * Bit 4-19:
    * This represents the CCA Function ID: It is of type - tU16, the maximum value this shall hold is FFFF
    *
    * Bit 20-35:
    * This signifies the Command counter, The value is set by the client and is of type - tU16, the maximum
    * value this shall hold is FFFF
    *
    * Bit 36-51:
    * These signifies the Source App Id, It is of type - tU16, the maximum value this shall hold is FFFF
    *
    * Bit 52-63:
    * 000 - Reserved
    *
    * Based on the above parameters the token is generated.
    *
    */

   retrieveMessageHeader(inMsg, msgHeader);

   // The _u16DestAppID -> Refers to the App ID of the client
   ipcToken = ((tU64)msgHeader._u16DestAppID * SOURCE_APP_ID_OFFSET_POS)
         + ((tU64)msgHeader._u16CmdCtr * COMMAND_COUNTER_OFFSET_POS)
         + ((tU64)msgHeader._u16Fid * FUNCTION_ID_OFFSET_POS)
         + IpcRequestType::CCA;

   ETG_TRACE_USR4(("Generated token for CCA request is: %u", ipcToken));

   // Insert the request into the the CCA request map.
   // Insertion fails if the key(ipcToken) already exists in the map and the return bool type will be false.
   std::pair<std::map<tU64, CcaMsgHeader>::iterator, bool> insertReturnType;
   insertReturnType = _ccaRequestMap.insert(std::pair<tU64, CcaMsgHeader>(ipcToken, msgHeader));

   if (false == insertReturnType.second)
   {
      ETG_TRACE_USR4(("ipcToken '%u' is already present in the map", ipcToken));
      ipcToken = 0;
   }
   printCcaRequestMap();
}

void CcaFunction::printCcaRequestMap()
{
   ETG_TRACE_USR4(("CcaFunction::printCcaRequestMap Entered"));

   for (auto& iter: _ccaRequestMap)
   {
      ETG_TRACE_USR4(("_ccaRequestMap.key : %u", iter.first));

      ETG_TRACE_USR4(("_u16DestAppID: %u", iter.second._u16DestAppID));
      ETG_TRACE_USR4(("_u16CmdCtr: %u", iter.second._u16CmdCtr));
      ETG_TRACE_USR4(("_u16RegisterId: %u", iter.second._u16RegisterId));
      ETG_TRACE_USR4(("_u16Fid: %u", iter.second._u16Fid));
      ETG_TRACE_USR4(("_u8OpCode: %u", iter.second._u8OpCode));
   }
}

void CcaFunction::removeEntryFromRequestMap(tU64 ccaToken, bool allEntries)
{
   ETG_TRACE_USR4(("CcaFunction::removeEntryFromRequestMap Token ID to be removed: %u", ccaToken));

   if(true == allEntries)
   {
      _ccaRequestMap.clear();
   }
   else
   {
      if(0 == _ccaRequestMap.erase(ccaToken))
      {
         ETG_TRACE_ERR(("CcaFunction::removeEntryFromRequestMap Invalid Token ID"));
      }
   }
   printCcaRequestMap();
}

CcaFunction::~CcaFunction()
{
   ETG_TRACE_USR4(("CcaFunction::~CcaFunction entered"));

   removeEntryFromRequestMap(0, true);

   _pService = nullptr;
}
