/**
 * @file CcaVoiceRecognitionExtendedHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaVoiceRecognitionExtendedHandler class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaVoiceRecognitionExtendedHandler.h"
#include "IpcWrapper.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaVoiceRecognitionExtendedHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

CcaVoiceRecognitionExtendedHandler::CcaVoiceRecognitionExtendedHandler(ahl_tclBaseOneThreadService* pAhlService)
: CcaProperty(pAhlService)
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::CcaVoiceRecognitionExtendedHandler entered"));
}

CcaVoiceRecognitionExtendedHandler::~CcaVoiceRecognitionExtendedHandler()
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::~CcaVoiceRecognitionExtendedHandler entered"));

   _vrExtendedStatusInfoMap.clear();
}

void CcaVoiceRecognitionExtendedHandler::onOpcodeGet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::onOpcodeGet entered"));
   (void) pInMsg;

   BdAddress deviceAddress = "";
   ::com::bosch::pmapp::DeviceDetailsListHandler::getInstance().getActiveDeviceAddress(deviceAddress);
   PmResult pmresult = pm_ipc_wrapper::IpcWrapper::getInstance().getVoiceRecognitionStatus(deviceAddress, act);

   if(PM_RESULT_OK != pmresult._pmResultCode)
   {
      most_telfi_tclMsgBTDeviceVoiceRecognitionExtendedStatus oBTDeviceVoiceRecExtendedStatus;

      oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle = PM_DEVICEHANDLE_ZERO;
      oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive = false ;

      // TODO: Need to check as of how to fill these two variables
      oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = false;
      oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState.enType =
            (most_fi_tcl_e8_TelSiriAvailabilityState::tenType::FI_EN_E8NOT_AVAILABLE);

      updateOpcodeStatus(act, oBTDeviceVoiceRecExtendedStatus, PM_PROPERTYID_BTDEVICEVOICERECOGNITIONEXTENDED);
      oBTDeviceVoiceRecExtendedStatus.vDestroy();
   }

}

void CcaVoiceRecognitionExtendedHandler::onOpcodeSet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::onOpcodeSet entered"));

   most_telfi_tclMsgBTDeviceVoiceRecognitionExtendedPureSet voiceRecognitionExtendedPureSet;

   getDataFromAmt(pInMsg, voiceRecognitionExtendedPureSet);

   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler:: BTDeviceVoiceRecActive : %u",
         voiceRecognitionExtendedPureSet.bBTDeviceVoiceRecActive));

   DeviceHandle receivedVRDevHandle = voiceRecognitionExtendedPureSet.u8DeviceHandle;
   ETG_TRACE_USR3((" receivedVRDevHandle: %u", receivedVRDevHandle));

   BdAddress deviceAddress = "";
   com::bosch::pmapp::DeviceDetailsListHandler::getInstance().getDeviceAddress(receivedVRDevHandle, deviceAddress);
   StartStop startStopVR = voiceRecognitionExtendedPureSet.bBTDeviceVoiceRecActive;

   PmResult pmresult = pm_ipc_wrapper::IpcWrapper::getInstance().postStartStopVoiceRecognitionRequest2PmCore(
         deviceAddress, startStopVR, act);

   if(PM_RESULT_OK != pmresult._pmResultCode)
   {
      ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::onOpcodeSet failed"));
      PmCoreResponseData pmCoreResponseData(pmresult, act);
      processOpcodeSetError(&pmCoreResponseData);
   }

   voiceRecognitionExtendedPureSet.vDestroy();
}

void CcaVoiceRecognitionExtendedHandler::onPropertyUpdate(PropertyUpdate* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::onPropertyUpdate entered"));
   dispatchProperty(propertyUpdate);
}

void CcaVoiceRecognitionExtendedHandler::processOpcodeGetResponse(PmCoreResponseData* responseMsg)
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::processOpcodeGetResponse entered"));
   dispatchProperty(responseMsg);
}

template<typename TPropertyType>
void CcaVoiceRecognitionExtendedHandler::dispatchProperty(TPropertyType* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::dispatchProperty entered"));

   VoiceRecognitionPropertyUpdate* voiceRecognitionPropertyUpdate =
         static_cast<VoiceRecognitionPropertyUpdate*> (propertyUpdate);

   most_telfi_tclMsgBTDeviceVoiceRecognitionExtendedStatus oBTDeviceVoiceRecExtendedStatus;

   DeviceHandle deviceHandle = PM_DEVICEHANDLE_ZERO;
   ::com::bosch::pmapp::DeviceDetailsListHandler::getInstance().getDeviceHandle
         (voiceRecognitionPropertyUpdate->_deviceAddress, deviceHandle);

   oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle = deviceHandle;
   oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive = false ;
   oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = false;
   oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState.enType =
         (most_fi_tcl_e8_TelSiriAvailabilityState::tenType::FI_EN_E8NOT_AVAILABLE);

   auto iter = _vrExtendedStatusInfoMap.find(deviceHandle);

   switch(voiceRecognitionPropertyUpdate->_updatedFieldMarker)
   {
      case VR_AVAILABILITY_STATUS:
      {
         if (PM_DEVICEHANDLE_ZERO != deviceHandle)
         {
            if (voiceRecognitionPropertyUpdate->_vrAvailabilityStatus)
            {
               _vrExtendedStatusInfoMap.emplace_hint(_vrExtendedStatusInfoMap.end(), deviceHandle,
                     EnhancedVoiceRecognitionFeature());

               oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = true;
            }
            else
            {
               if (_vrExtendedStatusInfoMap.end() != iter)
               {
                  _vrExtendedStatusInfoMap.erase(iter);
               }
            }
         }
      }
      break;

      case ENHANCEDVOICERECOGNITION_FEATURE_STATUS:
      {
         if (_vrExtendedStatusInfoMap.end() != iter)
         {
            oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = true;
            iter->second = voiceRecognitionPropertyUpdate->_enhancedVoiceRecognitionFeature;

            if ("Siri" == iter->second._enhancedVRFeature)
            {
               // Filling the SIRI availability status
               most_fi_tcl_e8_TelSiriAvailabilityState e8SiriAvailabilityState;
               setEnhancedVRAvailabilityStatusForCCAMost(iter->second, e8SiriAvailabilityState);
               oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState = e8SiriAvailabilityState;
            }
            else
            {
               ETG_TRACE_USR4(("Wrt CCA if it is NON siri, the update is not needed as of now"));
               oBTDeviceVoiceRecExtendedStatus.vDestroy();
               return;
            }
         }
      }
      break;

      case VOICERECOGNITION_STATUS:
      {
         if (_vrExtendedStatusInfoMap.end() != iter)
         {
            if (DEFAULT_ACT == voiceRecognitionPropertyUpdate->_u64TokenId)
            {
               ETG_TRACE_USR4(("This is an update from PM core"));

               VRStatus vrStatus = voiceRecognitionPropertyUpdate->_voiceRecognitionStatus._vrStatus;

               if ((VR_SESSION_STARTING == vrStatus) || (VR_SESSION_ENDING == vrStatus))
               {
                  ETG_TRACE_USR4(("Ignoring the intermediate VR status"));

                  oBTDeviceVoiceRecExtendedStatus.vDestroy();
                  return;
               }
            }

            oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported = true;

            // Filling the SIRI availability status
            most_fi_tcl_e8_TelSiriAvailabilityState e8SiriAvailabilityState;
            setEnhancedVRAvailabilityStatusForCCAMost(iter->second, e8SiriAvailabilityState);
            oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState = e8SiriAvailabilityState;

            if (VR_SESSION_ACTIVE == voiceRecognitionPropertyUpdate->_voiceRecognitionStatus._vrStatus)
            {
               oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive = true;
            }
         }
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("Default case"));
      }
   }

   ETG_TRACE_USR3(( "oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported= '%u'",
         oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecSupported));
   ETG_TRACE_USR3(( "oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive= '%u'",
         oBTDeviceVoiceRecExtendedStatus.bBTDeviceVoiceRecActive));
   ETG_TRACE_USR3(("oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState: '%u'",
         (oBTDeviceVoiceRecExtendedStatus.e8SiriAvailabilityState.enType)));
   ETG_TRACE_USR3(( "oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle= '%u'",
         oBTDeviceVoiceRecExtendedStatus.u8DeviceHandle));

   updateOpcodeStatus(voiceRecognitionPropertyUpdate->_u64TokenId, oBTDeviceVoiceRecExtendedStatus,
         voiceRecognitionPropertyUpdate->u16GetFunctionID());
   oBTDeviceVoiceRecExtendedStatus.vDestroy();
}

void CcaVoiceRecognitionExtendedHandler::processOpcodeSetError(PmCoreResponseData* responseMsg)
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::processOpcodeSetError entered"));

   most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode;

//   ETG_TRACE_USR3((" most_fi_tcl_e8_ErrorCode::FI_EN_BUSY"));
//   e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_BUSY;

   ETG_TRACE_USR3((" most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE"));
   e8ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;

   most_telfi_tclMsgBTDeviceVoiceRecognitionExtendedError oBTDeviceVoiceRecognitionExtME;
   oBTDeviceVoiceRecognitionExtME.e8ErrorCode.enType = e8ErrorCode;

   updateOpcodeStatus(responseMsg->_u64TokenId, oBTDeviceVoiceRecognitionExtME);
}

void CcaVoiceRecognitionExtendedHandler::setEnhancedVRAvailabilityStatusForCCAMost
(const EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature,
      most_fi_tcl_e8_TelSiriAvailabilityState& e8SiriAvailabilityState)
{
   ETG_TRACE_USR4(("CcaVoiceRecognitionExtendedHandler::setEnhancedVRAvailabilityStatusForCCAMost entered"));

   e8SiriAvailabilityState.enType =
            (most_fi_tcl_e8_TelSiriAvailabilityState::tenType::FI_EN_E8NOT_AVAILABLE);

      if ("Siri" == enhancedVoiceRecognitionFeature._enhancedVRFeature)
      {
         switch (enhancedVoiceRecognitionFeature._enhancedVRAvailability)
         {
            case ENHANCED_VR_ENABLED:
            {
               e8SiriAvailabilityState.enType =
                     (most_fi_tcl_e8_TelSiriAvailabilityState::tenType::FI_EN_E8AVAILABLE_ENABLED);
            }
            break;
            case ENHANCED_VR_DISABLED:
            {
               e8SiriAvailabilityState.enType =
                     (most_fi_tcl_e8_TelSiriAvailabilityState::tenType::FI_EN_E8AVAILABLE_DISABLED);
            }
            break;

            default:
            {
               ETG_TRACE_USR4(("Default case"));
            }
         }
      }


}
