/**
 * @file CcaCallStatusNoticeHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaCallStatusNoticeHandler class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaCallStatusNoticeHandler.h"
#include "IpcWrapper.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaCallStatusNoticeHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

CcaCallStatusNoticeHandler::CcaCallStatusNoticeHandler(ahl_tclBaseOneThreadService* pAhlService)
: CcaProperty(pAhlService)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeHandler::CcaCallStatusNoticeHandler entered"));
}

void CcaCallStatusNoticeHandler::onOpcodeGet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeHandler::onOpcodeGet entered"));
   (void) pInMsg;

   PmResult pmresult = pm_ipc_wrapper::IpcWrapper::getInstance().getCallStatusList(act);

   if(PM_RESULT_OK != pmresult._pmResultCode)
   {
      most_telfi_tclMsgCallStatusNoticeStatus oCallStatusNotice;
      updateOpcodeStatus(act, oCallStatusNotice, PM_PROPERTYID_CALLSTATUSNOTICE);
      oCallStatusNotice.vDestroy();
   }
}

void CcaCallStatusNoticeHandler::onOpcodeSet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_ERR(("Received an invalid opcode - SET for MOST_TELFI_C_U16_CALLSTATUSNOTICE, FID: %u",
         pInMsg->u16GetFunctionID()));

   // Set is not possible, Hence removing the token from the table.
   removeEntryFromRequestMap(act);
}

void CcaCallStatusNoticeHandler::onPropertyUpdate(PropertyUpdate* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeHandler::onPropertyUpdate entered"));
   dispatchProperty(propertyUpdate);
}

void CcaCallStatusNoticeHandler::processOpcodeGetResponse(PmCoreResponseData* responseMsg)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeHandler::processOpcodeGetResponse entered"));
   dispatchProperty(responseMsg);
}

template<typename TPropertyType>
void CcaCallStatusNoticeHandler::dispatchProperty(TPropertyType* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeHandler::dispatchProperty entered"));

   CallsReportListPropertyUpdate* callsReportListPropertyUpdate =
         static_cast<CallsReportListPropertyUpdate*> (propertyUpdate);

   com::bosch::pmapp::CallsReportList& callsReportList = callsReportListPropertyUpdate->_callsReportList;

   CcaCallStatusList ccaCallStatusListFull, ccaCallStatusList;
   CcaCallStatusListHandler::getInstance().onCallsReportListChanged(callsReportList, ccaCallStatusListFull);

   DeviceHandle activeDeviceHandle = CcaCallStatusListHandler::getInstance().getActiveDeviceHandle();

   if (MAX_DEVICE_CALL_INSTANCES_COUNT_FOR_CCA < ccaCallStatusListFull.size())
   {
      // More than 2 call instances are present. Hence only Active device call status should be updated
      for (auto& iter : ccaCallStatusListFull)
      {
         if (iter._deviceHandle == activeDeviceHandle)
         {
            ccaCallStatusList.emplace_back(iter);
         }
      }
   }
   else
   {
      ccaCallStatusList = ccaCallStatusListFull;
   }

   most_telfi_tclMsgCallStatusNoticeStatus oCallStatusNotice;

   CallInstance callInstance = 0x00;

   //IMPORTANT:
   //When there are no calls currently active, the Status response of this property cannot be an empty stream as an
   //empty payload MHP message is incompatible with the current implementation of NetServices. Instead, when no calls
   //are active, two empty streams are updated

   for (auto& ccaCallStatusListIter : ccaCallStatusList)
   {
      most_fi_tcl_TelCallStatusNoticeStreamItem oStreamItem;

      oStreamItem.u16CallInstance = (static_cast<tU16> (callInstance++));
      oStreamItem.u8DeviceHandle = ccaCallStatusListIter._deviceHandle;

      oStreamItem.u8CallDurationHr = ccaCallStatusListIter._callAttributes._durationHr;
      oStreamItem.u8CallDurationMin = ccaCallStatusListIter._callAttributes._durationMin;
      oStreamItem.u8CallDurationSec = ccaCallStatusListIter._callAttributes._durationSec;

      oStreamItem.bUsingVehicleAudio = ccaCallStatusListIter._audioActiveStatus;

      most_fi_tcl_e8_TelCallStatus::tenType mostCallState;

      if ((ccaCallStatusListIter._multiparty) && ("Active" == ccaCallStatusListIter._callAttributes._state))
      {
         mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8CONFERENCE;
      }
      else
      {
         if ("Active" == ccaCallStatusListIter._callAttributes._state)
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8ACTIVE;
         }
         else if ("Held" == ccaCallStatusListIter._callAttributes._state)
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8ON_HOLD;
         }
         else if (("Dialing" == ccaCallStatusListIter._callAttributes._state) ||
               ("Alerting" == ccaCallStatusListIter._callAttributes._state))
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8DIALING;
         }
         else if ("Incoming" == ccaCallStatusListIter._callAttributes._state)
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8RINGTONE;
         }
         else if ("Waiting" == ccaCallStatusListIter._callAttributes._state)
         {
            //TODO: FI_EN_E8BUSY is changed as FI_EN_E8RINGTONE as per FC_Phone implementation
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8RINGTONE;
         }
         else if ("Disconnecting" == ccaCallStatusListIter._callAttributes._state)
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8DISCONNECTING;
         }
         else
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8IDLE;
         }
      }

      oStreamItem.e8CallStatus.enType = mostCallState;

      ETG_TRACE_USR3(("oCallStatusNoticeStreamItem.u16CallInstance: %u ", oStreamItem.u16CallInstance));
      ETG_TRACE_USR1(("oCallStatusNoticeStreamItem.e8CallStatus.enType: %d",oStreamItem.e8CallStatus.enType));
      ETG_TRACE_USR1(("oCallStatusNoticeStreamItem.bUsingVehicleAudio: %d",oStreamItem.bUsingVehicleAudio));
      ETG_TRACE_USR1(("oCallStatusNoticeStreamItem.u8DeviceHandle: %d",oStreamItem.u8DeviceHandle));

      ETG_TRACE_USR1(("oCallStatusNoticeStreamItem.u8CallDurationHr: %d",oStreamItem.u8CallDurationHr));
      ETG_TRACE_USR1(("oCallStatusNoticeStreamItem.u8CallDurationMin: %d",oStreamItem.u8CallDurationMin));
      ETG_TRACE_USR1(("oCallStatusNoticeStreamItem.u8CallDurationSec: %d",oStreamItem.u8CallDurationSec));

      oCallStatusNotice.oCallStatusNoticeStream.oItems.push_back(oStreamItem);
   }

   updateOpcodeStatus(callsReportListPropertyUpdate->_u64TokenId, oCallStatusNotice,
         callsReportListPropertyUpdate->u16GetFunctionID());
   oCallStatusNotice.vDestroy();
}

void CcaCallStatusNoticeHandler::processOpcodeSetError(PmCoreResponseData* pInMsg)
{
   (void) pInMsg;
}
