/**
 * @file CcaCallStatusNoticeExtendedHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaCallStatusNoticeExtendedHandler class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaCallStatusNoticeExtendedHandler.h"
#include "CcaCallStatusListHandler.h"
#include "IpcWrapper.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaCallStatusNoticeExtendedHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

CcaCallStatusNoticeExtendedHandler::CcaCallStatusNoticeExtendedHandler(ahl_tclBaseOneThreadService* pAhlService)
: CcaProperty(pAhlService)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeExtendedHandler::CcaCallStatusNoticeExtendedHandler entered"));
}

void CcaCallStatusNoticeExtendedHandler::onOpcodeGet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeExtendedHandler::onOpcodeGet entered"));
   (void) pInMsg;

   PmResult pmresult = pm_ipc_wrapper::IpcWrapper::getInstance().getCallStatusList(act);
   if(PM_RESULT_OK != pmresult._pmResultCode)
   {
      most_telfi_tclMsgCallStatusNoticeExtendedStatus oCallStatusNoticeExtended;
      updateOpcodeStatus(act, oCallStatusNoticeExtended, PM_PROPERTYID_CALLSTATUSNOTICEEXTENDED);
      oCallStatusNoticeExtended.vDestroy();
   }
}

void CcaCallStatusNoticeExtendedHandler::onOpcodeSet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_ERR(("Received an invalid opcode - SET for MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED, FID: %u",
         pInMsg->u16GetFunctionID()));

   // Set is not possible, Hence removing the token from the table.
   removeEntryFromRequestMap(act);
}

void CcaCallStatusNoticeExtendedHandler::onPropertyUpdate(PropertyUpdate* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeExtendedHandler::onPropertyUpdate entered"));
   dispatchProperty(propertyUpdate);
}

void CcaCallStatusNoticeExtendedHandler::processOpcodeGetResponse(PmCoreResponseData* responseMsg)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeExtendedHandler::processOpcodeGetResponse entered"));
   dispatchProperty(responseMsg);
}

template<typename TPropertyType>
void CcaCallStatusNoticeExtendedHandler::dispatchProperty(TPropertyType* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaCallStatusNoticeExtendedHandler::dispatchProperty entered"));

   CallsReportListPropertyUpdate* callsReportListPropertyUpdate =
         static_cast<CallsReportListPropertyUpdate*> (propertyUpdate);

   com::bosch::pmapp::CallsReportList& callsReportList = callsReportListPropertyUpdate->_callsReportList;

   CcaCallStatusList ccaCallStatusList;
   CcaCallStatusListHandler::getInstance().onCallsReportListChanged(callsReportList, ccaCallStatusList);

   //IMPORTANT:
   //When there are no calls currently active, the Status response of this property cannot be an empty stream as an
   //empty payload MHP message is incompatible with the current implementation of NetServices. Instead, when no calls
   //are active, two empty streams are updated

   CallInstance callInstance = 0x00;

   most_telfi_tclMsgCallStatusNoticeExtendedStatus oCallStatusNoticeExtended;

   for (auto& ccaCallStatusListIter : ccaCallStatusList)
   {
      most_fi_tcl_TelCallStatusNoticeExtendedStreamItem oStreamItem;

      oStreamItem.u16CallInstance = (static_cast<tU16> (callInstance++));
      oStreamItem.u8DeviceHandle = ccaCallStatusListIter._deviceHandle;

      oStreamItem.u8CallDurationHr = ccaCallStatusListIter._callAttributes._durationHr;
      oStreamItem.u8CallDurationMin = ccaCallStatusListIter._callAttributes._durationMin;
      oStreamItem.u8CallDurationSec = ccaCallStatusListIter._callAttributes._durationSec;

      oStreamItem.bUsingVehicleAudio = ccaCallStatusListIter._audioActiveStatus;

      oStreamItem.sTelephoneNumber.bSet(ccaCallStatusListIter._telephoneNumber.c_str());
      oStreamItem.sName.bSet("");

      // In CCA, the contact name should be updated to the clients only if it is set via DialExtended IF.
      if (ccaCallStatusListIter._callAttributes._isContactProvidedByClient)
      {
         if (ccaCallStatusListIter._callAttributes._lastName.empty())
         {
            oStreamItem.sName.bSet((ccaCallStatusListIter._callAttributes._firstName).c_str());
         }
         else
         {
            // TODO: Preferred sort order to be retrieved from PB if needed in CCA
            oStreamItem.sName.bSet((ccaCallStatusListIter._callAttributes._firstName + " " +
                  ccaCallStatusListIter._callAttributes._lastName).c_str());
         }
      }

      most_fi_tcl_e8_TelCallStatus::tenType mostCallState;

      if ((ccaCallStatusListIter._multiparty) && ("Active" == ccaCallStatusListIter._callAttributes._state))
      {
         mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8CONFERENCE;
      }
      else
      {
         if ("Active" == ccaCallStatusListIter._callAttributes._state)
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8ACTIVE;
         }
         else if ("Held" == ccaCallStatusListIter._callAttributes._state)
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8ON_HOLD;
         }
         else if (("Dialing" == ccaCallStatusListIter._callAttributes._state) ||
               ("Alerting" == ccaCallStatusListIter._callAttributes._state))
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8DIALING;
         }
         else if ("Incoming" == ccaCallStatusListIter._callAttributes._state)
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8RINGTONE;
         }
         else if ("Waiting" == ccaCallStatusListIter._callAttributes._state)
         {
            //TODO: FI_EN_E8BUSY is changed as FI_EN_E8RINGTONE as per FC_Phone implementation
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8RINGTONE;
         }
         else if ("Disconnecting" == ccaCallStatusListIter._callAttributes._state)
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8DISCONNECTING;
         }
         else
         {
            mostCallState = most_fi_tcl_e8_TelCallStatus::tenType::FI_EN_E8IDLE;
         }
      }

      oStreamItem.e8CallStatus.enType = mostCallState;

      ETG_TRACE_USR3(("oStreamItem.u16CallInstance: %u ", oStreamItem.u16CallInstance));
      ETG_TRACE_USR1(("oStreamItem.e8CallStatus.enType: %d",oStreamItem.e8CallStatus.enType));
      ETG_TRACE_USR1(("oStreamItem.bUsingVehicleAudio: %d",oStreamItem.bUsingVehicleAudio));
      ETG_TRACE_USR1(("oStreamItem.u8DeviceHandle: %d",oStreamItem.u8DeviceHandle));
      ETG_TRACE_USR1(("oStreamItem.sTelephoneNumber: %s",oStreamItem.sTelephoneNumber.szGet()));
      ETG_TRACE_USR1(("oStreamItem.sName: %s",oStreamItem.sName.szGet()));

      ETG_TRACE_USR1(("oStreamItem.u8CallDurationHr: %d",oStreamItem.u8CallDurationHr));
      ETG_TRACE_USR1(("oStreamItem.u8CallDurationMin: %d",oStreamItem.u8CallDurationMin));
      ETG_TRACE_USR1(("oStreamItem.u8CallDurationSec: %d",oStreamItem.u8CallDurationSec));

      oCallStatusNoticeExtended.oCallStatusNoticeExtendedStream.oItems.push_back(oStreamItem);
   }

   ETG_TRACE_USR1(("oCallStatusNoticeExtended.u8TotalCallInstances: %u", callInstance));

   oCallStatusNoticeExtended.u8TotalCallInstances = callInstance;

   updateOpcodeStatus(callsReportListPropertyUpdate->_u64TokenId, oCallStatusNoticeExtended,
         callsReportListPropertyUpdate->u16GetFunctionID());
   oCallStatusNoticeExtended.vDestroy();
}

void CcaCallStatusNoticeExtendedHandler::processOpcodeSetError(PmCoreResponseData* pInMsg)
{
   (void) pInMsg;
}
