/**
 * @file CcaActivePhoneDeviceHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaActivePhoneDeviceHandler class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaActivePhoneDeviceHandler.h"
#include "CcaCallStatusListHandler.h"
#include "IpcWrapper.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaActivePhoneDeviceHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

CcaActivePhoneDeviceHandler::CcaActivePhoneDeviceHandler(ahl_tclBaseOneThreadService* pAhlService)
: CcaProperty(pAhlService), _activeDeviceHandle(PM_DEVICEHANDLE_ZERO)
{
   ETG_TRACE_USR4(("CcaActivePhoneDeviceHandler::CcaActivePhoneDeviceHandler entered"));
}

void CcaActivePhoneDeviceHandler::onOpcodeGet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_USR4(("CcaActivePhoneDeviceHandler::onOpcodeGet entered"));
   (void) pInMsg;

   PmResult pmresult = pm_ipc_wrapper::IpcWrapper::getInstance().getActivePassiveDeviceList(act);

   if(PM_RESULT_OK != pmresult._pmResultCode)
   {
      most_telfi_tclMsgActivePhoneDeviceStatus oActivePhoneDeviceStatus;
      oActivePhoneDeviceStatus.u8DeviceHandle = _activeDeviceHandle;

      updateOpcodeStatus(act, oActivePhoneDeviceStatus, PM_PROPERTYID_ACTIVEPHONEDEVICE);
      oActivePhoneDeviceStatus.vDestroy();
   }
}

void CcaActivePhoneDeviceHandler::onOpcodeSet(amt_tclServiceData* pInMsg, const ActType act)
{
   ETG_TRACE_ERR(("Received an invalid opcode - SET for MOST_TELFI_C_U16_ACTIVEPHONEDEVICE, FID: %u",
         pInMsg->u16GetFunctionID()));

   // Set is not possible, Hence removing the token from the table.
   removeEntryFromRequestMap(act);
}

void CcaActivePhoneDeviceHandler::onPropertyUpdate(PropertyUpdate* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaActivePhoneDeviceHandler::onPropertyUpdate entered"));
   dispatchProperty(propertyUpdate);
}

void CcaActivePhoneDeviceHandler::processOpcodeGetResponse(PmCoreResponseData* responseMsg)
{
   ETG_TRACE_USR4(("CcaActivePhoneDeviceHandler::processOpcodeGetResponse entered"));
   dispatchProperty(responseMsg);
}

template<typename TPropertyType>
void CcaActivePhoneDeviceHandler::dispatchProperty(TPropertyType* propertyUpdate)
{
   ETG_TRACE_USR4(("CcaActivePhoneDeviceHandler::dispatchProperty entered"));

   ActivePassiveDeviceListPropertyUpdate* activePassiveDeviceListPropertyUpdate =
         static_cast<ActivePassiveDeviceListPropertyUpdate*> (propertyUpdate);

   ActivePassiveDeviceListMap activePassiveDeviceListMap =
         activePassiveDeviceListPropertyUpdate->_activePassiveDeviceListObj._activePassiveDeviceList;

   DeviceHandle activeDeviceHandle = PM_DEVICEHANDLE_ZERO;

   for (auto& iter : activePassiveDeviceListMap)
   {
      if (DEVICEROLE_ACTIVE == iter.second)
      {
         ETG_TRACE_USR4(("Active device address: %s", iter.first.c_str()));
         ::com::bosch::pmapp::DeviceDetailsListHandler::getInstance().getDeviceHandle(
               iter.first, activeDeviceHandle);
         break;
      }
   }

   // The below function is called to keep track of the connected devices in "CcaCallStatusListHandler"
   CcaCallStatusListHandler::getInstance().onActivePassiveDeviceListChanged(activePassiveDeviceListMap,
         activeDeviceHandle);

   bool updateProperty = true;

   if (DEFAULT_ACT == activePassiveDeviceListPropertyUpdate->_u64TokenId)
   {
      ETG_TRACE_USR4(("This is an update from PM core"));

      if (activeDeviceHandle != _activeDeviceHandle)
      {
         _activeDeviceHandle = activeDeviceHandle;
      }
      else
      {
         updateProperty = false;
      }
   }

   if (updateProperty)
   {
      most_telfi_tclMsgActivePhoneDeviceStatus oActivePhoneDeviceStatus;
      oActivePhoneDeviceStatus.u8DeviceHandle = _activeDeviceHandle;
      ETG_TRACE_USR1(("oActivePhoneDeviceStatus.u8DeviceHandle :%u", oActivePhoneDeviceStatus.u8DeviceHandle));

      updateOpcodeStatus(activePassiveDeviceListPropertyUpdate->_u64TokenId, oActivePhoneDeviceStatus,
            activePassiveDeviceListPropertyUpdate->u16GetFunctionID());
      oActivePhoneDeviceStatus.vDestroy();
   }
}

void CcaActivePhoneDeviceHandler::processOpcodeSetError(PmCoreResponseData* pInMsg)
{
   (void) pInMsg;
}
