/**
 * @file CcaMethod.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaMethod class methods
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "CcaMethod.h"
#include "PmAppTrace.h"

using namespace pmcore;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CcaMethod.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS   TR_CLASS_PM_SERVICE
#endif
#endif

CcaMethod::CcaMethod(ahl_tclBaseOneThreadService* pService)
: CcaFunction(pService)
{
   ETG_TRACE_USR4(("CcaMethod::CcaMethod entered"));
}

void CcaMethod::handleMessage(amt_tclServiceData* pInMsg)
{
   ETG_TRACE_USR4(("CcaMethod::handleMessage entered for CCA message"));
   
   if(nullptr != pInMsg)
   {
      switch (pInMsg->u8GetOpCode())
      {
         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
         {
            ETG_TRACE_USR4(("CcaMethod::handleMessage AMT_C_U8_CCAMSG_OPCODE_METHODSTART"));

            /*
            * @see CcaFunction::generateToken() for details on token generation
            * If the map has the token already, which means the request is in progress for the same request.
            * Hint:- For consecutive multiple requests the command counter should be differently set by the client
            */
            tU64 ccaToken = 0;
            CcaMsgHeader msgHeader;

            generateToken(pInMsg, ccaToken, msgHeader);

            if(ccaToken > 0)
            {
               onMethodStart(pInMsg, ccaToken);
            }
            else
            {
               ETG_TRACE_ERR(("onMethodStart - Posting error since the same request is in process"));
               sendMethodError(msgHeader, most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
            }
         }
         break;

         default:
         {
            ETG_TRACE_USR4(("CcaMethod::handleMessage default"));
            break;
         }
      }
   }
}

void CcaMethod::handleMessage(PmCoreResponseData* pmCoreResponseData)
{
   ETG_TRACE_USR4(("CcaMethod::handleMessage entered for CPP response message"));
   
   if(pmCoreResponseData)
   {
      processMethodResponse(pmCoreResponseData);
   }
   else
   {
      ETG_TRACE_ERR(("CcaMethod::handleMessage pCppResponse is NULL"));
   }
}

void CcaMethod::sendMethodResult(const tU64 ccaToken)
{
   ETG_TRACE_USR4(("CcaMethod::sendMethodResult entered with ccaToken: %u", ccaToken));
   CcaMsgHeader msgHeaderInfo;

   bool entryFound = retrieveMessageHeader(ccaToken, msgHeaderInfo);

   if(entryFound)
   {
      msgHeaderInfo._u8OpCode = AMT_C_U8_CCAMSG_OPCODE_METHODRESULT;
      sendEmptyMessage(msgHeaderInfo);
      removeEntryFromRequestMap(ccaToken);
   }
   else
   {
      // This portion of code will be reached only due to error in coding.
      // Hint:- Check for the token value sent from the respective Message Handler
      ETG_TRACE_ERR(("CcaMethod::sendMethodResult invalid token sent from message handler for empty message"));
   }
}

void CcaMethod::sendMethodResult(const tU64 ccaToken, const fi_tclTypeBase& oOutData)
{
   ETG_TRACE_USR4(("CcaMethod::sendMethodResult entered with ccaToken: %u", ccaToken));
   CcaMsgHeader msgHeaderInfo;

   bool entryFound = retrieveMessageHeader(ccaToken, msgHeaderInfo);

   if(entryFound)
   {
      msgHeaderInfo._u8OpCode = AMT_C_U8_CCAMSG_OPCODE_METHODRESULT;
      sendMessage(msgHeaderInfo, oOutData);
      removeEntryFromRequestMap(ccaToken);
   }
   else
   {
      // This portion of code will be reached only due to error in coding.
      // Hint:- Check for the token value sent from the respective Message Handler
      ETG_TRACE_ERR(("CcaMethod::sendMethodResult invalid token sent from message handler"));
   }
}

void CcaMethod::sendMethodError(const CcaMsgHeader& msgHeaderInfo, tU16 u16ErrorCode)
{
   sendErrorMessage(msgHeaderInfo, u16ErrorCode);
}

void CcaMethod::sendMethodError(const tU64 ccaToken, tU16 u16ErrorCode)
{
   CcaMsgHeader msgHeaderInfo;
   bool entryFound = retrieveMessageHeader(ccaToken, msgHeaderInfo);

   if(entryFound)
   {
      sendErrorMessage(msgHeaderInfo, u16ErrorCode);
      removeEntryFromRequestMap(ccaToken);
   }
   else
   {
      // This portion of code will be reached only due to error in coding.
      // Hint:- Check for the token value sent from the respective Message Handler
      ETG_TRACE_ERR(("CcaMethod::sendMethodError invalid token sent from message handler in posting error message"));
   }
}

void CcaMethod::onPropertyUpdate(PropertyUpdate* /*pPropertyUpdate*/)
{
   // Do Nothing, this method is relevant only for CcaProperties
}

void CcaMethod::handlePmResult(const PmResult pmresult, const ActType ccaToken)
{
   if(PM_RESULT_OK != pmresult._pmResultCode)
   {
      ETG_TRACE_USR4(("Method start is not processed"));
      ETG_TRACE_USR4(("Result code: %u", pmresult._pmResultCode));
      ETG_TRACE_USR4(("Result message: %s", pmresult._pmResultMessage.c_str()));

      tU16 errorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;

      switch(pmresult._pmResultCode)
      {
         case PM_RESULT_ERR_INVALID_PARAMETER :
         {
            errorCode = most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE;
         }
         break;

         default:
         {
            errorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE;
         }
      }

      sendMethodError(ccaToken, errorCode);
   }
   else
   {
      (void) ccaToken;
   }
}

CcaMethod::~CcaMethod()
{

}
