/**
 * @file CcaMethod.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaMethod class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef CcaMethod_h
#define CcaMethod_h

#include "CcaFunction.h"

/*
 * This class acts as base class for all CCA method handlers
 *
 */
class CcaMethod : public CcaFunction
{

public:

   /**
    * Constructor of CcaMethod class
    *
    * @param[in] ahl_tclBaseOneThreadService*  pointer to the CCA service class
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CcaMethod(ahl_tclBaseOneThreadService* pService);

   /**
    * Destructor of CcaMethod class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual ~CcaMethod();

   /**
    * Performs opcode check for incoming message and invokes the message handler in the
    * derived class
    *
    * @param[in] amt_tclServiceData* pInMsg - Message in CCA format
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void handleMessage(amt_tclServiceData* pInMsg) override;

   /**
    * Process Method response from CPP interfaces.
    *
    * @param[in]  PmCoreResponseData* - pointer to the response data from CPP layer
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void handleMessage(PmCoreResponseData* pCppResponse) override;

   /**
    * Method which posts method result to the CCA framework
    *
    * @param[in] tU64 ccaToken - Token generated internally within IpcWrapper to map request / response between
    *                            IPC wrapper and PmCore
    *            fi_tclTypeBase& oOutData - Message data to be sent
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void sendMethodResult(const tU64 ccaToken, const fi_tclTypeBase& oOutData);

   /**
    * Method which posts method error to the CCA framework
    *
    * @param[in] tU64 ccaToken - Token generated internally within IpcWrapper to map request / response between
    *                            IPC wrapper and PmCore
    *            tU16 u16ErrorCode - Error codes outlined in MOST standard
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void sendMethodError(const tU64 ccaToken, tU16 u16ErrorCode);

   /**
    * Method which posts method error to the CCA framework
    *
    * @param[in] tU64 ccaToken - Token generated internally within IpcWrapper to map request / response between
    *                            IPC wrapper and PmCore
    *            tU16 u16ErrorCode - Error codes outlined in MOST standard
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void sendMethodError(const CcaMsgHeader&, tU16 u16ErrorCode);

   /**
    * This method shall be overridden in the property handlers, not required for methods
    * Just nullify the abstractness.
    *
    * @param[in] PropertyUpdate* - pointer to the property update data
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onPropertyUpdate(PropertyUpdate* pPropertyUpdate) override;

protected:

   /**
    * This method delegates the message requests to be handled in the derived class
    * 
    * @param[in]  amt_tclServiceData* pInMsg - Message in CCA format
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onMethodStart(amt_tclServiceData* pInMsg,tU64 ccaToken) = 0;

   /**
    * This method has to be overridden in the derived classes message handlers to process the response
    * data from the core layer(CPP interface) and update the requested clients
    *
    * @param[in]  PmCoreResponseData* - Pointer to the response data from CPP layer
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void processMethodResponse(PmCoreResponseData* pInMsg) = 0;

   /**
    * Method which posts messages back to the CCA framework like Method Result, Property Status for a GET request,.
    *
    * @param[in] tU64 ccaToken - Token generated internally within IpcWrapper to map request / response between
    *                            IPC wrapper and PmCore
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void sendMethodResult(const tU64 ccaToken);


   /**
    * Method which handles the "pmcore::PmResult" parameter received and manipulates and checks for the error.
    * In case of error, Method error is posted.
    *
    * @param[in] tU64 ccaToken - Token generated internally within IpcWrapper to map request / response between
    *                            IPC wrapper and PmCore
    *            pmcore::PmResult pmresult - This is the output parameter received by posting request to IPC wrapper.
    *                                In case of error, Method error is posted.
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void handlePmResult(const pmcore::PmResult pmresult, const pmcore::ActType ccaToken);
};

#endif // CcaMethod_h
