/**
 * @file CcaMessageHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CcaMessageHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef CcaMessageHandler_h
#define CcaMessageHandler_h

#include "IpcMessageHandler.h"
#include "AhlService.h"
#include "CcaFunction.h"
#include "CcaActiveCallInHandsetHandler.h"
#include "CcaActivePhoneDeviceHandler.h"
#include "CcaAudioTransferFailedEventHandler.h"
#include "CcaAutoWaitingModeOnOffListHandler.h"
#include "CcaCallOperationFailedEventHandler.h"
#include "CcaCallStatusNoticeExtendedHandler.h"
#include "CcaCallStatusNoticeHandler.h"
#include "CcaCellDeviceNameHandler.h"
#include "CcaCellNetworkStatusHandler.h"
#include "CcaCellSignalStrengthHandler.h"
#include "CcaCellStateOfChargeHandler.h"
#include "CcaDeviceRingtoneListHandler.h"
#include "CcaExternalVRStateHandler.h"
#include "CcaHandsFreeEnhancedCallControlHandler.h"
#include "CcaHFServiceHandler.h"
#include "CcaInComingCallEventHandler.h"
#include "CcaIncomingCallRejectedEventHandler.h"
#include "CcaMicrophoneMuteStateHandler.h"
#include "CcaNetworkOperatorHandler.h"
#include "CcaOutgoingCallFailedEventHandler.h"
#include "CcaRingtoneMuteStateHandler.h"
#include "CcaRingtonePlaybackStatusHandler.h"
#include "CcaSendDTMFFailedEventHandler.h"
#include "CcaSlcStatusHandler.h"
#include "CcaSuppressRingtoneOnOffListHandler.h"
#include "CcaSystemWideRingtonesListHandler.h"
#include "CcaVehicleAudioHandler.h"
#include "CcaVoiceRecognitionExtendedHandler.h"
#include "CcaVoiceRecognitionHandler.h"
#include "CcaWaitingModeFilePathHandler.h"
#include "CcaWaitingModeStateHandler.h"
#include "CcaAcceptCallReqHandler.h"
#include "CcaActivateExternalVRReqHandler.h"
#include "CcaAutoWaitingModeOnOffReqHandler.h"
#include "CcaCancelOutgoingCallReqHandler.h"
#include "CcaConferenceInReqHandler.h"
#include "CcaDeActivateExternalVRReqHandler.h"
#include "CcaDialExtendedReqHandler.h"
#include "CcaDialReqHandler.h"
#include "CcaHangUpCallReqHandler.h"
#include "CcaHoldCallReqHandler.h"
#include "CcaIgnoreCallReqHandler.h"
#include "CcaPlayRingtoneReqHandler.h"
#include "CcaQueryTelephoneNumberReqHandler.h"
#include "CcaRedialReqHandler.h"
#include "CcaRejectCallReqHandler.h"
#include "CcaResumeCallReqHandler.h"
#include "CcaSendDTMFReqHandler.h"
#include "CcaSetActivePhoneReqHandler.h"
#include "CcaSetRingtoneReqHandler.h"
#include "CcaSpeedDialReqHandler.h"
#include "CcaSplitCallReqHandler.h"
#include "CcaStartStopWaitingModeReqHandler.h"
#include "CcaStopRingtoneReqHandler.h"
#include "CcaSuppressRingtoneOnOffReqHandler.h"
#include "CcaSwapCallReqHandler.h"
#include "CcaTransferCallToHandsetReqHandler.h"
#include "CcaTransferCallToVehicleReqHandler.h"
#include "CcaVoiceMailReqHandler.h"

#include <map>

/*
 * This class acts as the gateway between the CCA messages and the CCA message
 * handlers.
 */
class CcaMessageHandler final: public IpcMessageHandler<amt_tclServiceData>,
public IpcMessageHandler<PmCoreResponseData>
{

public:

   /**
    * Process Method request from different IPC. This method forwards the message to the respective message handlers
    * to process the CCA message
    *
    * @param[in] amt_tclServiceData* - Pointer to the CCA message data
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void processMessage(amt_tclServiceData* pInMessage) override;

   /**
    * Process Method response from CPP interfaces. This method forwards the response data from the core layer to the
    * method handlers
    *
    * @param[in] PmCoreResponseData* - pointer to the response data from CPP layer
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void processMessage(PmCoreResponseData* pmCoreResponseData) override;

   /**
    * Insert the function ID & corresponding function handler in the _functionMap
    *
    * @param[in] tU16 u16FktId - unique function ID for every CCA method/property
    *            CcaFunction*  pointer to the CcaFunction class
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void addFunction(tU16 u16FktId, CcaFunction* fnHandler);

   /**
    * Process the property updates from CPP interface
    *
    * @param[in] PropertyUpdate* - pointer to the property update data structure
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onPropertyUpdate(PropertyUpdate* pPropertyUpdate) override;

   /**
    * Constructor of CcaMessageHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CcaMessageHandler();

   /**
    * Constructor of CcaMessageHandler class
    *
    * @param[in]  AhlService* - pointer to the PM CCA Service
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CcaMessageHandler(AhlService* pAhlService);

   /**
    * Destructor of CcaMessageHandler class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ~CcaMessageHandler();

   /**
     * Copy constructor of CcaMessageHandler class
     *
     * @param[in] CcaMessageHandler& - Reference to CcaMessageHandler
     * @param[out]
     * @param[in,out]
     *
     * @return
     *
     */
   CcaMessageHandler(const CcaMessageHandler&);

   /**
     * Copy constructor of CcaMessageHandler class
     *
     * @param[in] CcaMessageHandler& - Reference to CcaMessageHandler
     * @param[out]
     * @param[in,out]
     *
     * @return CcaMessageHandler& - Reference to CcaMessageHandler
     *
     */
   CcaMessageHandler& operator=(const CcaMessageHandler&);

private:

   /**
    * Instantiate all the Handler class of CCA messages (both methods and properties)
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void createCcaMsgHandlers();

   /**
    * Returns the pointer to the message handler mapped to a function ID
    *
    * @param[in]  tU16 u16FktId - unique ID of a function
    * @param[out]
    * @param[in,out]
    *
    * @return  CcaFunction* - pointer to the message handler
    *
    */
   CcaFunction* getMsgHandlerFromFktId(tU16 u16FktId);

   /**
    * Returns the pointer to the message handler mapped to a function ID
    *
    * @param[in]  tU64 tokenTd - unique token ID generated
    * @param[out]
    * @param[in,out]
    *
    * @return  CcaFunction* - pointer to the message handler
    *
    */
   CcaFunction* getMsgHandlerFromTokenId(tU64 tokenTd);

   /**
    * Delete all the Handler class of CCA messages (both methods and properties)
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void deleteCcaMsgHandlers();

   std::map<tU16, CcaFunction* > _functionMap; /**< Map the function Id with the handler function */

   AhlService* _pAhlService;  /**< Pointer to the PM CCA Service */

   //Property handlers
   CcaActiveCallInHandsetHandler* _ccaActiveCallInHandsetHandler; /**< Instance of Active call in handset class */

   CcaActivePhoneDeviceHandler* _ccaActivePhoneDeviceHandler; /**< Instance of Active phone device class */

   CcaAudioTransferFailedEventHandler* _ccaAudioTransferFailedEventHandler; /**< Instance of Audio transfer Failed Event class */

   CcaAutoWaitingModeOnOffListHandler* _ccaAutoWaitingModeOnOffListHandler; /**< Instance of AutoWaitingMode on off list class */

   CcaCallOperationFailedEventHandler* _ccaCallOperationFailedEventHandler; /**< Instance of CallOperationFailedEventHandler class */

   CcaCallStatusNoticeExtendedHandler* _ccaCallStatusNoticeExtendedHandler; /**< Instance of Call status notice extended class */

   CcaCallStatusNoticeHandler* _ccaCallStatusNoticeHandler; /**< Instance of Call status notice class */

   CcaCellDeviceNameHandler* _ccaCellDeviceNameHandler; /**< Instance of Cell device Name class */

   CcaCellNetworkStatusHandler* _ccaCellNetworkStatusHandler; /**< Instance of Cell Network status class */

   CcaCellSignalStrengthHandler* _ccaCellSignalStrengthHandler; /**< Instance of Cell signal strength class */

   CcaCellStateOfChargeHandler* _ccaCellStateOfChargeHandler; /**< Instance of Cell state of charge class */

   CcaDeviceRingtoneListHandler* _ccaDeviceRingtoneListHandler; /**< Instance of Device Ringtone List class */

   CcaExternalVRStateHandler* _ccaExternalVRStateHandler; /**< Instance of external VR state class */

   CcaHandsFreeEnhancedCallControlHandler* _ccaHandsFreeEnhancedCallControlHandler; /**< Instance of Handsfree enhanced call control class */

   CcaHFServiceHandler* _ccaHFServiceHandler; /**< Instance of HF service class */

   CcaInComingCallEventHandler* _ccaInComingCallEventHandler; /**< Instance of Incoming call event class */

   CcaIncomingCallRejectedEventHandler* _ccaIncomingCallRejectedEventHandler; /**< Instance of Incoming call reject event class */

   CcaMicrophoneMuteStateHandler* _ccaMicrophoneMuteStateHandler; /**< Instance of Microphone mute state class */

   CcaNetworkOperatorHandler* _ccaNetworkOperatorHandler; /**< Instance of Network operator class */

   CcaOutgoingCallFailedEventHandler* _ccaOutgoingCallFailedEventHandler; /**< Instance of Outgoing call failed event class */

   CcaRingtoneMuteStateHandler* _ccaRingtoneMuteStateHandler; /**< Instance of Ringtone mute state class */

   CcaRingtonePlaybackStatusHandler* _ccaRingtonePlaybackStatusHandler; /**< Instance of Ringtone playback status class */

   CcaSendDTMFFailedEventHandler* _ccaSendDTMFFailedEventHandler; /**< Instance of Send DTMF Failed Event class */

   CcaSlcStatusHandler* _ccaSlcStatusHandler; /**< Instance of SLC status class */

   CcaSuppressRingtoneOnOffListHandler* _ccaSuppressRingtoneOnOffListHandler; /**< Instance of SuppressRingtoneOnOff List class */

   CcaSystemWideRingtonesListHandler* _ccaSystemWideRingtonesListHandler; /**< Instance of System Wide Ringtones List class */

   CcaVehicleAudioHandler* _ccaVehicleAudioHandler; /**< Instance of Vehicle audio class */

   CcaVoiceRecognitionExtendedHandler* _ccaVoiceRecognitionExtendedHandler; /**< Instance of voice recognition Extended class */

   CcaVoiceRecognitionHandler* _ccaVoiceRecognitionHandler; /**< Instance of voice recognition class */

   CcaWaitingModeFilePathHandler* _ccaWaitingModeFilePathHandler; /**< Instance of Waiting mode file path class */

   CcaWaitingModeStateHandler* _ccaWaitingModeStateHandler; /**< Instance of Waiting Mode state class */

   //Method handlers
   CcaAcceptCallReqHandler* _ccaAcceptCallReqHandler; /** < Instance of AcceptCall request handler class */

   CcaActivateExternalVRReqHandler* _ccaActivateExternalVRReqHandler; /** < Instance of ActivateExternalVR request handler class */

   CcaAutoWaitingModeOnOffReqHandler* _ccaAutoWaitingModeOnOffReqHandler; /** < Instance of AutoWaitingModeOnOff request handler class */

   CcaCancelOutgoingCallReqHandler* _ccaCancelOutgoingCallReqHandler; /** < Instance of CcaCancelOutgoingCall request handler class */

   CcaConferenceInReqHandler* _ccaConferenceInReqHandler; /** < Instance of CcaConferenceIn request handler class*/

   CcaDeActivateExternalVRReqHandler* _ccaDeActivateExternalVRReqHandler; /** < Instance of DeActivateExternalVR request handler class */

   CcaDialExtendedReqHandler* _ccaDialExtendedReqHandler; /**< Instance of DialExtended request handler class */

   CcaDialReqHandler* _ccaDialReqHandler; /**< Instance of Dial request handler class */

   CcaHangUpCallReqHandler* _ccaHangUpCallReqHandler; /** < Instance of CcaHangUpCall request handler class */

   CcaHoldCallReqHandler* _ccaHoldCallReqHandler; /** < Instance of CcaHoldCall request handler class */

   CcaIgnoreCallReqHandler* _ccaIgnoreCallReqHandler; /** < Instance of CcaIgnoreCall request handler class */

   CcaPlayRingtoneReqHandler* _ccaPlayRingtoneReqHandler;  /** < Instance of Play Ringtone request handler class */

   CcaQueryTelephoneNumberReqHandler* _ccaQueryTelephoneNumberReqHandler; /** < Instance of CcaQueryTelephoneNumber request handler class */

   CcaRedialReqHandler* _ccaRedialReqHandler; /** < Instance of CcaRedial request handler class */

   CcaRejectCallReqHandler* _ccaRejectCallReqHandler; /** < Instance of CcaRejectCall request handler class */

   CcaResumeCallReqHandler* _ccaResumeCallReqHandler; /** < Instance of CcaResumeCall request handler class */

   CcaSendDTMFReqHandler* _ccaSendDTMFReqHandler; /** < Instance of CcaSendDTMF request handler class */

   CcaSetActivePhoneReqHandler* _ccaSetActivePhoneReqHandler; /** < Instance of CcaSetActivePhone request handler class */

   CcaSetRingtoneReqHandler* _ccaSetRingtoneReqHandler;  /** < Instance of Set Ringtone request handler class */

   CcaSpeedDialReqHandler* _ccaSpeedDialReqHandler; /** < Instance of CcaSpeedDial request handler class */

   CcaSplitCallReqHandler* _ccaSplitCallReqHandler; /** < Instance of CcaSplitCall request handler class */

   CcaStartStopWaitingModeReqHandler* _ccaStartStopWaitingModeReqHandler; /** < Instance of CcaStartStopWaitingMode request handler class */

   CcaStopRingtoneReqHandler* _ccaStopRingtoneReqHandler;  /** < Instance of Stop Ringtone request handler class */

   CcaSuppressRingtoneOnOffReqHandler* _ccaSuppressRingtoneOnOffReqHandler;  /** < Instance of SuppressRingtoneOnOff request handler class */

   CcaSwapCallReqHandler* _ccaSwapCallReqHandler; /** < Instance of CcaSwapCall request handler class */

   CcaTransferCallToHandsetReqHandler* _ccaTransferCallToHandsetReqHandler; /** < Instance of CcaTransferCallToHandset request handler class */

   CcaTransferCallToVehicleReqHandler* _ccaTransferCallToVehicleReqHandler; /** < Instance of CcaTransferCallToVehicle request handler class */

   CcaVoiceMailReqHandler* _ccaVoiceMailReqHandler; /** < Instance of CcaVoiceMail request handler class */

};

#endif // CcaMessageHandler_h
