/**
 * @file CcaFunction.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the Cca Function class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This class is the base class for all the CCA related methods
 * and property handlers. All the CCA related message handlers has to inherit
 * this class
 *
 */

#ifndef CcaFunction_h
#define CcaFunction_h

#include "amt_fw.h"
#include "ahl_fw.h"
#include "fi_msgfw_fw.h"
#include "conn_most_fi_fw.h"

#include "IpcMsgTypes.h"
#include "DeviceDetailsListHandler.h"
#include "CcaCallStatusListHandler.h"

#include <map>

/*
 * This class takes care of handling the CCA messages and also involves
 * in retrieving the CCA message format from AhlServer
 *
 */
class CcaFunction
{

public:

   /**
    * Whenever a new CCA message is received for PmApp, this method would get invoked
    * from the CcaMessageHandler and the derived classes has to override this function to act
    * upon the received message. The received message could either be a property or method.
    *
    * @param[in]  amt_tclServiceData* pInMsg - Message in CCA format
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void handleMessage(amt_tclServiceData* pInMsg) = 0;

   /**
    * Process Method response from CPP interfaces. This method has to be overridden in the
    * derived classes message handlers to process the response data from the core layer
    * and update the requested clients
    *
    * @param[in]  PmCoreResponseData* - pointer to the response data from CPP layer
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void handleMessage(PmCoreResponseData* pmCoreResponseMsg) = 0;

   /**
    * This method shall be overridden in the derived class property handlers,
    * where the clients shall be updated with the status.
    *
    * @param[in] PropertyUpdate* - pointer to the property update data
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onPropertyUpdate(PropertyUpdate* pPropertyUpdate) = 0;

   /**
    * Remove the request entry / entries from the CCA Request Map
    *
    * @param[in] tU64 - Remove the token ID and its corresponding value from _ccaRequestMap
    *            bool - Remove all entries in the _ccaRequestMap
    * @param[out]
    * @param[in,out]
    *
    * @return void
    *
    */
   void removeEntryFromRequestMap(tU64 ccaToken, bool allEntries = false);

   /**
    * Constructor of CcaFunction class - When instances of the derived classes are created
    * the function and service IDs are passed
    *
    * @param[in]  tU16 u16FunctionID - unique ID for every CCA - Property and Method
    *             ahl_tclBaseOneThreadService* pService pointer to the CCA base class service
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CcaFunction(ahl_tclBaseOneThreadService* pService);

   /**
    * Destructor of CcaFunction class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual ~CcaFunction();

   /**
    * This function returns the size of the CCARequestMap.
    * This is called only from GTest to check whether all the elements are cleared after sending the response.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return size of the CCARequestMap
    *
    */
   static tU64 getRequestMapSize()
   {
      return _ccaRequestMap.size();
   }

protected:

   /**
    * Method which posts messages back to the CCA framework like Method Result,
    * Property Status,..
    *
    * @param[in]  CcaMsgHeader& msgHeaderInfo - contains CCA message header information
    *             fi_tclTypeBase& oOutData - Message data to be sent
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void sendMessage(const CcaMsgHeader& msgHeaderInfo, const fi_tclTypeBase& oOutData);

   /**
    * Method to post empty messages (without any data) to the CCA framework
    *
    * @param[in]  CcaMsgHeader& msgHeaderInfo - Contains CCA message header information
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void sendEmptyMessage(const CcaMsgHeader& msgHeaderInfo);

   /**
    * Method to extract data from the FI message format
    *
    * @param[in]  amt_tclServiceData* pFIMsg - Message in CCA format
    * @param[out] fi_tclTypeBase& oFIData - Converted data from CCA format
    * @param[in,out]
    *
    * @return
    *
    */
   void getDataFromAmt(amt_tclServiceData* pFIMsg, fi_tclTypeBase& oFIData);

   /**
    * Method to send error message to CCA framework
    *
    * @param[in]  CcaMsgHeader& msgHeaderInfo - Has the neccessary CCA message header
    *             information
    *             tU16 u16ErrorCode - Error codes outlined in MOST standard
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void sendErrorMessage(const CcaMsgHeader& msgHeaderInfo, tU16 u16Errorcode);

   /**
    * Retrieve the CCA Message header information from the received CCA message. Header information includes:
    * Source ID, Function Id, Command Counter, Opcode & Register ID.
    *
    * @param[in] amt_tclServiceData* - Message in CCA format
    * @param[out] CcaMsgHeader& - Populate the CCA message header information to this object
    * @param[in,out]
    *
    * @return
    *
    */
   void retrieveMessageHeader(const amt_tclServiceData* pFIMsg, CcaMsgHeader& msgHeaderInfo);

   /**
    * Retrieve the CCA Message header information from the ccaRequestmap. Header information includes:
    * Source ID, Function Id, Command Counter, Opcode & Register ID.
    *
    * @param[in] tU64 - asynchronous token generated within PM-IpcWrapper layer
    * @param[out] CcaMsgHeader& - Populate the CCA message header information to this object
    * @param[in,out]
    *
    * @return
    *
    */
   bool retrieveMessageHeader(const tU64 ccaToken, CcaMsgHeader& msgHeaderInfo);

   /**
    * Generate the token for message request based on the parameters - SourceAppID, CmdCounter, FunctionID
    *
    * @param[in] const amt_tclServiceData* - CCA message type
    * @param[out] tU64& - Token generated based on the CCA message
    * @param[in,out]
    *
    * @return void
    *
    */
   void generateToken(amt_tclServiceData* pInMsg, tU64& ipcToken, CcaMsgHeader& msgHeader);

   ahl_tclBaseOneThreadService* _pService;   /**< CCA service instance */

   static std::map<tU64, CcaMsgHeader> _ccaRequestMap; /**< Map the generated token with CCA Header Information */

private:

   /**
    * Used to get the Major version of the PM Application
    *
    * @param[in]
    * @param[out] tU16 - Major version of the PM Application
    * @param[in,out]
    *
    * @return tU16 - Major version of the PM Application
    *
    */
   tU16 getMajorVersion();

   void printCcaRequestMap();
};

#endif // CcaFunction_h
