/**
 * @file YakinduSmVoiceRecIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmVoiceRecIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice recognition
 *
 * @ingroup PmCore
 */

#include "YakinduSmVoiceRecIf.h"
#include "PmVoiceRecHandler.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/YakinduSmVoiceRecIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   YakinduSmVoiceRecIf::YakinduSmVoiceRecIf():_deviceToVoiceRecHandlerSmMap()
   {
   }

   YakinduSmVoiceRecIf::~YakinduSmVoiceRecIf()
   {
      _deviceToVoiceRecHandlerSmMap.clear();
   }

   void YakinduSmVoiceRecIf::onDeviceConnected(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceRecIf::onDeviceConnected"));

      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);

      if(it == _deviceToVoiceRecHandlerSmMap.end())
      {
         PmVoiceRecHandler* pmVoiceRecHandler = new PmVoiceRecHandler(deviceAddress);
         if(nullptr != pmVoiceRecHandler)
         {
            _deviceToVoiceRecHandlerSmMap.insert(std::pair<BdAddress, PmVoiceRecHandler*>(deviceAddress, pmVoiceRecHandler));
         }
      }
      else
      {
         ETG_TRACE_USR4(("SM already exists for device %s", deviceAddress.c_str()));
      }
   }

   void YakinduSmVoiceRecIf::onDeviceDisconnected(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceRecIf::onDeviceDisconnected"));

      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->onDeviceDisconnection();
         delete it->second;
         _deviceToVoiceRecHandlerSmMap.erase(it);
      }
   }

   PmResult YakinduSmVoiceRecIf::startStopVoiceRecognitionRequest(IN const BdAddress& bdAddress, IN const StartStop startStopVR)
   {
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      auto it = _deviceToVoiceRecHandlerSmMap.find(bdAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         pmResult = it->second->startStopVoiceRecognitionRequest(startStopVR);
      }

      return pmResult;
   }

   PmResult YakinduSmVoiceRecIf::startStopExtVoiceRecognitionRequest(IN const BdAddress& bdAddress,
         IN const StartStop startStopExtVR)
   {
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      auto it = _deviceToVoiceRecHandlerSmMap.find(bdAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         pmResult = it->second->startStopExtVoiceRecognitionRequest(startStopExtVR);
      }

      return pmResult;
   }

   void YakinduSmVoiceRecIf::onAgVoiceRecStateUpdate(IN const BdAddress& deviceAddress,
         IN const VRStatus vrStatus)
   {
      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->onAgVoiceRecStateUpdate(vrStatus);
      }
   }

   void YakinduSmVoiceRecIf::onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
         IN const AudioManagerEventType amEvent)
   {
      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->onAudioManagerEventUpdate(amEvent);
      }
   }

   void YakinduSmVoiceRecIf::onScoConnectionStatusChanged(IN const BdAddress& deviceAddress,
         IN const SCOStatus scoStatus)
   {
      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->onScoConnectionStatusChanged(scoStatus);
      }
   }

   void YakinduSmVoiceRecIf::onCallAdded(IN const BdAddress& deviceAddress)
   {
      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->onCallAdded();
      }
   }

   void YakinduSmVoiceRecIf::switchToPassiveRequest(IN const BdAddress& deviceAddress,
         const pmaudiomanager::AudioChannel audioChannelToAcquire)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceRecIf::switchToPassiveRequest"));

      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->switchToPassiveRequest(audioChannelToAcquire);
      }
   }

   void YakinduSmVoiceRecIf::switchToPassiveClientRequest(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceRecIf::switchToPassiveClientRequest"));

      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->switchToPassiveClientRequest();
      }
   }

   void YakinduSmVoiceRecIf::stopExtendedVR(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceRecIf::stopExtendedVR"));

      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->stopExtendedVR();
      }
   }

   void YakinduSmVoiceRecIf::startStopVoiceRecognitionError(IN const BdAddress& deviceAddress, IN const StartStop vrValue,
         IN PmResult& pmResult)
   {
      ETG_TRACE_USR4(("YakinduSmVoiceRecIf::startStopVoiceRecognitionError"));

      auto it = _deviceToVoiceRecHandlerSmMap.find(deviceAddress);
      if(it != _deviceToVoiceRecHandlerSmMap.end())
      {
         it->second->startStopVoiceRecognitionError(vrValue, pmResult);
      }
   }
}
