/**
 * @file VRController.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the VRController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the VRController interfaces of PmCore.
 *
 * @ingroup PmCore
 */

#include "VRController.h"
#include "PmCoreMainController.h"
#include "PmAudioManagerWrapper.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PmCoreIfMessageResult.h"
#include "YakinduSmVoiceRecIf.h"
#include "EvoBtStackWrapper.h"
#include "BtStackWrapperTypesInternal.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/VRController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   std::map<pmcore::BdAddress, pmcore::VoiceRecognitionStatus> VRController::_deviceVRStatus;
   std::map<pmcore::BdAddress, pmcore::ExtVoiceRecognitionStatus> VRController::_deviceExtVRStatus;
   std::map<pmcore::BdAddress, pmcore::EnhancedVoiceRecognitionFeature> VRController::_enhancedVRSupportedDevices;

   VRController::VRController() :
      _propertyIdList()
   {
      ETG_TRACE_USR1(("VRController"));

      _smVoiceRecIf = new YakinduSmVoiceRecIf;

      subscribeToBtStackEventNotifier();
   }

   VRController::~VRController()
   {
      ETG_TRACE_USR1(("~VRController"));
      PropertyUpdateNotifierToCore::getInstance().detachControllerInNotifierList(_propertyIdList, this);
      _propertyIdList.clear();

      _enhancedVRSupportedDevices.clear();

      if(nullptr != _smVoiceRecIf)
      {
         delete _smVoiceRecIf;
         _smVoiceRecIf = nullptr;
      }
   }

   void VRController::subscribeToBtStackEventNotifier()
   {
      ETG_TRACE_USR1(("subscribeToBtStackEventNotifier"));

      // Pushing the default properties for both CallController & VRController
      for( unsigned int index = BTS_UPDATE_AUDIO; index < BTS_UPDATE_END_VR_CONTROLLER; index++)
      {
         if(index != BTS_UPDATE_END_CALL_CONTROLLER)
         {
            _propertyIdList.push_back(static_cast<PmCorePropertyAndEventId>(index));
         }
      }

      // Pushing the other interested properties and signals
      _propertyIdList.push_back(ON_DEVICE_CONNECTED);
      _propertyIdList.push_back(ON_DEVICE_DISCONNECTED);
      _propertyIdList.push_back(BTS_SIGNAL_SCO_CONNECT_REQUEST);
      _propertyIdList.push_back(BTS_SIGNAL_VOICE_CALL_ADDED_END_VR);
      _propertyIdList.push_back(BTS_UPDATE_PROXY_SERVICE_AVAILABILITY);

      // Events from PM Audiomanager
      _propertyIdList.push_back(AM_VR_CHANNEL_ACQUISITION_SUCCESS);
      _propertyIdList.push_back(AM_VR_CHANNEL_ACQUISITION_FAILURE);

      _propertyIdList.push_back(AM_VR_CHANNEL_PLAYAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_VR_CHANNEL_PLAYAUDIO_FAILURE);

      _propertyIdList.push_back(AM_VR_CHANNEL_STOPAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_VR_CHANNEL_STOPAUDIO_FAILURE);

      _propertyIdList.push_back(AM_VR_CHANNEL_PAUSEAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_VR_CHANNEL_PAUSEAUDIO_FAILURE);

      PropertyUpdateNotifierToCore::getInstance().attachControllerToNotifierList(_propertyIdList, this);
   }

   void VRController::onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
         IN std::shared_ptr<void> propertyDetails)
   {
      ETG_TRACE_USR4(("VRController::onPropertyUpdate propertyId : %d",
            ETG_CENUM(PmCorePropertyAndEventId, propertyId)));
      //TODO: Handler functions needs to be implemented and mapped to switch cases.

      if(nullptr == propertyDetails)
      {
         ETG_TRACE_ERR(("VRController::onPropertyUpdate with empty details"));
         return;
      }

      switch(propertyId)
      {
         case BTS_UPDATE_VOICE_RECOGNITION_STATUS:
         {
            std::shared_ptr<PropertyDetails<VoiceRecognitionStatus>> property =
                  std::static_pointer_cast<PropertyDetails<VoiceRecognitionStatus>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();
            VRStatus vrStatus = property->getMessage()._vrStatus;

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate VRStatus : %u", vrStatus));

            if(nullptr != _smVoiceRecIf)
            {
               // Try to acquire Active device slot if available
               DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
               ActiveSwitchingRequest activeSwitchingRequest(deviceAddress,
                     TEMP_ROLE_SWITCH, PendingRequestInfo(), VR_CONTROLLER);
               DeviceRoleSwitchResultEnum deviceRoleSwitchResultEnum = CANNOT_BE_MADE_ACTIVE;

               if(VR_SESSION_ACTIVE == vrStatus)
               {
                  deviceRoleSwitchResultEnum =
                     (deviceInfoHandler.acquireFreeActiveDeviceSlot(deviceAddress, activeSwitchingRequest));
               }

               if (((PmCoreMainController::getInstance().getCallController().isCallsInIdleState(deviceAddress))) &&
                     (((VR_SESSION_ACTIVE == vrStatus) &&
                     (CANNOT_BE_MADE_ACTIVE != deviceRoleSwitchResultEnum)) ||
                     (VR_SESSION_IDLE == vrStatus)))
               {
                  _smVoiceRecIf->onAgVoiceRecStateUpdate(deviceAddress, vrStatus);
               }
            }
         }
         break;

         case BTS_UPDATE_ECHO_CANCELING_NOISE_REDUCTION:
            break;

         case BTS_UPDATE_SIRI_STATUS:
         {
            //TODO: check if seperate property for VoiceRec and SIRI to be maintained??
            std::shared_ptr<PropertyDetails<uint8_t>> property =
                  std::static_pointer_cast<PropertyDetails<uint8_t>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();
            uint8_t siriStatus = property->getMessage();

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate SiriStatus : %u", siriStatus));

            auto iter = _enhancedVRSupportedDevices.find(deviceAddress);

            switch (siriStatus)
            {
               case 0:
               {
                  if ((_enhancedVRSupportedDevices.end() != iter) && ("Siri" == iter->second._enhancedVRFeature))
                  {
                     // Only if there is a change, then it is updated.
                     PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().
                           doOnEnhancedVoiceRecognitionFeatureChanged(deviceAddress, EnhancedVoiceRecognitionFeature());
                     _enhancedVRSupportedDevices.erase(iter);
                  }
               }
               break;

               case 1:
               {
                  if (_enhancedVRSupportedDevices.end() == iter)
                  {
                     // SiriNR needs to be enabled
                     evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSiriSetNRRequest(deviceAddress,
                           ENABLE_SIRI_NR, PM_DEFAULT_ACT);

                     _enhancedVRSupportedDevices.emplace_hint(_enhancedVRSupportedDevices.end(), deviceAddress,
                           EnhancedVoiceRecognitionFeature("Siri", ENHANCED_VR_DEFAULT));
                  }
                  else
                  {
                     if (ENHANCED_VR_DISABLED == iter->second._enhancedVRAvailability)
                     {
                        iter->second._enhancedVRAvailability = ENHANCED_VR_ENABLED;
                        PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().
                              doOnEnhancedVoiceRecognitionFeatureChanged(deviceAddress, iter->second);
                     }
                  }
               }
               break;

               case 2:
               {
                  if (_enhancedVRSupportedDevices.end() == iter)
                  {
                     iter = _enhancedVRSupportedDevices.emplace_hint(_enhancedVRSupportedDevices.end(),
                           deviceAddress, EnhancedVoiceRecognitionFeature("Siri", ENHANCED_VR_DISABLED));
                  }
                  else
                  {
                     iter->second._enhancedVRAvailability = ENHANCED_VR_DISABLED;
                  }

                  PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().
                        doOnEnhancedVoiceRecognitionFeatureChanged(deviceAddress, iter->second);
               }
               break;

               default:
               {
                  ETG_TRACE_USR4(("BTS_UPDATE_SIRI_STATUS- Default case"));
               }
            }
         }
         break;

         case BTS_UPDATE_NOISE_REDUCTION:
         {
            std::shared_ptr<PropertyDetails<uint8>> property =
                  std::static_pointer_cast<PropertyDetails<uint8>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();
            uint8_t siriNRStatus = property->getMessage();

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", property->getBdAddress().c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate siriNRStatus : %u", siriNRStatus));

            auto iter = _enhancedVRSupportedDevices.find(deviceAddress);
            if (iter != _enhancedVRSupportedDevices.end())
            {
               if (ENABLED_SIRI_NR == siriNRStatus)
               {
                  if (iter->second._enhancedVRAvailability != ENHANCED_VR_ENABLED)
                  {
                     iter->second._enhancedVRAvailability = ENHANCED_VR_ENABLED;
                     PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().
                           doOnEnhancedVoiceRecognitionFeatureChanged(deviceAddress, iter->second);

                     evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSetPropertyRequest(
                                    ::ccdbusif::evolution::IF_SIRI, deviceAddress, "EyesFreeMode", true);
                  }
               }
               else if (DISABLED_SIRI_NR == siriNRStatus)
               {
                  if (iter->second._enhancedVRAvailability == ENHANCED_VR_ENABLED)
                  {
                     iter->second._enhancedVRAvailability = ENHANCED_VR_DISABLED;
                     PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().
                           doOnEnhancedVoiceRecognitionFeatureChanged(deviceAddress, iter->second);
                  }
               }
            }
         }
         break;

         case BTS_UPDATE_EYES_FREE_MODE:
         {
            std::shared_ptr<PropertyDetails<bool>> property =
                              std::static_pointer_cast<PropertyDetails<bool>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();
            bool efmStatus = property->getMessage();

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate efmStatus : %u", efmStatus));
         }
         break;

         case BTS_UPDATE_SIRI_SUPPORTED_TYPE:
         {
            std::shared_ptr<PropertyDetails<uint16>> property =
                              std::static_pointer_cast<PropertyDetails<uint16>>(propertyDetails);
         }
         break;

         case BTS_UPDATE_SCO_TYPE:
         {
            std::shared_ptr<PropertyDetails<SCOConnection>> property =
                  std::static_pointer_cast<PropertyDetails<SCOConnection>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();
            SCOStatus scoStatus = property->getMessage()._scoStatus;

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate ScoStatus : %u", scoStatus));

            if(nullptr != _smVoiceRecIf)
            {
               if (! (SCO_NOT_ESTABLISHED == scoStatus))
               {
                  auto iter = _enhancedVRSupportedDevices.find(deviceAddress);

                  if (_enhancedVRSupportedDevices.end() != iter)
                  {
                     if (("Siri" == iter->second._enhancedVRFeature)
                           && (ENHANCED_VR_ENABLED == iter->second._enhancedVRAvailability))
                     {
                        switch (scoStatus)
                        {
                           case SCO_WIDEBAND:
                              scoStatus = SCO_WIDEBAND_SIRI;
                              break;
                           case SCO_NARROWBAND:
                              scoStatus = SCO_NARROWBAND_SIRI;
                              break;
                           default:
                              ETG_TRACE_USR4(("ScoStatus- default case"));
                        }
                     }
                  }
               }
               _smVoiceRecIf->onScoConnectionStatusChanged(deviceAddress, scoStatus);
            }
         }
         break;

         case ON_DEVICE_CONNECTED:
         {
            std::shared_ptr<PropertyDetails<BasicDeviceDetails>> property =
                  std::static_pointer_cast<PropertyDetails<BasicDeviceDetails>>(propertyDetails);

            DeviceIdentification deviceIdentification = property->getMessage()._deviceIdentification;
            BdAddress deviceAddress = property->getMessage()._deviceAddress;

            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate deviceHandle : %u", property->getMessage()._deviceHandle));
            ETG_TRACE_USR4(("onPropertyUpdate vendorId : %u", deviceIdentification._vendorId));
            ETG_TRACE_USR4(("onPropertyUpdate vendorIdSource : %u", deviceIdentification._vendorIdSource));

            if(nullptr != _smVoiceRecIf)
            {
               static_cast<YakinduSmVoiceRecIf*>(_smVoiceRecIf)->onDeviceConnected(deviceAddress);
            }

            _deviceVRStatus[deviceAddress] = VoiceRecognitionStatus();
            _deviceExtVRStatus[deviceAddress] = ExtVoiceRecognitionStatus();
         }
         break;

         case ON_DEVICE_DISCONNECTED:
         {
            std::shared_ptr<PropertyDetails<BdAddress>> property =
                  std::static_pointer_cast<PropertyDetails<BdAddress>>(propertyDetails);

            BdAddress deviceAddress = property->getMessage();
            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", deviceAddress.c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               static_cast<YakinduSmVoiceRecIf*>(_smVoiceRecIf)->onDeviceDisconnected(deviceAddress);
            }

            auto enhancedVRSupportedDevicesIter = _enhancedVRSupportedDevices.find(deviceAddress);
            if (_enhancedVRSupportedDevices.end() != enhancedVRSupportedDevicesIter)
            {
               _enhancedVRSupportedDevices.erase(enhancedVRSupportedDevicesIter);
            }

            auto deviceVRStatusIter = _deviceVRStatus.find(deviceAddress);
            if (deviceVRStatusIter != _deviceVRStatus.end())
            {
               _deviceVRStatus.erase(deviceAddress);
            }

            auto deviceExtVRStatusIter = _deviceExtVRStatus.find(deviceAddress);
            if (deviceExtVRStatusIter != _deviceExtVRStatus.end())
            {
               _deviceExtVRStatus.erase(deviceAddress);
            }
         }
         break;

         case BTS_UPDATE_PROXY_SERVICE_AVAILABILITY:
         {
            std::shared_ptr<PropertyDetails<evobtstackwrapper::ProxyServiceAvailability>> property =
                  std::static_pointer_cast<PropertyDetails<evobtstackwrapper::ProxyServiceAvailability>>(propertyDetails);

            if((::ccdbusif::evolution::IF_SIRI == property->getMessage()._interface) &&
                  (true == property->getMessage()._serviceAvailability))
            {
               if (true == PmCoreMainController::getInstance().getDeviceInfoHandler().isAppleDevice(property->getBdAddress()))
               {
                   SiriType supportedTypeForSiri = com::bosch::pmcommon::PmConfiguration::getInstance().getSupportedTypeForSIRI();

                  // Siri needs to be enabled for an Apple device
                  evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSiriEnableRequest(property->getBdAddress(),
                          supportedTypeForSiri, PM_DEFAULT_ACT);
               }
            }
         }
         break;

         case AM_VR_CHANNEL_ACQUISITION_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_VR_CHANNEL_ACQUISITION_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               _smVoiceRecIf->onAudioManagerEventUpdate(property->getBdAddress(), VOICEREC_CHANNEL_GRANTED);
            }
         }
         break;

         case AM_VR_CHANNEL_ACQUISITION_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_VR_CHANNEL_ACQUISITION_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               _smVoiceRecIf->onAudioManagerEventUpdate(property->getBdAddress(), VOICEREC_CHANNEL_DENIED);
            }
         }
         break;

         case AM_VR_CHANNEL_PLAYAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_VR_CHANNEL_PLAYAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               _smVoiceRecIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_SUCCESS);
            }
         }
         break;

         case AM_VR_CHANNEL_PLAYAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_VR_CHANNEL_PLAYAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               _smVoiceRecIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_FAILURE);
            }
         }
         break;

         case AM_VR_CHANNEL_STOPAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_VR_CHANNEL_STOPAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               AudioManagerEventType audioManagerEventType = STOP_AUDIO_SUCCESS;

               if (STREAMING_STOPPED_DUE_TO_EXT_TRIGGER == property->getMessage())
               {
                  audioManagerEventType = VOICEREC_CHANNEL_RELEASED;
               }

               _smVoiceRecIf->onAudioManagerEventUpdate(property->getBdAddress(), audioManagerEventType);
            }
         }
         break;

         case AM_VR_CHANNEL_STOPAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_VR_CHANNEL_STOPAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               _smVoiceRecIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_FAILURE);
            }
         }
         break;

         case AM_VR_CHANNEL_PAUSEAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_VR_CHANNEL_PAUSEAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               _smVoiceRecIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_SUCCESS);
            }
         }
         break;

         case AM_VR_CHANNEL_PAUSEAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_VR_CHANNEL_PAUSEAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            if(nullptr != _smVoiceRecIf)
            {
               _smVoiceRecIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_FAILURE);
            }
         }
         break;

         case BTS_SIGNAL_SCO_CONNECT_REQUEST:
         {
            std::shared_ptr<PropertyDetails<uint8_t>> property =
                              std::static_pointer_cast<PropertyDetails<uint8_t>>(propertyDetails);
         }
         break;

         case BTS_UPDATE_AUDIO:
         {
            std::shared_ptr<PropertyDetails<AudioDirection>> property =
                  std::static_pointer_cast<PropertyDetails<AudioDirection>>(propertyDetails);
         }
         break;

         case BTS_UPDATE_AUDIO_PATH:
         {
            std::shared_ptr<PropertyDetails<AudioPath>> property =
                              std::static_pointer_cast<PropertyDetails<AudioPath>>(propertyDetails);
         }
         break;

         case BTS_UPDATE_AUDIO_INDICATOR:
         {
            std::shared_ptr<PropertyDetails<uint8>> property =
                              std::static_pointer_cast<PropertyDetails<uint8>>(propertyDetails);
         }
         break;

         case BTS_UPDATE_SPEAKER_VOLUME:
         {
            std::shared_ptr<PropertyDetails<VolumeType>> property =
                              std::static_pointer_cast<PropertyDetails<VolumeType>>(propertyDetails);
         }
         break;

         case BTS_UPDATE_MICROPHONE_VOLUME:
         {
            std::shared_ptr<PropertyDetails<VolumeType>> property =
                              std::static_pointer_cast<PropertyDetails<VolumeType>>(propertyDetails);
         }
         break;

         case BTS_SIGNAL_VOICE_CALL_ADDED_END_VR:
         {
            std::shared_ptr<PropertyDetails<BTSVoiceCallInfo>> property =
                  std::static_pointer_cast<PropertyDetails<BTSVoiceCallInfo>>(propertyDetails);

            // Channel should be released on processing this event
            _smVoiceRecIf->onCallAdded(property->getBdAddress());
         }
         break;

         default:
         {
            ETG_TRACE_ERR(("VRController::Invalid propertyId"));
         }
         break;
      }
   }

   void VRController::getVoiceRecognitionStatusRequest(IN const BdAddress& bdAddress, IN const ActType& act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      VoiceRecognitionStatus voiceRecognitionStatus;

      auto iter = _deviceVRStatus.find(bdAddress);
      if (iter != _deviceVRStatus.end())
      {
         voiceRecognitionStatus = _deviceVRStatus[bdAddress];
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetVoiceRecognitionStatusResponse(
            pmResult, bdAddress, voiceRecognitionStatus, act);
   }

   void VRController::getExtVoiceRecognitionStatusRequest(IN const BdAddress& bdAddress, IN const ActType& act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      ExtVoiceRecognitionStatus extVoiceRecognitionStatus;

      auto iter = _deviceExtVRStatus.find(bdAddress);
      if (iter != _deviceExtVRStatus.end())
      {
         extVoiceRecognitionStatus = _deviceExtVRStatus[bdAddress];
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetExternalVRStateResponse(
            pmResult, bdAddress, extVoiceRecognitionStatus, act);
   }

   bool VRController::isVRStatusIdle(const BdAddress& bdAddress)
   {
      bool vrIdleStatus = true;

      auto iter = _deviceVRStatus.find(bdAddress);
      if (iter != _deviceVRStatus.end())
      {
         if (VR_SESSION_IDLE != _deviceVRStatus[bdAddress]._vrStatus)
            vrIdleStatus = false;
      }

      ETG_TRACE_USR4(("vrIdleStatus : %u for deviceAddress: %20s", vrIdleStatus, bdAddress.c_str()));

      return vrIdleStatus;
   }

   bool VRController::isExternalVRStatusIdle(const BdAddress& bdAddress)
   {
      bool externalVRIdleStatus = true;

      auto iter = _deviceExtVRStatus.find(bdAddress);
      if (iter != _deviceExtVRStatus.end())
      {
         if (EXT_VR_SESSION_IDLE != _deviceExtVRStatus[bdAddress]._extVRStatus)
            externalVRIdleStatus = false;
      }

      ETG_TRACE_USR4(("externalVRIdleStatus : %u for deviceAddress: %20s", externalVRIdleStatus, bdAddress.c_str()));

      return externalVRIdleStatus;
   }

   void VRController::switchToPassive(const BdAddress& deviceAddress,
         const pmaudiomanager::AudioChannel audioChannelToAcquire)
   {
      ETG_TRACE_USR1(("VRController::switchToPassive entered with deviceAddress: %s", deviceAddress.c_str()));

      if(nullptr != _smVoiceRecIf)
      {
         _smVoiceRecIf->switchToPassiveRequest(deviceAddress, audioChannelToAcquire);
      }
   }

   void VRController::stopExtendedVR(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR1(("VRController::stopExtendedVR entered with deviceAddress: %s", deviceAddress.c_str()));

      if(nullptr != _smVoiceRecIf)
      {
         _smVoiceRecIf->stopExtendedVR(deviceAddress);
      }
   }

   void VRController::switchToPassiveClientRequest(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR1(("VRController::switchToPassiveClientRequest entered with deviceAddress: %s",
            deviceAddress.c_str()));

      if(nullptr != _smVoiceRecIf)
      {
         _smVoiceRecIf->switchToPassiveClientRequest(deviceAddress);
      }
   }

   PmResult VRController::onVoiceRecognitionStatusChanged(const BdAddress& bdAddress,
         const VoiceRecognitionStatus& voiceRecognitionStatus)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      auto iter = _deviceVRStatus.find(bdAddress);
      if (iter != _deviceVRStatus.end())
      {
         _deviceVRStatus[bdAddress] = voiceRecognitionStatus;

         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnVoiceRecognitionStatusChanged(
               bdAddress, voiceRecognitionStatus);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      return pmResult;
   }

   PmResult VRController::onExtVoiceRecognitionStatusChanged(const BdAddress& bdAddress,
         const ExtVoiceRecognitionStatus& extVoiceRecognitionStatus)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      auto iter = _deviceExtVRStatus.find(bdAddress);
      if (iter != _deviceExtVRStatus.end())
      {
         _deviceExtVRStatus[bdAddress] = extVoiceRecognitionStatus;

         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnExternalVRStateChanged(
               bdAddress, extVoiceRecognitionStatus);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      return pmResult;
   }

   PmResult VRController::startStopVoiceRecognitionRequest(IN const BdAddress& bdAddress,
         IN const StartStop startStopVR)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      if(nullptr != _smVoiceRecIf)
      {
         pmResult = _smVoiceRecIf->startStopVoiceRecognitionRequest(bdAddress, startStopVR);
      }

      return pmResult;
   }

   PmResult VRController::startStopExtVoiceRecognitionRequest(IN const BdAddress& bdAddress,
         IN const StartStop startStopExtVR)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      if ((PmCoreMainController::getInstance().getCallController().isCallsInIdleState(bdAddress)) &&
            (isVRStatusIdle(bdAddress)) &&
            (! (PmCoreMainController::getInstance().getDeviceInfoHandler().isSCOEstablished(bdAddress))))
      {
         // Calls are in Idle state and SCO is not established. Hence starting External VR

         if(nullptr != _smVoiceRecIf)
         {
            pmResult = _smVoiceRecIf->startStopExtVoiceRecognitionRequest(bdAddress, startStopExtVR);
         }
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_HFP;
      }

      return pmResult;
   }


   void VRController::getEnhancedVoiceRecognitionFeatureRequest(IN const BdAddress& bdAddress,
         IN const ActType& act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      EnhancedVoiceRecognitionFeature enhancedVoiceRecognitionFeature;

      auto iter = _enhancedVRSupportedDevices.find(bdAddress);
      if (iter != _enhancedVRSupportedDevices.end())
      {
         enhancedVoiceRecognitionFeature = _enhancedVRSupportedDevices[bdAddress];
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetEnhancedVoiceRecognitionFeatureResponse(
            pmResult, bdAddress, enhancedVoiceRecognitionFeature, act);
   }

   PmResult VRController::startStopVoiceRecognitionResult(IN const BdAddress& bdAddress, IN const StartStop vrValue,
         IN const BTSResult& btsResult)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      (void)bdAddress; //To remove gen4 warning

      if (BTS_REQ_SUCCESS != btsResult._btsRequestResult)
      {
         ETG_TRACE_USR4(("VR Set Property Failure for bdAddress: %s", bdAddress.c_str()));

         ETG_TRACE_USR4(("btsResult._btsRequestResult: %u", btsResult._btsRequestResult));
         ETG_TRACE_USR4(("btsResult._errorMessage: %s", btsResult._errorMessage.c_str()));

         convertBTSResultToPmResult(btsResult, pmResult);

         if(nullptr != _smVoiceRecIf)
         {
            _smVoiceRecIf->startStopVoiceRecognitionError(bdAddress, vrValue, pmResult);
         }
      }

      return pmResult;
   }

   PmResult VRController::siriEnableResult(IN const BdAddress& bdAddress, IN const BTSResult& btsResult)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      (void)bdAddress; //To remove gen4 warning

      if (BTS_REQ_SUCCESS != btsResult._btsRequestResult)
      {
         ETG_TRACE_USR4(("siriEnableResult Failure for bdAddress: %s", bdAddress.c_str()));

         ETG_TRACE_USR4(("btsResult._btsRequestResult: %u", btsResult._btsRequestResult));
         ETG_TRACE_USR4(("btsResult._errorMessage: %s", btsResult._errorMessage.c_str()));
      }

      return pmResult;
   }

   PmResult VRController::siriSetNRResult(IN const BdAddress& bdAddress, IN const BTSResult& btsResult)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      (void)bdAddress; //To remove gen4 warning

      if (BTS_REQ_SUCCESS != btsResult._btsRequestResult)
      {
         ETG_TRACE_USR4(("siriSetNRResult Failure for bdAddress: %s", bdAddress.c_str()));

         ETG_TRACE_USR4(("btsResult._btsRequestResult: %u", btsResult._btsRequestResult));
         ETG_TRACE_USR4(("btsResult._errorMessage: %s", btsResult._errorMessage.c_str()));
      }

      return pmResult;
   }

   void VRController::prepareAudioRouteResponse(
         IN std::shared_ptr<PmCoreIfMessage_PrepareAudioRouteResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("VRController::prepareAudioRouteResponse- BdAddress(): %s",
            pmCoreIfMessage->getBdAddress().c_str()));
   }

   void VRController::playAudioResponse(IN std::shared_ptr<PmCoreIfMessage_PlayAudioResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("VRController::playAudioResponse- BdAddress(): %s",
            pmCoreIfMessage->getBdAddress().c_str()));
   }

   void VRController::stopAudioResponse(IN std::shared_ptr<PmCoreIfMessage_StopAudioResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("VRController::stopAudioResponse- BdAddress(): %s",
            pmCoreIfMessage->getBdAddress().c_str()));
   }

} // namespace pmcore
