/**
 * @file PmVoiceRecHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmVoiceRecHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice recognition
 *
 * @ingroup PmCore
 */

#ifndef PmVoiceRecHandler_h
#define PmVoiceRecHandler_h

#include "EvoBtStackWrapper.h"
#include "PmCoreIfMessageResult.h"

#include "../src-gen/PmVoiceRecHandlerSm.h"
#include "AsfTimer.h"

namespace pmcore
{
   class PmVoiceRecHandler final: public PmVoiceRecHandlerSm
   {
   public:
      /**
       * Parameterized Constructor of PmVoiceRecHandler class
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmVoiceRecHandler(const BdAddress& deviceAddress);

      /**
       * Destructor of PmVoiceRecHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~PmVoiceRecHandler();

      /**
       * This function is used to get the deviceAddress of the SM instance
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BdAddress - device address
       */
      const BdAddress& getDeviceAddress() { return _deviceAddress; }

      /**
       * Member function overridden from ISmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult startStopVoiceRecognitionRequest(IN const StartStop startStopVR);

      /**
       * Member function overridden from ISmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult startStopExtVoiceRecognitionRequest(IN const StartStop startStopVR);

      /**
       * This function is used to post the VR Status to state machine.
       *
       * @param[in] vrStatus - VR status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAgVoiceRecStateUpdate(IN const VRStatus vrStatus);

      /**
       * This function is used to post the AudioManager event to state machine.
       *
       * @param[in] amEvent - AudioManager event update
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerEventUpdate(IN const AudioManagerEventType amEvent);

      /**
       * This function is used to post the SCO connection status to state machine.
       *
       * @param[in] scoStatus - SCO connection status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onScoConnectionStatusChanged(IN const SCOStatus scoStatus);

      /**
       * This method is used to notify VR SM about the received new call.
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onCallAdded();

      /**
       * This method is used to notify timer elapsed callback to the call Controller state machine
       *
       * @param[in] data              - timer data
       * @param[in] timerCallbackData - timer callback data
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void timerElapsed(com::bosch::pmcommon::TimerData<std::string,ms_t,RepeatCount> data,
            com::bosch::pmcommon::AsfTimerCallbackData timerCallbackData);

      /**
       * This method is used to switch a device to Passive
       *
       * @param[in]  audioChannelToAcquire- The audio channel to be acquired after stopping VR
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveRequest(const pmaudiomanager::AudioChannel audioChannelToAcquire);

      /**
       * Stop Extended VR on Media Ready to Play
       *
       * @param[in]
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopExtendedVR();

      /**
       * This method is used to handle VR Activation/DeActivation failure received from BT stack
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void startStopVoiceRecognitionError(const StartStop vrValue, PmResult& pmResult);

      /**
       * This method is used to switch a device to Passive based on client request
       *
       * @param[in
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveClientRequest();

      /**
       * This method is used to update the SM about the device disconnection
       *
       * @param[in
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onDeviceDisconnection();

   private:
      /**
       * Constructor of PmVoiceRecHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmVoiceRecHandler();

      /**
       * SmUserActionIf class inherits the functions of SCI_UserAction_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmUserActionIf final: public SCI_UserAction_OCB
      {
      public:
         SmUserActionIf(PmVoiceRecHandler& pmVoiceRecHandler): _pmVoiceRecHandler(pmVoiceRecHandler){}

         ~SmUserActionIf(){}

         virtual void activateVR() override;

         virtual void deactivateVR() override;

         virtual void tempFunc(sc_boolean success) override;

      private:
         PmVoiceRecHandler& _pmVoiceRecHandler;
      };

      /**
       * SmAMRequestIf class inherits the functions of SCI_AMRequest_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAMRequestIf final: public SCI_AMRequest_OCB
      {
      public:
         SmAMRequestIf(PmVoiceRecHandler& pmVoiceRecHandler): _pmVoiceRecHandler(pmVoiceRecHandler){}

         ~SmAMRequestIf(){}

         virtual void requestAudioChannel() override;

         virtual void playHfAudio(SCOStatus scoStatus) override;

         virtual void stopOrPauseHfAudio(AudioChannel audioChannelId) override;

         virtual void stopAudio() override;

      private:
         PmVoiceRecHandler& _pmVoiceRecHandler;
      };

      /**
       * SmSmHandlerIf class inherits the functions of SCI_SmHandler_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmSmHandlerIf final: public SCI_SmHandler_OCB
      {
      public:
         SmSmHandlerIf(PmVoiceRecHandler& pmVoiceRecHandler): _pmVoiceRecHandler(pmVoiceRecHandler){}

         ~SmSmHandlerIf(){}

         virtual void updateVRState(VRStatusEnumType vrState) override;

         virtual void updateVRStartError(void * pmResult) override;

         virtual void updateVRStopError(void * pmResult) override;

         virtual void updateVRInterruptedReason(VRInterruptedReason vrInterruptedReason) override;

         virtual void updateExternalVRState(ExternalVRStatusEnumType extVRState) override;

         virtual void updateExternalVRStartError(void * pmResult) override;

         virtual void updateExtVRInterruptedReason(VRInterruptedReason extVRInterruptedReason) override;

         virtual void updateSwitchToPassiveError(void * pmResult) override;

         virtual void updateSwitchToPassiveNotHandledError() override;

         virtual void updateswitchToPassiveResponse(sc_string msg) override;

         virtual void processVRStopped() override;

         virtual void postGetCallsRequest2Btstack() override;

      private:
         PmVoiceRecHandler& _pmVoiceRecHandler;
      };

      /**
       * SmTimerIf class inherits the functions of SCI_Timer_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmTimerIf final: public SCI_Timer_OCB
      {
      public:
         SmTimerIf(PmVoiceRecHandler& pmVoiceRecHandler): _pmVoiceRecHandler(pmVoiceRecHandler){}

         ~SmTimerIf(){}

         virtual void stopExtendVRTimer() override;

         virtual void startExtendVRTimer() override;

      private:
         PmVoiceRecHandler& _pmVoiceRecHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final : public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(PmVoiceRecHandler& pmVoiceRecHandler):
            _pmVoiceRecHandler(pmVoiceRecHandler){}

         virtual ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         PmVoiceRecHandler&    _pmVoiceRecHandler;
      };

      BdAddress         _deviceAddress;   /**< device address */
      SmUserActionIf*   _smUserActionIf;  /**< pointer for UserAction interface */
      SmAMRequestIf*    _smAMRequestIf;   /**< pointer for AudioManagerRequest interface */
      SmSmHandlerIf*    _smSmHandlerIf;   /**< pointer for SmHandler interface */
      SmTimerIf*        _smTimerIf;       /**< pointer for Timer interface */
      SmTraceObserverIf<PmVoiceRecHandlerSmStates>* _smTraceObserverIf; /**< pointer for TraceObserver interface */

      com::bosch::pmcommon::ITimer<PmVoiceRecHandler, std::string, ms_t, RepeatCount>* _extendVRSessionTimer;

      // This is used to avoid the unnecessary posting of De-activation VR request to the stack before
      // receiving the response for the same
      // TODO: _deActVRinProgress - should be reset on receiving the error for VR Deactivation
      // ie., when raising the event "raise_vR_DEACTIVATION_FAILURE".
      bool _deActVRinProgress;

      // This parameter is used to avoid the repetitive updates if the same state from State chart
      static VRStatusEnumType _vrState;

      static bool _temp;
   };
}

#endif //YakinduSmVoiceCallIf_h
