/**
 * @file ISmVoiceRecIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the interface definition of the ISmVoiceRecIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice recognition
 *
 * @ingroup PmCore
 */

#ifndef ISmVoiceRecIf_h
#define ISmVoiceRecIf_h

#include "PmCoreIfTypes.h"
#include "BtStackWrapperTypes.h"
#include "PmCoreIfMessageRequest.h"
#include "PmCoreIfMessageResult.h"

namespace pmcore
{
   class ISmVoiceRecIf
   {
   public:

      /**
       * Destructor of ISmVoiceRecIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual ~ISmVoiceRecIf(){}

      // Request calls
      /**
       * This method is used to post the StartStop VoiceRecognition Request to the VR controller state machine
       *
       * @param[in] bdAddress    - BT address of the device
       * @param[in] startStopVR  - Flag for start stop VR
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      virtual PmResult startStopVoiceRecognitionRequest(IN const BdAddress& bdAddress,
            IN const StartStop startStopVR) = 0;

      /**
       * This method is used to post the StartStop External VoiceRecognition Request to the VR controller state machine
       *
       * @param[in] bdAddress    - BT address of the device
       * @param[in] startStopExtVR  - Flag for start stop External VR
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      virtual PmResult startStopExtVoiceRecognitionRequest(IN const BdAddress& bdAddress,
            IN const StartStop startStopExtVR) = 0;

      // **************************************************************************************************
      // Result calls
      /**
       * This function is used to post the VR Status to state machine.
       *
       * @param[in] deviceAddress - device address
       * @param[in] vrStatus - VR status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAgVoiceRecStateUpdate(IN const BdAddress& deviceAddress, IN const VRStatus vrStatus) = 0;

      /**
       * This method is used to post the AudioManager update to the voice call state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] amEvent - audio manager event update
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
            IN const AudioManagerEventType amEvent) = 0;

      /**
       * This method is used to update the SCO status to the voice call state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] scoStatus - sco status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onScoConnectionStatusChanged(IN const BdAddress& deviceAddress, IN const SCOStatus scoStatus) = 0;

      /**
       * This method is used to notify VR SM about the received new call.
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onCallAdded(IN const BdAddress& deviceAddress) = 0;

      /**
       * This method is used to switch a device to Passive and hold the VR channel
       *
       * @param[in] deviceAddress - device address
       * @param[in] audioChannelToAcquire - The audio channel to be acquired after stopping VR
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToPassiveRequest(IN const BdAddress& deviceAddress,
            const pmaudiomanager::AudioChannel audioChannelToAcquire) = 0;

      /**
       * This method is used to switch a device to Passive based on the client request and hence the VR should be
       * stopped and the acquired channel should be released.
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToPassiveClientRequest(IN const BdAddress& deviceAddress) = 0;

      /**
       * This method is used to stop extended VR when a media ready to play status is received from Media player
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void stopExtendedVR(IN const BdAddress& deviceAddress) = 0;

      /**
       * This method is used to handle VR Activation/DeActivation failure received from BT stack
       *
       * @param[in] deviceAddress - device address
       * @param[in] vrValue       - VR value sent in DBUS SetProprety request
       * @param[in] PmResult      - device address
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void startStopVoiceRecognitionError(IN const BdAddress& deviceAddress, IN const StartStop vrValue,
            IN PmResult& pmResult) = 0;
   };
}

#endif //ISmVoiceRecIf_h
