/**
 * @file PmRingtoneHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmRingtoneHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#include "PmRingtoneHandler.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PropertyDetails.h"
#include "PmAudioManagerWrapper.h"
#include "PmCoreMainController.h"
#include "DeviceInfoHandler.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmRingtoneHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   PmRingtoneHandler::PmRingtoneHandler()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::PmRingtoneHandler Constructor"));
      _deviceAddress = "";
      _smUserActionIf = nullptr;
      _smSmHandlerIf = nullptr;
      _smAudioManagerRequestIf = nullptr;
      _smTraceObserverIf = nullptr;
   }

   PmRingtoneHandler::PmRingtoneHandler(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::PmRingtoneHandler Constructor"));

      _deviceAddress = deviceAddress;

      _smUserActionIf = new PmRingtoneHandler::SmUserActionIf(*this);

      if(nullptr != _smUserActionIf)
      {
         setSCI_UserAction_OCB(_smUserActionIf);
      }

      _smSmHandlerIf = new PmRingtoneHandler::SmSmHandlerIf(*this);

      if(nullptr != _smSmHandlerIf)
      {
         setSCI_SmHandler_OCB(_smSmHandlerIf);
      }

      _smAudioManagerRequestIf = new PmRingtoneHandler::SmAudioManagerRequestIf(*this);

      if(nullptr != _smAudioManagerRequestIf)
      {
         setSCI_AudioManagerRequest_OCB(_smAudioManagerRequestIf);
      }

      _smTraceObserverIf = new PmRingtoneHandler::SmTraceObserverIf<PmRingtoneHandlerSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      //initialize the state machine
      init();

      enter();
   }

   PmRingtoneHandler::~PmRingtoneHandler()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler: Destructor"));

      if(nullptr != _smUserActionIf)
      {
         delete _smUserActionIf;
         _smUserActionIf = nullptr;
      }

      if(nullptr != _smSmHandlerIf)
      {
         delete _smSmHandlerIf;
         _smSmHandlerIf = nullptr;
      }

      if(nullptr != _smAudioManagerRequestIf)
      {
         delete _smAudioManagerRequestIf;
         _smAudioManagerRequestIf = nullptr;
      }

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }
   }

   void PmRingtoneHandler::onAudioManagerEventUpdate(IN const AudioManagerEventType amEvent)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::onAudioManagerEventUpdate : %u",
            ETG_CENUM(AudioManagerEventType, amEvent)));

      switch(amEvent)
      {
         case RINGTONE_CHANNEL_GRANTED:
            getSCI_AudioManagerResponse()->raise_rINGTONE_CHANNEL_GRANTED();
            break;
         case RINGTONE_CHANNEL_RELEASED:
            getSCI_AudioManagerResponse()->raise_rINGTONE_CHANNEL_RELEASED();
            break;
         case RINGTONE_CHANNEL_DENIED:
            getSCI_AudioManagerResponse()->raise_rINGTONE_CHANNEL_DENIED();
            break;
         case PLAY_AUDIO_SUCCESS:
            getSCI_AudioManagerResponse()->raise_pLAY_AUDIO_SUCCESS();
            break;
         case PLAY_AUDIO_FAILURE:
            getSCI_AudioManagerResponse()->raise_pLAY_AUDIO_FAILURE();
            break;
         case PAUSE_AUDIO_SUCCESS:
            getSCI_AudioManagerResponse()->raise_pAUSE_AUDIO_SUCCESS();
            break;
         case PAUSE_AUDIO_FAILURE:
            getSCI_AudioManagerResponse()->raise_pAUSE_AUDIO_FAILURE();
            break;
         case STOP_AUDIO_SUCCESS:
            getSCI_AudioManagerResponse()->raise_sTOP_AUDIO_SUCCESS();
            break;
         case STOP_AUDIO_FAILURE:
            getSCI_AudioManagerResponse()->raise_sTOP_AUDIO_FAILURE();
            break;
         default:
            ETG_TRACE_ERR(("onAudioManagerEventUpdate Invalid Event"));
            break;
      }
   }

   void PmRingtoneHandler::playRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::playRingtoneRequest"));

      getSCI_UserAction()->set_ringtoneId(pmCoreIfMessage->getRingtoneId());
      getSCI_UserAction()->set_playCount(pmCoreIfMessage->getPlayCount());

      getSCI_UserAction()->raise_pLAY_RINGTONE();

      //Send success response to requested client
      PmResult pmResult(PM_RESULT_OK, "");
      pmCoreIfMessage->setPmResult(pmResult);
   }

   void PmRingtoneHandler::switchToWaitForStopAudioResponse(
         IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::switchToWaitForStopAudioResponse"));

      getSCI_UserAction()->set_ringtoneId(pmCoreIfMessage->getRingtoneId());
      getSCI_UserAction()->set_playCount(pmCoreIfMessage->getPlayCount());

      getSCI_UserAction()->raise_wAIT_FOR_STOP_RESP_AND_PLAY_RINGTONE();

      //Send success response to requested client
      PmResult pmResult(PM_RESULT_OK, "");
      pmCoreIfMessage->setPmResult(pmResult);
   }

   void PmRingtoneHandler::switchToWaitForPlayAudioResponse(
         IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::switchToWaitForPlayAudioResponse"));

      getSCI_UserAction()->set_ringtoneId(pmCoreIfMessage->getRingtoneId());
      getSCI_UserAction()->set_playCount(pmCoreIfMessage->getPlayCount());

      getSCI_UserAction()->raise_wAIT_FOR_PLAY_RT_RESP_AND_PLAY_RINGTONE();

      //Send success response to requested client
      PmResult pmResult(PM_RESULT_OK, "");
      pmCoreIfMessage->setPmResult(pmResult);
   }

   void PmRingtoneHandler::switchToWaitForChannelAcquisition(
         IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::switchToWaitForChannelAcquisition"));

      getSCI_UserAction()->set_ringtoneId(pmCoreIfMessage->getRingtoneId());
      getSCI_UserAction()->set_playCount(pmCoreIfMessage->getPlayCount());

      getSCI_UserAction()->raise_wAIT_FOR_RT_CH_ACQUISITION_AND_PLAY_RINGTONE();

      //Send success response to requested client
      PmResult pmResult(PM_RESULT_OK, "");
      pmCoreIfMessage->setPmResult(pmResult);
   }

   void PmRingtoneHandler::pauseRingtoneRequest()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::pauseRingtoneRequest"));
      getSCI_UserAction()->raise_pAUSE_RINGTONE();
   }

   void PmRingtoneHandler::stopRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::stopRingtoneRequest"));

      getSCI_UserAction()->raise_sTOP_RINGTONE(pmCoreIfMessage.get());

      //Send success response to requested client
      PmResult pmResult(PM_RESULT_OK, "");
      pmCoreIfMessage->setPmResult(pmResult);
   }

   void PmRingtoneHandler::SmAudioManagerRequestIf::requestRingtoneAudio()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::requestRingtoneAudio"));

      BdAddress bdAddress = _pmRingtoneHandler.getDeviceAddress();

      //Request origin from CallController
      if(0 != bdAddress.compare(DEVICE_ADDRESS_ALL))
      {
         //Request RingTone channel to play the vehicle ring tone
         PmAudioManagerWrapper::getInstance().sendPrepareAudioRoute(bdAddress,
               pmaudiomanager::AM_VEHICLERINGTONE, CALL_CONTROLLER);
      }
      //Request origin from Client
      else
      {
         //Request RingTone channel to play the vehicle ring tone
         PmAudioManagerWrapper::getInstance().sendPrepareAudioRoute(bdAddress,
               pmaudiomanager::AM_VEHICLERINGTONE, MAIN_CONTROLLER);
      }
   }

   void PmRingtoneHandler::SmAudioManagerRequestIf::playRingtone()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::playRingtone"));

      BdAddress bdAddress = _pmRingtoneHandler.getDeviceAddress();
      FilePath filePath;

      //Request origin from Client
      if(0 == bdAddress.compare(DEVICE_ADDRESS_ALL))
      {
         PmCoreMainController::getInstance().getPmSettings().getRingtoneName(
               _pmRingtoneHandler.getSCI_UserAction()->get_ringtoneId(), filePath);

         //Requesting audio manager to play audio
         PmAudioManagerWrapper::getInstance().sendPlayAudio(bdAddress, filePath,
               MAIN_CONTROLLER, _pmRingtoneHandler.getSCI_UserAction()->get_playCount());
      }
      //Request origin from CallController
      else
      {
         PmCoreMainController::getInstance().getPmSettings().getRingtoneName(bdAddress, filePath);
         //Requesting audio manager to play audio

         PmAudioManagerWrapper::getInstance().sendPlayAudio(bdAddress, filePath,
               CALL_CONTROLLER, _pmRingtoneHandler.getSCI_UserAction()->get_playCount());
      }
   }

   void PmRingtoneHandler::SmAudioManagerRequestIf::pauseRingtone()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::pauseRingtone"));

      //Requesting audio manager to pause audio
      //Request origin from Client
      PmAudioManagerWrapper::getInstance().sendPauseAudio(_pmRingtoneHandler.getDeviceAddress());
   }

   void PmRingtoneHandler::SmAudioManagerRequestIf::resumeRingtone()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::resumeRingtone"));

      FilePath filePath;
      PmCoreMainController::getInstance().getPmSettings().getRingtoneName(
            _pmRingtoneHandler.getDeviceAddress(), filePath);

      //Requesting audio manager to play audio
      PmAudioManagerWrapper::getInstance().sendPlayAudio(_pmRingtoneHandler.getDeviceAddress(), filePath,
            MAIN_CONTROLLER);
   }

   void PmRingtoneHandler::SmUserActionIf::stopRingtone(PmCoreIfMessage_StopRingtoneRequest * msg)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::stopRingtone"));

      //Request origin from CallController
      if(0 != (msg->getBdAddress().compare(DEVICE_ADDRESS_ALL)))
      {
         //Requesting audio manager to stop audio
         PmAudioManagerWrapper::getInstance().sendStopAudio(msg->getBdAddress(), CALL_CONTROLLER);
      }
      //Request origin from Client
      else
      {
         //Requesting audio manager to stop audio
         PmAudioManagerWrapper::getInstance().sendStopAudio(msg->getBdAddress(), MAIN_CONTROLLER);
      }
   }

   void PmRingtoneHandler::setRingtoneMuteStateRequest(IN const BdAddress& bdAddress, IN const MuteState ringtoneMuteState)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::setRingtoneMuteStateRequest"));
      (void) bdAddress;

      if(ringtoneMuteState)
      {
         getSCI_UserAction()->raise_mUTE_RINGTONE();
      }
      else
      {
         getSCI_UserAction()->raise_uNMUTE_RINGTONE();
      }
   }

   void PmRingtoneHandler::switchToActiveRequest()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::switchToActiveRequest"));
      getSCI_SmHandler()->raise_sWITCH_TO_ACTIVE();
   }

   void PmRingtoneHandler::processSourceSwitch()
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::processSourceSwitch"));
      getSCI_SmHandler()->raise_mOVE_TO_IDLE();
   }

   void PmRingtoneHandler::SmSmHandlerIf::sendPlayResponse(PmResultMessage result)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::sendPlayResponse"));

      if(_pmRingtoneHandler.getDeviceAddress().compare("") != 0)
      {
         //UnMute is successful
         if(result.compare("PLAY_OK") ==0)
         {
            PmCoreMainController::getInstance().getRingtoneController().updateRingtoneMuteState(
                  _pmRingtoneHandler.getDeviceAddress(), false);
         }
         else if(result.compare("PLAY_ERROR") == 0)
         {
            //TODO: Send error event
         }
      }
   }

   void PmRingtoneHandler::SmSmHandlerIf::sendPauseResponse(PmResultMessage result)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::sendPauseResponse"));

      //Mute is successful
      if(result.compare("PAUSE_OK") == 0)
      {
         PmCoreMainController::getInstance().getRingtoneController().updateRingtoneMuteState(
               _pmRingtoneHandler.getDeviceAddress(), true);
      }
      else if(result.compare("PAUSE_ERROR") == 0)
      {
         //TODO: Send error event
      }
   }

   void PmRingtoneHandler::SmSmHandlerIf::updateRingtoneStatus(RingtonePlaybackStateSMType smState)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::updateRingtoneStatus"));

      if (DEVICE_ADDRESS_ALL == _pmRingtoneHandler._deviceAddress)
      {
         PmCoreMainController::getInstance().getRingtoneController().
               onRingtonePlaybackStateChanged(smState);
      }
   }

   template<typename T>
   void PmRingtoneHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::stateEntered()::%u for deviceAddress: %20s",
            ETG_CENUM(PmRingtoneHandlerSmStates, state), _pmRingtoneHandler.getDeviceAddress().c_str()));
   }

   template<typename T>
   void PmRingtoneHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("PmRingtoneHandler::stateExited()::%u for deviceAddress: %20s",
            ETG_CENUM(PmRingtoneHandlerSmStates, state), _pmRingtoneHandler.getDeviceAddress().c_str()));
   }
} // namespace pmcore
