/**
 * @file ISmRingtoneIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the interface definition of the SmRingtoneIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#ifndef ISmRingtoneIf_h
#define ISmRingtoneIf_h

#include "PmCoreIfTypes.h"
#include "BtStackWrapperTypes.h"
#include "PmCoreIfMessageRequest.h"

namespace pmcore
{
   class ISmRingtoneIf
   {
   public:

      /**
       * Destructor of ISmRingtoneIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual ~ISmRingtoneIf(){}

      /**
       * This method is used to post the AudioManager update to the voice call state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] amEvent - audio manager event update
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
            IN const AudioManagerEventType amEvent) = 0;

      // Request calls
      /**
       * This method is used to post the Play Ringtone request to the Ringtone state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for PlayRingToneRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void playRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to process the Play Ringtone request after stop audio response is received
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToWaitForStopAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage) = 0;

      /**
       * This method informs SM to wait for Play audio response to replay the RT
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToWaitForPlayAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to process the Play Ringtone request after RT channel acquisition
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToWaitForChannelAcquisition(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to pause request to the RT SM and also to know whether the
       * corresponding RT SM is waiting for stop audio response
       *
       * @param[in] deviceAddress - Device address of the device for which pause ringtone need to be processed
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void pauseRingtoneRequest(IN const BdAddress deviceAddress) = 0;

      /**
       * This method is used to post the Stop Ringtone Request to the Ringtone state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for StopRingToneRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void stopRingtoneRequest(IN std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the Set ringTone to the Ringtone state machine
       *
       * @param[in] ringtoneMuteState - Flag for MuteState
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setRingtoneMuteStateRequest(IN const BdAddress& bdAddress, IN const MuteState ringtoneMuteState) = 0;

      /**
       * This method is used to post the switch to Active request to the SM
       *
       * @param[in] bdAddress - Device address of the connected device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToActiveRequest(IN const BdAddress& bdAddress) = 0;


      /**
       * This method is used to inform the SM about the acquisition of a channel other than Ring-tone channel
       * for the given device address
       *
       * @param[in] bdAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void processSourceSwitch(IN const BdAddress& bdAddress) = 0;

      // **************************************************************************************************
      // Result calls
   };
}

#endif //ISmRingtoneIf_h
