/**
 * @file YakinduSmVolumeControlIf.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the YakinduSmVolumeControlIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#include "YakinduSmVolumeControlIf.h"
#include "RVCHandler.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/YakinduSmVolumeControlIf.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   YakinduSmVolumeControlIf::YakinduSmVolumeControlIf():_deviceToRVCHandlerSmMap(),_deviceIdToDevAddrMap()
   {
   }

   YakinduSmVolumeControlIf::~YakinduSmVolumeControlIf()
   {
      _deviceToRVCHandlerSmMap.clear();
      _deviceIdToDevAddrMap.clear();
   }

   BdAddress YakinduSmVolumeControlIf::getBdAddressFromDeviceHandle(IN const DeviceHandle& deviceHandle)
   {
      ETG_TRACE_USR4(("YakinduSmVolumeControlIf::getBdAddressFromDeviceHandle"));

      BdAddress bdAddr("");
      auto it = _deviceIdToDevAddrMap.find(deviceHandle);
      if(it != _deviceIdToDevAddrMap.end())
      {
         bdAddr = it->second;
      }
      return bdAddr;
   }

   void YakinduSmVolumeControlIf::onDeviceConnected(IN const BdAddress& deviceAddress, IN const DeviceHandle& deviceHandle)
   {
      ETG_TRACE_USR4(("YakinduSmVolumeControlIf::onDeviceConnected"));

      auto it = _deviceToRVCHandlerSmMap.find(deviceAddress);

      if(it == _deviceToRVCHandlerSmMap.end())
      {
         RVCHandler* rvcHandler = new RVCHandler(deviceAddress);
         if(nullptr != rvcHandler)
         {
            _deviceToRVCHandlerSmMap.emplace(deviceAddress, rvcHandler);
            _deviceIdToDevAddrMap.emplace(deviceHandle, deviceAddress);
            //Sending device connect event
            rvcHandler->onDeviceConnect();
         }
      }
      else
      {
         ETG_TRACE_USR4(("SM already exists for device %s", deviceAddress.c_str()));
      }
   }

   void YakinduSmVolumeControlIf::onDeviceDisconnected(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVolumeControlIf::onDeviceDisconnected"));

      //Sending SLC OFF event
      auto it = _deviceToRVCHandlerSmMap.find(deviceAddress);
      if(it != _deviceToRVCHandlerSmMap.end())
      {
         delete it->second;
         _deviceToRVCHandlerSmMap.erase(it);
      }

      for (auto iter = _deviceIdToDevAddrMap.begin(); iter != _deviceIdToDevAddrMap.end(); ++iter)
      {
         if (iter->second == deviceAddress)
         {
            _deviceIdToDevAddrMap.erase(iter);
            break;
         }
      }
   }

   void YakinduSmVolumeControlIf::onVolumeUpdateFromAm(IN const BdAddress& deviceAddress, IN const VolumeLevel& volume)
   {
      ETG_TRACE_USR4(("YakinduSmVolumeControlIf::onVolumeUpdateFromAm"));

      auto iter = _deviceToRVCHandlerSmMap.find(deviceAddress);
      if(iter != _deviceToRVCHandlerSmMap.end())
      {
         iter->second->onVolumeUpdateFromAm(volume);
      }
   }

   void YakinduSmVolumeControlIf::onVolumeUpdateFromAg(IN const BdAddress& deviceAddress, IN const VolumeLevel& volume)
   {
      ETG_TRACE_USR4(("YakinduSmVolumeControlIf::onVolumeUpdateFromAg"));

      auto it = _deviceToRVCHandlerSmMap.find(deviceAddress);
      if(it != _deviceToRVCHandlerSmMap.end())
      {
         it->second->onVolumeUpdateFromAg(volume);
      }
   }

   void YakinduSmVolumeControlIf::onTimerEventUpdate(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("YakinduSmVolumeControlIf::onTimerEventUpdate"));

      auto it = _deviceToRVCHandlerSmMap.find(deviceAddress);
      if(it != _deviceToRVCHandlerSmMap.end())
      {
         it->second->onTimerEventUpdate();
      }
   }
}
