/**
 * @file RVCHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the RVCHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#include "RVCHandler.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PmCoreIfMessageCreator.h"
#include "PropertyDetails.h"
#include "PmAudioManagerWrapper.h"
#include "PmCoreMainController.h"
#include "DeviceInfoHandler.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/RVCHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   RVCHandler::RVCHandler()
   {
      _deviceAddress = "";
      _smAgVolCtrlIf = nullptr;
      _smAudioManagerVolCtrlIf = nullptr;
      _smVolCtrlSmHandlerIf = nullptr;
      _smVolCtrlTimerIf = nullptr;
      _smTraceObserverIf = nullptr;
      _agLastsetValue = COUNT_ZERO;
      _agCurrentVolume = COUNT_ZERO;
      _amLastsetValue = COUNT_ZERO;
      _amCurrentVolume = COUNT_ZERO;
   }

   RVCHandler::RVCHandler(const BdAddress& deviceAddress)
   {
      _deviceAddress = deviceAddress;

      _smAgVolCtrlIf = new RVCHandler::SmAgVolCtrlIf(*this);

      if(nullptr != _smAgVolCtrlIf)
      {
         setSCI_AgVolCtrlIf_OCB(_smAgVolCtrlIf);
      }

      _smAudioManagerVolCtrlIf = new RVCHandler::SmAudioManagerVolCtrlIf(*this);

      if(nullptr != _smAudioManagerVolCtrlIf)
      {
         setSCI_AudioManagerVolCtrlIf_OCB(_smAudioManagerVolCtrlIf);
      }

      _smVolCtrlSmHandlerIf = new RVCHandler::SmVolCtrlSmHandlerIf(*this);

      if(nullptr != _smVolCtrlSmHandlerIf)
      {
         setSCI_VolCtrlSmHandlerIf_OCB(_smVolCtrlSmHandlerIf);
      }

      _smVolCtrlTimerIf = new RVCHandler::SmVolCtrlTimerIf(*this);

      if(nullptr != _smVolCtrlTimerIf)
      {
         setSCI_VolCtrlTimerIf_OCB(_smVolCtrlTimerIf);
      }

      _smTraceObserverIf = new RVCHandler::SmTraceObserverIf<RVCSmStates>(*this);

      if(nullptr != _smTraceObserverIf)
      {
         setTraceObserver(_smTraceObserverIf);
      }

      _agLastsetValue = COUNT_ZERO;
      _agCurrentVolume = COUNT_ZERO;
      _amLastsetValue = COUNT_ZERO;
      _amCurrentVolume = COUNT_ZERO;

      //initialize the state machine
      init();

      enter();
   }

   RVCHandler::~RVCHandler()
   {
      if(nullptr != _smAgVolCtrlIf)
      {
         delete _smAgVolCtrlIf;
         _smAgVolCtrlIf = nullptr;
      }

      if(nullptr != _smAudioManagerVolCtrlIf)
      {
         delete _smAudioManagerVolCtrlIf;
         _smAudioManagerVolCtrlIf = nullptr;
      }

      if(nullptr != _smVolCtrlSmHandlerIf)
      {
         delete _smVolCtrlSmHandlerIf;
         _smVolCtrlSmHandlerIf = nullptr;
      }

      if(nullptr != _smVolCtrlTimerIf)
      {
         delete _smVolCtrlTimerIf;
         _smVolCtrlTimerIf = nullptr;
      }

      if(nullptr != _smTraceObserverIf)
      {
         delete _smTraceObserverIf;
         _smTraceObserverIf = nullptr;
      }
   }

   VolumeLevel RVCHandler::convertAmVolumeToAgVolume(IN const VolumeLevel& volume)
   {
      ETG_TRACE_USR4(("convertAudioVolumeToAgVolume entered"));
      pmaudiomanager::GeniviAMVolumeLevel minAMVolumeLevel = com::bosch::pmcommon::PmConfiguration::getInstance().getMinAMVolumeLevel();
      pmaudiomanager::GeniviAMVolumeLevel maxAMVolumeLevel = com::bosch::pmcommon::PmConfiguration::getInstance().getMaxAMVolumeLevel();

      VolumeLevel amRange = static_cast<VolumeLevel>((maxAMVolumeLevel > minAMVolumeLevel) ? (maxAMVolumeLevel - minAMVolumeLevel) : 1);
      VolumeLevel hfRange = MAX_HFP_SCO_VOICE_CALL_VOLUME - MIN_HFP_SCO_VOICE_CALL_VOLUME;
      VolumeLevel hfOffset = static_cast<VolumeLevel>((hfRange * (volume - minAMVolumeLevel)) / amRange);
      VolumeLevel agVol = static_cast<VolumeLevel>(MIN_HFP_SCO_VOICE_CALL_VOLUME + hfOffset);
      ETG_TRACE_USR4(("Volume converted from AM:%d to AG:%d",volume,agVol));
      return agVol;
   }

   VolumeLevel RVCHandler::convertAgVolumeToAudioVolume(IN const VolumeLevel& volume)
   {
      ETG_TRACE_USR4(("convertAgVolumeToAudioVolume entered"));
      pmaudiomanager::GeniviAMVolumeLevel minAMVolumeLevel = com::bosch::pmcommon::PmConfiguration::getInstance().getMinAMVolumeLevel();
      pmaudiomanager::GeniviAMVolumeLevel maxAMVolumeLevel = com::bosch::pmcommon::PmConfiguration::getInstance().getMaxAMVolumeLevel();

      VolumeLevel amRange = static_cast<VolumeLevel>((maxAMVolumeLevel > minAMVolumeLevel) ? (maxAMVolumeLevel - minAMVolumeLevel) : 1);
      VolumeLevel hfRange = MAX_HFP_SCO_VOICE_CALL_VOLUME - MIN_HFP_SCO_VOICE_CALL_VOLUME;
      VolumeLevel amOffset = static_cast<VolumeLevel>((amRange * (volume - MIN_HFP_SCO_VOICE_CALL_VOLUME)) / hfRange);
      VolumeLevel amVol = static_cast<VolumeLevel>(minAMVolumeLevel + amOffset);
      ETG_TRACE_USR4(("Volume converted from AG:%d to AM:%d",volume,amVol));
      return amVol;
   }

   void RVCHandler::onDeviceConnect()
   {
      ETG_TRACE_USR4(("RVCHandler::onDeviceConnect() entered"));
      PmAudioManagerWrapper::getInstance().getAudioSinkVolume(_deviceAddress, GAM_SINK_ID);
   }

   void RVCHandler::onVolumeUpdateFromAm(IN const VolumeLevel& volume)
   {
      ETG_TRACE_USR4(("RVCHandler::onVolumeUpdateFromAm() entered"));

      //Storing the incoming value
      _amCurrentVolume = volume;

      getSCI_AudioManagerVolCtrlUpdate()->raise_vOLUME_CHANGED();
   }

   void RVCHandler::onVolumeUpdateFromAg(IN const VolumeLevel& volume)
   {
      ETG_TRACE_USR4(("RVCHandler::onVolumeUpdateFromAg() entered"));

      //Storing the incoming value
      _agCurrentVolume = volume;

      getSCI_AgVolCtrlUpdate()->raise_vOLUME_CHANGED();
   }

   void RVCHandler::onTimerEventUpdate()
   {
      ETG_TRACE_USR4(("RVCHandler::onTimerEventUpdate() entered"));
      getSCI_VolCtrlTimerUpdate()->raise_tIMER_CALLBACK();
   }

   void RVCHandler::SmVolCtrlSmHandlerIf::handleIdle()
   {
      ETG_TRACE_USR4(("RVCHandler::handleIdle() entered"));

      PmCoreMainController::getInstance().getRVController().stopAsfTimer(_rvcHandler._deviceAddress);
      _rvcHandler._agLastsetValue = COUNT_ZERO;
      _rvcHandler._agCurrentVolume = COUNT_ZERO;
      _rvcHandler._amLastsetValue = COUNT_ZERO;
      _rvcHandler._amCurrentVolume = COUNT_ZERO;
   }

   void RVCHandler::SmVolCtrlTimerIf::startTimer()
   {
      ETG_TRACE_USR4(("RVCHandler::startTimer() entered"));
      RvcTimeout rvcResponseTimeout = com::bosch::pmcommon::PmConfiguration::getInstance().getRvcResponseTimeout();
      PmCoreMainController::getInstance().getRVController().startAsfTimer(
            _rvcHandler._deviceAddress, rvcResponseTimeout, TIMER_REPEAT_COUNT_ONE);
   }

   void RVCHandler::SmVolCtrlTimerIf::stopTimer()
   {
      ETG_TRACE_USR4(("RVCHandler::stopTimer() entered"));
      PmCoreMainController::getInstance().getRVController().stopAsfTimer(_rvcHandler._deviceAddress);
   }

   void RVCHandler::SmAgVolCtrlIf::setAgSpeakerVolume()
   {
      ETG_TRACE_USR4(("RVCHandler::setAgSpeakerVolume() entered"));

      VolumeLevel volumeToBeSet = _rvcHandler.convertAmVolumeToAgVolume(_rvcHandler._amCurrentVolume);

      //No property changed received for setting same value and latest received value from stack
      if((_rvcHandler._agLastsetValue != volumeToBeSet) && (_rvcHandler._agCurrentVolume != volumeToBeSet))
      {
         _rvcHandler._agLastsetValue = volumeToBeSet;
         evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSetPropertyRequest(::ccdbusif::evolution::IF_CALL_VOLUME,
               _rvcHandler.getDeviceAddress(),"SpeakerVolume", volumeToBeSet);
      }
   }

   void RVCHandler::SmAudioManagerVolCtrlIf::setAmSpeakerVolume()
   {
      ETG_TRACE_USR4(("RVCHandler::setAmSpeakerVolume() entered"));

      VolumeLevel volumeToBeSet = _rvcHandler.convertAgVolumeToAudioVolume(_rvcHandler._agCurrentVolume);

      //No property changed received for setting same value or less than Default value
      if((_rvcHandler._amLastsetValue != volumeToBeSet) && (_rvcHandler._amCurrentVolume != volumeToBeSet))
      {
         pmaudiomanager::AmPropertyType2VolumeMap deviceHandleToVolume;

         _rvcHandler._amLastsetValue = volumeToBeSet;
         if(true == PmCoreMainController::getInstance().getVRController().isVRStatusIdle(
               _rvcHandler.getDeviceAddress()))
         {
            pmaudiomanager::GeniviAMPropertyType gamDefaultPhoneSourceID = com::bosch::pmcommon::PmConfiguration::getInstance().getGAMDefaultPhoneSourceID();
            deviceHandleToVolume.emplace(gamDefaultPhoneSourceID,volumeToBeSet);
            PmAudioManagerWrapper::getInstance().setAudioSinkVolume(_rvcHandler._deviceAddress, GAM_SINK_ID, deviceHandleToVolume);
         }
         else
         {
            pmaudiomanager::GeniviAMPropertyType gamVoiceRecSourceID = com::bosch::pmcommon::PmConfiguration::getInstance().getGAMVoiceRecSourceID();
            deviceHandleToVolume.emplace(gamVoiceRecSourceID,volumeToBeSet);
            PmAudioManagerWrapper::getInstance().setAudioSinkVolume(_rvcHandler._deviceAddress, GAM_SINK_ID, deviceHandleToVolume);
         }
      }
   }

   template<typename T>
   void RVCHandler::SmTraceObserverIf<T>::stateEntered(T state)
   {
      ETG_TRACE_USR4(("RVCHandler::BDA::%20s::stateEntered()::%u", _RVCHandler._deviceAddress.c_str(),
            ETG_CENUM(RVCHandlerSmStates,state)));
   }

   template<typename T>
   void RVCHandler::SmTraceObserverIf<T>::stateExited(T state)
   {
      ETG_TRACE_USR4(("RVCHandler::BDA::%20s::stateExited()::%u", _RVCHandler._deviceAddress.c_str(),
            ETG_CENUM(RVCHandlerSmStates,state)));
   }

}
