/**
 * @file RVController.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the RVController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the RVController interfaces of PmCore
 *
 * @ingroup PmCore
 */

#ifndef RVController_h
#define RVController_h

#include "PmCoreIfTypes.h"
#include "PropertyUpdateHandler.h"
#include "ITimer.h"
#include "AsfTimer.h"
#include "AsfTimerCallbackData.h"

namespace pmcore
{
   class ISmVolumeControlIf;

   class RVController : public PropertyUpdateHandler
   {
   public:
      /**
       * Constructor of RVController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      RVController();

      /**
       * Destructor of RVController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~RVController();

      /**
       * Copy constructor of RVController class
       *
       * @param[in] ref - Reference to RVController
       * @param[out]
       * @param[in,out]
       *
       * @return
       *
       */
      RVController(const RVController& ref);

      /**
       * operator= overloaded function
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return Returns the object of the RVController
       *
       */
      RVController& operator=(const RVController& ref);

      /**
       * This method is used to get property changes update from BT Stack
       * This method is overridden from PropertyUpdateHandler class
       *
       * @param[in] propertyId      - Id of the property or event
       * @param[in] propertyDetails - Details of the property or event
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
            IN std::shared_ptr<void> propertyDetails) override;

      /**
       * This method is used to start call timer for new active and held calls
       *
       * @param[in] token  - BTaddress of the device
       * @param[in] duration - Duration of the timer
       * @param[in] repeatCount - repetition count
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void startAsfTimer(std::string token, ms_t duration, RepeatCount repeatCount);

      /**
       * This method is used to stop call timer for disconnected calls
       *
       * @param[in] token - BTaddress of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopAsfTimer(std::string token);

      /**
       * This method is used to delete call timer instance for disconnected calls
       *
       * @param[in] token - BTaddress of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void deleteAsfTimer(std::string token);

      /**
       * This method is used to notify timer elapsed callback to the Remote volume Controller state machine
       *
       * @param[in] data              - timer data
       * @param[in] timerCallbackData - timer callback data
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void timerElapsed(com::bosch::pmcommon::TimerData<std::string,ms_t,RepeatCount> data,
            com::bosch::pmcommon::AsfTimerCallbackData timerCallbackData);

      // **************************************************************************************************
      // Request calls

   private:
      /**
       * This method is used to subscribe interested properties in BT stack event notifier.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void subscribeToBtStackEventNotifier();

      PropertyIdList                           _propertyIdList;          /**< list of property id's */
      ISmVolumeControlIf*                      _smVolumeControlIf;        /**< pointer for remote volume sm interface */
      std::map< std::string, com::bosch::pmcommon::ITimer<RVController,std::string,ms_t,RepeatCount>*> _asfTimerInstanceMap;
   };
}

#endif // RVController_h

