/**
 * @file RVCHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the RVCHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#ifndef RVCHandler_h
#define RVCHandler_h

#include "EvoBtStackWrapper.h"
#include "../src-gen/RVCSm.h"

namespace pmcore
{
   class RVCHandler final: public RVCSm
   {
   public:
      /**
       * Parameterized Constructor of RVCHandler class
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      RVCHandler(const BdAddress& deviceAddress);

      /**
       * Destructor of RVCHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~RVCHandler();

      /**
       * This function is used to get the deviceAddress of the SM instance
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BdAddress - device address
       */
      const BdAddress& getDeviceAddress() { return _deviceAddress; }

      /**
       * This function is used to post the Device connection update to the SM instance
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void onDeviceConnect();

      /**
       * This function is used to post the Audio volume update to the SM instance
       *
       * @param[in] volumeControlData - Volume control data
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void onVolumeUpdateFromAm(IN const VolumeLevel& volume);

      /**
       * This function is used to post the AG volume update to the SM instance
       *
       * @param[in] volumeControlData - Volume control data
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void onVolumeUpdateFromAg(IN const VolumeLevel& volume);

      /**
       * This function is used to convert the Audio Volume into Ag Volume
       *
       * @param[in] amVolume - Volume level
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VolumeLevel convertAmVolumeToAgVolume(IN const VolumeLevel& volume);

      /**
       * This function is used to convert the Ag Volume into Audio Volume
       *
       * @param[in] amVolume - volume level
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      VolumeLevel convertAgVolumeToAudioVolume(IN const VolumeLevel& volume);

      /**
       * This function is used to update the Timer callback events
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void onTimerEventUpdate();

   private:
      /**
       * Constructor of RVCHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      RVCHandler();

      /**
       * SmAgVolCtrlUpdateIf class inherits the functions of SCI_AgVolCtrlUpdate_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAgVolCtrlIf final: public SCI_AgVolCtrlIf_OCB
      {
      public:
         SmAgVolCtrlIf(RVCHandler& rvcHandler): _rvcHandler(rvcHandler){}

         ~SmAgVolCtrlIf(){}

         virtual void setAgSpeakerVolume() override;

      private:
         RVCHandler& _rvcHandler;
      };

      /**
       * SmAudioManagerVolCtrlUpdateIf class inherits the functions of SCI_AudioManagerVolCtrlUpdate_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAudioManagerVolCtrlIf final: public SCI_AudioManagerVolCtrlIf_OCB
      {
      public:
         SmAudioManagerVolCtrlIf(RVCHandler& rvcHandler): _rvcHandler(rvcHandler){}

         ~SmAudioManagerVolCtrlIf(){}

         virtual void setAmSpeakerVolume() override;

      private:
         RVCHandler& _rvcHandler;
      };

      /**
       * SmVolCtrlTimerIf class inherits the functions of SCI_VolCtrlTimerIf_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmVolCtrlTimerIf final: public SCI_VolCtrlTimerIf_OCB
      {
      public:
         SmVolCtrlTimerIf(RVCHandler& rvcHandler): _rvcHandler(rvcHandler){}

         ~SmVolCtrlTimerIf(){}

         virtual void startTimer() override;

         virtual void stopTimer() override;

      private:
         RVCHandler& _rvcHandler;
      };

      /**
       * VolCtrlSmHandlerIf class inherits the functions of SCI_VolCtrlSmHandler_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmVolCtrlSmHandlerIf final: public SCI_VolCtrlSmHandlerIf_OCB
      {
      public:
         SmVolCtrlSmHandlerIf(RVCHandler& rvcHandler): _rvcHandler(rvcHandler){}

         ~SmVolCtrlSmHandlerIf(){}

         virtual void handleIdle() override;

      private:
         RVCHandler& _rvcHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of ysc::TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final: public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(RVCHandler& RVCHandler): _RVCHandler(RVCHandler){}

         ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         RVCHandler& _RVCHandler;
      };

      BdAddress                                  _deviceAddress;             /**< device address */
      SmAgVolCtrlIf*                             _smAgVolCtrlIf;             /**< pointer for AgVolCtrlUpdate interface */
      SmAudioManagerVolCtrlIf*                   _smAudioManagerVolCtrlIf;   /**< pointer for SmAudioManagerVolCtrlUpdateIf interface */
      SmVolCtrlTimerIf*                          _smVolCtrlTimerIf;          /**< pointer for SmAudioManagerVolCtrlUpdateIf interface */
      SmVolCtrlSmHandlerIf*                      _smVolCtrlSmHandlerIf;      /**< pointer for SmAudioManagerVolCtrlUpdateIf interface */
      SmTraceObserverIf<RVCSmStates>*            _smTraceObserverIf;         /**< pointer for TraceObserver interface */
      VolumeLevel                                _agLastsetValue;            /**< Last set value ofAg */
      VolumeLevel                                _agCurrentVolume;           /**< Current volume of Ag */
      VolumeLevel                                _amLastsetValue;            /**< Last set value of Am */
      VolumeLevel                                _amCurrentVolume;           /**< Current volume of Am*/
   };
}

#endif //YakinduSmVoiceCallIf_h
