/**
 * @file PropertyUpdateNotifierToCore.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PropertyUpdateNotifierToCore class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmCore
 */

#ifndef PropertyUpdateNotifierToCore_h
#define PropertyUpdateNotifierToCore_h

#include "PmCoreIfTypes.h"
#include "PropertyUpdateHandler.h"
#include "PmSingleton.h"
#include "PropertyDetails.h"
namespace pmcore
{
   class PropertyUpdateNotifierToCore : public PmSingleton <PropertyUpdateNotifierToCore>
   {
   public:

      /**
       * This method is used to handle property changed update
       *
       * @param[in] propertyId       - ID of the property or event
       * @param[in] propertyDetails  - reference to PropertyDetails
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template <class DetailsType>
      void notifyControllers(IN const PmCorePropertyAndEventId propertyId,
            IN std::shared_ptr<PropertyDetails<DetailsType>> propertyDetails)
      {
         if (_propertyToControllersMap.find(propertyId) != _propertyToControllersMap.end())
         {
            ControllerList controllersList = _propertyToControllersMap[propertyId];

            ControllerList::iterator itVec;
            for (itVec = controllersList.begin(); itVec != controllersList.end(); ++itVec)
            {
               (*itVec)->onPropertyUpdate(propertyId, propertyDetails);
            }
         }
      }

      /**
       * This method is used to attach Controller to Notifier list with the PropertyUpdateNotifierToCore
       *
       * @param[in] propertyIdList   - List of propertyIDs.
       * @param[in] propertyHandler  - Pointer to the PropertyUpdateHandler instance
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void attachControllerToNotifierList(IN const PropertyIdList& propertyIdList,
            IN const PropertyUpdateHandler* propertyHandler);

      /**
       * This method is used to detach Controller in Notifier list with the PropertyUpdateNotifierToCore
       *
       * @param[in] propertyIdList      - List of propertyIDs.
       * @param[in] propertyHandler     - Pointer to the PropertyUpdateHandler instance
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void detachControllerInNotifierList(IN const PropertyIdList& propertyIdList,
            IN const PropertyUpdateHandler* propertyHandler);

      /**
       * This method is used to print the contents of Property Handlers Map
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void printPropertyToControllersMap();

   private:
      friend class PmSingleton<PropertyUpdateNotifierToCore>;

      /**
       * Constructor of PropertyUpdateNotifierToCore class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PropertyUpdateNotifierToCore();

      /**
       * Destructor of PropertyUpdateNotifierToCore class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~PropertyUpdateNotifierToCore();

      /**
       * Copy construcor of PropertyUpdateNotifierToCore class
       *
       * @param[in] PropertyUpdateNotifierToCore - reference to PropertyUpdateNotifierToCore instance
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PropertyUpdateNotifierToCore(IN const PropertyUpdateNotifierToCore&);

      /**
       * Assignment operator of PropertyUpdateNotifierToCore class
       *
       * @param[in] PropertyUpdateNotifierToCore - reference to PropertyUpdateNotifierToCore instance
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      const PropertyUpdateNotifierToCore& operator=(IN const PropertyUpdateNotifierToCore&);

      PropertyToControllersMap _propertyToControllersMap;    /**< property to controllers map */
   };
}
#endif // PropertyUpdateNotifierToCore_h
