/**
 * @file PmCoreIfMessageResult.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @briefThis file contains the definition of all PmCoreIfMessageResult
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the implementation for all PmCore result Messages
 *
 * @ingroup PmCore
 */

#include "PmCoreIfMessageResult.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmCoreIfMessageResult.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   PmCoreIfMessageResult::PmCoreIfMessageResult(IN const PmCoreIfMsgId messageId, IN const BTSResult& btsResult,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
               PmCoreIfMessage(messageId, act, origin),
               _btsResult(btsResult)
   {
      ETG_TRACE_USR1(("PmCoreIfMessageResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessageResult::PmCoreIfMessageResult(IN const PmCoreIfMessageResult& other) :
         PmCoreIfMessage(other),
         _btsResult(other._btsResult)
   {
      ETG_TRACE_USR1(("PmCoreIfMessageResult: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessageResult::~PmCoreIfMessageResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessageResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessageResult& first, PmCoreIfMessageResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessage&>(first), static_cast<PmCoreIfMessage&>(second));
      swap(first._btsResult, second._btsResult);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessageResult& PmCoreIfMessageResult::operator=(PmCoreIfMessageResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessageResult::operator==(const PmCoreIfMessageResult& rhs) const
   {
      return ((PmCoreIfMessage::operator ==(rhs))
            && (this->_btsResult == rhs._btsResult));
   }

   void PmCoreIfMessageResult::traceMessage(void) const
   {
      PmCoreIfMessage::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BTSRequestResult = %d, ErrorCode = %d, ErrorMessage = \"%50s\" ",
            (const void*) this, ETG_CENUM(BTSRequestResult, _btsResult._btsRequestResult),
            ETG_CENUM(::ccdbusif::evolution::ErrorCode, _btsResult._errorCode), _btsResult._errorMessage.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_RequestPhoneNumberResult::PmCoreIfMessage_RequestPhoneNumberResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_REQUEST_PHONE_NUMBER_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _telephoneNumber("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RequestPhoneNumberResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_RequestPhoneNumberResult::PmCoreIfMessage_RequestPhoneNumberResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const TelephoneNumber& telephoneNumber, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_REQUEST_PHONE_NUMBER_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _telephoneNumber(telephoneNumber)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RequestPhoneNumberResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_RequestPhoneNumberResult::PmCoreIfMessage_RequestPhoneNumberResult(
         IN const PmCoreIfMessage_RequestPhoneNumberResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _telephoneNumber(other._telephoneNumber)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RequestPhoneNumberResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_RequestPhoneNumberResult::~PmCoreIfMessage_RequestPhoneNumberResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RequestPhoneNumberResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_RequestPhoneNumberResult& first, PmCoreIfMessage_RequestPhoneNumberResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._telephoneNumber, second._telephoneNumber);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_RequestPhoneNumberResult& PmCoreIfMessage_RequestPhoneNumberResult::operator=(
         PmCoreIfMessage_RequestPhoneNumberResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_RequestPhoneNumberResult::operator==(const PmCoreIfMessage_RequestPhoneNumberResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_telephoneNumber == rhs._telephoneNumber));
   }

   void PmCoreIfMessage_RequestPhoneNumberResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", TelephoneNumber = \"%50s\" ",
            (const void*) this, _bdAddress.c_str(), _telephoneNumber.c_str()));
   }

   // *****************************************************************************************************************
   PmCoreIfMessage_GetSubscriberResult::PmCoreIfMessage_GetSubscriberResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_GET_SUBSCRIBER_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _numberType(NUMBER_TYPE_UNKNOWN),
         _telephoneNumber("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSubscriberResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_GetSubscriberResult::PmCoreIfMessage_GetSubscriberResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const NumberType numberType, IN const TelephoneNumber& telephoneNumber,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_GET_SUBSCRIBER_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _numberType(numberType),
         _telephoneNumber(telephoneNumber)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSubscriberResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_GetSubscriberResult::PmCoreIfMessage_GetSubscriberResult(
         IN const PmCoreIfMessage_GetSubscriberResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _numberType(other._numberType),
         _telephoneNumber(other._telephoneNumber)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSubscriberResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSubscriberResult::~PmCoreIfMessage_GetSubscriberResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSubscriberResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetSubscriberResult& first, PmCoreIfMessage_GetSubscriberResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._numberType, second._numberType);
      swap(first._telephoneNumber, second._telephoneNumber);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetSubscriberResult& PmCoreIfMessage_GetSubscriberResult::operator=(
         PmCoreIfMessage_GetSubscriberResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetSubscriberResult::operator==(const PmCoreIfMessage_GetSubscriberResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_numberType == rhs._numberType)
            && (this->_telephoneNumber == rhs._telephoneNumber));
   }

   void PmCoreIfMessage_GetSubscriberResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", NumberType =%d, TelephoneNumber = \"%50s\" ",
            (const void*) this, _bdAddress.c_str(), _numberType, _telephoneNumber.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SendCustomATCommandResult::PmCoreIfMessage_SendCustomATCommandResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SEND_CUSTOM_AT_COMMAND_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendCustomATCommandResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SendCustomATCommandResult::PmCoreIfMessage_SendCustomATCommandResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SEND_CUSTOM_AT_COMMAND_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendCustomATCommandResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SendCustomATCommandResult::PmCoreIfMessage_SendCustomATCommandResult(
         IN const PmCoreIfMessage_SendCustomATCommandResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendCustomATCommandResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SendCustomATCommandResult::~PmCoreIfMessage_SendCustomATCommandResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendCustomATCommandResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SendCustomATCommandResult& first, PmCoreIfMessage_SendCustomATCommandResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SendCustomATCommandResult& PmCoreIfMessage_SendCustomATCommandResult::operator=(
         PmCoreIfMessage_SendCustomATCommandResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SendCustomATCommandResult::operator==(
         const PmCoreIfMessage_SendCustomATCommandResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_SendCustomATCommandResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************
   PmCoreIfMessage_HFUpdateIndicatorsValueResult::PmCoreIfMessage_HFUpdateIndicatorsValueResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HF_UPDATE_INDICATORS_VALUE_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HFUpdateIndicatorsValueResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HFUpdateIndicatorsValueResult::PmCoreIfMessage_HFUpdateIndicatorsValueResult(
         IN const BTSResult& btsResult, IN const BdAddress& bdAddress, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HF_UPDATE_INDICATORS_VALUE_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HFUpdateIndicatorsValueResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HFUpdateIndicatorsValueResult::PmCoreIfMessage_HFUpdateIndicatorsValueResult(
         IN const PmCoreIfMessage_HFUpdateIndicatorsValueResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HFUpdateIndicatorsValueResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HFUpdateIndicatorsValueResult::~PmCoreIfMessage_HFUpdateIndicatorsValueResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HFUpdateIndicatorsValueResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_HFUpdateIndicatorsValueResult& first, PmCoreIfMessage_HFUpdateIndicatorsValueResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_HFUpdateIndicatorsValueResult& PmCoreIfMessage_HFUpdateIndicatorsValueResult::operator=(
         PmCoreIfMessage_HFUpdateIndicatorsValueResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_HFUpdateIndicatorsValueResult::operator==(
         const PmCoreIfMessage_HFUpdateIndicatorsValueResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_HFUpdateIndicatorsValueResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetModemsResult::PmCoreIfMessage_GetModemsResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_GET_MODEMS_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetModemsResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_GetModemsResult::PmCoreIfMessage_GetModemsResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_GET_MODEMS_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetModemsResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_GetModemsResult::PmCoreIfMessage_GetModemsResult(IN const PmCoreIfMessage_GetModemsResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetModemsResult: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_GetModemsResult::~PmCoreIfMessage_GetModemsResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetModemsResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_GetModemsResult& first, PmCoreIfMessage_GetModemsResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetModemsResult& PmCoreIfMessage_GetModemsResult::operator=(
         PmCoreIfMessage_GetModemsResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetModemsResult::operator==(const PmCoreIfMessage_GetModemsResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetModemsResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************
   PmCoreIfMessage_AcceptSCOConnectResult::PmCoreIfMessage_AcceptSCOConnectResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_ACCEPT_SCO_CONNECT_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptSCOConnectResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AcceptSCOConnectResult::PmCoreIfMessage_AcceptSCOConnectResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_ACCEPT_SCO_CONNECT_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptSCOConnectResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AcceptSCOConnectResult::PmCoreIfMessage_AcceptSCOConnectResult(
         IN const PmCoreIfMessage_AcceptSCOConnectResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptSCOConnectResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AcceptSCOConnectResult::~PmCoreIfMessage_AcceptSCOConnectResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptSCOConnectResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_AcceptSCOConnectResult& first, PmCoreIfMessage_AcceptSCOConnectResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_AcceptSCOConnectResult& PmCoreIfMessage_AcceptSCOConnectResult::operator=(
         PmCoreIfMessage_AcceptSCOConnectResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_AcceptSCOConnectResult::operator==(const PmCoreIfMessage_AcceptSCOConnectResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_AcceptSCOConnectResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();
      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SiriEnableResult::PmCoreIfMessage_SiriEnableResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SIRI_ENABLE_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SiriEnableResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SiriEnableResult::PmCoreIfMessage_SiriEnableResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SIRI_ENABLE_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SiriEnableResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SiriEnableResult::PmCoreIfMessage_SiriEnableResult(
         IN const PmCoreIfMessage_SiriEnableResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SiriEnableResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SiriEnableResult::~PmCoreIfMessage_SiriEnableResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SiriEnableResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_SiriEnableResult& first, PmCoreIfMessage_SiriEnableResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SiriEnableResult& PmCoreIfMessage_SiriEnableResult::operator=(
         PmCoreIfMessage_SiriEnableResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SiriEnableResult::operator==(const PmCoreIfMessage_SiriEnableResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_SiriEnableResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SiriSetNRResult::PmCoreIfMessage_SiriSetNRResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SIRI_SET_NR_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SiriSetNRResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SiriSetNRResult::PmCoreIfMessage_SiriSetNRResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SIRI_SET_NR_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SiriSetNRResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SiriSetNRResult::PmCoreIfMessage_SiriSetNRResult(IN const PmCoreIfMessage_SiriSetNRResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SiriSetNRResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SiriSetNRResult::~PmCoreIfMessage_SiriSetNRResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SiriSetNRResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_SiriSetNRResult& first, PmCoreIfMessage_SiriSetNRResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SiriSetNRResult& PmCoreIfMessage_SiriSetNRResult::operator=(
         PmCoreIfMessage_SiriSetNRResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SiriSetNRResult::operator==(const PmCoreIfMessage_SiriSetNRResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_SiriSetNRResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetCallsResult::PmCoreIfMessage_GetCallsResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_GET_CALLS_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _btsVoiceCallInfoList()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetCallsResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_GetCallsResult::PmCoreIfMessage_GetCallsResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const BTSVoiceCallInfoList& btsVoiceCallInfoList, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_GET_CALLS_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _btsVoiceCallInfoList(btsVoiceCallInfoList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetCallsResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_GetCallsResult::PmCoreIfMessage_GetCallsResult(IN const PmCoreIfMessage_GetCallsResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _btsVoiceCallInfoList(other._btsVoiceCallInfoList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetCallsResult: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_GetCallsResult::~PmCoreIfMessage_GetCallsResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetCallsResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_GetCallsResult& first, PmCoreIfMessage_GetCallsResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._btsVoiceCallInfoList,second._btsVoiceCallInfoList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetCallsResult& PmCoreIfMessage_GetCallsResult::operator=(
         PmCoreIfMessage_GetCallsResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetCallsResult::operator==(const PmCoreIfMessage_GetCallsResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_btsVoiceCallInfoList == rhs._btsVoiceCallInfoList));
   }

   void PmCoreIfMessage_GetCallsResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));

      for(auto it = _btsVoiceCallInfoList.begin(); it != _btsVoiceCallInfoList.end(); it++)
      {
         ETG_TRACE_USR4(("traceMessage: CallInstance    = %d", it->_callInfo._instance));
         ETG_TRACE_USR4(("traceMessage: CallState       = \"%50s\" ", it->_callInfo._state.c_str()));
         ETG_TRACE_USR4(("traceMessage: TelephoneNumber = \"%50s\" ", it->_callInfo._telephoneNumber.c_str()));
      }
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_DialResult::PmCoreIfMessage_DialResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_DIAL_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _callInstance(CALL_INSTANCE_DEFAULT),
         _telephoneNumber("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_DialResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_DialResult::PmCoreIfMessage_DialResult(IN const BTSResult& btsResult, IN const BdAddress& bdAddress,
         IN const CallInstance callInstance, IN const TelephoneNumber telephoneNumber,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_DIAL_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _callInstance(callInstance),
         _telephoneNumber(telephoneNumber)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_DialResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_DialResult::PmCoreIfMessage_DialResult(IN const PmCoreIfMessage_DialResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _callInstance(other._callInstance),
         _telephoneNumber(other._telephoneNumber)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_DialResult: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_DialResult::~PmCoreIfMessage_DialResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_DialResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_DialResult& first, PmCoreIfMessage_DialResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstance, second._callInstance);
      swap(first._telephoneNumber, second._telephoneNumber);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_DialResult& PmCoreIfMessage_DialResult::operator=(PmCoreIfMessage_DialResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_DialResult::operator==(const PmCoreIfMessage_DialResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstance == rhs._callInstance)
            && (this->_telephoneNumber == rhs._telephoneNumber));
   }

   void PmCoreIfMessage_DialResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", CallInstance = %d, TelephoneNumber = %s ",
            (const void*) this, _bdAddress.c_str(),_callInstance,_telephoneNumber.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SpeedDialResult::PmCoreIfMessage_SpeedDialResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SPEED_DIAL_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _callInstance(CALL_INSTANCE_DEFAULT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SpeedDialResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SpeedDialResult::PmCoreIfMessage_SpeedDialResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const CallInstance callInstance, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SPEED_DIAL_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _callInstance(callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SpeedDialResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SpeedDialResult::PmCoreIfMessage_SpeedDialResult(IN const PmCoreIfMessage_SpeedDialResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _callInstance(other._callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SpeedDialResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SpeedDialResult::~PmCoreIfMessage_SpeedDialResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SpeedDialResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_SpeedDialResult& first, PmCoreIfMessage_SpeedDialResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SpeedDialResult& PmCoreIfMessage_SpeedDialResult::operator=(PmCoreIfMessage_SpeedDialResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SpeedDialResult::operator==(const PmCoreIfMessage_SpeedDialResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   void PmCoreIfMessage_SpeedDialResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", CallInstance = %d ",
            (const void*) this, _bdAddress.c_str(), _callInstance));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_RedialResult::PmCoreIfMessage_RedialResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_REDIAL_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _callInstance(CALL_INSTANCE_DEFAULT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RedialResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_RedialResult::PmCoreIfMessage_RedialResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const CallInstance callInstance, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_REDIAL_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _callInstance(callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RedialResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_RedialResult::PmCoreIfMessage_RedialResult(IN const PmCoreIfMessage_RedialResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _callInstance(other._callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RedialResult: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_RedialResult::~PmCoreIfMessage_RedialResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RedialResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_RedialResult& first, PmCoreIfMessage_RedialResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_RedialResult& PmCoreIfMessage_RedialResult::operator=(PmCoreIfMessage_RedialResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_RedialResult::operator==(const PmCoreIfMessage_RedialResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   void PmCoreIfMessage_RedialResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", CallInstance = %d ",
            (const void*) this, _bdAddress.c_str(), _callInstance));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SwapCallResult::PmCoreIfMessage_SwapCallResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SWAP_CALL_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SwapCallResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SwapCallResult::PmCoreIfMessage_SwapCallResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SWAP_CALL_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SwapCallResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SwapCallResult::PmCoreIfMessage_SwapCallResult(IN const PmCoreIfMessage_SwapCallResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SwapCallResult: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SwapCallResult::~PmCoreIfMessage_SwapCallResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SwapCallResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_SwapCallResult& first, PmCoreIfMessage_SwapCallResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SwapCallResult& PmCoreIfMessage_SwapCallResult::operator=(PmCoreIfMessage_SwapCallResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SwapCallResult::operator==(const PmCoreIfMessage_SwapCallResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_SwapCallResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_ReleaseAndAcceptResult::PmCoreIfMessage_ReleaseAndAcceptResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_RELEASE_AND_ACCEPT_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_ReleaseAndAcceptResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_ReleaseAndAcceptResult::PmCoreIfMessage_ReleaseAndAcceptResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_RELEASE_AND_ACCEPT_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_ReleaseAndAcceptResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_ReleaseAndAcceptResult::PmCoreIfMessage_ReleaseAndAcceptResult(
         IN const PmCoreIfMessage_ReleaseAndAcceptResult& other) : PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_ReleaseAndAcceptResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_ReleaseAndAcceptResult::~PmCoreIfMessage_ReleaseAndAcceptResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_ReleaseAndAcceptResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_ReleaseAndAcceptResult& first, PmCoreIfMessage_ReleaseAndAcceptResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_ReleaseAndAcceptResult& PmCoreIfMessage_ReleaseAndAcceptResult::operator=(
         PmCoreIfMessage_ReleaseAndAcceptResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_ReleaseAndAcceptResult::operator==(const PmCoreIfMessage_ReleaseAndAcceptResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_ReleaseAndAcceptResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_ReleaseAndSwapResult::PmCoreIfMessage_ReleaseAndSwapResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_RELEASE_AND_SWAP_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_ReleaseAndSwapResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_ReleaseAndSwapResult::PmCoreIfMessage_ReleaseAndSwapResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_RELEASE_AND_SWAP_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_ReleaseAndSwapResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_ReleaseAndSwapResult::PmCoreIfMessage_ReleaseAndSwapResult(
         IN const PmCoreIfMessage_ReleaseAndSwapResult& other) : PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_ReleaseAndSwapResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_ReleaseAndSwapResult::~PmCoreIfMessage_ReleaseAndSwapResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_ReleaseAndSwapResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_ReleaseAndSwapResult& first, PmCoreIfMessage_ReleaseAndSwapResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_ReleaseAndSwapResult& PmCoreIfMessage_ReleaseAndSwapResult::operator=(
         PmCoreIfMessage_ReleaseAndSwapResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_ReleaseAndSwapResult::operator==(const PmCoreIfMessage_ReleaseAndSwapResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_ReleaseAndSwapResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_HoldAndAcceptResult::PmCoreIfMessage_HoldAndAcceptResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HOLD_AND_ACCEPT_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HoldAndAcceptResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_HoldAndAcceptResult::PmCoreIfMessage_HoldAndAcceptResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HOLD_AND_ACCEPT_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HoldAndAcceptResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_HoldAndAcceptResult::PmCoreIfMessage_HoldAndAcceptResult(
         IN const PmCoreIfMessage_HoldAndAcceptResult& other) : PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HoldAndAcceptResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HoldAndAcceptResult::~PmCoreIfMessage_HoldAndAcceptResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HoldAndAcceptResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_HoldAndAcceptResult& first, PmCoreIfMessage_HoldAndAcceptResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_HoldAndAcceptResult& PmCoreIfMessage_HoldAndAcceptResult::operator=(
         PmCoreIfMessage_HoldAndAcceptResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_HoldAndAcceptResult::operator==(const PmCoreIfMessage_HoldAndAcceptResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_HoldAndAcceptResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_HangupAllResult::PmCoreIfMessage_HangupAllResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HANGUP_ALL_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupAllResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_HangupAllResult::PmCoreIfMessage_HangupAllResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HANGUP_ALL_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupAllResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_HangupAllResult::PmCoreIfMessage_HangupAllResult(IN const PmCoreIfMessage_HangupAllResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupAllResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HangupAllResult::~PmCoreIfMessage_HangupAllResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupAllResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_HangupAllResult& first, PmCoreIfMessage_HangupAllResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_HangupAllResult& PmCoreIfMessage_HangupAllResult::operator=(PmCoreIfMessage_HangupAllResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_HangupAllResult::operator==(const PmCoreIfMessage_HangupAllResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_HangupAllResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SplitCallsResult::PmCoreIfMessage_SplitCallsResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SPLIT_CALLS_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _callInstanceList()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SplitCallsResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SplitCallsResult::PmCoreIfMessage_SplitCallsResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const CallInstanceList& callInstanceList, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SPLIT_CALLS_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _callInstanceList(callInstanceList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SplitCallsResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SplitCallsResult::PmCoreIfMessage_SplitCallsResult(
         IN const PmCoreIfMessage_SplitCallsResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _callInstanceList(other._callInstanceList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SplitCallsResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SplitCallsResult::~PmCoreIfMessage_SplitCallsResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SplitCallsResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_SplitCallsResult& first, PmCoreIfMessage_SplitCallsResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstanceList, second._callInstanceList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SplitCallsResult& PmCoreIfMessage_SplitCallsResult::operator=(
         PmCoreIfMessage_SplitCallsResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SplitCallsResult::operator==(const PmCoreIfMessage_SplitCallsResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstanceList == rhs._callInstanceList));
   }

   void PmCoreIfMessage_SplitCallsResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_MergeCallsResult::PmCoreIfMessage_MergeCallsResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_MERGE_CALLS_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _callInstanceList()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_MergeCallsResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_MergeCallsResult::PmCoreIfMessage_MergeCallsResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const CallInstanceList& callInstanceList, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_MERGE_CALLS_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _callInstanceList(callInstanceList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_MergeCallsResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_MergeCallsResult::PmCoreIfMessage_MergeCallsResult(IN const PmCoreIfMessage_MergeCallsResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _callInstanceList(other._callInstanceList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_MergeCallsResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_MergeCallsResult::~PmCoreIfMessage_MergeCallsResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_MergeCallsResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_MergeCallsResult& first, PmCoreIfMessage_MergeCallsResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstanceList, second._callInstanceList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_MergeCallsResult& PmCoreIfMessage_MergeCallsResult::operator=(
         PmCoreIfMessage_MergeCallsResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_MergeCallsResult::operator==(const PmCoreIfMessage_MergeCallsResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstanceList == rhs._callInstanceList));
   }

   void PmCoreIfMessage_MergeCallsResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_HangupMultipartyResult::PmCoreIfMessage_HangupMultipartyResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HANGUP_MULTIPARTY_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupMultipartyResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HangupMultipartyResult::PmCoreIfMessage_HangupMultipartyResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HANGUP_MULTIPARTY_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupMultipartyResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HangupMultipartyResult::PmCoreIfMessage_HangupMultipartyResult(
         IN const PmCoreIfMessage_HangupMultipartyResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupMultipartyResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HangupMultipartyResult::~PmCoreIfMessage_HangupMultipartyResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupMultipartyResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_HangupMultipartyResult& first, PmCoreIfMessage_HangupMultipartyResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_HangupMultipartyResult& PmCoreIfMessage_HangupMultipartyResult::operator=(
         PmCoreIfMessage_HangupMultipartyResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_HangupMultipartyResult::operator==(const PmCoreIfMessage_HangupMultipartyResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_HangupMultipartyResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SendDTMFResult::PmCoreIfMessage_SendDTMFResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SEND_DTMF_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendDTMFResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SendDTMFResult::PmCoreIfMessage_SendDTMFResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_SEND_DTMF_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendDTMFResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SendDTMFResult::PmCoreIfMessage_SendDTMFResult(IN const PmCoreIfMessage_SendDTMFResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendDTMFResult: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SendDTMFResult::~PmCoreIfMessage_SendDTMFResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendDTMFResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_SendDTMFResult& first, PmCoreIfMessage_SendDTMFResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SendDTMFResult& PmCoreIfMessage_SendDTMFResult::operator=(PmCoreIfMessage_SendDTMFResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SendDTMFResult::operator==(const PmCoreIfMessage_SendDTMFResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_SendDTMFResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_HangupCallResult::PmCoreIfMessage_HangupCallResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HANGUP_CALL_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _callInstance(CALL_INSTANCE_DEFAULT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupCallResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_HangupCallResult::PmCoreIfMessage_HangupCallResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const CallInstance callInstance, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HANGUP_CALL_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _callInstance(callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupCallResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_HangupCallResult::PmCoreIfMessage_HangupCallResult(
         IN const PmCoreIfMessage_HangupCallResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _callInstance(other._callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupCallResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HangupCallResult::~PmCoreIfMessage_HangupCallResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupCallResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_HangupCallResult& first, PmCoreIfMessage_HangupCallResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_HangupCallResult& PmCoreIfMessage_HangupCallResult::operator=(
         PmCoreIfMessage_HangupCallResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_HangupCallResult::operator==(const PmCoreIfMessage_HangupCallResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   void PmCoreIfMessage_HangupCallResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", CallInstance = %d",
            (const void*) this, _bdAddress.c_str(), _callInstance));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_AcceptCallResult::PmCoreIfMessage_AcceptCallResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_ACCEPT_CALL_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _callInstance(CALL_INSTANCE_DEFAULT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptCallResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AcceptCallResult::PmCoreIfMessage_AcceptCallResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const CallInstance callInstance,IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_ACCEPT_CALL_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _callInstance(callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptCallResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_AcceptCallResult::PmCoreIfMessage_AcceptCallResult(
         IN const PmCoreIfMessage_AcceptCallResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _callInstance(other._callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptCallResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AcceptCallResult::~PmCoreIfMessage_AcceptCallResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptCallResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_AcceptCallResult& first, PmCoreIfMessage_AcceptCallResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_AcceptCallResult& PmCoreIfMessage_AcceptCallResult::operator=(
         PmCoreIfMessage_AcceptCallResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_AcceptCallResult::operator==(const PmCoreIfMessage_AcceptCallResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   void PmCoreIfMessage_AcceptCallResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", CallInstance = %d",
            (const void*) this, _bdAddress.c_str(), _callInstance));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_HoldIncomingCallResult::PmCoreIfMessage_HoldIncomingCallResult() :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HOLD_INCOMING_CALL_RESULT, BTSResult(), PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
         _bdAddress(""),
         _callInstance(CALL_INSTANCE_DEFAULT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HoldIncomingCallResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HoldIncomingCallResult::PmCoreIfMessage_HoldIncomingCallResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const CallInstance callInstance, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_HOLD_INCOMING_CALL_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress),
         _callInstance(callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HoldIncomingCallResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HoldIncomingCallResult::PmCoreIfMessage_HoldIncomingCallResult(
         IN const PmCoreIfMessage_HoldIncomingCallResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _callInstance(other._callInstance)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HoldIncomingCallResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HoldIncomingCallResult::~PmCoreIfMessage_HoldIncomingCallResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HoldIncomingCallResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_HoldIncomingCallResult& first, PmCoreIfMessage_HoldIncomingCallResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstance, second._callInstance);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_HoldIncomingCallResult& PmCoreIfMessage_HoldIncomingCallResult::operator=(
         PmCoreIfMessage_HoldIncomingCallResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_HoldIncomingCallResult::operator==(const PmCoreIfMessage_HoldIncomingCallResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstance == rhs._callInstance));
   }

   void PmCoreIfMessage_HoldIncomingCallResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", CallInstance = %d",
            (const void*) this, _bdAddress.c_str(), _callInstance));
   }

   // *****************************************************************************************************************
   // PM Audiomanager related response messages
   // *****************************************************************************************************************

   PmCoreIfMessageAmResult::PmCoreIfMessageAmResult(IN const PmCoreIfMsgId messageId, 
         IN const pmaudiomanager::AmResult& amResult, 
         IN const PmCoreIfMsgOrigin origin) : 
         PmCoreIfMessage(messageId, PM_DEFAULT_ACT, origin), _amResult(amResult)
   {
      ETG_TRACE_USR1(("PmCoreIfMessageAmResult: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessageAmResult::PmCoreIfMessageAmResult(IN const PmCoreIfMessageAmResult& other) : 
         PmCoreIfMessage(other), _amResult(other._amResult)
   {
      ETG_TRACE_USR1(("PmCoreIfMessageAmResult: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessageAmResult::~PmCoreIfMessageAmResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessageResult: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessageAmResult& first, PmCoreIfMessageAmResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessage&>(first), static_cast<PmCoreIfMessage&>(second));
      swap(first._amResult, second._amResult);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessageAmResult& PmCoreIfMessageAmResult::operator=(PmCoreIfMessageAmResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessageAmResult::operator==(const PmCoreIfMessageAmResult& rhs) const
   {
      return ((PmCoreIfMessage::operator ==(rhs))
            && (this->_amResult == rhs._amResult));
   }

   void PmCoreIfMessageAmResult::traceMessage(void) const
   {
      PmCoreIfMessage::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] AmResult- ErrorCode = %d, ErrorMessage = \"%50s\" ", 
            (const void*) this, ETG_CENUM(AmResultCode, _amResult._amResultCode), _amResult._amResultMessage.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_PrepareAudioRouteResponse::PmCoreIfMessage_PrepareAudioRouteResponse() :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_PREPARE_AUDIO_ROUTE_RESPONSE, pmaudiomanager::AmResult(),
               PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF),
         _bdAddress(""),
         _controllerOriginEnumType(MAIN_CONTROLLER)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PrepareAudioRouteResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PrepareAudioRouteResponse::PmCoreIfMessage_PrepareAudioRouteResponse(IN const pmaudiomanager::AmResult& amResult,
         IN const BdAddress& bdAddress,
         IN const ControllerOriginEnum controllerOriginEnumType,
         IN const PmCoreIfMsgOrigin origin) :
               PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_PREPARE_AUDIO_ROUTE_RESPONSE, amResult, origin),
               _bdAddress(bdAddress), _controllerOriginEnumType(controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PrepareAudioRouteResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PrepareAudioRouteResponse::PmCoreIfMessage_PrepareAudioRouteResponse(IN const PmCoreIfMessage_PrepareAudioRouteResponse& other) :
         PmCoreIfMessageAmResult(other),
         _bdAddress(other._bdAddress),
         _controllerOriginEnumType(other._controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PrepareAudioRouteResponse: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PrepareAudioRouteResponse::~PmCoreIfMessage_PrepareAudioRouteResponse()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PrepareAudioRouteResponse: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_PrepareAudioRouteResponse& first, PmCoreIfMessage_PrepareAudioRouteResponse& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageAmResult&>(first), static_cast<PmCoreIfMessageAmResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_PrepareAudioRouteResponse& PmCoreIfMessage_PrepareAudioRouteResponse::operator=
         (PmCoreIfMessage_PrepareAudioRouteResponse other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_PrepareAudioRouteResponse::operator==(const PmCoreIfMessage_PrepareAudioRouteResponse& rhs) const
   {
      return ((PmCoreIfMessageAmResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType));
   }

   void PmCoreIfMessage_PrepareAudioRouteResponse::traceMessage(void) const
   {
      PmCoreIfMessageAmResult::traceMessage();
      ETG_TRACE_USR4(("traceMessage: [0x%p] bdAddress = \"%50s\", ControllerOriginEnumType = %d",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(ControllerOriginEnum, _controllerOriginEnumType)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_PlayAudioResponse::PmCoreIfMessage_PlayAudioResponse() :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_PLAY_AUDIO_RESPONSE, pmaudiomanager::AmResult(),
               PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF),
         _bdAddress(""),
         _controllerOriginEnumType(MAIN_CONTROLLER)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PlayAudioResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PlayAudioResponse::PmCoreIfMessage_PlayAudioResponse(IN const pmaudiomanager::AmResult& amResult,
         IN const BdAddress& bdAddress,
         IN const ControllerOriginEnum controllerOriginEnumType,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_PLAY_AUDIO_RESPONSE, amResult, origin),
         _bdAddress(bdAddress),
         _controllerOriginEnumType(controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PlayAudioResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PlayAudioResponse::PmCoreIfMessage_PlayAudioResponse(IN const PmCoreIfMessage_PlayAudioResponse& other) :
         PmCoreIfMessageAmResult(other),
         _bdAddress(other._bdAddress),
         _controllerOriginEnumType(other._controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PlayAudioResponse: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PlayAudioResponse::~PmCoreIfMessage_PlayAudioResponse()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PlayAudioResponse: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_PlayAudioResponse& first, PmCoreIfMessage_PlayAudioResponse& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageAmResult&>(first), static_cast<PmCoreIfMessageAmResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_PlayAudioResponse& PmCoreIfMessage_PlayAudioResponse::operator=
         (PmCoreIfMessage_PlayAudioResponse other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_PlayAudioResponse::operator==(const PmCoreIfMessage_PlayAudioResponse& rhs) const
   {
      return ((PmCoreIfMessageAmResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType));
   }

   void PmCoreIfMessage_PlayAudioResponse::traceMessage(void) const
   {
      PmCoreIfMessageAmResult::traceMessage();
      ETG_TRACE_USR4(("traceMessage: [0x%p] bdAddress = \"%50s\", ControllerOriginEnumType = %d",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(ControllerOriginEnum, _controllerOriginEnumType)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_StopAudioResponse::PmCoreIfMessage_StopAudioResponse() :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_STOP_AUDIO_RESPONSE, pmaudiomanager::AmResult(),
               PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF),
         _bdAddress(""),
         _controllerOriginEnumType(MAIN_CONTROLLER)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StopAudioResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StopAudioResponse::PmCoreIfMessage_StopAudioResponse(IN const pmaudiomanager::AmResult& amResult,
         IN const BdAddress& bdAddress,
         IN const ControllerOriginEnum controllerOriginEnumType,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_STOP_AUDIO_RESPONSE, amResult, origin),
         _bdAddress(bdAddress),
         _controllerOriginEnumType(controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StopAudioResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StopAudioResponse::PmCoreIfMessage_StopAudioResponse(IN const PmCoreIfMessage_StopAudioResponse& other) :
         PmCoreIfMessageAmResult(other),
         _bdAddress(other._bdAddress),
         _controllerOriginEnumType(other._controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StopAudioResponse: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StopAudioResponse::~PmCoreIfMessage_StopAudioResponse()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StopAudioResponse: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_StopAudioResponse& first, PmCoreIfMessage_StopAudioResponse& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageAmResult&>(first), static_cast<PmCoreIfMessageAmResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_StopAudioResponse& PmCoreIfMessage_StopAudioResponse::operator=
         (PmCoreIfMessage_StopAudioResponse other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_StopAudioResponse::operator==(const PmCoreIfMessage_StopAudioResponse& rhs) const
   {
      return ((PmCoreIfMessageAmResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType));
   }

   void PmCoreIfMessage_StopAudioResponse::traceMessage(void) const
   {
      PmCoreIfMessageAmResult::traceMessage();
      ETG_TRACE_USR4(("traceMessage: [0x%p] bdAddress = \"%50s\", ControllerOriginEnumType = %d",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(ControllerOriginEnum, _controllerOriginEnumType)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SetMicMuteStateResponse::PmCoreIfMessage_SetMicMuteStateResponse() :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_SET_MIC_MUTE_STATE_RESPONSE, pmaudiomanager::AmResult(),
               PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF),
         _bdAddress(""),
         _controllerOriginEnumType(MAIN_CONTROLLER)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetMicMuteStateResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetMicMuteStateResponse::PmCoreIfMessage_SetMicMuteStateResponse(IN const pmaudiomanager::AmResult& amResult,
         IN const BdAddress& bdAddress,
         IN const ControllerOriginEnum controllerOriginEnumType,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_SET_MIC_MUTE_STATE_RESPONSE, amResult, origin),
         _bdAddress(bdAddress),
         _controllerOriginEnumType(controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetMicMuteStateResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetMicMuteStateResponse::PmCoreIfMessage_SetMicMuteStateResponse(IN const PmCoreIfMessage_SetMicMuteStateResponse& other) :
         PmCoreIfMessageAmResult(other),
         _bdAddress(other._bdAddress),
         _controllerOriginEnumType(other._controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetMicMuteStateResponse: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetMicMuteStateResponse::~PmCoreIfMessage_SetMicMuteStateResponse()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetMicMuteStateResponse: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetMicMuteStateResponse& first, PmCoreIfMessage_SetMicMuteStateResponse& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageAmResult&>(first), static_cast<PmCoreIfMessageAmResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetMicMuteStateResponse& PmCoreIfMessage_SetMicMuteStateResponse::operator=
         (PmCoreIfMessage_SetMicMuteStateResponse other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetMicMuteStateResponse::operator==(const PmCoreIfMessage_SetMicMuteStateResponse& rhs) const
   {
      return ((PmCoreIfMessageAmResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType));
   }

   void PmCoreIfMessage_SetMicMuteStateResponse::traceMessage(void) const
   {
      PmCoreIfMessageAmResult::traceMessage();
      ETG_TRACE_USR4(("traceMessage: [0x%p] bdAddress = \"%50s\", ControllerOriginEnumType = %d",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(ControllerOriginEnum, _controllerOriginEnumType)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_PauseAudioResponse::PmCoreIfMessage_PauseAudioResponse() :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_PAUSE_AUDIO_RESPONSE, pmaudiomanager::AmResult(),
               PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF),
         _bdAddress(""),
         _controllerOriginEnumType(MAIN_CONTROLLER)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PauseAudioResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PauseAudioResponse::PmCoreIfMessage_PauseAudioResponse(IN const pmaudiomanager::AmResult& amResult,
         IN const BdAddress& bdAddress,
         IN const ControllerOriginEnum controllerOriginEnumType,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_PAUSE_AUDIO_RESPONSE, amResult, origin),
         _bdAddress(bdAddress),
         _controllerOriginEnumType(controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PauseAudioResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PauseAudioResponse::PmCoreIfMessage_PauseAudioResponse(IN const PmCoreIfMessage_PauseAudioResponse& other) :
         PmCoreIfMessageAmResult(other),
         _bdAddress(other._bdAddress),
         _controllerOriginEnumType(other._controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PauseAudioResponse: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PauseAudioResponse::~PmCoreIfMessage_PauseAudioResponse()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PauseAudioResponse: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_PauseAudioResponse& first, PmCoreIfMessage_PauseAudioResponse& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageAmResult&>(first), static_cast<PmCoreIfMessageAmResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_PauseAudioResponse& PmCoreIfMessage_PauseAudioResponse::operator=
         (PmCoreIfMessage_PauseAudioResponse other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_PauseAudioResponse::operator==(const PmCoreIfMessage_PauseAudioResponse& rhs) const
   {
      return ((PmCoreIfMessageAmResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType));
   }

   void PmCoreIfMessage_PauseAudioResponse::traceMessage(void) const
   {
      PmCoreIfMessageAmResult::traceMessage();
      ETG_TRACE_USR4(("traceMessage: [0x%p] bdAddress = \"%50s\", ControllerOriginEnumType = %d",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(ControllerOriginEnum, _controllerOriginEnumType)));
   }
   // *****************************************************************************************************************

   PmCoreIfMessage_TransferAudioResult::PmCoreIfMessage_TransferAudioResult() :
                 PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_TRANSFER_AUDIO_RESULT, BTSResult(), PM_DEFAULT_ACT,
                       PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
                       _bdAddress(""), _audioDirection(AUDIO_DIRECTION_HANDSFREE)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_TransferAudioResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_TransferAudioResult::PmCoreIfMessage_TransferAudioResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const AudioDirectionEnumType audioDirection, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_TRANSFER_AUDIO_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress), _audioDirection(audioDirection)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_TransferAudioResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_TransferAudioResult::PmCoreIfMessage_TransferAudioResult(
         IN const PmCoreIfMessage_TransferAudioResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress), _audioDirection(other._audioDirection)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_TransferAudioResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_TransferAudioResult::~PmCoreIfMessage_TransferAudioResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_TransferAudioResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_TransferAudioResult& first, PmCoreIfMessage_TransferAudioResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._audioDirection, second._audioDirection);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_TransferAudioResult& PmCoreIfMessage_TransferAudioResult::operator=(
         PmCoreIfMessage_TransferAudioResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_TransferAudioResult::operator==(const PmCoreIfMessage_TransferAudioResult& rhs) const
           {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_audioDirection == rhs._audioDirection));
           }

   void PmCoreIfMessage_TransferAudioResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" and AudioDirection: %u",
            (const void*) this, _bdAddress.c_str(), _audioDirection));
   }
   // *****************************************************************************************************************

   PmCoreIfMessage_StartStopVoiceRecognitionResult::PmCoreIfMessage_StartStopVoiceRecognitionResult() :
                 PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_START_STOP_VOICE_RECOGNITION_RESULT, BTSResult(), PM_DEFAULT_ACT,
                       PM_CORE_IF_MSG_ORIGIN_BT_STACK_IF),
                       _bdAddress(""), _vrValue(false)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopVoiceRecognitionResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StartStopVoiceRecognitionResult::PmCoreIfMessage_StartStopVoiceRecognitionResult(IN const BTSResult& btsResult,
         IN const BdAddress& bdAddress, IN const StartStop vrValue, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageResult(PM_CORE_IF_MSG_ID_START_STOP_VOICE_RECOGNITION_RESULT, btsResult, act, origin),
         _bdAddress(bdAddress), _vrValue(vrValue)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopVoiceRecognitionResult: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StartStopVoiceRecognitionResult::PmCoreIfMessage_StartStopVoiceRecognitionResult(
         IN const PmCoreIfMessage_StartStopVoiceRecognitionResult& other) :
         PmCoreIfMessageResult(other),
         _bdAddress(other._bdAddress),
         _vrValue(other._vrValue)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopVoiceRecognitionResult: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StartStopVoiceRecognitionResult::~PmCoreIfMessage_StartStopVoiceRecognitionResult()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopVoiceRecognitionResult: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_StartStopVoiceRecognitionResult& first, PmCoreIfMessage_StartStopVoiceRecognitionResult& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageResult&>(first), static_cast<PmCoreIfMessageResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._vrValue, second._vrValue);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_StartStopVoiceRecognitionResult& PmCoreIfMessage_StartStopVoiceRecognitionResult::operator=(
         PmCoreIfMessage_StartStopVoiceRecognitionResult other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_StartStopVoiceRecognitionResult::operator==(const PmCoreIfMessage_StartStopVoiceRecognitionResult& rhs) const
   {
      return ((PmCoreIfMessageResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_vrValue == rhs._vrValue));
   }

   void PmCoreIfMessage_StartStopVoiceRecognitionResult::traceMessage(void) const
   {
      PmCoreIfMessageResult::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" VrStatus = %d",
            (const void*) this, _bdAddress.c_str(), _vrValue));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetAudioSinkVolumeResponse::PmCoreIfMessage_GetAudioSinkVolumeResponse() :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_GET_AUDIO_SINK_VOLUME_RESPONSE, pmaudiomanager::AmResult(),
               PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF),
         _bdAddress(""),
         _controllerOriginEnumType(RV_CONTROLLER)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetAudioSinkVolumeResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetAudioSinkVolumeResponse::PmCoreIfMessage_GetAudioSinkVolumeResponse(IN const pmaudiomanager::AmResult& amResult,
         IN const BdAddress& bdAddress,
         IN const ControllerOriginEnum controllerOriginEnumType,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_GET_AUDIO_SINK_VOLUME_RESPONSE, amResult, origin),
         _bdAddress(bdAddress),
         _controllerOriginEnumType(controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetAudioSinkVolumeResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetAudioSinkVolumeResponse::PmCoreIfMessage_GetAudioSinkVolumeResponse(IN const PmCoreIfMessage_GetAudioSinkVolumeResponse& other) :
         PmCoreIfMessageAmResult(other),
         _bdAddress(other._bdAddress),
         _controllerOriginEnumType(other._controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetAudioSinkVolumeResponse: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetAudioSinkVolumeResponse::~PmCoreIfMessage_GetAudioSinkVolumeResponse()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetAudioSinkVolumeResponse: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetAudioSinkVolumeResponse& first, PmCoreIfMessage_GetAudioSinkVolumeResponse& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageAmResult&>(first), static_cast<PmCoreIfMessageAmResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetAudioSinkVolumeResponse& PmCoreIfMessage_GetAudioSinkVolumeResponse::operator=
         (PmCoreIfMessage_GetAudioSinkVolumeResponse other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetAudioSinkVolumeResponse::operator==(const PmCoreIfMessage_GetAudioSinkVolumeResponse& rhs) const
   {
      return ((PmCoreIfMessageAmResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType));
   }

   void PmCoreIfMessage_GetAudioSinkVolumeResponse::traceMessage(void) const
   {
      PmCoreIfMessageAmResult::traceMessage();
      ETG_TRACE_USR4(("traceMessage: [0x%p] bdAddress = \"%50s\", ControllerOriginEnumType = %d",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(ControllerOriginEnum, _controllerOriginEnumType)));
   }
   // *****************************************************************************************************************

   PmCoreIfMessage_SetAudioSinkVolumeResponse::PmCoreIfMessage_SetAudioSinkVolumeResponse() :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_SET_AUDIO_SINK_VOLUME_RESPONSE, pmaudiomanager::AmResult(),
               PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF),
         _bdAddress(""),
         _controllerOriginEnumType(RV_CONTROLLER)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetAudioSinkVolumeResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetAudioSinkVolumeResponse::PmCoreIfMessage_SetAudioSinkVolumeResponse(IN const pmaudiomanager::AmResult& amResult,
         IN const BdAddress& bdAddress,
         IN const ControllerOriginEnum controllerOriginEnumType,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageAmResult(PM_CORE_IF_MSG_ID_SET_AUDIO_SINK_VOLUME_RESPONSE, amResult, origin),
         _bdAddress(bdAddress),
         _controllerOriginEnumType(controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetAudioSinkVolumeResponse: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetAudioSinkVolumeResponse::PmCoreIfMessage_SetAudioSinkVolumeResponse(IN const PmCoreIfMessage_SetAudioSinkVolumeResponse& other) :
         PmCoreIfMessageAmResult(other),
         _bdAddress(other._bdAddress),
         _controllerOriginEnumType(other._controllerOriginEnumType)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetAudioSinkVolumeResponse: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetAudioSinkVolumeResponse::~PmCoreIfMessage_SetAudioSinkVolumeResponse()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetAudioSinkVolumeResponse: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetAudioSinkVolumeResponse& first, PmCoreIfMessage_SetAudioSinkVolumeResponse& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageAmResult&>(first), static_cast<PmCoreIfMessageAmResult&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetAudioSinkVolumeResponse& PmCoreIfMessage_SetAudioSinkVolumeResponse::operator=
         (PmCoreIfMessage_SetAudioSinkVolumeResponse other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetAudioSinkVolumeResponse::operator==(const PmCoreIfMessage_SetAudioSinkVolumeResponse& rhs) const
   {
      return ((PmCoreIfMessageAmResult::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType));
   }

   void PmCoreIfMessage_SetAudioSinkVolumeResponse::traceMessage(void) const
   {
      PmCoreIfMessageAmResult::traceMessage();
      ETG_TRACE_USR4(("traceMessage: [0x%p] bdAddress = \"%50s\", ControllerOriginEnumType = %d",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(ControllerOriginEnum, _controllerOriginEnumType)));
   }
} // namespace pmcore
