/**
 * @file PmCoreIfMessageRequest.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @briefThis file contains the definition of all PmCoreIfMessageRequest
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the implementation for all PmCore request Messages
 *
 * @ingroup PmCore
 */

#include "PmCoreIfMessageRequest.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmCoreIfMessageRequest.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{

   PmCoreIfMessageRequest::PmCoreIfMessageRequest(
         IN const PmCoreIfMsgId messageId, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
               PmCoreIfMessage(messageId, act, origin),
               _pmResult(PM_RESULT_ERR_GENERAL, "")
   {
      ETG_TRACE_USR1(("PmCoreIfMessageRequest: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessageRequest::PmCoreIfMessageRequest(
         IN const PmCoreIfMessageRequest& other) : PmCoreIfMessage(other),
               _pmResult(other._pmResult)
   {
      ETG_TRACE_USR1(("PmCoreIfMessageRequest: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessageRequest::~PmCoreIfMessageRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessageRequest: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessageRequest& first, PmCoreIfMessageRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessage&>(first), static_cast<PmCoreIfMessage&>(second));
      swap(first._pmResult, second._pmResult);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessageRequest& PmCoreIfMessageRequest::operator=(PmCoreIfMessageRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessageRequest::operator==(const PmCoreIfMessageRequest& rhs) const
   {
      return ((PmCoreIfMessage::operator ==(rhs))
            && (this->_pmResult == rhs._pmResult));
   }

   void PmCoreIfMessageRequest::traceMessage(void) const
   {
      PmCoreIfMessage::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] PmResultCode = %u, PmResultMessage = \"%50s\"",
            (const void*) this, ETG_CENUM(PmResultCode, _pmResult._pmResultCode), _pmResult._pmResultMessage.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SetSystemState::PmCoreIfMessage_SetSystemState() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_SYSTEM_STATE, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _systemState(PM_SPM_SYSTEM_OFF)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetSystemState: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SetSystemState::PmCoreIfMessage_SetSystemState(
         IN const SystemState systemState, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_SYSTEM_STATE, act, origin),
         _systemState(systemState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetSystemState: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SetSystemState::PmCoreIfMessage_SetSystemState(
         IN const PmCoreIfMessage_SetSystemState& other) : PmCoreIfMessageRequest(other),
         _systemState(other._systemState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetSystemState: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetSystemState::~PmCoreIfMessage_SetSystemState()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetSystemState: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetSystemState& first, PmCoreIfMessage_SetSystemState& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._systemState, second._systemState);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetSystemState& PmCoreIfMessage_SetSystemState::operator=(
         PmCoreIfMessage_SetSystemState other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetSystemState::operator==(const PmCoreIfMessage_SetSystemState& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_systemState == rhs._systemState));
   }

   void PmCoreIfMessage_SetSystemState::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] SystemState = %u", (const void*) this,
            ETG_CENUM(SystemState, _systemState)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SetVehicleConfiguration::PmCoreIfMessage_SetVehicleConfiguration() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_VEHICLE_CONFIGURATION, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _vehicleConfig()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetVehicleConfiguration: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SetVehicleConfiguration::PmCoreIfMessage_SetVehicleConfiguration(
         IN const VehicleConfiguration& vehicleConfig, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_VEHICLE_CONFIGURATION, act, origin),
         _vehicleConfig(vehicleConfig)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetVehicleConfiguration: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_SetVehicleConfiguration::PmCoreIfMessage_SetVehicleConfiguration(
         IN const PmCoreIfMessage_SetVehicleConfiguration& other) : PmCoreIfMessageRequest(other),
         _vehicleConfig(other._vehicleConfig)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetVehicleConfiguration: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetVehicleConfiguration::~PmCoreIfMessage_SetVehicleConfiguration()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetVehicleConfiguration: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetVehicleConfiguration& first, PmCoreIfMessage_SetVehicleConfiguration& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._vehicleConfig, second._vehicleConfig);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetVehicleConfiguration& PmCoreIfMessage_SetVehicleConfiguration::operator=(
         PmCoreIfMessage_SetVehicleConfiguration other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetVehicleConfiguration::operator==(const PmCoreIfMessage_SetVehicleConfiguration& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_vehicleConfig == rhs._vehicleConfig));
   }

   void PmCoreIfMessage_SetVehicleConfiguration::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] vehicle BdAddress = \"%50s\"", (const void*) this, _vehicleConfig._vehicleBTAddress.c_str()));
      ETG_TRACE_USR4(("traceMessage: [0x%p] vehicle product ID = \"%x\"", (const void*) this, _vehicleConfig._productID));
      ETG_TRACE_USR4(("traceMessage: [0x%p] vehicle vendor ID = \"%x\"", (const void*) this, _vehicleConfig._vendorID));
      ETG_TRACE_USR4(("traceMessage: [0x%p] vehicle vendorIDSource = \"%x\"", (const void*) this, _vehicleConfig._vendorIDSource));
      ETG_TRACE_USR4(("traceMessage: [0x%p] vehicle product version = \"%x\"", (const void*) this, _vehicleConfig._productVersion));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_OnDevicePaired::PmCoreIfMessage_OnDevicePaired() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ON_DEVICE_PAIRED, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _pairedDeviceInfo()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDevicePaired: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_OnDevicePaired::PmCoreIfMessage_OnDevicePaired(
         IN const BasicDeviceDetails& pairedDeviceInfo, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ON_DEVICE_PAIRED, act, origin),
         _pairedDeviceInfo(pairedDeviceInfo)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDevicePaired: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_OnDevicePaired::PmCoreIfMessage_OnDevicePaired(
         IN const PmCoreIfMessage_OnDevicePaired& other) : PmCoreIfMessageRequest(other),
         _pairedDeviceInfo(other._pairedDeviceInfo)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDevicePaired: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_OnDevicePaired::~PmCoreIfMessage_OnDevicePaired()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDevicePaired: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_OnDevicePaired& first, PmCoreIfMessage_OnDevicePaired& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._pairedDeviceInfo, second._pairedDeviceInfo);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_OnDevicePaired& PmCoreIfMessage_OnDevicePaired::operator=(
         PmCoreIfMessage_OnDevicePaired other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_OnDevicePaired::operator==(const PmCoreIfMessage_OnDevicePaired& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_pairedDeviceInfo == rhs._pairedDeviceInfo));
   }

   void PmCoreIfMessage_OnDevicePaired::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      (const_cast <PmCoreIfMessage_OnDevicePaired*> (this))->_pairedDeviceInfo.printDeviceIdentity();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_OnDeviceConnected::PmCoreIfMessage_OnDeviceConnected() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ON_DEVICE_CONNECTED, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _connectedDeviceInfo()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceConnected: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_OnDeviceConnected::PmCoreIfMessage_OnDeviceConnected(
         IN const BasicDeviceDetails& connectedDeviceInfo, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ON_DEVICE_CONNECTED, act, origin),
         _connectedDeviceInfo(connectedDeviceInfo)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceConnected: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_OnDeviceConnected::PmCoreIfMessage_OnDeviceConnected(
         IN const PmCoreIfMessage_OnDeviceConnected& other) : PmCoreIfMessageRequest(other),
         _connectedDeviceInfo(other._connectedDeviceInfo)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceConnected: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_OnDeviceConnected::~PmCoreIfMessage_OnDeviceConnected()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceConnected: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_OnDeviceConnected& first, PmCoreIfMessage_OnDeviceConnected& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._connectedDeviceInfo, second._connectedDeviceInfo);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_OnDeviceConnected& PmCoreIfMessage_OnDeviceConnected::operator=(
         PmCoreIfMessage_OnDeviceConnected other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_OnDeviceConnected::operator==(const PmCoreIfMessage_OnDeviceConnected& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_connectedDeviceInfo == rhs._connectedDeviceInfo));
   }

   void PmCoreIfMessage_OnDeviceConnected::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      (const_cast <PmCoreIfMessage_OnDeviceConnected*> (this))->_connectedDeviceInfo.printDeviceIdentity();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_OnDeviceDisconnected::PmCoreIfMessage_OnDeviceDisconnected() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ON_DEVICE_DISCONNECTED, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceDisconnected: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_OnDeviceDisconnected::PmCoreIfMessage_OnDeviceDisconnected(
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ON_DEVICE_DISCONNECTED, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceDisconnected: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_OnDeviceDisconnected::PmCoreIfMessage_OnDeviceDisconnected(
         IN const PmCoreIfMessage_OnDeviceDisconnected& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceDisconnected: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_OnDeviceDisconnected::~PmCoreIfMessage_OnDeviceDisconnected()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceDisconnected: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_OnDeviceDisconnected& first, PmCoreIfMessage_OnDeviceDisconnected& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_OnDeviceDisconnected& PmCoreIfMessage_OnDeviceDisconnected::operator=(
         PmCoreIfMessage_OnDeviceDisconnected other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_OnDeviceDisconnected::operator==(const PmCoreIfMessage_OnDeviceDisconnected& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_OnDeviceDisconnected::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_OnDeviceDeleted::PmCoreIfMessage_OnDeviceDeleted() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ON_DEVICE_DELETED, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceDeleted: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_OnDeviceDeleted::PmCoreIfMessage_OnDeviceDeleted(IN const BdAddress& bdAddress,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ON_DEVICE_DELETED, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceDeleted: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_OnDeviceDeleted::PmCoreIfMessage_OnDeviceDeleted(
         IN const PmCoreIfMessage_OnDeviceDeleted& other) :
         PmCoreIfMessageRequest(other), _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceDeleted: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_OnDeviceDeleted::~PmCoreIfMessage_OnDeviceDeleted()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_OnDeviceDeleted: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_OnDeviceDeleted& first, PmCoreIfMessage_OnDeviceDeleted& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_OnDeviceDeleted& PmCoreIfMessage_OnDeviceDeleted::operator=(
         PmCoreIfMessage_OnDeviceDeleted other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_OnDeviceDeleted::operator==(const PmCoreIfMessage_OnDeviceDeleted& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_OnDeviceDeleted::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetSystemWideRingtoneListRequest::PmCoreIfMessage_GetSystemWideRingtoneListRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SYSTEM_WIDE_RINGTONE_LIST_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSystemWideRingtoneListRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetSystemWideRingtoneListRequest::PmCoreIfMessage_GetSystemWideRingtoneListRequest(
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SYSTEM_WIDE_RINGTONE_LIST_REQUEST, act, origin)

   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSystemWideRingtoneListRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSystemWideRingtoneListRequest::PmCoreIfMessage_GetSystemWideRingtoneListRequest(
         IN const PmCoreIfMessage_GetSystemWideRingtoneListRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSystemWideRingtoneListRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSystemWideRingtoneListRequest::~PmCoreIfMessage_GetSystemWideRingtoneListRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSystemWideRingtoneListRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetSystemWideRingtoneListRequest& first,
         PmCoreIfMessage_GetSystemWideRingtoneListRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetSystemWideRingtoneListRequest& PmCoreIfMessage_GetSystemWideRingtoneListRequest::operator=(
         PmCoreIfMessage_GetSystemWideRingtoneListRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetSystemWideRingtoneListRequest::operator==(
         const PmCoreIfMessage_GetSystemWideRingtoneListRequest& rhs) const
   {
      return (PmCoreIfMessageRequest::operator ==(rhs));
   }

   void PmCoreIfMessage_GetSystemWideRingtoneListRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetDeviceRingtoneListRequest::PmCoreIfMessage_GetDeviceRingtoneListRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_DEVICE_RINGTONE_LIST_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetDeviceRingtoneListRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetDeviceRingtoneListRequest::PmCoreIfMessage_GetDeviceRingtoneListRequest(IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_DEVICE_RINGTONE_LIST_REQUEST, act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetDeviceRingtoneListRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetDeviceRingtoneListRequest::PmCoreIfMessage_GetDeviceRingtoneListRequest(
         IN const PmCoreIfMessage_GetDeviceRingtoneListRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetDeviceRingtoneListRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetDeviceRingtoneListRequest::~PmCoreIfMessage_GetDeviceRingtoneListRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetDeviceRingtoneListRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetDeviceRingtoneListRequest& first, PmCoreIfMessage_GetDeviceRingtoneListRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetDeviceRingtoneListRequest& PmCoreIfMessage_GetDeviceRingtoneListRequest::operator=(
         PmCoreIfMessage_GetDeviceRingtoneListRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetDeviceRingtoneListRequest::operator==(
         const PmCoreIfMessage_GetDeviceRingtoneListRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs)));
   }

   void PmCoreIfMessage_GetDeviceRingtoneListRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SetRingtoneRequest::PmCoreIfMessage_SetRingtoneRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_RINGTONE_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _ringtoneId(0u)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetRingtoneRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetRingtoneRequest::PmCoreIfMessage_SetRingtoneRequest(IN const BdAddress& bdAddress,
         IN const RingtoneId ringtoneId, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_RINGTONE_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _ringtoneId(ringtoneId)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetRingtoneRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetRingtoneRequest::PmCoreIfMessage_SetRingtoneRequest(
         IN const PmCoreIfMessage_SetRingtoneRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _ringtoneId(other._ringtoneId)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetRingtoneRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetRingtoneRequest::~PmCoreIfMessage_SetRingtoneRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetRingtoneRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetRingtoneRequest& first, PmCoreIfMessage_SetRingtoneRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._ringtoneId, second._ringtoneId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetRingtoneRequest& PmCoreIfMessage_SetRingtoneRequest::operator=(
         PmCoreIfMessage_SetRingtoneRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetRingtoneRequest::operator==(const PmCoreIfMessage_SetRingtoneRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_ringtoneId == rhs._ringtoneId));
   }

   void PmCoreIfMessage_SetRingtoneRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", RingtoneId = %u",
            (const void*) this, _bdAddress.c_str(), _ringtoneId));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_PlayRingtoneRequest::PmCoreIfMessage_PlayRingtoneRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_PLAY_RINGTONE_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(DEVICE_ADDRESS_ALL),
         _ringtoneId(0u),
         _playCount(0u)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PlayRingtoneRequest: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_PlayRingtoneRequest::PmCoreIfMessage_PlayRingtoneRequest(IN const BdAddress& bdAddress,
         IN const RingtoneId ringtoneId,IN const PlayCount playcount, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_PLAY_RINGTONE_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _ringtoneId(ringtoneId),
         _playCount(playcount)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PlayRingtoneRequest: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_PlayRingtoneRequest::PmCoreIfMessage_PlayRingtoneRequest(
         IN const PmCoreIfMessage_PlayRingtoneRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _ringtoneId(other._ringtoneId),
         _playCount(other._playCount)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PlayRingtoneRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_PlayRingtoneRequest::~PmCoreIfMessage_PlayRingtoneRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_PlayRingtoneRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_PlayRingtoneRequest& first, PmCoreIfMessage_PlayRingtoneRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._ringtoneId, second._ringtoneId);
      swap(first._playCount, second._playCount);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_PlayRingtoneRequest& PmCoreIfMessage_PlayRingtoneRequest::operator=(
         PmCoreIfMessage_PlayRingtoneRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_PlayRingtoneRequest::operator==(const PmCoreIfMessage_PlayRingtoneRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            &&(this->_bdAddress == rhs._bdAddress)
            && (this->_ringtoneId == rhs._ringtoneId)
            && (this->_playCount == rhs._playCount));
   }

   void PmCoreIfMessage_PlayRingtoneRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", RingtoneId = %u PlayCount = %u",
            (const void*) this, _bdAddress.c_str(), _ringtoneId, _playCount));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_StopRingtoneRequest::PmCoreIfMessage_StopRingtoneRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_STOP_RINGTONE_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(DEVICE_ADDRESS_ALL)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StopRingtoneRequest: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_StopRingtoneRequest::PmCoreIfMessage_StopRingtoneRequest(IN const BdAddress& bdAddress,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_STOP_RINGTONE_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StopRingtoneRequest: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_StopRingtoneRequest::PmCoreIfMessage_StopRingtoneRequest(
         IN const PmCoreIfMessage_StopRingtoneRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StopRingtoneRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StopRingtoneRequest::~PmCoreIfMessage_StopRingtoneRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StopRingtoneRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_StopRingtoneRequest& first, PmCoreIfMessage_StopRingtoneRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_StopRingtoneRequest& PmCoreIfMessage_StopRingtoneRequest::operator=(
         PmCoreIfMessage_StopRingtoneRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_StopRingtoneRequest::operator==(const PmCoreIfMessage_StopRingtoneRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
               &&(this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_StopRingtoneRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\" ", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest::PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SUPPRESS_RINGTONE_ON_OFF_LIST_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest::PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest(
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SUPPRESS_RINGTONE_ON_OFF_LIST_REQUEST, act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest::PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest(
         IN const PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest: [0x%p] copy-creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest::~PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest: [0x%p] destroying PM Core IF message "
            "object", (void*) this));
   }

   void swap(PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& first,
         PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest::operator=(
         PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest::operator==(
         const PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs)));
   }

   void PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SuppressRingtoneOnOffRequest::PmCoreIfMessage_SuppressRingtoneOnOffRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SUPPRESS_RINGTONE_ON_OFF_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _suppressRingtoneState(false)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SuppressRingtoneOnOffRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SuppressRingtoneOnOffRequest::PmCoreIfMessage_SuppressRingtoneOnOffRequest(
         IN const BdAddress& bdAddress, IN const SuppressRingtoneState suppressRingtoneState, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SUPPRESS_RINGTONE_ON_OFF_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _suppressRingtoneState(suppressRingtoneState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SuppressRingtoneOnOffRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SuppressRingtoneOnOffRequest::PmCoreIfMessage_SuppressRingtoneOnOffRequest(
         IN const PmCoreIfMessage_SuppressRingtoneOnOffRequest& other) :  PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _suppressRingtoneState(other._suppressRingtoneState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SuppressRingtoneOnOffRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SuppressRingtoneOnOffRequest::~PmCoreIfMessage_SuppressRingtoneOnOffRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SuppressRingtoneOnOffRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SuppressRingtoneOnOffRequest& first,
         PmCoreIfMessage_SuppressRingtoneOnOffRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._suppressRingtoneState, second._suppressRingtoneState);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SuppressRingtoneOnOffRequest& PmCoreIfMessage_SuppressRingtoneOnOffRequest::operator=(
         PmCoreIfMessage_SuppressRingtoneOnOffRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SuppressRingtoneOnOffRequest::operator==(
         const PmCoreIfMessage_SuppressRingtoneOnOffRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_suppressRingtoneState == rhs._suppressRingtoneState));
   }

   void PmCoreIfMessage_SuppressRingtoneOnOffRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", SuppressRingtoneState = \"%10s\"",
            (const void*) this, _bdAddress.c_str(), _suppressRingtoneState ? "TRUE" : "FALSE"));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest::PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_AUTO_WAITING_MODE_ON_OFF_LIST_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest::PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest(
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_AUTO_WAITING_MODE_ON_OFF_LIST_REQUEST, act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest::PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest(
         IN const PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest: [0x%p] copy-creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest::~PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest: [0x%p] destroying PM Core IF message "
            "object", (void*) this));
   }

   void swap(PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& first,
         PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest::operator=(
         PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest::operator==(
         const PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs)));
   }

   void PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_AutoWaitingModeOnOffRequest::PmCoreIfMessage_AutoWaitingModeOnOffRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_AUTO_WAITING_MODE_ON_OFF_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _autoWaitingModeState(false)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AutoWaitingModeOnOffRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_AutoWaitingModeOnOffRequest::PmCoreIfMessage_AutoWaitingModeOnOffRequest(
         IN const BdAddress& bdAddress, IN const AutoWaitingModeState autoWaitingModeState, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_AUTO_WAITING_MODE_ON_OFF_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _autoWaitingModeState(autoWaitingModeState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AutoWaitingModeOnOffRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AutoWaitingModeOnOffRequest::PmCoreIfMessage_AutoWaitingModeOnOffRequest(
         IN const PmCoreIfMessage_AutoWaitingModeOnOffRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _autoWaitingModeState(other._autoWaitingModeState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AutoWaitingModeOnOffRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AutoWaitingModeOnOffRequest::~PmCoreIfMessage_AutoWaitingModeOnOffRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AutoWaitingModeOnOffRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_AutoWaitingModeOnOffRequest& first, PmCoreIfMessage_AutoWaitingModeOnOffRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._autoWaitingModeState, second._autoWaitingModeState);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_AutoWaitingModeOnOffRequest& PmCoreIfMessage_AutoWaitingModeOnOffRequest::operator=(
         PmCoreIfMessage_AutoWaitingModeOnOffRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_AutoWaitingModeOnOffRequest::operator==(
         const PmCoreIfMessage_AutoWaitingModeOnOffRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_autoWaitingModeState == rhs._autoWaitingModeState));
   }

   void PmCoreIfMessage_AutoWaitingModeOnOffRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", AutoWaitingModeState = \"%10s\"",
            (const void*) this, _bdAddress.c_str(), _autoWaitingModeState ? "TRUE" : "FALSE"));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SetWaitingModeFilePathRequest::PmCoreIfMessage_SetWaitingModeFilePathRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_WAITING_MODE_FILE_PATH_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _waitingModeFilePath("","")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetWaitingModeFilePathRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SetWaitingModeFilePathRequest::PmCoreIfMessage_SetWaitingModeFilePathRequest(
         IN const WaitingModeFilePath& waitingModeFilePath, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_WAITING_MODE_FILE_PATH_REQUEST, act, origin),
         _waitingModeFilePath(waitingModeFilePath)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetWaitingModeFilePathRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetWaitingModeFilePathRequest::PmCoreIfMessage_SetWaitingModeFilePathRequest(
         IN const PmCoreIfMessage_SetWaitingModeFilePathRequest& other) : PmCoreIfMessageRequest(other),
         _waitingModeFilePath(other._waitingModeFilePath)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetWaitingModeFilePathRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetWaitingModeFilePathRequest::~PmCoreIfMessage_SetWaitingModeFilePathRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetWaitingModeFilePathRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetWaitingModeFilePathRequest& first,
         PmCoreIfMessage_SetWaitingModeFilePathRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._waitingModeFilePath, second._waitingModeFilePath);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetWaitingModeFilePathRequest& PmCoreIfMessage_SetWaitingModeFilePathRequest::operator=(
         PmCoreIfMessage_SetWaitingModeFilePathRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetWaitingModeFilePathRequest::operator==(
         const PmCoreIfMessage_SetWaitingModeFilePathRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_waitingModeFilePath == rhs._waitingModeFilePath));
   }

   void PmCoreIfMessage_SetWaitingModeFilePathRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] FilePathNB = \"%80s\", FilePathWB = \"%80s\"",
            (const void*) this, _waitingModeFilePath._filePathNB.c_str(), _waitingModeFilePath._filePathNB.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetWaitingModeFilePathRequest::PmCoreIfMessage_GetWaitingModeFilePathRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_WAITING_MODE_FILE_PATH_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetWaitingModeFilePathRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetWaitingModeFilePathRequest::PmCoreIfMessage_GetWaitingModeFilePathRequest(IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_WAITING_MODE_FILE_PATH_REQUEST, act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetWaitingModeFilePathRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetWaitingModeFilePathRequest::PmCoreIfMessage_GetWaitingModeFilePathRequest(
         IN const PmCoreIfMessage_GetWaitingModeFilePathRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetWaitingModeFilePathRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetWaitingModeFilePathRequest::~PmCoreIfMessage_GetWaitingModeFilePathRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetWaitingModeFilePathRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetWaitingModeFilePathRequest& first,
         PmCoreIfMessage_GetWaitingModeFilePathRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetWaitingModeFilePathRequest& PmCoreIfMessage_GetWaitingModeFilePathRequest::operator=(
         PmCoreIfMessage_GetWaitingModeFilePathRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetWaitingModeFilePathRequest::operator==(
         const PmCoreIfMessage_GetWaitingModeFilePathRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs)));
   }

   void PmCoreIfMessage_GetWaitingModeFilePathRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetPhoneCallAudioActiveRequest::PmCoreIfMessage_GetPhoneCallAudioActiveRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_PHONE_CALL_AUDIO_ACTIVE_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetPhoneCallAudioActiveRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetPhoneCallAudioActiveRequest::PmCoreIfMessage_GetPhoneCallAudioActiveRequest(IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_PHONE_CALL_AUDIO_ACTIVE_REQUEST, act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetPhoneCallAudioActiveRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetPhoneCallAudioActiveRequest::PmCoreIfMessage_GetPhoneCallAudioActiveRequest(
         IN const PmCoreIfMessage_GetPhoneCallAudioActiveRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetPhoneCallAudioActiveRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetPhoneCallAudioActiveRequest::~PmCoreIfMessage_GetPhoneCallAudioActiveRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetPhoneCallAudioActiveRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetPhoneCallAudioActiveRequest& first,
         PmCoreIfMessage_GetPhoneCallAudioActiveRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetPhoneCallAudioActiveRequest& PmCoreIfMessage_GetPhoneCallAudioActiveRequest::operator=(
         PmCoreIfMessage_GetPhoneCallAudioActiveRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetPhoneCallAudioActiveRequest::operator==(
         const PmCoreIfMessage_GetPhoneCallAudioActiveRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs)));
   }

   void PmCoreIfMessage_GetPhoneCallAudioActiveRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetActivePassiveDeviceListRequest::PmCoreIfMessage_GetActivePassiveDeviceListRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_ACTIVE_PASSIVE_DEVICE_LIST_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetActivePassiveDeviceListRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetActivePassiveDeviceListRequest::PmCoreIfMessage_GetActivePassiveDeviceListRequest(
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_ACTIVE_PASSIVE_DEVICE_LIST_REQUEST, act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetActivePassiveDeviceListRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetActivePassiveDeviceListRequest::PmCoreIfMessage_GetActivePassiveDeviceListRequest(
         IN const PmCoreIfMessage_GetActivePassiveDeviceListRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetActivePassiveDeviceListRequest: [0x%p] copy-creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_GetActivePassiveDeviceListRequest::~PmCoreIfMessage_GetActivePassiveDeviceListRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetActivePassiveDeviceListRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetActivePassiveDeviceListRequest& first,
         PmCoreIfMessage_GetActivePassiveDeviceListRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetActivePassiveDeviceListRequest& PmCoreIfMessage_GetActivePassiveDeviceListRequest::operator=(
         PmCoreIfMessage_GetActivePassiveDeviceListRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetActivePassiveDeviceListRequest::operator==(
         const PmCoreIfMessage_GetActivePassiveDeviceListRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs)));
   }

   void PmCoreIfMessage_GetActivePassiveDeviceListRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SetActivePhoneRequest::PmCoreIfMessage_SetActivePhoneRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_ACTIVE_PHONE_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _deviceAddressList()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetActivePhoneRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SetActivePhoneRequest::PmCoreIfMessage_SetActivePhoneRequest(
         IN const BdAddressList& deviceAddressList, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_ACTIVE_PHONE_REQUEST, act, origin),
         _deviceAddressList(deviceAddressList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetActivePhoneRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetActivePhoneRequest::PmCoreIfMessage_SetActivePhoneRequest(
         IN const PmCoreIfMessage_SetActivePhoneRequest& other) :
         PmCoreIfMessageRequest(other),
         _deviceAddressList(other._deviceAddressList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetActivePhoneRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetActivePhoneRequest::~PmCoreIfMessage_SetActivePhoneRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetActivePhoneRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetActivePhoneRequest& first, PmCoreIfMessage_SetActivePhoneRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._deviceAddressList, second._deviceAddressList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetActivePhoneRequest& PmCoreIfMessage_SetActivePhoneRequest::operator=(
         PmCoreIfMessage_SetActivePhoneRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetActivePhoneRequest::operator==(const PmCoreIfMessage_SetActivePhoneRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_deviceAddressList == rhs._deviceAddressList));
   }

   void PmCoreIfMessage_SetActivePhoneRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      BdAddressList::const_iterator iter;
      for (iter = _deviceAddressList.begin() ; iter != _deviceAddressList.end(); ++iter )
      {
         ETG_TRACE_USR4(("traceMessage: BdAddress = %s", iter->c_str()));
      }
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_RestoreDefaultSettingRequest::PmCoreIfMessage_RestoreDefaultSettingRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_RESTORE_DEFAULT_SETTING_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RestoreDefaultSettingRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_RestoreDefaultSettingRequest::PmCoreIfMessage_RestoreDefaultSettingRequest(
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_RESTORE_DEFAULT_SETTING_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RestoreDefaultSettingRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_RestoreDefaultSettingRequest::PmCoreIfMessage_RestoreDefaultSettingRequest(
         IN const PmCoreIfMessage_RestoreDefaultSettingRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RestoreDefaultSettingRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_RestoreDefaultSettingRequest::~PmCoreIfMessage_RestoreDefaultSettingRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RestoreDefaultSettingRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_RestoreDefaultSettingRequest& first, PmCoreIfMessage_RestoreDefaultSettingRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_RestoreDefaultSettingRequest& PmCoreIfMessage_RestoreDefaultSettingRequest::operator=(
         PmCoreIfMessage_RestoreDefaultSettingRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_RestoreDefaultSettingRequest::operator==(
         const PmCoreIfMessage_RestoreDefaultSettingRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_RestoreDefaultSettingRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetSupportedFeaturesRequest::PmCoreIfMessage_GetSupportedFeaturesRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SUPPORTED_FEATURES_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSupportedFeaturesRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetSupportedFeaturesRequest::PmCoreIfMessage_GetSupportedFeaturesRequest(
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SUPPORTED_FEATURES_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSupportedFeaturesRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSupportedFeaturesRequest::PmCoreIfMessage_GetSupportedFeaturesRequest(
         IN const PmCoreIfMessage_GetSupportedFeaturesRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSupportedFeaturesRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSupportedFeaturesRequest::~PmCoreIfMessage_GetSupportedFeaturesRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSupportedFeaturesRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetSupportedFeaturesRequest& first, PmCoreIfMessage_GetSupportedFeaturesRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetSupportedFeaturesRequest& PmCoreIfMessage_GetSupportedFeaturesRequest::operator=(
         PmCoreIfMessage_GetSupportedFeaturesRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetSupportedFeaturesRequest::operator==(
         const PmCoreIfMessage_GetSupportedFeaturesRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetSupportedFeaturesRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetNetworkStatusRequest::PmCoreIfMessage_GetNetworkStatusRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_NETWORK_STATUS_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetNetworkStatusRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetNetworkStatusRequest::PmCoreIfMessage_GetNetworkStatusRequest(IN const BdAddress& bdAddress,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_NETWORK_STATUS_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetNetworkStatusRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetNetworkStatusRequest::PmCoreIfMessage_GetNetworkStatusRequest(
         IN const PmCoreIfMessage_GetNetworkStatusRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetNetworkStatusRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetNetworkStatusRequest::~PmCoreIfMessage_GetNetworkStatusRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetNetworkStatusRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetNetworkStatusRequest& first, PmCoreIfMessage_GetNetworkStatusRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetNetworkStatusRequest& PmCoreIfMessage_GetNetworkStatusRequest::operator=(
         PmCoreIfMessage_GetNetworkStatusRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetNetworkStatusRequest::operator==(const PmCoreIfMessage_GetNetworkStatusRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetNetworkStatusRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************
   PmCoreIfMessage_GetSignalStrengthRequest::PmCoreIfMessage_GetSignalStrengthRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SIGNAL_STRENGTH_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSignalStrengthRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetSignalStrengthRequest::PmCoreIfMessage_GetSignalStrengthRequest(IN const BdAddress& bdAddress,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SIGNAL_STRENGTH_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSignalStrengthRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSignalStrengthRequest::PmCoreIfMessage_GetSignalStrengthRequest(
         IN const PmCoreIfMessage_GetSignalStrengthRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSignalStrengthRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSignalStrengthRequest::~PmCoreIfMessage_GetSignalStrengthRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSignalStrengthRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetSignalStrengthRequest& first, PmCoreIfMessage_GetSignalStrengthRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetSignalStrengthRequest& PmCoreIfMessage_GetSignalStrengthRequest::operator=(
         PmCoreIfMessage_GetSignalStrengthRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetSignalStrengthRequest::operator==(const PmCoreIfMessage_GetSignalStrengthRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetSignalStrengthRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetBatteryChargeLevelRequest::PmCoreIfMessage_GetBatteryChargeLevelRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_BATTERY_CHARGE_LEVEL_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetBatteryChargeLevelRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetBatteryChargeLevelRequest::PmCoreIfMessage_GetBatteryChargeLevelRequest(
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_BATTERY_CHARGE_LEVEL_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetBatteryChargeLevelRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetBatteryChargeLevelRequest::PmCoreIfMessage_GetBatteryChargeLevelRequest(
         IN const PmCoreIfMessage_GetBatteryChargeLevelRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetBatteryChargeLevelRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetBatteryChargeLevelRequest::~PmCoreIfMessage_GetBatteryChargeLevelRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetBatteryChargeLevelRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetBatteryChargeLevelRequest& first, PmCoreIfMessage_GetBatteryChargeLevelRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetBatteryChargeLevelRequest& PmCoreIfMessage_GetBatteryChargeLevelRequest::operator=(
         PmCoreIfMessage_GetBatteryChargeLevelRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetBatteryChargeLevelRequest::operator==(
         const PmCoreIfMessage_GetBatteryChargeLevelRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetBatteryChargeLevelRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetNetworkOperatorRequest::PmCoreIfMessage_GetNetworkOperatorRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_NETWORK_OPERATOR_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetNetworkOperatorRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetNetworkOperatorRequest::PmCoreIfMessage_GetNetworkOperatorRequest(IN const BdAddress& bdAddress,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_NETWORK_OPERATOR_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetNetworkOperatorRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetNetworkOperatorRequest::PmCoreIfMessage_GetNetworkOperatorRequest(
         IN const PmCoreIfMessage_GetNetworkOperatorRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetNetworkOperatorRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetNetworkOperatorRequest::~PmCoreIfMessage_GetNetworkOperatorRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetNetworkOperatorRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetNetworkOperatorRequest& first, PmCoreIfMessage_GetNetworkOperatorRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetNetworkOperatorRequest& PmCoreIfMessage_GetNetworkOperatorRequest::operator=(
         PmCoreIfMessage_GetNetworkOperatorRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetNetworkOperatorRequest::operator==(
         const PmCoreIfMessage_GetNetworkOperatorRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetNetworkOperatorRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_VoiceMailListRequest::PmCoreIfMessage_VoiceMailListRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_VOICE_MAIL_LIST_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_VoiceMailListRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_VoiceMailListRequest::PmCoreIfMessage_VoiceMailListRequest(IN const BdAddress& bdAddress,
         IN const ActType act,IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_VOICE_MAIL_LIST_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_VoiceMailListRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_VoiceMailListRequest::PmCoreIfMessage_VoiceMailListRequest(
         IN const PmCoreIfMessage_VoiceMailListRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_VoiceMailListRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_VoiceMailListRequest::~PmCoreIfMessage_VoiceMailListRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_VoiceMailListRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_VoiceMailListRequest& first, PmCoreIfMessage_VoiceMailListRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_VoiceMailListRequest& PmCoreIfMessage_VoiceMailListRequest::operator=(
         PmCoreIfMessage_VoiceMailListRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_VoiceMailListRequest::operator==(const PmCoreIfMessage_VoiceMailListRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_VoiceMailListRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetCallStatusListRequest::PmCoreIfMessage_GetCallStatusListRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_CALL_STATUS_LIST_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetCallStatusListRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetCallStatusListRequest::PmCoreIfMessage_GetCallStatusListRequest(IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) : PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_CALL_STATUS_LIST_REQUEST,
               act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetCallStatusListRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetCallStatusListRequest::PmCoreIfMessage_GetCallStatusListRequest(
         IN const PmCoreIfMessage_GetCallStatusListRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetCallStatusListRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetCallStatusListRequest::~PmCoreIfMessage_GetCallStatusListRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetCallStatusListRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetCallStatusListRequest& first, PmCoreIfMessage_GetCallStatusListRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetCallStatusListRequest& PmCoreIfMessage_GetCallStatusListRequest::operator=(
         PmCoreIfMessage_GetCallStatusListRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetCallStatusListRequest::operator==(const PmCoreIfMessage_GetCallStatusListRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs)));
   }

   void PmCoreIfMessage_GetCallStatusListRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetMicrophoneMuteStateRequest::PmCoreIfMessage_GetMicrophoneMuteStateRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_MICROPHONE_MUTE_STATE_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetMicrophoneMuteStateRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetMicrophoneMuteStateRequest::PmCoreIfMessage_GetMicrophoneMuteStateRequest(IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_MICROPHONE_MUTE_STATE_REQUEST, act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetMicrophoneMuteStateRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetMicrophoneMuteStateRequest::PmCoreIfMessage_GetMicrophoneMuteStateRequest(
         IN const PmCoreIfMessage_GetMicrophoneMuteStateRequest& other) : PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetMicrophoneMuteStateRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetMicrophoneMuteStateRequest::~PmCoreIfMessage_GetMicrophoneMuteStateRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetMicrophoneMuteStateRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetMicrophoneMuteStateRequest& first,
         PmCoreIfMessage_GetMicrophoneMuteStateRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetMicrophoneMuteStateRequest& PmCoreIfMessage_GetMicrophoneMuteStateRequest::operator=(
         PmCoreIfMessage_GetMicrophoneMuteStateRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetMicrophoneMuteStateRequest::operator==(
         const PmCoreIfMessage_GetMicrophoneMuteStateRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs)));
   }

   void PmCoreIfMessage_GetMicrophoneMuteStateRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SetMicrophoneMuteStateRequest::PmCoreIfMessage_SetMicrophoneMuteStateRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_MICROPHONE_MUTE_STATE_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _microphoneMuteState(false)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetMicrophoneMuteStateRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SetMicrophoneMuteStateRequest::PmCoreIfMessage_SetMicrophoneMuteStateRequest(
         IN const MuteState microphoneMuteState, IN const ActType act,IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_MICROPHONE_MUTE_STATE_REQUEST, act, origin),
         _microphoneMuteState(microphoneMuteState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetMicrophoneMuteStateRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetMicrophoneMuteStateRequest::PmCoreIfMessage_SetMicrophoneMuteStateRequest(
         IN const PmCoreIfMessage_SetMicrophoneMuteStateRequest& other) : PmCoreIfMessageRequest(other),
         _microphoneMuteState(other._microphoneMuteState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetMicrophoneMuteStateRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetMicrophoneMuteStateRequest::~PmCoreIfMessage_SetMicrophoneMuteStateRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetMicrophoneMuteStateRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetMicrophoneMuteStateRequest& first,
         PmCoreIfMessage_SetMicrophoneMuteStateRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._microphoneMuteState, second._microphoneMuteState);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetMicrophoneMuteStateRequest& PmCoreIfMessage_SetMicrophoneMuteStateRequest::operator=(
         PmCoreIfMessage_SetMicrophoneMuteStateRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetMicrophoneMuteStateRequest::operator==(
         const PmCoreIfMessage_SetMicrophoneMuteStateRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_microphoneMuteState == rhs._microphoneMuteState));
   }

   void PmCoreIfMessage_SetMicrophoneMuteStateRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] MicrophoneMuteState = \"%10s\"", (const void*) this,
            _microphoneMuteState ? "TRUE" : "FALSE"));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_MergeCallsRequest::PmCoreIfMessage_MergeCallsRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_MERGE_CALLS_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _mergeCallList()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_MergeCallsRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_MergeCallsRequest::PmCoreIfMessage_MergeCallsRequest(IN const BdAddress& bdAddress,
         IN const CallInstanceList& mergeCallList, IN const ActType act,IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_MERGE_CALLS_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _mergeCallList(mergeCallList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_MergeCallsRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_MergeCallsRequest::PmCoreIfMessage_MergeCallsRequest(
         IN const PmCoreIfMessage_MergeCallsRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _mergeCallList(other._mergeCallList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_MergeCallsRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_MergeCallsRequest::~PmCoreIfMessage_MergeCallsRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_MergeCallsRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_MergeCallsRequest& first, PmCoreIfMessage_MergeCallsRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._mergeCallList, second._mergeCallList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_MergeCallsRequest& PmCoreIfMessage_MergeCallsRequest::operator=(
         PmCoreIfMessage_MergeCallsRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_MergeCallsRequest::operator==(const PmCoreIfMessage_MergeCallsRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_mergeCallList == rhs._mergeCallList));
   }

   void PmCoreIfMessage_MergeCallsRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      CallInstanceList::const_iterator it;

      ETG_TRACE_USR4(("traceMessage: BdAddress = \"%50s\"", _bdAddress.c_str()));

      for (it = _mergeCallList.begin() ; it != _mergeCallList.end(); ++it )
      {
         ETG_TRACE_USR4(("traceMessage: CallInstance = %u", *it));
      }
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SplitCallsRequest::PmCoreIfMessage_SplitCallsRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SPLIT_CALLS_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _splitCallList()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SplitCallsRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SplitCallsRequest::PmCoreIfMessage_SplitCallsRequest(IN const BdAddress& bdAddress,
         IN const CallInstanceList& splitCallList, IN const ActType act,IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SPLIT_CALLS_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _splitCallList(splitCallList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SplitCallsRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SplitCallsRequest::PmCoreIfMessage_SplitCallsRequest(
         IN const PmCoreIfMessage_SplitCallsRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _splitCallList(other._splitCallList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SplitCallsRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SplitCallsRequest::~PmCoreIfMessage_SplitCallsRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SplitCallsRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SplitCallsRequest& first, PmCoreIfMessage_SplitCallsRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._splitCallList, second._splitCallList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SplitCallsRequest& PmCoreIfMessage_SplitCallsRequest::operator=(
         PmCoreIfMessage_SplitCallsRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SplitCallsRequest::operator==(const PmCoreIfMessage_SplitCallsRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_splitCallList == rhs._splitCallList));
   }

   void PmCoreIfMessage_SplitCallsRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      CallInstanceList::const_iterator it;

      ETG_TRACE_USR4(("traceMessage: BdAddress = \"%50s\"", _bdAddress.c_str()));

      for (it = _splitCallList.begin() ; it != _splitCallList.end(); ++it )
      {
         ETG_TRACE_USR4(("traceMessage: CallInstance = %u", *it));
      }
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_HangupCallsRequest::PmCoreIfMessage_HangupCallsRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_HANGUP_CALLS_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _hangupCallList()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupCallsRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_HangupCallsRequest::PmCoreIfMessage_HangupCallsRequest(IN const BdAddress& bdAddress,
         IN const CallInstanceList& hangupCallList, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_HANGUP_CALLS_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _hangupCallList(hangupCallList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupCallsRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HangupCallsRequest::PmCoreIfMessage_HangupCallsRequest(
         IN const PmCoreIfMessage_HangupCallsRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _hangupCallList(other._hangupCallList)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupCallsRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_HangupCallsRequest::~PmCoreIfMessage_HangupCallsRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_HangupCallsRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_HangupCallsRequest& first, PmCoreIfMessage_HangupCallsRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._hangupCallList, second._hangupCallList);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_HangupCallsRequest& PmCoreIfMessage_HangupCallsRequest::operator=(
         PmCoreIfMessage_HangupCallsRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_HangupCallsRequest::operator==(const PmCoreIfMessage_HangupCallsRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_hangupCallList == rhs._hangupCallList));
   }

   void PmCoreIfMessage_HangupCallsRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      CallInstanceList::const_iterator it;

      ETG_TRACE_USR4(("traceMessage: BdAddress = \"%50s\"", _bdAddress.c_str()));

      for (it = _hangupCallList.begin() ; it != _hangupCallList.end(); ++it )
      {
         ETG_TRACE_USR4(("traceMessage: CallInstance = %u", *it));
      }
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_RedialRequest::PmCoreIfMessage_RedialRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_REDIAL_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _hideCallerId(DEFAULT_CALLERID)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RedialRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_RedialRequest::PmCoreIfMessage_RedialRequest(IN const BdAddress& bdAddress,
         IN const HideCallerId hideCallerId, IN const ActType act,IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_REDIAL_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _hideCallerId(hideCallerId)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RedialRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_RedialRequest::PmCoreIfMessage_RedialRequest(IN const PmCoreIfMessage_RedialRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _hideCallerId(other._hideCallerId)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RedialRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_RedialRequest::~PmCoreIfMessage_RedialRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_RedialRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_RedialRequest& first, PmCoreIfMessage_RedialRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._hideCallerId, second._hideCallerId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_RedialRequest& PmCoreIfMessage_RedialRequest::operator=(PmCoreIfMessage_RedialRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_RedialRequest::operator==(const PmCoreIfMessage_RedialRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_hideCallerId == rhs._hideCallerId));
   }

   void PmCoreIfMessage_RedialRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", HideCallerId = %u",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(HideCallerIdEnumType, _hideCallerId)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetSCOConnectionRequest::PmCoreIfMessage_GetSCOConnectionRequest() :
      PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SCO_CONNECTION_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
      _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSCOConnectionRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetSCOConnectionRequest::PmCoreIfMessage_GetSCOConnectionRequest(IN const BdAddress& bdAddress,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_SCO_CONNECTION_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSCOConnectionRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSCOConnectionRequest::PmCoreIfMessage_GetSCOConnectionRequest(
         IN const PmCoreIfMessage_GetSCOConnectionRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSCOConnectionRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetSCOConnectionRequest::~PmCoreIfMessage_GetSCOConnectionRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetSCOConnectionRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetSCOConnectionRequest& first, PmCoreIfMessage_GetSCOConnectionRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetSCOConnectionRequest& PmCoreIfMessage_GetSCOConnectionRequest::operator=(
         PmCoreIfMessage_GetSCOConnectionRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetSCOConnectionRequest::operator==(
         const PmCoreIfMessage_GetSCOConnectionRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetSCOConnectionRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetRingtoneMuteStateRequest::PmCoreIfMessage_GetRingtoneMuteStateRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_RINGTONE_MUTE_STATE_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetRingtoneMuteStateRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetRingtoneMuteStateRequest::PmCoreIfMessage_GetRingtoneMuteStateRequest(
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_RINGTONE_MUTE_STATE_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetRingtoneMuteStateRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetRingtoneMuteStateRequest::PmCoreIfMessage_GetRingtoneMuteStateRequest(
         IN const PmCoreIfMessage_GetRingtoneMuteStateRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetRingtoneMuteStateRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetRingtoneMuteStateRequest::~PmCoreIfMessage_GetRingtoneMuteStateRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetRingtoneMuteStateRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetRingtoneMuteStateRequest& first, PmCoreIfMessage_GetRingtoneMuteStateRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetRingtoneMuteStateRequest& PmCoreIfMessage_GetRingtoneMuteStateRequest::operator=(
         PmCoreIfMessage_GetRingtoneMuteStateRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetRingtoneMuteStateRequest::operator==(
         const PmCoreIfMessage_GetRingtoneMuteStateRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetRingtoneMuteStateRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SetRingtoneMuteStateRequest::PmCoreIfMessage_SetRingtoneMuteStateRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_RINGTONE_MUTE_STATE_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _ringtoneMuteState(false)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetRingtoneMuteStateRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SetRingtoneMuteStateRequest::PmCoreIfMessage_SetRingtoneMuteStateRequest(
         IN const BdAddress& bdAddress, IN const MuteState ringtoneMuteState, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SET_RINGTONE_MUTE_STATE_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _ringtoneMuteState(ringtoneMuteState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetRingtoneMuteStateRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetRingtoneMuteStateRequest::PmCoreIfMessage_SetRingtoneMuteStateRequest(
         IN const PmCoreIfMessage_SetRingtoneMuteStateRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _ringtoneMuteState(other._ringtoneMuteState)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetRingtoneMuteStateRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SetRingtoneMuteStateRequest::~PmCoreIfMessage_SetRingtoneMuteStateRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SetRingtoneMuteStateRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SetRingtoneMuteStateRequest& first, PmCoreIfMessage_SetRingtoneMuteStateRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._ringtoneMuteState, second._ringtoneMuteState);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SetRingtoneMuteStateRequest& PmCoreIfMessage_SetRingtoneMuteStateRequest::operator=(
         PmCoreIfMessage_SetRingtoneMuteStateRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SetRingtoneMuteStateRequest::operator==(
         const PmCoreIfMessage_SetRingtoneMuteStateRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_ringtoneMuteState == rhs._ringtoneMuteState));
   }

   void PmCoreIfMessage_SetRingtoneMuteStateRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", RingtoneMuteState = \"%10s\"",
            (const void*) this, _bdAddress.c_str(), _ringtoneMuteState ? "TRUE" : "FALSE"));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetWaitingModeStateRequest::PmCoreIfMessage_GetWaitingModeStateRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_WAITING_MODE_STATE_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetWaitingModeStateRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetWaitingModeStateRequest::PmCoreIfMessage_GetWaitingModeStateRequest(IN const BdAddress& bdAddress,
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_WAITING_MODE_STATE_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetWaitingModeStateRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetWaitingModeStateRequest::PmCoreIfMessage_GetWaitingModeStateRequest(
         IN const PmCoreIfMessage_GetWaitingModeStateRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetWaitingModeStateRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetWaitingModeStateRequest::~PmCoreIfMessage_GetWaitingModeStateRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetWaitingModeStateRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetWaitingModeStateRequest& first, PmCoreIfMessage_GetWaitingModeStateRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetWaitingModeStateRequest& PmCoreIfMessage_GetWaitingModeStateRequest::operator=(
         PmCoreIfMessage_GetWaitingModeStateRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetWaitingModeStateRequest::operator==(
         const PmCoreIfMessage_GetWaitingModeStateRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetWaitingModeStateRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetRingtonePlaybackStatusRequest::PmCoreIfMessage_GetRingtonePlaybackStatusRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_RINGTONE_PLAYBACK_STATUS_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetRingtonePlaybackStatusRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetRingtonePlaybackStatusRequest::PmCoreIfMessage_GetRingtonePlaybackStatusRequest(
         IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_RINGTONE_PLAYBACK_STATUS_REQUEST, act, origin)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetRingtonePlaybackStatusRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetRingtonePlaybackStatusRequest::PmCoreIfMessage_GetRingtonePlaybackStatusRequest(
         IN const PmCoreIfMessage_GetRingtonePlaybackStatusRequest& other) :
         PmCoreIfMessageRequest(other)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetRingtonePlaybackStatusRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetRingtonePlaybackStatusRequest::~PmCoreIfMessage_GetRingtonePlaybackStatusRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetRingtonePlaybackStatusRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetRingtonePlaybackStatusRequest& first, PmCoreIfMessage_GetRingtonePlaybackStatusRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetRingtonePlaybackStatusRequest& PmCoreIfMessage_GetRingtonePlaybackStatusRequest::operator=(
         PmCoreIfMessage_GetRingtonePlaybackStatusRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetRingtonePlaybackStatusRequest::operator==(
         const PmCoreIfMessage_GetRingtonePlaybackStatusRequest& rhs) const
   {
      return (PmCoreIfMessageRequest::operator ==(rhs));
   }

   void PmCoreIfMessage_GetRingtonePlaybackStatusRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();
   }

   // *****************************************************************************************************************
   PmCoreIfMessage_StartStopWaitingModeRequest::PmCoreIfMessage_StartStopWaitingModeRequest() :
      PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_START_STOP_WAITING_MODE_REQUEST, PM_DEFAULT_ACT,
            PM_CORE_IF_MSG_ORIGIN_CLIENT),
      _bdAddress(""),
      _startStopWaitingMode(false)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopWaitingModeRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_StartStopWaitingModeRequest::PmCoreIfMessage_StartStopWaitingModeRequest(
         IN const BdAddress& bdAddress, IN const StartStop startStopWaitingMode, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_START_STOP_WAITING_MODE_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _startStopWaitingMode(startStopWaitingMode)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopWaitingModeRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StartStopWaitingModeRequest::PmCoreIfMessage_StartStopWaitingModeRequest(
         IN const PmCoreIfMessage_StartStopWaitingModeRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _startStopWaitingMode(other._startStopWaitingMode)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopWaitingModeRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StartStopWaitingModeRequest::~PmCoreIfMessage_StartStopWaitingModeRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopWaitingModeRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_StartStopWaitingModeRequest& first, PmCoreIfMessage_StartStopWaitingModeRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._startStopWaitingMode, second._startStopWaitingMode);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_StartStopWaitingModeRequest& PmCoreIfMessage_StartStopWaitingModeRequest::operator=(
         PmCoreIfMessage_StartStopWaitingModeRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_StartStopWaitingModeRequest::operator==(
         const PmCoreIfMessage_StartStopWaitingModeRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_startStopWaitingMode == rhs._startStopWaitingMode));
   }

   void PmCoreIfMessage_StartStopWaitingModeRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", StartStopWaitingMode = \"%10s\"",
            (const void*) this, _bdAddress.c_str(), _startStopWaitingMode ? "TRUE" : "FALSE"));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_DialRequest::PmCoreIfMessage_DialRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_DIAL_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _telephoneNumber(""),
         _hideCallerId(DEFAULT_CALLERID)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_DialRequest: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_DialRequest::PmCoreIfMessage_DialRequest(IN const BdAddress& bdAddress,
         IN const TelephoneNumber& telephoneNumber, IN const HideCallerId hideCallerId, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_DIAL_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _telephoneNumber(telephoneNumber),
         _hideCallerId(hideCallerId)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_DialRequest: [0x%p] creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_DialRequest::PmCoreIfMessage_DialRequest(IN const PmCoreIfMessage_DialRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _telephoneNumber(other._telephoneNumber),
         _hideCallerId(other._hideCallerId)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_DialRequest: [0x%p] copy-creating PM Core IF message object", (void*) this));
   }

   PmCoreIfMessage_DialRequest::~PmCoreIfMessage_DialRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_DialRequest: [0x%p] destroying PM Core IF message object", (void*) this));
   }

   void swap(PmCoreIfMessage_DialRequest& first, PmCoreIfMessage_DialRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._telephoneNumber, second._telephoneNumber);
      swap(first._hideCallerId, second._hideCallerId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_DialRequest& PmCoreIfMessage_DialRequest::operator=(PmCoreIfMessage_DialRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_DialRequest::operator==(const PmCoreIfMessage_DialRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_telephoneNumber == rhs._telephoneNumber)
            && (this->_hideCallerId == rhs._hideCallerId));
   }

   void PmCoreIfMessage_DialRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", TelephoneNumber = \"%50s\", HideCallerId = %u",
            (const void*) this, _bdAddress.c_str(), _telephoneNumber.c_str(),
            ETG_CENUM(HideCallerIdEnumType, _hideCallerId)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_AcceptCallRequest::PmCoreIfMessage_AcceptCallRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ACCEPT_CALL_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _callInstance(0x00u),
         _acceptOperation(0x00u)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptCallRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_AcceptCallRequest::PmCoreIfMessage_AcceptCallRequest(IN const BdAddress& bdAddress,
         IN const CallInstance callInstance, IN const AcceptOperation operation, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) : PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_ACCEPT_CALL_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _callInstance(callInstance),
         _acceptOperation(operation)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptCallRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AcceptCallRequest::PmCoreIfMessage_AcceptCallRequest(
         IN const PmCoreIfMessage_AcceptCallRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _callInstance(other._callInstance),
         _acceptOperation(other._acceptOperation)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptCallRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_AcceptCallRequest::~PmCoreIfMessage_AcceptCallRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_AcceptCallRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_AcceptCallRequest& first, PmCoreIfMessage_AcceptCallRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._callInstance, second._callInstance);
      swap(first._acceptOperation, second._acceptOperation);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_AcceptCallRequest& PmCoreIfMessage_AcceptCallRequest::operator=(
         PmCoreIfMessage_AcceptCallRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_AcceptCallRequest::operator==(const PmCoreIfMessage_AcceptCallRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_callInstance == rhs._callInstance)
            && (this->_acceptOperation == rhs._acceptOperation));
   }

   void PmCoreIfMessage_AcceptCallRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", CallInstance = %u, AcceptOperation = %u",
            (const void*) this, _bdAddress.c_str(), _callInstance, ETG_CENUM(AcceptOperationEnumType, _acceptOperation)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SwapCallRequest::PmCoreIfMessage_SwapCallRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SWAP_CALL_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _swapOperation(0x00u)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SwapCallRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SwapCallRequest::PmCoreIfMessage_SwapCallRequest(IN const BdAddress& bdAddress,
         IN const SwapOperation operation, IN const ActType act,IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SWAP_CALL_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _swapOperation(operation)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SwapCallRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SwapCallRequest::PmCoreIfMessage_SwapCallRequest(IN const PmCoreIfMessage_SwapCallRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _swapOperation(other._swapOperation)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SwapCallRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SwapCallRequest::~PmCoreIfMessage_SwapCallRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SwapCallRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SwapCallRequest& first, PmCoreIfMessage_SwapCallRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._swapOperation, second._swapOperation);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SwapCallRequest& PmCoreIfMessage_SwapCallRequest::operator=(PmCoreIfMessage_SwapCallRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SwapCallRequest::operator==(const PmCoreIfMessage_SwapCallRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_swapOperation == rhs._swapOperation));
   }

   void PmCoreIfMessage_SwapCallRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", SwapOperation = %u",
            (const void*) this, _bdAddress.c_str(), ETG_CENUM(SwapOperationEnumType, _swapOperation)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SpeedDialRequest::PmCoreIfMessage_SpeedDialRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SPEED_DIAL_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _memDialIndex(0x00u),
         _hideCallerId(DEFAULT_CALLERID)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SpeedDialRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SpeedDialRequest::PmCoreIfMessage_SpeedDialRequest(IN const BdAddress& bdAddress,
         IN const MemDialIndex memDialIndex, IN const HideCallerId hideCallerId, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) : PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SPEED_DIAL_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _memDialIndex(memDialIndex),
         _hideCallerId(hideCallerId)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SpeedDialRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SpeedDialRequest::PmCoreIfMessage_SpeedDialRequest(IN const PmCoreIfMessage_SpeedDialRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _memDialIndex(other._memDialIndex),
         _hideCallerId(other._hideCallerId)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SpeedDialRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SpeedDialRequest::~PmCoreIfMessage_SpeedDialRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SpeedDialRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SpeedDialRequest& first, PmCoreIfMessage_SpeedDialRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._memDialIndex, second._memDialIndex);
      swap(first._hideCallerId, second._hideCallerId);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SpeedDialRequest& PmCoreIfMessage_SpeedDialRequest::operator=(
         PmCoreIfMessage_SpeedDialRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SpeedDialRequest::operator==(const PmCoreIfMessage_SpeedDialRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_memDialIndex == rhs._memDialIndex)
            && (this->_hideCallerId == rhs._hideCallerId));
   }

   void PmCoreIfMessage_SpeedDialRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", MemDialIndex = %u, HideCallerId = %u",
            (const void*) this, _bdAddress.c_str(), _memDialIndex, ETG_CENUM(HideCallerIdEnumType, _hideCallerId)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_SendDTMFRequest::PmCoreIfMessage_SendDTMFRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SEND_DTMF_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _tones("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendDTMFRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_SendDTMFRequest::PmCoreIfMessage_SendDTMFRequest(IN const BdAddress& bdAddress,
         IN const DTMFTones& tones, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_SEND_DTMF_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _tones(tones)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendDTMFRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SendDTMFRequest::PmCoreIfMessage_SendDTMFRequest(IN const PmCoreIfMessage_SendDTMFRequest& other) :
         PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _tones(other._tones)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendDTMFRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_SendDTMFRequest::~PmCoreIfMessage_SendDTMFRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_SendDTMFRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_SendDTMFRequest& first, PmCoreIfMessage_SendDTMFRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._tones, second._tones);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_SendDTMFRequest& PmCoreIfMessage_SendDTMFRequest::operator=(PmCoreIfMessage_SendDTMFRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_SendDTMFRequest::operator==(const PmCoreIfMessage_SendDTMFRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_tones == rhs._tones));
   }

   void PmCoreIfMessage_SendDTMFRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", Tone = \"%5s\"", (const void*) this,
            _bdAddress.c_str(), _tones.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_TransferAudioRequest::PmCoreIfMessage_TransferAudioRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_TRANSFER_AUDIO_REQUEST, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _audioDirection(AUDIO_DIRECTION_HANDSFREE)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_TransferAudioRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_TransferAudioRequest::PmCoreIfMessage_TransferAudioRequest(IN const BdAddress& bdAddress,
         IN const AudioDirection audioDirection, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_TRANSFER_AUDIO_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _audioDirection(audioDirection)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_TransferAudioRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_TransferAudioRequest::PmCoreIfMessage_TransferAudioRequest(
         IN const PmCoreIfMessage_TransferAudioRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _audioDirection(other._audioDirection)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_TransferAudioRequest: [0x%p] copy-creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_TransferAudioRequest::~PmCoreIfMessage_TransferAudioRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_TransferAudioRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_TransferAudioRequest& first, PmCoreIfMessage_TransferAudioRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._audioDirection, second._audioDirection);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_TransferAudioRequest& PmCoreIfMessage_TransferAudioRequest::operator=(
         PmCoreIfMessage_TransferAudioRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_TransferAudioRequest::operator==(const PmCoreIfMessage_TransferAudioRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_audioDirection == rhs._audioDirection));
   }

   void PmCoreIfMessage_TransferAudioRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", AudioDirection = %u", (const void*) this,
            _bdAddress.c_str(), ETG_CENUM(AudioDirectionEnumType, _audioDirection)));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetVoiceRecognitionStatusRequest::PmCoreIfMessage_GetVoiceRecognitionStatusRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_VOICE_RECOGNITION_STATUS_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetVoiceRecognitionStatusRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetVoiceRecognitionStatusRequest::PmCoreIfMessage_GetVoiceRecognitionStatusRequest(
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_VOICE_RECOGNITION_STATUS_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetVoiceRecognitionStatusRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetVoiceRecognitionStatusRequest::PmCoreIfMessage_GetVoiceRecognitionStatusRequest(
         IN const PmCoreIfMessage_GetVoiceRecognitionStatusRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetVoiceRecognitionStatusRequest: [0x%p] copy-creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_GetVoiceRecognitionStatusRequest::~PmCoreIfMessage_GetVoiceRecognitionStatusRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetVoiceRecognitionStatusRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetVoiceRecognitionStatusRequest& first,
         PmCoreIfMessage_GetVoiceRecognitionStatusRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetVoiceRecognitionStatusRequest& PmCoreIfMessage_GetVoiceRecognitionStatusRequest::operator=(
         PmCoreIfMessage_GetVoiceRecognitionStatusRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetVoiceRecognitionStatusRequest::operator==(
         const PmCoreIfMessage_GetVoiceRecognitionStatusRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetVoiceRecognitionStatusRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest::PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_EXTERNAL_VOICE_RECOGNITION_STATUS_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT), _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest::PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest(
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_EXTERNAL_VOICE_RECOGNITION_STATUS_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest::PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest(
         IN const PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest: [0x%p] copy-creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest::~PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& first,
         PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest::operator=(
         PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest::operator==(
         const PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_StartStopVoiceRecognitionRequest::PmCoreIfMessage_StartStopVoiceRecognitionRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_START_STOP_VOICE_RECOGNITION_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _startStopVR(false)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopVoiceRecognitionRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_StartStopVoiceRecognitionRequest::PmCoreIfMessage_StartStopVoiceRecognitionRequest(
         IN const BdAddress& bdAddress, IN const StartStop startStopVR, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_START_STOP_VOICE_RECOGNITION_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _startStopVR(startStopVR)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopVoiceRecognitionRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StartStopVoiceRecognitionRequest::PmCoreIfMessage_StartStopVoiceRecognitionRequest(
         IN const PmCoreIfMessage_StartStopVoiceRecognitionRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _startStopVR(other._startStopVR)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopVoiceRecognitionRequest: [0x%p] copy-creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_StartStopVoiceRecognitionRequest::~PmCoreIfMessage_StartStopVoiceRecognitionRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopVoiceRecognitionRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_StartStopVoiceRecognitionRequest& first,
         PmCoreIfMessage_StartStopVoiceRecognitionRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._startStopVR, second._startStopVR);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_StartStopVoiceRecognitionRequest& PmCoreIfMessage_StartStopVoiceRecognitionRequest::operator=(
         PmCoreIfMessage_StartStopVoiceRecognitionRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_StartStopVoiceRecognitionRequest::operator==(
         const PmCoreIfMessage_StartStopVoiceRecognitionRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_startStopVR == rhs._startStopVR));
   }

   void PmCoreIfMessage_StartStopVoiceRecognitionRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", StartStopVR = \"%10s\"",
            (const void*) this, _bdAddress.c_str(), _startStopVR ? "TRUE" : "FALSE"));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest::PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_START_STOP_EXTERNAL_VOICE_RECOGNITION_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress(""),
         _startStopExtVR(false)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest::PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(
         IN const BdAddress& bdAddress, IN const StartStop extStartStopVR, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_START_STOP_EXTERNAL_VOICE_RECOGNITION_REQUEST, act, origin),
         _bdAddress(bdAddress),
         _startStopExtVR(extStartStopVR)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest::PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(
         IN const PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress),
         _startStopExtVR(other._startStopExtVR)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest: [0x%p] copy-creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest::~PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& first,
         PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
      swap(first._startStopExtVR, second._startStopExtVR);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest::operator=(
         PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest::operator==(
         const PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress)
            && (this->_startStopExtVR == rhs._startStopExtVR));
   }

   void PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\", StartStopVR = \"%10s\"",
            (const void*) this, _bdAddress.c_str(), _startStopExtVR ? "TRUE" : "FALSE"));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_stopExtendedVRRequest::PmCoreIfMessage_stopExtendedVRRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_STOP_EXTENDED_VOICE_RECOGNITION_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_stopExtendedVRRequest: [0x%p] creating PM Core IF message object",
                  (void*) this));
   }

   PmCoreIfMessage_stopExtendedVRRequest::PmCoreIfMessage_stopExtendedVRRequest(
         IN const BdAddress& bdAddress, IN const ActType act,
         IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_STOP_EXTENDED_VOICE_RECOGNITION_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_stopExtendedVRRequest: [0x%p] creating PM Core IF message object",
            (void*) this));
   }

   PmCoreIfMessage_stopExtendedVRRequest::PmCoreIfMessage_stopExtendedVRRequest(
         IN const PmCoreIfMessage_stopExtendedVRRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_stopExtendedVRRequest: [0x%p] copy-creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_stopExtendedVRRequest::~PmCoreIfMessage_stopExtendedVRRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_stopExtendedVRRequest: [0x%p] destroying PM Core IF message object",
            (void*) this));
   }

   void swap(PmCoreIfMessage_stopExtendedVRRequest& first,
         PmCoreIfMessage_stopExtendedVRRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_stopExtendedVRRequest& PmCoreIfMessage_stopExtendedVRRequest::operator=(
         PmCoreIfMessage_stopExtendedVRRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_stopExtendedVRRequest::operator==(
         const PmCoreIfMessage_stopExtendedVRRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_stopExtendedVRRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"",
            (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

   PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest::PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest() :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_ENHANCED_VOICE_RECOGNITION_FEATURE_REQUEST, PM_DEFAULT_ACT,
               PM_CORE_IF_MSG_ORIGIN_CLIENT),
         _bdAddress("")
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest: [0x%p] creating PM Core "
            "IF message object", (void*) this));
   }

   PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest::PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest(
         IN const BdAddress& bdAddress, IN const ActType act, IN const PmCoreIfMsgOrigin origin) :
         PmCoreIfMessageRequest(PM_CORE_IF_MSG_ID_GET_ENHANCED_VOICE_RECOGNITION_FEATURE_REQUEST, act, origin),
         _bdAddress(bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest: [0x%p] creating PM Core IF message "
            "object", (void*) this));
   }

   PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest::PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest(
         IN const PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& other) : PmCoreIfMessageRequest(other),
         _bdAddress(other._bdAddress)
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest: [0x%p] copy-creating PM Core IF message"
            " object", (void*) this));
   }

   PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest::~PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest()
   {
      ETG_TRACE_USR1(("PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest: [0x%p] destroying PM Core IF message "
            "object", (void*) this));
   }

   void swap(PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& first,
         PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& second)
   {
      using std::swap;

      swap(static_cast<PmCoreIfMessageRequest&>(first), static_cast<PmCoreIfMessageRequest&>(second));
      swap(first._bdAddress, second._bdAddress);
   }

   //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
   PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest::
   operator=( PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest other)
   {
      swap(*this, other);

      return *this;
   }

   bool PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest::operator==(
         const PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& rhs) const
   {
      return ((PmCoreIfMessageRequest::operator ==(rhs))
            && (this->_bdAddress == rhs._bdAddress));
   }

   void PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest::traceMessage(void) const
   {
      PmCoreIfMessageRequest::traceMessage();

      ETG_TRACE_USR4(("traceMessage: [0x%p] BdAddress = \"%50s\"", (const void*) this, _bdAddress.c_str()));
   }

   // *****************************************************************************************************************

}// namespace pmcore
