/**
 * @file PmCoreIfMessageRequest.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the PmCoreIfMessageRequest
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @detailss
 *
 * @ingroup PmCore
 */

#ifndef PmCoreIfMessageRequest_h
#define PmCoreIfMessageRequest_h

#include "PmCoreIfMessage.h"
#include "PmCoreIfTypes.h"

/* Note:
 * Function headers are not added to avoid large file length */

namespace pmcore
{
   class PmCoreIfMessageRequest : public PmCoreIfMessage
   {
   public:

      friend void swap(PmCoreIfMessageRequest& first, PmCoreIfMessageRequest& second);

      explicit PmCoreIfMessageRequest(IN const PmCoreIfMsgId messageId,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessageRequest(IN const PmCoreIfMessageRequest& other);

      virtual ~PmCoreIfMessageRequest();

      PmCoreIfMessageRequest& operator=(PmCoreIfMessageRequest other);

      bool operator==(const PmCoreIfMessageRequest& rhs) const;

      /**
       * This method is used to set the PmResult
       *
       * @param[in] pmResult - PmCore result
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      inline void setPmResult(IN const PmResult& pmResult)
      {
         _pmResult = pmResult;
      }

      /**
       * This method is used to get the PmResult
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult
       */
      inline const PmResult getPmResult(void) const
      {
         return _pmResult;
      }

      virtual void traceMessage(void) const override;

   private:
      PmResult     _pmResult;  /**< PmResult used only for Request messages */
   };

   class PmCoreIfMessage_SetSystemState final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SetSystemState& first, PmCoreIfMessage_SetSystemState& second);

      explicit PmCoreIfMessage_SetSystemState();

      explicit PmCoreIfMessage_SetSystemState(IN const SystemState systemState,
            IN const ActType act = PM_DEFAULT_ACT, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SetSystemState(IN const PmCoreIfMessage_SetSystemState& other);

      virtual ~PmCoreIfMessage_SetSystemState();

      PmCoreIfMessage_SetSystemState& operator=(PmCoreIfMessage_SetSystemState other);

      bool operator==(const PmCoreIfMessage_SetSystemState& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setSystemState(IN const SystemState systemState)
      {
         _systemState = systemState;
      }

      inline SystemState getSystemState(void) const
      {
         return _systemState;
      }

   private:
      SystemState     _systemState;    /**< System state */
   };

   class PmCoreIfMessage_SetVehicleConfiguration final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SetVehicleConfiguration& first, PmCoreIfMessage_SetVehicleConfiguration& second);

      explicit PmCoreIfMessage_SetVehicleConfiguration();

      explicit PmCoreIfMessage_SetVehicleConfiguration(IN const VehicleConfiguration& vehicleConfig,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SetVehicleConfiguration(IN const PmCoreIfMessage_SetVehicleConfiguration& other);

      virtual ~PmCoreIfMessage_SetVehicleConfiguration();

      PmCoreIfMessage_SetVehicleConfiguration& operator=(PmCoreIfMessage_SetVehicleConfiguration other);

      bool operator==(const PmCoreIfMessage_SetVehicleConfiguration& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setVehicleConfig(IN const VehicleConfiguration& vehicleConfig)
      {
         _vehicleConfig = vehicleConfig;
      }

      inline const VehicleConfiguration& getVehicleConfig(void) const
      {
         return _vehicleConfig;
      }

   private:
      VehicleConfiguration _vehicleConfig;    /**< configuration of the vehicle*/
   };

   class PmCoreIfMessage_OnDevicePaired final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_OnDevicePaired& first, PmCoreIfMessage_OnDevicePaired& second);

      explicit PmCoreIfMessage_OnDevicePaired();

      explicit PmCoreIfMessage_OnDevicePaired(IN const BasicDeviceDetails& pairedDeviceInfo,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_OnDevicePaired(IN const PmCoreIfMessage_OnDevicePaired& other);

      virtual ~PmCoreIfMessage_OnDevicePaired();

      PmCoreIfMessage_OnDevicePaired& operator=(PmCoreIfMessage_OnDevicePaired other);

      bool operator==(const PmCoreIfMessage_OnDevicePaired& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setPairedDeviceInfo(IN const BasicDeviceDetails& pairedDeviceInfo)
      {
         _pairedDeviceInfo = pairedDeviceInfo;
      }

      inline const BasicDeviceDetails& getPairedDeviceInfo(void) const
      {
         return _pairedDeviceInfo;
      }

   private:
      BasicDeviceDetails     _pairedDeviceInfo;    /**< Paired device details */
   };

   class PmCoreIfMessage_OnDeviceConnected final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_OnDeviceConnected& first, PmCoreIfMessage_OnDeviceConnected& second);

      explicit PmCoreIfMessage_OnDeviceConnected();

      explicit PmCoreIfMessage_OnDeviceConnected(IN const BasicDeviceDetails& connectedDeviceInfo,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_OnDeviceConnected(IN const PmCoreIfMessage_OnDeviceConnected& other);

      virtual ~PmCoreIfMessage_OnDeviceConnected();

      PmCoreIfMessage_OnDeviceConnected& operator=(PmCoreIfMessage_OnDeviceConnected other);

      bool operator==(const PmCoreIfMessage_OnDeviceConnected& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setConnectedDeviceInfo(IN const BasicDeviceDetails& connectedDeviceInfo)
      {
         _connectedDeviceInfo = connectedDeviceInfo;
      }

      inline const BasicDeviceDetails& getConnectedDeviceInfo(void) const
      {
         return _connectedDeviceInfo;
      }

   private:
      BasicDeviceDetails     _connectedDeviceInfo;    /**< Address of BT device*/
   };

   class PmCoreIfMessage_OnDeviceDisconnected final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_OnDeviceDisconnected& first, PmCoreIfMessage_OnDeviceDisconnected& second);

      explicit PmCoreIfMessage_OnDeviceDisconnected();

      explicit PmCoreIfMessage_OnDeviceDisconnected(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_OnDeviceDisconnected(IN const PmCoreIfMessage_OnDeviceDisconnected& other);

      virtual ~PmCoreIfMessage_OnDeviceDisconnected();

      PmCoreIfMessage_OnDeviceDisconnected& operator=(PmCoreIfMessage_OnDeviceDisconnected other);

      bool operator==(const PmCoreIfMessage_OnDeviceDisconnected& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress     _bdAddress;     /**< Address of BT device*/
   };

   class PmCoreIfMessage_OnDeviceDeleted final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_OnDeviceDeleted& first, PmCoreIfMessage_OnDeviceDeleted& second);

      explicit PmCoreIfMessage_OnDeviceDeleted();

      explicit PmCoreIfMessage_OnDeviceDeleted(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_OnDeviceDeleted(IN const PmCoreIfMessage_OnDeviceDeleted& other);

      virtual ~PmCoreIfMessage_OnDeviceDeleted();

      PmCoreIfMessage_OnDeviceDeleted& operator=(PmCoreIfMessage_OnDeviceDeleted other);

      bool operator==(const PmCoreIfMessage_OnDeviceDeleted& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress      _bdAddress;     /**< Address of BT device*/
   };

   class PmCoreIfMessage_GetSystemWideRingtoneListRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetSystemWideRingtoneListRequest& first,
            PmCoreIfMessage_GetSystemWideRingtoneListRequest& second);

      explicit PmCoreIfMessage_GetSystemWideRingtoneListRequest();

      explicit PmCoreIfMessage_GetSystemWideRingtoneListRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetSystemWideRingtoneListRequest(
            IN const PmCoreIfMessage_GetSystemWideRingtoneListRequest& other);

      virtual ~PmCoreIfMessage_GetSystemWideRingtoneListRequest();

      PmCoreIfMessage_GetSystemWideRingtoneListRequest& operator=(
            PmCoreIfMessage_GetSystemWideRingtoneListRequest other);

      bool operator==(const PmCoreIfMessage_GetSystemWideRingtoneListRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_GetDeviceRingtoneListRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetDeviceRingtoneListRequest& first,
            PmCoreIfMessage_GetDeviceRingtoneListRequest& second);

      explicit PmCoreIfMessage_GetDeviceRingtoneListRequest();

      explicit PmCoreIfMessage_GetDeviceRingtoneListRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetDeviceRingtoneListRequest(IN const PmCoreIfMessage_GetDeviceRingtoneListRequest& other);

      virtual ~PmCoreIfMessage_GetDeviceRingtoneListRequest();

      PmCoreIfMessage_GetDeviceRingtoneListRequest& operator=(PmCoreIfMessage_GetDeviceRingtoneListRequest other);

      bool operator==(const PmCoreIfMessage_GetDeviceRingtoneListRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_SetRingtoneRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SetRingtoneRequest& first, PmCoreIfMessage_SetRingtoneRequest& second);

      explicit PmCoreIfMessage_SetRingtoneRequest();

      explicit PmCoreIfMessage_SetRingtoneRequest(IN const BdAddress& bdAddress,
            IN const RingtoneId ringtoneId,IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SetRingtoneRequest(IN const PmCoreIfMessage_SetRingtoneRequest& other);

      virtual ~PmCoreIfMessage_SetRingtoneRequest();

      PmCoreIfMessage_SetRingtoneRequest& operator=(PmCoreIfMessage_SetRingtoneRequest other);

      bool operator==(const PmCoreIfMessage_SetRingtoneRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setRingtoneId(IN const RingtoneId ringtoneId)
      {
         _ringtoneId = ringtoneId;
      }

      inline RingtoneId getRingtoneId(void) const
      {
         return _ringtoneId;
      }

   private:
      BdAddress          _bdAddress;       /**< Address of BT device*/
      RingtoneId         _ringtoneId;      /**< ID of the ring tone*/
   };

   class PmCoreIfMessage_PlayRingtoneRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_PlayRingtoneRequest& first, PmCoreIfMessage_PlayRingtoneRequest& second);

      explicit PmCoreIfMessage_PlayRingtoneRequest();

      explicit PmCoreIfMessage_PlayRingtoneRequest(IN const BdAddress& bdAddress, IN const RingtoneId ringtoneId,
            IN const PlayCount, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_PlayRingtoneRequest(IN const PmCoreIfMessage_PlayRingtoneRequest& other);

      virtual ~PmCoreIfMessage_PlayRingtoneRequest();

      PmCoreIfMessage_PlayRingtoneRequest& operator=(PmCoreIfMessage_PlayRingtoneRequest other);

      bool operator==(const PmCoreIfMessage_PlayRingtoneRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setRingtoneId(IN const RingtoneId ringtoneId)
      {
         _ringtoneId = ringtoneId;
      }

      inline RingtoneId getRingtoneId(void) const
      {
         return _ringtoneId;
      }

      inline void setPlayCount(IN const PlayCount playCount)
      {
         _playCount = playCount;
      }

      inline PlayCount getPlayCount(void) const
      {
         return _playCount;
      }

   private:
      BdAddress           _bdAddress;      /**< Address of BT device*/
      RingtoneId          _ringtoneId;     /**< ID of the ring tone*/
      PlayCount           _playCount;      /**< Play count for the ringtone*/
   };

   class PmCoreIfMessage_StopRingtoneRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_StopRingtoneRequest& first, PmCoreIfMessage_StopRingtoneRequest& second);

      explicit PmCoreIfMessage_StopRingtoneRequest();

      explicit PmCoreIfMessage_StopRingtoneRequest(IN const BdAddress& bdAddress, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_StopRingtoneRequest(IN const PmCoreIfMessage_StopRingtoneRequest& other);

      virtual ~PmCoreIfMessage_StopRingtoneRequest();

      PmCoreIfMessage_StopRingtoneRequest& operator=(PmCoreIfMessage_StopRingtoneRequest other);

      bool operator==(const PmCoreIfMessage_StopRingtoneRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }
   private:
      BdAddress           _bdAddress;      /**< Address of BT device*/
   };

   class PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& first,
            PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& second);

      explicit PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest();

      explicit PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest(
            IN const PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& other);

      virtual ~PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest();

      PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& operator=(
            PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest other);

      bool operator==(const PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_SuppressRingtoneOnOffRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SuppressRingtoneOnOffRequest& first,
            PmCoreIfMessage_SuppressRingtoneOnOffRequest& second);

      explicit PmCoreIfMessage_SuppressRingtoneOnOffRequest();

      explicit PmCoreIfMessage_SuppressRingtoneOnOffRequest(IN const BdAddress& bdAddress,
            IN const SuppressRingtoneState suppressRingtoneState, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SuppressRingtoneOnOffRequest(IN const PmCoreIfMessage_SuppressRingtoneOnOffRequest& other);

      virtual ~PmCoreIfMessage_SuppressRingtoneOnOffRequest();

      PmCoreIfMessage_SuppressRingtoneOnOffRequest& operator=(PmCoreIfMessage_SuppressRingtoneOnOffRequest other);

      bool operator==(const PmCoreIfMessage_SuppressRingtoneOnOffRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setSuppressRingtoneState(IN const SuppressRingtoneState suppressRingtoneState)
      {
         _suppressRingtoneState = suppressRingtoneState;
      }

      inline SuppressRingtoneState getSuppressRingtoneState(void) const
      {
         return _suppressRingtoneState;
      }

   private:
      BdAddress                _bdAddress;                 /**< Address of BT device*/
      SuppressRingtoneState    _suppressRingtoneState;     /**< flag for suppress ring tone state*/
   };

   class PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& first,
            PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& second);

      explicit PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest();

      explicit PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest(
            IN const PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& other);

      virtual ~PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest();

      PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& operator=(
            PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest other);

      bool operator==(const PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_AutoWaitingModeOnOffRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_AutoWaitingModeOnOffRequest& first,
            PmCoreIfMessage_AutoWaitingModeOnOffRequest& second);

      explicit PmCoreIfMessage_AutoWaitingModeOnOffRequest();

      explicit PmCoreIfMessage_AutoWaitingModeOnOffRequest(IN const BdAddress& bdAddress,
            IN const AutoWaitingModeState autoWaitingModeState,IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_AutoWaitingModeOnOffRequest(IN const PmCoreIfMessage_AutoWaitingModeOnOffRequest& other);

      virtual ~PmCoreIfMessage_AutoWaitingModeOnOffRequest();

      PmCoreIfMessage_AutoWaitingModeOnOffRequest& operator=(PmCoreIfMessage_AutoWaitingModeOnOffRequest other);

      bool operator==(const PmCoreIfMessage_AutoWaitingModeOnOffRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setAutoWaitingModeState(IN const AutoWaitingModeState autoWaitingModeState)
      {
         _autoWaitingModeState = autoWaitingModeState;
      }

      inline AutoWaitingModeState getAutoWaitingModeState(void) const
      {
         return _autoWaitingModeState;
      }
   private:
      BdAddress                _bdAddress;                /**< Address of BT device*/
      AutoWaitingModeState     _autoWaitingModeState;     /**< flag for auto waiting mode state*/
   };

   class PmCoreIfMessage_SetWaitingModeFilePathRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void
      swap(PmCoreIfMessage_SetWaitingModeFilePathRequest& first, PmCoreIfMessage_SetWaitingModeFilePathRequest& second);

      explicit PmCoreIfMessage_SetWaitingModeFilePathRequest();

      explicit PmCoreIfMessage_SetWaitingModeFilePathRequest(IN const WaitingModeFilePath& waitingModeFilePath,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SetWaitingModeFilePathRequest(IN const PmCoreIfMessage_SetWaitingModeFilePathRequest& other);

      virtual ~PmCoreIfMessage_SetWaitingModeFilePathRequest();

      PmCoreIfMessage_SetWaitingModeFilePathRequest& operator=(
            PmCoreIfMessage_SetWaitingModeFilePathRequest other);

      bool operator==(const PmCoreIfMessage_SetWaitingModeFilePathRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setWaitingModeFilePath(IN const WaitingModeFilePath& waitingModeFilePath)
      {
         _waitingModeFilePath = waitingModeFilePath;
      }

      inline const WaitingModeFilePath& getWaitingModeFilePath(void) const
      {
         return _waitingModeFilePath;
      }

   private:
      WaitingModeFilePath      _waitingModeFilePath; /**< Instance of WaitingModeFilePath */
   };

   class PmCoreIfMessage_GetWaitingModeFilePathRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetWaitingModeFilePathRequest& first,
            PmCoreIfMessage_GetWaitingModeFilePathRequest& second);

      explicit PmCoreIfMessage_GetWaitingModeFilePathRequest();

      explicit PmCoreIfMessage_GetWaitingModeFilePathRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetWaitingModeFilePathRequest(IN const PmCoreIfMessage_GetWaitingModeFilePathRequest& other);

      virtual ~PmCoreIfMessage_GetWaitingModeFilePathRequest();

      PmCoreIfMessage_GetWaitingModeFilePathRequest& operator=(PmCoreIfMessage_GetWaitingModeFilePathRequest other);

      bool operator==(const PmCoreIfMessage_GetWaitingModeFilePathRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_GetPhoneCallAudioActiveRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetPhoneCallAudioActiveRequest& first,
            PmCoreIfMessage_GetPhoneCallAudioActiveRequest& second);

      explicit PmCoreIfMessage_GetPhoneCallAudioActiveRequest();

      explicit PmCoreIfMessage_GetPhoneCallAudioActiveRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetPhoneCallAudioActiveRequest(IN const PmCoreIfMessage_GetPhoneCallAudioActiveRequest& other);

      virtual ~PmCoreIfMessage_GetPhoneCallAudioActiveRequest();

      PmCoreIfMessage_GetPhoneCallAudioActiveRequest& operator=(PmCoreIfMessage_GetPhoneCallAudioActiveRequest other);

      bool operator==(const PmCoreIfMessage_GetPhoneCallAudioActiveRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_GetActivePassiveDeviceListRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetActivePassiveDeviceListRequest& first,
            PmCoreIfMessage_GetActivePassiveDeviceListRequest& second);

      explicit PmCoreIfMessage_GetActivePassiveDeviceListRequest();

      explicit PmCoreIfMessage_GetActivePassiveDeviceListRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetActivePassiveDeviceListRequest(
            IN const PmCoreIfMessage_GetActivePassiveDeviceListRequest& other);

      virtual ~PmCoreIfMessage_GetActivePassiveDeviceListRequest();

      PmCoreIfMessage_GetActivePassiveDeviceListRequest& operator=(
            PmCoreIfMessage_GetActivePassiveDeviceListRequest other);

      bool operator==(const PmCoreIfMessage_GetActivePassiveDeviceListRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_SetActivePhoneRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SetActivePhoneRequest& first, PmCoreIfMessage_SetActivePhoneRequest& second);

      explicit PmCoreIfMessage_SetActivePhoneRequest();

      explicit PmCoreIfMessage_SetActivePhoneRequest(IN const BdAddressList& deviceAddressList,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SetActivePhoneRequest(IN const PmCoreIfMessage_SetActivePhoneRequest& other);

      virtual ~PmCoreIfMessage_SetActivePhoneRequest();

      PmCoreIfMessage_SetActivePhoneRequest& operator=(PmCoreIfMessage_SetActivePhoneRequest other);

      bool operator==(const PmCoreIfMessage_SetActivePhoneRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddressList(IN const BdAddressList& deviceAddressList)
      {
         _deviceAddressList = deviceAddressList;
      }

      inline const BdAddressList& getBdAddressList(void) const
      {
         return _deviceAddressList;
      }

   private:
      BdAddressList     _deviceAddressList;      /**< List of BT device address*/
   };

   class PmCoreIfMessage_RestoreDefaultSettingRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_RestoreDefaultSettingRequest& first,
            PmCoreIfMessage_RestoreDefaultSettingRequest& second);

      explicit PmCoreIfMessage_RestoreDefaultSettingRequest();

      explicit PmCoreIfMessage_RestoreDefaultSettingRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_RestoreDefaultSettingRequest(IN const PmCoreIfMessage_RestoreDefaultSettingRequest& other);

      virtual ~PmCoreIfMessage_RestoreDefaultSettingRequest();

      PmCoreIfMessage_RestoreDefaultSettingRequest& operator=(PmCoreIfMessage_RestoreDefaultSettingRequest other);

      bool operator==(const PmCoreIfMessage_RestoreDefaultSettingRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress         _bdAddress;      /**< Address of the BT device*/
   };

   class PmCoreIfMessage_GetSupportedFeaturesRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetSupportedFeaturesRequest& first,
            PmCoreIfMessage_GetSupportedFeaturesRequest& second);

      explicit PmCoreIfMessage_GetSupportedFeaturesRequest();

      explicit PmCoreIfMessage_GetSupportedFeaturesRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetSupportedFeaturesRequest(IN const PmCoreIfMessage_GetSupportedFeaturesRequest& other);

      virtual ~PmCoreIfMessage_GetSupportedFeaturesRequest();

      PmCoreIfMessage_GetSupportedFeaturesRequest& operator=(PmCoreIfMessage_GetSupportedFeaturesRequest other);

      bool operator==(const PmCoreIfMessage_GetSupportedFeaturesRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress         _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_GetNetworkStatusRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetNetworkStatusRequest& first,
            PmCoreIfMessage_GetNetworkStatusRequest& second);

      explicit PmCoreIfMessage_GetNetworkStatusRequest();

      explicit PmCoreIfMessage_GetNetworkStatusRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetNetworkStatusRequest(IN const PmCoreIfMessage_GetNetworkStatusRequest& other);

      virtual ~PmCoreIfMessage_GetNetworkStatusRequest();

      PmCoreIfMessage_GetNetworkStatusRequest& operator=(PmCoreIfMessage_GetNetworkStatusRequest other);

      bool operator==(const PmCoreIfMessage_GetNetworkStatusRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress          _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_GetSignalStrengthRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetSignalStrengthRequest& first,
            PmCoreIfMessage_GetSignalStrengthRequest& second);

      explicit PmCoreIfMessage_GetSignalStrengthRequest();

      explicit PmCoreIfMessage_GetSignalStrengthRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetSignalStrengthRequest(IN const PmCoreIfMessage_GetSignalStrengthRequest& other);

      virtual ~PmCoreIfMessage_GetSignalStrengthRequest();

      PmCoreIfMessage_GetSignalStrengthRequest& operator=(PmCoreIfMessage_GetSignalStrengthRequest other);

      bool operator==(const PmCoreIfMessage_GetSignalStrengthRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress            _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_GetBatteryChargeLevelRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetBatteryChargeLevelRequest& first,
            PmCoreIfMessage_GetBatteryChargeLevelRequest& second);

      explicit PmCoreIfMessage_GetBatteryChargeLevelRequest();

      explicit PmCoreIfMessage_GetBatteryChargeLevelRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetBatteryChargeLevelRequest(IN const PmCoreIfMessage_GetBatteryChargeLevelRequest& other);

      virtual ~PmCoreIfMessage_GetBatteryChargeLevelRequest();

      PmCoreIfMessage_GetBatteryChargeLevelRequest& operator=(PmCoreIfMessage_GetBatteryChargeLevelRequest other);

      bool operator==(const PmCoreIfMessage_GetBatteryChargeLevelRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress            _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_GetNetworkOperatorRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetNetworkOperatorRequest& first,
            PmCoreIfMessage_GetNetworkOperatorRequest& second);

      explicit PmCoreIfMessage_GetNetworkOperatorRequest();

      explicit PmCoreIfMessage_GetNetworkOperatorRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetNetworkOperatorRequest(IN const PmCoreIfMessage_GetNetworkOperatorRequest& other);

      virtual ~PmCoreIfMessage_GetNetworkOperatorRequest();

      PmCoreIfMessage_GetNetworkOperatorRequest& operator=(PmCoreIfMessage_GetNetworkOperatorRequest other);

      bool operator==(const PmCoreIfMessage_GetNetworkOperatorRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress          _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_VoiceMailListRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_VoiceMailListRequest& first, PmCoreIfMessage_VoiceMailListRequest& second);

      explicit PmCoreIfMessage_VoiceMailListRequest();

      explicit PmCoreIfMessage_VoiceMailListRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_VoiceMailListRequest(IN const PmCoreIfMessage_VoiceMailListRequest& other);

      virtual ~PmCoreIfMessage_VoiceMailListRequest();

      PmCoreIfMessage_VoiceMailListRequest& operator=(PmCoreIfMessage_VoiceMailListRequest other);

      bool operator==(const PmCoreIfMessage_VoiceMailListRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress         _bdAddress;      /**< Address of the BT device*/
   };

   class PmCoreIfMessage_GetCallStatusListRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetCallStatusListRequest& first,
            PmCoreIfMessage_GetCallStatusListRequest& second);

      explicit PmCoreIfMessage_GetCallStatusListRequest();

      explicit PmCoreIfMessage_GetCallStatusListRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetCallStatusListRequest(IN const PmCoreIfMessage_GetCallStatusListRequest& other);

      virtual ~PmCoreIfMessage_GetCallStatusListRequest();

      PmCoreIfMessage_GetCallStatusListRequest& operator=(PmCoreIfMessage_GetCallStatusListRequest other);

      bool operator==(const PmCoreIfMessage_GetCallStatusListRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_GetMicrophoneMuteStateRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetMicrophoneMuteStateRequest& first,
            PmCoreIfMessage_GetMicrophoneMuteStateRequest& second);

      explicit PmCoreIfMessage_GetMicrophoneMuteStateRequest();

      explicit PmCoreIfMessage_GetMicrophoneMuteStateRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetMicrophoneMuteStateRequest(IN const PmCoreIfMessage_GetMicrophoneMuteStateRequest& other);

      virtual ~PmCoreIfMessage_GetMicrophoneMuteStateRequest();

      PmCoreIfMessage_GetMicrophoneMuteStateRequest& operator=(PmCoreIfMessage_GetMicrophoneMuteStateRequest other);

      bool operator==(const PmCoreIfMessage_GetMicrophoneMuteStateRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_SetMicrophoneMuteStateRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SetMicrophoneMuteStateRequest& first,
            PmCoreIfMessage_SetMicrophoneMuteStateRequest& second);

      explicit PmCoreIfMessage_SetMicrophoneMuteStateRequest();

      explicit PmCoreIfMessage_SetMicrophoneMuteStateRequest(IN const MuteState microphoneMuteState,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SetMicrophoneMuteStateRequest(IN const PmCoreIfMessage_SetMicrophoneMuteStateRequest& other);

      virtual ~PmCoreIfMessage_SetMicrophoneMuteStateRequest();

      PmCoreIfMessage_SetMicrophoneMuteStateRequest& operator=(PmCoreIfMessage_SetMicrophoneMuteStateRequest other);

      bool operator==(const PmCoreIfMessage_SetMicrophoneMuteStateRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setMuteState(IN const MuteState muteState)
      {
         _microphoneMuteState = muteState;
      }

      inline MuteState getMuteState(void) const
      {
         return _microphoneMuteState;
      }

   private:
      MuteState         _microphoneMuteState;      /**< Mute state of the Microphone*/
   };

   class PmCoreIfMessage_MergeCallsRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_MergeCallsRequest& first, PmCoreIfMessage_MergeCallsRequest& second);

      explicit PmCoreIfMessage_MergeCallsRequest();

      explicit PmCoreIfMessage_MergeCallsRequest(IN const BdAddress& bdAddress,
            IN const CallInstanceList& mergeCallList, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_MergeCallsRequest(IN const PmCoreIfMessage_MergeCallsRequest& other);

      virtual ~PmCoreIfMessage_MergeCallsRequest();

      PmCoreIfMessage_MergeCallsRequest& operator=(PmCoreIfMessage_MergeCallsRequest other);

      bool operator==(const PmCoreIfMessage_MergeCallsRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setCallInstanceList(IN const CallInstanceList& mergeCallList)
      {
         _mergeCallList = mergeCallList;
      }

      inline const CallInstanceList& getCallInstanceList(void) const
      {
         return _mergeCallList;
      }

   private:
      BdAddress          _bdAddress;                /**< device address */
      CallInstanceList   _mergeCallList;            /**< list of call instances */
   };

   class PmCoreIfMessage_SplitCallsRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SplitCallsRequest& first, PmCoreIfMessage_SplitCallsRequest& second);

      explicit PmCoreIfMessage_SplitCallsRequest();

      explicit PmCoreIfMessage_SplitCallsRequest(IN const BdAddress& bdAddress,
            IN const CallInstanceList& splitCallList, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SplitCallsRequest(IN const PmCoreIfMessage_SplitCallsRequest& other);

      virtual ~PmCoreIfMessage_SplitCallsRequest();

      PmCoreIfMessage_SplitCallsRequest& operator=(PmCoreIfMessage_SplitCallsRequest other);

      bool operator==(const PmCoreIfMessage_SplitCallsRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setCallInstanceList(IN const CallInstanceList& splitCallList)
      {
         _splitCallList = splitCallList;
      }

      inline const CallInstanceList& getCallInstanceList(void) const
      {
         return _splitCallList;
      }

   private:
      BdAddress          _bdAddress;                /**< device address */
      CallInstanceList   _splitCallList;            /**< list of call instances */
   };

   class PmCoreIfMessage_HangupCallsRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_HangupCallsRequest& first, PmCoreIfMessage_HangupCallsRequest& second);

      explicit PmCoreIfMessage_HangupCallsRequest();

      explicit PmCoreIfMessage_HangupCallsRequest(IN const BdAddress& bdAddress,
            IN const CallInstanceList& hangupCallList, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_HangupCallsRequest(IN const PmCoreIfMessage_HangupCallsRequest& other);

      virtual ~PmCoreIfMessage_HangupCallsRequest();

      PmCoreIfMessage_HangupCallsRequest& operator=(PmCoreIfMessage_HangupCallsRequest other);

      bool operator==(const PmCoreIfMessage_HangupCallsRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setCallInstanceList(IN const CallInstanceList& hangupCallList)
      {
         _hangupCallList = hangupCallList;
      }

      inline const CallInstanceList& getCallInstanceList(void) const
      {
         return _hangupCallList;
      }

   private:
      BdAddress          _bdAddress;                /**< device address */
      CallInstanceList   _hangupCallList;           /**< list of call instances */
   };

   class PmCoreIfMessage_RedialRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_RedialRequest& first, PmCoreIfMessage_RedialRequest& second);

      explicit PmCoreIfMessage_RedialRequest();

      explicit PmCoreIfMessage_RedialRequest(IN const BdAddress& bdAddress, IN const HideCallerId hideCallerId,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_RedialRequest(IN const PmCoreIfMessage_RedialRequest& other);

      virtual ~PmCoreIfMessage_RedialRequest();

      PmCoreIfMessage_RedialRequest& operator=(PmCoreIfMessage_RedialRequest other);

      bool operator==(const PmCoreIfMessage_RedialRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setHideCallerId(IN const HideCallerId hideCallerId)
      {
         _hideCallerId = hideCallerId;
      }

      inline HideCallerId getHideCallerId(void) const
      {
         return _hideCallerId;
      }

   private:
      BdAddress         _bdAddress;        /**< Address of the BT device*/
      HideCallerId      _hideCallerId;     /**< Instance of HideCallerId*/
   };

   class PmCoreIfMessage_GetSCOConnectionRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetSCOConnectionRequest& first,
            PmCoreIfMessage_GetSCOConnectionRequest& second);

      explicit PmCoreIfMessage_GetSCOConnectionRequest();

      explicit PmCoreIfMessage_GetSCOConnectionRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetSCOConnectionRequest(IN const PmCoreIfMessage_GetSCOConnectionRequest& other);

      virtual ~PmCoreIfMessage_GetSCOConnectionRequest();

      PmCoreIfMessage_GetSCOConnectionRequest& operator=(PmCoreIfMessage_GetSCOConnectionRequest other);

      bool operator==(const PmCoreIfMessage_GetSCOConnectionRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress         _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_GetRingtoneMuteStateRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetRingtoneMuteStateRequest& first,
            PmCoreIfMessage_GetRingtoneMuteStateRequest& second);

      explicit PmCoreIfMessage_GetRingtoneMuteStateRequest();

      explicit PmCoreIfMessage_GetRingtoneMuteStateRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetRingtoneMuteStateRequest(IN const PmCoreIfMessage_GetRingtoneMuteStateRequest& other);

      virtual ~PmCoreIfMessage_GetRingtoneMuteStateRequest();

      PmCoreIfMessage_GetRingtoneMuteStateRequest& operator=(PmCoreIfMessage_GetRingtoneMuteStateRequest other);

      bool operator==(const PmCoreIfMessage_GetRingtoneMuteStateRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress          _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_SetRingtoneMuteStateRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SetRingtoneMuteStateRequest& first,
            PmCoreIfMessage_SetRingtoneMuteStateRequest& second);

      explicit PmCoreIfMessage_SetRingtoneMuteStateRequest();

      explicit PmCoreIfMessage_SetRingtoneMuteStateRequest(IN const BdAddress& bdAddress,
            IN const MuteState ringtoneMuteState, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SetRingtoneMuteStateRequest(IN const PmCoreIfMessage_SetRingtoneMuteStateRequest& other);

      virtual ~PmCoreIfMessage_SetRingtoneMuteStateRequest();

      PmCoreIfMessage_SetRingtoneMuteStateRequest& operator=(PmCoreIfMessage_SetRingtoneMuteStateRequest other);

      bool operator==(const PmCoreIfMessage_SetRingtoneMuteStateRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setMuteState(IN const MuteState ringtoneMuteState)
      {
         _ringtoneMuteState = ringtoneMuteState;
      }

      inline MuteState getMuteState(void) const
      {
         return _ringtoneMuteState;
      }

   private:
      BdAddress         _bdAddress;              /**< Address of the BT device*/
      MuteState         _ringtoneMuteState;      /**< flag for Mute or Unmute*/
   };

   class PmCoreIfMessage_GetWaitingModeStateRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetWaitingModeStateRequest& first, PmCoreIfMessage_GetWaitingModeStateRequest& second);

      explicit PmCoreIfMessage_GetWaitingModeStateRequest();

      explicit PmCoreIfMessage_GetWaitingModeStateRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetWaitingModeStateRequest(IN const PmCoreIfMessage_GetWaitingModeStateRequest& other);

      virtual ~PmCoreIfMessage_GetWaitingModeStateRequest();

      PmCoreIfMessage_GetWaitingModeStateRequest& operator=(PmCoreIfMessage_GetWaitingModeStateRequest other);

      bool operator==(const PmCoreIfMessage_GetWaitingModeStateRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress          _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_GetRingtonePlaybackStatusRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetRingtonePlaybackStatusRequest& first,
            PmCoreIfMessage_GetRingtonePlaybackStatusRequest& second);

      explicit PmCoreIfMessage_GetRingtonePlaybackStatusRequest();

      explicit PmCoreIfMessage_GetRingtonePlaybackStatusRequest(IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetRingtonePlaybackStatusRequest(IN const PmCoreIfMessage_GetRingtonePlaybackStatusRequest& other);

      virtual ~PmCoreIfMessage_GetRingtonePlaybackStatusRequest();

      PmCoreIfMessage_GetRingtonePlaybackStatusRequest& operator=(PmCoreIfMessage_GetRingtonePlaybackStatusRequest other);

      bool operator==(const PmCoreIfMessage_GetRingtonePlaybackStatusRequest& rhs) const;

      virtual void traceMessage(void) const override;
   };

   class PmCoreIfMessage_StartStopWaitingModeRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_StartStopWaitingModeRequest& first, PmCoreIfMessage_StartStopWaitingModeRequest& second);

      explicit PmCoreIfMessage_StartStopWaitingModeRequest();

      explicit PmCoreIfMessage_StartStopWaitingModeRequest(IN const BdAddress& bdAddress,
            IN const StartStop startStopWaitingMode, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_StartStopWaitingModeRequest(IN const PmCoreIfMessage_StartStopWaitingModeRequest& other);

      virtual ~PmCoreIfMessage_StartStopWaitingModeRequest();

      PmCoreIfMessage_StartStopWaitingModeRequest& operator=(PmCoreIfMessage_StartStopWaitingModeRequest other);

      bool operator==(const PmCoreIfMessage_StartStopWaitingModeRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setStartStopWaitingMode(IN const StartStop startStopWaitingMode)
      {
         _startStopWaitingMode = startStopWaitingMode;
      }

      inline StartStop getStartStopWaitingMode(void) const
      {
         return _startStopWaitingMode;
      }

   private:
      BdAddress         _bdAddress;              /**< Address of the BT device*/
      StartStop         _startStopWaitingMode;   /**< start or stop flag of Waiting Mode*/
   };

   class PmCoreIfMessage_DialRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_DialRequest& first, PmCoreIfMessage_DialRequest& second);

      explicit PmCoreIfMessage_DialRequest();

      explicit PmCoreIfMessage_DialRequest(IN const BdAddress& bdAddress, IN const TelephoneNumber& telephoneNumber,
            IN const HideCallerId hideCallerId, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_DialRequest(IN const PmCoreIfMessage_DialRequest& other);

      virtual ~PmCoreIfMessage_DialRequest();

      PmCoreIfMessage_DialRequest& operator=(PmCoreIfMessage_DialRequest other);

      bool operator==(const PmCoreIfMessage_DialRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setTelephoneNumber(IN const TelephoneNumber& telephoneNumber)
      {
         _telephoneNumber = telephoneNumber;
      }

      inline const TelephoneNumber& getTelephoneNumber(void) const
      {
         return _telephoneNumber;
      }

      inline void setHideCallerId(IN const HideCallerId hideCallerId)
      {
         _hideCallerId = hideCallerId;
      }

      inline HideCallerId getHideCallerId(void) const
      {
         return _hideCallerId;
      }

   private:
      BdAddress         _bdAddress;         /**< Instance of BdAddress */
      TelephoneNumber   _telephoneNumber;   /**< Instance of TelephoneNumber */
      HideCallerId      _hideCallerId;      /**< Instance of HideCallerId */
   };

   class PmCoreIfMessage_AcceptCallRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_AcceptCallRequest& first, PmCoreIfMessage_AcceptCallRequest& second);

      explicit PmCoreIfMessage_AcceptCallRequest();

      explicit PmCoreIfMessage_AcceptCallRequest(IN const BdAddress& bdAddress, IN const CallInstance callInstance,
            IN const AcceptOperation operation, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_AcceptCallRequest(IN const PmCoreIfMessage_AcceptCallRequest& other);

      virtual ~PmCoreIfMessage_AcceptCallRequest();

      PmCoreIfMessage_AcceptCallRequest& operator=(PmCoreIfMessage_AcceptCallRequest other);

      bool operator==(const PmCoreIfMessage_AcceptCallRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setCallInstance(IN const CallInstance callInstance)
      {
         _callInstance = callInstance;
      }

      inline CallInstance getCallInstance(void) const
      {
         return _callInstance;
      }

      inline void setAcceptOperation(IN const AcceptOperation acceptOperation)
      {
         _acceptOperation = acceptOperation;
      }

      inline AcceptOperation getAcceptOperation(void) const
      {
         return _acceptOperation;
      }

   private:
      BdAddress         _bdAddress;         /**< Address of the BT device*/
      CallInstance      _callInstance;      /**< Call instance of the call*/
      AcceptOperation   _acceptOperation;   /**< type of Accept operation*/
   };

   class PmCoreIfMessage_SwapCallRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SwapCallRequest& first, PmCoreIfMessage_SwapCallRequest& second);

      explicit PmCoreIfMessage_SwapCallRequest();

      explicit PmCoreIfMessage_SwapCallRequest(IN const BdAddress& bdAddress, IN const SwapOperation operation,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SwapCallRequest(IN const PmCoreIfMessage_SwapCallRequest& other);

      virtual ~PmCoreIfMessage_SwapCallRequest();

      PmCoreIfMessage_SwapCallRequest& operator=(PmCoreIfMessage_SwapCallRequest other);

      bool operator==(const PmCoreIfMessage_SwapCallRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setSwapOperation(IN const SwapOperation swapOperation)
      {
         _swapOperation = swapOperation;
      }

      inline SwapOperation getSwapOperation(void) const
      {
         return _swapOperation;
      }

   private:
      BdAddress         _bdAddress;         /**< Address of the BT device*/
      SwapOperation     _swapOperation;     /**< type of Swap operation*/
   };

   class PmCoreIfMessage_SpeedDialRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SpeedDialRequest& first, PmCoreIfMessage_SpeedDialRequest& second);

      explicit PmCoreIfMessage_SpeedDialRequest();

      explicit PmCoreIfMessage_SpeedDialRequest(IN const BdAddress& bdAddress, IN const MemDialIndex memDialIndex,
            IN const HideCallerId hideCallerId, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SpeedDialRequest(IN const PmCoreIfMessage_SpeedDialRequest& other);

      virtual ~PmCoreIfMessage_SpeedDialRequest();

      PmCoreIfMessage_SpeedDialRequest& operator=(PmCoreIfMessage_SpeedDialRequest other);

      bool operator==(const PmCoreIfMessage_SpeedDialRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setMemDialIndex(IN const MemDialIndex& memDialIndex)
      {
         _memDialIndex = memDialIndex;
      }

      inline const MemDialIndex& getMemDialIndex(void) const
      {
         return _memDialIndex;
      }

      inline void setHideCallerId(IN const HideCallerId hideCallerId)
      {
         _hideCallerId = hideCallerId;
      }

      inline HideCallerId getHideCallerId(void) const
      {
         return _hideCallerId;
      }

   private:
      BdAddress         _bdAddress;         /**< Address of the BT device*/
      MemDialIndex      _memDialIndex;      /**< Memory dial index*/
      HideCallerId      _hideCallerId;      /**< Instance of HideCallerId*/
   };

   class PmCoreIfMessage_SendDTMFRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_SendDTMFRequest& first, PmCoreIfMessage_SendDTMFRequest& second);

      explicit PmCoreIfMessage_SendDTMFRequest();

      explicit PmCoreIfMessage_SendDTMFRequest(IN const BdAddress& bdAddress, IN const DTMFTones& tones,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_SendDTMFRequest(IN const PmCoreIfMessage_SendDTMFRequest& other);

      virtual ~PmCoreIfMessage_SendDTMFRequest();

      PmCoreIfMessage_SendDTMFRequest& operator=(PmCoreIfMessage_SendDTMFRequest other);

      bool operator==(const PmCoreIfMessage_SendDTMFRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setDTMFTones(IN const DTMFTones& tones)
      {
         _tones = tones;
      }

      inline const DTMFTones& getDTMFTones(void) const
      {
         return _tones;
      }

   private:
      BdAddress         _bdAddress;         /**< Address of the BT device*/
      DTMFTones         _tones;             /**< DTMF tones*/
   };

   class PmCoreIfMessage_TransferAudioRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_TransferAudioRequest& first, PmCoreIfMessage_TransferAudioRequest& second);

      explicit PmCoreIfMessage_TransferAudioRequest();

      explicit PmCoreIfMessage_TransferAudioRequest(IN const BdAddress& bdAddress,
            IN const AudioDirection audioDirection, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_TransferAudioRequest(IN const PmCoreIfMessage_TransferAudioRequest& other);

      virtual ~PmCoreIfMessage_TransferAudioRequest();

      PmCoreIfMessage_TransferAudioRequest& operator=(PmCoreIfMessage_TransferAudioRequest other);

      bool operator==(const PmCoreIfMessage_TransferAudioRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setAudioDirection(IN const AudioDirection audioDirection)
      {
         _audioDirection = audioDirection;
      }

      inline AudioDirection getAudioDirection(void) const
      {
         return _audioDirection;
      }

   private:
      BdAddress         _bdAddress;         /**< Address of the BT device*/
      AudioDirection    _audioDirection;    /**< Requested audio direction*/
   };

   class PmCoreIfMessage_GetVoiceRecognitionStatusRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetVoiceRecognitionStatusRequest& first,
            PmCoreIfMessage_GetVoiceRecognitionStatusRequest& second);

      explicit PmCoreIfMessage_GetVoiceRecognitionStatusRequest();

      explicit PmCoreIfMessage_GetVoiceRecognitionStatusRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetVoiceRecognitionStatusRequest(
            IN const PmCoreIfMessage_GetVoiceRecognitionStatusRequest& other);

      virtual ~PmCoreIfMessage_GetVoiceRecognitionStatusRequest();

      PmCoreIfMessage_GetVoiceRecognitionStatusRequest& operator=(
            PmCoreIfMessage_GetVoiceRecognitionStatusRequest other);

      bool operator==(const PmCoreIfMessage_GetVoiceRecognitionStatusRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress          _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& first,
            PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& second);

      explicit PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest();

      explicit PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest(
            IN const PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& other);

      virtual ~PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest();

      PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& operator=(
            PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest other);

      bool operator==(const PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress          _bdAddress;       /**< BT address of the vehicle*/
   };

   class PmCoreIfMessage_StartStopVoiceRecognitionRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_StartStopVoiceRecognitionRequest& first,
            PmCoreIfMessage_StartStopVoiceRecognitionRequest& second);

      explicit PmCoreIfMessage_StartStopVoiceRecognitionRequest();

      explicit PmCoreIfMessage_StartStopVoiceRecognitionRequest(IN const BdAddress& bdAddress,
            IN const StartStop startStopVR, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_StartStopVoiceRecognitionRequest(
            IN const PmCoreIfMessage_StartStopVoiceRecognitionRequest& other);

      virtual ~PmCoreIfMessage_StartStopVoiceRecognitionRequest();

      PmCoreIfMessage_StartStopVoiceRecognitionRequest& operator=(
            PmCoreIfMessage_StartStopVoiceRecognitionRequest other);

      bool operator==(const PmCoreIfMessage_StartStopVoiceRecognitionRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setStartStopVR(IN const StartStop startStop)
      {
         _startStopVR = startStop;
      }

      inline StartStop getStartStopVR(void) const
      {
         return _startStopVR;
      }

   private:
      BdAddress         _bdAddress;         /**< Address of the BT device*/
      StartStop         _startStopVR;       /**< start or stop flag of VR*/
   };

   class PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& first,
            PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& second);

      explicit PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest();

      explicit PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(IN const BdAddress& bdAddress,
            IN const StartStop extStartStopVR, IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(
            IN const PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& other);

      virtual ~PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest();

      PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& operator=(
            PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest other);

      bool operator==(const PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

      inline void setStartStopExtVR(IN const StartStop startStop)
      {
         _startStopExtVR = startStop;
      }

      inline StartStop getStartStopExtVR(void) const
      {
         return _startStopExtVR;
      }

   private:
      BdAddress         _bdAddress;         /**< Address of the BT device*/
      StartStop         _startStopExtVR;       /**< start or stop flag of VR*/
   };

   class PmCoreIfMessage_stopExtendedVRRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_stopExtendedVRRequest& first,
            PmCoreIfMessage_stopExtendedVRRequest& second);

      explicit PmCoreIfMessage_stopExtendedVRRequest();

      explicit PmCoreIfMessage_stopExtendedVRRequest(IN const BdAddress& bdAddress,
            IN const ActType act,
            IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_stopExtendedVRRequest(
            IN const PmCoreIfMessage_stopExtendedVRRequest& other);

      virtual ~PmCoreIfMessage_stopExtendedVRRequest();

      PmCoreIfMessage_stopExtendedVRRequest& operator=(
            PmCoreIfMessage_stopExtendedVRRequest other);

      bool operator==(const PmCoreIfMessage_stopExtendedVRRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress         _bdAddress;         /**< Address of the BT device */
   };

   class PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest final : public PmCoreIfMessageRequest
   {
   public:

      friend void swap(PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& first,
            PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& second);

      explicit PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest();

      explicit PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest(IN const BdAddress& bdAddress,
            IN const ActType act, IN const PmCoreIfMsgOrigin origin = PM_CORE_IF_MSG_ORIGIN_CLIENT);

      PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest(
            IN const PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& other);

      virtual ~PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest();

      PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& operator=(
            PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest other);

      bool operator==(const PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest& rhs) const;

      virtual void traceMessage(void) const override;

      inline void setBdAddress(IN const BdAddress& bdAddress)
      {
         _bdAddress = bdAddress;
      }

      inline const BdAddress& getBdAddress(void) const
      {
         return _bdAddress;
      }

   private:
      BdAddress         _bdAddress;       /**< BT address of the vehicle*/
   };

} //namespace pmcore

#endif // PmCoreIfMessageRequest_h
