/**
 * @file PmAudioManagerWrapper.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmAudioManagerWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the definitions for the wrapper interface to PM AudioManager in PM Core.
 *
 * @ingroup PmCore
 */

#include "PmAudioManagerWrapper.h"
#include "PmAudioManagerRequestIf.h"
#include "PmCoreIfMessageCreator.h"
#include "PmCoreMainController.h"
#include "PropertyUpdateNotifierToCore.h"
#include <algorithm>
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PmAudioManagerWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

using namespace pmaudiomanager;

namespace pmcore
{

   std::map<pmaudiomanager::AmSessionId, RequestInfo> PmAudioManagerWrapper::_amSessionIdRequestInfoMap;
   BdAddressList PmAudioManagerWrapper::_activeDevicesList;
   std::vector<PrepareAudioRouteRequestJob> PmAudioManagerWrapper::_preparingAudioRouteJobsList;

   PmAudioManagerWrapper::PmAudioManagerWrapper() : _amWrapperResourceLock(),
         _iPmAudioManagerRequestIf(nullptr)
   {
      ETG_TRACE_USR4(("PmAudioManagerWrapper"));
   }

   PmAudioManagerWrapper::~PmAudioManagerWrapper()
   {
      ETG_TRACE_USR4(("~PmAudioManagerWrapper"));

      _iPmAudioManagerRequestIf = nullptr;

      _amSessionIdRequestInfoMap.clear();
      _activeDevicesList.clear();
      _preparingAudioRouteJobsList.clear();
   }

   void PmAudioManagerWrapper::setActiveDevicesList(IN const BdAddressList activeDevicesList)
   {
      ETG_TRACE_USR4(("setActiveDevicesList"));

      // TODO: When the element in the _amSessionIdRequestInfoMap turns to be passive,
      // the corresponding changes need to be checked.
      _activeDevicesList = activeDevicesList;

      ETG_TRACE_USR4(("ActiveDevicesList: "));
      for_each(_activeDevicesList.begin(), _activeDevicesList.end(),
            [](BdAddress & obj) {
         ETG_TRACE_USR4(("DeviceAddress: %s", obj.c_str()));
      });
   }

   AmResult PmAudioManagerWrapper::preparePmAudioManager(ahl_tclBaseOneThreadApp* mainApplication)
   {
      ETG_TRACE_USR4(("preparePmAudioManager"));

      _iPmAudioManagerRequestIf = &(PmAudioManagerRequestIf::getInstance());

      // Initializing PM_Audiomanager
      AmResult amResult = _iPmAudioManagerRequestIf->intializePmAudioManager(mainApplication);

      if (amResult._amResultCode == AM_RESULT_OK)
      {
         amResult = _iPmAudioManagerRequestIf->registerPmAudioManagerCallbackIf(this);
      }

      return amResult;
   }

   void PmAudioManagerWrapper::sendSetAmState(IN const AmState amState)
   {
      ETG_TRACE_USR4(("setAmState: amState: %u", amState));

      _iPmAudioManagerRequestIf->setAmState(amState);
   }

   AmResult PmAudioManagerWrapper::sendPrepareAudioRoute(IN const BdAddress& deviceAddress,
         IN const AudioChannel audioChannelId, IN const ControllerOriginEnum controllerOriginEnumType)
   {
      ETG_TRACE_USR4(("sendPrepareAudioRoute: deviceAddress: %s", deviceAddress.c_str()));
      ETG_TRACE_USR4(("sendPrepareAudioRoute: audioChannelId: %u", audioChannelId));

      AmResult amResult(AM_RESULT_OK, "");

      if ((true == isActiveDevice(deviceAddress)) || (0 == deviceAddress.compare(DEVICE_ADDRESS_ALL)))
      {
         AmSessionId amSessionId = AM_SESSION_ID_DEFAULT;

         // If amSessionId is "AM_SESSION_ID_DEFAULT", it implies that the request is new.
         // Hence the PM_AudioManager has to set the new Am Session ID.
         // Else the existing Session Id is used for this communication also.
         getAmSessionId(deviceAddress, amSessionId);

         if (AM_SESSION_ID_DEFAULT == amSessionId)
         {
            PrepareAudioRouteRequestJob recentJob;
            getRecentJobFromPreparingAudioRouteJobsList(deviceAddress, recentJob);

            // Handling 7(a)
            if (!(recentJob._deviceAddress.empty()))
            {
               ETG_TRACE_USR4(("Device already present in the _preparingAudioRouteJobsList"));

               // If the request from the same device address is for the recent request audio channel, then
               // it is responded and returned here itself.
               // If the request from the same device address is for the different audio channel other than recent one,
               // then the request is added to the list and on receiving the response, the session ID is used and the
               // request is posted to PM AudioManager.
               if (audioChannelId == recentJob._audioChannelId)
               {
                  ETG_TRACE_USR4(("Req received for the same recent AudioChannel"));
                  return amResult;
               }
            }

            // Handling 6(a), 7(b)(i) and 7(c)(i)
            insertElementIntoPreparingAudioRouteJobsList(PrepareAudioRouteRequestJob(deviceAddress, audioChannelId,
                  controllerOriginEnumType, PrepareAudioRouteRequestJob::PREPARE_AUDIO_ROUTE));

            if (isPrepareAudioRouteRequestDisposable())
            {
               amResult = _iPmAudioManagerRequestIf->prepareAudioRoute(amSessionId, audioChannelId);
            }
         }
         else
         {
            // Handling 6(b)
            amResult = _iPmAudioManagerRequestIf->prepareAudioRoute(amSessionId, audioChannelId);
            updateAmSessionIdRequestInfoMap(amSessionId, deviceAddress, audioChannelId, controllerOriginEnumType);
         }
      }
      else if (CLIENT_REQUEST == deviceAddress)
      {
         // TODO: This follow will be handled later.
         // updateAmSessionIdRequestInfoMap(AM_SESSION_ID_CLIENTREQUEST, deviceAddress, controllerOriginEnumType);
         // amResult = _iPmAudioManagerRequestIf->prepareAudioRoute(AM_SESSION_ID_CLIENTREQUEST, audioChannelId);
      }
      else
      {
         ETG_TRACE_USR4(("Passive Device"));

         // For Passive device, the response and the AmNotificationEvent are simulated
         sendAmResponse(deviceAddress, amResult, AM_MSG_PREPARE_AUDIO_PROUTE, controllerOriginEnumType);
         sendAmNotificationEvent(deviceAddress, audioChannelId, StreamingState(StreamingStateEnumType::NOT_PLAYING),
               AmEventDetails(AUDIO_STATE_CHANGE, ""));
      }

      return amResult;
   }

   AmResult PmAudioManagerWrapper::sendPlayAudio(IN const BdAddress& deviceAddress, IN const FilePath& filePath,
         IN const ControllerOriginEnum controllerOriginEnumType, IN const PlayCount playCount,
         IN const ToneType toneType)
   {
      ETG_TRACE_USR4(("sendPlayAudio: deviceAddress: %s", deviceAddress.c_str()));
      ETG_TRACE_USR4(("sendPlayAudio: filePath: %s", filePath.c_str()));
      ETG_TRACE_USR4(("playCount: %u", playCount));

      AmResult amResult(AM_RESULT_OK, "");

      AmSessionId amSessionId = AM_SESSION_ID_DEFAULT;
      getAmSessionId(deviceAddress, amSessionId);

      if (AM_SESSION_ID_DEFAULT != amSessionId)
      {
         amResult = _iPmAudioManagerRequestIf->playAudio(amSessionId, filePath, playCount, toneType);
      }
      else
      {
         ETG_TRACE_USR4(("Passive Device"));

         // For Passive device, the response and the AmNotificationEvent are simulated.
         // Only for Vehicle Ringtone, sendPlayAudio() shall be requested by the Passive device.
         // Because in other cases SCO will not be available to invoke sendPlayAudio()
         sendAmResponse(deviceAddress, amResult, AM_MSG_PLAY_AUDIO, controllerOriginEnumType);

         sendAmNotificationEvent(deviceAddress, AM_VEHICLERINGTONE,
               StreamingState(StreamingStateEnumType::PLAYING), AmEventDetails());
      }

      return amResult;
   }

   AmResult PmAudioManagerWrapper::sendPlayAudio(IN const BdAddress& deviceAddress,
         IN const SCOStatus scoStatus)
   {
      ETG_TRACE_USR4(("sendPlayAudio: deviceAddress: %s", deviceAddress.c_str()));
      ETG_TRACE_USR4(("sendPlayAudio: scoStatus: %u", scoStatus));

      AmResult amResult(AM_RESULT_OK, "");

      AmSessionId amSessionId = AM_SESSION_ID_DEFAULT;
      getAmSessionId(deviceAddress, amSessionId);

      if (AM_SESSION_ID_DEFAULT != amSessionId)
      {
         SamplingType samplingType = UNKNOWN;
         switch (scoStatus)
         {
            case SCO_NARROWBAND:
            {
               samplingType = NARROWBAND;
            }
            break;

            case SCO_WIDEBAND:
            {
               samplingType = WIDEBAND;
            }
            break;

            case SCO_NARROWBAND_SIRI:
            {
               samplingType = NARROWBAND_SIRI;
            }
            break;

            case SCO_WIDEBAND_SIRI:
            {
               samplingType = WIDEBAND_SIRI;
            }
            break;

            default:
               samplingType = UNKNOWN;
         }

         amResult = _iPmAudioManagerRequestIf->playAudio(amSessionId, samplingType);
      }
      else
      {
         ETG_TRACE_USR4(("Code flow should not come here for a valid usecase scenario"));
         amResult._amResultCode = AM_RESULT_ERR_INVALID_PARAMETER;
      }

      return amResult;
   }

   AmResult PmAudioManagerWrapper::sendStopAudio(IN const BdAddress& deviceAddress,
         IN const ControllerOriginEnum controllerOriginEnumType)
   {
      ETG_TRACE_USR4(("sendStopAudio: deviceAddress: %s", deviceAddress.c_str()));

      AmResult amResult(AM_RESULT_OK, "");

      AmSessionId amSessionId = AM_SESSION_ID_DEFAULT;
      getAmSessionId(deviceAddress, amSessionId);

      if (AM_SESSION_ID_DEFAULT != amSessionId)
      {
         amResult = _iPmAudioManagerRequestIf->stopAudio(amSessionId);
      }
      else
      {
         if (isActiveDevice(deviceAddress))
         {
            // Active device with a Default AmSessionId implies that the item is waiting in the
            // "_amSessionIdRequestInfoMap" to get prepareAudioRouteResponse or a valid AmSessionId.
            // Now the received request is pushed into "_preparingAudioRouteJobsList" to be processed
            // after getting the response for the previous request from the same device

            // Check if prepareAudio request is already pushed into "_preparingAudioRouteJobsList".
            // Else the response and the AmNotificationEvent are simulated.

            /* STOP_AUDIO_SUCCESS event is simulated to handle the below scenario:
             * 1. Incoming call in In-band supported phone. Hence Phone channel is requested
             * 2. The Phone channel is denied
             * 3. since the Phone channel is denied, phoneAudioState of PmCallHandler SCT is set to
             *    "PHONE_CHANNEL_NOT_ALLOCATED"
             * 4. Now the call is ended. And hence stopAudio is posted by the SCT and the SCT waits for the
             *    response.
             * 5. Hence the Stopaudio response is simulated from here.
             */

            auto iter = std::find_if(_preparingAudioRouteJobsList.rbegin(), _preparingAudioRouteJobsList.rend(),
                  [&deviceAddress](PrepareAudioRouteRequestJob const& obj){return obj._deviceAddress == deviceAddress;});

            if (_preparingAudioRouteJobsList.rend() != iter)
            {
               insertElementIntoPreparingAudioRouteJobsList(PrepareAudioRouteRequestJob(deviceAddress,
                     AM_UNKNOWN, controllerOriginEnumType,
                     PrepareAudioRouteRequestJob::STOP_AUDIO));
            }
            else
            {
               sendAmResponse(deviceAddress, amResult, AM_MSG_STOP_AUDIO, controllerOriginEnumType);

               sendAmNotificationEvent(deviceAddress, AM_VEHICLERINGTONE,
                     StreamingState(StreamingStateEnumType::STOPPED), AmEventDetails());
            }
         }
         else
         {
            // For Passive device, the response and the AmNotificationEvent are simulated.
            // To release the Vehicle Ringtone channel for Passive device, this interface is invoked
            // and hence the responses are simulated.
            sendAmResponse(deviceAddress, amResult, AM_MSG_STOP_AUDIO, controllerOriginEnumType);

            sendAmNotificationEvent(deviceAddress, AM_VEHICLERINGTONE,
                  StreamingState(StreamingStateEnumType::STOPPED), AmEventDetails());
         }
      }

      return amResult;
   }

   AmResult PmAudioManagerWrapper::sendSetMicMuteState(IN const MuteState muteState)
   {
      ETG_TRACE_USR4(("sendSetMicMuteState: muteState: %u", muteState));

      AmResult amResult(AM_RESULT_OK, "");
      amResult = _iPmAudioManagerRequestIf->setMicMuteState(muteState);
      return amResult;
   }

   AmResult PmAudioManagerWrapper::sendPauseAudio(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("sendPauseAudio: deviceAddress: %s", deviceAddress.c_str()));

      AmResult amResult(AM_RESULT_OK, "");

      AmSessionId amSessionId = AM_SESSION_ID_DEFAULT;
      getAmSessionId(deviceAddress, amSessionId);

      if (AM_SESSION_ID_DEFAULT != amSessionId)
      {
         amResult = _iPmAudioManagerRequestIf->pauseAudio(amSessionId);
      }
      else
      {
         ETG_TRACE_USR4(("Code flow should not come here for a valid usecase scenario"));
         amResult._amResultCode = AM_RESULT_ERR_INVALID_PARAMETER;
      }

      return amResult;
   }

   AmResult PmAudioManagerWrapper::getAudioSinkVolume(IN const BdAddress& /*deviceAddress*/, IN const GeniviAMSinkId sinkId)
   {
      ETG_TRACE_USR4(("getAudioSinkVolume"));

      AmResult amResult(AM_RESULT_OK, "");
      if(nullptr != _iPmAudioManagerRequestIf)
      {
         amResult = _iPmAudioManagerRequestIf->getAudioSinkVolume(VALUE_ONE, sinkId);
      }
      return amResult;
      //TODO:Check how to map deviceAddress with amSessionId
//      AmSessionId amSessionId = AM_SESSION_ID_DEFAULT;
//      getAmSessionId(deviceAddress, amSessionId);
   }

   AmResult PmAudioManagerWrapper::setAudioSinkVolume(IN const BdAddress& /*deviceAddress*/, IN const GeniviAMSinkId sinkId,
         IN const AmPropertyType2VolumeMap propertyVal)
   {
      ETG_TRACE_USR4(("setAudioSinkVolume"));

      AmResult amResult(AM_RESULT_OK, "");
      if(nullptr != _iPmAudioManagerRequestIf)
      {
         amResult = _iPmAudioManagerRequestIf->setAudioSinkVolume(VALUE_ONE, sinkId, propertyVal);
      }

      //TODO:Check how to map deviceAddress with amSessionId
//      AmSessionId amSessionId = AM_SESSION_ID_DEFAULT;
//      getAmSessionId(deviceAddress, amSessionId);
//
//      if (AM_SESSION_ID_DEFAULT != amSessionId)
//      {
//         amResult = _iPmAudioManagerRequestIf->setAudioSinkVolume(amSessionId, sinkId, propertyVal);
//      }
//      else
//      {
//         ETG_TRACE_USR4(("Code flow should not come here for a valid usecase scenario"));
//         amResult._amResultCode = AM_RESULT_ERR_INVALID_PARAMETER;
//      }
      return amResult;
   }

   void PmAudioManagerWrapper::prepareAudioRouteResponse(IN const AmSessionId amSessionId,
         IN const AmResult amResult)
   {
      ETG_TRACE_USR4(("prepareAudioRouteResponse: amSessionId: %u", amSessionId));

      BdAddress deviceAddress = "";
      getDeviceAddress(amSessionId, deviceAddress);

      if (deviceAddress.empty())
      {
         if (!isPreparingAudioRouteJobsListEmpty())
         {
            PrepareAudioRouteRequestJob frontEndJob;
            getFrontElementFromPreparingAudioRouteJobsList(frontEndJob);

            deviceAddress = frontEndJob._deviceAddress;
            sendAmResponse(deviceAddress, amResult, AM_MSG_PREPARE_AUDIO_PROUTE,
                  frontEndJob._controllerOriginEnumType);

            if (AM_RESULT_OK == amResult._amResultCode)
            {
               updateAmSessionIdRequestInfoMap(amSessionId, frontEndJob._deviceAddress,
                     frontEndJob._audioChannelId, frontEndJob._controllerOriginEnumType);
            }

            removeFrontElementFromPreparingAudioRouteJobsList();
         }

         // Next request should be processed from the list - "_preparingAudioRouteJobsList"
         // only if the received response has got a new valid AmSessionId
         processRequestFromJobsList(deviceAddress, amSessionId, amResult);
      }
      else
      {
         sendAmResponse(deviceAddress, amResult, AM_MSG_PREPARE_AUDIO_PROUTE);
      }
   }

   void PmAudioManagerWrapper::playAudioResponse(IN const AmSessionId amSessionId,
         IN const AmResult amResult)
   {
      ETG_TRACE_USR4(("playAudioResponse: amSessionId: %u", amSessionId));

      BdAddress deviceAddress = "";
      getDeviceAddress(amSessionId, deviceAddress);

      if (!deviceAddress.empty())
      {
         sendAmResponse(deviceAddress, amResult, AM_MSG_PLAY_AUDIO);
      }
   }

   void PmAudioManagerWrapper::stopAudioResponse(IN const AmSessionId amSessionId,
         IN const AmResult amResult)
   {
      ETG_TRACE_USR4(("stopAudioResponse: amSessionId: %u", amSessionId));

      BdAddress deviceAddress = "";
      getDeviceAddress(amSessionId, deviceAddress);

      if (!deviceAddress.empty())
      {
         sendAmResponse(deviceAddress, amResult, AM_MSG_STOP_AUDIO);
      }
   }

   void PmAudioManagerWrapper::setMicMuteStateResponse(IN const AmSessionId amSessionId,
         IN const AmResult amResult)
   {
      ETG_TRACE_USR4(("setMicMuteStateResponse: amSessionId: %u", amSessionId));

      BdAddress deviceAddress = "";
      getDeviceAddress(amSessionId, deviceAddress);

      if (!deviceAddress.empty())
      {
         sendAmResponse(deviceAddress, amResult, AM_MSG_SET_MIC_MUTE_STATE);
      }
   }

   void PmAudioManagerWrapper::pauseAudioResponse(IN const AmSessionId amSessionId,
         IN const AmResult amResult)
   {
      ETG_TRACE_USR4(("pauseAudioResponse: amSessionId: %u", amSessionId));

      BdAddress deviceAddress = "";
      getDeviceAddress(amSessionId, deviceAddress);

      if (!deviceAddress.empty())
      {
         sendAmResponse(deviceAddress, amResult, AM_MSG_PAUSE_AUDIO);
      }
   }

   void PmAudioManagerWrapper::getAudioSinkVolumeResponse(IN const AmSessionId amSessionId, IN const AmResult amResult)
   {
      ETG_TRACE_USR4(("getAudioSinkVolumeResponse: amSessionId: %u", amSessionId));

      //TODO:Check how to map deviceAddress with amSessionId
      BdAddress deviceAddress = "";
      getDeviceAddress(amSessionId, deviceAddress);

      if (!deviceAddress.empty())
      {
         sendAmResponse(deviceAddress, amResult, AM_MSG_GET_AUDIO_SINK);
      }
   }

   void PmAudioManagerWrapper::setAudioSinkVolumeResponse(IN const AmSessionId amSessionId, IN const AmResult amResult)
   {
      ETG_TRACE_USR4(("setAudioSinkVolumeResponse: amSessionId: %u", amSessionId));

      //TODO:Check how to map deviceAddress with amSessionId
      BdAddress deviceAddress = "";
      getDeviceAddress(amSessionId, deviceAddress);

      if (!deviceAddress.empty())
      {
         sendAmResponse(deviceAddress, amResult, AM_MSG_SET_AUDIO_SINK);
      }
   }

   void PmAudioManagerWrapper::updateAudioSinkVolumeList(IN const AmSessionId amSessionId, IN const AmResult amResult,
            IN const GeniviAMSinkId sinkId, IN const AmPropertyType2VolumeMap propertyList)
   {
      ETG_TRACE_USR4(("updateAudioSinkVolumeList: amSessionId: %u", amSessionId));
      (void)amResult;
      (void)sinkId;

      //Send update to RV Controller
      BdAddress deviceAddress("");
      std::shared_ptr<PropertyDetails<AmPropertyType2VolumeMap>> propertyDetails =
            std::make_shared<PropertyDetails<AmPropertyType2VolumeMap>>(deviceAddress, propertyList);

      PropertyUpdateNotifierToCore::getInstance().notifyControllers(AM_UPDATE_AUDIO_VOLUME,
            propertyDetails);
   }

   void PmAudioManagerWrapper::amNotificationEvent(IN const AmSessionId amSessionId,
         IN const AudioChannel audioChannelId,
         IN const StreamingState streamingState,
         IN const AmEventDetails amEventDetails)
   {
      ETG_TRACE_USR4(("amNotificationEvent: amSessionId: %u", amSessionId));

      BdAddress deviceAddress = "";
      getDeviceAddress(amSessionId, deviceAddress);

      if (!deviceAddress.empty())
      {
         sendAmNotificationEvent(deviceAddress, audioChannelId, streamingState, amEventDetails);

         /* Removing the element from the STL
          * 1. when the stop audio is successfully done (channel is also successfully released for that device)
          * 2. When the audio streaming is stopped due to external trigger
          * 3. When prepareAudioRoute request was posted successfully and the response is also success,
          *    but the channel acquisition was actually Failed.
          */
         if(SESSION_REMOVED == amEventDetails._event)
         {
            eraseAmSessionIdRequestInfoMapElement(amSessionId);
         }
      }
      else
      {
         ETG_TRACE_USR4(("Invalid amSessionId received"));
      }
   }

   void PmAudioManagerWrapper::eraseAmSessionIdRequestInfoMapElement(const AmSessionId amSessionId)
   {
      ETG_TRACE_USR4(("eraseAmSessionIdRequestInfoMapElement"));

      Locker locker(&_amWrapperResourceLock);

      _amSessionIdRequestInfoMap.erase(amSessionId);
      printAmSessionIdRequestInfoMap();
   }

   void PmAudioManagerWrapper::getDeviceAddress(const AmSessionId amSessionId, BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("getDeviceAddress"));

      deviceAddress = "";

      Locker locker(&_amWrapperResourceLock);

      printAmSessionIdRequestInfoMap();

      auto iter = _amSessionIdRequestInfoMap.find(amSessionId);
      if (_amSessionIdRequestInfoMap.end() != iter)
      {
         deviceAddress = iter->second._deviceAddress;
      }
   }

   void PmAudioManagerWrapper::getAmSessionId(const BdAddress& deviceAddress, pmaudiomanager::AmSessionId& amSessionId)
   {
      ETG_TRACE_USR4(("getAmSessionId"));

      amSessionId = AM_SESSION_ID_DEFAULT;

      Locker locker(&_amWrapperResourceLock);

      printAmSessionIdRequestInfoMap();

      auto iter = std::find_if(_amSessionIdRequestInfoMap.begin(), _amSessionIdRequestInfoMap.end(),
            [&deviceAddress](std::pair<AmSessionId, RequestInfo> const& obj)
            {return obj.second._deviceAddress == deviceAddress;});

      if (_amSessionIdRequestInfoMap.end() != iter)
      {
         amSessionId = iter->first;
      }
   }

   void PmAudioManagerWrapper::getControllerOrigin(const BdAddress& deviceAddress,
         ControllerOriginEnum& controllerOriginEnumType)
   {
      ETG_TRACE_USR4(("getControllerOrigin"));

      Locker locker(&_amWrapperResourceLock);

      printAmSessionIdRequestInfoMap();

      controllerOriginEnumType = MAIN_CONTROLLER;

      auto iter = std::find_if(_amSessionIdRequestInfoMap.begin(), _amSessionIdRequestInfoMap.end(),
            [&deviceAddress](std::pair<AmSessionId, RequestInfo> const& obj)
            {return obj.second._deviceAddress == deviceAddress;});

      if (_amSessionIdRequestInfoMap.end() != iter)
      {
         controllerOriginEnumType = iter->second._controllerOriginEnumType;
      }
   }

   void PmAudioManagerWrapper::updateAmSessionIdRequestInfoMap(AmSessionId amSessionId, BdAddress deviceAddress,
         AudioChannel audioChannelId, ControllerOriginEnum controllerOriginEnumType)
   {
      ETG_TRACE_USR4(("updateAmSessionIdRequestInfoMap"));

      Locker locker(&_amWrapperResourceLock);

      auto iter = std::find_if(_amSessionIdRequestInfoMap.begin(), _amSessionIdRequestInfoMap.end(),
            [&deviceAddress](std::pair<AmSessionId, RequestInfo> const& obj)
            {return obj.second._deviceAddress == deviceAddress;});

      if (_amSessionIdRequestInfoMap.end() != iter)
      {
         // Updating the Controller origin
         iter->second._controllerOriginEnumType = controllerOriginEnumType;
         iter->second._audioChannelId = audioChannelId;
      }
      else
      {
         _amSessionIdRequestInfoMap.emplace_hint(_amSessionIdRequestInfoMap.end(), amSessionId,
               RequestInfo(deviceAddress, audioChannelId, controllerOriginEnumType));
      }

      printAmSessionIdRequestInfoMap();
   }

   void PmAudioManagerWrapper::printAmSessionIdRequestInfoMap()
   {
      ETG_TRACE_USR4(("printAmSessionIdRequestInfoMap"));

      for(auto iter : _amSessionIdRequestInfoMap)
      {
         ETG_TRACE_USR4(("AmSessionId: %u", ETG_CENUM(AmSessionId, iter.first)));

         ETG_TRACE_USR4(("DeviceAddress: %s", iter.second._deviceAddress.c_str()));
         ETG_TRACE_USR4(("AudioChannelId: %u", ETG_CENUM(AudioChannel, iter.second._audioChannelId)));
         ETG_TRACE_USR4(("ControllerOriginEnumType: %u",
               ETG_CENUM(ControllerOriginEnum, iter.second._controllerOriginEnumType)));
      }
   }

   void PmAudioManagerWrapper::printPreparingAudioRouteJobsList()
   {
      ETG_TRACE_USR4(("printPreparingAudioRouteJobsList"));

      for_each(_preparingAudioRouteJobsList.begin(), _preparingAudioRouteJobsList.end(),
            [](PrepareAudioRouteRequestJob & obj) {
         ETG_TRACE_USR4(("DeviceAddress: %s", obj._deviceAddress.c_str()));
         ETG_TRACE_USR4(("AudioChannelId: %u", ETG_CENUM(AudioChannel, obj._audioChannelId)));
         ETG_TRACE_USR4(("ControllerOriginEnumType: %u",
               ETG_CENUM(ControllerOriginEnum, obj._controllerOriginEnumType)));
         ETG_TRACE_USR4(("OperationToPerform: %u",
               ETG_CENUM(OperationToPerform, obj._operationToPerform)));
      });
   }

   void PmAudioManagerWrapper::sendAmResponse(const BdAddress& deviceAddress,
         const AmResult& amResult, const AmMessageIdEnum amMessageId)
   {
      ETG_TRACE_USR4(("sendAmResponse"));

      ControllerOriginEnum controllerOriginEnumType = MAIN_CONTROLLER;
      getControllerOrigin(deviceAddress, controllerOriginEnumType);

      sendAmResponse(deviceAddress, amResult, amMessageId, controllerOriginEnumType);
   }

   void PmAudioManagerWrapper::sendAmResponse(const BdAddress& deviceAddress,
         const AmResult& amResult, const AmMessageIdEnum amMessageId,
         ControllerOriginEnum controllerOriginEnumType)
   {
      ETG_TRACE_USR4(("sendAmResponse: deviceAddress: %s", deviceAddress.c_str()));

      std::shared_ptr<PmCoreIfMessage> pmCoreIfMessage = nullptr;

      ETG_TRACE_USR4(("controllerOriginEnumType: %u", controllerOriginEnumType));

      switch (amMessageId)
      {
         case AM_MSG_PREPARE_AUDIO_PROUTE:
         {
            pmCoreIfMessage = getNewPmCoreIfMessage_PrepareAudioRouteResponse(amResult,
                  deviceAddress, controllerOriginEnumType, PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF);
         }
         break;

         case AM_MSG_PLAY_AUDIO:
         {
            pmCoreIfMessage = getNewPmCoreIfMessage_PlayAudioResponse(amResult,
                  deviceAddress, controllerOriginEnumType, PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF);
         }
         break;

         case AM_MSG_STOP_AUDIO:
         {
            pmCoreIfMessage = getNewPmCoreIfMessage_StopAudioResponse(amResult,
                  deviceAddress, controllerOriginEnumType, PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF);
         }
         break;

         case AM_MSG_SET_MIC_MUTE_STATE:
         {
            pmCoreIfMessage = getNewPmCoreIfMessage_SetMicMuteStateResponse(amResult,
                  deviceAddress, controllerOriginEnumType, PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF);
         }
         break;

         case AM_MSG_PAUSE_AUDIO:
         {
            pmCoreIfMessage = getNewPmCoreIfMessage_PauseAudioResponse(amResult,
                  deviceAddress, controllerOriginEnumType, PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF);
         }
         break;
         case AM_MSG_GET_AUDIO_SINK:
         {
            pmCoreIfMessage = getNewPmCoreIfMessage_GetAudioSinkVolumeResponse(amResult,
                  deviceAddress, controllerOriginEnumType, PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF);
         }
         break;
         case AM_MSG_SET_AUDIO_SINK:
         {
            pmCoreIfMessage = getNewPmCoreIfMessage_SetAudioSinkVolumeResponse(amResult,
                  deviceAddress, controllerOriginEnumType, PM_CORE_IF_MSG_ORIGIN_PM_AUDIOMANAGER_IF);
         }
         break;

         default:
            ETG_TRACE_USR4(("sendPmCoreIfMessage: default case"));
      }

      if (pmCoreIfMessage)
      {
         pmCoreIfMessage->traceMessage();
         PmCoreMainController::getInstance().handlePmCoreIfMessage(pmCoreIfMessage);
      }
   }

   void PmAudioManagerWrapper::sendAmNotificationEvent(const BdAddress& deviceAddress,
         const AudioChannel audioChannelId,
         const StreamingState streamingState, const AmEventDetails amEventDetails)
   {
      ETG_TRACE_USR4(("sendAmNotificationEvent: deviceAddress: %s", deviceAddress.c_str()));

      PmCorePropertyAndEventId eventToControllers = PROPERTY_AND_EVENTID_LAST;
      AmErrorMessage amErrorMessage = "";

      // The Event to be posted to the SM should be decided here based on the parameters received
      switch (amEventDetails._event)
      {
         case AUDIO_STATE_CHANGE:
         {
            ETG_TRACE_USR4(("AUDIO_STATE_CHANGE"));
            switch (streamingState)
            {
               case StreamingStateEnumType::NOT_PLAYING:
                  setAudioChannelEvent(audioChannelId, AM_MSG_CHANNEL_ACQUISITION, AM_RESPONSE_SUCCESS,
                        eventToControllers);
                  break;
               case STOPPED:
                  setAudioChannelEvent(audioChannelId, AM_MSG_STOP_AUDIO, AM_RESPONSE_SUCCESS,
                        eventToControllers);

                  if (STREAMING_STOPPED_DUE_TO_EXT_TRIGGER == amEventDetails._message)
                  {
                     amErrorMessage = STREAMING_STOPPED_DUE_TO_EXT_TRIGGER;
                  }
                  break;
               case StreamingStateEnumType::PLAYING:
                  if ("Unmuted" == amEventDetails._message)
                  {
                     eventToControllers = AM_SET_MIC_UNMUTE_SUCCESS;
                  }
                  else
                  {
                     setStreamingStateEvent(audioChannelId, AM_MSG_PLAY_AUDIO, AM_RESPONSE_SUCCESS, eventToControllers);
                  }
                  break;
               case StreamingStateEnumType::PAUSED:
                  setStreamingStateEvent(audioChannelId, AM_MSG_PAUSE_AUDIO, AM_RESPONSE_SUCCESS, eventToControllers);
                  if (STREAMING_PAUSED_DUE_TO_EXT_TRIGGER == amEventDetails._message)
                  {
                     amErrorMessage = STREAMING_PAUSED_DUE_TO_EXT_TRIGGER;
                  }
                  break;
               case StreamingStateEnumType::MUTED:
                  eventToControllers = AM_SET_MIC_MUTE_SUCCESS;
                  break;
               case StreamingStateEnumType::DEMUTED:
                  eventToControllers = AM_SET_MIC_UNMUTE_SUCCESS;
                  break;
               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         case REQUEST_FAILURE:
         {
            ETG_TRACE_USR4(("REQUEST_FAILURE"));

            AmMessageIdEnum amMessageId;
            std::string failedEvent = "";
            uint64 delimiterPos = amEventDetails._message.find("-");
            if (delimiterPos != std::string::npos)
            {
               failedEvent = amEventDetails._message.substr(0, delimiterPos);
               amErrorMessage = amEventDetails._message.substr(delimiterPos + 1);
               setAmMessageIdEnum(failedEvent, amMessageId);

               switch (amMessageId)
               {
                  case AM_MSG_CHANNEL_ACQUISITION: // Intentional fall through
                  case AM_MSG_STOP_AUDIO:
                  {
                     setAudioChannelEvent(audioChannelId, amMessageId, AM_RESPONSE_FAILURE,
                           eventToControllers);
                  }
                  break;

                  case AM_MSG_PLAY_AUDIO: // Intentional fall through
                  case AM_MSG_PAUSE_AUDIO:
                  {
                     setStreamingStateEvent(audioChannelId, amMessageId, AM_RESPONSE_FAILURE, eventToControllers);
                  }
                  break;
                  case AM_MSG_SET_MIC_MUTE:
                  {
                     eventToControllers = AM_SET_MIC_MUTE_FAILURE;
                  }
                  break;
                  case AM_MSG_SET_MIC_UNMUTE:
                  {
                     eventToControllers = AM_SET_MIC_UNMUTE_FAILURE;
                  }
                  break;
                  default:
                     eventToControllers = PROPERTY_AND_EVENTID_LAST;
               }
            }
         }
         break;
      }

      //Send update to Controllers- Call Controller and VR Controller
      std::shared_ptr<PropertyDetails<AmErrorMessage>> propertyDetails =
            std::make_shared<PropertyDetails<AmErrorMessage>>(deviceAddress, amErrorMessage);

      PropertyUpdateNotifierToCore::getInstance().notifyControllers(eventToControllers, propertyDetails);
   }

   void PmAudioManagerWrapper::setAmMessageIdEnum(const std::string& failedEvent, AmMessageIdEnum& amMessageId)
   {
      ETG_TRACE_USR4(("setAmMessageIdEnum"));

      if ("ChannelAcquisition" == failedEvent)
      {
         amMessageId = AM_MSG_CHANNEL_ACQUISITION;
      }
      else if ("PlayAudio" == failedEvent)
      {
         amMessageId = AM_MSG_PLAY_AUDIO;
      }
      else if ("StopAudio" == failedEvent)
      {
         amMessageId = AM_MSG_STOP_AUDIO;
      }
      else if ("SetMicMute" == failedEvent)
      {
         amMessageId = AM_MSG_SET_MIC_MUTE;
      }
      else if ("SetMicUnmute" == failedEvent)
      {
         amMessageId = AM_MSG_SET_MIC_UNMUTE;
      }
      else if ("PauseAudio" == failedEvent)
      {
         amMessageId = AM_MSG_PAUSE_AUDIO;
      }
      else
      {
         amMessageId = AM_MSG_CHANNEL_ACQUISITION;
      }
   }

   void PmAudioManagerWrapper::setAudioChannelEvent(const AudioChannel audioChannelId,
         const AmMessageIdEnum amMessageId,
         const AmResponse amResponse, PmCorePropertyAndEventId& eventToControllers)
   {
      ETG_TRACE_USR4(("setAudioChannelEvent"));

      switch(audioChannelId)
      {
         case AM_VEHICLERINGTONE:
         {
            switch (amMessageId)
            {
               case AM_MSG_CHANNEL_ACQUISITION:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_RINGTONE_CHANNEL_ACQUISITION_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_RINGTONE_CHANNEL_ACQUISITION_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_RINGTONE_CHANNEL_ACQUISITION_FAILURE;
                  }
               }
               break;

               case AM_MSG_STOP_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_RINGTONE_CHANNEL_STOPAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE;
                  }
               }
               break;

               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         case AM_PHONEAUDIO:
         {
            switch (amMessageId)
            {
               case AM_MSG_CHANNEL_ACQUISITION:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_PHONE_CHANNEL_ACQUISITION_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_PHONE_CHANNEL_ACQUISITION_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_PHONE_CHANNEL_ACQUISITION_FAILURE;
                  }
               }
               break;

               case AM_MSG_STOP_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_PHONE_CHANNEL_STOPAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_PHONE_CHANNEL_STOPAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_PHONE_CHANNEL_STOPAUDIO_FAILURE;
                  }
               }
               break;

               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         case AM_WAITINGMODE:
         {
            switch (amMessageId)
            {
               case AM_MSG_CHANNEL_ACQUISITION:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_ACQUISITION_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_ACQUISITION_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_ACQUISITION_FAILURE;
                  }
               }
               break;

               case AM_MSG_STOP_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_STOPAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_STOPAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_STOPAUDIO_FAILURE;
                  }
               }
               break;

               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         case AM_VOICERECOGNITION:
         {
            switch (amMessageId)
            {
               case AM_MSG_CHANNEL_ACQUISITION:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_VR_CHANNEL_ACQUISITION_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_VR_CHANNEL_ACQUISITION_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_VR_CHANNEL_ACQUISITION_FAILURE;
                  }
               }
               break;
               case AM_MSG_STOP_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_VR_CHANNEL_STOPAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_VR_CHANNEL_STOPAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_VR_CHANNEL_STOPAUDIO_FAILURE;
                  }
               }
               break;

               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         default:
            eventToControllers = PROPERTY_AND_EVENTID_LAST;
      }
   }

   void PmAudioManagerWrapper::setStreamingStateEvent(const AudioChannel audioChannelId,
         const AmMessageIdEnum amMessageId,
         const AmResponse amResponse, PmCorePropertyAndEventId& eventToControllers)
   {
      ETG_TRACE_USR4(("setStreamingStateEvent"));

      switch(audioChannelId)
      {
         case AM_VEHICLERINGTONE:
         {
            switch (amMessageId)
            {
               case AM_MSG_PLAY_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_RINGTONE_CHANNEL_PLAYAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_RINGTONE_CHANNEL_PLAYAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_RINGTONE_CHANNEL_PLAYAUDIO_FAILURE;
                  }
               }
               break;

               case AM_MSG_PAUSE_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_RINGTONE_CHANNEL_PAUSEAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_RINGTONE_CHANNEL_PAUSEAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_RINGTONE_CHANNEL_PAUSEAUDIO_FAILURE;
                  }
               }
               break;

               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         case AM_PHONEAUDIO:
         {
            switch (amMessageId)
            {
               case AM_MSG_PLAY_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_PHONE_CHANNEL_PLAYAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_PHONE_CHANNEL_PLAYAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_PHONE_CHANNEL_PLAYAUDIO_FAILURE;
                  }
               }
               break;

               case AM_MSG_PAUSE_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_PHONE_CHANNEL_PAUSEAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_PHONE_CHANNEL_PAUSEAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_PHONE_CHANNEL_PAUSEAUDIO_FAILURE;
                  }
               }
               break;

               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         case AM_WAITINGMODE:
         {
            switch (amMessageId)
            {
               case AM_MSG_PLAY_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_PLAYAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_PLAYAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_PLAYAUDIO_FAILURE;
                  }
               }
               break;

               case AM_MSG_PAUSE_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_FAILURE;
                  }
               }
               break;

               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         case AM_VOICERECOGNITION:
         {
            switch (amMessageId)
            {
               case AM_MSG_PLAY_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_VR_CHANNEL_PLAYAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_VR_CHANNEL_PLAYAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_VR_CHANNEL_PLAYAUDIO_FAILURE;
                  }
               }
               break;
               case AM_MSG_PAUSE_AUDIO:
               {
                  switch (amResponse)
                  {
                     case AM_RESPONSE_SUCCESS:
                        eventToControllers = AM_VR_CHANNEL_PAUSEAUDIO_SUCCESS;
                        break;
                     case AM_RESPONSE_FAILURE:
                        eventToControllers = AM_VR_CHANNEL_PAUSEAUDIO_FAILURE;
                        break;
                     default:
                        eventToControllers = AM_VR_CHANNEL_PAUSEAUDIO_FAILURE;
                  }
               }
               break;

               default:
                  eventToControllers = PROPERTY_AND_EVENTID_LAST;
            }
         }
         break;

         default:
            eventToControllers = PROPERTY_AND_EVENTID_LAST;
      }
   }

   bool PmAudioManagerWrapper::isActiveDevice(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("isActiveDevice: deviceAddress: %s", deviceAddress.c_str()));

      if (std::find(_activeDevicesList.begin(), _activeDevicesList.end(), deviceAddress) != _activeDevicesList.end())
      {
         return true;
      }

      return false;
   }

   bool PmAudioManagerWrapper::isPrepareAudioRouteRequestDisposable()
   {
      ETG_TRACE_USR4(("isPrepareAudioRouteRequestDisposable"));

      Locker locker(&_amWrapperResourceLock);
      if (CONTAINER_SIZE_ONE == _preparingAudioRouteJobsList.size())
      {
         return true;
      }
      return false;
   }

   void PmAudioManagerWrapper::insertElementIntoPreparingAudioRouteJobsList(
         const PrepareAudioRouteRequestJob& prepareAudioRouteRequestJob,
         const STLInsertionPosition pos)
   {
      ETG_TRACE_USR4(("insertElementIntoPreparingAudioRouteJobsList: pos: %u", pos));

      Locker locker(&_amWrapperResourceLock);

      if (POS_BACK == pos)
      {
         _preparingAudioRouteJobsList.emplace_back(prepareAudioRouteRequestJob);
      }
      else if (POS_FRONT == pos)
      {
         _preparingAudioRouteJobsList.emplace(_preparingAudioRouteJobsList.begin(), prepareAudioRouteRequestJob);
      }

      printPreparingAudioRouteJobsList();
   }

   void PmAudioManagerWrapper::getRecentJobFromPreparingAudioRouteJobsList(const BdAddress& deviceAddress,
         PrepareAudioRouteRequestJob& recentJob)
   {
      ETG_TRACE_USR4(("getRecentJobFromPreparingAudioRouteJobsList: deviceAddress: %s", deviceAddress.c_str()));

      recentJob = PrepareAudioRouteRequestJob();

      Locker locker(&_amWrapperResourceLock);

      printPreparingAudioRouteJobsList();

      auto iter = std::find_if(_preparingAudioRouteJobsList.rbegin(), _preparingAudioRouteJobsList.rend(),
            [&deviceAddress](PrepareAudioRouteRequestJob const& obj){return obj._deviceAddress == deviceAddress;});

      if (_preparingAudioRouteJobsList.rend() != iter)
      {
         recentJob = *iter;
      }
   }

   void PmAudioManagerWrapper::getFrontElementFromPreparingAudioRouteJobsList(
         PrepareAudioRouteRequestJob& frontEndJob)
   {
      ETG_TRACE_USR4(("getFrontElementFromPreparingAudioRouteJobsList"));

      frontEndJob = PrepareAudioRouteRequestJob();

      Locker locker(&_amWrapperResourceLock);

      printPreparingAudioRouteJobsList();

      if (!_preparingAudioRouteJobsList.empty())
      {
         frontEndJob = _preparingAudioRouteJobsList.front();
      }
   }

   void PmAudioManagerWrapper::removeFrontElementFromPreparingAudioRouteJobsList()
   {
      ETG_TRACE_USR4(("removeFrontElementFromPreparingAudioRouteJobsList"));

      Locker locker(&_amWrapperResourceLock);

      if (!_preparingAudioRouteJobsList.empty())
      {
         _preparingAudioRouteJobsList.erase(_preparingAudioRouteJobsList.begin());
      }
      printPreparingAudioRouteJobsList();
   }

   bool PmAudioManagerWrapper::isPreparingAudioRouteJobsListEmpty()
   {
      ETG_TRACE_USR4(("isPreparingAudioRouteJobsListEmpty"));

      Locker locker(&_amWrapperResourceLock);

      return (_preparingAudioRouteJobsList.empty());
   }

   void PmAudioManagerWrapper::processRequestFromJobsList(const BdAddress& deviceAddress,
         const AmSessionId amSessionId, const AmResult& receivedAmResult)
   {
      ETG_TRACE_USR4(("processRequestFromJobsList"));

      Locker locker(&_amWrapperResourceLock);

      bool processNextDeviceRequest = true;

      if (!_preparingAudioRouteJobsList.empty())
      {
         // Get the recent request for the device address if present
         auto iter = std::find_if(_preparingAudioRouteJobsList.rbegin(), _preparingAudioRouteJobsList.rend(),
               [&deviceAddress](PrepareAudioRouteRequestJob const& obj){return obj._deviceAddress == deviceAddress;});

         if (_preparingAudioRouteJobsList.rend() != iter)
         {
            PrepareAudioRouteRequestJob recentRequestFromSameDevice(*iter);

            // Removing all the other requests in the list received from the same device address
            _preparingAudioRouteJobsList.erase(std::remove_if(_preparingAudioRouteJobsList.begin(),
                  _preparingAudioRouteJobsList.end(),
                  [&deviceAddress](PrepareAudioRouteRequestJob const& obj){return (obj._deviceAddress == deviceAddress);}),
                  _preparingAudioRouteJobsList.end());

            printPreparingAudioRouteJobsList();

            ETG_TRACE_USR4(("Request received for different channel from the same device address"));

            AmResult recentRequestResult(AM_RESULT_OK, "");

            if (PrepareAudioRouteRequestJob::STOP_AUDIO == recentRequestFromSameDevice._operationToPerform)
            {
               // Posting StopAudio request only if the received prepareAudioRoute response is successful.
               if (receivedAmResult._amResultCode == AM_RESULT_OK)
               {
                  // Handling [7(b)(ii)]
                  recentRequestResult = _iPmAudioManagerRequestIf->stopAudio(amSessionId);

                  if (AM_RESULT_OK != recentRequestResult._amResultCode)
                  {
                     sendAmResponse(recentRequestFromSameDevice._deviceAddress, recentRequestResult, AM_MSG_STOP_AUDIO,
                           recentRequestFromSameDevice._controllerOriginEnumType);
                  }
               }
               else
               {
                  // Handling [7(b)(ii)(1)]
                  // PrepareAudioRequest was failure. And now StopAudio need to be processed.
                  // Hence simply update the response as StopAudio SUCCESS.

                  sendAmResponse(deviceAddress, AmResult(AM_RESULT_OK, ""), AM_MSG_STOP_AUDIO,
                        recentRequestFromSameDevice._controllerOriginEnumType);
                  sendAmNotificationEvent(deviceAddress, recentRequestFromSameDevice._audioChannelId,
                        StreamingState(StreamingStateEnumType::STOPPED), AmEventDetails(AUDIO_STATE_CHANGE, ""));
               }
            }
            else
            {
               // Handling [7(b)(ii)(2)]
               recentRequestResult =
                     _iPmAudioManagerRequestIf->prepareAudioRoute(amSessionId, recentRequestFromSameDevice._audioChannelId);

               if (AM_RESULT_OK != recentRequestResult._amResultCode)
               {
                  // If posting the request fails, then the response is simulated from here itself.
                  sendAmResponse(deviceAddress, recentRequestResult, AM_MSG_PREPARE_AUDIO_PROUTE,
                        recentRequestFromSameDevice._controllerOriginEnumType);
               }
               else
               {
                  // The recent request is posted successfully
                  if (receivedAmResult._amResultCode == AM_RESULT_OK)
                  {
                     updateAmSessionIdRequestInfoMap(amSessionId, deviceAddress, recentRequestFromSameDevice._audioChannelId,
                           recentRequestFromSameDevice._controllerOriginEnumType);
                  }
                  else
                  {
                     // The received PrepareaudioResponse failure case
                     // Handling [7(b)(ii)(2)]
                     _preparingAudioRouteJobsList.emplace(_preparingAudioRouteJobsList.begin(), recentRequestFromSameDevice);
                     printPreparingAudioRouteJobsList();

                     // The current request is posted with AM_SESSION_ID_DEFAULT.
                     // Hence the next request should not be posted now.
                     processNextDeviceRequest = false;
                  }
               }
            }
         }

         if (processNextDeviceRequest)
         {
            processNextDeviceRequestFromJobsList();
         }
      }
   }

   void PmAudioManagerWrapper::processNextDeviceRequestFromJobsList()
   {
      ETG_TRACE_USR4(("processNextDeviceRequestFromJobsList"));

      bool requestPosted = false;

      while ((!_preparingAudioRouteJobsList.empty()) && (!requestPosted))
      {
         BdAddress frontEndDeviceAddress = _preparingAudioRouteJobsList.front()._deviceAddress;

         // Retrieving the recent request for that DeviceAddress
         auto iter = std::find_if(_preparingAudioRouteJobsList.rbegin(), _preparingAudioRouteJobsList.rend(),
               [&frontEndDeviceAddress](PrepareAudioRouteRequestJob const& obj)
               {return obj._deviceAddress == frontEndDeviceAddress;});

         if (_preparingAudioRouteJobsList.rend() != iter)
         {
            PrepareAudioRouteRequestJob job(*iter);

            // Removing all the requests in the list received from the same device address
            _preparingAudioRouteJobsList.erase(std::remove_if(_preparingAudioRouteJobsList.begin(),
                  _preparingAudioRouteJobsList.end(),
                  [&frontEndDeviceAddress](PrepareAudioRouteRequestJob const& obj)
                  {return (obj._deviceAddress == frontEndDeviceAddress);}),
                  _preparingAudioRouteJobsList.end());

            if (PrepareAudioRouteRequestJob::STOP_AUDIO == job._operationToPerform)
            {
               // Stop Audio received before processing PrepareAudioRoute request.
               // Hence replying with Success response for StopAudio request.
               sendAmResponse(job._deviceAddress, AmResult(AM_RESULT_OK, ""), AM_MSG_STOP_AUDIO,
                     job._controllerOriginEnumType);
               sendAmNotificationEvent(job._deviceAddress, job._audioChannelId,
                     StreamingState(StreamingStateEnumType::STOPPED), AmEventDetails(AUDIO_STATE_CHANGE, ""));
            }
            else
            {
               AmResult sendPrepareAudioRouteResult =
                     _iPmAudioManagerRequestIf->prepareAudioRoute(AM_SESSION_ID_DEFAULT, job._audioChannelId);

               // Responding with the Error response if Error is received when posting the request
               if (AM_RESULT_OK != sendPrepareAudioRouteResult._amResultCode)
               {
                  sendAmResponse(job._deviceAddress, sendPrepareAudioRouteResult, AM_MSG_PREPARE_AUDIO_PROUTE,
                        job._controllerOriginEnumType);
               }
               else
               {
                  requestPosted = true;

                  // Inserting this job into the front of the list since the request is posted successfully.
                  _preparingAudioRouteJobsList.emplace(_preparingAudioRouteJobsList.begin(), job);
               }
            }
         }

         printPreparingAudioRouteJobsList();
      }
   }

   void PmAudioManagerWrapper::swapDeviceAddress(const BdAddress replaceDeviceAddress,
         const BdAddress deviceAddress, const ControllerOriginEnum controllerOriginEnum)
   {
      ETG_TRACE_USR4(("swapDeviceAddress entered with replaceDeviceAddress: %20s and deviceAddress: %20s",
            replaceDeviceAddress.c_str(), deviceAddress.c_str()));

      printAmSessionIdRequestInfoMap();

      auto iter = std::find_if(_amSessionIdRequestInfoMap.begin(), _amSessionIdRequestInfoMap.end(),
            [&replaceDeviceAddress](std::pair<AmSessionId, RequestInfo> const& obj)
            {return obj.second._deviceAddress == replaceDeviceAddress;});

      if (_amSessionIdRequestInfoMap.end() != iter)
      {
         iter->second._deviceAddress = deviceAddress;
         iter->second._controllerOriginEnumType = controllerOriginEnum;

         // Before replacing, the corresponding SM should be updated with STOP_AUDIO_SUCCESS event
         // so that it shall be moved to IDLE state.
         sendAmNotificationEvent(replaceDeviceAddress, AM_PHONEAUDIO, StreamingState(StreamingStateEnumType::STOPPED),
               AmEventDetails());
      }
   }

} /* namespace pmcore */
