/**
 * @file PmAudioManagerWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the declaration of the PmAudioManagerWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the wrapper interface to PM AudioManager in PM Core
 *
 * The responsibilities of this class are listed below:
 * 1. The Audio Activation requests are processed only for the Active devices.
 *    If the request is received for a Passive device, then
 *    - the result is updated as success and the Response message and the amNotification event are simulated
 *      and updated for that Passive device.
 *
 * 2. When a Vehicle ringtone is playing and if an Active device requests for Audio Activation of the
 *    Vehicle Ringtone, then
 *       (a)   the request is posted to PM AudioManager which will take care of stopping the current
 *             ringtone file and playing the received filePath.
 *             Also it will take care of updating the Am Session Ids.
 *
 * 3. When the vehicle ringtone is playing for a particular Active device and if an other Active
 *    device is requesting for VEHICLE RINGTONE,
 *       (a)   If multiple Ringtone channels are supported, then this code needs to be revisited
 *             and adapted accordingly.
 *
 * 4. When the vehicle ringtone is playing for a particular Active device and if the same device
 *    is requesting for an other channel say PHONE channel, then the same Audio session Id is
 *    used for this communication also.
 *    (PM_Audiomanager will take care of releasing the VEHICLE RINGTONE channel and the final status
 *    of the request is responded with the amNotificationEvent signal)
 *
 * 5. When the vehicle ringtone is playing for a particular Active device and if an other Active device is
 *    requesting for an other channel say PHONE channel,
 *       (a)   If multiple channels are supported, then this code needs to be revisited
 *             and adapted accordingly.
 *
 * 6. If a PrepareAudioRoute request is received for a device
 *       (a)   for the first time, the request job is pushed into
 *             "_preparingAudioRouteJobsList" vector and removed when the corresponding response is received.
 *       (b)   after a successful acquisition of a channel,
 *             then the valid AmSessionId corresponding to this device, will be used to post this request
 *             to PM Audio-manager.
 *
 * 7.  The below cases are considered when PM Audio-manager and PM Core are running in a different thread.
 *       (a)   When PM Core is waiting for PrepareAudioRoute response from PM Audiomanager and if
 *             PM AudioManager Wrapper receives another PrepareAudioRoute from the same device for the same
 *             Audio channel Id, then the request is simply returned with Success and the response and
 *             amNotifiaction event will be sent on receiving the PrepareAudioRoute response of the previously
 *             posted request.
 *
 *       (b)   When PM Core is waiting for PrepareAudioRoute response from PM Audiomanager and if
 *             PM AudioManager Wrapper receives another PrepareAudioRoute for different Audio channel Id
 *             or StopAudio request from the same device, then
 *             (i)   that request is also pushed into the "_preparingAudioRouteJobsList" vector.
 *             (ii)  On receiving the PrepareAudioRoute response for the prior request, the recent request is
 *                   posted to PM Audiomanager and all the requests corresponding to this device address are
 *                   removed from the "_preparingAudioRouteJobsList" vector.
 *                   If the received response is FAILURE one, and if the recent request is
 *                   (1) StopAudio request, then StopAudio response is sent as success immediately
 *                         and the corresponding amNotificationEvent is also simulated
 *                         since prepareAudioRoute itself is a failure one.
 *                   (2) PrepareAudioRoute request for an other channel, then that recent request is posted
 *                         to PM AudioManager with AmSessionId as "0".
 *                         And this element will be emplaced at the front of the "_preparingAudioRouteJobsList"
 *                         vector.
 *
 *       (c)   When PM Core is waiting for PrepareAudioRoute response from PM Audiomanager and if
 *             PM AudioManager Wrapper receives another request from a different device, then
 *             (i)   that request is also pushed back into the "_preparingAudioRouteJobsList" vector.
 *             (ii)  And will be processed only on receiving the response for the prior device request.
 *                   On processing the response of the prior request, if the recent request from other device is-
 *                   (1) StopAudio request, then StopAudio response is sent as success immediately
 *                         and the corresponding amNotificationEvent is also simulated
 *                         since prepareAudioRoute itself is not yet processed for that device.
 *                   (2) PrepareAudioRoute request, then that recent request is posted
 *                         to PM AudioManager with AmSessionId as "0"
 *
 * @ingroup PmCore
 */

#ifndef PmAudioManagerWrapper_h
#define PmAudioManagerWrapper_h

#include "PmSingleton.h"
#include "PmCoreIfTypes.h"
#include "IPmAudioManagerCallbackIf.h"
#include "IPmAudioManagerRequestIf.h"
#include "PmAudioManagerIfTypes.h"
#include "Lock.h"

#define STOPAUDIO_EVENT_WITH_DELIMITER "StopAudio-"
#define CHANNEL_ACQUISITION_EVENT_WITH_DELIMITER "ChannelAcquisition-"

#define STREAMING_STOPPED_DUE_TO_EXT_TRIGGER "streaming stopped due to external trigger"
#define STREAMING_PAUSED_DUE_TO_EXT_TRIGGER "streaming paused due to external trigger"

namespace pmcore
{

   /*
    * Type of Am messages received in PM AudioManager
    */
   typedef enum
   {
      AM_MSG_PREPARE_AUDIO_PROUTE = 0UL,
      AM_MSG_CHANNEL_ACQUISITION,
      AM_MSG_PLAY_AUDIO,
      AM_MSG_STOP_AUDIO,
      AM_MSG_PAUSE_AUDIO,
      AM_MSG_SET_MIC_MUTE_STATE,
      AM_MSG_SET_MIC_UNMUTE,
      AM_MSG_SET_MIC_MUTE,
      AM_MSG_GET_AUDIO_SINK,
      AM_MSG_SET_AUDIO_SINK
   }AmMessageIdEnum;

   /**
    * Am Response message Enum
    */
   typedef enum
   {
      AM_RESPONSE_SUCCESS = 0x00u, /**< request was successful */
      AM_RESPONSE_FAILURE,          /**< request/operation failed */

      AM_RESPONSE_LAST             /**< end marker, must be the last one */
   }AmResponse;

   /**
    * Am Response message Enum
    */
   typedef enum
   {
      POS_FRONT = 0x00u, // Element will inserted into the front of the STL
      POS_BACK,          // Element will inserted at the back of the STL
   }STLInsertionPosition;

   /*
    * PrepareAudioRoute request information are stored in this class
    */
   class PrepareAudioRouteRequestJob
   {
   public:
      typedef enum {
         PREPARE_AUDIO_ROUTE, // Default value
         STOP_AUDIO           // Set when the StopAudio request received before receiving PrepareAudioRouteResponse
      }OperationToPerform;

      PrepareAudioRouteRequestJob() : _deviceAddress(""), _audioChannelId(pmaudiomanager::AM_UNKNOWN),
            _controllerOriginEnumType(MAIN_CONTROLLER), _operationToPerform(PREPARE_AUDIO_ROUTE)
      {

      }

      PrepareAudioRouteRequestJob(BdAddress deviceAddress, pmaudiomanager::AudioChannel audioChannelId,
            ControllerOriginEnum controllerOriginEnumType, OperationToPerform operationToPerform) :
               _deviceAddress(deviceAddress), _audioChannelId(audioChannelId),
               _controllerOriginEnumType(controllerOriginEnumType), _operationToPerform(operationToPerform)
      {

      }

      PrepareAudioRouteRequestJob(const PrepareAudioRouteRequestJob& ref) :
         _deviceAddress(ref._deviceAddress), _audioChannelId(ref._audioChannelId),
         _controllerOriginEnumType(ref._controllerOriginEnumType),
         _operationToPerform(ref._operationToPerform)
      {

      }

      friend void swap(PrepareAudioRouteRequestJob& first, PrepareAudioRouteRequestJob& second)
      {
         using std::swap;

         swap(first._deviceAddress, second._deviceAddress);
         swap(first._audioChannelId, second._audioChannelId);
         swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
         swap(first._operationToPerform, second._operationToPerform);
      }

      //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
      PrepareAudioRouteRequestJob& operator=(PrepareAudioRouteRequestJob ref)
      {
         swap(*this, ref);
         return *this;
      }

      bool operator==(const PrepareAudioRouteRequestJob& rhs) const
         {
         return ((this->_deviceAddress == rhs._deviceAddress)
               && (this->_audioChannelId == rhs._audioChannelId)
               && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType)
               && (this->_operationToPerform == rhs._operationToPerform));
         }

      BdAddress _deviceAddress;
      pmaudiomanager::AudioChannel _audioChannelId;
      ControllerOriginEnum _controllerOriginEnumType;
      OperationToPerform _operationToPerform;
   };

   /**
    * DevAddressSessionIdCtrlOrigin class definition
    */
   class RequestInfo
   {
   public:
      RequestInfo() : _deviceAddress(""), _audioChannelId(pmaudiomanager::AM_UNKNOWN),
      _controllerOriginEnumType(MAIN_CONTROLLER)
   {

   }

      RequestInfo(BdAddress deviceAddress, pmaudiomanager::AudioChannel audioChannelId,
            ControllerOriginEnum controllerOriginEnumType) :
               _deviceAddress(deviceAddress), _audioChannelId(audioChannelId),
               _controllerOriginEnumType(controllerOriginEnumType)
      {

      }

      RequestInfo(const RequestInfo& ref) :
         _deviceAddress(ref._deviceAddress), _audioChannelId(ref._audioChannelId),
         _controllerOriginEnumType(ref._controllerOriginEnumType)
      {

      }

      friend void swap(RequestInfo& first, RequestInfo& second)
      {
         using std::swap;

         swap(first._deviceAddress, second._deviceAddress);
         swap(first._audioChannelId, second._audioChannelId);
         swap(first._controllerOriginEnumType, second._controllerOriginEnumType);
      }

      //lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
      RequestInfo& operator=(RequestInfo ref)
      {
         swap(*this, ref);
         return *this;
      }

      bool operator==(const RequestInfo& rhs) const
         {
         return ((this->_deviceAddress == rhs._deviceAddress)
               && (this->_audioChannelId == rhs._audioChannelId)
               && (this->_controllerOriginEnumType == rhs._controllerOriginEnumType));
         }

      BdAddress _deviceAddress;
      pmaudiomanager::AudioChannel _audioChannelId;
      ControllerOriginEnum _controllerOriginEnumType;
   };

   /**
    * PmAudioManagerWrapper class definition
    */
   class PmAudioManagerWrapper : public PmSingleton <PmAudioManagerWrapper>,
   public pmaudiomanager::IPmAudioManagerCallbackIf
   {
   public:
      /**
       * This interface is used to instantiate and initialize PM Audiomanager and to register callback interface with
       * PM Audiomanager.
       * If the AmResult is successful, then it implies that the PM_Audiomanager is active and
       * accessible.
       * If the AmResult is failure, it implies that the PM_Audiomanager is not ready and
       * should not be accessed.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      pmaudiomanager::AmResult preparePmAudioManager(ahl_tclBaseOneThreadApp* mainApplication);

      /**
       * THis interface is used to set the state of the PM Audiomanager
       *
       * @param[in] AmState amState- Provides the SPM system state / BT System state to AM
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void sendSetAmState(IN const pmaudiomanager::AmState amState);

      /**
       * This interface is used to prepare/acquire the audio channel before the actual Audio streaming shall be started.
       *
       * @param[in]  BdAddress deviceAddress - Bluetooth address of the device
       *             AudioChannel audioChannelId - Required Audio channel Id
       *             ControllerOriginEnum controllerOriginEnumType - Origin of the controller
       * @param[out]
       * @param[in,out]
       *
       * @return AmResult- Result of the operation
       */
      pmaudiomanager::AmResult sendPrepareAudioRoute(IN const BdAddress& deviceAddress,
            IN const pmaudiomanager::AudioChannel audioChannelId,
            IN const ControllerOriginEnum controllerOriginEnumType);

      /**
       * This interface is used to play the audio from the specified file path for the given number of times.
       *
       * @param[in]  BdAddress deviceAddress - Bluetooth address of the device
       *             FilePath - Valid file path of the audio file that has to be played
       *             ControllerOriginEnum controllerOriginEnumType - Origin of the controller
       *             PlayCount- Denotes the number of times the file has to be played.
       *                This parameter shall be set to FF (Play repeatedly)
       *                - when the vehicle ringtone for an incoming call has to be played [OR]
       *                - when the waiting mode tone has to be played
       * @param[out]
       * @param[in,out]
       *
       * @return AmResult- Result of the operation
       */
      pmaudiomanager::AmResult sendPlayAudio(IN const BdAddress& deviceAddress, IN const FilePath& filePath,
            IN const ControllerOriginEnum controllerOriginEnumType,
            IN const PlayCount playCount = DEFAULT_FILE_PLAY_COUNT,
            IN const pmaudiomanager::ToneType toneType = pmaudiomanager::RING_TONE);

      /**
       * This interface is used to acquire audio channel and then to start the audio streaming.
       *
       * @param[in]  BdAddress deviceAddress - Bluetooth address of the device
       *             AudioChannel - ID of the audio channel to be acquired to start the audio streaming.
       *             SCOStatus - Denotes the SCO type.
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      pmaudiomanager::AmResult sendPlayAudio(IN const BdAddress& deviceAddress,
            IN const SCOStatus scoStatus);

      /**
       * This interface is used to release the acquired audio channel and will stop the audio streaming if streaming.
       *
       * @param[in]  BdAddress deviceAddress - Bluetooth address of the device
       *             ControllerOriginEnum controllerOriginEnumType - Origin of the controller
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      pmaudiomanager::AmResult sendStopAudio(IN const BdAddress& deviceAddress,
            IN const ControllerOriginEnum controllerOriginEnumType);

      /**
       * This interface is used to mute/de-mute the microphone of the IVI system.
       *
       * @param[in]  MuteState muteState - MUte state operation to be performed.
       *               0x00 - unmute, 0x01 - mute
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      pmaudiomanager::AmResult sendSetMicMuteState(IN const MuteState muteState);

      /**
       * This interface is used to pause the audio streaming in the corresponding audio channel.
       *
       * @param[in] BdAddress deviceAddress - Bluetooth address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      pmaudiomanager::AmResult sendPauseAudio(IN const BdAddress& deviceAddress);

      /**
       * This interface is used to get the audio sink volume from Genivi Audio Manager.
       *
       * @param[in] BdAddress deviceAddress - Bluetooth address of the device
       * @param[in] GeniviAMSinkId sinkId - sink ID
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      pmaudiomanager::AmResult getAudioSinkVolume(IN const BdAddress& deviceAddress, IN const pmaudiomanager::GeniviAMSinkId sinkId);

      /**
       * This interface is used to set the audio sink volume from Genivi Audio Manager.
       *
       * @param[in] BdAddress deviceAddress - Bluetooth address of the device
       * @param[in] GeniviAMSinkId sinkId - sink ID
       * @param[in] AmPropertyType2VolumeMap propertyVal - Map of AmProperty to volume
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      pmaudiomanager::AmResult setAudioSinkVolume(IN const BdAddress& deviceAddress, IN const pmaudiomanager::GeniviAMSinkId sinkId,
            IN const pmaudiomanager::AmPropertyType2VolumeMap propertyVal);

      /**
       * Overriden from IPmAudioManagerCallbackIf class
       *
       * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
       *                if the request is successful, then the new session ID will be created by PM AM
       *                for the corresponding audio session and will update it.
       *            AmResult - Result of the operation
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void prepareAudioRouteResponse(IN const pmaudiomanager::AmSessionId amSessionId,
            IN const pmaudiomanager::AmResult amResult) override;

      /**
       * Overriden from IPmAudioManagerCallbackIf class
       * Response interface for the playAudio request
       *
       * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
       *                if the request is successful, then the new session ID will be created by PM AM
       *                for the corresponding audio session and will update it.
       *            AmResult - Result of the operation
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void playAudioResponse(IN const pmaudiomanager::AmSessionId amSessionId,
            IN const pmaudiomanager::AmResult amResult) override;

      /**
       * Overriden from IPmAudioManagerCallbackIf class
       * Response for stopAudio interface.
       * Response will be sent as success once the request is considered for processing.
       * AmSessionId will become invalid once the stopAudio is processed
       * i.e. when the amNotificationEvent is sent with the StreamingState set as STOPPED
       *
       * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
       *                if the request is successful, then the new session ID will be created by PM AM
       *                for the corresponding audio session and will update it.
       *            AmResult - Result of the operation
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void stopAudioResponse(IN const pmaudiomanager::AmSessionId amSessionId,
            IN const pmaudiomanager::AmResult amResult) override;

      /**
       * Overriden from IPmAudioManagerCallbackIf class
       * This API is used to notify:
       *    1. Audio streaming / playing status
       *    2. Error response when the acquisition (or) release of AudioChannel failed.
       *    3. Update when the audio is stopped / paused due to a trigger from ARL (Audio Routing Library)
       *
       * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
       *                if the request is successful, then the new session ID will be created by PM AM
       *                for the corresponding audio session and will update it.
       *            AudioChannel - Prepare the audio route for the requested audio channel.
       *            StreamingState - Gives the Streaming state for the corresponding Audio session
       *            AmEventDetails - Gives the details of the event
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void amNotificationEvent(IN const pmaudiomanager::AmSessionId amSessionId,
            IN const pmaudiomanager::AudioChannel audioChannelId,
            IN const pmaudiomanager::StreamingState streamingState,
            IN const pmaudiomanager::AmEventDetails amEventDetails) override;

      /**
       * Overriden from IPmAudioManagerCallbackIf class
       * Response for setMicMuteState interface. The final Mute state is sent through the amNotificationEvent method.
       *
       * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
       *                if the request is successful, then the new session ID will be created by PM AM
       *                for the corresponding audio session and will update it.
       *            AmResult - Result of the operation
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void setMicMuteStateResponse(IN const pmaudiomanager::AmSessionId amSessionId,
            IN const pmaudiomanager::AmResult amResult) override;

      /**
       * Overriden from IPmAudioManagerCallbackIf class
       * Response for pauseAudio interface.
       * Response will be sent as success once the request is considered for processing.
       * AmSessionId will become invalid once the stopAudio is processed
       * i.e. when the amNotificationEvent is sent with the StreamingState set as NOT_PLAYING
       *
       * @param[in] AmSessionId - If the corresponding request was called with AmSessionId as "0" and
       *                if the request is successful, then the new session ID will be created by PM AM
       *                for the corresponding audio session and will update it.
       *            AmResult - Result of the operation
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual void pauseAudioResponse(IN const pmaudiomanager::AmSessionId amSessionId,
            IN const pmaudiomanager::AmResult amResult) override;

      /**
       * Response for get the audio sink volume from Genivi Audio Manager.
       *
       * @param[in] amSessionId - Session Id of the request
       * @param[in] amResult - Result of the operation
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      virtual void getAudioSinkVolumeResponse(IN const pmaudiomanager::AmSessionId amSessionId, IN const pmaudiomanager::AmResult amResult) override;

      /**
       * Response for set the audio sink volume from Genivi Audio Manager.
       *
       * @param[in] amSessionId - Session Id of the request
       * @param[in] amResult - Result of the operation
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      virtual void setAudioSinkVolumeResponse(IN const pmaudiomanager::AmSessionId amSessionId, IN const pmaudiomanager::AmResult amResult) override;

      /**
       * Property update from Genivi Audio Manager for set and get of volume properties.
       *
       * @param[in] amSessionId - Session Id of the request
       * @param[in] amResult - Result of the operation
       * @param[in] sinkId - Sink Id
       * @param[in] propertyList - Map of AmProperty to volume
       * @param[out]
       * @param[in,out]
       *
       * @return pmaudiomanager::AmResult- Result of the operation
       */
      virtual void updateAudioSinkVolumeList(IN const pmaudiomanager::AmSessionId amSessionId, IN const pmaudiomanager::AmResult amResult,
               IN const pmaudiomanager::GeniviAMSinkId sinkId, IN const pmaudiomanager::AmPropertyType2VolumeMap propertyList) override;

      /**
       * This function is called by PM Core Main controller to set the list of Active devices.
       * Because the Audio activation request shall be processed only for the active devices.
       *
       * @param[in] BdAddressList bdAddressList - List of addresses of active devices
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void setActiveDevicesList(IN const BdAddressList activeDevicesList);

      /**
       * This function is sued to alter the device address in the "_amSessionIdRequestInfoMap"
       * to the given device address
       *
       * @param[in]  replaceDeviceAddress - device address which should be altered
       *             deviceAddress - device address to be altered with
       *             controllerOriginEnum - Controller origin of the new device address to be altered
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void swapDeviceAddress(const BdAddress replaceDeviceAddress, const BdAddress deviceAddress,
            const ControllerOriginEnum controllerOriginEnum);

   private:
      friend class PmSingleton<PmAudioManagerWrapper>;

      /**
       * Constructor of PmAudioManagerWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmAudioManagerWrapper();

      /**
       * Destructor of PmAudioManagerWrapper class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~PmAudioManagerWrapper();

      /**
       * This method is used to send the PM Core If message from the response of PM Audio manager after
       * retrieving the ControllerOriginEnumType.
       *
       * @param[in] BdAddress deviceAddress - Address of the device requesting for audio activation/deactivation.
       *            AmSessionId amSessionId - Audio session ID
       *            AmMessageIdEnum - Response type received from PM Audiomanager
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void sendAmResponse(const BdAddress& deviceAddress, const pmaudiomanager::AmResult& amResult,
            const AmMessageIdEnum amMessageId);

      /**
       * This method is used to create and send the PM Core If message from the response of PM Audio manager
       *
       * @param[in] BdAddress deviceAddress - Address of the device requesting for audio activation/deactivation.
       *            AmSessionId amSessionId - Audio session ID
       *            AmMessageIdEnum - Response type received from PM Audiomanager
       *            ControllerOriginEnum controllerOriginEnumType - Origin of the controller
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void sendAmResponse(const BdAddress& deviceAddress, const pmaudiomanager::AmResult& amResult,
            const AmMessageIdEnum amMessageId, ControllerOriginEnum controllerOriginEnumType);

      /**
       * This method is used to create and send the PM Core If message from the AmNotificationEvent of PM Audio manager
       *
       * @param[in] BdAddress deviceAddress - Address of the device requesting for audio activation/deactivation.
       *            AudioChannel audioChannelId - Id of the audio channel
       *            StreamingState streamingState - Current Streaming state of the channel
       *            AmEventDetails eventInfo - Information about the event
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void sendAmNotificationEvent(const BdAddress& deviceAddress,
            const pmaudiomanager::AudioChannel audioChannelId,
            const pmaudiomanager::StreamingState streamingState, const pmaudiomanager::AmEventDetails amEventDetails);

      /**
       * This method is used to remove the element from the STL "_amSessionIdRequestInfoMap"
       * based on the given AmSessionId
       *
       * @param[in]  pmaudiomanager::AmSessionId - AmSession Id key value whose element need to be deleted.
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void eraseAmSessionIdRequestInfoMapElement(const pmaudiomanager::AmSessionId amSessionId);

      /**
       * This method is used to get the device address from the received AmSession Id.
       *
       * @param[in] AmSessionId amSessionId - Audio Session Id for which the device address is needed
       * @param[out] BdAddress& deviceAddress - Ref to Device address
       * @param[in,out]
       *
       * @return
       */
      void getDeviceAddress(const pmaudiomanager::AmSessionId amSessionId, BdAddress& deviceAddress);

      /**
       * This method is used to get the Audio Session ID from the received device address
       *
       * @param[in] BdAddress& deviceAddress - Bluetooth device address
       * @param[out] AmSessionId& amSessionId - Ref to audio Session Id
       * @param[in,out]
       *
       * @return
       */
      void getAmSessionId(const BdAddress& deviceAddress, pmaudiomanager::AmSessionId& amSessionId);

      /**
       * This method is used to get the Controller origin info from the received device address
       *
       * @param[in] BdAddress& deviceAddress - Bluetooth device address
       * @param[out] ControllerOriginEnum& controllerOriginEnumType - Ref to ControllerOrigin info
       * @param[in,out]
       *
       * @return
       */
      void getControllerOrigin(const BdAddress& deviceAddress, ControllerOriginEnum& controllerOriginEnumType);

      /**
       * This method is used to insert element into the container "_amSessionIdRequestInfoMap"
       * or to update the details.
       *
       * @param[in]  AmSessionId& amSessionId - Ref to audio Session Id
       *             BdAddress& deviceAddress - Bluetooth device address
       *             pmaudiomanager::AudioChannel audioChannelId - Audio channel Id
       *             ControllerOriginEnum controllerOriginEnumType - Origin of the controller
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void updateAmSessionIdRequestInfoMap(pmaudiomanager::AmSessionId amSessionId, BdAddress deviceAddress,
            pmaudiomanager::AudioChannel audioChannelId, ControllerOriginEnum controllerOriginEnumType);

      /**
       * This method is used to print the STL "_amSessionIdRequestInfoMap"
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void printAmSessionIdRequestInfoMap();

      /**
       * This method is used to get the recent job present in the STL
       * "_preparingAudioRouteJobsList" for the given device address
       *
       * @param[in]  BdAddress& deviceAddress - Device address for which the recent job is needed
       * @param[out] PrepareAudioRouteRequestJob& recentJob - Recent job in the STL
       * @param[in,out]
       *
       * @return
       */
      void getRecentJobFromPreparingAudioRouteJobsList(const BdAddress& deviceAddress,
            PrepareAudioRouteRequestJob& recentJob);

      /**
       * This method is used to insert an element into "_preparingAudioRouteJobsList" at the
       * given position in the STL
       *
       * @param[in]  PrepareAudioRouteRequestJob& prepareAudioRouteRequestJob - element to be inserted
       *             STLInsertionPosition pos - Position at the STL where the element should be inserted
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void insertElementIntoPreparingAudioRouteJobsList(const PrepareAudioRouteRequestJob& prepareAudioRouteRequestJob,
            const STLInsertionPosition pos = POS_BACK);

      /**
       * This method is used to know whether the STL "_preparingAudioRouteJobsList" is empty
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return TRUE- STL is empty, FALSE- STL is not empty
       */
      bool isPreparingAudioRouteJobsListEmpty();

      /**
       * This method is used to remove the front element from the STL "_preparingAudioRouteJobsList"
       * STL "_preparingAudioRouteJobsList"
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void removeFrontElementFromPreparingAudioRouteJobsList();

      /**
       * This method is used to get the front element from the STL "_preparingAudioRouteJobsList"
       *
       * @param[in]
       * @param[out] PrepareAudioRouteRequestJob& frontEndJob - Reference Job to be filled
       * @param[in,out]
       *
       * @return
       */
      void getFrontElementFromPreparingAudioRouteJobsList(
            PrepareAudioRouteRequestJob& frontEndJob);

      /**
       * This method is used to know whether the prepareAudioRouteRequest shall be posted to
       * PM Audiomanager. The request shall be posted only if there is only one element in the
       * STL "_preparingAudioRouteJobsList"
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return TRUE- Shall post the request to PM Audiomanager, FALSE- Should not post the request now.
       */
      bool isPrepareAudioRouteRequestDisposable();

      /**
       * This method is used to process the next device's request from the STL "_preparingAudioRouteJobsList"
       * if present
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void processNextDeviceRequestFromJobsList();

      /**
       * This method is used to process the request from the STL "_preparingAudioRouteJobsList" corresponding
       * to the received device address
       *
       * @param[in]  BdAddress& deviceAddress - DeviceAddress
       *             pmaudiomanager::AmSessionId amSessionId - AmSessionId to be used when posting the next request
       *             pmaudiomanager::AmResult receivedAmResult - Result of the recent prepareAudioRoute request
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void processRequestFromJobsList(const BdAddress& deviceAddress, const pmaudiomanager::AmSessionId amSessionId,
            const pmaudiomanager::AmResult& receivedAmResult);

      /**
       * This method is used to print the vector "_preparingAudioRouteJobsList"
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void printPreparingAudioRouteJobsList();

      /**
       * This method is used to know whether the given device address is an Active device
       *
       * @param[in] BdAddress& deviceAddress - Bluetooth device address
       * @param[out]
       * @param[in,out]
       *
       * @return Bool - Returns TRUE if the device is Active, else False
       */
      bool isActiveDevice(const BdAddress& deviceAddress);

      /**
       * This method is used to set the Event Id with the given AudioChannelId and its corresponding acquisition
       * state and response message. The events related to Audio channel Acquisition and release are handled
       * in this function
       *
       * @param[in]  pmaudiomanager::AudioChannel& audioChannelId - Audio channel Id
       *             AmMessageIdEnum& amMessageId - Type of response message
       *             AmResponse& amResponse - SUCCESS or FAILURE response from PM Audiomanager
       * @param[out] PmCorePropertyAndEventId& eventToControllers - Event Id
       * @param[in,out]
       *
       * @return
       */
      void setAudioChannelEvent(const pmaudiomanager::AudioChannel audioChannelId,
            const AmMessageIdEnum amMessageId, const AmResponse amResponse,
            PmCorePropertyAndEventId& eventToControllers);

      /**
       * This method is used to set the Event Id with the given AudioChannelId and its corresponding streaming
       * state and response message. The events related to streaming state such as play audio and pause audio
       * are handled in this function
       *
       * @param[in]  pmaudiomanager::AudioChannel& audioChannelId - Audio channel Id
       *             AmMessageIdEnum& amMessageId - Type of response message
       *             AmResponse& amResponse - SUCCESS or FAILURE response from PM Audiomanager
       * @param[out] PmCorePropertyAndEventId& eventToControllers - Event Id
       * @param[in,out]
       *
       * @return
       */
      void setStreamingStateEvent(const pmaudiomanager::AudioChannel audioChannelId,
            const AmMessageIdEnum amMessageId, const AmResponse amResponse,
            PmCorePropertyAndEventId& eventToControllers);

      /**
       * This method is used to set the AmMessageId based on the failedEvent string
       *
       * @param[in]  std::string& failedEvent - Name of the Failed event
       * @param[out] AmMessageIdEnum& amMessageId - Type of response message
       * @param[in,out]
       *
       * @return
       */
      void setAmMessageIdEnum(const std::string& failedEvent, AmMessageIdEnum& amMessageId);

      // Lock for _amSessionIdRequestInfoMap and _preparingAudioRouteJobsList
      LockForever _amWrapperResourceLock;

      // Pointer to PM AudioManager reaquest interface
      pmaudiomanager::IPmAudioManagerRequestIf* _iPmAudioManagerRequestIf;

      /* This STL is used to store the session ID, its corresponding device address and
       * and the corresponding controller origin info
       */
      static std::map<pmaudiomanager::AmSessionId, RequestInfo> _amSessionIdRequestInfoMap;

      /* Active devices list is maintained so that the Audio Activation request shall be
       * processed only for the Active devices.
       */
      static BdAddressList _activeDevicesList;

      /*
       * When sendPrepareAudioRoute request is received from a state machine, the corresponding request job is
       * added to this list and then the request is posted to PM Audiomanager.
       * The request is posted to PM Audiomanager only when the size of this list is equal to one
       * after pushing the request.
       *
       * And when the next request is received, it is also added to this list. But not to the PM Audiomanager.
       * On receiving the response, the first element is retrieved and added to _amSessionIdRequestInfoMap
       * with the received AmSession Id and the corresponding device address and then the
       * corresponding element is removed from the list.
       * And now the next earliest element is retrieved from this list and the corresponding request is
       * posted to PM Audiomanager.
       */
      static std::vector<PrepareAudioRouteRequestJob> _preparingAudioRouteJobsList;
   };
}

#endif //PmAudioManagerWrapper_h
