/**
 * @file PmCoreMainController.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmCoreMainController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmCore
 */

#ifndef PmCoreMainController_h
#define PmCoreMainController_h

#include "PmSingleton.h"
#include "PmCoreIfTypes.h"
#include "PmSettings.h"
#include "DeviceInfoHandler.h"
#include "CallController.h"
#include "VRController.h"
#include "RingtoneController.h"
#include "RVController.h"
#include "PmCoreCallbackIfWrapper.h"

namespace pmcore
{
   class PmCoreIfMessage;

   //Generic methods
   class PmCoreIfMessage_SetSystemState;
   class PmCoreIfMessage_SetVehicleConfiguration;
   class PmCoreIfMessage_OnDevicePaired;
   class PmCoreIfMessage_OnDeviceConnected;
   class PmCoreIfMessage_OnDeviceDisconnected;
   class PmCoreIfMessage_OnDeviceDeleted;

   //Requests
   class PmCoreIfMessage_GetSystemWideRingtoneListRequest;
   class PmCoreIfMessage_GetDeviceRingtoneListRequest;
   class PmCoreIfMessage_SetRingtoneRequest;
   class PmCoreIfMessage_PlayRingtoneRequest;
   class PmCoreIfMessage_StopRingtoneRequest;
   class PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest;
   class PmCoreIfMessage_SuppressRingtoneOnOffRequest;
   class PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest;
   class PmCoreIfMessage_AutoWaitingModeOnOffRequest;
   class PmCoreIfMessage_SetWaitingModeFilePathRequest;
   class PmCoreIfMessage_GetWaitingModeFilePathRequest;
   class PmCoreIfMessage_GetPhoneCallAudioActiveRequest;
   class PmCoreIfMessage_GetActivePassiveDeviceListRequest;
   class PmCoreIfMessage_SetActivePhoneRequest;
   class PmCoreIfMessage_RestoreDefaultSettingRequest;
   class PmCoreIfMessage_GetSupportedFeaturesRequest;
   class PmCoreIfMessage_GetNetworkStatusRequest;
   class PmCoreIfMessage_GetSignalStrengthRequest;
   class PmCoreIfMessage_GetBatteryChargeLevelRequest;
   class PmCoreIfMessage_GetNetworkOperatorRequest;
   class PmCoreIfMessage_VoiceMailListRequest;
   class PmCoreIfMessage_GetCallStatusListRequest;
   class PmCoreIfMessage_GetMicrophoneMuteStateRequest;
   class PmCoreIfMessage_SetMicrophoneMuteStateRequest;
   class PmCoreIfMessage_MergeCallsRequest;
   class PmCoreIfMessage_SplitCallsRequest;
   class PmCoreIfMessage_HangupCallsRequest;
   class PmCoreIfMessage_RedialRequest;
   class PmCoreIfMessage_GetSCOConnectionRequest;
   class PmCoreIfMessage_GetRingtoneMuteStateRequest;
   class PmCoreIfMessage_SetRingtoneMuteStateRequest;
   class PmCoreIfMessage_GetWaitingModeStateRequest;
   class PmCoreIfMessage_StartStopWaitingModeRequest;
   class PmCoreIfMessage_DialRequest;
   class PmCoreIfMessage_AcceptCallRequest;
   class PmCoreIfMessage_SwapCallRequest;
   class PmCoreIfMessage_SpeedDialRequest;
   class PmCoreIfMessage_SendDTMFRequest;
   class PmCoreIfMessage_TransferAudioRequest;
   class PmCoreIfMessage_GetVoiceRecognitionStatusRequest;
   class PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest;
   class PmCoreIfMessage_StartStopVoiceRecognitionRequest;
   class PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest;
   class PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest;
   class PmCoreIfMessage_stopExtendedVRRequest;

   //Results
   class PmCoreIfMessage_RequestPhoneNumberResult;
   class PmCoreIfMessage_GetSubscriberResult;
   class PmCoreIfMessage_SendCustomATCommandResult;
   class PmCoreIfMessage_HFUpdateIndicatorsValueResult;
   class PmCoreIfMessage_GetModemsResult;
   class PmCoreIfMessage_AcceptSCOConnectResult;
   class PmCoreIfMessage_SiriEnableResult;
   class PmCoreIfMessage_SiriSetNRResult;
   class PmCoreIfMessage_GetCallsResult;
   class PmCoreIfMessage_DialResult;
   class PmCoreIfMessage_SpeedDialResult;
   class PmCoreIfMessage_RedialResult;
   class PmCoreIfMessage_SwapCallResult;
   class PmCoreIfMessage_ReleaseAndAcceptResult;
   class PmCoreIfMessage_ReleaseAndSwapResult;
   class PmCoreIfMessage_HoldAndAcceptResult;
   class PmCoreIfMessage_HangupAllResult;
   class PmCoreIfMessage_SplitCallsResult;
   class PmCoreIfMessage_MergeCallsResult;
   class PmCoreIfMessage_HangupMultipartyResult;
   class PmCoreIfMessage_SendDTMFResult;
   class PmCoreIfMessage_HangupCallResult;
   class PmCoreIfMessage_AcceptCallResult;
   class PmCoreIfMessage_HoldIncomingCallResult;
   class PmCoreIfMessage_TransferAudioResult;
   class PmCoreIfMessage_StartStopVoiceRecognitionResult;

   // Response from PM_AudioManager
   class PmCoreIfMessage_PrepareAudioRouteResponse;
   class PmCoreIfMessage_PlayAudioResponse;
   class PmCoreIfMessage_StopAudioResponse;
   class PmCoreIfMessage_SetMicMuteStateResponse;
   class PmCoreIfMessage_PauseAudioResponse;

   class PmCoreMainController : public PmSingleton <PmCoreMainController>
   {
   public:
      /**
       * This method is used to register with the CallbackIfWrapper.
       *
       * @param[in] pmCoreCallbackIf - Pointer to the IPmCoreCallbackIf class
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult (PmResultCode, PmResultMessage)
       */
      PmResult registerCallback(IN IPmCoreCallbackIf* pmCoreCallbackIf);

      /**
       * This method is used to get the PmCore Callback Wrapper
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmCoreCallbackIfWrapper - Reference to the PmCoreCallbackIfWrapper
       */
      PmCoreCallbackIfWrapper& getPmCoreCallbackIfWrapper(void);

      /**
       * This method is used to get the PmSettings
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return PmSettings - Reference to the PmSettings
       */
      PmSettings& getPmSettings(void);

      /**
       * This method is used to get the DeviceInfoHandler
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return DeviceInfoHandler - Reference to the DeviceInfoHandler
       */
      DeviceInfoHandler& getDeviceInfoHandler(void);

      /**
       * This method is used to get the CallController
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return CallController - Reference to the CallController
       */
      CallController& getCallController(void);

      /**
       * This method is used to get the VRController
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return VRController - Reference to the VRController
       */
      VRController& getVRController(void);

      /**
       * This method is used to get the VRController
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return VRController - Reference to the VRController
       */
      RingtoneController& getRingtoneController(void);

      /**
       * This method is used to get the RVController
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return RVController - Reference to the RVController
       */
      RVController& getRVController(void);

      /**
       * This method is used to handle the incoming request and result messages to the PmCoreMainController.
       *
       * @param[in] pmCoreIfMessage - Pointer to the PmCoreIfMessage class
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage(IN std::shared_ptr<PmCoreIfMessage>& pmCoreIfMessage);

      /**
       * This method is called from other controller to post response for a request
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void postResponseForRequest(IN const std::string& operation, IN const PmResult pmResult,
            IN const BdAddress deviceAddress, IN const CallInstance callInstance,
            IN const ActType act);

   private:
      friend class PmSingleton<PmCoreMainController>;

      /**
       * This method is used to handle the Set SystemState call from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SetSystemState message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SetSystemState(
            IN std::shared_ptr<PmCoreIfMessage_SetSystemState> pmCoreIfMessage);

      /**
       * This method is used to handle the SetVehicleConfiguration call from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SetVehicleConfiguration message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SetVehicleConfiguration(
            IN std::shared_ptr<PmCoreIfMessage_SetVehicleConfiguration> pmCoreIfMessage);

      /**
       * This method is used to handle the OnDevicePaired call from client
       *
       * @param[in] pmCoreIfMessage - pointer to the OnDevicePaired message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_OnDevicePaired(
            IN std::shared_ptr<PmCoreIfMessage_OnDevicePaired> pmCoreIfMessage);

      /**
       * This method is used to handle the OnDeviceConnected call from client
       *
       * @param[in] pmCoreIfMessage - pointer to the OnDeviceConnected message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_OnDeviceConnected(
            IN std::shared_ptr<PmCoreIfMessage_OnDeviceConnected> pmCoreIfMessage);

      /**
       * This method is used to handle the OnDeviceDisconnected request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the OnDeviceDisconnectedRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_OnDeviceDisconnected(
            IN std::shared_ptr<PmCoreIfMessage_OnDeviceDisconnected> pmCoreIfMessage);

      /**
       * This method is used to handle the OnDeviceDeleted call from client
       *
       * @param[in] pmCoreIfMessage - pointer to the OnDeviceDeleted message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_OnDeviceDeleted(
            IN std::shared_ptr<PmCoreIfMessage_OnDeviceDeleted> pmCoreIfMessage);

      /**
       * This method is used to handle the GetSystemWideRingtoneList request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetSystemWideRingtoneListRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetSystemWideRingtoneListRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetSystemWideRingtoneListRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetDeviceRingtoneList request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetDeviceRingtoneListRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetDeviceRingtoneListRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetDeviceRingtoneListRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SetRingTone request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SetRingtoneRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SetRingtoneRequest(
            IN std::shared_ptr<PmCoreIfMessage_SetRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the PlayRingTone request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the PlayRingtoneRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_PlayRingtoneRequest(
            IN std::shared_ptr<PmCoreIfMessage_PlayRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the StopRingTone request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the StopRingtoneRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_StopRingtoneRequest(
            IN std::shared_ptr<PmCoreIfMessage_StopRingtoneRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetSuppressRingtoneOnOffList request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetSuppressRingtoneOnOffListRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetSuppressRingtoneOnOffListRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetSuppressRingtoneOnOffListRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SuppressRingtoneOnOff request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SuppressRingtoneOnOffRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SuppressRingtoneOnOffRequest(
            IN std::shared_ptr<PmCoreIfMessage_SuppressRingtoneOnOffRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetAutoWaitingModeOnOffList request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetAutoWaitingModeOnOffListRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetAutoWaitingModeOnOffListRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetAutoWaitingModeOnOffListRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the AutoWaitingModeOnOff request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the AutoWaitingModeOnOffRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_AutoWaitingModeOnOffRequest(
            IN std::shared_ptr<PmCoreIfMessage_AutoWaitingModeOnOffRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SetWaitingModeFilePath request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SetWaitingModeFilePathRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SetWaitingModeFilePathRequest(
            IN std::shared_ptr<PmCoreIfMessage_SetWaitingModeFilePathRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetWaitingModeFilePath request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetWaitingModeFilePathRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetWaitingModeFilePathRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetWaitingModeFilePathRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetPhoneCallAudioActive request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetPhoneCallAudioActiveRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetPhoneCallAudioActiveRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetPhoneCallAudioActiveRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetActivePassiveDeviceList request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetActivePassiveDeviceListRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetActivePassiveDeviceListRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetActivePassiveDeviceListRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SetActivePhone request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SetActivePhoneRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SetActivePhoneRequest(
            IN std::shared_ptr<PmCoreIfMessage_SetActivePhoneRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the RestoreDefaultSetting request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the RestoreDefaultSettingRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_RestoreDefaultSettingRequest(
            IN std::shared_ptr<PmCoreIfMessage_RestoreDefaultSettingRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetSupportedFeatures request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetSupportedFeaturesRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetSupportedFeaturesRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetSupportedFeaturesRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetNetworkStatus request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetNetworkStatusRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetNetworkStatusRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetNetworkStatusRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetSignalStrength request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetSignalStrengthRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetSignalStrengthRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetSignalStrengthRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetBatteryChargeLevel request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetBatteryChargeLevelRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetBatteryChargeLevelRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetBatteryChargeLevelRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetNetworkOperator request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetNetworkOperatorRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetNetworkOperatorRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetNetworkOperatorRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the VoiceMailList request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the VoiceMailListRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_VoiceMailListRequest(
            IN std::shared_ptr<PmCoreIfMessage_VoiceMailListRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetCallStatusList request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetCallStatusListRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetCallStatusListRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetCallStatusListRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetMicrophoneMuteState request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetMicrophoneMuteStateRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetMicrophoneMuteStateRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetMicrophoneMuteStateRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SetMicrophoneMuteState request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SetMicrophoneMuteStateRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SetMicrophoneMuteStateRequest(
            IN std::shared_ptr<PmCoreIfMessage_SetMicrophoneMuteStateRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the MergeCalls request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the MergeCallsRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_MergeCallsRequest(
            IN std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SplitCalls request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SplitCallsRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SplitCallsRequest(
            IN std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the HangupCalls request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the HangupCallsRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_HangupCallsRequest(
            IN std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the Redial request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the RedialRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_RedialRequest(
            IN std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetSCOConnection request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetSCOConnectionRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetSCOConnectionRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetSCOConnectionRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetRingtoneMuteState request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetRingtoneMuteStateRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetRingtoneMuteStateRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetRingtoneMuteStateRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SetRingtoneMuteState request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SetRingtoneMuteStateRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SetRingtoneMuteStateRequest(
            IN std::shared_ptr<PmCoreIfMessage_SetRingtoneMuteStateRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetWaitingModeState request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetWaitingModeStateRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetWaitingModeStateRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetWaitingModeStateRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetRingtonePlaybackStatus request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetRingtonePlaybackStatus message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetRingtonePlaybackStatusRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetRingtonePlaybackStatusRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the StartStopWaitingMode request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the StartStopWaitingModeRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_StartStopWaitingModeRequest(
            IN std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the Dial request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the DialRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_DialRequest(
            IN std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the AcceptCall request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the AcceptCallRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_AcceptCallRequest(
            IN std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SwapCall request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SwapCallRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SwapCallRequest(
            IN std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SpeedDial request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SpeedDialRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SpeedDialRequest(
            IN std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the SendDTMF request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the SendDTMFRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SendDTMFRequest(
            IN std::shared_ptr<PmCoreIfMessage_SendDTMFRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the TransferAudio request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the TransferAudioRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_TransferAudioRequest(
            IN std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetVoiceRecognitionStatus request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetVoiceRecognitionStatusRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetVoiceRecognitionStatusRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetVoiceRecognitionStatusRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetExtVoiceRecognitionStatus request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetExtVoiceRecognitionStatusRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetExtVoiceRecognitionStatusRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetExtVoiceRecognitionStatusRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the StartStopVoiceRecognition request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the StartStopVoiceRecognitionRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_StartStopVoiceRecognitionRequest(
            IN std::shared_ptr<PmCoreIfMessage_StartStopVoiceRecognitionRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the ExtStartStopVoiceRecognition request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the ExtStartStopVoiceRecognitionRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_StartStopExternalVoiceRecognitionRequest(
            IN std::shared_ptr<PmCoreIfMessage_StartStopExternalVoiceRecognitionRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the StopExtendedVoiceRecognition request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the StopExtendedVoiceRecognition message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_stopExtendedVRRequest(
            IN std::shared_ptr<PmCoreIfMessage_stopExtendedVRRequest> pmCoreIfMessage);

      /**
       * This method is used to handle the GetEnhancedVoiceRecognitionFeature request from client
       *
       * @param[in] pmCoreIfMessage - pointer to the GetEnhancedVoiceRecognitionFeatureRequest message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest(
            IN std::shared_ptr<PmCoreIfMessage_GetEnhancedVoiceRecognitionFeatureRequest> pmCoreIfMessage);

      /*
       * This method is used to handle the Request PhoneNumber Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the RequestPhoneNumberResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_RequestPhoneNumberResult(
            IN std::shared_ptr<PmCoreIfMessage_RequestPhoneNumberResult> pmCoreIfMessage);

      /*
       * This method is used to handle the Get Subscriber Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the GetSubscriberResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetSubscriberResult(
            IN std::shared_ptr<PmCoreIfMessage_GetSubscriberResult> pmCoreIfMessage);

      /*
       * This method is used to handle the Send Custom AT Command Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the SendCustomATCommandResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SendCustomATCommandResult(
            IN std::shared_ptr<PmCoreIfMessage_SendCustomATCommandResult> pmCoreIfMessage);

      /*
       * This method is used to handle the HF Update Indicators Value Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the HFUpdateIndicatorsValueResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_HFUpdateIndicatorsValueResult(
            IN std::shared_ptr<PmCoreIfMessage_HFUpdateIndicatorsValueResult> pmCoreIfMessage);

      /*
       * This method is used to handle the Get Modems Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the GetModemsResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetModemsResult(
            IN std::shared_ptr<PmCoreIfMessage_GetModemsResult> pmCoreIfMessage);

      /*
       * This method is used to handle the Accept SCO Connect Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the AcceptSCOConnectResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_AcceptSCOConnectResult(
            IN std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage);

      /*
       * This method is used to handle the Siri Enable Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the SiriEnableResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SiriEnableResult(
            IN std::shared_ptr<PmCoreIfMessage_SiriEnableResult> pmCoreIfMessage);

      /*
       * This method is used to handle the Siri Set NR Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the SiriSetNRResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SiriSetNRResult(
            IN std::shared_ptr<PmCoreIfMessage_SiriSetNRResult> pmCoreIfMessage);

      /*
       * This method is used to handle the Get Calls Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the GetCallsResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_GetCallsResult(
            IN std::shared_ptr<PmCoreIfMessage_GetCallsResult> pmCoreIfMessage);

      /*
       * This method is used to handle the Dial Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the DialResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_DialResult(
            IN std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage);

      /**
       * This method is used to handle the SpeedDial result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the SpeedDialResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SpeedDialResult(
            IN std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage);

      /**
       * This method is used to handle the Redial result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the RedialResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_RedialResult(
            IN std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage);

      /**
       * This method is used to handle the SwapCall result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the SwapCallResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SwapCallResult(
            IN std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage);

      /**
       * This method is used to handle the ReleaseAndAccept result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the ReleaseAndAcceptResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_ReleaseAndAcceptResult(
            IN std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage);

      /**
       * This method is used to handle the ReleaseAndSwap result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the ReleaseAndSwapResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_ReleaseAndSwapResult(
            IN std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage);

      /**
       * This method is used to handle the HoldAndAccept result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the HoldAndAcceptResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_HoldAndAcceptResult(
            IN std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage);

      /**
       * This method is used to handle the HangupAll result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the HangupAllResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_HangupAllResult(
            IN std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage);

      /**
       * This method is used to handle the SplitCalls result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the SplitCallsResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SplitCallsResult(
            IN std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage);

      /**
       * This method is used to handle the MergeCalls result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the MergeCallsResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_MergeCallsResult(
            IN std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage);

      /**
       * This method is used to handle the Hangup Multiparty Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the HangupMultipartyResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_HangupMultipartyResult(
            IN std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage);

      /**
       * This method is used to handle the SendDTMF result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the SendDTMFResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SendDTMFResult(
            IN std::shared_ptr<PmCoreIfMessage_SendDTMFResult> pmCoreIfMessage);

      /**
       * This method is used to handle the HangupCall result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the HangupCallResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_HangupCallResult(
            IN std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage);

      /**
       * This method is used to handle the AcceptCall result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the AcceptCallResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_AcceptCallResult(
            IN std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage);

      /**
       * This method is used to handle the Hold IncomingCall Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the HoldIncomingCallResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_HoldIncomingCallResult(
            IN std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage);

      /**
       * This method is used to handle the Transfer Audio Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the TransferAudioResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_TransferAudioResult(
            IN std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage);

      /**
       * This method is used to handle the VR Property Set Result from stack
       *
       * @param[in] pmCoreIfMessage - pointer to the VRPropertySetResult message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_StartStopVoiceRecognitionResult(
            IN std::shared_ptr<PmCoreIfMessage_StartStopVoiceRecognitionResult> pmCoreIfMessage);


      /**
       * This method is used to handle the Prepare Audio Route response from PM Audiomanager
       *
       * @param[in] pmCoreIfMessage - pointer to the PrepareAudioRouteResponse message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_PrepareAudioRouteResponse(
            IN std::shared_ptr<PmCoreIfMessage_PrepareAudioRouteResponse> pmCoreIfMessage);

      /**
       * This method is used to handle the PlayAudio Response from PM Audiomanager
       *
       * @param[in] pmCoreIfMessage - pointer to the PlayAudioResponse message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_PlayAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_PlayAudioResponse> pmCoreIfMessage);

      /**
       * This method is used to handle the StopAudio response from PM Audiomanager
       *
       * @param[in] pmCoreIfMessage - pointer to the StopAudioResponse message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_StopAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_StopAudioResponse> pmCoreIfMessage);

      /**
       * This method is used to handle the SetMicMute response from PM Audiomanager
       *
       * @param[in] pmCoreIfMessage - pointer to the SetMicMuteResponse message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_SetMicMuteStateResponse(
            IN std::shared_ptr<PmCoreIfMessage_SetMicMuteStateResponse> pmCoreIfMessage);

      /**
       * This method is used to handle the pause Audio response from PM Audiomanager
       *
       * @param[in] pmCoreIfMessage - pointer to the PauseAudioResponse message
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void handlePmCoreIfMessage_PauseAudioResponse(
            IN std::shared_ptr<PmCoreIfMessage_PauseAudioResponse> pmCoreIfMessage);

      /**
       * Constructor of PmCoreMainController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmCoreMainController();

      /**
       * Destructor of PmCoreMainController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~PmCoreMainController();

      /**
       * Copy construcor of PmCoreMainController class
       *
       * @param[in] PmCoreMainController - reference to PmCoreMainController instance
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmCoreMainController(const PmCoreMainController&);

      /**
       * Assignment operator of PmCoreMainController class
       *
       * @param[in] PmCoreMainController - reference to PmCoreMainController instance
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      const PmCoreMainController& operator=(const PmCoreMainController&);

      PmSettings                     _pmSettings;              /**< Instance of PmSettings */
      DeviceInfoHandler              _deviceInfoHandler;       /**< Instance of DeviceInfoHandler */
      CallController                 _callController;          /**< Instance of CallController */
      VRController                   _vrController;            /**< Instance of VRController */
      RingtoneController             _ringtoneController;      /**< Instance of RingtoneController */
      RVController                   _rVController;            /**< Instance of RVController */
      PmCoreCallbackIfWrapper*       _pmCoreCallbackIfWrapper; /**< Instance of PmCoreCallbackIfWrapper */
   };
}

#endif // PmCoreMainController_h
