/**
 * @file DeviceInfoHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the DeviceInfoHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmCore
 */

#include "PmCoreRequestIf.h"
#include "DeviceInfoHandler.h"
#include "PmCoreMainController.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PropertyDetails.h"
#include "EvoBtStackWrapper.h"
#include "PmCoreIfMessageResult.h"
#include "PmAudioManagerWrapper.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceInfoHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

using namespace pmaudiomanager;

namespace pmcore
{
   DeviceInfoHandler::DeviceInfoHandler() :
         _deviceInfoList(),
         _activePassiveDeviceList(),
         _propertyIdList(),
         _deviceSCOStatus(),
         _devicesToBeMadePassive()
   {
      ETG_TRACE_USR4(("DeviceInfoHandler() entered"));
      subscribeToBtStackEventNotifier();
   }

   DeviceInfoHandler::~DeviceInfoHandler()
   {
      ETG_TRACE_USR4(("~DeviceInfoHandler() entered"));

      PropertyUpdateNotifierToCore::getInstance().detachControllerInNotifierList(_propertyIdList, this);
      _deviceInfoList.clear();
      _activePassiveDeviceList.clear();
      _propertyIdList.clear();
      _deviceSCOStatus.clear();
      _devicesToBeMadePassive.clear();
   }

   void DeviceInfoHandler::subscribeToBtStackEventNotifier()
   {
      ETG_TRACE_USR1(("subscribeToBtStackEventNotifier"));

      // Pushing the default properties for DeviceInfoHandler
      for (unsigned int index = BTS_UPDATE_SUPPORTED_FEATURES; index < BTS_UPDATE_END_DEVICE_INFO_HANDLER; index++)
      {
         _propertyIdList.push_back(static_cast<PmCorePropertyAndEventId>(index));
      }

      // Pushing the other interested properties and events
      _propertyIdList.push_back(ON_DEVICE_CONNECTED);
      _propertyIdList.push_back(ON_DEVICE_DISCONNECTED);
      _propertyIdList.push_back(ON_DEVICE_DELETED);
      _propertyIdList.push_back(BTS_UPDATE_SCO_TYPE);

      PropertyUpdateNotifierToCore::getInstance().attachControllerToNotifierList(_propertyIdList, this);
   }

   void DeviceInfoHandler::onDeviceConnected(const DeviceHandle deviceHandle, const BdAddress& deviceAddress)
   {
      if(!addDeviceToDeviceInfoList(deviceHandle, deviceAddress))
      {
         ETG_TRACE_USR4(("Adding device to DeviceInfoList failed (or) device already exist"));
      }

      if(addDeviceToActivePassiveDeviceList(deviceAddress))
      {
         updateActivePassiveDeviceList();
      }
   }

   void DeviceInfoHandler::onDeviceDisconnected(const BdAddress& deviceAddress)
   {
      if(deviceAddress.compare(DEVICE_ADDRESS_ALL) == 0)
      {
         //Do for all devices
         bool retValue = false;
         DeviceInfoList::iterator it;

         for(it = _deviceInfoList.begin(); it != _deviceInfoList.end(); it++)
         {
            updatePropertiesWithDefaultValues(it->first);

            retValue |= removeDeviceFromActivePassiveDeviceList(it->first);
         }

         if(true == retValue)
         {
            updateActivePassiveDeviceList();
         }

         clearDeviceInfoList();

         // Since the device itself disconnects, the map "_devicesToBeMadePassive" shall be cleared.
         PmResult pmResult(PM_RESULT_ERR_DEVICE_NOT_EXIST, "");
         for (auto& iter : _devicesToBeMadePassive)
         {
            processSwitchToPassiveResponse(iter.first, pmResult, false);
         }
         _devicesToBeMadePassive.clear();
      }
      else
      {
         updatePropertiesWithDefaultValues(deviceAddress);

         DeviceRole disconnectedDeviceRole = DEVICEROLE_DEFAULT;
         auto activePassiveDeviceListIter = _activePassiveDeviceList.find(deviceAddress);
         if(activePassiveDeviceListIter != _activePassiveDeviceList.end())
            disconnectedDeviceRole = activePassiveDeviceListIter->second;

         // In case of setting a Passive device to Active device when Active device is disconnected, the flow
         // checks "_deviceInfoList". And hence the disconnected device is first removed from "_deviceInfoList"
         // and then from "_activePassiveDeviceList"
         if(!removeDeviceFromDeviceInfoList(deviceAddress))
         {
            ETG_TRACE_USR4(("Removing device from DeviceInfoList failed (or) device already removed"));
         }

         //Do for specific device
         if(removeDeviceFromActivePassiveDeviceList(deviceAddress))
         {
            // clearing the item from "_devicesToBeMadePassive", if present
            auto iter = std::find_if(_devicesToBeMadePassive.begin(), _devicesToBeMadePassive.end(),
                  [&deviceAddress](std::pair<BdAddress, ActiveSwitchingRequest> const& obj)
                  {return obj.second._deviceAddress == deviceAddress;});
            if (_devicesToBeMadePassive.end() != iter)
            {
               processSwitchToPassiveResponse(deviceAddress, PmResult(PM_RESULT_ERR_DEVICE_NOT_EXIST, ""));
               _devicesToBeMadePassive.erase(iter);
            }

            if ((DEVICEROLE_ACTIVE == disconnectedDeviceRole) && (0 != _activePassiveDeviceList.size()))
            {
               for(auto it = _activePassiveDeviceList.begin(); it != _activePassiveDeviceList.end(); ++it)
               {
                  // IMPORTANT: As of now, the first met PASSIVE device is made ACTIVE
                  if(DEVICEROLE_PASSIVE == it->second)
                  {
                     BdAddressList activeDevicesList;
                     getActiveDevicesList(activeDevicesList);

                     ActivePassiveDeviceListMap setActivePassiveDeviceList;

                     for (auto& activeDeviceAddress : activeDevicesList)
                     {
                        setActivePassiveDeviceList.emplace_hint(setActivePassiveDeviceList.end(),
                              activeDeviceAddress, DEVICEROLE_ACTIVE);
                     }
                     setActivePassiveDeviceList.emplace_hint(setActivePassiveDeviceList.end(),
                           it->first, DEVICEROLE_ACTIVE);

                     setDeviceRole(setActivePassiveDeviceList);
                     break;
                  }
               }
            }
            else
            {
               updateActivePassiveDeviceList();
            }
         }
      }
      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));
   }

   void DeviceInfoHandler::updatePropertiesWithDefaultValues(const BdAddress& deviceAddress)
   {
      //Set DeviceInfo details to default
      SupportedFeatures supportedFeatures;
      onSupportedFeaturesSignal(deviceAddress, supportedFeatures);

      NetworkStatus networkStatus;
      onNetworkStatusSignal(deviceAddress, networkStatus);

      SignalStrength signalStrength;
      onSignalStrengthSignal(deviceAddress, signalStrength);

      BatteryChargeLevel batteryChargeLevel;
      onBatteryChargeLevelSignal(deviceAddress, batteryChargeLevel);

      NetworkOperator networkOperator;
      onNetworkOperatorSignal(deviceAddress, networkOperator);
   }

   void DeviceInfoHandler::onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
         IN std::shared_ptr<void> propertyDetails)
   {
      ETG_TRACE_USR4(("DeviceInfoHandler::onPropertyUpdate propertyId : %d",
            ETG_CENUM(PmCorePropertyAndEventId, propertyId)));
      //TODO: Handler functions needs to be implemented and mapped to switch cases.

      if(nullptr == propertyDetails)
      {
         ETG_TRACE_ERR(("DeviceInfoHandler::onPropertyUpdate with empty details"));
         return;
      }

      switch(propertyId)
      {
         case ON_DEVICE_CONNECTED:
         {
            std::shared_ptr<PropertyDetails<BasicDeviceDetails>> property =
                  std::static_pointer_cast<PropertyDetails<BasicDeviceDetails>>(propertyDetails);

            BdAddress deviceAddress = property->getMessage()._deviceAddress;
            DeviceIdentification deviceIdentification = property->getMessage()._deviceIdentification;

            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate deviceHandle : %u", property->getMessage()._deviceHandle));
            ETG_TRACE_USR4(("onPropertyUpdate vendorId : %u", deviceIdentification._vendorId));
            ETG_TRACE_USR4(("onPropertyUpdate vendorIdSource : %u", deviceIdentification._vendorIdSource));

            onDeviceConnected(property->getMessage()._deviceHandle, deviceAddress);

            bool isAppleDevice = (PM_BLUETOOTH_SIG_ID_SOURCE == deviceIdentification._vendorIdSource
                  && PM_APPLE_BLUETOOTH_SIG_VENDOR_ID == deviceIdentification._vendorId) ||
                        (PM_USB_IMPLEMENTER_FORUM_ID_SOURCE == deviceIdentification._vendorIdSource
                              && PM_APPLE_USB_IMPLEMENTER_VENDOR_ID == deviceIdentification._vendorId);

            setAppleDevice(deviceAddress, isAppleDevice);

            _deviceSCOStatus[property->getMessage()._deviceAddress] = SCO_NOT_ESTABLISHED;
         }
         break;
         case ON_DEVICE_DISCONNECTED:
         {
            std::shared_ptr<PropertyDetails<BdAddress>> property =
                  std::static_pointer_cast<PropertyDetails<BdAddress>>(propertyDetails);

            BdAddress deviceAddress = property->getMessage();
            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", deviceAddress.c_str()));

            onDeviceDisconnected(deviceAddress);

            auto deviceSCOStatusIter = _deviceSCOStatus.find(deviceAddress);
            if (deviceSCOStatusIter != _deviceSCOStatus.end())
            {
               _deviceSCOStatus.erase(deviceAddress);
            }
         }
         break;
         case ON_DEVICE_DELETED:
         {
            std::shared_ptr<PropertyDetails<BdAddress>> property =
                  std::static_pointer_cast<PropertyDetails<BdAddress>>(propertyDetails);
            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getMessage().c_str()));
         }
         break;

         case BTS_UPDATE_SCO_TYPE:
         {
            std::shared_ptr<PropertyDetails<SCOConnection>> property =
                  std::static_pointer_cast<PropertyDetails<SCOConnection>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();
            SCOStatus scoStatus = property->getMessage()._scoStatus;
            SCOStatus prevScoStatus = SCO_DEFAULT;

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", property->getBdAddress().c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate ScoStatus : %u", scoStatus));

            auto deviceSCOStatusIter = _deviceSCOStatus.find(deviceAddress);
            if (deviceSCOStatusIter != _deviceSCOStatus.end())
            {
               prevScoStatus = _deviceSCOStatus[deviceAddress];
               _deviceSCOStatus[deviceAddress] = scoStatus;
            }

            ETG_TRACE_USR4(("onPropertyUpdate ScoStatus : %u", scoStatus));
            ETG_TRACE_USR4(("onPropertyUpdate prevScoStatus : %u", prevScoStatus));

            if ((SCO_NOT_ESTABLISHED == scoStatus) && ((prevScoStatus != SCO_NOT_ESTABLISHED) && (prevScoStatus != SCO_DEFAULT)))
            {
               // This case is useful when SCO is simply established and disconnected by mistake for a Passive device
               if (isDeviceInIdleState(deviceAddress))
               {
                  processSwitchToPassiveResponse(deviceAddress, PmResult());
               }
            }
         }
         break;

         case BTS_UPDATE_SUPPORTED_FEATURES:
         {
            std::shared_ptr<PropertyDetails<SupportedFeatures>> property =
                  std::static_pointer_cast<PropertyDetails<SupportedFeatures>>(propertyDetails);

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", property->getBdAddress().c_str()));

            for(auto it = property->getMessage()._supportedFeatures.begin();
                  it != property->getMessage()._supportedFeatures.end(); it++)
            {
               ETG_TRACE_USR4(("onPropertyUpdate SupportedFeatures : %s", it->c_str()));
            }

            onSupportedFeaturesSignal(property->getBdAddress(), property->getMessage());
         }
         break;
         case BTS_UPDATE_INBAND_RINGING:
            break;
         case BTS_UPDATE_BATTERY_CHARGE_LEVEL:
         {
            std::shared_ptr<PropertyDetails<BatteryChargeLevel>> property =
                  std::static_pointer_cast<PropertyDetails<BatteryChargeLevel>>(propertyDetails);
            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", property->getBdAddress().c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate BatteryChargeLevel : %u", property->getMessage()._batteryChargeLevel));
            onBatteryChargeLevelSignal(property->getBdAddress(), property->getMessage());
         }
         break;
         case BTS_UPDATE_CHLD_FEATURE:
            break;
         case BTS_UPDATE_NETWORK_STATUS:
         {
            std::shared_ptr<PropertyDetails<NetworkStatus>> property =
                  std::static_pointer_cast<PropertyDetails<NetworkStatus>>(propertyDetails);
            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", property->getBdAddress().c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate NetworkStatus : %u", property->getMessage()._networkStatus));
            onNetworkStatusSignal(property->getBdAddress(), property->getMessage());
         }
         break;
         case BTS_UPDATE_SIGNAL_STRENGTH:
         {
            std::shared_ptr<PropertyDetails<SignalStrength>> property =
                  std::static_pointer_cast<PropertyDetails<SignalStrength>>(propertyDetails);
            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", property->getBdAddress().c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate SignalStrength : %u", property->getMessage()._signalStrength));
            onSignalStrengthSignal(property->getBdAddress(), property->getMessage());
         }
         break;
         case BTS_UPDATE_CURRENT_OPERATOR_NAME:
         {
            std::shared_ptr<PropertyDetails<NetworkOperator>> property =
                  std::static_pointer_cast<PropertyDetails<NetworkOperator>>(propertyDetails);
            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", property->getBdAddress().c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate NetworkOperator : %s", property->getMessage()._networkOperator.c_str()));
            onNetworkOperatorSignal(property->getBdAddress(), property->getMessage());
         }
         break;
         case BTS_UPDATE_MOBILE_COUNTRY_CODE:
            break;
         default:
            ETG_TRACE_ERR(("DeviceInfoHandler::Invalid propertyId"));
            break;
      }
   }

   bool DeviceInfoHandler::isFeatureSupported(const BdAddress& deviceAddress, const std::string& feature)
   {
      bool featureSupported = false;
      SupportedFeatures supportedFeatures;
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.getSupportedFeatures(supportedFeatures);

         auto FeatureListIter = std::find_if(supportedFeatures._supportedFeatures.begin(),
               supportedFeatures._supportedFeatures.end(), [&feature](std::string const& obj)
               {return obj == feature;});

         if(supportedFeatures._supportedFeatures.end() != FeatureListIter)
         {
            featureSupported = true;
         }
      }

      return featureSupported;
   }

   void DeviceInfoHandler::getSupportedFeatures(const BdAddress& deviceAddress, const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      SupportedFeatures supportedFeatures;

      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.getSupportedFeatures(supportedFeatures);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetSupportedFeaturesResponse(
            pmResult, deviceAddress, supportedFeatures, act);
   }

   void DeviceInfoHandler::onSupportedFeaturesSignal(const BdAddress& deviceAddress,
         const SupportedFeatures& supportedFeatures)
   {
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.setSupportedFeatures(supportedFeatures);
      }
      else
      {
         //this should never happen
         ETG_TRACE_USR4(("SupportedFeatures update received for non-existing device"));
      }
   }

   void DeviceInfoHandler::getNetworkStatus(const BdAddress& deviceAddress, const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      NetworkStatus networkStatus;

      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.getNetworkStatus(networkStatus);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetNetworkStatusResponse(
            pmResult, deviceAddress, networkStatus, act);
   }

   void DeviceInfoHandler::onNetworkStatusSignal(const BdAddress& deviceAddress, const NetworkStatus& networkStatus)
   {
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.setNetworkStatus(networkStatus);
      }
      else
      {
         //this should never happen
         ETG_TRACE_USR4(("NetworkStatus update received for non-existing device"));
      }
   }

   void DeviceInfoHandler::getSignalStrength(const BdAddress& deviceAddress, const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      SignalStrength signalStrength;

      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.getSignalStrength(signalStrength);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetSignalStrengthResponse(
            pmResult, deviceAddress, signalStrength, act);
   }

   void DeviceInfoHandler::onSignalStrengthSignal(const BdAddress& deviceAddress, const SignalStrength& signalStrength)
   {
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.setSignalStrength(signalStrength);
      }
      else
      {
         //this should never happen
         ETG_TRACE_USR4(("SignalStrength update received for non-existing device"));
      }
   }

   void DeviceInfoHandler::getBatteryChargeLevel(const BdAddress& deviceAddress, const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      BatteryChargeLevel batteryChargeLevel;

      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.getBatteryChargeLevel(batteryChargeLevel);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetBatteryChargeLevelResponse(
            pmResult, deviceAddress, batteryChargeLevel, act);
   }

   void DeviceInfoHandler::onBatteryChargeLevelSignal(const BdAddress& deviceAddress,
         const BatteryChargeLevel& batteryChargeLevel)
   {
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.setBatteryChargeLevel(batteryChargeLevel);
      }
      else
      {
         //this should never happen
         ETG_TRACE_USR4(("BatteryChargeLevel update received for non-existing device"));
      }
   }

   void DeviceInfoHandler::getNetworkOperator(const BdAddress& deviceAddress, const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      NetworkOperator networkOpertor;

      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.getNetworkOperator(networkOpertor);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetNetworkOperatorResponse(
            pmResult, deviceAddress, networkOpertor, act);
   }

   void DeviceInfoHandler::onNetworkOperatorSignal(const BdAddress& deviceAddress,
         const NetworkOperator& networkOpertaor)
   {
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.setNetworkOperator(networkOpertaor);
      }
      else
      {
         //this should never happen
         ETG_TRACE_USR4(("NetworkOperator update received for non-existing device"));
      }
   }

   bool DeviceInfoHandler::isAppleDevice(IN const BdAddress& deviceAddress)
   {
      bool isAppleDevice = false;

      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         isAppleDevice = it->second.isAppleDevice();
      }

      return isAppleDevice;
   }

   void DeviceInfoHandler::setAppleDevice(IN const BdAddress& deviceAddress, IN const bool isAppleDevice)
   {
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.setAppleDevice(isAppleDevice);
      }
      else
      {
         //this should never happen
         ETG_TRACE_USR4(("setAppleDevice called for non-existing device"));
      }
   }

   PmResult DeviceInfoHandler::getDeviceRole(const BdAddress& deviceAddress, DeviceRole& deviceRole, const ActType act)
   {
      (void)(act); // might be required in future.
      PmResult pmResult(PM_RESULT_OK, "");

      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         it->second.getDeviceRole(deviceRole);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      return pmResult;
   }

   PmResult DeviceInfoHandler::setDeviceRoleRequest(ActivePassiveDeviceListMap& setActivePassiveDeviceList)
   {
      ETG_TRACE_USR4(("setDeviceRoleRequest() entered"));

      // Since the user himself/herself sets the preference now, the map "_devicesToBeMadePassive" shall be cleared.
      PmResult pmResult(PM_RESULT_ERR_USER_PREF_CHANGED_FOR_DEVICE_ROLE, "");

      for (auto& iter : _devicesToBeMadePassive)
      {
         processSwitchToPassiveResponse(iter.first, pmResult, false);
      }
      _devicesToBeMadePassive.clear();
      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));

      return setDeviceRole(setActivePassiveDeviceList, true);
   }

   PmResult DeviceInfoHandler::setDeviceRole(ActivePassiveDeviceListMap& setActivePassiveDeviceList,
         bool requestFromClient)
   {
      ETG_TRACE_USR4(("setDeviceRole() entered"));

      PmResult pmResult(PM_RESULT_OK, "");

      if(0 < setActivePassiveDeviceList.size())
      {
         DevicesCount maxNumActiveDevices = com::bosch::pmcommon::PmConfiguration::getInstance().getMaxNumActiveDevices();
         BdAddressList passiveToActiveDevices, activeToPassiveDevices, activeToPassiveDevicesClientRequest;

         //first get the current active passive device list
         ActivePassiveDeviceListMap prevActivePassiveDeviceList = _activePassiveDeviceList;

         //clear the active passive device list
         clearActivePassiveDeviceList();

         //check all available devices, set active role to received devices. all others will be set as passive.
         DeviceInfoList::iterator deviceInfoListIter;
         for(deviceInfoListIter = _deviceInfoList.begin(); deviceInfoListIter != _deviceInfoList.end();
               deviceInfoListIter++)
         {
            DeviceRole deviceRole = DEVICEROLE_PASSIVE;
            ActivePassiveDeviceListMap::iterator it = setActivePassiveDeviceList.find(deviceInfoListIter->first);

            if(it != setActivePassiveDeviceList.end())
            {
               DeviceRole currentDeviceRole = DEVICEROLE_PASSIVE;
               deviceInfoListIter->second.getDeviceRole(currentDeviceRole);

               if((DEVICEROLE_ACTIVE == it->second) && (getNumberOfActiveDevices() < maxNumActiveDevices))
               {
                  deviceRole = DEVICEROLE_ACTIVE;

                  if (DEVICEROLE_PASSIVE == currentDeviceRole)
                  {
                     ETG_TRACE_USR4(("Passive to Active Switching"));
                     passiveToActiveDevices.emplace_back(it->first);
                  }
               }

               // Fixed for an issue. While switching between Active Passive devices from VR active mode,
               // a device is selected from devices list by mistake. By this time, the device switching happened and
               // the VR needs to be stopped in this case for the Passive device.
               else if ((DEVICEROLE_PASSIVE == it->second) && (DEVICEROLE_ACTIVE == currentDeviceRole))
               {
                  if (requestFromClient)
                  {
                     activeToPassiveDevicesClientRequest.emplace_back(it->first);
                  }
                  activeToPassiveDevices.emplace_back(it->first);
               }

               setActivePassiveDeviceList.erase(it);
            }

            //set the received role to corresponding device
            _activePassiveDeviceList.insert(std::pair<BdAddress, DeviceRole>(deviceInfoListIter->first, deviceRole));

            deviceInfoListIter->second.setDeviceRole(deviceRole);
         }

         printActivePassiveDeviceList();
         //update active passive device list if any changes in the device role
         if(prevActivePassiveDeviceList != _activePassiveDeviceList)
         {
            updateActivePassiveDeviceList();
         }

         // On updating the original active passive devices to the clients and PM Audiomanager, the SMs shall request
         // for the corresponding channels
         // And activeToPassiveDevices is not used as of now.

         for (auto& deviceAddress : passiveToActiveDevices)
         {
            auto iter = std::find_if(_devicesToBeMadePassive.begin(), _devicesToBeMadePassive.end(),
                  [&deviceAddress](std::pair<BdAddress, ActiveSwitchingRequest> const& obj)
                  {return obj.second._deviceAddress == deviceAddress;});

            if ((_devicesToBeMadePassive.end() != iter) &&
                  (isSwitchReasonAcceptCallOrStartWaitMode(iter->second._switchingReason)))
            {
               ETG_TRACE_USR4(("The same device is already waiting to be switched to Active"));
            }
            else
            {
               PmCoreMainController::getInstance().getCallController().switchToActive(deviceAddress);
            }
         }
         for (auto& deviceAddress : activeToPassiveDevicesClientRequest)
         {
            PmCoreMainController::getInstance().getVRController().switchToPassiveClientRequest(deviceAddress);
         }

         if(0 < setActivePassiveDeviceList.size())
         {
            //received list contains unknown devices
            pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
         }
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
      }

      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));
      return pmResult;
   }

   DeviceRoleSwitchResultEnum DeviceInfoHandler::acquireFreeActiveDeviceSlot(const BdAddress& deviceAddress,
         const ActiveSwitchingRequest activeSwitchingRequest)
   {
      ETG_TRACE_USR4(("acquireFreeActiveDeviceSlot() entered with deviceAddress: %20s", deviceAddress.c_str()));

      DeviceRole deviceRole = DEVICEROLE_DEFAULT;
      getDeviceRole(deviceAddress, deviceRole);

      DeviceRoleSwitchResultEnum deviceRoleSwitchResult = CANNOT_BE_MADE_ACTIVE;

      if (DEVICEROLE_ACTIVE == deviceRole)
      {
         ETG_TRACE_USR4(("Active device"));
         deviceRoleSwitchResult = ALREADY_ACTIVE;
      }
      else if ((DEVICEROLE_PASSIVE == deviceRole) || (DEVICE_ADDRESS_ALL == deviceAddress))
      {
         ETG_TRACE_USR4(("Passive device or DEVICE_ADDRESS_ALL"));

         // Get the active devices list
         BdAddressList activeDevicesList;
         getActiveDevicesList(activeDevicesList);

         if (activeDevicesList.empty())
         {
            ETG_TRACE_USR4(("No Active device connected"));
            deviceRoleSwitchResult = SWITCHED_TO_ACTIVE;
         }
         else
         {
            for (auto& activeDeviceAddress : activeDevicesList)
            {
               CallController& callController = PmCoreMainController::getInstance().getCallController();
               if (!isSCOEstablished(activeDeviceAddress))
               {
                  if ((callController.isCallsInIdleState(activeDeviceAddress)) &&
                        (PmCoreMainController::getInstance().getVRController().isVRStatusIdle(activeDeviceAddress)))
                  {
                     if (DEVICE_ADDRESS_ALL == deviceAddress)
                     {
                        deviceRoleSwitchResult = SWITCHED_TO_ACTIVE;
                        break;
                     }

                     ActiveSwitchingRequest activeSwitchingRequestFinal(activeSwitchingRequest);

                     if (TEMP_ROLE_SWITCH == activeSwitchingRequest._switchingReason)
                     {
                        activeSwitchingRequestFinal._deviceAddress = activeDeviceAddress;
                        // Active device is in an Idle state
                        _devicesToBeMadePassive.emplace_hint(_devicesToBeMadePassive.end(), deviceAddress,
                              activeSwitchingRequestFinal);
                     }
                     // For accept call and the similar actions, the following is the apt parameters
                     else
                     {
                        _devicesToBeMadePassive.emplace_hint(_devicesToBeMadePassive.end(), activeDeviceAddress,
                              activeSwitchingRequestFinal);
                     }

                     ActivePassiveDeviceListMap activePassiveDevicelist;
                     for(auto& activeDevicesListIter : activeDevicesList)
                     {
                        if (activeDeviceAddress != activeDevicesListIter)
                        {
                           activePassiveDevicelist.emplace_hint(activePassiveDevicelist.end(), activeDevicesListIter,
                                 DEVICEROLE_ACTIVE);
                        }
                     }
                     activePassiveDevicelist.emplace_hint(activePassiveDevicelist.end(), deviceAddress, DEVICEROLE_ACTIVE);

                     PmResult pmResult = setDeviceRole(activePassiveDevicelist);

                     // Swapping the device address if present for the device address "activeDeviceAddress"
                     PmAudioManagerWrapper::getInstance().swapDeviceAddress(activeDeviceAddress, deviceAddress, CALL_CONTROLLER);

                     // since these are user prompted actions, switching back again is not preferred.
                     if (TEMP_ROLE_SWITCH != activeSwitchingRequest._switchingReason)
                     {
                        clearElementFromDeviceToBeMadePassiveMap(activeDeviceAddress);
                     }

                     if (PM_RESULT_OK == pmResult._pmResultCode)
                     {
                        deviceRoleSwitchResult = SWITCHED_TO_ACTIVE;
                     }
                     break;
                  }
               }
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("DEVICEROLE_DEFAULT"));
      }

      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));
      ETG_TRACE_USR4(("deviceRoleSwitchResult: %u", deviceRoleSwitchResult));

      return deviceRoleSwitchResult;
   }

   void DeviceInfoHandler::clearElementFromDeviceToBeMadePassiveMap(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("clearElementFromDeviceToBeMadePassiveMap() entered with deviceAddress: %s",
            deviceAddress.c_str()));

      auto iter = _devicesToBeMadePassive.find(deviceAddress);

      if (iter != _devicesToBeMadePassive.end())
      {
         _devicesToBeMadePassive.erase(iter);
      }

      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));
   }

   DeviceRoleSwitchResultEnum DeviceInfoHandler::acquireActiveDeviceSlot(
         const ActiveSwitchingRequest activeSwitchingRequest)
   {
      ETG_TRACE_USR4(("acquireActiveDeviceSlot() entered with deviceAddress: %s",
            activeSwitchingRequest._deviceAddress.c_str()));

      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));

      // Get the active devices list
      BdAddressList activeDevicesList;
      getActiveDevicesList(activeDevicesList);

      DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
      CallController& callController = PmCoreMainController::getInstance().getCallController();
      VRController& vrController = PmCoreMainController::getInstance().getVRController();
      BdAddress deviceToBeMadePassive = "";

      auto iter = std::find_if(_devicesToBeMadePassive.begin(), _devicesToBeMadePassive.end(),
            [&activeSwitchingRequest](std::pair<BdAddress, ActiveSwitchingRequest> const& obj)
            {return obj.second._deviceAddress == activeSwitchingRequest._deviceAddress;});

      if (_devicesToBeMadePassive.end() != iter)
      {
         ETG_TRACE_USR4(("The same device is already waiting to be switched to Active"));
         if (activeSwitchingRequest == iter->second)
         {
            ETG_TRACE_USR4(("Same request"));
            return SWITCHING_TO_ACTIVE_IN_PROGRESS;
         }
         else
         {
            if (TEMP_ROLE_SWITCH == iter->second._switchingReason)
            {
               ETG_TRACE_USR4(("TEMP_ROLE_SWITCH"));

               ControllerOriginEnum controllerOriginEnum = iter->second._passiveSwitchingControllerType;

               iter->second = activeSwitchingRequest;
               iter->second._prevPassiveSwitchingControllerType = controllerOriginEnum;

               AudioChannel audioChannelToAcquire = AM_PHONEAUDIO;

               if (ACCEPTING_CALL == iter->second._switchingReason)
               {
                  iter->second._switchingReason = TEMP_ROLE_SWITCH_AND_THEN_ACCEPTING_CALL;
                  audioChannelToAcquire = AM_PHONEAUDIO;
               }
               else if (ACTIVATING_VR == iter->second._switchingReason)
               {
                  iter->second._switchingReason = TEMP_ROLE_SWITCH_AND_THEN_ACTIVATING_VR;
                  audioChannelToAcquire = AM_VOICERECOGNITION;
               }
               else if (STARTING_WAITING_MODE == iter->second._switchingReason)
               {
                  iter->second._switchingReason = TEMP_ROLE_SWITCH_AND_THEN_STARTING_WAITING_MODE;
                  audioChannelToAcquire = AM_WAITINGMODE;
               }

               if (deviceInfoHandler.isSCOEstablished(iter->first))
               {
                  if (CALL_CONTROLLER == controllerOriginEnum)
                  {
                     callController.switchToPassive(iter->first);
                  }
                  else if (VR_CONTROLLER == iter->second._passiveSwitchingControllerType)
                  {
                     vrController.switchToPassive(iter->first, audioChannelToAcquire);
                  }

                  return SWITCHING_TO_ACTIVE_IN_PROGRESS;
               }
               // SCO is not established which implies that the required channel is released or in the releasing phase
               // by the corresponding SM
               BdAddressList activeDevicesList2;
               getActiveDevicesList(activeDevicesList2);

               ActivePassiveDeviceListMap setActivePassiveDeviceList2;

               for (auto& activeDeviceAddress : activeDevicesList2)
               {
                  if (activeDeviceAddress != iter->first)
                     setActivePassiveDeviceList2.emplace_hint(setActivePassiveDeviceList2.end(),
                           activeDeviceAddress, DEVICEROLE_ACTIVE);
               }
               setActivePassiveDeviceList2.emplace_hint(setActivePassiveDeviceList2.end(),
                     activeSwitchingRequest._deviceAddress, DEVICEROLE_ACTIVE);

               setDeviceRole(setActivePassiveDeviceList2);

               clearElementFromDeviceToBeMadePassiveMap(iter->first);

               // Swapping the device address if present for the device address "iter->first"
               PmAudioManagerWrapper::getInstance().swapDeviceAddress(iter->first,
                     activeSwitchingRequest._deviceAddress, CALL_CONTROLLER);

               return SWITCHED_TO_ACTIVE;
            }
            else
            {
               ETG_TRACE_USR4(("VR req or accept call in passive device is already in progress. Hence returning"));
               return CANNOT_BE_MADE_ACTIVE;
            }
         }

      }

      if ((ACCEPTING_CALL == activeSwitchingRequest._switchingReason) ||
            (STARTING_WAITING_MODE == activeSwitchingRequest._switchingReason))
      {
         for (auto& activeDeviceAddress : activeDevicesList)
         {
            if (!callController.isCallsInIdleState(activeDeviceAddress))
            {
               if (deviceInfoHandler.isSCOEstablished(activeDeviceAddress))
               {
                  /* "_devicesToBeMadePassive" is filled here itself to handle a special scenario.
                   * An active call is present in AG in an active device. Now accept the incoming call
                   * in a Passive device. The event "SWITCH_TO_PASSIVE" is posted to SM and since SCO
                   * is not present, the function "processSwitchToPassiveResponse" is called immediately
                   * and that function checks the device address in "_devicesToBeMadePassive"
                   */
                  deviceToBeMadePassive = activeDeviceAddress;
                  _devicesToBeMadePassive.emplace_hint(_devicesToBeMadePassive.end(),
                        deviceToBeMadePassive, activeSwitchingRequest);

                  callController.switchToPassive(activeDeviceAddress);
                  return SWITCHING_TO_ACTIVE_IN_PROGRESS;
               }

               // SCO is not established which implies that the required channel is released or in the releasing phase
               // by the corresponding SM
               BdAddressList activeDevicesList;
               getActiveDevicesList(activeDevicesList);

               ActivePassiveDeviceListMap setActivePassiveDeviceList;

               for (auto& it : activeDevicesList)
               {
                  if (it != activeDeviceAddress)
                     setActivePassiveDeviceList.emplace_hint(setActivePassiveDeviceList.end(), it, DEVICEROLE_ACTIVE);
               }
               setActivePassiveDeviceList.emplace_hint(setActivePassiveDeviceList.end(),
                     activeSwitchingRequest._deviceAddress, DEVICEROLE_ACTIVE);

               // Element is added and removed from the map "deviceToBeMadePassiveMap". So that while
               // setting the device as Active, SWITCH_TO_ACTIVE is not posted to the SM. Else the SM will try to play
               // incoming call before answering the call.
               _devicesToBeMadePassive.emplace_hint(_devicesToBeMadePassive.end(),
                     activeDeviceAddress, activeSwitchingRequest);

               setDeviceRole(setActivePassiveDeviceList);

               clearElementFromDeviceToBeMadePassiveMap(activeDeviceAddress);

               // Swapping the device address if present for the "activeDeviceAddress"
               PmAudioManagerWrapper::getInstance().swapDeviceAddress(activeDeviceAddress,
                     activeSwitchingRequest._deviceAddress, CALL_CONTROLLER);

               return SWITCHED_TO_ACTIVE;
            }
         }

         for (auto& activeDeviceAddress : activeDevicesList)
         {
            if (!vrController.isVRStatusIdle(activeDeviceAddress))
            {
               AudioChannel audioChannelToAcquire = AM_PHONEAUDIO;
               if (STARTING_WAITING_MODE == activeSwitchingRequest._switchingReason)
                  audioChannelToAcquire = AM_WAITINGMODE;

               vrController.switchToPassive(activeDeviceAddress, audioChannelToAcquire);
               deviceToBeMadePassive = activeDeviceAddress;
               break;
            }
         }

      }

      else if (ACTIVATING_VR == activeSwitchingRequest._switchingReason)
      {
         for (auto& activeDeviceAddress : activeDevicesList)
         {
            if (!vrController.isVRStatusIdle(activeDeviceAddress))
            {
               vrController.switchToPassive(activeDeviceAddress, AM_VOICERECOGNITION);
               deviceToBeMadePassive = activeDeviceAddress;
               break;
            }
         }
      }

      if (!deviceToBeMadePassive.empty())
      {
         _devicesToBeMadePassive.emplace_hint(_devicesToBeMadePassive.end(),
               deviceToBeMadePassive, activeSwitchingRequest);
         return SWITCHED_TO_ACTIVE;
      }

      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));

      return CANNOT_BE_MADE_ACTIVE;
   }

   void DeviceInfoHandler::processVRRemoved(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("processVRRemoved() entered with deviceAddress: %s", deviceAddress.c_str()));

      auto iter = _devicesToBeMadePassive.find(deviceAddress);
      if (iter != _devicesToBeMadePassive.end())
      {
         // Only if the calls are in Idle state, the device Role should be switched.
         // Else the switching should happen only when the call ends
         CallController& callController = PmCoreMainController::getInstance().getCallController();
         if (callController.isCallsInIdleState(deviceAddress))
         {
            processActiveSwitching(deviceAddress, iter->second);
            _devicesToBeMadePassive.erase(iter);
         }
      }

      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));
   }

   void DeviceInfoHandler::processAllCallsRemoved(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("processAllCallsRemoved() entered with deviceAddress: %s", deviceAddress.c_str()));

      auto iter = _devicesToBeMadePassive.find(deviceAddress);
      if (iter != _devicesToBeMadePassive.end())
      {
         processActiveSwitching(deviceAddress, iter->second);
         _devicesToBeMadePassive.erase(iter);
      }

      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));
   }

   void DeviceInfoHandler::getActiveDevicesList(BdAddressList& activeDeviceList)
   {
      for(ActivePassiveDeviceListMap::iterator it = _activePassiveDeviceList.begin();
            it != _activePassiveDeviceList.end(); it++)
      {
         if(DEVICEROLE_ACTIVE == it->second)
         {
            activeDeviceList.push_back(it->first);
         }
      }
   }

   void DeviceInfoHandler::getActivePassiveDeviceList(const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      ActivePassiveDeviceListMap activePassiveDeviceList;
      activePassiveDeviceList = _activePassiveDeviceList;

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetActivePassiveDeviceListResponse(pmResult,
            activePassiveDeviceList, act);
   }

   void DeviceInfoHandler::updateActivePassiveDeviceList()
   {
      ETG_TRACE_USR4(("updateActivePassiveDeviceList() entered"));

      // To clients
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnActivePassiveDeviceListChanged(
            _activePassiveDeviceList);

      // Updating active devices list in PM Audio manager wrapper
      BdAddressList activeDevicesList;
      getActiveDevicesList(activeDevicesList);
      PmAudioManagerWrapper::getInstance().setActiveDevicesList(activeDevicesList);
   }

   void DeviceInfoHandler::getVoiceMailListResponse(
         std::shared_ptr<PmCoreIfMessage_GetSubscriberResult> pmCoreIfMessage)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      VoiceMailList voiceMailList;

      if(BTS_REQ_SUCCESS == pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         DeviceInfoList::iterator it = _deviceInfoList.find(pmCoreIfMessage->getBdAddress());
         if(it != _deviceInfoList.end())
         {
            if(true != pmCoreIfMessage->getTelephoneNumber().empty())
            {
               VoiceMail voiceMail;
               voiceMail._numberType = pmCoreIfMessage->getNumberType();
               voiceMail._telephoneNumber = pmCoreIfMessage->getTelephoneNumber();

               voiceMailList.push_back(voiceMail);

               it->second.setVoiceMailList(voiceMailList);
            }
         }
         else
         {
            pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
         }
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
      }

      //Send the received VoiceMailList to client
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doVoiceMailListResponse(pmResult,
            pmCoreIfMessage->getBdAddress(), voiceMailList, pmCoreIfMessage->getAct());
   }

   void DeviceInfoHandler::getVoiceMailList(const BdAddress& deviceAddress, const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         //invoke GetSubscribers method request to stack
         evobtstackwrapper::EvoBtStackWrapper::getInstance().sendGetSubscriberRequest(deviceAddress, act);
      }
      else
      {
         pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
      }

      //if any error received from stack for the request, then send error response here
      if(PM_RESULT_OK != pmResult._pmResultCode)
      {
         VoiceMailList voiceMailList;
         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doVoiceMailListResponse(pmResult,
               deviceAddress, voiceMailList, act);
      }
   }

   PmResult DeviceInfoHandler::getDeviceInfoList(DeviceInfoList& deviceInfoList)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      deviceInfoList = _deviceInfoList;

      return pmResult;
   }

   void DeviceInfoHandler::getSCOConnectionRequest(IN const BdAddress& deviceAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      SCOStatus scoStatus = SCO_DEFAULT;

      auto deviceSCOStatusIter = _deviceSCOStatus.find(deviceAddress);
      if (deviceSCOStatusIter != _deviceSCOStatus.end())
      {
         scoStatus = _deviceSCOStatus[deviceAddress];
      }

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetSCOConnectionResponse(
            pmResult, deviceAddress, SCOConnection(scoStatus), act);
   }

   bool DeviceInfoHandler::addDeviceToActivePassiveDeviceList(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("addDeviceToActivePassiveDeviceList() deviceAddress : %s", deviceAddress.c_str()));

      printActivePassiveDeviceList();

      bool retValue = true;
      DeviceRole deviceRole = DEVICEROLE_PASSIVE;
      DevicesCount maxNumActiveDevices = com::bosch::pmcommon::PmConfiguration::getInstance().getMaxNumActiveDevices();

      ActivePassiveDeviceListMap::iterator it = _activePassiveDeviceList.find(deviceAddress);

      if(it != _activePassiveDeviceList.end())
      {
         ETG_TRACE_USR4(("Device is already available in the ActivePassiveDeviceList"));
         retValue = false;
      }
      else
      {
         //1. if first device connected, then the same should be updated as active.
         //2. if number of active devices are less than the maximum allowed limit,
         //   then the device role should be updated as active.
         //3. if number of active devices are reached the maximum limit allowed,
         //   then the device role should be updated as passive.

         if(getNumberOfActiveDevices() < maxNumActiveDevices)
         {
            deviceRole = DEVICEROLE_ACTIVE;
         }

         _activePassiveDeviceList.emplace_hint(_activePassiveDeviceList.end(), deviceAddress, deviceRole);

         auto deviceInfoListIter = _deviceInfoList.find(deviceAddress);
         if(deviceInfoListIter != _deviceInfoList.end())
         {
            deviceInfoListIter->second.setDeviceRole(deviceRole);
         }
      }

      printActivePassiveDeviceList();
      return retValue;
   }

   bool DeviceInfoHandler::removeDeviceFromActivePassiveDeviceList(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("removeDeviceFromActivePassiveDeviceList() deviceAddress : %s", deviceAddress.c_str()));

      printActivePassiveDeviceList();

      bool retValue = true;
      ActivePassiveDeviceListMap::iterator it = _activePassiveDeviceList.find(deviceAddress);

      if(it != _activePassiveDeviceList.end())
      {
         _activePassiveDeviceList.erase(it);
      }
      else
      {
         ETG_TRACE_USR4(("Device is not available in the ActivePassiveDeviceList"));
         retValue = false;
      }

      printActivePassiveDeviceList();
      return retValue;
   }

   void DeviceInfoHandler::clearActivePassiveDeviceList()
   {
      ETG_TRACE_USR4(("clearActivePassiveDeviceList() entered"));

      printActivePassiveDeviceList();
      _activePassiveDeviceList.clear();
   }

   DevicesCount DeviceInfoHandler::getNumberOfActiveDevices() const
   {
      DevicesCount count = 0;

      for (auto& activePassiveDeviceListElement : _activePassiveDeviceList)
      {
         if (DEVICEROLE_ACTIVE == activePassiveDeviceListElement.second)
         {
            ++count;
         }
      }

      ETG_TRACE_USR4(("getNumberOfActiveDevices- count : %u", count));
      return count;
   }

   bool DeviceInfoHandler::addDeviceToDeviceInfoList(const DeviceHandle deviceHandle, const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("addDeviceToDeviceInfoList() deviceAddress : %s", deviceAddress.c_str()));

      bool retValue = true;
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         ETG_TRACE_USR4(("Device is already available in the DeviceInfoList"));
         retValue = false;
      }
      else
      {
         DeviceInfo deviceInfo(deviceHandle, deviceAddress);
         _deviceInfoList.insert(std::pair<BdAddress, DeviceInfo>(deviceAddress, deviceInfo));
      }

      return retValue;
   }

   bool DeviceInfoHandler::removeDeviceFromDeviceInfoList(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("removeDeviceFromDeviceInfoList() deviceAddress : %s", deviceAddress.c_str()));

      bool retValue = true;
      DeviceInfoList::iterator it = _deviceInfoList.find(deviceAddress);

      if(it != _deviceInfoList.end())
      {
         _deviceInfoList.erase(it);
      }
      else
      {
         ETG_TRACE_USR4(("Device is not available in the DeviceInfoList"));
         retValue = false;
      }

      return retValue;
   }

   void DeviceInfoHandler::clearDeviceInfoList()
   {
      ETG_TRACE_USR4(("clearDeviceInfoList() entered"));

      _deviceInfoList.clear();
   }

   bool DeviceInfoHandler::processSwitchToPassiveResponse(const BdAddress deviceAddress,
         const PmResult pmResult, const bool erasePair)
   {
      ETG_TRACE_USR4(("processSwitchToPassiveResponse() entered"));

      auto iter = _devicesToBeMadePassive.find(deviceAddress);
      if (_devicesToBeMadePassive.end() == iter)
      {
         return false;
      }

      if ("INPROGRESS" == pmResult._pmResultMessage)
      {
         ETG_TRACE_USR4(("processSwitchToPassiveResponse() : INPROGRESS"));
         return true;
      }

      bool eraseIter = true;

      ActiveSwitchingRequest activeSwitchingRequest = iter->second;

      if (PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("processSwitchToPassiveResponse Error response: %u",
               ETG_CENUM(PmResultCode, pmResult._pmResultCode)));

         ETG_TRACE_USR4(("processSwitchToPassiveResponse: switchingReason: %u",
               ETG_CENUM(SwitchingReason, activeSwitchingRequest._switchingReason)));

         switch (activeSwitchingRequest._switchingReason)
         {
            case ACCEPTING_CALL:
            {
               PmCoreMainController::getInstance().postResponseForRequest("AcceptCall", pmResult,
                     iter->second._deviceAddress, iter->second._pendingRequestInfo._callInstance,
                     iter->second._pendingRequestInfo._act);

               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = iter->second._deviceAddress;
               notificationEvent._eventName = ACCEPT_CALL_FAILED;
               notificationEvent._reason = pmResult._pmResultMessage;
               notificationEvent._callInstanceList.push_back(iter->second._pendingRequestInfo._callInstance);
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);
            }
            break;

            case ACTIVATING_VR:
            {
               PmCoreMainController::getInstance().postResponseForRequest("StartVoiceRecognition", pmResult,
                     iter->second._deviceAddress, iter->second._pendingRequestInfo._callInstance,
                     iter->second._pendingRequestInfo._act);

               // TODO: Update Error for VR activation request
            }
            break;

            case STARTING_WAITING_MODE:
            {
               PmCoreMainController::getInstance().postResponseForRequest("StartWaitingMode", pmResult,
                     iter->second._deviceAddress, iter->second._pendingRequestInfo._callInstance,
                     iter->second._pendingRequestInfo._act);

               // TODO: Update Error for Starting waiting mode
            }
            break;

            case TEMP_ROLE_SWITCH_AND_THEN_ACCEPTING_CALL:
            {
               PmCoreMainController::getInstance().postResponseForRequest("AcceptCall", pmResult,
                     iter->second._deviceAddress, iter->second._pendingRequestInfo._callInstance,
                     iter->second._pendingRequestInfo._act);

               NotificationEvent notificationEvent;
               notificationEvent._bdAddress = iter->second._deviceAddress;
               notificationEvent._eventName = ACCEPT_CALL_FAILED;
               notificationEvent._reason = pmResult._pmResultMessage;
               notificationEvent._callInstanceList.push_back(iter->second._pendingRequestInfo._callInstance);
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(
                     notificationEvent);

               iter->second._switchingReason = TEMP_ROLE_SWITCH;
               iter->second._passiveSwitchingControllerType = iter->second._prevPassiveSwitchingControllerType;
               eraseIter = false;
               // because the currently active device should be made passive once the call is completely ended
            }
            break;

            case TEMP_ROLE_SWITCH_AND_THEN_ACTIVATING_VR:
            {
               PmCoreMainController::getInstance().postResponseForRequest("StartVoiceRecognition", pmResult,
                     iter->second._deviceAddress, iter->second._pendingRequestInfo._callInstance,
                     iter->second._pendingRequestInfo._act);

               // TODO: Update Error for VR activation request

               iter->second._switchingReason = TEMP_ROLE_SWITCH;
               iter->second._passiveSwitchingControllerType = iter->second._prevPassiveSwitchingControllerType;
               eraseIter = false;
               // because the currently active device should be made passive once the call is completely ended
            }
            break;

            case TEMP_ROLE_SWITCH_AND_THEN_STARTING_WAITING_MODE:
            {
               PmCoreMainController::getInstance().postResponseForRequest("StartWaitingMode", pmResult,
                     iter->second._deviceAddress, iter->second._pendingRequestInfo._callInstance,
                     iter->second._pendingRequestInfo._act);

               // TODO: Update Error for VR activation request
               iter->second._switchingReason = TEMP_ROLE_SWITCH;
               iter->second._passiveSwitchingControllerType = iter->second._prevPassiveSwitchingControllerType;
               eraseIter = false;
               // because the currently active device should be made passive once the call is completely ended
            }
            break;

            default:
            {
               ETG_TRACE_USR4(("activeSwitchingRequest._switchingReason: Default case"));
            }
         }
      }
      else
      {
         processActiveSwitching(deviceAddress, activeSwitchingRequest);
      }

      if (erasePair && eraseIter)
      {
         _devicesToBeMadePassive.erase(iter);
      }

      ETG_TRACE_USR4(("_devicesToBeMadePassive.size(): %u", _devicesToBeMadePassive.size()));

      return true;
   }

   void DeviceInfoHandler::processActiveSwitching(const BdAddress devicetoPassive,
         const ActiveSwitchingRequest activeSwitchingRequest)
   {
      ETG_TRACE_USR4(("processActiveSwitching() entered with devicetoPassive: %s", devicetoPassive.c_str()));

      BdAddressList activeDevicesList;
      getActiveDevicesList(activeDevicesList);

      ActivePassiveDeviceListMap activePassiveDevicelist;
      for(auto& activeDevicesListIter : activeDevicesList)
      {
         if (activeDevicesListIter != devicetoPassive)
            activePassiveDevicelist.emplace_hint(activePassiveDevicelist.end(), activeDevicesListIter,
                  DEVICEROLE_ACTIVE);
      }
      ETG_TRACE_USR4(("activeSwitchingRequest._deviceAddress: %s", activeSwitchingRequest._deviceAddress.c_str()));
      activePassiveDevicelist.emplace_hint(activePassiveDevicelist.end(), activeSwitchingRequest._deviceAddress,
            DEVICEROLE_ACTIVE);

      // IMPORTANT: Consider this scenario: All calls were removed in an Active device and stop Audio success is also
      // received. But the session is not yet removed. Hence if this swapping is not done here, a new session will be
      // requested by PM AudioManagerWrapper.
      // And in the below if else statements, this swap is done. Hence it is done generically here.
      PmAudioManagerWrapper::getInstance().swapDeviceAddress(devicetoPassive,
            activeSwitchingRequest._deviceAddress, CALL_CONTROLLER);

      setDeviceRole(activePassiveDevicelist);

      ETG_TRACE_USR4(("activeSwitchingRequest._switchingReason: %u",
            ETG_CENUM(SwitchingReason, activeSwitchingRequest._switchingReason)));

      if (ACCEPTING_CALL == activeSwitchingRequest._switchingReason ||
            TEMP_ROLE_SWITCH_AND_THEN_ACCEPTING_CALL == activeSwitchingRequest._switchingReason)
      {
         ETG_TRACE_USR4(("ACCEPTING_CALL"));

         PendingRequestInfo acceptCallRequest = activeSwitchingRequest._pendingRequestInfo;
         PmInterface::getInstance().acceptCall(activeSwitchingRequest._deviceAddress, acceptCallRequest._callInstance,
               acceptCallRequest._acceptOperation, acceptCallRequest._act);
      }
      else if ((ACTIVATING_VR == activeSwitchingRequest._switchingReason) ||
            TEMP_ROLE_SWITCH_AND_THEN_ACTIVATING_VR == activeSwitchingRequest._switchingReason)
      {
         ETG_TRACE_USR4(("ACTIVATING_VR"));

         PmInterface::getInstance().startStopVoiceRecognition(activeSwitchingRequest._deviceAddress, true,
               activeSwitchingRequest._pendingRequestInfo._act);
      }
      else if ((STARTING_WAITING_MODE == activeSwitchingRequest._switchingReason) ||
            TEMP_ROLE_SWITCH_AND_THEN_STARTING_WAITING_MODE == activeSwitchingRequest._switchingReason)
      {
         ETG_TRACE_USR4(("STARTING_WAITING_MODE"));

         PmInterface::getInstance().startStopWaitingMode(activeSwitchingRequest._deviceAddress, true,
               activeSwitchingRequest._pendingRequestInfo._act);
      }
   }

   void DeviceInfoHandler::printActivePassiveDeviceList()
   {
      ETG_TRACE_USR4(("printActivePassiveDeviceList() entered"));

      for (auto& iter : _activePassiveDeviceList)
      {
         ETG_TRACE_USR4(("ActivePassiveDeviceList: DeviceAddress: %s", iter.first.c_str()));
         ETG_TRACE_USR4(("ActivePassiveDeviceList: DeviceRole: %u", ETG_CENUM(DeviceRoleEnumType,
               iter.second)));
      }
   }

   bool DeviceInfoHandler::isSCOEstablished(const BdAddress& deviceAddress)
   {
      auto iter = _deviceSCOStatus.find(deviceAddress);
      if (iter != _deviceSCOStatus.end())
      {
         if ((SCO_NARROWBAND == _deviceSCOStatus[deviceAddress]) ||
               (SCO_WIDEBAND == _deviceSCOStatus[deviceAddress]))
         {
            return true;
         }
      }
      return false;
   }

   bool DeviceInfoHandler::isSwitchReasonAcceptCallOrStartWaitMode(const SwitchingReason switchingReason)
   {
      ETG_TRACE_USR4(("isSwitchReasonAcceptCallOrStartWaitMode() entered"));

      if ((ACCEPTING_CALL == switchingReason) || (TEMP_ROLE_SWITCH_AND_THEN_ACCEPTING_CALL == switchingReason) ||
            (STARTING_WAITING_MODE == switchingReason) ||
            (TEMP_ROLE_SWITCH_AND_THEN_STARTING_WAITING_MODE == switchingReason))
      {
         return true;
      }
      return false;
   }

   bool DeviceInfoHandler::isDeviceInIdleState(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("isDeviceInIdleState() entered with deviceAddress: %20s", deviceAddress.c_str()));

      bool deviceInIdleState = false;

      if ((PmCoreMainController::getInstance().getCallController().isCallsInIdleState(deviceAddress)) &&
            (PmCoreMainController::getInstance().getVRController().isVRStatusIdle(deviceAddress)) &&
            (PmCoreMainController::getInstance().getVRController().isExternalVRStatusIdle(deviceAddress)))
      {
         deviceInIdleState = true;
      }
      return deviceInIdleState;
   }

}
