/**
 * @file DbHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the DbHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup PmCore
 */

#include "DbHandler.h"
#include "FileUtils.h"
#include <TimeTrace.h>
#include <errno.h>
#include <sys/stat.h>
#include <openssl/sha.h>
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DbHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

namespace pmcore
{
   #define QUICK_CHECK 1

   #define PM_DB_FILEPATH      "/var/opt/bosch/dynamic/connectivity/db/PM.db"
   #define PM_DB_RECREATEFILE  "/var/opt/bosch/dynamic/connectivity/db/PM.db.recreate"
   #define PM_DB_USERNAME      "aid_phone"
   #define PM_DB_GROUPNAME     "aid_phone"

   #define PM_DB_TABLENAME_SYSTEMWIDESETTINGS "SystemWideSettings"
   #define PM_DB_TABLENAME_DEVICESETTINGS     "DeviceSettings"
   #define PM_DB_TABLENAME_SCHEMAVERSION      "SchemaVersion"

   #define PM_DB_COLUMNNAME_ACTIVERINGTONEID  "ActiveRingtoneId"
   #define PM_DB_COLUMNNAME_SUPPRESSRINGTONE  "SuppressRingtone"
   #define PM_DB_COLUMNNAME_AUTOWAITINGMODE   "AutoWaitingMode"
   #define PM_DB_COLUMNNAME_DEVICEHANDLE      "DeviceHandle"
   #define PM_DB_COLUMNNAME_DEVICEADDRESS     "DeviceAddress"
   #define PM_DB_COLUMNNAME_RINGTONEID        "RingtoneId"
   #define PM_DB_COLUMNNAME_VERSION           "Version"
   #define PM_DB_COLUMNNAME_INTEGRITY         "Integrity"
   #define PM_DB_COLUMNNAME_SCHEMAHASH        "SchemaHash"

   #define PM_DB_SCHEMA_VERSION           1

   #define PM_DB_BIND_INDEX_1             1
   #define PM_DB_BIND_INDEX_2             2
   #define PM_DB_BIND_INDEX_3             3
   #define PM_DB_BIND_INDEX_4             4
   #define PM_DB_BIND_INDEX_5             5

   #define PM_DB_GETRESULT_COLUMN_0       0
   #define PM_DB_GETRESULT_COLUMN_1       1
   #define PM_DB_GETRESULT_COLUMN_2       2

   #define PM_DB_CHECK_QUERY_ERROR(sqlErrorCode) checkDbQueryError(sqlErrorCode, __LINE__, __FILE__, __FUNCTION__);
   #define PM_DB_ERROR_NOSUCHTABLE               "no such table"

   DbHandler::DbHandler(): _dbName(PM_DB_FILEPATH)
   {
      ETG_TRACE_USR1(("DbHandler: Constructor"));
   }

   DbHandler::~DbHandler()
   {
      ETG_TRACE_USR1(("DbHandler: Destructor"));
   }

   bool DbHandler::openDatabase()
   {
      ETG_TRACE_USR1(("openDatabase() entered"));

      bool result = false;

      // check if a previous run wants us to recreate the database because of a malformed error
      FILE *fp;
      fp = fopen(PM_DB_RECREATEFILE, "r");

      if (!fp)
      {
         ETG_TRACE_FATAL(("openDatabase() file \"%s\" does not exist => DB is not requested to be recreated",
               PM_DB_RECREATEFILE));
         ETG_TRACE_ERRMEM(("openDatabase() file \"%s\" does not exist => DB is not requested to be recreated",
                        PM_DB_RECREATEFILE));

         result = open(_dbName);

         if(true == result)
         {
            ETG_TRACE_USR4(("openDatabase() success"));

            result = checkSchemaHash();

            result = result && checkDatabaseIntegrity();

            if(false == result)
            {
               ETG_TRACE_USR4(("openDatabase() database integrity failed"));
               (void)close();
            }
            else
            {
               checkAndAdaptDbSettings();
            }
         }
         else
         {
            ETG_TRACE_ERR(("openDatabase() open error"));
         }
      }
      else
      {
         // recreate file exists => DB file has to be recreated
         ETG_TRACE_FATAL(("openDatabase() file \"%s\" exists, DB is requested to be recreated", PM_DB_RECREATEFILE));
         ETG_TRACE_ERRMEM(("openDatabase() file \"%s\" exists, DB is requested to be recreated", PM_DB_RECREATEFILE));

         // close and remove the recreation trigger file
         fclose(fp);

         ETG_TRACE_FATAL(("openDatabase() removing file \"%s\"", PM_DB_RECREATEFILE));

         int ret = remove(PM_DB_RECREATEFILE);

         if (0 == ret)
         {
            ETG_TRACE_FATAL(("openDatabase() removed file \"%s\"", PM_DB_RECREATEFILE));
         }
         else
         {
            ETG_TRACE_FATAL(("openDatabase() could not remove file \"%s\"", PM_DB_RECREATEFILE));
         }
      }

      if(false == result)
      {
         ETG_TRACE_ERR(("openDatabase() recreating the database"));

         result = recreateDatabase();
      }

      return result;
   }

   void DbHandler::closeDatabase()
   {
      ETG_TRACE_USR1(("closeDatabase() entered"));

      (void)close();
   }

   bool DbHandler::addDeviceInDb(const DeviceHandle deviceHandle, const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR1(("addDeviceInDb() DeviceHandle : %d, DeviceAddress : %s", deviceHandle, deviceAddress.c_str()));

      std::string query = "SELECT " PM_DB_COLUMNNAME_DEVICEHANDLE " FROM " PM_DB_TABLENAME_DEVICESETTINGS
            " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS " = \"" + deviceAddress + "\"";

      bool result = false;
      int deviceHandleInDb = 0;
      int queryResult = SQLITE_ERROR;

      queryResult = prepare(query);

      if(SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            if(SQLITE_ROW == queryResult)
            {
               getInt(PM_DB_GETRESULT_COLUMN_0, deviceHandleInDb);
            }
            else
            {
               break;
            }
         }
         queryResult = finalize();

         if(deviceHandle == deviceHandleInDb)
         {
            ETG_TRACE_ERR(("addDeviceInDb() device already exists"));
            result = false;
         }
         else
         {
            if(0 == deviceHandleInDb)
            {
               ETG_TRACE_USR1(("addDeviceInDb() adding new device to database"));

               // Check SystemWideSetting table and align the DeviceSetting
               int ringtoneIdInDb = -1;
               int suppressRingtoneInDb = 0;
               int autoWaitingModeInDb = 0;

               query = "SELECT " PM_DB_COLUMNNAME_ACTIVERINGTONEID ", "
                     PM_DB_COLUMNNAME_SUPPRESSRINGTONE ", "
                     PM_DB_COLUMNNAME_AUTOWAITINGMODE " FROM " PM_DB_TABLENAME_SYSTEMWIDESETTINGS;

               queryResult = prepare(query);

               if(SQLITE_OK == queryResult)
               {
                  while(1)
                  {
                     queryResult = step();

                     if(SQLITE_ROW == queryResult)
                     {
                        getInt(PM_DB_GETRESULT_COLUMN_0, ringtoneIdInDb);
                        getInt(PM_DB_GETRESULT_COLUMN_1, suppressRingtoneInDb);
                        getInt(PM_DB_GETRESULT_COLUMN_2, autoWaitingModeInDb);
                     }
                     else
                     {
                        break;
                     }
                  }
                  queryResult = finalize();
               }
               else
               {
                  ETG_TRACE_ERR(("addDeviceInDb() fetching SystemWideSetting failed"));
               }

               query = "INSERT INTO " PM_DB_TABLENAME_DEVICESETTINGS " ( "
                     PM_DB_COLUMNNAME_DEVICEHANDLE ", "
                     PM_DB_COLUMNNAME_DEVICEADDRESS ", "
                     PM_DB_COLUMNNAME_RINGTONEID ", "
                     PM_DB_COLUMNNAME_SUPPRESSRINGTONE ", "
                     PM_DB_COLUMNNAME_AUTOWAITINGMODE " ) "
                     "VALUES ( " + std::to_string(deviceHandle) + ", \"" + deviceAddress + "\", " +
                     std::to_string(ringtoneIdInDb) + ", " + std::to_string(0) + ", " + std::to_string(0) + " );";
            }
            else
            {
               ETG_TRACE_USR1(("addDeviceInDb() device handle updated"));

               query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET "
                     PM_DB_COLUMNNAME_DEVICEHANDLE " = " + std::to_string(deviceHandle) +
                     " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS " = \"" + deviceAddress + "\"";
            }

            queryResult = exec(query);

            if(SQLITE_OK != queryResult)
            {
               ETG_TRACE_ERR(("addDeviceInDb() execute query failed"));
               PM_DB_CHECK_QUERY_ERROR(queryResult);
            }
            else
            {
               result = true;
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("addDeviceInDb() prepare query failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return result;
   }

   bool DbHandler::removeDeviceFromDb(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR1(("removeDeviceFromDb() DeviceAddress : %s", deviceAddress.c_str()));

      bool result = true;
      int queryResult = SQLITE_ERROR;
      std::string query;

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         query = "DELETE FROM " PM_DB_TABLENAME_DEVICESETTINGS;
      }
      else
      {
         query = "DELETE FROM " PM_DB_TABLENAME_DEVICESETTINGS
               " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS " = \"" + deviceAddress + "\"";
      }

      queryResult = exec(query);

      if(SQLITE_OK != queryResult)
      {
         ETG_TRACE_ERR(("removeDeviceFromDb() execute query failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
         result = false;
      }

      return result;
   }

   bool DbHandler::getDevicesInDb(std::map<DeviceHandle, BdAddress>& deviceList)
   {
      ETG_TRACE_USR1(("getDevicesInDb() entered"));

      bool result = true;

      std::string query("SELECT " PM_DB_COLUMNNAME_DEVICEHANDLE " , "
            PM_DB_COLUMNNAME_DEVICEADDRESS " FROM " PM_DB_TABLENAME_DEVICESETTINGS);

      int queryResult = prepare(query);

      if(SQLITE_OK == queryResult)
      {
         while (1)
         {
            queryResult = step();

            if(SQLITE_ROW == queryResult)
            {
               int deviceHandle = 0;
               BdAddress deviceAddress;

               getInt(PM_DB_GETRESULT_COLUMN_0, deviceHandle);
               getString(PM_DB_GETRESULT_COLUMN_1, deviceAddress);

               if((0 != deviceHandle) && (false == deviceAddress.empty()))
               {
                  deviceList.insert(std::pair<DeviceHandle, BdAddress>((DeviceHandle)deviceHandle, deviceAddress));
               }
            }
            else
            {
               break;
            }
         }
         queryResult = finalize();
      }
      else
      {
         ETG_TRACE_ERR(("getDevicesInDb() prepare query failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
         result = false;
      }

      return result;
   }

   bool DbHandler::restoreDefaultSetting(const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR1(("restoreDefaultSetting() DeviceAddress : %s", deviceAddress.c_str()));

      std::string query;
      int queryResult = SQLITE_ERROR;
      bool result = true;

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_DELETE_ALL))
      {
         if(removeDeviceFromDb(DEVICE_ADDRESS_ALL))
         {
            ETG_TRACE_USR1(("restoreDefaultSetting() successfully deleted all devices in DeviceSettings"));

            query = "UPDATE " PM_DB_TABLENAME_SYSTEMWIDESETTINGS " SET " PM_DB_COLUMNNAME_ACTIVERINGTONEID " = -1, "
                  PM_DB_COLUMNNAME_SUPPRESSRINGTONE " = 0, "
                  PM_DB_COLUMNNAME_AUTOWAITINGMODE " = 0;";

            queryResult = exec(query);
         }
         else
         {
            ETG_TRACE_ERR(("restoreDefaultSetting() failed to delete all devices in DeviceSettings"));

            result = false;
         }
      }
      else if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         //Update the setting for all devices
         query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET " PM_DB_COLUMNNAME_RINGTONEID " = -1,"
               PM_DB_COLUMNNAME_SUPPRESSRINGTONE " = 0, "
               PM_DB_COLUMNNAME_AUTOWAITINGMODE " = 0;";

         queryResult = exec(query);

         if(SQLITE_OK == queryResult)
         {
            ETG_TRACE_USR1(("restoreDefaultSetting() successful for all devices in DeviceSettings"));

            query = "UPDATE " PM_DB_TABLENAME_SYSTEMWIDESETTINGS " SET " PM_DB_COLUMNNAME_ACTIVERINGTONEID " = -1, "
                  PM_DB_COLUMNNAME_SUPPRESSRINGTONE " = 0, "
                  PM_DB_COLUMNNAME_AUTOWAITINGMODE " = 0;";

            queryResult = exec(query);
         }
         else
         {
            ETG_TRACE_ERR(("restoreDefaultSetting() failed for all devices in DeviceSettings"));

            result = false;
         }
      }
      else
      {
         query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET "
               PM_DB_COLUMNNAME_RINGTONEID " = -1,"
               PM_DB_COLUMNNAME_SUPPRESSRINGTONE " = 0, "
               PM_DB_COLUMNNAME_AUTOWAITINGMODE " = 0"
               " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS " = \"" + deviceAddress + "\"";

         queryResult = exec(query);
      }

      if((true == result) && (SQLITE_OK == queryResult))
      {
         ETG_TRACE_USR1(("restoreDefaultSetting() success"));
      }
      else
      {
         ETG_TRACE_ERR(("restoreDefaultSetting() failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
         result = false;
      }

      return result;
   }

   bool DbHandler::setRingtoneId(const BdAddress& deviceAddress, const RingtoneId ringtoneId)
   {
      ETG_TRACE_USR1(("setRingtoneId() DeviceAddress : %15s, RingtoneId : %d", deviceAddress.c_str(), ringtoneId));

      std::string query;
      int queryResult = SQLITE_ERROR;
      bool result = true;

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         // Update the setting for all devices
         query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET " PM_DB_COLUMNNAME_RINGTONEID " = " +
               std::to_string(ringtoneId);

         queryResult = exec(query);

         if(SQLITE_OK == queryResult)
         {
            ETG_TRACE_USR1(("setRingtoneId() successful for all devices in DeviceSettings"));

            //Update the same in SystemWideSettingTable
            query = "UPDATE " PM_DB_TABLENAME_SYSTEMWIDESETTINGS " SET " PM_DB_COLUMNNAME_ACTIVERINGTONEID " = " +
                  std::to_string(ringtoneId);

            queryResult = exec(query);
         }
         else
         {
            ETG_TRACE_ERR(("setRingtoneId() failed for all devices in DeviceSettings"));

            result = false;
         }
      }
      else
      {
         query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET " PM_DB_COLUMNNAME_RINGTONEID " = " +
               std::to_string(ringtoneId) + " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS " = \"" + deviceAddress + "\"";

         queryResult = exec(query);
      }

      if(SQLITE_OK == queryResult)
      {
         ETG_TRACE_USR1(("setRingtoneId() success"));
      }
      else
      {
         ETG_TRACE_ERR(("setRingtoneId() failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
         result = false;
      }

      return result;
   }

   bool DbHandler::getRingtoneId(const BdAddress& deviceAddress, RingtoneId& ringtoneId)
   {
      ETG_TRACE_USR1(("getRingtoneId() DeviceAddress : %s", deviceAddress.c_str()));

      std::string query;
      int queryResult = SQLITE_ERROR;
      bool retValue = false;

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         query = "SELECT " PM_DB_COLUMNNAME_ACTIVERINGTONEID " FROM " PM_DB_TABLENAME_SYSTEMWIDESETTINGS;
      }
      else
      {
         query = "SELECT " PM_DB_COLUMNNAME_RINGTONEID " FROM " PM_DB_TABLENAME_DEVICESETTINGS
               " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS " = \"" + deviceAddress + "\"";
      }

      queryResult = prepare(query);

      if (SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            if(SQLITE_ROW == queryResult)
            {
               //this should be run only once
               int ringtoneIdInDb = 0;

               getInt(PM_DB_GETRESULT_COLUMN_0, ringtoneIdInDb);

               ringtoneId = (RingtoneId)ringtoneIdInDb;
            }
            else if(SQLITE_DONE == queryResult)
            {
               retValue = true;
               break;
            }
            else
            {
               break;
            }
         }

         (void)finalize();
      }
      else
      {
         ETG_TRACE_ERR(("getRingtoneId() failed"));

         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return retValue;
   }

   bool DbHandler::getDeviceRingtoneIdList(DeviceRingtoneListType& deviceRingtoneList)
   {
      ETG_TRACE_USR1(("getDeviceRingtoneIdList() entered"));

      int queryResult = SQLITE_ERROR;
      bool retValue = false;

      std::string query("SELECT " PM_DB_COLUMNNAME_DEVICEADDRESS "," PM_DB_COLUMNNAME_RINGTONEID " FROM "
            PM_DB_TABLENAME_DEVICESETTINGS);

      queryResult = prepare(query);

      if (SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            if(SQLITE_ROW == queryResult)
            {
               int ringtoneIdInDb = 0;
               DeviceRingtone deviceRingtone;

               getString(PM_DB_GETRESULT_COLUMN_0, deviceRingtone._deviceAddress);
               getInt(PM_DB_GETRESULT_COLUMN_1, ringtoneIdInDb);

               deviceRingtone._ringtoneId = (RingtoneId)ringtoneIdInDb;

               deviceRingtoneList.push_back(deviceRingtone);
            }
            else if(SQLITE_DONE == queryResult)
            {
               retValue = true;
               break;
            }
            else
            {
               break;
            }
         }

         (void)finalize();
      }
      else
      {
         ETG_TRACE_ERR(("getDeviceRingtoneIdList() failed"));

         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return retValue;
   }

   bool DbHandler::setSuppressRingtoneSetting(const BdAddress& deviceAddress, const bool suppressRingtone)
   {
      ETG_TRACE_USR1(("setSuppressRingtoneSetting() DeviceAddress : %15s, SuppressRingtone : %d",
            deviceAddress.c_str(), suppressRingtone));

      std::string query;
      int queryResult = SQLITE_ERROR;
      bool result = true;

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         //Update the setting for all devices
         query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET " PM_DB_COLUMNNAME_SUPPRESSRINGTONE " = " +
               std::to_string((int)suppressRingtone);

         queryResult = exec(query);

         if(SQLITE_OK == queryResult)
         {
            ETG_TRACE_USR1(("setSuppressRingtoneSetting() successful for all devices in DeviceSettings"));

            //Update the same in SystemWideSettingTable
            query = "UPDATE " PM_DB_TABLENAME_SYSTEMWIDESETTINGS " SET " PM_DB_COLUMNNAME_SUPPRESSRINGTONE " = " +
                  std::to_string((int)suppressRingtone);

            queryResult = exec(query);
         }
         else
         {
            ETG_TRACE_ERR(("setSuppressRingtoneSetting() failed for all devices in DeviceSettings"));

            result = false;
         }
      }
      else
      {
         query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET " PM_DB_COLUMNNAME_SUPPRESSRINGTONE " = " +
               std::to_string((int)suppressRingtone) + " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS "=\"" + deviceAddress + "\"";

         queryResult = exec(query);
      }

      if((true == result) && (SQLITE_OK == queryResult))
      {
         ETG_TRACE_USR1(("setSuppressRingtoneSetting() success"));
      }
      else
      {
         ETG_TRACE_ERR(("setSuppressRingtoneSetting() failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
         result = false;
      }

      return result;
   }

   bool DbHandler::getSuppressRingtoneSetting(const BdAddress& deviceAddress, bool& suppressRingtone)
   {
      ETG_TRACE_USR1(("getSuppressRingtoneSetting() DeviceAddress : %s", deviceAddress.c_str()));

      std::string query;
      int queryResult = SQLITE_ERROR;
      bool retValue = false;

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         query = "SELECT " PM_DB_COLUMNNAME_SUPPRESSRINGTONE " FROM " PM_DB_TABLENAME_SYSTEMWIDESETTINGS;
      }
      else
      {
         query = "SELECT " PM_DB_COLUMNNAME_SUPPRESSRINGTONE " FROM " PM_DB_TABLENAME_DEVICESETTINGS
               " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS " = \"" + deviceAddress + "\"";
      }

      queryResult = prepare(query);

      if (SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            if(SQLITE_ROW == queryResult)
            {
               //this should be run only once
               int suppressRingtoneInDb = 0;

               getInt(PM_DB_GETRESULT_COLUMN_0, suppressRingtoneInDb);

               suppressRingtone = (bool)suppressRingtoneInDb;
               retValue = true;
            }
            else
            {
               break;
            }
         }

         (void)finalize();
      }
      else
      {
         ETG_TRACE_ERR(("getSuppressRingtoneSetting() failed"));

         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return retValue;
   }

   bool DbHandler::getSuppressRingtoneSettingList(SuppressRingtoneOnOffListMap& suppressRingtoneOnOffList)
   {
      ETG_TRACE_USR1(("getSuppressRingtoneSettingList() entered"));

      int queryResult = SQLITE_ERROR;
      bool retValue = false;

      std::string query("SELECT " PM_DB_COLUMNNAME_DEVICEADDRESS "," PM_DB_COLUMNNAME_SUPPRESSRINGTONE " FROM "
            PM_DB_TABLENAME_DEVICESETTINGS);

      queryResult = prepare(query);

      if (SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            if(SQLITE_ROW == queryResult)
            {
               int suppressRingtoneInDb = 0;
               BdAddress deviceAddress;

               getString(PM_DB_GETRESULT_COLUMN_0, deviceAddress);
               getInt(PM_DB_GETRESULT_COLUMN_1, suppressRingtoneInDb);

               suppressRingtoneOnOffList.insert(std::pair<BdAddress, SuppressRingtoneState>(
                     deviceAddress, (SuppressRingtoneState)suppressRingtoneInDb));
            }
            else if(SQLITE_DONE == queryResult)
            {
               retValue = true;
               break;
            }
            else
            {
               break;
            }
         }

         (void)finalize();
      }
      else
      {
         ETG_TRACE_ERR(("getSuppressRingtoneSettingList() failed"));

         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return retValue;
   }

   bool DbHandler::setAutoWaitingModeSetting(const BdAddress& deviceAddress, const bool autoWaitingMode)
   {
      ETG_TRACE_USR1(("setAutoWaitingModeSetting() DeviceAddress : %15s, AutoWaitingMode : %d",
            deviceAddress.c_str(), autoWaitingMode));

      std::string query;
      int queryResult = SQLITE_ERROR;
      bool result = true;

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         //Update the setting for all devices
         query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET " PM_DB_COLUMNNAME_AUTOWAITINGMODE " = " +
               std::to_string((int)autoWaitingMode);

         queryResult = exec(query);

         if(SQLITE_OK == queryResult)
         {
            ETG_TRACE_USR1(("setAutoWaitingModeSetting() successful for all devices in DeviceSettings"));

            //Update the same in SystemWideSettingTable
            query = "UPDATE " PM_DB_TABLENAME_SYSTEMWIDESETTINGS " SET " PM_DB_COLUMNNAME_AUTOWAITINGMODE " = " +
                  std::to_string((int)autoWaitingMode);

            queryResult = exec(query);
         }
         else
         {
            ETG_TRACE_ERR(("setAutoWaitingModeSetting() failed for all devices in DeviceSettings"));

            result = false;
         }
      }
      else
      {
         query = "UPDATE " PM_DB_TABLENAME_DEVICESETTINGS " SET " PM_DB_COLUMNNAME_AUTOWAITINGMODE " = " +
               std::to_string((int)autoWaitingMode) + " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS "=\"" + deviceAddress + "\"";

         queryResult = exec(query);
      }

      if((true == result) && (SQLITE_OK == queryResult))
      {
         ETG_TRACE_USR1(("setAutoWaitingModeSetting() success"));
      }
      else
      {
         ETG_TRACE_ERR(("setAutoWaitingModeSetting() failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
         result = false;
      }

      return result;
   }

   bool DbHandler::getAutoWaitingModeSetting(const BdAddress& deviceAddress, bool& autoWaitingMode)
   {
      ETG_TRACE_USR1(("getAutoWaitingModeSetting() DeviceAddress : %s", deviceAddress.c_str()));

      std::string query;
      int queryResult = SQLITE_ERROR;
      bool retValue = false;

      if(0 == deviceAddress.compare(DEVICE_ADDRESS_ALL))
      {
         query = "SELECT " PM_DB_COLUMNNAME_AUTOWAITINGMODE " FROM " PM_DB_TABLENAME_SYSTEMWIDESETTINGS;
      }
      else
      {
         query = "SELECT " PM_DB_COLUMNNAME_AUTOWAITINGMODE " FROM " PM_DB_TABLENAME_DEVICESETTINGS
               " WHERE " PM_DB_COLUMNNAME_DEVICEADDRESS " = \"" + deviceAddress + "\"";
      }

      queryResult = prepare(query);

      if (SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            if(SQLITE_ROW == queryResult)
            {
               //this should be run only once
               int autoWaitingModeInDb = 0;

               getInt(PM_DB_GETRESULT_COLUMN_0, autoWaitingModeInDb);

               autoWaitingMode = (bool)autoWaitingModeInDb;
               retValue = true;
            }
            else
            {
               break;
            }
         }

         (void)finalize();
      }
      else
      {
         ETG_TRACE_ERR(("getAutoWaitingModeSetting() failed"));

         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return retValue;
   }

   bool DbHandler::getAutoWaitingModeSettingList(AutoWaitingModeOnOffListMap& autoWaitingModeOnOffList)
   {
      ETG_TRACE_USR1(("getAutoWaitingModeSettingList() entered"));

      int queryResult = SQLITE_ERROR;
      bool retValue = false;

      std::string query = "SELECT " PM_DB_COLUMNNAME_DEVICEADDRESS "," PM_DB_COLUMNNAME_AUTOWAITINGMODE " FROM "
            PM_DB_TABLENAME_DEVICESETTINGS;

      queryResult = prepare(query);

      if (SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            if(SQLITE_ROW == queryResult)
            {
               int autoWaitingModeInDb = 0;
               BdAddress deviceAddress;

               getString(PM_DB_GETRESULT_COLUMN_0, deviceAddress);
               getInt(PM_DB_GETRESULT_COLUMN_1, autoWaitingModeInDb);

               autoWaitingModeOnOffList.insert(std::pair<BdAddress,AutoWaitingModeState>(
                     deviceAddress, (AutoWaitingModeState)autoWaitingModeInDb));
            }
            else if(SQLITE_DONE == queryResult)
            {
               retValue = true;
               break;
            }
            else
            {
               break;
            }
         }

         (void)finalize();
      }
      else
      {
         ETG_TRACE_ERR(("getAutoWaitingModeSetting() failed"));

         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return retValue;
   }

   bool DbHandler::recreateDatabase()
   {
      ETG_TRACE_USR1(("recreateDatabase() entered"));

      //1. Remove PM.db file in the System
      ETG_TRACE_FATAL(("recreateDatabase() removing database file \"%s\"", _dbName.c_str()));

      int ret = remove(_dbName.c_str()); // if not exist, don't care

      if(0 == ret)
      {
         ETG_TRACE_FATAL(("recreateDatabase() removed file \"%s\"", _dbName.c_str()));
      }
      else
      {
         ETG_TRACE_FATAL(("recreateDatabase() error while deleting \"%s\"", _dbName.c_str() ));
      }

      //2. Open/Create Database (PM.db) and Configure Connection
      bool result = open(_dbName);

      if (true == result)
      {
         //3. Check and Adapt Database File Permissions
         checkAndAdaptDbSettings();

         //4. Create Database Schema
         result = result && createDatabaseTables();

         //5. Reset Database Integrity
         result = result && resetDatabaseIntegrity();

         //6. Insert SchemaHash
         result = result && insertSchemaHash();
      }
      else
      {
         ETG_TRACE_FATAL(("recreateDatabase: open/configure database failed"));
         ETG_TRACE_ERRMEM(("recreateDatabase: open/configure database failed"));

         close(); //Close database
      }

      return result;
   }

   void DbHandler::checkAndAdaptDbSettings()
   {
      ETG_TRACE_USR1(("checkAndAdaptDbSettings() entered"));

      int result = 0;

      result = com::bosch::pmcommon::checkAndAdaptFilePermission(_dbName, PM_DB_USERNAME,
            PM_DB_GROUPNAME, (S_IRUSR | S_IWUSR | S_IRGRP));

      if( result != 0 )
      {
         ETG_TRACE_ERR(("checkAndAdaptDbSettings() failed"));
      }
      else
      {
         ETG_TRACE_USR4(("checkAndAdaptDbSettings() success"));
      }
   }

   bool DbHandler::checkDatabaseIntegrity()
   {
      ETG_TRACE_USR1(("checkDatabaseIntegrity() entered"));

      bool result = false;
      int queryResult = SQLITE_ERROR;
      int integrityOk = 0;
      std::string query;

      // create the explain query string.
 #if QUICK_CHECK
      query = "PRAGMA quick_check;";
 #else
      query = "PRAGMA integrity_check;";
 #endif

      // prepare the update statement.
      queryResult = prepare(query);

      if (SQLITE_OK == queryResult)
      {
         TimeTrace ticks("IntegrityCheck");

         int lineNo = 1;

         while(1)
         {
            // do the explain.
            queryResult = step();

            if (SQLITE_ROW == queryResult)
            {
               std::string sqlText;

               getString(PM_DB_GETRESULT_COLUMN_0, sqlText);

               // check if it is OK.
               if((lineNo == 1) && (false == sqlText.empty()) && (0 == sqlText.compare("ok")))
               {
                  ETG_TRACE_USR2(("checkDatabaseIntegrity:integrityCheck : %s", sqlText.c_str()));

                  integrityOk = 1;
               }
               else
               {
                  ETG_TRACE_ERR(("checkDatabaseIntegrity:integrityError : %s", sqlText.c_str()));
               }
            }
            else
            {
               break;
            }

            lineNo++;
         }

         // finalize pragma query.
         queryResult = finalize();

         ticks.elapsed();
      }
      else
      {
         ETG_TRACE_ERR(("checkDatabaseIntegrity failed to prepare the integrity update statement"));
      }

      if (integrityOk)
      {
         query = "SELECT"
               " [" PM_DB_COLUMNNAME_VERSION "],"
               " [" PM_DB_COLUMNNAME_INTEGRITY "]"
               " FROM"
               " [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "];";

         // Prepare the database integrity check statement.
         queryResult = prepare(query);

         if (SQLITE_OK == queryResult)
         {
            while(1)
            {
               // Execute the integrity check statement.
               queryResult = step();

               // Retrieve the selected data if available.
               if (SQLITE_ROW == queryResult)
               {
                  // Check whether the schema version is valid.
                  int schemaValue;

                  getInt(PM_DB_GETRESULT_COLUMN_0, schemaValue);

                  if (PM_DB_SCHEMA_VERSION == schemaValue)
                  {
                     // Check the database integrity status.
                     getInt(PM_DB_GETRESULT_COLUMN_1, schemaValue);

                     if (0 == schemaValue)
                     {
                        // The database integrity is invalid.
                        ETG_TRACE_ERR(("checkDatabaseIntegrity database integrity is invalid"));
                     }
                     else
                     {
                        ETG_TRACE_USR4(("checkDatabaseIntegrity database integrity is valid"));
                        result = true;
                     }
                  }
                  else
                  {
                     ETG_TRACE_ERR(("checkDatabaseIntegrity invalid database schema version"));
                  }
               }
               else
               {
                  break;
               }
            }

            // Destroy the integrity check statement.
            queryResult = finalize();
         }
         else
         {
            // Failed to prepare the integrity check statement.
            ETG_TRACE_ERR(("checkDatabaseIntegrity failed to prepare the integrity check statement"));
         }
      }
      else
      {
         ETG_TRACE_ERR(("checkDatabaseIntegrity DB integrity damaged"));
      }

      // Check for success
      if((SQLITE_DONE != queryResult) && (SQLITE_OK != queryResult))
      {
         ETG_TRACE_ERR(("checkDatabaseIntegrity failed integrity check : %d / %s", queryResult,
               sqlite3_errstr(queryResult)));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return result;
   }

   bool DbHandler::checkSchemaHash()
   {
      ETG_TRACE_USR1(("checkSchemaHash() entered"));

      int queryResult = SQLITE_ERROR;
      bool result = false;
      std::string DefaultSchemaHash="0";

      std::string query = "SELECT"
            " [" PM_DB_COLUMNNAME_VERSION "],"
            " [" PM_DB_COLUMNNAME_INTEGRITY "],"
            " [" PM_DB_COLUMNNAME_SCHEMAHASH "]"
            " FROM"
            " [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "];";

      queryResult = prepare(query);

      // Check for success
      if (SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            // Retrieve the selected data if available.
            if (SQLITE_ROW == queryResult)
            {
               std::string schemaHash;

               getString(PM_DB_GETRESULT_COLUMN_2, schemaHash);

               if (0 == schemaHash.compare(recreateDBSchemaHash()))
               {
                  ETG_TRACE_FATAL(("checkSchemaHash: calculated and stored DB schema hashes match => OK"));
                  result = true;
               }
               else
               {
                  ETG_TRACE_FATAL(("checkSchemaHash: calculated DB schema hash differs from the one"
                        " stored in file => not OK  "));
               }
            }
            else
            {
               break;
            }
         }

         // Destroy the schema hash statement
         queryResult = finalize();
      }
      else
      {
         ETG_TRACE_ERR(("checkSchemaHash: failed to prepare the schema hash check statement"));
      }

      // Check for success
      if (SQLITE_OK != queryResult)
      {
         // Failed to finalize the schema hash statement
         ETG_TRACE_ERR(("checkSchemaHash: failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return result;
   }

   bool DbHandler::resetDatabaseIntegrity()
   {
      ETG_TRACE_USR1(("resetDatabaseIntegrity() entered"));

      bool result = false;
      int queryResult = SQLITE_ERROR;

      std::string query = "UPDATE"
            " [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "]"
            " SET"
            " [" PM_DB_COLUMNNAME_INTEGRITY "] = 1;";

      // Execute SQL query for integrity reset.
      queryResult = exec(query);

      // Check for success
      if (SQLITE_OK == queryResult)
      {
         ETG_TRACE_USR1(("resetDatabaseIntegrity() success"));
         result = true;
      }
      else
      {
         // Failed to execute SQL query.
         ETG_TRACE_USR1(("resetDatabaseIntegrity() failed"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return result;
   }

   std::string DbHandler::recreateDBSchemaHash()
   {
      ETG_TRACE_USR1(("recreateDBSchemaHash() entered"));

      char computedSchemaSHA256[SHA256_DIGEST_LENGTH+1];
      char* buf = new char[32];
      std::string str;
      std::string schema = "DROP TABLE IF EXISTS [MAIN].[" PM_DB_TABLENAME_SYSTEMWIDESETTINGS "];"
            "DROP TABLE IF EXISTS [MAIN].[" PM_DB_TABLENAME_DEVICESETTINGS "];"
            "DROP TABLE IF EXISTS [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "];"
            "CREATE TABLE IF NOT EXISTS [" PM_DB_TABLENAME_SYSTEMWIDESETTINGS "] ("
            "[" PM_DB_COLUMNNAME_ACTIVERINGTONEID "] INTEGER DEFAULT -1,"
            "[" PM_DB_COLUMNNAME_SUPPRESSRINGTONE "] BOOLEAN DEFAULT 0,"
            "[" PM_DB_COLUMNNAME_AUTOWAITINGMODE "] BOOLEAN DEFAULT 0);"
            "INSERT OR REPLACE INTO [MAIN].[" PM_DB_TABLENAME_SYSTEMWIDESETTINGS "] ("
            " [" PM_DB_COLUMNNAME_ACTIVERINGTONEID "],"
            " [" PM_DB_COLUMNNAME_SUPPRESSRINGTONE "],"
            " [" PM_DB_COLUMNNAME_AUTOWAITINGMODE "])"
            " VALUES (-1, 0, 0);"
            "CREATE TABLE IF NOT EXISTS [" PM_DB_TABLENAME_DEVICESETTINGS "] ("
            "[" PM_DB_COLUMNNAME_DEVICEHANDLE "] INTEGER DEFAULT 0,"
            "[" PM_DB_COLUMNNAME_DEVICEADDRESS "] TEXT CONSTRAINT [" PM_DB_COLUMNNAME_DEVICEADDRESS "]"
            " UNIQUE ON CONFLICT IGNORE,"
            "[" PM_DB_COLUMNNAME_RINGTONEID "] INTEGER DEFAULT -1,"
            "[" PM_DB_COLUMNNAME_SUPPRESSRINGTONE "] BOOLEAN DEFAULT 0,"
            "[" PM_DB_COLUMNNAME_AUTOWAITINGMODE "] BOOLEAN DEFAULT 0);"
            "CREATE TABLE IF NOT EXISTS [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "] ("
            " [" PM_DB_COLUMNNAME_VERSION "]    INTEGER PRIMARY KEY NOT NULL,"
            " [" PM_DB_COLUMNNAME_INTEGRITY "]  BOOLEAN NOT NULL,"
            " [" PM_DB_COLUMNNAME_SCHEMAHASH "] TEXT DEFAULT '');"
            "INSERT OR REPLACE INTO [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "] ("
            " [" PM_DB_COLUMNNAME_VERSION "],"
            " [" PM_DB_COLUMNNAME_INTEGRITY "],"
            " [" PM_DB_COLUMNNAME_SCHEMAHASH "])"
            "VALUES("
            " 1,"
            " 1,"
            " '0');";

      SHA256(reinterpret_cast<const unsigned char*>(schema.c_str()), schema.size(), reinterpret_cast<unsigned char*>(computedSchemaSHA256));
      computedSchemaSHA256[SHA256_DIGEST_LENGTH] = '\0';

      for(int i=0; i < SHA256_DIGEST_LENGTH; i++)
      {
         sprintf(buf,"%02x",computedSchemaSHA256[i]);
         str.append(buf);
      }

      delete [] buf;

      return str;
   }

   bool DbHandler::insertSchemaHash()
   {
      ETG_TRACE_USR1(("insertSchemaHash() entered"));

      bool result = false;
      int queryResult = SQLITE_ERROR;
      std::string DefaultSchemaHash="0";
      std::string query("SELECT"
            " [" PM_DB_COLUMNNAME_VERSION "],"
            " [" PM_DB_COLUMNNAME_INTEGRITY "],"
            " [" PM_DB_COLUMNNAME_SCHEMAHASH "]"
            " FROM"
            " [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "];");

      queryResult = prepare(query);

      // Check for success
      if (SQLITE_OK == queryResult)
      {
         while(1)
         {
            queryResult = step();

            // Check for success
            // Retrieve the selected data if available.
            if (SQLITE_ROW == queryResult)
            {
               std::string schemaHash = "";

               getString(PM_DB_GETRESULT_COLUMN_2, schemaHash);

               if (0 == DefaultSchemaHash.compare(schemaHash))
               {
                  // DB contains the default DB schema hash, DB schema hash has to be calculated and stored into DB
                  ETG_TRACE_FATAL(("insertSchemaHash: DB contains the default schema hash, calculating new value."));

                  schemaHash = recreateDBSchemaHash();

                  query = "UPDATE"
                        " [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "]"
                        " SET"
                        " [" PM_DB_COLUMNNAME_SCHEMAHASH "] = '" + schemaHash +
                        "' WHERE"
                        " [" PM_DB_COLUMNNAME_SCHEMAHASH "] = '0';";

                  queryResult = exec(query);

                  // Check for success
                  if (SQLITE_OK != queryResult)
                  {
                     // Failed to execute SQL query.
                     ETG_TRACE_FATAL(("insertSchemaHash: could not store calculated DB schema hash value in DB"));
                  }
                  else
                  {
                     ETG_TRACE_FATAL(("insertSchemaHash: stored the calculated DB schema hash value in DB => OK"));
                     result = true;
                  }
               }
            }
            else
            {
               break;
            }
         }

         queryResult = finalize();
      }

      // Check for success
      if((SQLITE_DONE != queryResult) && (SQLITE_OK != queryResult))
      {
         ETG_TRACE_ERR(("insertSchemaHash failed : %d / %s", queryResult, sqlite3_errstr(queryResult)));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return result;
   }

   bool DbHandler::createDatabaseTables()
   {
      ETG_TRACE_FATAL(("createDatabaseTables() entered"));

      bool result = true;

      // Start DB creation monitoring
      // write a file onto the disk to issue a data base file delete on next restart of system
      FILE *fp;
      fp = fopen(PM_DB_RECREATEFILE, "w");

      if (nullptr != fp)
      {
         ETG_TRACE_ERRMEM(("createDatabaseTables: created file \"%s\"", PM_DB_RECREATEFILE));
         fclose(fp);
      }
      else
      {
         ETG_TRACE_ERRMEM(("createDatabaseTables: could not open/create file \"%s\"", PM_DB_RECREATEFILE));
      }

      //Create Database schema for all tables
      result = result && createSystemWideSettingsTable();
      result = result && createDeviceSettingsTable();
      result = result && createSchemaVersionTable();

      // End DB creation monitoring
      // remove the recreation trigger file in case of success
      if (true == result)
      {
         if(0 == remove(PM_DB_RECREATEFILE))
         {
            ETG_TRACE_FATAL(("createDatabaseTables() removed file \"%s\"", PM_DB_RECREATEFILE));
            ETG_TRACE_ERRMEM(("createDatabaseTables() removed file \"%s\"", PM_DB_RECREATEFILE));
         }
         else
         {
            ETG_TRACE_FATAL(("createDatabaseTables() error while deleting \"%s\"", PM_DB_RECREATEFILE));
         }
      }
      else
      {
         ETG_TRACE_ERR(("createDatabaseTables() error while creating tables"));
      }

      return result;
   }

   bool DbHandler::checkSystemWideSettingsTableSchema()
   {
      ETG_TRACE_USR1(("checkSystemWideSettingsTableSchema() entered"));

      //TODO: imlpement if required

      return true;
   }

   bool DbHandler::createSystemWideSettingsTable()
   {
      ETG_TRACE_USR1(("createSystemWideSettingsTable() entered"));

      bool result = false;
      int queryResult = SQLITE_ERROR;

      //drop the system wide settings if table exists
      std::string query("DROP TABLE IF EXISTS [MAIN].[" PM_DB_TABLENAME_SYSTEMWIDESETTINGS "];");

      queryResult = exec(query); //ignore, if error returned

      //create the table if not exist
      query = "CREATE TABLE IF NOT EXISTS [" PM_DB_TABLENAME_SYSTEMWIDESETTINGS "] ("
            "[" PM_DB_COLUMNNAME_ACTIVERINGTONEID "] INTEGER DEFAULT -1,"
            "[" PM_DB_COLUMNNAME_SUPPRESSRINGTONE "] BOOLEAN DEFAULT 0,"
            "[" PM_DB_COLUMNNAME_AUTOWAITINGMODE "] BOOLEAN DEFAULT 0);";

      queryResult = exec(query);

      if(SQLITE_OK == queryResult)
      {
         ETG_TRACE_USR4(("createSystemWideSettingsTable() table created"));

         //insert the default values in system wide settings
         query = "INSERT OR REPLACE INTO [MAIN].[" PM_DB_TABLENAME_SYSTEMWIDESETTINGS "] ("
               " [" PM_DB_COLUMNNAME_ACTIVERINGTONEID "],"
               " [" PM_DB_COLUMNNAME_SUPPRESSRINGTONE "],"
               " [" PM_DB_COLUMNNAME_AUTOWAITINGMODE "])"
               " VALUES (-1, 0, 0);";

         queryResult = exec(query);

         if(SQLITE_OK == queryResult)
         {
            ETG_TRACE_USR4(("createSystemWideSettingsTable() default values set"));
            result = true;
         }
         else
         {
            ETG_TRACE_ERR(("createSystemWideSettingsTable() default values not set"));
            PM_DB_CHECK_QUERY_ERROR(queryResult);
         }
      }
      else
      {
         ETG_TRACE_ERR(("createSystemWideSettingsTable() table not created"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return result;
   }

   bool DbHandler::checkDeviceSettingsTableSchema()
   {
      ETG_TRACE_USR1(("checkDeviceSettingsTableSchema() entered"));

      //TODO: imlpement if required

      return true;
   }

   bool DbHandler::createDeviceSettingsTable()
   {
      ETG_TRACE_USR1(("createDeviceSettingsTable() entered"));

      bool result = false;
      int queryResult = SQLITE_ERROR;

      std::string query("DROP TABLE IF EXISTS [MAIN].[" PM_DB_TABLENAME_DEVICESETTINGS "];");

      queryResult = exec(query); //ignore, if error returned

      query = "CREATE TABLE IF NOT EXISTS [" PM_DB_TABLENAME_DEVICESETTINGS "] ("
            "[" PM_DB_COLUMNNAME_DEVICEHANDLE "] INTEGER DEFAULT 0,"
            "[" PM_DB_COLUMNNAME_DEVICEADDRESS "] TEXT CONSTRAINT [" PM_DB_COLUMNNAME_DEVICEADDRESS "]"
            " UNIQUE ON CONFLICT IGNORE,"
            "[" PM_DB_COLUMNNAME_RINGTONEID "] INTEGER DEFAULT -1,"
            "[" PM_DB_COLUMNNAME_SUPPRESSRINGTONE "] BOOLEAN DEFAULT 0,"
            "[" PM_DB_COLUMNNAME_AUTOWAITINGMODE "] BOOLEAN DEFAULT 0);";

      queryResult = exec(query);

      if(SQLITE_OK == queryResult)
      {
         ETG_TRACE_ERR(("createDeviceSettingsTable() table created"));
         result = true;
      }
      else
      {
         ETG_TRACE_ERR(("createDeviceSettingsTable() table not created"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return result;
   }

   bool DbHandler::checkSchemaVersionTableSchema()
   {
      ETG_TRACE_USR1(("checkSchemaVersionTableSchema() entered"));

      //TODO: imlpement if required

      return true;
   }

   bool DbHandler::createSchemaVersionTable()
   {
      ETG_TRACE_USR1(("createSchemaVersionTable() entered"));

      bool result = false;
      int queryResult = SQLITE_ERROR;

      std::string query("DROP TABLE IF EXISTS [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "];");

      queryResult = exec(query); //ignore, if error returned

      query = "CREATE TABLE IF NOT EXISTS [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "] ("
            " [" PM_DB_COLUMNNAME_VERSION "]    INTEGER PRIMARY KEY NOT NULL,"
            " [" PM_DB_COLUMNNAME_INTEGRITY "]  BOOLEAN NOT NULL,"
            " [" PM_DB_COLUMNNAME_SCHEMAHASH "] TEXT DEFAULT '');";

      queryResult = exec(query);

      if(SQLITE_OK == queryResult)
      {
         ETG_TRACE_USR4(("createSchemaVersionTable() table created"));

         query = "INSERT OR REPLACE INTO [MAIN].[" PM_DB_TABLENAME_SCHEMAVERSION "] ("
               " [" PM_DB_COLUMNNAME_VERSION "],"
               " [" PM_DB_COLUMNNAME_INTEGRITY "],"
               " [" PM_DB_COLUMNNAME_SCHEMAHASH "])"
               "VALUES("
               " 1,"
               " 1,"
               " '0');";

         queryResult = exec(query);

         if(SQLITE_OK == queryResult)
         {
            ETG_TRACE_USR4(("createSchemaVersionTable() default values set"));
            result = true;
         }
         else
         {
            ETG_TRACE_ERR(("createSchemaVersionTable() default values not set"));
            PM_DB_CHECK_QUERY_ERROR(queryResult);
         }
      }
      else
      {
         ETG_TRACE_ERR(("createSchemaVersionTable() table not created"));
         PM_DB_CHECK_QUERY_ERROR(queryResult);
      }

      return result;
   }

   void DbHandler::checkDbQueryError(const int sqlErrorCode, const int lineNo, const std::string& fileName,
         const std::string& functionName)
   {
      ETG_TRACE_USR1(("checkDbQueryError() SqlErrorCode : %d, ErrMsg : %s", sqlErrorCode,
            sqlite3_errstr(sqlErrorCode)));
      ETG_TRACE_USR4(("checkDbQueryError() FileName: %100s, LineNo: %d, FuncName: %50s", fileName.c_str(),
            lineNo, functionName.c_str()));

      int createRecreateFile = 0;

      // check if the database as returned a massive error
      if (sqlErrorCode == SQLITE_CORRUPT)
      {
         ETG_TRACE_USR1(("checkDbQueryError() database corrupted"));

         // in this case, database must be recreated anyway
         createRecreateFile = 1;
      }
      else if (sqlErrorCode == SQLITE_ERROR)
      {
         // this is a general error
         std::string errMsg;

         getErrMsg(errMsg);

         if (false == errMsg.empty())
         {
            if ((std::string::npos != errMsg.find(PM_DB_ERROR_NOSUCHTABLE)) &&
                  (std::string::npos != errMsg.find(PM_DB_TABLENAME_SYSTEMWIDESETTINGS)) &&
                  (std::string::npos != errMsg.find(PM_DB_TABLENAME_DEVICESETTINGS)) &&
                  (std::string::npos != errMsg.find(PM_DB_TABLENAME_SCHEMAVERSION)))
            {
               ETG_TRACE_USR1(("checkDbQueryError() table missing in database"));

               // do only a recreate if a table was missing
               createRecreateFile = 1;
            }
            else
            {
               ETG_TRACE_ERR(("checkDbQueryError() query error"));
            }
         }
      }

      if (createRecreateFile)
      {
         // write a file onto the disk to issue a database file delete on next restart of system
         FILE *fp;
         fp = fopen(PM_DB_RECREATEFILE, "w");

         if (nullptr != fp)
         {
            ETG_TRACE_ERRMEM(("checkDbQueryError: created file \"%s\"", PM_DB_RECREATEFILE));
            fclose(fp);
         }
         else
         {
            ETG_TRACE_ERRMEM(("checkDbQueryError: could not open/create file \"%s\"", PM_DB_RECREATEFILE));
         }
      }
   }
}
