/**
 * @file DbHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the DbHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file handles the db related information and updating the same
 *          to clients if any changes in the same
 *
 * @ingroup PmCore
 */

#ifndef DbHandler_h
#define DbHandler_h

#include "PmSingleton.h"
#include "Database.h"
#include "PmInterfaceTypesInternal.h"
#include "PmCoreIfTypes.h"

namespace pmcore
{
   class DbHandler : public Database, public PmSingleton<DbHandler>
   {
   public:
      /**
       * This method is used to open the database and checks database schema hash, integrity, file permissions.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool openDatabase();

      /**
       * This method is used to close the database.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void closeDatabase();

      /**
       * This method is used to add the device to database.
       *
       * @param[in] deviceHandle - device handle
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool addDeviceInDb(const DeviceHandle deviceHandle, const BdAddress& deviceAddress);

      /**
       * This method is used to remove the device from database.
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool removeDeviceFromDb(const BdAddress& deviceAddress);

      /**
       * This method is used to get the available devices from database.
       *
       * @param[in] deviceList - device list
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool getDevicesInDb(std::map<DeviceHandle, BdAddress>& deviceList);

      /**
       * This method is used to restore the both system wide and device specific settings.
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool restoreDefaultSetting(const BdAddress& deviceAddress);

      /**
       * This method is used to set the ringtone setting for both system wide and device specific settings.
       *
       * @param[in] deviceAddress - device address
       * @param[in] ringtoneId - ringtone id
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool setRingtoneId(const BdAddress& deviceAddress, const RingtoneId ringtoneId);

      /**
       * This method is used to get the ringtone setting for both system wide and device specific settings.
       *
       * @param[in] deviceAddress - device address
       * @param[out] ringtoneId - ringtone id
       * @param[in,out]
       *
       * @return bool
       */
      bool getRingtoneId(const BdAddress& deviceAddress, RingtoneId& ringtoneId);

      /**
       * This method is used to get the device ringtone list.
       *
       * @param[in]
       * @param[out] deviceRingtoneList - device ringtone list
       * @param[in,out]
       *
       * @return bool
       */
      bool getDeviceRingtoneIdList(DeviceRingtoneListType& deviceRingtoneList);

      /**
       * This method is used to set the suppress ringtone setting for both system wide and device specific settings.
       *
       * @param[in] deviceAddress - device address
       * @param[in] suppressRingtone - suppress ringtone state
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool setSuppressRingtoneSetting(const BdAddress& deviceAddress, const bool suppressRingtone);

      /**
       * This method is used to get the suppress ringtone setting for both system wide and device specific settings.
       *
       * @param[in] deviceAddress - device address
       * @param[out] suppressRingtone - suppress ringtone state
       * @param[in,out]
       *
       * @return bool
       */
      bool getSuppressRingtoneSetting(const BdAddress& deviceAddress, bool& suppressRingtone);

      /**
       * This method is used to get the suppress ringtone on off list.
       *
       * @param[in]
       * @param[out] suppressRingtoneOnOffList - suppress ringtone on off list
       * @param[in,out]
       *
       * @return bool
       */
      bool getSuppressRingtoneSettingList(SuppressRingtoneOnOffListMap& suppressRingtoneOnOffList);

      /**
       * This method is used to set the auto waiting mode setting for both system wide and device specific settings.
       *
       * @param[in] deviceAddress - device address
       * @param[in] autoWaitingMode - auto waiting mode state
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool setAutoWaitingModeSetting(const BdAddress& deviceAddress, const bool autoWaitingMode);

      /**
       * This method is used to get the auto waiting mode setting for both system wide and device specific settings.
       *
       * @param[in] deviceAddress - device address
       * @param[out] autoWaitingMode - auto waiting mode state
       * @param[in,out]
       *
       * @return bool
       */
      bool getAutoWaitingModeSetting(const BdAddress& deviceAddress, bool& autoWaitingMode);

      /**
       * This method is used to get the auto waiting mode on off list.
       *
       * @param[in]
       * @param[out] autoWaitingModeOnOffList - auto waiting mode on off list
       * @param[in,out]
       *
       * @return bool
       */
      bool getAutoWaitingModeSettingList(AutoWaitingModeOnOffListMap& autoWaitingModeOnOffList);

   private:
      friend class PmSingleton<DbHandler>;

      /**
       * Constructor of DbHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      DbHandler();

      /**
       * Copy Constructor of DbHandler class => must not be used.
       *
       * @param[in] other: reference to copy
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      DbHandler(const DbHandler& other);

      /**
       * Overloading '=' operator => must not be used.
       *
       * @param[in] other: reference to copy
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      DbHandler& operator=(DbHandler other);

      /**
       * Destructor of DbHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~DbHandler();

      /**
       * This method is used to recreate the database in case of any failures.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool recreateDatabase();

      /**
       * This method is used to check the database file permission and adapts to the required permission.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void checkAndAdaptDbSettings();

      /**
       * This method is used to check the database integrity.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool checkDatabaseIntegrity();

      /**
       * This method is used to check the database schema hash value.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool checkSchemaHash();

      /**
       * This method is used to reset the database integrity.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool resetDatabaseIntegrity();

      /**
       * This method is used to recreate the database schema hash value.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return string
       */
      std::string recreateDBSchemaHash();

      /**
       * This method is used to update the database schema hash value.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool insertSchemaHash();

      /**
       * This method is used to create the database tables.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool createDatabaseTables();

      /**
       * This method is used to check the database system wide settings table schema.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool checkSystemWideSettingsTableSchema();

      /**
       * This method is used to create the database system wide settings table.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool createSystemWideSettingsTable();

      /**
       * This method is used to check the database device settings table schema.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool checkDeviceSettingsTableSchema();

      /**
       * This method is used to create the database device settings table.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool createDeviceSettingsTable();

      /**
       * This method is used to check the database schema version table schema.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool checkSchemaVersionTableSchema();

      /**
       * This method is used to create the database schema version table.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool createSchemaVersionTable();

      /**
       * This method is used to check the database query error and triggers the database recreation if needed.
       *
       * @param[in] sqlErrorCode - sql error
       * @param[in] lineNo - file line number
       * @param[in] fileName - file name
       * @param[in] functionName - function name
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void checkDbQueryError(const int sqlErrorCode, const int lineNo = 0, const std::string& fileName = "file_name",
            const std::string& functionName = "function_name");

      std::string  _dbName;  /**< Database Name */
   };

} // namespace pmcore

#endif // DbHandler_h
