/**
 * @file Database.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the Database class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file handles the database related operations
 *
 * @ingroup PmCore
 */

#ifndef Database_h
#define Database_h

#include "Lock.h"
#include <sqlite3.h>
#include <string>

namespace pmcore
{
   class Database
   {
   public:
      /**
       * Constructor of Database class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      Database();

      /**
       * Destructor of Database class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~Database();

      /**
       * This method is used to check whether any column selected in database step query execution.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool columnSelected();

      /**
       * This method is used to get the integer value in the selected column.
       *
       * @param[in] column - selected column
       * @param[out] value - integer value
       * @param[in,out]
       *
       * @return void
       */
      void getInt(const int column, int& value);

      /**
       * This method is used to get the string value in the selected column.
       *
       * @param[in] column - selected column
       * @param[out] value - string value
       * @param[in,out]
       *
       * @return void
       */
      void getString(const int column, std::string& value);

      /**
       * This method is used to get the received error when executing database queries.
       *
       * @param[in]
       * @param[out] value - error string
       * @param[in,out]
       *
       * @return void
       */
      void getErrMsg(std::string& value);

   protected:
      /**
       * This method is used to open the database and also configures the connection.
       *
       * @param[in] fileName - database file path
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool open(const std::string& fileName);

      /**
       * This method is used to close the database.
       *
       * @param[in] fileName - database file path
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool close();

      /**
       * This method is used to attach the database.
       *
       * @param[in] db - database file path
       * @param[in] alias - alias name
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void attach(const std::string& db, const std::string& alias);

      /**
       * This method is used to detach the database.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void detach();

      /**
       * This method is used to begin the database transaction.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return int
       */
      int beginTransaction();

      /**
       * This method is used to end the database transaction.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return int
       */
      int endTransaction();

      /**
       * This method is used to prepare the database query.
       *
       * @param[in] query - sqlite query
       * @param[out]
       * @param[in,out]
       *
       * @return int
       */
      int prepare(std::string& query);

      /**
       * This method is used to bind the integer value in the database query.
       *
       * @param[in] column - bind index
       * @param[in] value - integer value
       * @param[out]
       * @param[in,out]
       *
       * @return int
       */
      int bind(const int column, const int value);

      /**
       * This method is used to bind the string value in the database query.
       *
       * @param[in] column - bind index
       * @param[in] value - string value
       * @param[out]
       * @param[in,out]
       *
       * @return int
       */
      int bind(const int column, const std::string& value);

      /**
       * This method is used to execute the prepared database query step by step.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return int
       */
      int step();

      /**
       * This method is used to finalize the prepared database query.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return int
       */
      int finalize();

      /**
       * This method is used to execute the database query.
       *
       * @param[in] query - sqlite query
       * @param[out]
       * @param[in,out]
       *
       * @return int
       */
      int exec(std::string& query);

      //static int execCb(void *userData, int count, char **data, char **columns);

   private:
      /**
       * This method is used to configure the database connection.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return bool
       */
      bool configure();

      /**
       * This method will be called when database is busy to execute the given query.
       *
       * @param[in] userData - user data
       * @param[out] times - sleep times
       * @param[in,out]
       *
       * @return int
       */
      static int busyCallback(void *userData, int times);

      sqlite3*      _dbHandle;     /**< Unique DB Handle */
      sqlite3_stmt* _sqlStatement; /**< SQL Statement */
      int           _queryResult;  /**< Query Result */
      LockForever   _dbAccessLock; /**< Lock to synchronize the database access */
   };

} // namespace pmcore

#endif // Database_h
