/**
 * @file CallController.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CallController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CallController interfaces of PmCore.
 *
 * @ingroup PmCore
 */

#include "CallController.h"
#include "PmCoreMainController.h"
#include "PropertyUpdateNotifierToCore.h"
#include "PmCoreIfMessageRequest.h"
#include "PmCoreIfMessageResult.h"
#include "PropertyDetails.h"
#include "EvoBtStackWrapper.h"
#include "YakinduSmVoiceCallIf.h"
#include "PmConfiguration.h"
#include "PmCoreIfMessageCreator.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/CallController.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_CORE
#endif
#endif

#define CALL_COUNT_TWO 0x02

using namespace com::bosch::pmcommon;

namespace pmcore
{
   CallController::CallController() :
      _propertyIdList()
   {
      ETG_TRACE_USR1(("CallController"));

      _smVoiceCallIf = new YakinduSmVoiceCallIf;

      subscribeToBtStackEventNotifier();
   }

   CallController::~CallController()
   {
      ETG_TRACE_USR1(("~CallController"));

      PropertyUpdateNotifierToCore::getInstance().detachControllerInNotifierList(_propertyIdList, this);
      _propertyIdList.clear();

      if(nullptr != _smVoiceCallIf)
      {
         delete _smVoiceCallIf;
         _smVoiceCallIf = nullptr;
      }
   }

   void CallController::subscribeToBtStackEventNotifier()
   {
      ETG_TRACE_USR1(("subscribeToBtStackEventNotifier"));

      // Pushing the default properties for CallController
      for( unsigned int index = BTS_UPDATE_AUDIO; index < BTS_UPDATE_END_CALL_CONTROLLER; index++)
      {
         _propertyIdList.push_back(static_cast<PmCorePropertyAndEventId>(index));
      }

      // Pushing the other interested properties and events
      _propertyIdList.push_back(ON_DEVICE_CONNECTED);
      _propertyIdList.push_back(ON_DEVICE_DISCONNECTED);
      _propertyIdList.push_back(BTS_UPDATE_INBAND_RINGING);
      _propertyIdList.push_back(BTS_UPDATE_CHLD_FEATURE);
      _propertyIdList.push_back(BTS_UPDATE_MOBILE_COUNTRY_CODE);
      _propertyIdList.push_back(BTS_SIGNAL_SCO_CONNECT_REQUEST);
      _propertyIdList.push_back(BTS_SIGNAL_VOICE_CALL_ADDED);
      _propertyIdList.push_back(BTS_SIGNAL_VOICE_CALL_REMOVED);

      // Events from PM Audiomanager
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_ACQUISITION_SUCCESS);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_ACQUISITION_SUCCESS);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_ACQUISITION_FAILURE);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_ACQUISITION_SUCCESS);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_ACQUISITION_FAILURE);

      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_PLAYAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_PLAYAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_PLAYAUDIO_FAILURE);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_PLAYAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_PLAYAUDIO_FAILURE);

      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_STOPAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_STOPAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_STOPAUDIO_FAILURE);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_STOPAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_STOPAUDIO_FAILURE);

      _propertyIdList.push_back(AM_PHONE_CHANNEL_PAUSEAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_PHONE_CHANNEL_PAUSEAUDIO_FAILURE);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_SUCCESS);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_FAILURE);

      _propertyIdList.push_back(AM_PHONE_CHANNEL_RELEASED);
      _propertyIdList.push_back(AM_RINGTONE_CHANNEL_RELEASED);
      _propertyIdList.push_back(AM_WAITINGMODE_CHANNEL_RELEASED);

      _propertyIdList.push_back(AM_SET_MIC_MUTE_SUCCESS);
      _propertyIdList.push_back(AM_SET_MIC_MUTE_FAILURE);
      _propertyIdList.push_back(AM_SET_MIC_UNMUTE_SUCCESS);
      _propertyIdList.push_back(AM_SET_MIC_UNMUTE_FAILURE);

      PropertyUpdateNotifierToCore::getInstance().attachControllerToNotifierList(_propertyIdList, this);
   }

   void CallController::onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
         IN std::shared_ptr<void> propertyDetails)
   {
      ETG_TRACE_USR4(("CallController::onPropertyUpdate propertyId : %d",
            ETG_CENUM(PmCorePropertyAndEventId, propertyId)));
      //TODO: Handler functions needs to be implemented and mapped to switch cases.

      if(nullptr == propertyDetails)
      {
         ETG_TRACE_ERR(("CallController::onPropertyUpdate with empty details"));
         return;
      }

      switch(propertyId)
      {
         case BTS_SIGNAL_SCO_CONNECT_REQUEST:
         {
            BTSResult btsResult;
            std::shared_ptr<PropertyDetails<uint8_t>> property =
                  std::static_pointer_cast<PropertyDetails<uint8_t>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();

            // Try to acquire Active device slot if available
            DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
            ActiveSwitchingRequest activeSwitchingRequest(deviceAddress, TEMP_ROLE_SWITCH,
                  PendingRequestInfo(), CALL_CONTROLLER);

            DeviceRoleSwitchResultEnum acquireFreeActiveDeviceSlotResult =
                  deviceInfoHandler.acquireFreeActiveDeviceSlot(deviceAddress, activeSwitchingRequest);

            if (CANNOT_BE_MADE_ACTIVE != acquireFreeActiveDeviceSlotResult)
            {
               //TODO:Do ECNR configuration based on incoming SCOType and ready for audio channel request.
               // Respond to SCO connection via Accept SCO connect
               btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendAcceptSCOConnectRequest(
                     deviceAddress, 0 /*Act value for internal call to stack*/);

               if(btsResult._btsRequestResult != BTS_REQ_SUCCESS)
               {
                  ETG_TRACE_ERR(("CallController::onPropertyUpdate sendAcceptSCOConnectRequest failed for :%s",
                        deviceAddress));
               }
            }
         }
         break;
         case BTS_SIGNAL_VOICE_CALL_ADDED:
         {
            std::shared_ptr<PropertyDetails<BTSVoiceCallInfo>> property =
                  std::static_pointer_cast<PropertyDetails<BTSVoiceCallInfo>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();

            CallStatusListMap::iterator it = _callStatusList._callStatusList.find(deviceAddress);
            bool updateToSM = true;

            if(it != _callStatusList._callStatusList.end())
            {
               auto waitingModeIter = _waitingModeStateListMap.find(deviceAddress);
               bool autoRejectThirdIncomingCall =
                     com::bosch::pmcommon::PmConfiguration::getInstance().getAutoRejectThirdIncomingCall();

               if ((it->second._callInfoList.size() == CALL_COUNT_TWO) &&
                     ("Waiting" == property->getMessage()._callInfo._state) && autoRejectThirdIncomingCall)
               {
                  ETG_TRACE_USR4(("Rejecting third incoming call"));
                  _smVoiceCallIf->autoRejectCallRequest(deviceAddress,
                        AutoRejectCall(property->getMessage()._callInfo._instance,
                              property->getMessage()._callInfo._telephoneNumber, THIRD_INCOMING_CALL));

                  updateToSM = false;
               }
               else if (_waitingModeStateListMap.end() != waitingModeIter)
               {
                  ETG_TRACE_USR4(("Valid Waiting mode Iter"));

                  if ((WAITING_MODE_IDLE != waitingModeIter->second._waitingModeState) &&
                        ("Waiting" == property->getMessage()._callInfo._state))
                  {
                     ETG_TRACE_USR4(("Rejecting Call during waiting mode"));

                     _smVoiceCallIf->autoRejectCallRequest(deviceAddress,
                           AutoRejectCall(property->getMessage()._callInfo._instance,
                                 property->getMessage()._callInfo._telephoneNumber, WAITING_MODE_IN_PROGRESS));

                     updateToSM = false;
                  }
               }

               if (updateToSM)
               {
                  //already call exists in the same device, add this call into list
                  // The below code helps overcoming more than one Call Added signal for a single call.
                  CallInstance callInstance = property->getMessage()._callInfo._instance;

                  CallInfoList& callInfoList = it->second._callInfoList;
                  auto callInfoListIter = std::find_if(callInfoList.begin(), callInfoList.end(),
                           [&callInstance](CallInfo const& obj){return obj._instance == callInstance;});
                  if (callInfoList.end() == callInfoListIter)
                  {
                     it->second._multiparty &= property->getMessage()._multiparty;
                     it->second._callInfoList.push_back(property->getMessage()._callInfo);
                  }
                  else
                  {
                     ETG_TRACE_USR4(("Call Added signal already received for this instance"));
                     updateToSM = false;
                  }
               }
            }
            else
            {
               VRController& vrController = PmCoreMainController::getInstance().getVRController();
               if (! vrController.isExternalVRStatusIdle(deviceAddress))
               {
                  updateToSM = false;
               }
               else
               {
                  PropertyUpdateNotifierToCore::getInstance().notifyControllers(BTS_SIGNAL_VOICE_CALL_ADDED_END_VR,
                        property);

                  CallStatus callStatus;
                  callStatus._multiparty = property->getMessage()._multiparty;
                  callStatus._callInfoList.push_back(property->getMessage()._callInfo);

                  _callStatusList._callStatusList.insert(std::pair<BdAddress, CallStatus>(deviceAddress, callStatus));

                  CallState currentCallState = callStatus._callInfoList.front()._state;

                  // Device address of the active device to be filled in "acquireFreeActiveDeviceSlot"
                  ActiveSwitchingRequest activeSwitchingRequest("", TEMP_ROLE_SWITCH, PendingRequestInfo(),
                        CALL_CONTROLLER);

                  // Try to acquire Active device slot if available
                  DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
                  DeviceRoleSwitchResultEnum acquireFreeActiveDeviceSlotResult =
                        deviceInfoHandler.acquireFreeActiveDeviceSlot(deviceAddress, activeSwitchingRequest);

                  ETG_TRACE_USR4(("CallController::acquireFreeActiveDeviceSlotResult : %d",
                        ETG_CENUM(DeviceRoleSwitchResultEnum, acquireFreeActiveDeviceSlotResult)));

                  if (("Dialing" == currentCallState) || ("Alerting" == currentCallState))
                  {
                     // If the state is dialling or alerting and if the slot is already acquired
                     // due to SCO establishment, then after the call ends the device should remain as Active.
                     deviceInfoHandler.clearElementFromDeviceToBeMadePassiveMap(deviceAddress);
                  }

                  CallExistenceState callExistenceState = CALL_PRESENT_WITHOUT_SCO;
                  if (deviceInfoHandler.isSCOEstablished(deviceAddress))
                  {
                     callExistenceState = CALL_PRESENT_WITH_SCO;
                  }
                  else if(("Incoming" == currentCallState) || ("Waiting" == currentCallState))
                  {
                     callExistenceState = INCOMING_CALL_PRESENT;
                  }

                  if(CANNOT_BE_MADE_ACTIVE != acquireFreeActiveDeviceSlotResult)
                  {
                     PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnPmStateChanged(
                           deviceAddress, callExistenceState);
                  }
               }
            }

            if (updateToSM)
            {
               //Update should be triggered from SM
               generateAgCallStateEventToSM(deviceAddress);
            }
         }
         break;
         case BTS_SIGNAL_VOICE_CALL_REMOVED:
         {
            std::shared_ptr<PropertyDetails<CallInstance>> property =
                  std::static_pointer_cast<PropertyDetails<CallInstance>>(propertyDetails);

            CallStatusListMap::iterator it = _callStatusList._callStatusList.find(property->getBdAddress());

            if(it != _callStatusList._callStatusList.end())
            {
               CallInfoList::iterator itrList;
               for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
               {
                  if(itrList->_instance == property->getMessage())
                  {
                     itrList = it->second._callInfoList.erase(itrList);
                     break;
                  }
               }

               if (itrList == it->second._callInfoList.end())
               {
                  _smVoiceCallIf->onCallRemoved(property->getBdAddress(), property->getMessage());
               }

               if(1 == it->second._callInfoList.size())
               {
                  //TODO: Manually updated to false, since  only one call is present in device
                  it->second._multiparty = false;
               }
               else if(0 == it->second._callInfoList.size())
               {
                  //all calls removed, removed device from list
                  _callStatusList._callStatusList.erase(it);

                  //send mic unmute when all calls are removed
                  if((_callStatusList._callStatusList.empty()) && (0x01 == _microphoneMuteState._microphoneMuteState))
                  {
                      _microphoneMuteState._microphoneMuteState = 0x00;

                      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnMicrophoneMuteStateChanged(
                            _microphoneMuteState);
                  }

                  PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnPmStateChanged(
                        property->getBdAddress(), CALL_REMOVED);
               }
            }
            else
            {
               ETG_TRACE_USR4(("Device not exist"));
            }
         }
         break;

         case ON_DEVICE_CONNECTED:
         {
            std::shared_ptr<PropertyDetails<BasicDeviceDetails>> property =
                  std::static_pointer_cast<PropertyDetails<BasicDeviceDetails>>(propertyDetails);

            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getMessage()._deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate deviceHandle : %u", property->getMessage()._deviceHandle));
            ETG_TRACE_USR4(("onPropertyUpdate vendorId : %u", property->getMessage()._deviceIdentification._vendorId));
            ETG_TRACE_USR4(("onPropertyUpdate vendorIdSource : %u",
                  property->getMessage()._deviceIdentification._vendorIdSource));

            static_cast<YakinduSmVoiceCallIf*>(_smVoiceCallIf)->onDeviceConnected(property->getMessage()._deviceAddress);
         }
         break;

         case ON_DEVICE_DISCONNECTED:
         {
            std::shared_ptr<PropertyDetails<BdAddress>> property =
                  std::static_pointer_cast<PropertyDetails<BdAddress>>(propertyDetails);

            BdAddress deviceAddress = property->getMessage();

            ETG_TRACE_USR4(("onPropertyUpdate deviceAddress : %s", property->getMessage().c_str()));

            static_cast<YakinduSmVoiceCallIf*>(_smVoiceCallIf)->onDeviceDisconnected(property->getMessage());

            auto callStatusListIter = _callStatusList._callStatusList.find(deviceAddress);
            if(callStatusListIter != _callStatusList._callStatusList.end())
            {
               //all calls removed, removed device from list
               _callStatusList._callStatusList.erase(callStatusListIter);
            }
            onCallStatusListChanged();

            // If call removed is missed during device disconnection, this update will be used to inform clients about
            // calls removed in this device send a mic mute state if no pending calls found
            if((_callStatusList._callStatusList.empty()) && (0x01 == _microphoneMuteState._microphoneMuteState))
            {
                _microphoneMuteState._microphoneMuteState = 0x00;

                PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnMicrophoneMuteStateChanged(
                      _microphoneMuteState);
            }
            PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnPmStateChanged(
                  property->getBdAddress(), CALL_REMOVED);
         }
         break;

         case BTS_UPDATE_ALL_CALL_STATUS:
         {
            std::shared_ptr<PropertyDetails<uint8>> property =
                  std::static_pointer_cast<PropertyDetails<uint8>>(propertyDetails);
         }
         break;
         case BTS_UPDATE_VOICE_CALL_STATUS:
         {
            bool isCallStateUpdated = false;
            bool isMultipartyUpdated = false;
            bool sendUpdate = false;

            std::shared_ptr<PropertyDetails<VoiceCallProperty>> property =
                  std::static_pointer_cast<PropertyDetails<VoiceCallProperty>>(propertyDetails);

            CallStatusListMap::iterator it = _callStatusList._callStatusList.find(property->getBdAddress());

            if(it != _callStatusList._callStatusList.end())
            {
               CallInfoList::iterator itrList;
               for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
               {
                  if(itrList->_instance == property->getMessage()._callInstance)
                  {
                     switch(property->getMessage()._propertyId)
                     {
                        case VOICE_CALL_LINEIDENTIFICATION:
                        {
                           itrList->_telephoneNumber = *(static_cast<std::string*>(property->getMessage()._value));
                           sendUpdate = true;
                        }
                        break;
                        case VOICE_CALL_NAME:
                        {
                           //not used as of now
                        }
                        break;
                        case VOICE_CALL_STATE:
                        {
                           itrList->_state = *(static_cast<std::string*>(property->getMessage()._value));
                           if(0 == itrList->_state.compare("Disconnected"))
                           {
                              itrList->_state = "Idle";
                           }
                           isCallStateUpdated = true;
                        }
                        break;
                        case VOICE_CALL_MULTIPARTY:
                        {
                           it->second._multiparty = *(static_cast<bool*>(property->getMessage()._value));
                           isMultipartyUpdated = true;
                        }
                        break;
                        case VOICE_CALL_EMERGENCY:
                        {
                           //not used as of now
                        }
                        break;
                        case VOICE_CALL_DIRECTION:
                        {
                           itrList->_direction = *(static_cast<CallDirection*>(property->getMessage()._value));
                        }
                        break;
                        case VOICE_CALL_MODE:
                        {
                           itrList->_mode = *(static_cast<CallMode*>(property->getMessage()._value));
                        }
                        break;
                        case VOICE_CALL_TYPE:
                        {
                           itrList->_type = *(static_cast<NumberType*>(property->getMessage()._value));
                        }
                        break;
                        default:
                        {

                        }
                        break;
                     }
                     break;
                  }
               }
            }

            if((true == isCallStateUpdated) || (true == isMultipartyUpdated))
            {
               //Update should be triggered from SM if any change in state
               generateAgCallStateEventToSM(property->getBdAddress());
            }
            else if(true == sendUpdate)
            {
               //CallInfo update should be sent to Clients.
               onCallStatusListChanged();
            }
            else
            {
               ETG_TRACE_USR4(("CallStatusListChanged, but not updated to clients"));
            }
         }
         break;
         case BTS_UPDATE_SCO_TYPE:
         {
            std::shared_ptr<PropertyDetails<SCOConnection>> property =
                  std::static_pointer_cast<PropertyDetails<SCOConnection>>(propertyDetails);

            BdAddress deviceAddress = property->getBdAddress();
            SCOStatus scoStatus = property->getMessage()._scoStatus;

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress : %s", deviceAddress.c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate ScoStatus : %u", ETG_CENUM(SCOStatusEnumType, scoStatus)));

            DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
            DeviceRole deviceRole = DEVICEROLE_DEFAULT;
            deviceInfoHandler.getDeviceRole(deviceAddress, deviceRole);

            if (DEVICEROLE_ACTIVE == deviceRole)
            {
               if((!isCallsInIdleState(deviceAddress)) && (SCO_NOT_ESTABLISHED == scoStatus))
               {
                  _smVoiceCallIf->onScoConnectionStatusChanged(property->getBdAddress(), SCO_PAUSED);
               }
               else
               {
                  _smVoiceCallIf->onScoConnectionStatusChanged(deviceAddress, scoStatus);
                  updatePMStateChanged(deviceAddress, scoStatus);
               }
            }
            else if (DEVICEROLE_PASSIVE == deviceRole)
            {
               if ((SCO_NOT_ESTABLISHED != scoStatus) && (SCO_DEFAULT != scoStatus))
               {
                  ETG_TRACE_USR4(("SCO established for a Passive device. Hence transferring to AG"));
                  if(nullptr != _smVoiceCallIf)
                  {
                     std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage =
                           getNewPmCoreIfMessage_TransferAudioRequest(deviceAddress,
                                 AUDIO_DIRECTION_AUDIOGATEWAY, PM_DEFAULT_ACT);
                     _smVoiceCallIf->transferAudioRequest(pmCoreIfMessage);
                  }
               }
               else if((isCallsInIdleState(deviceAddress)) && (SCO_NOT_ESTABLISHED == scoStatus))
               {
                   _smVoiceCallIf->onScoConnectionStatusChanged(deviceAddress, scoStatus);
               }
            }
         }
         break;
         case BTS_UPDATE_AUDIO:
         {
            std::shared_ptr<PropertyDetails<AudioDirection>> property =
                  std::static_pointer_cast<PropertyDetails<AudioDirection>>(propertyDetails);
         }
         break;
         case BTS_UPDATE_AUDIO_PATH:
         {
            std::shared_ptr<PropertyDetails<AudioPath>> property =
                  std::static_pointer_cast<PropertyDetails<AudioPath>>(propertyDetails);
         }
         break;
         case BTS_UPDATE_AUDIO_INDICATOR:
         {
            std::shared_ptr<PropertyDetails<uint8>> property =
                  std::static_pointer_cast<PropertyDetails<uint8>>(propertyDetails);
         }
         break;
         case BTS_UPDATE_INBAND_RINGING:
         {
            std::shared_ptr<PropertyDetails<bool>> property =
                  std::static_pointer_cast<PropertyDetails<bool>>(propertyDetails);

            ETG_TRACE_USR4(("onPropertyUpdate BdAddress     : %s", property->getBdAddress().c_str()));
            ETG_TRACE_USR4(("onPropertyUpdate InbandRinging : %d", property->getMessage()));

            std::map<BdAddress, bool>::iterator it = _inBandRingingListMap.find(property->getBdAddress());
            if(it != _inBandRingingListMap.end())
            {
               if(property->getMessage() != it->second)
               {
                  //Updating the Inband ringing flag
                  it->second = property->getMessage();
               }
            }
            else
            {
               //Inserting the Inband ringing flag
               _inBandRingingListMap.insert(std::pair<BdAddress, bool>(property->getBdAddress(), property->getMessage()));
            }
         }
         break;
         case BTS_UPDATE_CHLD_FEATURE:
         {
            std::shared_ptr<PropertyDetails<ChldFeaturesType>> property =
                  std::static_pointer_cast<PropertyDetails<ChldFeaturesType>>(propertyDetails);
         }
         break;
         case BTS_UPDATE_MOBILE_COUNTRY_CODE:
         {
            std::shared_ptr<PropertyDetails<MobileCountryCode>> property =
                  std::static_pointer_cast<PropertyDetails<MobileCountryCode>>(propertyDetails);
         }
         break;
         case BTS_UPDATE_MICROPHONE_VOLUME:
         {
            std::shared_ptr<PropertyDetails<VolumeType>> property =
                  std::static_pointer_cast<PropertyDetails<VolumeType>>(propertyDetails);
         }
         break;

         case AM_RINGTONE_CHANNEL_ACQUISITION_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_ACQUISITION_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), RINGTONE_CHANNEL_GRANTED);
         }
         break;

         case AM_PHONE_CHANNEL_ACQUISITION_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_ACQUISITION_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PHONE_CHANNEL_GRANTED);
         }
         break;

         case AM_PHONE_CHANNEL_ACQUISITION_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_ACQUISITION_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PHONE_CHANNEL_DENIED);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_ACQUISITION_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_WAITINGMODE_CHANNEL_ACQUISITION_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), WAITINGMODE_CHANNEL_GRANTED);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_ACQUISITION_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_WAITINGMODE_CHANNEL_ACQUISITION_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), WAITINGMODE_CHANNEL_DENIED);
         }
         break;

         case AM_RINGTONE_CHANNEL_PLAYAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_PLAYAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_SUCCESS);
         }
         break;

         case AM_PHONE_CHANNEL_PLAYAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_PLAYAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_SUCCESS);
         }
         break;

         case AM_PHONE_CHANNEL_PLAYAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_PLAYAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_FAILURE);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_PLAYAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_WAITINGMODE_CHANNEL_PLAYAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_SUCCESS);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_PLAYAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_WAITINGMODE_CHANNEL_PLAYAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PLAY_AUDIO_FAILURE);
         }
         break;

         case AM_RINGTONE_CHANNEL_STOPAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_STOPAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_SUCCESS);
         }
         break;

         case AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_STOPAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_FAILURE);
         }
         break;

         case AM_PHONE_CHANNEL_STOPAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_STOPAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_SUCCESS);

            if (0x01 == _microphoneMuteState._microphoneMuteState)
            {
               _microphoneMuteState._microphoneMuteState = 0x00;
               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnMicrophoneMuteStateChanged(
                     _microphoneMuteState);
            }
         }
         break;

         case AM_PHONE_CHANNEL_STOPAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_STOPAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_FAILURE);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_STOPAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_WAITINGMODE_CHANNEL_STOPAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_SUCCESS);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_STOPAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_WAITINGMODE_CHANNEL_STOPAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), STOP_AUDIO_FAILURE);
         }
         break;

         case AM_RINGTONE_CHANNEL_PAUSEAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_PAUSEAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_SUCCESS);
         }
         break;

         case AM_RINGTONE_CHANNEL_PAUSEAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_RINGTONE_CHANNEL_PAUSEAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_FAILURE);
         }
         break;

         case AM_PHONE_CHANNEL_PAUSEAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_PAUSEAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_SUCCESS);
         }
         break;

         case AM_PHONE_CHANNEL_PAUSEAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_PHONE_CHANNEL_PAUSEAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_FAILURE);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_SUCCESS);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_WAITINGMODE_CHANNEL_PAUSEAUDIO_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PAUSE_AUDIO_FAILURE);
         }
         break;

         case AM_PHONE_CHANNEL_RELEASED:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("PHONE_CHANNEL_RELEASED- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), PHONE_CHANNEL_RELEASED);
         }
         break;

         case AM_RINGTONE_CHANNEL_RELEASED:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("RINGTONE_CHANNEL_RELEASED- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), RINGTONE_CHANNEL_RELEASED);
         }
         break;

         case AM_WAITINGMODE_CHANNEL_RELEASED:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("WAITINGMODE_CHANNEL_RELEASED- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _smVoiceCallIf->onAudioManagerEventUpdate(property->getBdAddress(), WAITINGMODE_CHANNEL_RELEASED);
         }
         break;

         case AM_SET_MIC_MUTE_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_SET_MIC_MUTE_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

               _microphoneMuteState._microphoneMuteState = 0x01;

               PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnMicrophoneMuteStateChanged(
                     _microphoneMuteState);
         }
         break;

         case AM_SET_MIC_MUTE_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_SET_MIC_MUTE_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            //TODO: Check on how to send error for MIC MUTE request failure
         }
         break;

         case AM_SET_MIC_UNMUTE_SUCCESS:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_SET_MIC_UNMUTE_SUCCESS- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            _microphoneMuteState._microphoneMuteState = 0x00;

            PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnMicrophoneMuteStateChanged(
                  _microphoneMuteState);
         }
         break;

         case AM_SET_MIC_UNMUTE_FAILURE:
         {
            std::shared_ptr<PropertyDetails<AmErrorMessage>> property =
                  std::static_pointer_cast<PropertyDetails<AmErrorMessage>>(propertyDetails);

            ETG_TRACE_USR4(("AM_SET_MIC_UNMUTE_FAILURE- DeviceAddress: %s",
                  property->getBdAddress().c_str()));

            //TODO: Check on how to send error for MIC UNMUTE request failure
         }
         break;

         default:
            ETG_TRACE_ERR(("CallController::Invalid propertyId"));
            break;
      }
   }

   // Request calls
   void CallController::getPhoneCallAudioActiveRequest(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetPhoneCallAudioActiveResponse(
            pmResult, _phoneCallAudioActive, act);
   }

   void CallController::getCallStatusListRequest(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetCallStatusListResponse(
            pmResult, _callStatusList, act);
   }

   void CallController::getMicrophoneMuteStateRequest(IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      // Response to client's request
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetMicrophoneMuteStateResponse(
            pmResult, _microphoneMuteState, act);
   }

   PmResult CallController::setMicrophoneMuteStateRequest(IN const MuteState muteState)
   {
      PmResult pmResult(PM_RESULT_ERR_GENERAL, "");

      if((muteState != _microphoneMuteState._microphoneMuteState) && (true == _phoneCallAudioActive._status))
      {
         (void)PmAudioManagerWrapper::getInstance().sendSetMicMuteState(muteState);
         pmResult._pmResultCode = PM_RESULT_OK;
      }

      return pmResult;
   }

   void CallController::mergeCallsRequest(IN std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->mergeCallsRequest(pmCoreIfMessage);
      }
   }

   void CallController::splitCallsRequest(IN std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->splitCallsRequest(pmCoreIfMessage);
      }
   }

   void CallController::hangupCallsRequest(IN std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->hangupCallsRequest(pmCoreIfMessage);
      }
      if(PM_RESULT_OK == pmCoreIfMessage->getPmResult()._pmResultCode)
      {
         CallStatusListMap::iterator it = _callStatusList._callStatusList.find(pmCoreIfMessage->getBdAddress());

         if(it != _callStatusList._callStatusList.end())
         {
            CallInfoList::iterator itrList;
            CallInstanceList instanceList = pmCoreIfMessage->getCallInstanceList();

            for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
            {
               if(find(instanceList.begin(), instanceList.end(), itrList->_instance)!=instanceList.end())
               {
                  itrList->_state = "Disconnecting";
                  onCallStatusListChanged();
               }
            }
         }
      }
   }

   void CallController::redialRequest(IN std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->redialRequest(pmCoreIfMessage);
      }
   }

   PmResult CallController::getWaitingModeStateRequest(IN const BdAddress& bdAddress, IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_ERR_DEVICE_NOT_EXIST, "");
      WaitingModeState waitingModeState(WAITING_MODE_IDLE);

      auto it = _waitingModeStateListMap.find(bdAddress);
      if(it != _waitingModeStateListMap.end())
      {
         waitingModeState = it->second;
         pmResult._pmResultCode = PM_RESULT_OK;
      }

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doGetWaitingModeStateResponse(pmResult,
            bdAddress, waitingModeState, act);

      return pmResult;
   }

   PmResult CallController::setWaitingModeFilePathRequest(IN const WaitingModeFilePath& waitingModeFilePath,
         IN const ActType act)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      _waitingModeFilePath._filePathNB  = waitingModeFilePath._filePathNB;
      _waitingModeFilePath._filePathWB  = waitingModeFilePath._filePathWB;

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doSetWaitingModeFilePathResponse(pmResult,act);

      return pmResult;
   }

   void CallController::getWaitingModeFilePath(WaitingModeFilePath& waitingModeFilePath)
   {
      waitingModeFilePath = _waitingModeFilePath;
   }

   void CallController::startStopWaitingModeRequest(IN std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->startStopWaitingModeRequest(pmCoreIfMessage);
      }
   }

   void CallController::dialRequest(IN std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::Dial BdAddress : \"%50s\" , TelephoneNumber : \"%50s\" ",
            pmCoreIfMessage->getBdAddress().c_str(), pmCoreIfMessage->getTelephoneNumber().c_str()));

      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->dialRequest(pmCoreIfMessage);
      }
   }

   void CallController::acceptCallRequest(IN std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->acceptCallRequest(pmCoreIfMessage);
      }
   }

   void CallController::swapCallRequest(IN std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->swapCallRequest(pmCoreIfMessage);
      }
   }

   void CallController::speedDialRequest(IN std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->speedDialRequest(pmCoreIfMessage);
      }
   }

   PmResult CallController::sendDTMFRequest(IN std::shared_ptr<PmCoreIfMessage_SendDTMFRequest> pmCoreIfMessage)
   {
      PmResult pmResult(PM_RESULT_OK, "");
      BTSResult btsResult(BTS_REQ_FAILED, ::ccdbusif::evolution::ERROR_UNKNOWN, "");

      btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendSendTonesRequest(pmCoreIfMessage->getBdAddress(),
            pmCoreIfMessage->getDTMFTones(), pmCoreIfMessage->getAct());

      if(btsResult._btsRequestResult != BTS_REQ_SUCCESS)
      {
         NotificationEvent notificationEvent;
         notificationEvent._bdAddress = pmCoreIfMessage->getBdAddress();
         notificationEvent._eventName = SEND_DTMF_FAILED;
         notificationEvent._reason = btsResult._errorMessage;
         pmResult._pmResultCode = PM_RESULT_ERR_GENERAL;
         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(notificationEvent);
      }

      return pmResult;
   }

   void CallController::transferAudioRequest(IN std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->transferAudioRequest(pmCoreIfMessage);
      }
   }

   // Result calls
   void CallController::requestPhoneNumberResult(IN const BTSResult btsResult, IN const BdAddress& bdAddress,
         IN const TelephoneNumber& telephoneNumber, IN const ActType act)
   {
      (void)bdAddress;       //To remove gen4 warning
      (void)telephoneNumber; //To remove gen4 warning
      (void)btsResult;       //To remove gen4 warning
      (void)act;
   }

   void CallController::acceptSCOConnectResult(
         IN std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->acceptSCOConnectResult(pmCoreIfMessage);
      }
   }

   void CallController::getCallsResult(IN std::shared_ptr<PmCoreIfMessage_GetCallsResult> pmCoreIfMessage)
   {
      if(pmCoreIfMessage->getBTSVoiceCallInfoList().size() == 0)
      {
         return;
      }

      auto it = _callStatusList._callStatusList.find(pmCoreIfMessage->getBdAddress());
      BTSVoiceCallInfoList btsVoiceCallInfoList = pmCoreIfMessage->getBTSVoiceCallInfoList();

      if(it == _callStatusList._callStatusList.end())
      {
         CallStatus callStatus;

         for(auto it1 = btsVoiceCallInfoList.begin(); it1 != btsVoiceCallInfoList.end(); it1++)
         {
            callStatus._multiparty &= it1->_multiparty;
            callStatus._callInfoList.push_back(it1->_callInfo);
         }

         _callStatusList._callStatusList.insert(std::pair<BdAddress, CallStatus>(
               pmCoreIfMessage->getBdAddress(), callStatus));
      }
      else
      {
         //already call status list updated. call instance availability to be checked then updated.
         for(auto& callInfo : btsVoiceCallInfoList)
         {
            auto it2 = std::find_if(it->second._callInfoList.begin(), it->second._callInfoList.end(),
                  [&callInfo](CallInfo const& obj){return obj._instance == callInfo._callInfo._instance;});

            if(it2 == it->second._callInfoList.end())
            {
               it->second._multiparty = callInfo._multiparty;
               it->second._callInfoList.push_back(callInfo._callInfo);
            }
         }
      }

      //Update should be triggered from SM
      if(_callStatusList._callStatusList.size() > 0)
      {
         generateAgCallStateEventToSM(pmCoreIfMessage->getBdAddress());
      }
   }

   void CallController::dialResult(IN std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::DialResult bdAddress :\"%50s\" ", pmCoreIfMessage->getBdAddress().c_str()));

      PmResult pmResult(PM_RESULT_OK, "");

      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->dialResult(pmCoreIfMessage);
      }

      convertBTSResultToPmResult(pmCoreIfMessage->getBTSResult(), pmResult);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doDialResponse(pmResult,
            pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getCallInstance(), pmCoreIfMessage->getAct());
   }

   void CallController::speedDialResult(IN std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::speedDialResult bdAddress :\"%50s\" ", pmCoreIfMessage->getBdAddress().c_str()));

      PmResult pmResult(PM_RESULT_OK, "");

      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->speedDialResult(pmCoreIfMessage);
      }

      convertBTSResultToPmResult(pmCoreIfMessage->getBTSResult(), pmResult);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doSpeedDialResponse(pmResult,
            pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getCallInstance(), pmCoreIfMessage->getAct());
   }

   void CallController::redialResult(IN std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::redialResult bdAddress :\"%50s\" ", pmCoreIfMessage->getBdAddress().c_str()));

      PmResult pmResult(PM_RESULT_OK, "");

      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->redialResult(pmCoreIfMessage);
      }

      convertBTSResultToPmResult(pmCoreIfMessage->getBTSResult(), pmResult);

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doRedialResponse(pmResult,
            pmCoreIfMessage->getBdAddress(), pmCoreIfMessage->getCallInstance(), pmCoreIfMessage->getAct());
   }

   void CallController::swapCallResult(IN std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->swapCallResult(pmCoreIfMessage);
      }
   }

   void CallController::releaseAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->releaseAndAcceptResult(pmCoreIfMessage);
      }
   }

   void CallController::releaseAndSwapResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->releaseAndSwapResult(pmCoreIfMessage);
      }
   }

   void CallController::holdAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->holdAndAcceptResult(pmCoreIfMessage);
      }
   }

   void CallController::hangupAllResult(IN std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->hangupAllResult(pmCoreIfMessage);
      }
   }

   void CallController::splitCallsResult(IN std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->splitCallsResult(pmCoreIfMessage);
      }
   }

   void CallController::mergeCallsResult(IN std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->mergeCallsResult(pmCoreIfMessage);
      }
   }

   void CallController::hangupMultipartyResult(IN std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->hangupMultipartyResult(pmCoreIfMessage);
      }
   }

   void CallController::sendDTMFResult(IN std::shared_ptr<PmCoreIfMessage_SendDTMFResult> pmCoreIfMessage)
   {
      PmResult pmResult(PM_RESULT_OK, "");

      convertBTSResultToPmResult(pmCoreIfMessage->getBTSResult(), pmResult);

      if(BTS_REQ_SUCCESS != pmCoreIfMessage->getBTSResult()._btsRequestResult)
      {
         NotificationEvent notificationEvent;
         notificationEvent._bdAddress = pmCoreIfMessage->getBdAddress();
         notificationEvent._eventName = SEND_DTMF_FAILED;
         notificationEvent._reason = pmCoreIfMessage->getBTSResult()._errorMessage;
         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnNotificationEvent(notificationEvent);
      }
      ETG_TRACE_USR4(("sendDTMFResult:PmResult : %u", ETG_CENUM(PmResultCode, pmResult._pmResultCode)));
   }

   void CallController::hangupCallResult(IN std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->hangupCallResult(pmCoreIfMessage);
      }
   }

   void CallController::acceptCallResult(IN std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->acceptCallResult(pmCoreIfMessage);
      }
   }

   void CallController::holdIncomingCallResult(IN std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->holdIncomingCallResult(pmCoreIfMessage);
      }
   }

   void CallController::transferAudioResult(IN std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->transferAudioResult(pmCoreIfMessage);
      }
   }

   void CallController::generateAgCallStateEventToSM(IN const BdAddress& bdAddress)
   {
      CallStatusListMap::iterator it = _callStatusList._callStatusList.find(bdAddress);
      CallStateEnumType callState = CALL_STATE_UNKNOWN;

      if(_smVoiceCallIf != nullptr)
      {
         static_cast<YakinduSmVoiceCallIf*>(_smVoiceCallIf)->setCallCount(bdAddress,
               static_cast<CallCount>(it->second._callInfoList.size()));
      }

      if(it != _callStatusList._callStatusList.end())
      {
         if(it->second._callInfoList.size() == VALUE_ONE)
         {
            CallInfoList::iterator itrList;
            for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
            {
               bool stopCallInstanceTimer = false;

               if(0 == itrList->_state.compare("Idle"))
               {
                  ETG_TRACE_USR4(("CALL_STATE_IDLE"));
                  callState = CALL_STATE_IDLE;

                  stopCallInstanceTimer = true;
               }
               else if(0 == itrList->_state.compare("Incoming"))
               {
                  //FIXME:
                  //Check if auto waiting mode is enabled for this device (directly post StartWaitingMode request)
                  //only if- All other devices are in Idle state (No calls and No VR)

                  callState = CALL_STATE_INCOMING;

                  RingtoneName ringtoneName;
                  PmCoreMainController::getInstance().getPmSettings().getRingtoneName(bdAddress, ringtoneName);

                  if(0 == ringtoneName.compare(RINGTONE_NAME_INBAND))
                  {
                     callState = CALL_STATE_INCOMING_INBAND;
                  }

                  stopCallInstanceTimer = true;
               }
               else if((0 == itrList->_state.compare("Alerting")) || (0 == itrList->_state.compare("Dialing")))
               {
                  callState = CALL_STATE_DIALING;
                  stopCallInstanceTimer = true;
               }
               else if(0 == itrList->_state.compare("Held"))
               {
                  callState = CALL_STATE_HELD;

                  std::string callInstanceToken = bdAddress + "/" + std::to_string(itrList->_instance);
                  startAsfTimer(callInstanceToken,ONE_SECOND_IN_MILLISECONDS, TIMER_REPEAT_COUNT_DEFAULT);
               }
               else if(0 == itrList->_state.compare("Active"))
               {
                  callState = CALL_STATE_ACTIVE;

                  std::string callInstanceToken = bdAddress + "/" + std::to_string(itrList->_instance);
                  startAsfTimer(callInstanceToken,ONE_SECOND_IN_MILLISECONDS, TIMER_REPEAT_COUNT_DEFAULT);
               }

               if (stopCallInstanceTimer)
               {
                  std::string callInstanceToken = bdAddress + "/" + std::to_string(itrList->_instance);
                  stopAsfTimer(callInstanceToken);
                  itrList->_durationHr = 0;
                  itrList->_durationMin = 0;
                  itrList->_durationSec = 0;
                  deleteAsfTimer(callInstanceToken);
               }
            }
         }
         else if(it->second._callInfoList.size() > VALUE_ONE)
         {
            unsigned int heldCount =0,activeCount=0;
            bool isIdle=false,isWaiting=false,isDialing=false;
            CallInfoList::iterator itrList;

            for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
            {
               bool stopCallInstanceTimer = false;

               if(0 == itrList->_state.compare("Idle"))
               {
                  ETG_TRACE_USR4(("CALL_STATE_IDLE"));
                  callState = CALL_STATE_IDLE;
                  isIdle = true;

                  stopCallInstanceTimer = true;
               }
               else if(0 == itrList->_state.compare("Incoming"))
               {
                  //Waiting call becomes incoming and Active call becomes Idle
                  ETG_TRACE_USR4(("CALL_STATE_INCOMING"));
                  isWaiting = true;

                  stopCallInstanceTimer = true;
               }
               else if(0 == itrList->_state.compare("Waiting"))
               {
                  ETG_TRACE_USR4(("CALL_STATE_CALL_X_WAITING"));
                  isWaiting = true;

                  stopCallInstanceTimer = true;
               }
               else if((0 == itrList->_state.compare("Alerting")) || (0 == itrList->_state.compare("Dialing")))
               {
                  ETG_TRACE_USR4(("CALL_STATE_CALL_X_DIALING"));
                  isDialing = true;

                  stopCallInstanceTimer = true;
               }
               else if(0 == itrList->_state.compare("Held"))
               {
                  ETG_TRACE_USR4(("heldCount++"));
                  heldCount++;

                  std::string callInstanceToken = bdAddress + "/" + std::to_string(itrList->_instance);
                  startAsfTimer(callInstanceToken,ONE_SECOND_IN_MILLISECONDS, TIMER_REPEAT_COUNT_DEFAULT);
               }
               else if(0 == itrList->_state.compare("Active"))
               {
                  ETG_TRACE_USR4(("activeCount++"));
                  activeCount++;

                  std::string callInstanceToken = bdAddress + "/" + std::to_string(itrList->_instance);
                  startAsfTimer(callInstanceToken,ONE_SECOND_IN_MILLISECONDS, TIMER_REPEAT_COUNT_DEFAULT);
               }

               if (stopCallInstanceTimer)
               {
                  std::string callInstanceToken = bdAddress + "/" + std::to_string(itrList->_instance);
                  stopAsfTimer(callInstanceToken);
                  itrList->_durationHr = 0;
                  itrList->_durationMin = 0;
                  itrList->_durationSec = 0;
                  deleteAsfTimer(callInstanceToken);
               }
            }

            if(isWaiting)
            {
               callState = CALL_STATE_CALL_X_WAITING;
            }
            if(isDialing)
            {
               callState = CALL_STATE_CALL_X_DIALING;
            }
            if(isWaiting && isIdle)
            {
               callState = CALL_STATE_INCOMING;

               RingtoneName ringtoneName;
               PmCoreMainController::getInstance().getPmSettings().getRingtoneName(bdAddress, ringtoneName);

               if(0 == ringtoneName.compare(RINGTONE_NAME_INBAND))
               {
                  callState = CALL_STATE_INCOMING_INBAND;
               }
            }
            if((activeCount > 0) && isIdle)
            {
               //Waiting call becomes Active and Active call becomes Idle(ReleaseAndAnswer)
               callState = CALL_STATE_ACTIVE;
            }
            if(isDialing && isIdle)
            {
               callState = CALL_STATE_DIALING;
            }
            if((heldCount == it->second._callInfoList.size()) || (activeCount == it->second._callInfoList.size()))
            {
               if(true == it->second._multiparty)
               {
                  ETG_TRACE_USR4(("CALL_STATE_CALLS_CONFERENCE"));
                  callState = CALL_STATE_CALLS_CONFERENCE;
               }
               else
               {
                  ETG_TRACE_USR4(("Intermediate Call Switching"));
               }
            }
            else if((heldCount + activeCount) == it->second._callInfoList.size())
            {
               if(false == it->second._multiparty)
               {
                  ETG_TRACE_USR4(("CALL_STATE_CALLS_SWAPPABLE"));
                  callState = CALL_STATE_CALLS_SWAPPABLE;
               }
               else
               {
                  ETG_TRACE_USR4(("Intermediate Call Switching"));
               }
            }
         }

         ETG_TRACE_USR4(("Generated AgCallState Event : %u", ETG_CENUM(CallStateEnumType, callState)));

         if((_smVoiceCallIf != nullptr) && (callState != CALL_STATE_UNKNOWN))
         {
            static_cast<YakinduSmVoiceCallIf*>(_smVoiceCallIf)->onAgCallStateUpdate(bdAddress, callState);
         }
      }
      else
      {
         ETG_TRACE_USR4(("Device not exist"));
      }
   }

   //TODO:Each time startAsfTimer is called from ACTIVE call state. This should be avoided
   void CallController::startAsfTimer(std::string token, ms_t duration, RepeatCount repeatCount)
   {
      ETG_TRACE_USR4(("CallController::startAsfTimer callInstanceToken: %s",token.c_str()));

      com::bosch::pmcommon::TimerData<std::string /*Token*/, ms_t /*Duration*/, RepeatCount> timerdata(
            token,(unsigned int)duration,(unsigned int)repeatCount);
      auto it = _asfTimerInstanceMap.find(token);
      if(it == _asfTimerInstanceMap.end())
      {
         ETG_TRACE_USR4(("starting the new timer at CALL_STATE_ACTIVE"));

         com::bosch::pmcommon::ITimer<CallController,std::string,ms_t,RepeatCount>* asfTimer = new AsfTimer<CallController,std::string>();
         //Inserting CallInstanceToken and asfTimer instance pair
         _asfTimerInstanceMap.emplace_hint(_asfTimerInstanceMap.end(),token, asfTimer);
         asfTimer->start(this,timerdata);
      }
      else
      {
         ETG_TRACE_USR4(("Timer instance found in _asfTimerInstanceMap"));

         if(it->second->isStopped())
         {
            ETG_TRACE_USR4(("starting the old timer at CALL_STATE_ACTIVE"));
            //Timer instance for the callInstanceToken is already available.
            //So reusing the same
            it->second->start(this,timerdata);
         }
      }
   }

   void CallController::stopAsfTimer(std::string token)
   {
      ETG_TRACE_USR4(("CallController::stopAsfTimer callInstanceToken: %s",token.c_str()));

      auto it = _asfTimerInstanceMap.find(token);

      if(it != _asfTimerInstanceMap.end())
      {
         if(true == it->second->isActive())
         {
            ETG_TRACE_USR4(("Stopping the active timer"));

            //Stop callInstance timer for IDLE call
            it->second->stop();
         }
      }
   }

   void CallController::deleteAsfTimer(std::string token)
   {
      ETG_TRACE_USR4(("CallController::deleteAsfTimer callInstanceToken: %s",token.c_str()));

      auto it = _asfTimerInstanceMap.find(token);

      if(it != _asfTimerInstanceMap.end())
      {
         ETG_TRACE_USR4(("CallController::deleteAsfTimer isActive :%d",it->second->isActive()));
         if(false == it->second->isActive())
         {
            ETG_TRACE_USR4(("Deleting the stopped timer"));

            //Deleting the timer instance
            delete it->second;

            //Erasing the timer instance
            _asfTimerInstanceMap.erase(it);
            ETG_TRACE_USR4(("CallController::_asfTimerInstanceMap.size() :%d",_asfTimerInstanceMap.size()));
         }
      }
   }

   void CallController::timerElapsed(com::bosch::pmcommon::TimerData<std::string,ms_t,RepeatCount> data,
		   com::bosch::pmcommon::AsfTimerCallbackData timerCallbackData)
   {
      ETG_TRACE_USR4(("CallController::timerCallbackData._repeatCount : %d", timerCallbackData._repeatCount));
      ETG_TRACE_USR4(("CallController::_timerInfo : %s", data._timerInfo.c_str()));

      std::size_t found = data._timerInfo.find(SCO_TIMER_ID);
      if (found != std::string::npos)
      {
         // SCO is in Paused state.On timer expiry, DISCONNECT the SCO. Also timer repeat count is 1.
         // Hence no need to stop the timer.
         BdAddress deviceAddress = data._timerInfo.substr(0,12);
         _smVoiceCallIf->onScoConnectionStatusChanged(deviceAddress, SCO_NOT_ESTABLISHED);
         updatePMStateChanged(deviceAddress, SCO_NOT_ESTABLISHED);
      }
      else
      {
         CallStatusListMap::iterator it = _callStatusList._callStatusList.find(data._timerInfo.substr(0,12));
         CallInfoList::iterator itrList;

         if(it != _callStatusList._callStatusList.end())
         {
            for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
            {
               if(std::to_string(itrList->_instance) == data._timerInfo.substr(13,1))
               {
                  //Repeat count received as ZERO on first timerElapsed
                  unsigned int repeatCount = timerCallbackData._repeatCount + 1;
                  //hours
                  repeatCount = repeatCount % NUMBER_OF_SECONDS_PER_DAY;
                  itrList->_durationHr = static_cast<Duration>(repeatCount / NUMBER_OF_SECONDS_PER_HOUR);

                  //minutes
                  repeatCount %= NUMBER_OF_SECONDS_PER_HOUR;
                  itrList->_durationMin = static_cast<Duration>(repeatCount / NUMBER_OF_SECONDS_PER_MINUTE);

                  //seconds
                  repeatCount %= NUMBER_OF_SECONDS_PER_MINUTE;
                  itrList->_durationSec = static_cast<Duration>(repeatCount);

                  ETG_TRACE_USR4(("CallController::itrList->_durationHr  :%u", itrList->_durationHr));
                  ETG_TRACE_USR4(("CallController::itrList->_durationMin :%u", itrList->_durationMin));
                  ETG_TRACE_USR4(("CallController::itrList->_durationSec :%u", itrList->_durationSec));

                  //Timer event to handle SCO connect,disconnect during INBAND call accept
                  if(itrList->_durationHr == 0 && itrList->_durationMin == 0 &&  itrList->_durationSec == INBAND_ACCEPT_WAIT_TIME)
                  {
                     ETG_TRACE_USR4(("CallController::onTimerEventUpdate"));
                     static_cast<YakinduSmVoiceCallIf*>(_smVoiceCallIf)->onTimerEventUpdate(data._timerInfo.substr(0,12));
                  }
               }
            }
         }

         onCallStatusListChanged();
      }
   }

   void CallController::onCallStatusListChanged()
   {
      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnCallStatusListChanged(_callStatusList);
   }

   void CallController::onWaitingModeStateChanged(IN const BdAddress& bdAddress,
         IN const WaitingModeState& waitingModeState)
   {
      ETG_TRACE_USR4(("CallController::onWaitingModeStateChanged :%d", waitingModeState._waitingModeState));

      PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnWaitingModeStateChanged(
            bdAddress, waitingModeState);
   }

   void CallController::updateWaitingModeState(IN const BdAddress& bdAddress,
         IN const WaitingModeState& waitingModeState)
   {
      ETG_TRACE_USR4(("CallController::updateWaitingModeState :%d", waitingModeState._waitingModeState));
      auto it = _waitingModeStateListMap.find(bdAddress);

      if(it != _waitingModeStateListMap.end())
      {
         if(waitingModeState != it->second)
         {
            it->second = waitingModeState;

            ETG_TRACE_USR4(("CallController::updateWaitingModeState update success"));
            onWaitingModeStateChanged(bdAddress, waitingModeState);
         }
      }
      else
      {
         ETG_TRACE_USR4(("CallController::updateWaitingModeState Inserting new element into a map"));

         //Inserting new element into a map
         _waitingModeStateListMap.emplace_hint(_waitingModeStateListMap.end(), bdAddress, waitingModeState);

         onWaitingModeStateChanged(bdAddress, waitingModeState);
      }
   }

   void CallController::acceptWaitingModeCall(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR4(("CallController::acceptWaitingModeCall"));
      CallStatusListMap::iterator it = _callStatusList._callStatusList.find(bdAddress);

      if(it != _callStatusList._callStatusList.end())
      {
         CallInfoList::iterator itrList;
         for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
         {
            if(0 == itrList->_state.compare("Incoming"))
            {
               BTSResult btsResult;

               //Accepting the waiting mode call.
               btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendAnswerRequest(bdAddress,
                     itrList->_instance, PM_DEFAULT_ACT);

               if(btsResult._btsRequestResult != BTS_REQ_SUCCESS)
               {
                  ETG_TRACE_ERR(("CallController::acceptWaitingModeCall BTS_REQ_FAILED"));
                  std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage = getNewPmCoreIfMessage_AcceptCallResult(btsResult,
                        bdAddress, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT);
                  acceptCallResult(pmCoreIfMessage);
               }
            }
         }
      }
   }

   void CallController::holdAndAcceptWaitingModeCall(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR4(("CallController::holdAndAcceptWaitingModeCall"));
      CallStatusListMap::iterator it = _callStatusList._callStatusList.find(bdAddress);

      if(it != _callStatusList._callStatusList.end())
      {
         CallInfoList::iterator itrList;
         for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
         {
            if(0 == itrList->_state.compare("Waiting"))
            {
               BTSResult btsResult;

               //Accepting the waiting mode call.
               btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendHoldAndAnswerRequest(bdAddress,
                     PM_DEFAULT_ACT);

               if(btsResult._btsRequestResult != BTS_REQ_SUCCESS)
               {
                  ETG_TRACE_ERR(("CallController::holdAndAcceptWaitingModeCall BTS_REQ_FAILED"));
                  std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage = getNewPmCoreIfMessage_HoldAndAcceptResult(btsResult,
                        bdAddress, PM_DEFAULT_ACT, PM_CORE_IF_MSG_ORIGIN_CLIENT);
                  holdAndAcceptResult(pmCoreIfMessage);
               }
            }
         }
      }
   }

   void CallController::onCallAudioActiveChanged(IN const BdAddress& bdAddress, IN const AudioActiveStatus& audioActiveStatus)
   {
      ETG_TRACE_USR1(("CallController::onCallAudioActiveChanged"));

      DeviceInfoHandler& deviceInfoHandler = PmCoreMainController::getInstance().getDeviceInfoHandler();
      DeviceRole deviceRole = DEVICEROLE_DEFAULT;
      deviceInfoHandler.getDeviceRole(bdAddress, deviceRole);

      if (DEVICEROLE_ACTIVE == deviceRole)
      {
         _phoneCallAudioActive._status = audioActiveStatus;
         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnPhoneCallAudioActiveChanged(
               _phoneCallAudioActive._status);
      }
   }

   void CallController::prepareAudioRouteResponse(
         IN std::shared_ptr<PmCoreIfMessage_PrepareAudioRouteResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::prepareAudioRouteResponse- BdAddress(): %s",
            pmCoreIfMessage->getBdAddress().c_str()));
   }

   void CallController::playAudioResponse(
         IN std::shared_ptr<PmCoreIfMessage_PlayAudioResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::playAudioResponse- BdAddress(): %s",
            pmCoreIfMessage->getBdAddress().c_str()));
   }

   void CallController::stopAudioResponse(
         IN std::shared_ptr<PmCoreIfMessage_StopAudioResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::stopAudioResponse- BdAddress(): %s",
            pmCoreIfMessage->getBdAddress().c_str()));
   }

   void CallController::setMicMuteStateResponse(
         IN std::shared_ptr<PmCoreIfMessage_SetMicMuteStateResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::setMicMuteStateResponse- BdAddress(): %s",
            pmCoreIfMessage->getBdAddress().c_str()));
   }

   void CallController::pauseAudioResponse(
         IN std::shared_ptr<PmCoreIfMessage_PauseAudioResponse> pmCoreIfMessage)
   {
      ETG_TRACE_USR1(("CallController::pauseAudioResponse- BdAddress(): %s",
            pmCoreIfMessage->getBdAddress().c_str()));
   }

   bool CallController::isInbandRinging(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR1(("CallController::isInbandRinging:deviceAddress: %s",deviceAddress.c_str()));

      bool isInbandRingingEnabled = false;
      auto it = _inBandRingingListMap.find(deviceAddress);

      if(it != _inBandRingingListMap.end())
      {
         isInbandRingingEnabled = it->second;
      }

      return isInbandRingingEnabled;
   }

   bool CallController::isCallsInIdleState(IN const BdAddress& deviceAddress)
   {
      auto iter = _callStatusList._callStatusList.find(deviceAddress);
      bool isCallsInIdleState = true;

      if (iter != _callStatusList._callStatusList.end())
      {
         isCallsInIdleState = false;
      }

      return isCallsInIdleState;
   }

   void CallController::switchToActive(IN const BdAddress& deviceAddress)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->switchToActiveRequest(deviceAddress);
      }
   }

   void CallController::switchToPassive(IN const BdAddress& deviceAddress)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->switchToPassiveRequest(deviceAddress);
      }
   }

   void CallController::switchToPassiveClientRequest(IN const BdAddress& deviceAddress)
   {
      if(nullptr != _smVoiceCallIf)
      {
         _smVoiceCallIf->switchToPassiveClientRequest(deviceAddress);
      }
   }

   void CallController::postGetCallsRequest2BtStack(IN const BdAddress& deviceAddress)
   {
      ETG_TRACE_USR4(("CallController::postGetCallsRequest2BtStack() entered"));

      BTSResult btsResult = evobtstackwrapper::EvoBtStackWrapper::getInstance().sendGetCallsRequest(deviceAddress,
            PM_DEFAULT_ACT);

      PmResult pmResult(PM_RESULT_OK, "");
      convertBTSResultToPmResult(btsResult, pmResult);

      if (PM_RESULT_OK != pmResult._pmResultCode)
      {
         ETG_TRACE_USR4(("CallController::postGetCallsRequest2BtStack(): Error in posting GetCalls request"));
      }
   }

   TelephoneNumber CallController::getTelephoneNumberFromCallInstance(IN const BdAddress& bdAddress,
         IN const CallInstance& callInstance)
   {
      CallStatusListMap::iterator it = _callStatusList._callStatusList.find(bdAddress);
      CallInfoList::iterator itrList;

      TelephoneNumber telephoneNumber = "";
      if(it != _callStatusList._callStatusList.end())
      {
         for(itrList = it->second._callInfoList.begin(); itrList != it->second._callInfoList.end(); itrList++)
         {
            if(itrList->_instance == callInstance)
            {
               telephoneNumber = itrList->_telephoneNumber;
               break;
            }
         }
      }

      return telephoneNumber;
   }

   void CallController::processSCODisconnection(IN const BdAddress& bdAddress)
   {
      ETG_TRACE_USR4(("CallController::processSCODisconnection() entered with bdAddress: %20s", bdAddress.c_str()));
      _smVoiceCallIf->onScoConnectionStatusChanged(bdAddress, SCO_NOT_ESTABLISHED);
      std::string token = bdAddress + "/" + SCO_TIMER_ID;
      stopAsfTimer(token);

      updatePMStateChanged(bdAddress, SCO_NOT_ESTABLISHED);
   }

   void CallController::updatePMStateChanged(IN const BdAddress& bdAddress, IN const SCOStatus scoStatus)
   {
      ETG_TRACE_USR4(("CallController::updatePMStateChanged() entered with bdAddress: %20s", bdAddress.c_str()));

      bool updateCallExistenceState = true;
      CallExistenceState callExistenceState = CALL_PRESENT_WITHOUT_SCO;

      if (isCallsInIdleState(bdAddress))
      {
         // No call present
         if ((SCO_NOT_ESTABLISHED != scoStatus) && (SCO_DEFAULT != scoStatus))
         {
            // SCO established. Do nothing
            updateCallExistenceState = false;
         }
         else
         {
            // Call might be removed or this may be due to VR. Hence this should be appropriately handled in the
            // corresponding handlers (While setting Phone sub state or setting BT Prof usage).
            callExistenceState = CALL_REMOVED;
         }
      }
      else
      {
         callExistenceState = CALL_PRESENT_WITHOUT_SCO;
         // Call is present
         if ((SCO_NOT_ESTABLISHED != scoStatus) && (SCO_DEFAULT != scoStatus))
         {
            // SCO established
            callExistenceState = CALL_PRESENT_WITH_SCO;
         }
      }

      if (updateCallExistenceState)
         PmCoreMainController::getInstance().getPmCoreCallbackIfWrapper().doOnPmStateChanged(
               bdAddress, callExistenceState);
   }

} // namespace pmcore
