/**
 * @file PmCallHandler.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the PmCallHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#ifndef PmCallHandler_h
#define PmCallHandler_h

#include "EvoBtStackWrapper.h"
#include "PmCoreIfMessageResult.h"
#include "../src-gen/PmCallStateHandlerSm.h"

namespace pmcore
{
   class PmCallHandler final: public PmCallStateHandlerSm
   {
   public:
      /**
       * Parameterized Constructor of PmCallHandler class
       *
       * @param[in] deviceAddress - device address
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmCallHandler(const BdAddress& deviceAddress);

      /**
       * Destructor of PmCallHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~PmCallHandler();

      /**
       * This function is used to get the deviceAddress of the SM instance
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return BdAddress - device address
       */
      const BdAddress& getDeviceAddress() { return _deviceAddress; }

      /**
       * This function is used to update the callcount to state machine.
       *
       * @param[in] callCount - Total call count
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setCallCount(IN const CallCount callCount);

      /**
       * This function is used to post the AgCallState event to state machine.
       *
       * @param[in] callState - AG call state update
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAgCallStateUpdate(IN const CallStateEnumType callState);

      /**
       * This function is used to post the AudioManager event to state machine.
       *
       * @param[in] amEvent - AudioManager event update
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onAudioManagerEventUpdate(IN const AudioManagerEventType amEvent);

      /**
       * This function is used to post the SCO connection status to state machine.
       *
       * @param[in] scoStatus - SCO connection status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onScoConnectionStatusChanged(IN const SCOStatus scoStatus);

      /**
       * This function is used to post the timer event to state machine.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onTimerEventUpdate();

      // Request calls
      /**
       * This function is used to post the merge call request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for MergeCallsRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void mergeCallsRequest(IN std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage);

      /**
       * This function is used to post the split call request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for SplitCallsRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void splitCallsRequest(IN std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage);

      /**
       * This function is used to post the hangup call request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupCallsRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupCallsRequest(IN std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage);

      /**
       * This function is used to post the merge redial request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for RedialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void redialRequest(IN std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage);

      /**
       * This function is used to post the start stop waiting mode request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for StartStopWaitingModeRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void startStopWaitingModeRequest(IN std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage);

      /**
       * This function is used to post the dial request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for DialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void dialRequest(IN std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage);

      /**
       * This function is used to post the accept call request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for AcceptCallRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptCallRequest(IN std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage);

      /**
       * This function is used to post the swap call request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for SwapCallRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void swapCallRequest(IN std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage);

      /**
       * This function is used to post the speed dial request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for SpeedDialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void speedDialRequest(IN std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage);

      /**
       * This function is used to post the transfer audio request event to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for TransferAudioRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void transferAudioRequest(IN std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage);

      /**
       * This function is used to post the transfer audio request directly to stack
       *
       * @param[in] audio - Audio direction. Should be "AG" or "HF"
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void transferAudioRequest(IN const std::string& audio);

      /**
       * This function is used to switch a device to Passive
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveRequest();

      /**
       * This function is used to switch a device to Active
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToActiveRequest();

      /**
       * This function is used to auto reject a call
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void autoRejectCallRequest(const AutoRejectCall autoRejectCall);

      // **************************************************************************************************
      // Result calls
      /**
       * This function is used to post the accept sco connect result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for AcceptSCOConnectResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptSCOConnectResult(IN std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage);

      /**
       * This function is used to post the dial result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for DialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void dialResult(IN std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage);

      /**
       * This function is used to post the speed dial result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for SpeedDialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void speedDialResult(IN std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage);

      /**
       * This function is used to post the redial result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for RedialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void redialResult(IN std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage);

      /**
       * This function is used to post the swap call result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for SwapCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void swapCallResult(IN std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage);

      /**
       * This function is used to post the release and accept result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for ReleaseAndAcceptResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void releaseAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage);

      /**
       * This function is used to post the release and swap result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for ReleaseAndSwapResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void releaseAndSwapResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage);

      /**
       * This function is used to post the hold and accept result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for HoldAndAcceptResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void holdAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage);

      /**
       * This function is used to post the hangup all result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupAllResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupAllResult(IN std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage);

      /**
       * This function is used to post the split calls result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for SplitCallsResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void splitCallsResult(IN std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage);

      /**
       * This function is used to post the merge calls result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for MergeCallsResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void mergeCallsResult(IN std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage);

      /**
       * This function is used to post the hangup multiparty result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupMultipartyResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupMultipartyResult(IN std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage);

      /**
       * This function is used to post the hangup call result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupCallResult(IN std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage);

      /**
       * This function is used to post the accept call result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for AcceptCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptCallResult(IN std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage);

      /**
       * This function is used to post the hold incoming call result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for HoldIncomingCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void holdIncomingCallResult(IN std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage);

      /**
       * This function is used to post the Transfer Audio Result to state machine.
       *
       * @param[in] pmCoreIfMessage - shared ptr for TransferAudioResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void transferAudioResult(IN std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage);

      /**
       * This method is used to inform about the removal of a call
       *
       * @param[in] callInstance - callInstance of the removed call
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onCallRemoved(IN const CallInstance callInstance);

   private:
      /**
       * Constructor of PmCallHandler class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      PmCallHandler();

      /**
       * This function is used to process the Error response received from Stack without posting it to SM.
       *
       * @param[in]  msg - PmCoreIfMsg
       *             errorInfo - Error info Enum
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void processErrorResponseFromStack(PmCoreIfMessage * msg, SmErrorEnumType errorInfo);

      /**
       * SmUserActionIf class inherits the functions of SCI_UserAction_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmUserActionIf final: public SCI_UserAction_OCB
      {
      public:
         SmUserActionIf(PmCallHandler& pmCallHandler): _pmCallHandler(pmCallHandler){}

         ~SmUserActionIf(){}

         virtual void dial(PmCoreIfMessage_DialRequest * msg) override;

         virtual void acquirePhoneAudio(PmCoreIfMessage_AcceptCallRequest * msg) override;

         virtual void speedDial(PmCoreIfMessage_SpeedDialRequest * msg) override;

         virtual void reDial(PmCoreIfMessage_RedialRequest * msg) override;

         virtual void hangupCalls(PmCoreIfMessage_HangupCallsRequest * msg) override;

         virtual void hangupAllCalls(PmCoreIfMessage_HangupCallsRequest * msg) override;

         virtual void swapCalls(PmCoreIfMessage_SwapCallRequest * msg) override;

         virtual void splitCalls(PmCoreIfMessage_SplitCallsRequest * msg) override;

         virtual void conference(PmCoreIfMessage_MergeCallsRequest * msg) override;

         virtual void transferAudio(PmCoreIfMessage_TransferAudioRequest * msg) override;

         virtual void muteRingtone(PmCoreIfMessage_SetRingtoneMuteStateRequest * msg) override;

         virtual void postTransferAudio(sc_string audio) override;

         virtual void postAcceptCall() override;

         virtual void postHoldAndAcceptCall() override;

         virtual void postCancelOutgoing() override;

         virtual void setTransferAudioToHfToSuccess(pmcore::PmCoreIfMessage_TransferAudioRequest * msg) override;

      private:
         PmCallHandler& _pmCallHandler;
      };

      /**
       * SmAgCallStateUpdateIf class inherits the functions of SCI_AgCallStateUpdate_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAgCallStateUpdateIf final: public SCI_AgCallStateUpdate_OCB
      {
      public:
         SmAgCallStateUpdateIf(PmCallHandler& pmCallHandler): _pmCallHandler(pmCallHandler){}

         ~SmAgCallStateUpdateIf(){}

         virtual void waitingCall() override;
      private:
         PmCallHandler& _pmCallHandler;
      };

      /**
       * SmSmHandlerIf class inherits the functions of SCI_SmHandler_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmSmHandlerIf final: public SCI_SmHandler_OCB
      {
      public:
         SmSmHandlerIf(PmCallHandler& pmCallHandler) : _pmCallHandler(pmCallHandler), _acceptCallParam(){}

         ~SmSmHandlerIf()
         {

         }

         virtual void stopSm() override;

         virtual void enterIdle() override;

         virtual void updatePhoneCallAudioActive(sc_boolean audioActive) override;

         virtual void updateCallStatus() override;

         virtual void updateVehicleAudioStastus() override;

         virtual void callDisconnected() override;

         virtual void updateWaitingModeState(WaitingModeStateEnumType waitingModeState) override;

         virtual void acceptWaitingModeCall() override;

         virtual void holdAndAcceptWaitingModeCall() override;

         virtual void postSwitchToPassiveResponse(PmResultMessage result) override;

         virtual StopAudioStateEnumType isStopAudioRequested() override;

         virtual sc_boolean isRingtoneSuppressed() override;

         virtual sc_boolean isInbandRinging() override;

         virtual void traceMessage(sc_string trace) override;

         virtual void generateCallStateEvent() override;

         virtual void acceptCallFinished() override;

         virtual void handleSwitchToActiveInOutbandRinging() override;

         virtual void processAllCallsRemoved() override;

         virtual sc_boolean isChannelDenied() override;

         virtual void setAcceptCallParam(pmcore::PmCoreIfMessage_AcceptCallRequest * msg) override;

         virtual pmcore::PmCoreIfMessage_AcceptCallRequest * getAcceptCallParam() override;

         virtual void printPhoneAudioState(PmCorePhoneAudioState audioState) override;

         virtual void updateErrorStatus(SmErrorEnumType errorInfo) override;

         virtual void updateCallErrorStatus(PmCoreIfMessage * msg, SmErrorEnumType errorInfo) override;

         virtual void handleSCODisconnection() override;

         virtual void startSCOTimer() override;

         virtual void stopSCOTimer() override;

      private:
         PmCallHandler& _pmCallHandler;
         pmcore::PmCoreIfMessage_AcceptCallRequest _acceptCallParam;
      };

      /**
       * SmAudioManagerRequestIf class inherits the functions of SCI_AudioManagerRequest_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAudioManagerRequestIf final: public SCI_AudioManagerRequest_OCB
      {
      public:
         SmAudioManagerRequestIf(PmCallHandler& pmCallHandler): _pmCallHandler(pmCallHandler){}

         ~SmAudioManagerRequestIf(){}

         virtual void requestPhoneAudio() override;

         virtual void requestWaitingModeAudio() override;

         virtual void playHfAudio(pmcore::SCOStatus scoStatus) override;

         virtual void playRingtone() override;

         virtual void playWaitingModeTone(SCOStatus scoStatus) override;

         virtual void pauseHfAudio() override;

         virtual void stopAudio() override;

      private:
         PmCallHandler& _pmCallHandler;
      };

      /**
       * SmAgResponseIf class inherits the functions of SCI_AgResponse_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmAgResponseIf final: public SCI_AgResponse_OCB
      {
      public:
         SmAgResponseIf(PmCallHandler& pmCallHandler): _pmCallHandler(pmCallHandler){}

         ~SmAgResponseIf(){}

         virtual void acceptError() override;

         virtual void dialError() override;

         virtual void swapError() override;

         virtual void hangupError() override;

         virtual void transferError() override;
      private:
         PmCallHandler& _pmCallHandler;
      };

      /**
       * SmTimerUpdateIf class inherits the functions of SCI_TimerUpdate_OCB class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      class SmTimerUpdateIf final: public SCI_TimerUpdate_OCB
      {
      public:
         SmTimerUpdateIf(PmCallHandler& pmCallHandler): _pmCallHandler(pmCallHandler){}

         ~SmTimerUpdateIf(){}

         virtual void handleTimerCallback() override;
      private:
         PmCallHandler& _pmCallHandler;
      };

      /**
       * SmTraceObserverIf class inherits the functions of ysc::TraceObserver class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      template<typename T>
      class SmTraceObserverIf final: public ysc::TraceObserver<T>
      {
      public:
         SmTraceObserverIf(PmCallHandler& pmCallHandler): _pmCallHandler(pmCallHandler){}

         ~SmTraceObserverIf(){}

         virtual void stateEntered(T state) override;

         virtual void stateExited(T state) override;

      private:
         PmCallHandler& _pmCallHandler;
      };

      BdAddress                                        _deviceAddress;            /**< device address */
      SmUserActionIf*                                  _smUserActionIf;           /**< pointer for UserAction interface */
      SmAgCallStateUpdateIf*                           _smAgCallStateUpdateIf;    /**< pointer for AgCallStateUpdate interface */
      SmSmHandlerIf*                                   _smSmHandlerIf;            /**< pointer for SmHandler interface */
      SmAudioManagerRequestIf*                         _smAudioManagerRequestIf;  /**< pointer for AudioManagerRequest interface */
      SmAgResponseIf*                                  _smAgResponseIf;           /**< pointer for AgResponse interface */
      SmTimerUpdateIf*                                 _smTimerUpdateIf;          /**< pointer for TimerUpdate interface */
      SmTraceObserverIf<PmCallStateHandlerSmStates>*   _smTraceObserverIf;        /**< pointer for TraceObserver interface */
      AutoRejectCall                                   _autoRejectCall;           /**< AutoReject reason */

      PmCoreIfMessage_AcceptCallRequest  _acceptCallRequest;
      PmCoreIfMessage_HangupCallsRequest _cancelOutgoingCallRequest;
      std::map<ActType,CallInstanceList> _actAndCallInstanceListMap;    /**< map for acknowledgment token and CallInstance */
   };
}

#endif //YakinduSmVoiceCallIf_h
