/**
 * @file ISmVoiceCallIf.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the interface definition of the SmVoiceCallIf class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the state machine interfaces of voice call
 *
 * @ingroup PmCore
 */

#ifndef ISmVoiceCallIf_h
#define ISmVoiceCallIf_h

#include "PmCoreIfTypes.h"
#include "BtStackWrapperTypes.h"
#include "PmCoreIfMessageRequest.h"
#include "PmCoreIfMessageResult.h"

namespace pmcore
{
   class ISmVoiceCallIf
   {
   public:

      /**
       * Destructor of ISmVoiceCallIf class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      virtual ~ISmVoiceCallIf(){}

      // Request calls
      /**
       * This method is used to post the merge calls request to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for MergeCallRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void mergeCallsRequest(IN std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the Split Calls Request to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SplitCallsRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void splitCallsRequest(IN std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the Hangup Calls Request to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupCallsRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void hangupCallsRequest(IN std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the Redial Request to the voice call state machine
       *
       * @param[in]  pmCoreIfMessage - shared ptr for RedialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void redialRequest(IN std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the StartStop WaitingMode Request to the voice call state machine
       *
       * @param[in]  pmCoreIfMessage - shared ptr for StartStopWaitingModeRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void startStopWaitingModeRequest(
            IN std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the dial request to the voice call state machine
       *
       * @param[in]  pmCoreIfMessage - shared ptr for DialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void dialRequest(IN std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the AcceptCall Request to the voice call state machine
       *
       * @param[in]  pmCoreIfMessage - shared ptr for AcceptCallRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void acceptCallRequest(IN std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the SwapCall Request to the voice call state machine
       *
       * @param[in]  pmCoreIfMessage - shared ptr for SwapCallRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void swapCallRequest(IN std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the SpeedDial Request to the voice call state machine
       *
       * @param[in]  pmCoreIfMessage - shared ptr for SpeedDialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void speedDialRequest(IN std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the Transfer Audio Request to the voice call state machine
       *
       * @param[in]  pmCoreIfMessage - shared ptr for TransferAudioRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void transferAudioRequest(IN std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage) = 0;

      /**
       * This method is used to post transfer audio request directly to BT Stack
       *
       * @param[in]  deviceAddress - BT device address
       *             audio - Audio path. Should be "AG" or "HF"
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void transferAudioRequest(IN const BdAddress& deviceAddress, IN const std::string& audio) = 0;

      // **************************************************************************************************
      // Result calls

      /**
       * This method is used to post the Accept SCO Connect Result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for AcceptSCOConnectResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void acceptSCOConnectResult(
            IN std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the dial result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for DialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void dialResult(IN std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the speed dial result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SpeedDialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void speedDialResult(IN std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the redial result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for RedialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void redialResult(IN std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the swap call result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SwapCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void swapCallResult(IN std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the release and accept result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for ReleaseAndAcceptResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void releaseAndAcceptResult(
            IN std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the release and swap result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for ReleaseAndSwapResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void releaseAndSwapResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the hold and accept result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HoldAndAcceptResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void holdAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the hangup all result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupAllResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void hangupAllResult(IN std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the split calls result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SplitCallsResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void splitCallsResult(IN std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the merge calls result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for MergeCallsResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void mergeCallsResult(IN std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the hangup multiparty result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupMultipartyResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void hangupMultipartyResult(
            IN std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the hangup call result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void hangupCallResult(IN std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the accept call result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for AcceptCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void acceptCallResult(IN std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the hold incoming call result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HoldIncomingCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void holdIncomingCallResult(
            IN std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to post the transfer Audio Result to the voice call state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for TransferAudioResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void transferAudioResult(
            IN std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage) = 0;

      /**
       * This method is used to set the call count to the voice call state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] callCount - call count
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void setCallCount(IN const BdAddress& deviceAddress, IN const CallCount callCount) = 0;

      /**
       * This method is used to post the AG call state update to the voice call state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] callState - call state
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAgCallStateUpdate(IN const BdAddress& deviceAddress, IN const CallStateEnumType callState) = 0;

      /**
       * This method is used to post the AudioManager update to the voice call state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] amEvent - audio manager event update
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onAudioManagerEventUpdate(IN const BdAddress& deviceAddress,
            IN const AudioManagerEventType amEvent) = 0;

      /**
       * This method is used to update the SCO status to the voice call state machine
       *
       * @param[in] deviceAddress - device address
       * @param[in] scoStatus - sco status
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onScoConnectionStatusChanged(IN const BdAddress& deviceAddress, IN const SCOStatus scoStatus) = 0;

      /**
       * This method is used to switch a device to Passive
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToPassiveRequest(IN const BdAddress& deviceAddress) = 0;

      /**
       * This method is used to switch the device to Passive
       * If the SCO is established for this device, it should be relinquished and the Phone channel should be released
       * if acquired. Because the device is no more an Active device to hold the Phone channel
       *
       * @param[in] bdAddress       - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToPassiveClientRequest(const BdAddress& deviceAddress) = 0;

      /**
       * This method is used to switch a device to Active
       *
       * @param[in] deviceAddress - device address
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void switchToActiveRequest(IN const BdAddress& deviceAddress) = 0;

      /**
       * This method is used to auto reject a call
       *
       * @param[in] deviceAddress - device address
       * @param[in] autoRejectCall - Reference to AutoRejectCall
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void autoRejectCallRequest(IN const BdAddress& deviceAddress,
            IN const AutoRejectCall autoRejectCall) = 0;

      /**
       * This method is used to inform about the removal of a call
       *
       * @param[in] deviceAddress - device address
       * @param[in] callInstance - instance of the removed call
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onCallRemoved(IN const BdAddress& deviceAddress,
            IN const CallInstance callInstance) = 0;
   };
}

#endif //ISmVoiceCallIf_h
