/**
 * @file CallController.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the CallController class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file provides the CallController interfaces of PmCore
 *
 * @ingroup PmCore
 */

#ifndef CallController_h
#define CallController_h

#include "PmCoreIfTypes.h"
#include "PropertyUpdateHandler.h"
#include "ITimer.h"
#include "AsfTimer.h"
#include "PmAudioManagerWrapper.h"
#include "AsfTimerCallbackData.h"

namespace pmcore
{
   class ISmVoiceCallIf;
   class PmCoreIfMessage_MergeCallsRequest;
   class PmCoreIfMessage_SplitCallsRequest;
   class PmCoreIfMessage_HangupCallsRequest;
   class PmCoreIfMessage_RedialRequest;
   class PmCoreIfMessage_StartStopWaitingModeRequest;
   class PmCoreIfMessage_DialRequest;
   class PmCoreIfMessage_AcceptCallRequest;
   class PmCoreIfMessage_SwapCallRequest;
   class PmCoreIfMessage_SpeedDialRequest;
   class PmCoreIfMessage_SendDTMFRequest;
   class PmCoreIfMessage_TransferAudioRequest;

   class PmCoreIfMessage_AcceptSCOConnectResult;
   class PmCoreIfMessage_GetCallsResult;
   class PmCoreIfMessage_DialResult;
   class PmCoreIfMessage_SpeedDialResult;
   class PmCoreIfMessage_RedialResult;
   class PmCoreIfMessage_SwapCallResult;
   class PmCoreIfMessage_ReleaseAndAcceptResult;
   class PmCoreIfMessage_ReleaseAndSwapResult;
   class PmCoreIfMessage_HoldAndAcceptResult;
   class PmCoreIfMessage_HangupAllResult;
   class PmCoreIfMessage_SplitCallsResult;
   class PmCoreIfMessage_MergeCallsResult;
   class PmCoreIfMessage_HangupMultipartyResult;
   class PmCoreIfMessage_SendDTMFResult;
   class PmCoreIfMessage_HangupCallResult;
   class PmCoreIfMessage_AcceptCallResult;
   class PmCoreIfMessage_HoldIncomingCallResult;
   class PmCoreIfMessage_TransferAudioResult;
   
   // Response messages from PM AudioManager
   class PmCoreIfMessage_PrepareAudioRouteResponse;
   class PmCoreIfMessage_PlayAudioResponse;
   class PmCoreIfMessage_StopAudioResponse;
   class PmCoreIfMessage_SetMicMuteStateResponse;
   class PmCoreIfMessage_PauseAudioResponse;

   class CallController : public PropertyUpdateHandler
   {
   public:
      /**
       * Constructor of CallController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      CallController();

      /**
       * Destructor of CallController class
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      ~CallController();

      /**
       * Copy constructor of CallController class
       *
       * @param[in] ref - Reference to CallController
       * @param[out]
       * @param[in,out]
       *
       * @return
       *
       */
      CallController(const CallController& ref);

      /**
       * operator= overloaded function
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return Returns the object of the CallController
       *
       */
      CallController& operator=(const CallController& ref);

      /**
       * This method is used to subscribe interested properties in BT stack event notifier.
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void subscribeToBtStackEventNotifier();

      /**
       * This method is used to get property changes update from BT Stack
       * This method is overridden from PropertyUpdateHandler class
       *
       * @param[in] propertyId      - Id of the property or event
       * @param[in] propertyDetails - Details of the property or event
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      virtual void onPropertyUpdate(IN const PmCorePropertyAndEventId propertyId,
            IN std::shared_ptr<void> propertyDetails) override;

      // **************************************************************************************************
      // Request calls
      /**
       * This method is used to post the Get Phone Call AudioActive Request to the call Controller state machine
       *
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getPhoneCallAudioActiveRequest(IN const ActType act);

      /**
       * This method is used to post the GetCall Status List request to the call Controller state machine
       *
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getCallStatusListRequest(IN const ActType act);

      /**
       * This method is used to post the Get Microphone MuteState Request to the call Controller state machine
       *
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getMicrophoneMuteStateRequest(IN const ActType act);

      /**
       * This method is used to post the Set Microphone MuteState Request to the call Controller state machine
       *
       * @param[in] muteState - Mute state of the microphone
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult setMicrophoneMuteStateRequest(IN const MuteState muteState);

      /**
       * This method is used to post the merge calls request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for MergeCallsRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void mergeCallsRequest(IN std::shared_ptr<PmCoreIfMessage_MergeCallsRequest> pmCoreIfMessage);

      /**
       * This method is used to post the Split Calls Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SplitCallsRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void splitCallsRequest(IN std::shared_ptr<PmCoreIfMessage_SplitCallsRequest> pmCoreIfMessage);

      /**
       * This method is used to post the Hangup Calls Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupCallsRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupCallsRequest(IN std::shared_ptr<PmCoreIfMessage_HangupCallsRequest> pmCoreIfMessage);

      /**
       * This method is used to post the Redial Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for RedialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void redialRequest(IN std::shared_ptr<PmCoreIfMessage_RedialRequest> pmCoreIfMessage);

      /**
       * This method is used to post the Set Ringtone MuteState Request to the call Controller state machine
       *
       * @param[in] bdAddress         - BT address of the device
       * @param[in] ringtoneMuteState - Mute state of the ringtone
       * @param[in] act               - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult setRingtoneMuteStateRequest(IN const BdAddress& bdAddress, IN const MuteState ringtoneMuteState,
            IN const ActType act);

      /**
       * This method is used to post the Get WaitingMode State Request to the call Controller state machine
       *
       * @param[in] bdAddress - BT address of the device
       * @param[in] act       - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult getWaitingModeStateRequest(IN const BdAddress& bdAddress, IN const ActType act);

      /**
       * This method is used to post the set WaitingMode FilePath Request to the call Controller state machine
       *
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult setWaitingModeFilePathRequest(IN const WaitingModeFilePath& waitingModeFilePath,
            IN const ActType act);

      /**
       * This method is used to get WaitingMode FilePath for the call Controller state machine
       *
       * @param[in]
       * @param[out] WaitingModeFilePath - File path to the waitingMode tone
       * @param[in,out]
       *
       * @return void
       */
       void getWaitingModeFilePath(WaitingModeFilePath& waitingModeFilePath);

      /**
       * This method is used to post the StartStop WaitingMode Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for StartStopWaitingModeRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void startStopWaitingModeRequest(IN std::shared_ptr<PmCoreIfMessage_StartStopWaitingModeRequest> pmCoreIfMessage);

      /**
       * This method is used to post the dial request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for DialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void dialRequest(std::shared_ptr<PmCoreIfMessage_DialRequest> pmCoreIfMessage);

      /**
       * This method is used to post the AcceptCall Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for AcceptCallRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptCallRequest(IN std::shared_ptr<PmCoreIfMessage_AcceptCallRequest> pmCoreIfMessage);

      /**
       * This method is used to post the SwapCall Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SwapCallRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void swapCallRequest(IN std::shared_ptr<PmCoreIfMessage_SwapCallRequest> pmCoreIfMessage);

      /**
       * This method is used to post the SpeedDial Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SpeedDialRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void speedDialRequest(std::shared_ptr<PmCoreIfMessage_SpeedDialRequest> pmCoreIfMessage);

      /**
       * This method is used to post the SendDTMF Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SendDTMFRequest
       * @param[out]
       * @param[in,out]
       *
       * @return PmResult - PmCore result message
       */
      PmResult sendDTMFRequest(IN std::shared_ptr<PmCoreIfMessage_SendDTMFRequest> pmCoreIfMessage);

      /**
       * This method is used to post the Transfer Audio Request to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for TransferAudioRequest
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void transferAudioRequest(IN std::shared_ptr<PmCoreIfMessage_TransferAudioRequest> pmCoreIfMessage);

      // **************************************************************************************************
      // Result calls
      /**
       * This method is used to post the request PhoneNumber Result to the call Controller state machine
       *
       * @param[in] btsResult - BT stack result for the request PhoneNumber Request
       * @param[in] bdAddress - BT address of the device
       * @param[in] telephoneNumber - telephone number
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void requestPhoneNumberResult(IN const BTSResult btsResult, IN const BdAddress& bdAddress,
            IN const TelephoneNumber& telephoneNumber, IN const ActType act);

      /**
       * This method is used to post the Accept SCO Connect Result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for AcceptSCOConnectResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptSCOConnectResult(IN std::shared_ptr<PmCoreIfMessage_AcceptSCOConnectResult> pmCoreIfMessage);

      /**
       * This method is used to post the get calls result to the call Controller state machine
       *
       * @param[in] btsResult - BT stack result for the get calls request
       * @param[in] bdAddress - BT address of the device
       * @param[in] act - acknowledgment token
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void getCallsResult(IN std::shared_ptr<PmCoreIfMessage_GetCallsResult> pmCoreIfMessage);

      /**
       * This method is used to post the dial result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for DialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void dialResult(IN std::shared_ptr<PmCoreIfMessage_DialResult> pmCoreIfMessage);

      /**
       * This method is used to post the speed dial result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SpeedDialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void speedDialResult(IN std::shared_ptr<PmCoreIfMessage_SpeedDialResult> pmCoreIfMessage);

      /**
       * This method is used to post the redial result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for RedialResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void redialResult(IN std::shared_ptr<PmCoreIfMessage_RedialResult> pmCoreIfMessage);

      /**
       * This method is used to post the swap call result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SwapCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void swapCallResult(IN std::shared_ptr<PmCoreIfMessage_SwapCallResult> pmCoreIfMessage);

      /**
       * This method is used to post the release and accept result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for ReleaseAndAcceptResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void releaseAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndAcceptResult> pmCoreIfMessage);

      /**
       * This method is used to post the release and swap result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for ReleaseAndSwapResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void releaseAndSwapResult(IN std::shared_ptr<PmCoreIfMessage_ReleaseAndSwapResult> pmCoreIfMessage);

      /**
       * This method is used to post the hold and accept result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HoldAndAcceptResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void holdAndAcceptResult(IN std::shared_ptr<PmCoreIfMessage_HoldAndAcceptResult> pmCoreIfMessage);

      /**
       * This method is used to post the hangup all result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupAllResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupAllResult(IN std::shared_ptr<PmCoreIfMessage_HangupAllResult> pmCoreIfMessage);

      /**
       * This method is used to post the split calls result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SplitCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void splitCallsResult(IN std::shared_ptr<PmCoreIfMessage_SplitCallsResult> pmCoreIfMessage);

      /**
       * This method is used to post the merge calls result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for MergeCallsResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void mergeCallsResult(IN std::shared_ptr<PmCoreIfMessage_MergeCallsResult> pmCoreIfMessage);

      /**
       * This method is used to post the hangup multiparty result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupMultipartyResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupMultipartyResult(IN std::shared_ptr<PmCoreIfMessage_HangupMultipartyResult> pmCoreIfMessage);

      /**
       * This method is used to post the send DTMF result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for SendDTMFResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void sendDTMFResult(IN std::shared_ptr<PmCoreIfMessage_SendDTMFResult> pmCoreIfMessage);

      /**
       * This method is used to post the hangup call result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HangupCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void hangupCallResult(IN std::shared_ptr<PmCoreIfMessage_HangupCallResult> pmCoreIfMessage);

      /**
       * This method is used to post the accept call result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for AcceptCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptCallResult(IN std::shared_ptr<PmCoreIfMessage_AcceptCallResult> pmCoreIfMessage);

      /**
       * This method is used to post the hold incoming call result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for HoldIncomingCallResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void holdIncomingCallResult(IN std::shared_ptr<PmCoreIfMessage_HoldIncomingCallResult> pmCoreIfMessage);

      /**
       * This method is used to post the transfer Audio Result to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for transferAudioResult
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void transferAudioResult(IN std::shared_ptr<PmCoreIfMessage_TransferAudioResult> pmCoreIfMessage);

      /**
       * This method is used to post the prepare audio route response to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for PrepareAudioRouteResponse
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void prepareAudioRouteResponse(IN std::shared_ptr<PmCoreIfMessage_PrepareAudioRouteResponse> pmCoreIfMessage);

      /**
       * This method is used to post the play Audio response to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for playAudioResponse
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void playAudioResponse(IN std::shared_ptr<PmCoreIfMessage_PlayAudioResponse> pmCoreIfMessage);

      /**
       * This method is used to post the stop Audio response to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for stopAudioResponse
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopAudioResponse(IN std::shared_ptr<PmCoreIfMessage_StopAudioResponse> pmCoreIfMessage);

      /**
       * This method is used to post the set Mic mute state response to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for setMicMuteStateResponse
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void setMicMuteStateResponse(IN std::shared_ptr<PmCoreIfMessage_SetMicMuteStateResponse> pmCoreIfMessage);

      /**
       * This method is used to post the pause Audio response to the call Controller state machine
       *
       * @param[in] pmCoreIfMessage - shared ptr for pauseAudioResponse
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void pauseAudioResponse(IN std::shared_ptr<PmCoreIfMessage_PauseAudioResponse> pmCoreIfMessage);

      /**
       * This method is used to start call timer for new active and held calls
       *
       * @param[in] token  - Combination of BTaddress and callInstance
       * @param[in] duration - Duration of the timer
       * @param[in] repeatCount - repetition count
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void startAsfTimer(std::string token, ms_t duration, RepeatCount repeatCount);

      /**
       * This method is used to stop call timer for disconnected calls
       *
       * @param[in] token  - Combination of BTaddress and callInstance
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void stopAsfTimer(std::string token);

      /**
       * This method is used to delete call timer instance for disconnected calls
       *
       * @param[in] token  - Combination of BTaddress and callInstance
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void deleteAsfTimer(std::string token);

      /**
       * This method is used to notify timer elapsed callback to the call Controller state machine
       *
       * @param[in] data              - timer data
       * @param[in] timerCallbackData - timer callback data
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void timerElapsed(com::bosch::pmcommon::TimerData<std::string,ms_t,RepeatCount> data,
            com::bosch::pmcommon::AsfTimerCallbackData timerCallbackData);

      /**
       * This method is used to post the CallAudioActive changes to the PmCore CallbackIfWrapper
       *
       * @param[in] bdAddress         - BT address of the device
       * @param[in] audioActiveStatus - status of Vehicle Audio
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onCallAudioActiveChanged(IN const BdAddress& bdAddress, IN const AudioActiveStatus& audioActiveStatus);

      /**
       * This method is used to post the CallStatus list to the PmCore CallbackIfWrapper
       *
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onCallStatusListChanged();

      /**
       * This method is used to get WaitingMode state from CallController StateMachine
       *
       * @param[in] bdAddress        - BT address of the device
       * @param[in] waitingModeState - state of waitingMode
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void updateWaitingModeState(IN const BdAddress& bdAddress, IN const WaitingModeState& waitingModeState);

      /**
       * This method is used to accept waitingModeCall from CallController StateMachine
       *
       * @param[in] bdAddress        - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void acceptWaitingModeCall(IN const BdAddress& bdAddress);

      /**
       * This method is used to accept waitingModeCall from CallController StateMachine
       *
       * @param[in] bdAddress        - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void holdAndAcceptWaitingModeCall(IN const BdAddress& bdAddress);

      /**
       * This method is used to check the Inband ringing supported state
       *
       * @param[in] bdAddress        - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return true- if Inband ringtone supported, false- if Inband ringtone not supported
       */
      bool isInbandRinging(const BdAddress& deviceAddress);

      /**
       * This method is used to check the Idle call status of a device
       *
       * @param[in] bdAddress        - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return true- if no call is present, false- if some call is present
       */
      bool isCallsInIdleState(const BdAddress& deviceAddress);

      /**
       * This method is used to switch the device to Active
       *
       * @param[in] bdAddress        - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToActive(const BdAddress& deviceAddress);

      /**
       * This method is used to switch the device to Passive
       * If the SCO is established for this device, it should be relinquished and the acquired Phone channel should be
       * set to Pause state
       *
       * @param[in] bdAddress       - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassive(const BdAddress& deviceAddress);

      /**
       * This method is used to switch the device to Passive
       * If the SCO is established for this device, it should be relinquished and the Phone channel should be released
       * if acquired. Because the device is no more an Active device to hold the Phone channel
       *
       * @param[in] bdAddress       - BT address of the device
       * @param[in]
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void switchToPassiveClientRequest(const BdAddress& deviceAddress);

      /**
       * This method is used to generate AgCallState events for CallController StateMachine
       *
       * @param[in] bdAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void generateAgCallStateEventToSM(IN const BdAddress& bdAddress);

      /**
       * This method is used to request Get calls interface of BT Stack for the given device address
       *
       * @param[in] bdAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void postGetCallsRequest2BtStack(const BdAddress& deviceAddress);

      /**
       * This method is used to get the TelephoneNumber for given CallInstance
       *
       * @param[in] callInstance - Instance of the VoiceCall
       * @param[out]
       * @param[in,out]
       *
       * @return TelephoneNumber -   TelephoneNumber of the voiceCall
       */
      TelephoneNumber getTelephoneNumberFromCallInstance(IN const BdAddress& bdAddress, IN const CallInstance& callInstance);

      /**
       * This method is used to process SCO disconnection from a notification from SCT
       *
       * @param[in] bdAddress - BT address of the device
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void processSCODisconnection(IN const BdAddress& bdAddress);

      /**
       * This method is used to check and update the PM state change based on SCO and call status
       *
       * @param[in]  bdAddress - BT address of the device
       *             scoStatus - SCO status
       * @param[out]
       * @param[in,out]
       *
       * @return
       */
      void updatePMStateChanged(IN const BdAddress& bdAddress, IN const SCOStatus scoStatus);

   private:

      /**
       * This method is used to post the WaitingModeState changes to the PmCore CallbackIfWrapper
       *
       * @param[in] bdAddress        - BT address of the device
       * @param[in] waitingModeState - state of waitingMode
       * @param[out]
       * @param[in,out]
       *
       * @return void
       */
      void onWaitingModeStateChanged(IN const BdAddress& bdAddress, IN const WaitingModeState& waitingModeState);

      PropertyIdList                           _propertyIdList;          /**< list of property id's */
      ISmVoiceCallIf*                          _smVoiceCallIf;           /**< pointer for voice call sm interface */
      CallStatusList                           _callStatusList;          /**< list of call status */
      std::map<BdAddress, WaitingModeState>    _waitingModeStateListMap; /**< waiting mode state list */
      std::map<BdAddress, bool>                _inBandRingingListMap;    /**< Inband ringing state list */
      MicrophoneMuteState                      _microphoneMuteState;     /**< Microphone MuteState */
      WaitingModeFilePath                      _waitingModeFilePath;     /**< WaitingMode filepath */

      //String is a combination of address and CallInstance seperated by '/'
      std::map< std::string, com::bosch::pmcommon::ITimer<CallController,std::string,ms_t,RepeatCount>*> _asfTimerInstanceMap;

      PhoneCallAudioActive _phoneCallAudioActive; //FIXME:this should be adopted to device specific
   };
}

#endif // CallController_h

