/**
 * @file ObjectPathInfo.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the ObjectPathInfo class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file maintains the object path of HFP interfaces for the connected devices.
 *
 * @ingroup BtStackWrapper
 */

#ifndef ObjectPathInfo_h
#define ObjectPathInfo_h

#include "PmInterfaceTypesInternal.h"
#include "BtStackWrapperTypesInternal.h"

namespace evobtstackwrapper {

/**
 *
 */
class ObjectPathInfo
{
public:
   /**
    * Constructor of ObjectPathInfo class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   ObjectPathInfo();

   /**
    * Destructor of ObjectPathInfo class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   ~ObjectPathInfo();

   /**
    * This method is used to set the Handsfree object path
    *
    * @param[in] objPath: handsfree object path
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setHandsfreeObjectPath(const ObjectPath& objPath);

   /**
    * This method is used to get the Handsfree object path
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return Handsfree object path
    */
   const ObjectPath& getHandsfreeObjectPath(void) const;

   /**
    * This method is used to clear the Handsfree object path
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void removeHandsfreeObjectPath(void);

   /**
    * This method is used to set the VoiceCall object path
    *
    * @param[in] callinstance: call instance
    * @param[in] objPath: voice call object path
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setVoiceCallObjectPath(const CallInstance callInstance, const ObjectPath& objPath);

   /**
    * This method is used to get the VoiceCall object path from CallInstance
    *
    * @param[in] callInstance: call instance
    * @param[out] objPath: voice call object path
    * @param[in,out]
    *
    * @return void
    */
   void getVoiceCallObjectPath(const CallInstance callInstance, ObjectPath& objPath);

   /**
    * This method is used to get the CallInstance from VoiceCall object path
    *
    * @param[in] objPath: voice call object path
    * @param[out] callInstance: call instance
    * @param[in,out]
    *
    * @return bool
    */
   bool getCallInstance(const ObjectPath& objPath, CallInstance& callInstance);

   /**
    * This method is used to clear the VoiceCall object path
    *
    * @param[in] callInstance: call instance
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void removeVoiceCallObjectPath(const CallInstance callInstance);

   /**
    * This method is used to set the NetworkOperator object path
    *
    * @param[in] networkOperatorInstance: network operator instance
    * @param[in] objPath: network operator object path
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setNetworkOperatorObjectPath(const NetworkOperatorInstance networkOperatorInstance, const ObjectPath& objPath);

   /**
    * This method is used to get the NetworkOperator object path from NetworkOperatorinstance
    *
    * @param[in] networkOperatorInstance: network operator instance
    * @param[out] objPath: network operator object path
    * @param[in,out]
    *
    * @return void
    */
   void getNetworkOperatorObjectPath(const NetworkOperatorInstance networkOperatorInstance, ObjectPath& objPath);

   /**
    * This method is used to get the NetworkOperatorinstance from NetworkOperator object path
    *
    * @param[in] objPath: network operator object path
    * @param[out] networkOperatorInstance: network operator instance
    * @param[in,out]
    *
    * @return bool
    */
   bool getNetworkOperatorInstance(const ObjectPath& objPath, NetworkOperatorInstance& networkOperatorInstance);

   /**
    * This method is used to clear the NetworkOperator object path
    *
    * @param[in] networkOperatorInstance: network operator instance
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void removeNetworkOperatorObjectPath(const NetworkOperatorInstance networkOperatorInstance);

protected:
   ObjectPath _handsfreeObjectPath; /**< common object path of call volume, handsfree, modem,
                                         network registration, siri and voice call manager proxies */
   std::map<CallInstance, ObjectPath> _voiceCallObjectPath;                  /** < object path of voice call proxy */
   std::map<NetworkOperatorInstance, ObjectPath> _networkOperatorObjectPath; /**< object path of network operator proxy*/
};

typedef std::map< ::pmcore::BdAddress, ObjectPathInfo> ObjectPathInfoList;

} //evobtstackwrapper

#endif //ObjectPathInfo_h
